/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file XMLParser.cxx
    \author Romain BONDUE
    \date 03/08/2006 */
#include <fstream>
#include <sstream>

#include "XMLParser.h"
#include "ErrorsCode.h"



namespace
{
    void MakeError (::GError** ppError, ::GMarkupParseContext* pContext,
                    int Code, const std::string& Label) throw()
    {
        int LineNumber, CharacterPosition;
        ::g_markup_parse_context_get_position (pContext, &LineNumber,
                                                &CharacterPosition);
        std::ostringstream Os;
        Os << Label << '\n' << "Line : " << LineNumber << '\t' << "Column : "
           << CharacterPosition;
        *ppError = ::g_error_new_literal (G_MARKUP_ERROR, Code,
                                          Os.str().c_str());
        
    } // MakeError()

} // anonymous namespace



const char* const nsUserData::XMLHeader
                                ("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
const char* const nsUserData::Markup2B ("</");
const char* const nsUserData::Markup2E (">\n");



void nsUserData::CXMLParser::Extract (const std::string& FileName)
                                                throw (nsErrors::CException)
{
    std::ifstream Is (FileName.c_str());
    if (!Is) return; // No file : nothing to load.

    const ::GMarkupParser Parser = {
        reinterpret_cast<void (*) (::GMarkupParseContext*, const char*,
                                   const char**,const char** AttributeValues,
                                   void*, ::GError**)> (&_StartElement),
        reinterpret_cast<void (*) (::GMarkupParseContext*, const char*,
                                   void*, ::GError**)> (&_EndElement),
        reinterpret_cast<void (*) (::GMarkupParseContext*, const char*,
                                   ::gsize, void*, ::GError**)> (&_Text),
        0, // &Ignored
        reinterpret_cast<void (*) (::GMarkupParseContext*, ::GError*,
                                   void*)> (&_Error)};
    ::GMarkupParseContext* const pContext  (::g_markup_parse_context_new
                                (&Parser, ::GMarkupParseFlags (0), this, 0));
    ::GError* pError (0);

    try
    {
        Is.exceptions (std::ios_base::badbit);
        const unsigned BufferSize (256);
        char Buffer [BufferSize];
        while (Is)
        {
            Is.read (Buffer, BufferSize);
            if (!::g_markup_parse_context_parse (pContext, Buffer, Is.gcount(),
                                                 &pError))
                throw nsErrors::CException (pError->message, pError->code);
        }
        if (!::g_markup_parse_context_end_parse (pContext, &pError))
            throw nsErrors::CException (pError->message,
                                    nsErrors::ProfilesExtractionExtraToken);
        ::g_markup_parse_context_free (pContext);
    }
    catch (const std::ios_base::failure& Exc)
    {
        ::g_markup_parse_context_free (pContext);
        throw nsErrors::CException (Exc.what(),
                                    nsErrors::ProfilesExtractionFailure);
    }
    catch (const nsErrors::CException&)
    {
        ::g_markup_parse_context_free (pContext);
        ::g_error_free (pError);
        throw;
    }

} // Extract()


void nsUserData::CXMLParser::_StartElement
                  (::GMarkupParseContext* pContext, const char* ElementName,
                   const char** AttributeNames, const char** AttributeValues,
                   CXMLParser* This, ::GError** ppError) throw()
{
    try{This->StartElement (ElementName, AttributeNames, AttributeValues);}
    catch (const nsErrors::CException& Exc)
    {
        MakeError (ppError, pContext, Exc.GetCode(), Exc.GetMsg());
    }
    catch (const std::bad_alloc& Exc)
    {
        MakeError (ppError, pContext, nsErrors::OutOfMemory, Exc.what());
    }

} // _StartElement()


void nsUserData::CXMLParser::_EndElement (::GMarkupParseContext* pContext,
                                          const char* ElementName,
                                          CXMLParser* This,
                                          ::GError** ppError) throw()
{
    try{This->EndElement (ElementName);}
    catch (const nsErrors::CException& Exc)
    {
        MakeError (ppError, pContext, Exc.GetCode(), Exc.GetMsg());
    }

} // _EndElement()


    // Text is not null-terminated.
void nsUserData::CXMLParser::_Text (::GMarkupParseContext* pContext,
                                    const char* Text, ::gsize Length,
                                    CXMLParser* This, ::GError** ppError)
                                                                        throw()
{     // This function is often called when we don't expect it.
    if (!Length) return;
    unsigned i (0);
    for ( ; i < Length ; ++i)   // We ignore indentation and things like that.
        if (!isspace (Text [i])) break;
    if (i == Length) return;

    try{This->Text (::g_markup_parse_context_get_element (pContext),
                    std::string (Text, 0, Length));}
    catch (const nsErrors::CException& Exc)
    {
        MakeError (ppError, pContext, Exc.GetCode(), Exc.GetMsg());
    }

} // _Text()


    // IgnoredText is not null-terminated.
/*void nsUserData::CXMLParser::Ignored (::GMarkupParseContext* pContext,
                                        const char* IgnoredText,
                                        unsigned Length, CXMLParser*,
                                        ::GError** ppError) throw()
{

} // Ignored()*/


void nsUserData::CXMLParser::_Error (::GMarkupParseContext* pContext,
                                     ::GError* ppError, CXMLParser* This)
                                                                        throw()
{
    This->Error();

} // Error()
