use failure;
use std::cell::RefCell;
use std::rc::Rc;

use super::markup_links;

use html5ever::rcdom::Node;
use html5ever::rcdom::NodeData;
use html5ever::rcdom::RcDom;
use html5ever::tendril::TendrilSink;
use html5ever::tree_builder::Attribute;
use html5ever::tree_builder::TreeBuilderOpts;
use html5ever::{parse_document, ParseOpts};

/// Each block contains the text formatted in pango format
#[derive(Debug, Clone, PartialEq)]
pub enum HtmlBlock {
    Text(String),
    Heading(u32, String),
    UList(Vec<String>),
    OList(Vec<String>),
    Code(String),
    Quote(Rc<Vec<HtmlBlock>>),
}

pub fn markup_html(s: &str) -> Result<Vec<HtmlBlock>, failure::Error> {
    let opts = ParseOpts {
        tree_builder: TreeBuilderOpts {
            drop_doctype: true,
            ..Default::default()
        },
        ..Default::default()
    };
    let dom = parse_document(RcDom::default(), opts)
        .from_utf8()
        .read_from(&mut s.as_bytes())?;

    let document = &dom.document;
    let html = &document.children.borrow()[0];
    let body = &html.children.borrow()[1];

    Ok(convert_node(body))
}

fn to_pango(t: &str) -> Option<&'static str> {
    let allowed = [
        "u", "del", "s", "em", "i", "strong", "b", "code", "a"
    ];

    if !allowed.contains(&t) {
        return None;
    }

    match t {
        "a" => Some("a"),
        "em" | "i" => Some("i"),
        "strong" | "b" => Some("b"),
        "del" | "s" => Some("s"),
        "u" => Some("u"),
        "code" => Some("tt"),
        _ => None,
    }
}

fn parse_link(node: &Node, attrs: &RefCell<Vec<Attribute>>) -> String {
    let mut link = "".to_string();
    for attr in attrs.borrow().iter() {
        let s = attr.name.local.to_string();
        match &s[..] {
            "href" => {
                link = attr.value.to_string();
            }
            _ => {}
        }
    }

    format!("<a href=\"{}\">{}</a>", link, get_text_content(node))
}

fn get_text_content(node: &Node) -> String {
    node.children.borrow().iter()
        .map(|node| match node.data {
            NodeData::Text { contents: ref c } => markup_links(&c.borrow()),
            NodeData::Element { name: ref n, attrs: ref a, .. } => {
                let inside = get_text_content(node);

                if &n.local == "a" {
                    return parse_link(node, a);
                }

                match to_pango(&n.local) {
                    Some(t) => format!("<{0}>{1}</{0}>", t, inside),
                    None => inside,
                }
            },
            _ => get_text_content(node)
        })
        .collect::<Vec<String>>()
        .concat()
}

fn get_plain_text_content(node: &Node) -> String {
    node.children.borrow().iter()
        .map(|node| match node.data {
            NodeData::Text { contents: ref c } => format!("{}", &c.borrow()),
            _ => get_text_content(node)
        })
        .collect::<Vec<String>>()
        .concat()
}

fn get_li_elements(node: &Node) -> Vec<String> {
    node.children.borrow().iter()
        .map(|node| match node.data {
            NodeData::Element { name: ref n, .. } if &n.local == "li" => {
                Some(get_text_content(node))
            },
            _ => None
        })
        .filter(|n| n.is_some())
        .map(|n| n.unwrap())
        .collect::<Vec<String>>()
}

fn convert_node(node: &Node) -> Vec<HtmlBlock> {
    let mut output = vec![];
    match node.data {
        NodeData::Text { contents: ref c } => {
            let s = markup_links(&c.borrow());
            if !s.trim().is_empty() {
                output.push(HtmlBlock::Text(s));
            }
        }

        NodeData::Element {
            name: ref n,
            attrs: ref a,
            ..
        } => {
            match &n.local as &str {
                "body" => {
                    for child in node.children.borrow().iter() {
                        for block in convert_node(child) {
                            output.push(block);
                        }
                    }
                }

                h if ["h1", "h2", "h3", "h4", "h5", "h6"].contains(&h) => {
                    let n: u32 = h[1..].parse().unwrap_or(6);
                    let text = get_text_content(node);
                    output.push(HtmlBlock::Heading(n, text));
                }

                "a" => {
                    let link = parse_link(node, a);
                    output.push(HtmlBlock::Text(link));
                }

                "pre" => {
                    let text = get_plain_text_content(node);
                    output.push(HtmlBlock::Code(text));
                }

                "ul" => {
                    let elements = get_li_elements(node);
                    output.push(HtmlBlock::UList(elements));
                }

                "ol" => {
                    let elements = get_li_elements(node);
                    output.push(HtmlBlock::OList(elements));
                }

                "blockquote" => {
                    let mut content = vec![];
                    for child in node.children.borrow().iter() {
                        for block in convert_node(child) {
                            content.push(block);
                        }
                    }
                    output.push(HtmlBlock::Quote(Rc::new(content)));
                }

                tag => {
                    let content = get_text_content(node);
                    let block = match to_pango(&tag) {
                        Some(t) => format!("<{0}>{1}</{0}>", t, content),
                        None => String::from(content),
                    };

                    output.push(HtmlBlock::Text(block));
                }
            };
        }
        _ => {}
    }

    // joining text nodes
    output.drain(..)
        .fold(vec![], |mut v, b| {
            let last = v.last().clone();
            if let (&HtmlBlock::Text(ref c), Some(HtmlBlock::Text(a))) = (&b, last) {
                let t = a.clone() + &c;
                v.pop();
                v.push(HtmlBlock::Text(t));
            } else {
                v.push(b);
            }

            v
        })
}

#[cfg(test)]
mod test {
    use super::*;
    use pretty_assertions::assert_eq;

    #[test]
    fn test_html_blocks() {
        let text = "<h1>heading 1 <em>italic</em></h1>\n<h2>heading 2</h2>\n<p>Some text with <em>markup</em> and <strong>other</strong> and more text and some <code>inline code</code>, that's all. And maybe some links http://google.es or <a href=\"http://gnome.org\">GNOME</a>.</p>\n<pre><code>Block text\n</code></pre>\n<ul>\n<li>This is a list</li>\n<li>second element</li>\n</ul>\n<ol>\n<li>another list</li>\n<li>that's all</li>\n</ol>\n";
        let blocks = markup_html(text);
        assert!(blocks.is_ok());
        let blocks = blocks.unwrap();
        println!("{:?}", blocks);
        assert_eq!(blocks.len(), 6);

        assert_eq!(blocks[0], HtmlBlock::Heading(1, "heading 1 <i>italic</i>".to_string()));
        assert_eq!(blocks[1], HtmlBlock::Heading(2, "heading 2".to_string()));

        assert_eq!(if let HtmlBlock::Text(_s) = &blocks[2] { true } else { false }, true);
        assert_eq!(if let HtmlBlock::Code(_s) = &blocks[3] { true } else { false }, true);
        assert_eq!(if let HtmlBlock::UList(_n) = &blocks[4] { true } else { false }, true);
        assert_eq!(if let HtmlBlock::OList(_n) = &blocks[5] { true } else { false }, true);
    }

    #[test]
    fn test_html_blocks_quote() {
        let text = "
<blockquote>
<h1>heading 1 <em>bold</em></h1>
<h2>heading 2</h2>
<p>Some text with <em>markup</em> and <strong>other</strong> and more things ~~strike~~ and more text and some <code>inline code</code>, that's all. And maybe some links http://google.es or &lt;a href=&quot;http://gnome.org&quot;&gt;GNOME&lt;/a&gt;, <a href=\"http://gnome.org\">GNOME</a>.</p>
<pre><code>`Block text
`
</code></pre>
<ul>
<li>This is a list</li>
<li>second element</li>
</ul>
</blockquote>
<p>quote :D</p>
";

        let blocks = markup_html(text);
        assert!(blocks.is_ok());
        let blocks = blocks.unwrap();
        assert_eq!(blocks.len(), 2);

        if let HtmlBlock::Quote(blks) = &blocks[0] {
            assert_eq!(blks.len(), 5);
        }
    }

    #[test]
    fn test_mxreply() {
        let text = "<mx-reply><blockquote><a href=\"https://matrix.to/#/!hwiGbsdSTZIwSRfybq:matrix.org/$1553513281991555ZdMuB:matrix.org\">In reply to</a> <a href=\"https://matrix.to/#/@afranke:matrix.org\">@afranke:matrix.org</a><br><a href=\"https://matrix.to/#/@gergely:polonkai.eu\">Gergely Polonkai</a>: we have https://gitlab.gnome.org/GNOME/fractal/issues/467 and https://gitlab.gnome.org/GNOME/fractal/issues/347 open, does your issue fit any of these two?</blockquote></mx-reply>#467 <em>might</em> be it, let me test a bit";

        let blocks = markup_html(text);
        assert!(blocks.is_ok());
        let blocks = blocks.unwrap();
        assert_eq!(blocks.len(), 1);
    }
}
