#!/bin/env ruby
# 
#----------------------------------------------------------------------
# 
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2 of the License, or (at your option)
# any later version. This program is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
# Public License for more details; it is available at
# <http://www.fsf.org/copyleft/gpl.html>, or by writing to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
# 
# Written by Gordon Miller <gmiller@bittwiddlers.com>. Inspired by and partly
# derived from the Python version by Michael Haggerty. If you find a problem
# or have a suggestion, please let me know at <gmiller@bittwiddlers.com>.
# Other feedback would also be appreciated.
#
#----------------------------------------------------------------------
#
# This file contains the implementations of the PlotItem objects. These are
# objects that specify the data that is to be plotted and its format. This
# file should not be included directly by an application, the Gnuplot.rb file
# should be included instead.
# 
#----------------------------------------------------------------------

class PlotItem
  attr_reader :basecommand, :options

  # order in which options need to be passed to gnuplot:
  @@optionSequence = ['binary', 'using', 'title', 'with']

  #
  # Construct a 'PlotItem' with the specified basecommand.  The basecommand
  # is the string that tells gnuplot what it's plotting.  For example, in
  # the command 'plot sin(x)' the basecommand is 'sin(x)'.  Probably not a
  # very intuitive name, but it works for now.
  # 
  # Keyword options:
  # 
  # 'with=<string>' -- choose how item will be plotted, e.g.,
  #                    with='points 3 3'.
  #
  # 'title=<string>' -- set the title to be associated with the item
  #                     in the plot legend.
  #
  # 'title=None' --     choose 'notitle' option (omit item from legend).
  # 
  # Note that omitting the title option is different than setting
  # `title=None'; the former chooses gnuplot's default whereas the
  # latter chooses `notitle'.
  # 
  
  def initialize (basecommand, opts)
    @basecommand = basecommand
    @options = opts
  end

  # Return the setting of an option.
  def getOption(name)
    return @options[name]
  end

  # Set an option's value
  def setOption(name, val)
    @options[ name ] = val
  end

  # Clear (unset) a plot option.  No error if option was not set.
  def clearOption(name)
    @options.delete (name)
  end

  # Build the 'plot' command to be sent to gnuplot.
  # 
  # Build and return the 'plot' command, with options, necessary
  # to display this item.
  # 
  def command
    cmd = @basecommand

    titleStr = @options["title"]
    cmd += case titleStr
	   when "notitle" then "notitle"
	   when nil then ""
	   else sprintf (" title '%s'", titleStr)
    end

    if ( @options["with"] != nil ) 
      cmd += sprintf (" with %s", @options["with"])
    end

    return cmd
  end

  # Pipe necessary inline data to gnuplot.
  # 
  # If the plot command requires data to be put on stdin (i.e.,
  # 'plot "-"'), this method should put that data there.  Can be
  # overridden in derived classes.
  # 
  def pipein(f)
  end
end

class DataSet < PlotItem
  # Represents a ruby data set to plot.

  def initialize (data, opts={})
    @data = data

    # Look for the xgrid and ygrid members of opts.  If they exist, set the
    # @xgrid and @ygrid members respectively and remove them from opts.
    if ( opts.has_key? "xgrid" )
      @xgrid = opts.delete ( "xgrid" )
    end
    
    if ( opts.has_key? "ygrid" )
      @ygrid = opts.delete ( "ygrid" )
    end
    
    super ("'-'", opts)
  end

  def pipein(f)
    writeData(f)
    f.write ("e\n")
  end

  def setGrid (xgrid, ygrid=nil)
    @xgrid = xgrid
    @ygrid = ygrid
  end

  def writeData (f)
    if ( @data.dim == 1 )
      @data.length.times { |i|
	x = if @xgrid then @xgrid[i].to_s else i.to_s end
	f.insert ( x + " " + @data[i].to_s )
      }
    elsif ( @data.dim == 2 )
      @data.shape[0].times { |i|
	x = if @xgrid then @xgrid[i] else i.to_f end
	@data.shape[1].times { |j|
	  y = if @ygrid then @ygrid[j] else j.to_f end
	  f.insert ( sprintf ("%f %f %s", x, y, @data[i,j].to_s) )
	}
	f.write("\n")
      }
    end
  end

end

class Func < PlotItem
  # Represents a mathematical expression to plot.
  # 
  # Func represents a mathematical expression that is to be computed by
  # gnuplot itself, as if you would type for example
  # 
  # gnuplot> plot sin(x)
  # 
  # into gnuplot itself.  The argument to the contructor is a string
  # that should be a mathematical expression.  Example:
  # 
  # g.plot(Func('sin(x)', with='line 3'))
  # 
  # or a shorthand example:
  # 
  # g.plot('sin(x)')
  
  def initialize (str, opts = {})
    # No need to do anything here, as the base class takes care of doing
    # everything for us.
    super (str, opts)
  end
end

# $Id: PlotItem.rb,v 1.3 2001/01/25 06:49:08 gmiller Exp $
