/*
*  Rubrica
*  file: rubrica.c
*  
*  Copyright (C) 2000-2003 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <time.h>
#include <gnome.h>
#include <gconf/gconf-client.h>

#include "rubrica.h"
#include "autosave.h"
#include "log.h"
#include "view.h"
#include "utils.h"
#include "prefer.h"
#include "main.h"
#include "file.h"
#include "dictionary.h"
#include "dialog.h"
#include "card.h"
#include "sort.h"

Rubrica *current_rubrica = NULL;
RubricaItem *buffer = NULL;
RubricaItem *rtemp  = NULL;

gboolean have_buffer = FALSE;
gboolean just_start = TRUE;


Rubrica *rubrica_new(void)
{
  Rubrica *rubrica = NULL;

  rubrica = (Rubrica *) g_malloc(sizeof(Rubrica));

  if (!rubrica)
    g_error(_("Out of memory! I can't allocate enough memory for rubrica"));
  
  rubrica->new      = TRUE;
  rubrica->modified = FALSE;
  rubrica->empty    = TRUE;

  rubrica->parent   = NULL;
  rubrica->view     = NULL;

  rubrica->name     = NULL;
  rubrica->path     = NULL;
  rubrica->tmpfile  = NULL;

  rubrica->items     = 0;
  rubrica->index     = -1;
  rubrica->list_path = NULL;
  rubrica->page      = -1;  

  return (Rubrica*) rubrica;
}


/* dispose memory for new card/address 
 */
RubricaItem* rubrica_item_new(void)
{
  RubricaItem *r;

  r = (RubricaItem*) g_malloc(sizeof(RubricaItem));

  if (r)
    rubrica_init_item(r);
  else
    {
      rubrica_dialog_error(_("Out of memory! I can't allocate enough " 
			     "memory for rubrica"));

      return NULL;
    }
  
  return (RubricaItem*) r;
}


void rubrica_init_item(RubricaItem *item)
{
  item->delete = TRUE;
  item->group.label = NULL;
  item->group.type = NO_GROUP;
  item->card  = NULL;

  item->personal.first = item->personal.middle = item->personal.last = 
    item->personal.profession = item->personal.prefix = 
    item->personal.title = NULL;

  item->personal.birthknow = FALSE;
  item->personal.birthdate = time(NULL);
      
  item->address.street = item->address.number = item->address.zip = 
    item->address.city = item->address.province = item->address.state = 
    item->address.country = NULL;

  item->net.web   = NULL;
  item->net.email = NULL;
  item->phone.telephone = NULL;

  item->company.name = item->company.street = item->company.number 
    = item->company.zip = item->company.city  = item->company.province
    = item->company.country = item->company.web = item->company.email
    = item->company.operator = item->company.fax = item->company.green 
    = item->company.customer_care = item->company.notes = NULL;
  
  item->work.assigment =  item->work.organization = 
    item->work.department = item->work.subdep = 
    item->work.secretary = item->work.telephone = NULL;
  
  item->notes.is_married = item->notes.has_children = item->notes.birthknow = 
    item->notes.anniverknow = FALSE;
  item->notes.birthdate = item->notes.anniverdate = time(NULL);
  item->notes.spouse = item->notes.children = item->notes.hobbies = 
    item->notes.notes = item->notes.pubkey = NULL;
}



void rubrica_init(void)
{
  g_print("\nRubrica: begin rubrica initialization...");

  rubrica_utils_add_pixmap_directory(RUBRICA_PIXMAP_DIR);

  rubrica_init_gconf();
  rubrica_update_time();
  rubrica_sort_init();

  g_print("\nRubrica: rubrica initialization done");
}


void rubrica_init_dictionary(void)
{
  gchar *dicthome = NULL;
  gchar *home, *homedir;
  RubricaLanguageType language;
  
  g_print("\nRubrica: begin dictionary initialization...");

  home = g_strdup(g_get_home_dir());
  homedir  = g_strdup_printf("%s/.rubrica/dictionary", home);
  dicthome = rubrica_preferences_get_dictionary_home();
  
  if (!dicthome ||
      (g_strcasecmp(dicthome, homedir) != 0))
    {
      dicthome = g_strdup_printf("%s/.rubrica/dictionary", home);
      rubrica_preferences_set_dictionary_home(dicthome);
    }
  
  /*      test is user want to use a default dictionary 
   */
  if (user_wants_default_dictionary()
      /*rubrica_preferences_get_use_default_dictionary()*/)
    {
      if(user_wants_italian_dictionary()
	 /* rubrica_preferences_get_use_italian_dictionary()*/)
	language = ITALIAN;
      else
	language = ENGLISH;
    }
  else
    language = USER;
  
  /*      test if this is first rubrica's run 
   */
  if (rubrica_run_first_time())
    rubrica_dictionary_init(ITALIAN);
  else
    rubrica_dictionary_init(language);  
  g_print("\nRubrica: dictionary initialization done...");
}


void 
rubrica_init_gconf(void)
{
  GConfClient *client;

  g_print("\nRubrica: loading rubrica configuration");
  
  client = gconf_client_get_default();

  gconf_client_add_dir(client, "/apps/rubrica", 
		       GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);
  gconf_client_add_dir(client, "/apps/rubrica/file", 
		       GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);
  gconf_client_add_dir(client, "/apps/rubrica/general", 
		       GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);
  gconf_client_add_dir(client, "/apps/rubrica/show",   
		       GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);
  gconf_client_add_dir(client, "/apps/rubrica/show/column", 
		       GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);
  gconf_client_add_dir(client, "/apps/rubrica/show/personal", 
		       GCONF_CLIENT_PRELOAD_NONE, NULL);
  gconf_client_add_dir(client, "/apps/rubrica/show/communicate", 
		       GCONF_CLIENT_PRELOAD_NONE, NULL);
  gconf_client_add_dir(client, "/apps/rubrica/show/job",    
		       GCONF_CLIENT_PRELOAD_NONE, NULL);
  gconf_client_add_dir(client, "/apps/rubrica/show/notes",  
		       GCONF_CLIENT_PRELOAD_NONE, NULL);
  gconf_client_add_dir(client, "/apps/rubrica/translation", 
		       GCONF_CLIENT_PRELOAD_NONE, NULL);
  gconf_client_add_dir(client, "/apps/rubrica/dictionary",  
		       GCONF_CLIENT_PRELOAD_NONE, NULL);

  g_print("\nRubrica: rubrica configuration loaded");
}


gboolean 
rubrica_start_with_user_addressbook()
{
  return rubrica_preferences_get_have_default_addressbook();
}


gchar*
rubrica_get_book()
{
  gchar *file;
  
  file = g_strdup(rubrica_preferences_get_book());
  
  return (gchar *)(file ? file : NULL);
}

void 
rubrica_update_time()
{
  time_t last_run;

  last_run = time(NULL);

  rubrica_preferences_set_last_run(last_run);
}


void rubrica_set_just_start(gboolean bool)
{
  just_start = bool;
}

gboolean rubrica_get_just_start(void)
{
  return just_start;
}

void rubrica_set_current_rubrica(Rubrica* rubrica)
{
  current_rubrica = rubrica;
}

Rubrica* rubrica_get_current_rubrica(void)
{
  Rubrica *rubrica;
  gint page;

  if (current_rubrica)
    return (Rubrica*) current_rubrica;

  page = rubrica_card_get_current_notebook_page();
  if (-1 == page)
    {
      g_print("\nRubrica: No pages here");

      return NULL;
    }

  rubrica = rubrica_get_nth_rubrica(page);

  return  (Rubrica*) rubrica;
}


/*  Get notebook child (tree view parent)

    ritorna il figlio (la rubrica) associata alla 
    pagina (parent) di notebook
    -- metodo privato --
    -- private --
*/
Rubrica *rubrica_get_child(GtkWidget *parent)
{
  return (Rubrica *) g_object_get_data(G_OBJECT(parent), "rubrica");
}


/* 
 *  ritorna la rubrica associata alla pagina "page" del notebook
 *  return the nth rubrica (stored in nth notebook's page) 
 */
Rubrica*
rubrica_get_nth_rubrica(gint n)
{
  GtkWidget *rubrica_parent;
  Rubrica *rubrica = NULL;

  rubrica_parent = rubrica_card_get_nth_parent(n);
  if (!rubrica_parent)
    return NULL;

  rubrica = rubrica_get_child(rubrica_parent);
  if (!rubrica)
    return NULL;

  return (Rubrica *) rubrica;
}


/*  return the nth item in addressbook 
    ritorna l'enesimo elemento (indirizzo) della rubrica 
    passata come parametro, se rubrica e' NULL, allora ritorna 
    l'ennesimo elemento della rubrica attualmente attiva.
*/
RubricaItem*
rubrica_get_nth_item(Rubrica *rubrica, gint n)
{
  gint len = 0;

  if (n < 0)
    return NULL;

  if (rubrica)
    len = rubrica->items;

  if (n > len)
    return NULL;

  return (RubricaItem*) rubrica_card_get_nth_item(rubrica->view, n);
}


/**
 *
 * Metodi operanti sugli elementi della struttuta Rubrica 
 *
 * metodi publici  
 * 
 **/

/*  is New ?
 */
gboolean 
rubrica_is_new(Rubrica *rubrica)
{
  return (gboolean) rubrica->new;
}


/*  is Empty ?
*/         
gboolean 
rubrica_is_empty(Rubrica *rubrica)
{
  return (gboolean) rubrica->empty;
}


/*  is Modified ?
*/
gboolean 
rubrica_is_modified(Rubrica *rubrica)
{
  return (gboolean) rubrica->modified;
}


/*  Get rubrica's path
*/
gchar*
rubrica_get_path(Rubrica *rubrica)
{
  return (gchar *) rubrica->path;
}


/*  Get rubrica's file name
*/
gchar*
rubrica_get_name(Rubrica *rubrica)
{
  return (gchar *) rubrica->name;
}


gchar*
rubrica_get_tmpfile(Rubrica *rubrica)
{
  return (gchar *) rubrica->tmpfile;  
}


/*  Get rubrica's items
*/
gint 
rubrica_get_items(Rubrica *rubrica)
{
  return (gint) rubrica->items;
}

/*  Get the index of currently displayed item
*/
gint 
rubrica_get_index(Rubrica *rubrica)
{
  return (gint) rubrica->index;
}


gchar *rubrica_get_list_path(Rubrica *rubrica)
{
  return (gchar*) rubrica->list_path;
}



/*  Get the parent of the rubrica struct (the notebook child)
    ritorna il genitore (la pagina di notebook) associata
    alla rubrica passata
    -- metodo privato --
    -- private --
*/
GtkWidget*
rubrica_get_rubrica_parent(Rubrica *rubrica)
{
  return rubrica->parent;
}


/*  Get GtkTreeView
*/
GtkWidget*
rubrica_get_view(Rubrica *rubrica)
{
  return (GtkWidget *) rubrica->view; 
}



/*  ritorna il numero della pagina di notebook in cui 
    si trova la treeview che mostra la rubrica
*/
gint 
rubrica_get_current_page(void)
{
  return (gint) rubrica_card_get_current_notebook_page();
}




/****
 *
 *      Set
 *
 ****/

/*  setta:
    true se la lista e' nuova
    false altrimenti  
 */
void 
rubrica_set_new(Rubrica *rubrica, gboolean state)
{
  rubrica->new = (gboolean) state;
}


/*  setta a:
    true se la lista e' modificata
    false altrimenti  
 */
void 
rubrica_set_modified(Rubrica *rubrica, gboolean modify)
{
  GtkWidget *app;
  GtkWidget *tab_box;
  GtkWidget *notebook;
  GtkWidget *green, *red;
  gchar *title, *name;
  gint page;
  
  app  = rubrica_get_app();
  name = rubrica_get_name(rubrica);
  page = rubrica_get_current_page();
  notebook = rubrica_card_get_notebook();

  tab_box = gtk_notebook_get_tab_label(GTK_NOTEBOOK(notebook), 
				       rubrica_card_get_nth_parent(page));
  red   = g_object_get_data(G_OBJECT(tab_box), "red");
  green = g_object_get_data(G_OBJECT(tab_box), "green");

  if (modify) 
    {
      title = g_strdup_printf(_("Rubrica -- %s changed --"), name);
      gtk_window_set_title(GTK_WINDOW(app), title);
      gtk_widget_hide(green);
      gtk_widget_show(red);
      g_free(title);
    }
  else      
    {
      gtk_window_set_title(GTK_WINDOW(app), "Rubrica");
      gtk_widget_show(green);
      gtk_widget_hide(red);
    }
  
  rubrica->modified = modify;
}


/*  setta a:
    true se la lista e' vuota
    false altrimenti  
 */
void 
rubrica_set_empty(Rubrica *rubrica, gboolean state)
{
  rubrica->empty = state;
}


/*  imposta la path a str
*/
void 
rubrica_set_path(Rubrica *rubrica, gchar *str)
{
  if (rubrica->path)
    g_free(rubrica->path);

  rubrica->path = g_strdup(str);
}


/*  imposta a str il nuovo nome della rubrica
*/
void 
rubrica_set_name(Rubrica *rubrica, gchar *str)
{
  if (rubrica->name)
    g_free(rubrica->name);

  rubrica->name = g_strdup(str);
  rubrica_card_set_tab_name(rubrica->name);
}


/*  imposta il nome del file temporaneo
*/
void rubrica_set_tmpfile(Rubrica *rubrica, gchar *str)
{
  if (rubrica->tmpfile)
    g_free(rubrica->tmpfile);

  rubrica->tmpfile = g_strdup(str);
}


/*  setta il numero di elementi a n
 */
void 
rubrica_set_items(Rubrica *rubrica, gint n)
{
  rubrica->items = n;
}

/*  setta l'indice dell'elemento mostrato
 */
void 
rubrica_set_index(Rubrica *rubrica, gint index)
{
  rubrica->index = index;
}


void  
rubrica_set_list_path(Rubrica *rubrica, gchar *path)
{
  if (rubrica->list_path != NULL)
    g_free(rubrica->list_path);
  
  rubrica->list_path = g_strdup(path);
}




/*   append an item in addressbook
*/
void 
rubrica_append_item(Rubrica *rubrica, RubricaItem *r)
{
  rubrica->items++;

  rubrica_set_new(rubrica, FALSE);
  rubrica_set_modified(rubrica, TRUE);
  rubrica_set_empty(rubrica, FALSE);

  rubrica_view_set_items(rubrica->items);
  rubrica_card_append_item(rubrica, r);
}

/*   insert an item in addressbook
*/
void 
rubrica_insert_item(Rubrica *rubrica, RubricaItem *r, gint index)
{
  rubrica->items++;

  rubrica_set_new(rubrica, FALSE);
  rubrica_set_modified(rubrica, TRUE);
  rubrica_set_empty(rubrica, FALSE);

  rubrica_view_set_items(rubrica->items);
  rubrica_card_insert_item(rubrica, r, index);
}


/*    modify a rubrica's item
*/
void 
rubrica_modify_item(Rubrica *rubrica, RubricaItem *r, gint index)
{
  rubrica_card_modify_item(rubrica, r, index);

  rubrica_set_new(rubrica, FALSE);
  rubrica_set_modified(rubrica, TRUE);
  rubrica_set_empty(rubrica, FALSE);
}


/*      get the item that will be deleted
*/
gboolean 
rubrica_delete_selected_item(Rubrica *rubrica)
{
  if (rubrica_is_empty(rubrica))
    {
      rubrica_dialog_message(_("Address book is empty. I can't continue\n"),
			     GTK_MESSAGE_ERROR, GTK_BUTTONS_OK); 

      return FALSE;
    }

  if(rubrica_card_remove_selected_item(rubrica))
    {  
      //   decremento il numero totale di indirizzi    
      rubrica->items--;

      rubrica_view_set_items(rubrica->items);
      rubrica_set_new(rubrica, FALSE);
      rubrica_set_modified(rubrica, TRUE);

      if (rubrica_get_items(rubrica) == 0)
	rubrica_set_empty(rubrica, TRUE);
      else
	rubrica_set_empty(rubrica, FALSE); 
            
      return TRUE;
    }

  return FALSE;
}
 
gboolean rubrica_delete_last_item(Rubrica *rubrica)
{
  if (rubrica_is_empty(rubrica))
    return FALSE;

  if(rubrica_card_remove_last_item(rubrica))
    {  
      //   decremento il numero totale di indirizzi    
      rubrica->items--;

      rubrica_view_set_items(rubrica->items);
      rubrica_set_new(rubrica, FALSE);
      rubrica_set_modified(rubrica, TRUE);

      if (rubrica_get_items(rubrica) == 0)
	rubrica_set_empty(rubrica, TRUE);
      else
	rubrica_set_empty(rubrica, FALSE); 
            
      return TRUE;
    }

  return FALSE;
}

gboolean rubrica_delete_nth_item(Rubrica *rubrica, gint n)
{
  if (rubrica_is_empty(rubrica))
    return FALSE;

  if(rubrica_card_remove_nth_item(rubrica, n))
    {  
      //   decremento il numero totale di indirizzi    
      rubrica->items--;

      rubrica_view_set_items(rubrica->items);
      rubrica_set_new(rubrica, FALSE);
      rubrica_set_modified(rubrica, TRUE);

      if (rubrica_get_items(rubrica) == 0)
	rubrica_set_empty(rubrica, TRUE);
      else
	rubrica_set_empty(rubrica, FALSE); 
            
      return TRUE;
    }

  return FALSE;
}



void 
rubrica_free_memory(Rubrica *rubrica)
{
  GtkTreeModel *model;

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(rubrica_get_view(rubrica)));
  gtk_tree_model_foreach(GTK_TREE_MODEL(model), 
			 rubrica_card_free_memory, rubrica);  
}

void rubrica_free_string(gchar *str)
{
  if (str)
    g_free(str);
}

void rubrica_free_item_data(RubricaItem *item)
{
  GList *alias;

  rubrica_free_string(item->card);
  rubrica_free_string(item->group.label);
  rubrica_free_string(item->personal.first);
  rubrica_free_string(item->personal.middle);
  rubrica_free_string(item->personal.last);
  rubrica_free_string(item->personal.profession);
  rubrica_free_string(item->personal.prefix);
  rubrica_free_string(item->personal.title);

  rubrica_free_string(item->address.street);
  rubrica_free_string(item->address.number);
  rubrica_free_string(item->address.zip);
  rubrica_free_string(item->address.city);
  rubrica_free_string(item->address.province);
  rubrica_free_string(item->address.state);
  rubrica_free_string(item->address.country);

  alias = item->net.web;
  while (alias)
    {
      rubrica_free_string(alias->data);
      
      alias = g_list_next(alias);	
    }

  alias = item->net.email;
  while (alias)
    {
      rubrica_free_string(alias->data);
      
      alias = g_list_next(alias);	
    }  
  
  alias = item->phone.telephone;
  while (alias)
    {
      TelNum *tn;
      
      if (alias->data)
	{
	  tn = (TelNum*) alias->data;
	  
	  rubrica_free_string(tn->type);
	  rubrica_free_string(tn->number);      
	}

      alias = g_list_next(alias);	
    }    

  rubrica_free_string(item->company.name);
  rubrica_free_string(item->company.street);
  rubrica_free_string(item->company.number);
  rubrica_free_string(item->company.zip);
  rubrica_free_string(item->company.city);
  rubrica_free_string(item->company.province);
  rubrica_free_string(item->company.country);
  rubrica_free_string(item->company.web);
  rubrica_free_string(item->company.email);
  rubrica_free_string(item->company.operator);
  rubrica_free_string(item->company.fax);
  rubrica_free_string(item->company.green);
  rubrica_free_string(item->company.customer_care);
  rubrica_free_string(item->company.notes);
  
  rubrica_free_string(item->work.assigment);
  rubrica_free_string(item->work.organization);
  rubrica_free_string(item->work.department);
  rubrica_free_string(item->work.subdep);
  rubrica_free_string(item->work.secretary);
  rubrica_free_string(item->work.telephone);

  rubrica_free_string(item->notes.spouse);
  rubrica_free_string(item->notes.children);
  rubrica_free_string(item->notes.hobbies);
  rubrica_free_string(item->notes.notes);
  rubrica_free_string(item->notes.pubkey);
}


/*    cut, 
      copy and
      paste      
*/
void 
rubrica_item_copy(Rubrica *rubrica)
{
  if (rubrica_is_empty(rubrica))
    {
      rubrica_dialog_error(_("Address book is empty. I can't continue"));
      
      return;
    }

  buffer = rubrica_card_item_copy(rubrica);
}


void 
rubrica_item_cut(Rubrica *rubrica)
{
  if (rubrica_is_empty(rubrica))
    {
      rubrica_dialog_error(_("Address book is empty. I can't continue"));
      
      return;
    }
  
  buffer = rubrica_card_item_cut(rubrica);
  if (buffer)
    {
      rubrica->items--;

      rubrica_set_new(rubrica, FALSE);
      rubrica_set_modified(rubrica, TRUE);
      rubrica_set_empty(rubrica, FALSE);

      rubrica_view_set_items(rubrica->items);    
    }
}


void 
rubrica_item_paste(Rubrica *rubrica)
{
  gint index;

  if (!buffer)
    {
      rubrica_dialog_message(_("I can't paste anything if you "
			       "haven't previously cut or copied"), 
			     GTK_MESSAGE_ERROR, GTK_BUTTONS_OK);

      return;
    }

  index = rubrica_get_index(rubrica);
  if (index  == -1)
    rubrica_append_item(rubrica, buffer);
  else
    rubrica_insert_item(rubrica, buffer, index);

  buffer = NULL;
}


/*  utility functions
*/
gboolean rubrica_others_are_modified()
{
  Rubrica *rubrica;
  gint cards, n;
  gboolean bool = FALSE;

  cards = rubrica_card_get_pages();

  for (n = 0; n < cards; n++)
    {
      rubrica = rubrica_get_nth_rubrica(n);

      bool |=  rubrica_is_modified(rubrica);   
    }

  return bool;  
}


void 
rubrica_close_rubrica(Rubrica *rubrica)
{    
  GtkWidget *dialog;
  gint scelta;
  gchar *mes;
  gint page;

  if (rubrica_is_modified(rubrica))
    {
      mes = g_strdup_printf(_("Warning. %s\n"
			      "Changes not saved will be lost.\n\n"
			      "Do you want continue?"), 
			    _("Addressbook has changed."));
      
      dialog = rubrica_dialog_query(mes, GTK_BUTTONS_YES_NO);
      gtk_widget_show(dialog);
      g_free(mes);

      scelta = gtk_dialog_run(GTK_DIALOG(dialog));
      
      switch(scelta)
	{
	case GTK_RESPONSE_YES:
	  gtk_widget_destroy(dialog);
	  /* delete addressbook, go down an delete */
	  break;
	  
	case GTK_RESPONSE_NO:
	default:
	  gtk_widget_destroy(dialog);
	  return;
	  break;
	}  
    }
  
  /*      clean view window 
   */
  rubrica_view_clean_view();  
  page = rubrica_get_current_page();
  rubrica_free_memory(rubrica);
  rubrica_card_remove_page(page);
  
  if (rubrica_card_get_pages() == 0)
    {
      gint page;

      page = rubrica_card_add_page_view(NULL);
    }
}
