/*
*  Rubrica
*  file: preview.c
*  
*  Copyright (C) 2000-2003 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <gnome.h>
#include <libxml/parser.h>
#include <libxml/parserInternals.h>
#include <gconf/gconf-client.h>

#include "rubrica_type.h"
#include "preview.h"
#include "dictionary.h"
#include "dialog.h"
#include "rubrica.h"
#include "prefer.h"
#include "main.h" 
#include "file.h"
#include "utils.h"
#include "ui.h"
#include "utils.h"

enum {
  PREVIEW_DATA_COLUMN,
  PREVIEW_COLUMNS
};

GtkWidget *previewFs;
GtkWidget *view;            /* tree view */
GtkWidget *rads;            /* rads -> Read ADdresseS */
gchar *filename;

gchar *doc_version;


GtkWidget*
rubrica_preview_create_preview(void)
{
  GtkTreeModel *model;

  GtkWidget *frame;
  GtkWidget *scrollwin;
  GtkWidget *viewport;

  GtkWidget *main_box;
  GtkWidget *vbox;
  GtkWidget *hbox;

  GtkWidget *label;

  GtkWidget *button;
  GtkWidget *ok_button;
  GtkWidget *cancel_button;

  GtkTooltips *tooltip;


  /* File selection */
  previewFs = gtk_file_selection_new (_("Rubrica Preview file selection"));
  g_object_set_data (G_OBJECT (previewFs), "previewFs", previewFs);
  gtk_container_set_border_width (GTK_CONTAINER (previewFs), 10);
  gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(previewFs));

  main_box = GTK_FILE_SELECTION (previewFs)->main_vbox;

  ok_button = GTK_FILE_SELECTION (previewFs)->ok_button;
  gtk_widget_show (ok_button);

  cancel_button = GTK_FILE_SELECTION (previewFs)->cancel_button;
  g_object_set_data(G_OBJECT(previewFs), "cancel_button", cancel_button);
  gtk_widget_show (cancel_button);

  tooltip = gtk_tooltips_new();
  rubrica_utils_add_tooltip(tooltip);
  gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltip), ok_button, 
		       _("Open the previewed file in rubrica"), NULL);

  frame = rubrica_ui_frame_new(_("Preview"));
  gtk_widget_set_usize(frame, 100, 150);
  gtk_container_set_border_width (GTK_CONTAINER (frame), 2);
  gtk_box_pack_start(GTK_BOX(main_box), frame, TRUE, TRUE, 2);
  gtk_widget_show (frame);

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add(GTK_CONTAINER (frame), vbox);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 2);
  gtk_widget_show (vbox);

  scrollwin = gtk_scrolled_window_new (NULL, NULL);
  gtk_box_pack_start (GTK_BOX (vbox), scrollwin, TRUE, TRUE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (scrollwin), 2);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrollwin), 
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_widget_show (scrollwin);

  viewport = gtk_viewport_new (NULL, NULL);
  gtk_container_add (GTK_CONTAINER (scrollwin), viewport);
  gtk_widget_show (viewport);

  /* Model/View */
  model = rubrica_preview_create_model();
  view = rubrica_preview_new_view_with_model(model);
  gtk_container_add(GTK_CONTAINER(viewport), view);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 2);
  gtk_widget_show(hbox);

  label = gtk_label_new(_("addresses in this addressbook:"));
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  rads = gtk_label_new("");
  gtk_box_pack_start(GTK_BOX(hbox), rads, FALSE, FALSE, 0);
  gtk_widget_show(rads);  

  button = gtk_button_new_with_label (_("Preview"));
  g_object_set_data(G_OBJECT (previewFs), "preview_open", button);
  gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, FALSE, 2);
  gtk_widget_show (button);

  g_signal_connect_swapped (G_OBJECT (previewFs), "delete_event",
			    G_CALLBACK(gtk_widget_destroy), previewFs);

  g_signal_connect (G_OBJECT (ok_button), "clicked",
		    G_CALLBACK(rubrica_preview_load_previewed), NULL);

  g_signal_connect_swapped (G_OBJECT (cancel_button), "clicked",
			    G_CALLBACK(gtk_widget_destroy), previewFs);
  
  g_signal_connect (G_OBJECT (button), "clicked",
		    G_CALLBACK(rubrica_preview_show_file), previewFs);
  
  return previewFs;
}

      
GtkTreeModel *rubrica_preview_create_model(void)
{
  GtkTreeStore *model;

  model = gtk_tree_store_new(PREVIEW_COLUMNS, 
			     G_TYPE_STRING);
  
  return GTK_TREE_MODEL(model); 
}


GtkWidget* rubrica_preview_new_view_with_model(GtkTreeModel *model)
{
  GtkWidget *treeview;
  GtkCellRenderer *cell;
  
  treeview  = gtk_tree_view_new_with_model(model);
  gtk_widget_show(treeview);

  g_object_unref(G_OBJECT(model));

  gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(treeview), FALSE);
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(treeview), FALSE);
  gtk_tree_view_columns_autosize(GTK_TREE_VIEW(treeview));

  /* tree columns 
   */
  cell = gtk_cell_renderer_text_new();
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(treeview), -1, 
					      NULL, cell, "text", 
					      PREVIEW_DATA_COLUMN, NULL); 

  return treeview;
}


/*    Open file, after shown it in preview 
      carica il file attualmente in preview in rubrica
*/
void
rubrica_preview_load_previewed(GtkButton *button, gpointer user_data)
{
  gboolean loaded;
  RubricaErrorType err;
  
  loaded =  rubrica_file_load_xmlfile(filename, &err);
  if (!loaded)
    rubrica_dialog_error_loading_file(filename, err);

  gtk_widget_destroy(previewFs);
}


/*  mostra in preview il file selezionato
*/
void
rubrica_preview_show_file(GtkButton *button, gpointer user_data)
{  
  GtkTreeStore *model;
  GtkWidget *fileDialog = user_data;
  xmlDocPtr doc;
  RubricaErrorType err;
  
  if (fileDialog != NULL) 
    {
      if (filename)
	g_free(filename);

      filename = g_strdup(gtk_file_selection_get_filename
			  (GTK_FILE_SELECTION(fileDialog))); 
      
      if (!filename)
	{
	  rubrica_dialog_error(_("You must select a rubrica's file"));

	  return;
	} 
      
      if ((doc = rubrica_file_open_doc(filename)) == NULL)
	return;

      if (!rubrica_file_is_rubrica_doc(doc, filename))
	return;

      model = GTK_TREE_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(view)));
      gtk_tree_store_clear(GTK_TREE_STORE(model));

      if (!rubrica_preview_display_file(doc, GTK_TREE_STORE(model), &err)) 
	{
//	TODO:  rubrica_dialog_file_error(print error message);
	}
    }  
}



/* show the parsed xml file in a tree 
*/
gboolean rubrica_preview_display_file(xmlDocPtr doc, GtkTreeStore *model,
				      RubricaErrorType *err) 
{ 
  GtkTreeIter top_iter, father, child;
  xmlNodePtr node;
  GConfClient *client;
  gint count = 0;
  gchar *str;

  client = gconf_client_get_default(); 

  /*    load dictionary
  */
  if (rubrica_preferences_get_use_default_dictionary())
    {
      if (rubrica_preferences_get_use_italian_dictionary())
	rubrica_dictionary_get_italian();
      else
	rubrica_dictionary_get_english();
    }
  else
    rubrica_dictionary_get_user_dictionary();

  gtk_tree_store_append(GTK_TREE_STORE(model), &top_iter, NULL);
  gtk_tree_store_set(GTK_TREE_STORE(model), &top_iter, 
		     PREVIEW_DATA_COLUMN,  g_basename(filename),
		     -1);
  
  node = doc->children;
  if (xmlIsBlankNode(node))
    node = node->next;
    
  if (node && node->name && g_strcasecmp(node->name, "Rubrica") == 0) 
    {
      node = node->children;      
      if (xmlIsBlankNode(node))
	node = node->next;
      
      while(node)
	{
	  gchar *field, *data, *tmp;	  
	  xmlNodePtr cur;
	  
	  if (node && node->name && g_strcasecmp(node->name, "Card") == 0)
	    {
	      field = gconf_client_get_string(client, RUBRICA_KEY_DICT_CARD,
					      NULL);
	      data  = (tmp = xmlGetProp(node, "name")) ? tmp : "";
	      rubrica_preview_print_node(model, &father, &top_iter,
					 field, data);
	      
	      if (tmp) g_free(tmp);
	      
	      rubrica_preview_load_card_node(model, client, node, &father);
	    }

	  cur = node->children;
	  if (xmlIsBlankNode(cur))
	    cur = cur->next;	    
	  if (cur && cur->name && g_strcasecmp(cur->name, "Data") == 0)
	    {
	      field = gconf_client_get_string(client, RUBRICA_KEY_DICT_DATA, 
					      NULL);
	      rubrica_preview_print_node(model, &child, &father, field, "");
	      rubrica_preview_load_data_node(model, client, cur, &child);
	    }
	  
	  cur = cur->next;
	  if (xmlIsBlankNode(cur))
	    cur = cur->next;	    
	  if (cur && cur->name && g_strcasecmp(cur->name, "Address") == 0)
	    {	      
	      field = gconf_client_get_string(client, RUBRICA_KEY_DICT_ADDRESS,
					      NULL);
	      rubrica_preview_print_node(model, &child, &father, field, "");
	      rubrica_preview_load_address_node(model, client, cur, &child);
	    }

	  cur = cur->next;
	  if (xmlIsBlankNode(cur))
	    cur = cur->next;	  	  
	  if (cur && cur->name && g_strcasecmp(cur->name, "WebAddresses") == 0)
	    { 
	      field = gconf_client_get_string(client,
					      RUBRICA_KEY_DICT_WEB_ADDRESS,
					      NULL);
	      rubrica_preview_print_node(model, &child, &father, field, "");
	      rubrica_preview_load_web_node(model, client, cur, &child);
	    }

	  cur = cur->next;
	  if (xmlIsBlankNode(cur))
	    cur = cur->next;	  
	  if (cur && cur->name && 
	      g_strcasecmp(cur->name, "EmailAddresses") == 0)
	    {  	      
	      field = gconf_client_get_string(client,
					      RUBRICA_KEY_DICT_EMAIL_ADDRESS,
					      NULL);
	      rubrica_preview_print_node(model, &child, &father, field, "");
	      rubrica_preview_load_email_node(model, client, cur, &child);
	    }

	  cur = cur->next;
	  if (xmlIsBlankNode(cur))
	    cur = cur->next;	  
	  if (cur && cur->name && 
	      g_strcasecmp(cur->name, "TelephoneNumbers") == 0)
	    {
	      field = gconf_client_get_string(client,
					      RUBRICA_KEY_DICT_TELEPHONE_LIST,
					      NULL);
	      rubrica_preview_print_node(model, &child, &father, field, "");
	      rubrica_preview_load_tel_node(model, client, cur, &child);
	    }
	      
	  cur = cur->next;
	  if (xmlIsBlankNode(cur))
	    cur = cur->next;	  
	  if (cur && cur->name && g_strcasecmp(cur->name, "Company") == 0)
	    {   
	      field = gconf_client_get_string(client, RUBRICA_KEY_DICT_COMPANY,
					      NULL);
	      rubrica_preview_print_node(model, &child, &father, field, "");
	      rubrica_preview_load_company_node(model, client, cur, &child);
	    }

	  cur = cur->next;
	  if (xmlIsBlankNode(cur))
	    cur = cur->next;	  
	  if (cur && cur->name && g_strcasecmp(cur->name, "Work") == 0)
	    {
	      field = gconf_client_get_string(client, RUBRICA_KEY_DICT_WORK, 
					      NULL);
	      rubrica_preview_print_node(model, &child, &father, field, "");
	      rubrica_preview_load_work_node(model, client, cur, &child);
	    }

	  cur = cur->next;
	  if (xmlIsBlankNode(cur))
	    cur = cur->next;	  
	  if (cur && cur->name && g_strcasecmp(cur->name, "Notes") == 0)
	    {
	      field = gconf_client_get_string(client, RUBRICA_KEY_DICT_NOTES, 
					      NULL);
	      rubrica_preview_print_node(model, &child, &father, field, "");
	      rubrica_preview_load_note_node(model, client, cur, &child);
	    }

	  count++;  
	  node = node->next;      
	  if (xmlIsBlankNode(node))
	    node = node->next;
	} 
    }  
  str = g_strdup_printf("%d", count);
  gtk_label_set_text(GTK_LABEL(rads), str);
  g_free(str);

  return FALSE; /* no errors */
}



void  rubrica_preview_load_card_node(GtkTreeStore *model, GConfClient *client,
				     xmlNodePtr node, GtkTreeIter *parent)
{
  GtkTreeIter child;
  gchar *tmp, *field, *data;

  field = gconf_client_get_string(client, 
				  RUBRICA_KEY_DICT_CARD_GROUP, NULL);
  data = (tmp = xmlGetProp(node, "group")) ? _(tmp) : "";
  rubrica_preview_print_node(model, &child, parent, field, data);

  if (tmp) 
    g_free(tmp);
}


void  rubrica_preview_load_data_node(GtkTreeStore *model, GConfClient *client,
				     xmlNodePtr node, GtkTreeIter *parent)
{
  GtkTreeIter child;
  gchar *tmp, *field, *data;
  gchar *bday;

  node = node->children;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      

  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_FIRST_NAME,  NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  
  if ((bday = rubrica_preview_get_date(node, "birthday")) != NULL)
    {
      gchar *tmp;
      
      tmp = g_strdup_printf("%s: %s\t%s", field, data, _("birthday")); 
      
      rubrica_preview_print_node(model, &child, parent, tmp, bday);      
      g_free(tmp);
    }
  else
    rubrica_preview_print_node(model, &child, parent, field, data);
  
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_MIDDLE_NAME, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
  
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_LAST_NAME, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
  
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_PROFESSION, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
      
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_NAME_PREFIX, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
      
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_TITLE, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
}  



void rubrica_preview_load_address_node(GtkTreeStore *model, 
				       GConfClient *client,
				       xmlNodePtr node, GtkTreeIter *parent)
{
  GtkTreeIter child;
  gchar *tmp, *field, *data;

  node = node->children;      
  if (xmlIsBlankNode(node))
    node = node->next;  	      

  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_STREET, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
  
  node = node->next; 	      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_STREET_NUMBER,NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
  
  node = node->next; 	      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_ZIP_CODE, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
  
  node = node->next; 	      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_CITY, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
  
  node = node->next; 	      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_PROVINCE, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);

  node = node->next; 	      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_STATE, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
  
  node = node->next; 	      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_COUNTRY, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);  
}


 
void  rubrica_preview_load_web_node(GtkTreeStore *model, GConfClient *client,
				    xmlNodePtr node, GtkTreeIter *parent)
{      
  GtkTreeIter child;
  gchar *field, *data;

  node = node->children;      
  if (xmlIsBlankNode(node))
    node = node->next; 	
      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_WEB, NULL);
  while (node)
    {		  
      data = xmlNodeGetContent(node);
      if (data)
	{
	  rubrica_preview_print_node(model, &child, parent, field, data);
	  node = node->next;      
	  if (xmlIsBlankNode(node))
	    node = node->next;    
	}
    }
}

void  rubrica_preview_load_email_node(GtkTreeStore *model, 
				      GConfClient *client,
				      xmlNodePtr node, GtkTreeIter *parent)
{
  GtkTreeIter child;
  gchar *field, *data;
 
  node = node->children;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      

  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_EMAIL, NULL);
  while (node)
    {		  
      data = xmlNodeGetContent(node);
      if (data)
	{
	  rubrica_preview_print_node(model, &child, parent, field, data);
	  node = node->next;      
	  if (xmlIsBlankNode(node))
	    node = node->next;    
	}
    }
}




void  rubrica_preview_load_tel_node(GtkTreeStore *model, GConfClient *client,
				    xmlNodePtr node, GtkTreeIter *parent)  
{
  GtkTreeIter child;
  gchar *field, *data;

  node = node->children;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      

  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_TELEPHONE_NUMBER, 
				  NULL);
  while (node)
    {		  
      gchar *tmp1;
      
      tmp1 = xmlGetProp(node, "type");			    
      data = xmlNodeGetContent(node);
      if (data)
	{
	  gchar *tmp;
	  
	  tmp = g_strdup_printf("%s (%s)", field, _(tmp1));
	  rubrica_preview_print_node(model, &child, parent, tmp, data);
	  g_free(tmp);
	  node = node->next;      
	  if (xmlIsBlankNode(node))
	    node = node->next;    
	}
    }
}
  
 
void  rubrica_preview_load_company_node(GtkTreeStore *model,
					GConfClient *client, xmlNodePtr node,
					GtkTreeIter *parent)
{
  GtkTreeIter child;
  gchar *field, *data, *tmp;

  node = node->children;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      

  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_COMPANY_NAME, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
      
  node = node->next; 	      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_STREET, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
      
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_STREET_NUMBER,NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
      
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_ZIP_CODE, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
            
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_CITY, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
    
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_PROVINCE, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
            
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_COUNTRY, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
            
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_WEB, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
      
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_EMAIL, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
      
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_OPERATOR, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
  
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_TELEPHONE_FAX,NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
      
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_GREEN, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
      
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_CUSTOMER_CARE,NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
  
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_NOTES, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
}


void  rubrica_preview_load_work_node(GtkTreeStore *model, GConfClient *client,
				     xmlNodePtr node, GtkTreeIter *parent)
{
  GtkTreeIter child;
  gchar *field, *data, *tmp;

  node = node->children;      
  if (xmlIsBlankNode(node))
    node = node->next; 	 
     
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_ASSIGMENT, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
  
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_ORGANIZATION, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
  
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_DEPARTMENT, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
  
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_SUB_DEPARTMENT,
				  NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
  
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_SECRETARY, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
  
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_SECRETARY_PHONE, 
				  NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
}


void  rubrica_preview_load_note_node(GtkTreeStore *model, GConfClient *client,
				     xmlNodePtr node, GtkTreeIter *parent)
{
  GtkTreeIter child;
  gboolean know_bday, anniversary;
  gchar *tmp, *field, *data;

  tmp = rubrica_utils_bool2str(xmlGetProp(node, "married"));
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_MARRIED, NULL);
  rubrica_preview_print_node(model, &child, parent, field, tmp);

  tmp = rubrica_utils_bool2str(xmlGetProp(node, "children")); 
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_CHILDREN, NULL);
  rubrica_preview_print_node(model, &child, parent, field, tmp);
  
  know_bday  = rubrica_utils_str2bool(xmlGetProp(node, "know_birthday"));
  anniversary = rubrica_utils_str2bool(xmlGetProp(node, "know_anniversary"));
  
  node = node->children;      
  if (xmlIsBlankNode(node))
    node = node->next; 	    
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_SPOUSE_NAME, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
    
  if (know_bday)
    {
      data = rubrica_utils_decode_date(xmlGetProp(node, "birthday"));
      field = gconf_client_get_string(client, RUBRICA_KEY_DICT_BIRTHDAY, NULL);

      rubrica_preview_print_node(model, &child, parent, field, data);
    }

  if (anniversary)
    {
      data = rubrica_utils_decode_date(xmlGetProp(node, "anniversary"));
      field = gconf_client_get_string(client, RUBRICA_KEY_DICT_ANNIVERSARY, 
				      NULL);
      rubrica_preview_print_node(model, &child, parent, field, data);
    }
				 
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	       
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_CHILDREN, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : ""; 
  rubrica_preview_print_node(model, &child, parent, field, data);
    
  node = node->next;      
  if (xmlIsBlankNode(node))
    node = node->next; 	      
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_HOBBY, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
    
  node = node->next;       
  if (xmlIsBlankNode(node))
    node = node->next; 	       
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_NOTES, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);

  node = node->next;       
  if (xmlIsBlankNode(node))
    node = node->next; 	       
  field = gconf_client_get_string(client, RUBRICA_KEY_DICT_PUBLIC_KEY, NULL);
  data = (tmp = xmlNodeGetContent(node)) ? tmp : "";
  rubrica_preview_print_node(model, &child, parent, field, data);
}
  


void rubrica_preview_print_node(GtkTreeStore *model, 
				GtkTreeIter *child, GtkTreeIter *parent,
				gchar *field, gchar* data)
{
  gchar *tmp;

  if (strcmp(data, "") != 0)
    tmp = g_strdup_printf("%s: %s", field, data);
  else
    tmp = g_strdup(field);

  gtk_tree_store_append(model, child, parent);
  gtk_tree_store_set(model, child, 
		     PREVIEW_DATA_COLUMN, tmp, -1);

  g_free(tmp);
}


gchar *rubrica_preview_get_date(xmlNodePtr node, gchar *key)
{
  return (gchar*) _(xmlGetProp(node, key));
}






