/*
*  program name: Rubrica <http://digilander.iol.it/nfragale/download.html>
*  file: export.c
*  
*  Copyright (C) 2000-2001 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <config.h>
#include <gnome.h>
#include <libxslt/xslt.h>
#include <libxslt/xsltInternals.h>
#include <libxslt/transform.h>
#include <libxslt/xsltutils.h>

#include "export.h"
#include "dialog.h"
#include "ui.h"

GtkWidget* export_entry;
GtkWidget* save_as_entry;

gchar* exporting_file = NULL;
gchar* exported_file = NULL;
gchar* stylesheet = NULL;

extern int xmlLoadExtDtdDefaultValue;

static RubricaOptionType export_options[] = {
  {"", N_("HTML file"),                  EXPORT_HTML},
  {"", N_("Text file"),                  EXPORT_TXT},
  {"", N_("GnomeCard addressbook"),      EXPORT_VCARD},
  {"", N_("Kde addressbook"),            EXPORT_KDE_VCARD},
  {"", N_("Csv (Mozilla) addressbook"),  EXPORT_MOZILLA_CSV},
  {"", N_("Csv (Netscape) addressbook"), EXPORT_NETSCAPE_CSV},
  {"", N_("VCard addressbook"),          EXPORT_VCARD},
  {"", N_("Pine addressbook"),           EXPORT_PINE},
  {NULL, NULL}
};


GtkWidget*
create_export_dialog (GtkWidget *parent)
{
  GtkWidget *export_dialog;
  GtkWidget *dialog_vbox;
  GtkWidget *frame;
  GtkWidget *frame2;
  GtkWidget *vbox;
  GtkWidget *save_vbox;
  GtkWidget *browse_hbox;
  GtkWidget *lbl;

  GtkWidget *pixbox;
  GtkWidget *pixmap;

  GtkWidget *option_box;
  GtkWidget *option_label;
  GtkWidget *option_menu;


  export_dialog = gtk_dialog_new();
  gtk_dialog_add_buttons(GTK_DIALOG(export_dialog), 			 
			 GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, 
			 GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,NULL);
  gtk_window_set_title(GTK_WINDOW(export_dialog), _("Rubrica export tool"));
  gtk_window_set_resizable (GTK_WINDOW (export_dialog), FALSE);
  
  gtk_dialog_set_has_separator (GTK_DIALOG(export_dialog), TRUE);
  gtk_window_set_position (GTK_WINDOW (export_dialog), GTK_WIN_POS_CENTER);
  gtk_window_set_policy (GTK_WINDOW (export_dialog), FALSE, FALSE, FALSE);

  dialog_vbox = GTK_DIALOG (export_dialog)->vbox;
  gtk_widget_show (dialog_vbox);

  frame = rubrica_ui_frame_new (_("Export"));
  gtk_box_pack_start (GTK_BOX (dialog_vbox), frame, TRUE, TRUE, 2);
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);

  vbox = gtk_vbox_new (FALSE, 5);
  gtk_widget_show (vbox);
  gtk_container_add (GTK_CONTAINER (frame), vbox);

  pixbox = rubrica_ui_box_new(HBOX, FALSE, 2, 2);
  gtk_box_pack_start(GTK_BOX(vbox), pixbox, FALSE, FALSE, 2);

  pixmap = rubrica_ui_image_new("export.png");
  gtk_box_pack_start(GTK_BOX(pixbox), pixmap, FALSE, FALSE, 2);

  option_box = rubrica_ui_box_new(HBOX, FALSE, 2, 2);
  gtk_box_pack_start(GTK_BOX(vbox), option_box, FALSE, FALSE, 2);

  option_label = gtk_label_new (_("Export as:"));
  gtk_widget_show(option_label);
  gtk_box_pack_start (GTK_BOX (option_box), option_label, FALSE, FALSE, 3);

  option_menu = rubrica_ui_option_menu_new(export_options);
  gtk_box_pack_start(GTK_BOX (option_box), option_menu, FALSE, FALSE, 3);
  gtk_option_menu_set_history(GTK_OPTION_MENU(option_menu), EXPORT_HTML); 
  stylesheet = g_strdup_printf("%s%sstylesheet%srub2html.xsl", 
			       RUBRICA_SCRIPT_DIR, G_DIR_SEPARATOR_S,
			       G_DIR_SEPARATOR_S);

  /*  Gnome FileSelector
   */
  frame2 = rubrica_ui_frame_new (_("Files"));
  gtk_box_pack_start (GTK_BOX (vbox), frame2, TRUE, TRUE, 5);
  gtk_container_set_border_width (GTK_CONTAINER (frame2), 5);

  save_vbox = gtk_vbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(frame2), save_vbox);
  gtk_widget_show(save_vbox);

  /* file to export */
  browse_hbox = gtk_hbox_new(FALSE,  5);
  gtk_box_pack_start(GTK_BOX(save_vbox), browse_hbox, TRUE, TRUE, 5);
  gtk_widget_show(browse_hbox);

  lbl = gtk_label_new(_("File to export:"));
  gtk_box_pack_start(GTK_BOX(browse_hbox), lbl, FALSE, FALSE, 5);
  gtk_widget_show(lbl);  

  export_entry = gnome_file_entry_new ("export_id", 
				       _("Select exporting file"));
  gtk_box_pack_start(GTK_BOX(browse_hbox), export_entry, TRUE, TRUE, 5);
  gtk_widget_show (export_entry);
  gtk_container_set_border_width (GTK_CONTAINER (export_entry), 5);
  gnome_file_entry_set_directory_entry(GNOME_FILE_ENTRY(export_entry), FALSE);
  gnome_file_entry_set_default_path(GNOME_FILE_ENTRY (export_entry), 
				    g_get_home_dir());
  gnome_file_entry_set_modal (GNOME_FILE_ENTRY (export_entry), TRUE);
  
  /* save as */
  browse_hbox = gtk_hbox_new(FALSE,  5);
  gtk_box_pack_start(GTK_BOX(save_vbox), browse_hbox, TRUE, TRUE, 5);
  gtk_widget_show(browse_hbox);

  lbl = gtk_label_new(_("Save exported file as:"));
  gtk_box_pack_start(GTK_BOX(browse_hbox), lbl, FALSE, FALSE, 5);
  gtk_widget_show(lbl);  

  save_as_entry = gnome_file_entry_new("exported_id",
				       _("Set exporting file name"));
  gtk_box_pack_start(GTK_BOX(browse_hbox), save_as_entry, TRUE, TRUE, 5);
  gtk_widget_show(save_as_entry);
  gtk_container_set_border_width (GTK_CONTAINER (save_as_entry), 5);
  gnome_file_entry_set_directory_entry(GNOME_FILE_ENTRY(save_as_entry), FALSE);
  gnome_file_entry_set_default_path(GNOME_FILE_ENTRY (save_as_entry), 
				    g_get_home_dir());
  gnome_file_entry_set_modal (GNOME_FILE_ENTRY (save_as_entry), TRUE);

  rubrica_export_init(option_menu, EXPORT_HTML);

  g_signal_connect(G_OBJECT(option_menu), "changed", 
		   G_CALLBACK(rubrica_export_option_menu_changed), NULL);

  return export_dialog;
}

void rubrica_export_init(GtkWidget *menu, RubricaExportType type)
{
  gtk_option_menu_set_history(GTK_OPTION_MENU(menu), type);
}



void 
rubrica_export_option_menu_changed(GtkOptionMenu *menu, gpointer data)
{
  gint opt;
  RubricaExportType type;

  opt = gtk_option_menu_get_history(GTK_OPTION_MENU(menu));

  if (opt == -1)
    return;

  if (stylesheet)
    {
      g_free(stylesheet);

      stylesheet = NULL;
    }

  type = (RubricaExportType) export_options[opt].type;

  switch (type)
    {
    case EXPORT_HTML:
      stylesheet = g_strdup_printf("%s%sstylesheet%srub2html.xsl", 
				   RUBRICA_SCRIPT_DIR, 
				   G_DIR_SEPARATOR_S,
				   G_DIR_SEPARATOR_S);
      break;

    case EXPORT_TXT:
      stylesheet = g_strdup_printf("%s%sstylesheet%srub2txt.xsl", 
				   RUBRICA_SCRIPT_DIR, 
				   G_DIR_SEPARATOR_S,
				   G_DIR_SEPARATOR_S);      
      break;

    case EXPORT_NETSCAPE_CSV:
      rubrica_dialog_will_do();

      return;
      break;

    case EXPORT_MOZILLA_CSV:
      rubrica_dialog_will_do();

      return;
      break;

    case EXPORT_KDE_VCARD:
      rubrica_dialog_will_do();

      return;
      break;

    case EXPORT_VCARD:
      stylesheet = g_strdup_printf("%s%sstylesheet%srub2vcard.xsl", 
				   RUBRICA_SCRIPT_DIR, 
				   G_DIR_SEPARATOR_S,
				   G_DIR_SEPARATOR_S);    
      break;

    case EXPORT_PINE:
      rubrica_dialog_will_do();

      return;
      break;

    default:
      break;
    }
  
  if (!g_file_exists(stylesheet))
    {
      rubrica_dialog_error(_("Can't find the stylesheet.\n"
			     "I can't export to this file format."));

      gtk_option_menu_set_history(GTK_OPTION_MENU(menu), EXPORT_HTML);
      g_free(stylesheet);

      return;
    }  
}


gchar *rubrica_export_get_exporting_file(void)
{
  GtkWidget *entry;
  G_CONST_RETURN gchar *fname;

  entry = gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (export_entry));
  fname = gtk_entry_get_text(GTK_ENTRY(entry));
  
  if (fname)
    {
      gchar *tmp;
      gsize read, written;
      GError *error;

      tmp = g_locale_from_utf8(fname, -1, &read, &written, &error);

      if (tmp)
	return (gchar *) tmp;
      else
	{
	  rubrica_dialog_error(_("Error during utf8 to locale "
				 "filename conversion"));
	  
	  return NULL;
	}
    }

  return NULL;
}

gchar *rubrica_export_get_exported_file(void)
{
  GtkWidget *entry;
  G_CONST_RETURN gchar *fname;

  entry = gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (save_as_entry));
  fname = gtk_entry_get_text(GTK_ENTRY(entry));
  
  if (fname)
    {
      gchar *tmp;
      gsize read, written;
      GError *error;

      tmp = g_locale_from_utf8(fname, -1, &read, &written, &error);

      if (tmp)
	return (gchar *) tmp;
      else
	{
	  rubrica_dialog_error(_("Error during utf8 to locale "
				 "filename conversion"));

	  return NULL;
	}
    }

  return NULL;
}


gboolean
rubrica_export_export_data (void)
{
  gboolean export = FALSE;
  gchar *tmp;

  tmp = rubrica_export_get_exporting_file();
  if (!tmp)
    {
      rubrica_dialog_error(_("You must select the file that want to export")); 
     
      return FALSE;
    }
  if (exporting_file)
    g_free(exporting_file);

  exporting_file = g_strdup(tmp);
  if (!g_file_exists(exporting_file))
    {
      gchar *msg = g_strdup_printf("File %s does not exists", exporting_file);

      rubrica_dialog_error(_(msg));
      g_free(msg);

      return FALSE;
    }

  tmp = rubrica_export_get_exported_file();
  if (!tmp)
    {
      rubrica_dialog_error(_("I need a name for exported file")); 
   
      return FALSE;
    }
  if (exported_file)
    g_free(exported_file);
  
  exported_file = g_strdup(tmp);
  if (g_file_exists(exported_file))
    {
      GtkWidget *query;
      gint response;

      query = rubrica_dialog_query(_("A file with this name exists\n"
				     "Do you want overwite?"),
				   GTK_BUTTONS_YES_NO);
      gtk_widget_show(query);

      response = gtk_dialog_run(GTK_DIALOG(query));
      gtk_widget_destroy(query);
      
      switch(response)
	{
	case GTK_RESPONSE_YES:
	  break;
	  
	case GTK_RESPONSE_NO:
	default:
	  return FALSE;
	  break;
	}
    }
  
  export = rubrica_export_engine(exporting_file, exported_file, stylesheet);
    
  if (export)
    {
      rubrica_dialog_message(_("File has been exported"), 
			     GTK_MESSAGE_INFO, GTK_BUTTONS_OK);

      if (stylesheet)
	g_free(stylesheet);

      return TRUE;
    }

  return FALSE;
}


/*
 * rubrica_export_engine based on:
 *
 * libxslt_tutorial.c: demo program for the XSL Transformation 1.0 engine
 *
 * based on xsltproc.c, by Daniel.Veillard@imag.fr 
 * by John Fleck 
 *
 */ 
gboolean  rubrica_export_engine(gchar *original, gchar *exported, 
				gchar *stylesheet)
{
  FILE *fout;
  xsltStylesheetPtr sheet = NULL;
  xmlDocPtr doc, export_doc;
  const char *params[1];

  if ((fout = fopen(exported, "w")) == NULL)
    {
      rubrica_dialog_error(_("Exporting don't performed.\n"
			     "I can't open the output file"));

      return FALSE;
    }

  xmlSubstituteEntitiesDefault(1);
  xmlLoadExtDtdDefaultValue = 1;
  params[0] = NULL;
  sheet = xsltParseStylesheetFile((const xmlChar *) stylesheet);
  
  doc = xmlParseFile(original);
  export_doc = xsltApplyStylesheet(sheet, doc, params);

  if (-1 == xsltSaveResultToFile(fout, export_doc, sheet))
    {
      rubrica_dialog_error(_("Exporting don't performed.\n"
			     "An error occured during file writing"));

      return FALSE;
    }
  
  xsltFreeStylesheet(sheet);
  xmlFreeDoc(export_doc);
  xmlFreeDoc(doc);
  
  xsltCleanupGlobals();
  xmlCleanupParser();

  return TRUE;
}

