#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <rsbac/types.h>
#include <rsbac/getname.h>
#include <rsbac/syscalls.h>
#include <rsbac/error.h>
#include <rsbac/helpers.h>
#include <rsbac/aci_data_structures.h>
#include "nls.h"
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

const char   set_prog[] = "attr_set_file_dir";
enum rsbac_attribute_t attr_list[RSBAC_FD_NR_ATTRIBUTES] = RSBAC_FD_ATTR_LIST;
int recurse = 0;
int verbose = 0;
int mignore = 0;
int exrdat = 0;
char * filename = NULL;

__s64 def_attr[RSBAC_FD_NR_ATTRIBUTES] = {
      SL_inherit, /* sec_level */
      RSBAC_NO_USER, /* mac_trusted_for_user */
      RSBAC_MAC_INHERIT_CAT_VECTOR, /* mac_categories */
      MA_inherit, /* mac_auto */
      FALSE, /* mac_prop_trusted */
      0, /* mac_file_flags */
      RSBAC_FC_OC_DEF, /* object_category */
      RSBAC_SIM_DT_DEF, /* data_type */
      0, /* pm_object_class */
      0, /* pm_tp */
      PO_none, /* pm_object_type */
      MS_unscanned, /* ms_scanned */
      FALSE, /* ms_trusted */
      MS_not_trusted, /* ms_sock_trusted_tcp */
      MS_not_trusted, /* ms_sock_trusted_udp */
      DEFAULT_MS_FD_NEED_SCAN, /* ms_need_scan */
      RSBAC_FF_DEF, /* ff_flags */
      RC_type_inherit_parent, /* rc_type_fd */
      RC_default_force_role, /* rc_force_role */
      RC_default_initial_role, /* rc_initial_role */
      FALSE, /* auth_may_setuid */
      FALSE, /* auth_may_set_cap */
      (rsbac_request_vector_t) -1,    /* log_array_low */
      (rsbac_request_vector_t) -1,    /* log_array_high */
      0,     /* log_program_based */
      FALSE, /* symlink_add_uid */
      FALSE, /* symlink_add_mac_level */
      FALSE, /* symlink_add_rc_role */
      LDD_inherit, /* linux_dac_disable */
      0,     /* min_caps */
      (rsbac_cap_vector_t) -1 /* max_caps */
  };

int process(char * name, FILE * tfile)
  {
    int res = 0;
    char tmp1[RSBAC_MAXNAMELEN];
    char tmp2[RSBAC_MAXNAMELEN];
    int j,k;
    struct stat buf;
    union rsbac_attribute_value_t value;

    if(verbose)
      printf(gettext("Processing FD '%s'\n"), name);
    if(   exrdat
       && !strcmp(name, "rsbac.dat")
      )
      return;
    for (j=0;j < RSBAC_FD_NR_ATTRIBUTES;j++)
      {
        value.dummy = -1;
        res = rsbac_get_attr_n(get_attr_module(attr_list[j]), T_FD, name, attr_list[j], &value, 0);
        if(res)
          {
            if(   (res != -RSBAC_EINVALIDMODULE)
               && (   verbose
                   || (res != -RSBAC_EINVALIDTARGET)
                  )
              )
              {
                get_error_name(tmp1,res);
                fprintf(stderr, "%s (%s): %s\n",
                        name,
                        get_attribute_name(tmp2,attr_list[j]),
                        tmp1);
              }
          }
        else
          switch(attr_list[j])
            {
              case A_log_array_low:
              case A_log_array_high:
                if (value.log_array_low != def_attr[j])
                  fprintf(tfile,
                          "%s -V %u FD \"%s\" %s %s\n",
                          set_prog,
                          RSBAC_VERSION_NR,
                          name,
                          get_attribute_name(tmp1,attr_list[j]),
                          u64tostrlog(tmp2,value.log_array_low));
                break;
              case A_log_program_based:
                if (value.log_program_based != def_attr[j])
                  fprintf(tfile,
                          "%s -V %u FD \"%s\" %s %s\n",
                          set_prog,
                          RSBAC_VERSION_NR,
                          name,
                          get_attribute_name(tmp1,attr_list[j]),
                          u64tostrlog(tmp2,value.log_program_based));
                break;
              case A_mac_categories:
                if (value.mac_categories != def_attr[j])
                  fprintf(tfile,
                          "%s -V %u FD \"%s\" %s %s\n",
                          set_prog,
                          RSBAC_VERSION_NR,
                          name,
                          get_attribute_name(tmp1,attr_list[j]),
                          u64tostrmac(tmp2,value.mac_categories));
                break;
              case A_mac_trusted_for_user:
                if (value.mac_trusted_for_user != def_attr[j])
                  fprintf(tfile,
                          "%s -V %u FD \"%s\" %s %u\n",
                          set_prog,
                          RSBAC_VERSION_NR,
                          name,
                          get_attribute_name(tmp1,attr_list[j]),
                          value.mac_trusted_for_user);
                break;
              case A_ff_flags:
                if (value.ff_flags != def_attr[j])
                  fprintf(tfile,
                          "%s -V %u FD \"%s\" %s %u\n",
                          set_prog,
                          RSBAC_VERSION_NR,
                          name,
                          get_attribute_name(tmp1,attr_list[j]),
                          value.ff_flags);
                break;
              case A_rc_type_fd:
              case A_rc_force_role:
              case A_rc_initial_role:
                if (value.rc_type_fd != def_attr[j])
                  fprintf(tfile,
                          "%s -V %u FD \"%s\" %s %u\n",
                          set_prog,
                          RSBAC_VERSION_NR,
                          name,
                          get_attribute_name(tmp1,attr_list[j]),
                          value.rc_type_fd);
                break;
              case A_security_level:
              case A_object_category:
              case A_data_type:
              case A_pm_object_type:
                if (value.security_level != def_attr[j])
                  fprintf(tfile,
                          "%s -V %u FD \"%s\" %s %u\n",
                          set_prog,
                          RSBAC_VERSION_NR,
                          name,
                          get_attribute_name(tmp1,attr_list[j]),
                          value.security_level);
                break;
              case A_max_caps:
              case A_min_caps:
                if (value.max_caps != def_attr[j])
                  fprintf(tfile,
                          "%s -V %u FD \"%s\" %s %s\n",
                          set_prog,
                          RSBAC_VERSION_NR,
                          name,
                          get_attribute_name(tmp1,attr_list[j]),
                          u32tostrcap(tmp2,value.max_caps));
                break;
              case A_ms_scanned:
                if (   !mignore
                    && (value.ms_scanned != def_attr[j])
                   )
                  fprintf(tfile,
                          "%s -V %u FD \"%s\" %s %u\n",
                          set_prog,
                          RSBAC_VERSION_NR,
                          name,
                          get_attribute_name(tmp1,attr_list[j]),
                          value.ms_scanned);
                break;
              case A_mac_auto:
              case A_mac_prop_trusted:
              case A_mac_file_flags:
              case A_ms_trusted:
              case A_ms_sock_trusted_tcp:
              case A_ms_sock_trusted_udp:
              case A_ms_need_scan:
              case A_linux_dac_disable:
                if (value.u_char_dummy != def_attr[j])
                  fprintf(tfile,
                          "%s -V %u FD \"%s\" %s %u\n",
                          set_prog,
                          RSBAC_VERSION_NR,
                          name,
                          get_attribute_name(tmp1,attr_list[j]),
                          value.u_char_dummy);
                break;
              case A_res_min:
              case A_res_max:
                for(k=0; k <= RSBAC_RES_MAX; k++)
                  {
                    if (value.res_array[k])
                      fprintf(tfile,
                              "%s -V %u FD \"%s\" %s %s %u\n",
                              set_prog,
                              RSBAC_VERSION_NR,
                              name,
                              get_attribute_name(tmp1,attr_list[j]),
                              get_res_name(tmp2,k),
                              value.res_array[k]);
                  }
                break;
              default:
                if(value.dummy != def_attr[j])
                  fprintf(tfile,
                          "%s -V %u FD \"%s\" %s %i\n",
                          set_prog,
                          RSBAC_VERSION_NR,
                          name,
                          get_attribute_name(tmp1,attr_list[j]),
                          value.dummy);
            }
      }
    if(   recurse
       && !lstat(name,&buf)
       && S_ISDIR(buf.st_mode))
      {
        DIR * dir_stream_p;
        struct dirent * dirent_p;
        char name2[PATH_MAX];

        if(S_ISLNK(buf.st_mode))
          return(0);
        if(!(dir_stream_p = opendir(name)))
          {
            fprintf(stderr, gettext("opendir for dir %s returned error: %s\n"),
                   name,
                   strerror(errno));
            return(-2);
          }
        while((dirent_p = readdir(dir_stream_p)))
          {
            if(   (strcmp(".",dirent_p->d_name))
               && (strcmp("..",dirent_p->d_name)) )
              {
                strcpy(name2,name);
                strcat(name2,"/");
                strcat(name2,dirent_p->d_name);
                process(name2, tfile);
              }
          }
        closedir(dir_stream_p);
      }
    return(0);
  }

int main(int argc, char ** argv)
{
  int res = 0;
  char tmp1[RSBAC_MAXNAMELEN],tmp2[RSBAC_MAXNAMELEN],tmp3[RSBAC_MAXNAMELEN];
  int i,j;
  FILE * tfile;
  char * progname;

  locale_init();
  
  progname = argv[0];
  while((argc > 1) && (argv[1][0] == '-'))
    {
      char * pos = argv[1];
      pos++;
      while(*pos)
        {
          switch(*pos)
            {
              case 'v':
                verbose++;
                break;
              case 'r':
                recurse=1;
                break;
              case 'm':
                mignore=1;
                break;
              case 'x':
                exrdat=1;
                break;
              case 'i':
                def_attr[0] = SL_unclassified; /* sec_level */
                def_attr[2] = RSBAC_MAC_DEF_CAT_VECTOR; /* mac_categories */
                def_attr[3] = MA_inherit; /* mac_auto */
                break;
              case 'o':
                if(argc > 2)
                  {
                    filename = argv[2];
                    argv++;
                    argc--;
                  }
                else
                  fprintf(stderr, gettext("%s: missing filename for parameter o\n"), progname);
                break;
              case 'a':
                printf(gettext("attributes and values in backup = see following list:\n"));
                for (j=0;j<RSBAC_FD_NR_ATTRIBUTES;j++)
                  {
                    get_switch_target_name(tmp1, get_attr_module(attr_list[j]));
                    get_attribute_name(tmp2,attr_list[j]);
                    get_attribute_param(tmp3,attr_list[j]);
                    printf("[%-4s] %s\n\t%s\n",tmp1,tmp2,tmp3);
                  }
                exit(0);
              default:
                fprintf(stderr, gettext("%s: unknown parameter %c\n"), progname, *pos);
            }
          pos++;
        }
      argv++;
      argc--;
    }

  if (argc > 1)
    {
      if(!filename)
        tfile = stdout;
      else
        {
          if (!(tfile=fopen(filename,"w")))
            {
              fprintf(stderr, gettext("opening target file returned error: %s\n"),
                      strerror(errno));
            }
        }
      if(verbose)
        {
          printf(gettext("%s: %i targets"), progname, argc - 2);
          if(recurse)
            printf(gettext(" - recursing"));
          printf("\n");
        }
      for (i=1;i < (argc);i++)
        {
          process(argv[i],tfile);
        }
      if(tfile != stdout)
        fclose(tfile);
    }
  else
    {
      printf(gettext("%s (RSBAC %s)\n***\n"), argv[0], VERSION);
      printf(gettext("Use: %s [options] file/dirname(s)\n"), progname);  
      printf(gettext("- should be called by user with full attribute read access,\n  e.g. root with all modules off\n"));
      printf(gettext("- -r = recurse in subdirs, -v = verbose, no symlinks followed,\n"));
      printf(gettext("- -m = ignore ms_scanned,\n"));
      printf(gettext("- -i = use MAC non-inherit values as default values,\n"));
      printf(gettext("- -o target-file = write to file, not stdout,\n"));
      printf(gettext("- -a = list attributes and values\n"));
    }
  return (res);
}
