"""
This file builds rpy.

It has been adapted to allow multiple versions of the rpy shared
library to be included in the build.  To enable this, set the
environment variable "RHOMES" to a colon-delimited
(semicolon-delimited on MS-Windows) list of R installation directories.

For linux csh derivatives (including tcsh) use something like

  setenv RHOME  /usr/local/R-2.1.0/lib/R:/usr/local/R-2.1.1/lib/R
  python setup.py bdist 

For linux sh derivatives (including bash) use something like

  RHOME=/usr/local/R-2.1.0/lib/R:/usr/local/R-2.1.1/lib/R
  export RHOME
  python setup.py bdist 

For windows, use something like
  
  set RHOME = c:\progra~1\R\rw2010\;c:\progra~1\R\rw2011
  python setup.py bdist_wininst

The resulting module will have multiple _rpy modules so it can be
used with multiple R versions.

[Multiple R code contributed by Bo Peng <ben.bob at gmail.com>]
"""

import os, os.path, sys, shutil
from distutils.core import setup, Extension
from distutils.sysconfig import *
from distutils.errors import *
import rpy_tools


try:
    if sys.platform=="win32":
        RHOMES = os.environ['RHOMES'].split(';')
    else:
        RHOMES = os.environ['RHOMES'].split(':')
except:
    RHOMES = []
print "RHOMES=", RHOMES

if not RHOMES:
    RHOMES = [rpy_tools.get_R_HOME(force_exec=False)]

modules = []
for RHOME in RHOMES:
   
    RHOME = RHOME.strip()

    # to avoid strict prototypes errors from R includes
    get_config_vars()['OPT'] = '-DNDEBUG -g -O3 -Wall'

    # get the Python version
    if sys.version[:3] >= '2.2':
        DEFINE = []
        UNDEF = ['PRE_2_2']
    else:
        DEFINE = [('PRE_2_2', None)]
        UNDEF = []

    # configure the R paths
    RVERSION = rpy_tools.get_R_VERSION(RHOME, force_exec=True)
    RVER     = rpy_tools.get_R_VERSION_CODE(RVERSION)

    print "### Using R verion %s installed at %s ###" % (RVERSION, RHOME)

    if sys.platform=="win32":
        DEFINE.append(('R_HOME', '\\"%s\\"' % RHOME.replace("\\","/") ))
    else:
        DEFINE.append(('R_HOME', '"%s"' % RHOME))

    r_libs = [
              os.path.join(RHOME, 'bin'), 
              os.path.join(RHOME, 'lib'),  # R 2.0.0 moved libR.so 
             ]

    print r_libs

    base_source_files = ["src/rpymodule.c", "src/R_eval.c",
                         "src/io.c"]

    exists = os.path.exists
    def mtime(f):
        """File modification time"""
        os.stat(f).st_mtime
        
    # Make one copy of the source files for this R version
    source_files = []
    for f in base_source_files:
        # file.c => file2010.c
        nfile = f[0:-2] + RVER + '.c'
        if (not exists(nfile)) or  ( mtime(f) > mtime(nfile) ):
            shutil.copy(f, nfile)
        source_files.append(nfile)

    if sys.platform=='win32':
        include_dirs = [ os.path.join(RHOME.strip(), 'include'),
                         'src' ]
        libraries=["Rdll","Advapi32"] 
        library_dirs = r_libs
        runtime_libs = []
        extra_compile_args= ["-wd4068"]  # Disable unknown pragma warning in pyconfig.h
        source_files = source_files + ["src\\setenv.c"]
    elif sys.platform=='darwin':
        include_dirs = [ os.path.join(RHOME.strip(), 'include'), 
                         'src' ]
        libraries=['R','Rlapack']
        library_dirs= r_libs
        runtime_libs = r_libs
        extra_compile_args=[]
    elif sys.platform=='osf1V5':
        include_dirs = [ os.path.join(RHOME.strip(), 'include'), 
                         'src' ]
        libraries=['R','Rlapack']
        library_dirs = r_libs
        runtime_libs = r_libs
        extra_compile_args=["-shared"]
        source_files = source_files + ["src/setenv.c"]
    else: # unix-like systems, this is known to work for Linux and Solaris
        include_dirs = [ os.path.join(RHOME.strip(), 'include'), 
                         'src' ]
        libraries=['R','Rlapack']
        library_dirs = r_libs
        runtime_libs = r_libs
        extra_compile_args=["-shared"]
        source_files = source_files + ["src/setenv.c"]

    # check whether Numeric is present
    try:
        import Numeric
        DEFINE.append(('WITH_NUMERIC', None))
    except ImportError:
        UNDEF.append('WITH_NUMERIC')

    # get the RPy version
    from rpy_version import rpy_version

    LONG_DESC = """RPy provides a robust Python interface to the R
    programming language.  It can manage all kinds of R objects and can
    execute arbitrary R functions. All the errors from the R language are
    converted to Python exceptions."""

    # use R version specific shared library name
    shlib_name = "_rpy%s" % RVER
    if sys.platform=="win32":
        DEFINE.append( ('RPY_SHNAME', '\\"_rpy%s\\"' % RVER ) )
    else:
        DEFINE.append( ('RPY_SHNAME', '"_rpy%s"' % RVER ) )
    DEFINE.append( ('INIT_RPY', 'init_rpy%s' % RVER ) )


    modules.append( Extension(
        shlib_name,
        source_files,
        include_dirs=include_dirs,
        libraries=libraries,
        library_dirs=library_dirs,
        define_macros=DEFINE,
        undef_macros=UNDEF,
        extra_compile_args=extra_compile_args,
        runtime_library_dirs = runtime_libs,
        ) )


setup(name="rpy",
      version=rpy_version,
      description="Python interface to the R language",
      maintainer="Gregory R. Warnes",
      maintainer_email="gregory.r.warnes@pfizer.com",
      url="http://rpy.sourceforge.net",
      license="GPL",
      long_description=LONG_DESC,
      py_modules=['rpy', 'rpy_io', 'rpy_version', 'rpy_tools'],
      ext_modules=modules
      )
