/*
    roxterm - GTK+ 2.0 terminal emulator with tabs
    Copyright (C) 2004 Tony Houghton <h@realh.co.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include "defns.h"

#include <sys/types.h>
#include <ctype.h>
#include <errno.h>
#include <pwd.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <vte/vte.h>
#include <gdk/gdkx.h>

#include "about.h"
#include "colourscheme.h"
#include "dlg.h"
#include "dragrcv.h"
#include "dynopts.h"
#include "encodings.h"
#include "globalopts.h"
#include "logo.h"
#include "optsfile.h"
#include "optsdbus.h"
#include "roxterm.h"
#include "multitab.h"
#include "shortcuts.h"
#include "uri.h"

typedef enum {
    ROXTerm_Match_Invalid,
    ROXTerm_Match_Complete,
    ROXTerm_Match_WWW,
    ROXTerm_Match_FTP,
    ROXTerm_Match_MailTo
} ROXTerm_MatchType;

typedef struct {
    int tag;
    ROXTerm_MatchType type;
} ROXTerm_MatchMap;

struct ROXTermData {
    /* We do not own references to win, tab or widget */
    MultiWin *win;
    MultiTab *tab;
    GtkWidget *widget;            /* VteTerminal */
    GtkWidget *hbox;
    GtkWidget *scrollbar;
    int geom_width, geom_height;
    int nonfs_width, nonfs_height;
    pid_t pid;
    /* We own a reference to colour_scheme */
    Options *colour_scheme;
    char *matched_url;
    ROXTerm_MatchType match_type;
    Options *profile;
    char *special_command;      /* Next window/tab opened from this one should
                                   run this command instead of default */
    char **commandv;            /* Copied from --execute args */
    char *directory;            /* Copied from global_options_directory */
    GArray *match_map;
    gboolean no_respawn;
    GtkWidget *im_submenu1, *im_submenu2, *im_submenu3;
            /* Input Methods submenus; one each for popups and menu bar */
    gboolean running;
    DragReceiveData *drd;

    gboolean hold_over_uri;    /* Whether we've just received a button press
                               over a URI which may turn into a drag or click
                                */
    int hold_x, hold_y;        /* Position of event above */
    gulong hold_handler_id;    /* Signal handler id for the above event */
    char *encoding;
    double target_zoom_factor, current_zoom_factor;
    int zoom_index;
    char *title_template;   /* Set if new terminal created by roxterm_launch
                                and --title was given. */
    gboolean post_exit_tag;
};

#define PROFILE_NAME_KEY "roxterm_profile_name"

static GList *roxterm_terms = NULL;

static DynamicOptions *roxterm_profiles = NULL;

static void roxterm_apply_profile(ROXTermData * roxterm, VteTerminal * vte);

/********************** Encodings ***************************/

static GKeyFile *roxterm_encodings = NULL;

inline static char **roxterm_list_encodings(void)
{
    return encodings_list(roxterm_encodings);
}

static void roxterm_apply_encoding(ROXTermData *roxterm, VteTerminal *vte)
{
    vte_terminal_set_encoding(vte, roxterm->encoding);
}

void roxterm_encoding_toggled(GtkCheckMenuItem *item, gpointer win)
{
    if (gtk_check_menu_item_get_active(item)
            && !multi_win_get_ignore_toggles(win))
    {
        ROXTermData *roxterm =
            multi_tab_get_user_data(multi_win_get_current_tab(win));
        const char *encoding = menutree_encoding_from_widget(item);

        g_free(roxterm->encoding);
        roxterm->encoding = encoding ? g_strdup(encoding) : NULL;
        roxterm_apply_encoding(roxterm, VTE_TERMINAL(roxterm->widget));
    }
}

static void roxterm_encodings_changed(MenuTree *mtree, 
        const char *what_happened,
        const char *current_name, const char *new_name)
{
    if (!strcmp(what_happened, OPTSDBUS_DELETED))
    {
        menutree_remove_encoding(mtree, current_name);
    }
    else if (!strcmp(what_happened, OPTSDBUS_ADDED))
    {
        menutree_add_encoding(mtree, current_name, roxterm_encoding_toggled);
    }
    else if (!strcmp(what_happened, OPTSDBUS_RENAMED))
    {
        menutree_change_encoding(mtree, current_name, new_name,
                roxterm_encoding_toggled);
    }
    else
    {
        g_critical(_("Action '%s' not recognised for encodings"),
                what_happened);
    }
}

/******************** End Encodings *************************/

/*********************** URI handling ***********************/

inline static void roxterm_match_add(ROXTermData *roxterm, VteTerminal *vte,
        const char *match, ROXTerm_MatchType type)
{
    ROXTerm_MatchMap map;

    map.type = type;
#ifdef HAVE_VTE_TERMINAL_MATCH_ADD_GREGEX
    map.tag = vte_terminal_match_add_gregex(vte,
            g_regex_new(match, 0, 0, NULL), 0);
#else
    map.tag = vte_terminal_match_add(vte, match);
#endif
    vte_terminal_match_set_cursor_type(vte, map.tag, GDK_HAND2);
    g_array_append_val(roxterm->match_map, map);
}

/* There isn't a portable standard way of expressing a word boundary (POSIX
 * 1003.2 had '[[:<:]]', GNU has '\<') so the most sensible thing to do seems
 * to be to ignore them and allow valid URIs to have any old rubbish tacked on
 * to the front of them (Claws Mail appears to do this).
 */
#define URLSTART ""

#define URLMSGIDSET "[-A-Z\\^_a-z{|}~!\"#$%&'()*+,./0-9;:=?`]"
#define URLEND "[^]'.}>) \t\r\n,\\\"]"
#define URLHOST "[A-Za-z0-9][-A-Za-z0-9.]*"
#define URLEXTHOST "[A-Za-z0-9][-.:!%$^*&~#A-Za-z0-9@]*"
#define URLFQDNTAIL "\\.[-A-Za-z0-9.]+"
#define URLFQDN "[A-Za-z0-9]\\.[-A-Za-z0-9.]+"
#define URLSCHEME "(telnet:|nntp:|file:/|https?:|ftps?:|webcal:)"
/*
#define URLUSER "[A-Za-z0-9][-.A-Za-z0-9]*"
#define URLUSERANDPASS URLUSER ":[-,?;.:/!%$^*&~\"#'A-Za-z0-9]+"
#define URLPORT ":[0-9]+"
*/
#define URLPATH "[/?][-A-Za-z0-9_$.+!*(),;:@&=?/~#%]*" URLEND

static const char *full_urls[] = {
    URLSTART URLSCHEME "//" URLEXTHOST,
    URLSTART URLSCHEME "//" URLEXTHOST URLPATH
};

static const char *web_urls[] = {
    URLSTART "www[0-9]*" URLFQDNTAIL,
    URLSTART "www[0-9]*" URLFQDNTAIL URLPATH
};

static const char *ftp_urls[] = {
    URLSTART "ftp[0-9]*" URLFQDNTAIL,
    URLSTART "ftp[0-9]*" URLFQDNTAIL URLPATH
};

#define URL_EMAIL_USER "[-A-Za-z.0-9]+"

#define URL_NEWS URLSTART "news:" URLMSGIDSET "+@" URLEXTHOST

static const char *mailto_urls[] = {
    URLSTART URL_EMAIL_USER "@" URLHOST,
    URLSTART "mailto:" URL_EMAIL_USER "@" URLHOST \
        "\\?[A-Za-z]+=" URLMSGIDSET "+"
};

static void roxterm_add_matches(ROXTermData *roxterm, VteTerminal *vte)
{
    int n;

    for (n = 0; n < G_N_ELEMENTS(full_urls); ++n)
        roxterm_match_add(roxterm, vte, full_urls[n], ROXTerm_Match_Complete);
    for (n = 0; n < G_N_ELEMENTS(web_urls); ++n)
        roxterm_match_add(roxterm, vte, web_urls[n], ROXTerm_Match_WWW);
    for (n = 0; n < G_N_ELEMENTS(ftp_urls); ++n)
        roxterm_match_add(roxterm, vte, ftp_urls[n], ROXTerm_Match_FTP);
    for (n = 0; n < G_N_ELEMENTS(mailto_urls); ++n)
        roxterm_match_add(roxterm, vte, mailto_urls[n], ROXTerm_Match_MailTo);
    roxterm_match_add(roxterm, vte, URL_NEWS, ROXTerm_Match_Complete);
}

static ROXTerm_MatchType roxterm_get_match_type(ROXTermData *roxterm, int tag)
{
    int n;

    for (n = 0; n < roxterm->match_map->len; ++n)
    {
        ROXTerm_MatchMap *m = &g_array_index(roxterm->match_map,
                ROXTerm_MatchMap, n);

        if (m->tag == tag)
            return m->type;
    }
    return ROXTerm_Match_Invalid;
}

/********************* End URI handling *********************/

static void roxterm_save_nonfs_dimensions(ROXTermData *roxterm)
{
    if (!roxterm->win || !multi_win_is_fullscreen(roxterm->win))
    {
        roxterm->nonfs_width = roxterm->geom_width;
        roxterm->nonfs_height = roxterm->geom_height;
    }
}

/* Foreground (and palette?) colour change doesn't cause a redraw so force it */
inline static void roxterm_force_redraw(ROXTermData *roxterm)
{
    if (roxterm->widget)
        gtk_widget_queue_draw(roxterm->widget);
}

static void roxterm_set_geometry(ROXTermData *roxterm, int width, int height)
{
    roxterm->geom_width = width;
    roxterm->geom_height = height;
    roxterm_save_nonfs_dimensions(roxterm);
}

static void roxterm_default_dimensions(ROXTermData *roxterm)
{
    roxterm_set_geometry(roxterm,
            options_lookup_int_with_default(roxterm->profile, "width", 80),
            options_lookup_int_with_default(roxterm->profile, "height", 24));
}

static ROXTermData *roxterm_data_clone(ROXTermData *old_gt)
{
    ROXTermData *new_gt = g_new(ROXTermData, 1);

    *new_gt = *old_gt;
    new_gt->widget = NULL;
    new_gt->win = NULL;
    new_gt->tab = NULL;
    new_gt->running = FALSE;
    new_gt->title_template = NULL;
    new_gt->current_zoom_factor = 1.0;
    if (old_gt->colour_scheme)
    {
        new_gt->colour_scheme = colour_scheme_lookup_and_ref
            (options_get_leafname(old_gt->colour_scheme));
    }
    if (old_gt->profile)
    {
        new_gt->profile = dynamic_options_lookup_and_ref(roxterm_profiles,
            options_get_leafname(old_gt->profile), "roxterm profile");
    }
    if (old_gt->widget)
    {
        if (!multi_win_is_fullscreen(old_gt->win))
        {
            roxterm_set_geometry(new_gt,
                    (VTE_TERMINAL(old_gt->widget))->column_count,
                    (VTE_TERMINAL(old_gt->widget))->row_count);
        }
        else
        {
            roxterm_set_geometry(new_gt,
                    old_gt->nonfs_width, old_gt->nonfs_height);
        }
    }
    else
    {
        roxterm_default_dimensions(new_gt);
    }
    /* special_command should be transferred to new data and removed from old
     * one */
    old_gt->special_command = NULL;

    /* Only inherit commandv from templates, not from running terminals */
    if (!old_gt->running && old_gt->commandv)
        new_gt->commandv = global_options_copy_strv(old_gt->commandv);
    else
        new_gt->commandv = NULL;
    if (old_gt->pid >= 0)
    {
        char *pidfile = g_strdup_printf("/proc/%d/cwd", old_gt->pid);

        new_gt->directory = g_file_read_link(pidfile, NULL);
        /* According to a comment in gnome-terminal readlink()
         * returns a NULL/empty string in Solaris but we can still use the
         * /proc link if it exists */
        if (!new_gt->directory || !new_gt->directory[0])
        {
            g_free(new_gt->directory);
            new_gt->directory = NULL;
            if (g_file_test(pidfile, G_FILE_TEST_IS_SYMLINK))
                new_gt->directory = g_file_read_link(pidfile, NULL);
            else
                g_free(pidfile);
        }
        else
        {
            g_free(pidfile);
        }
    }
    else
    {
        new_gt->directory = NULL;
    }
    if (!new_gt->directory && old_gt->directory)
    {
        new_gt->directory = g_strdup(old_gt->directory);
    }

    if (old_gt->encoding)
        new_gt->encoding = g_strdup(old_gt->encoding);

    new_gt->im_submenu1 = new_gt->im_submenu2 = NULL;

    new_gt->match_map = g_array_new(FALSE, FALSE, sizeof(ROXTerm_MatchMap));
    new_gt->matched_url = NULL;

    return new_gt;
}

extern char **environ;

/* This function was originally mostly copied from gnome-terminal */
static char **roxterm_get_environment(ROXTermData *roxterm)
{
    char **p;
    int i;
    char **result;
#define EXTRA_ENV_VARS 6
    
    for (p = environ; *p; ++p);
    i = p - environ;
    result = g_new (char *, i + 1 + EXTRA_ENV_VARS);

    for (i = 0, p = environ; *p; p++)
    {
        /* Strip all these out, we'll replace some of them */
        if (g_str_has_prefix(*p, "COLUMNS=") ||
            g_str_has_prefix(*p, "LINES=") ||
            g_str_has_prefix(*p, "WINDOWID=") ||
            g_str_has_prefix(*p, "TERM=") ||
            g_str_has_prefix(*p, "COLORTERM=") ||
            g_str_has_prefix(*p, "DISPLAY=") ||
            g_str_has_prefix(*p, "ROXTERM_ID="))
        {
            /* nothing: do not copy */
        }
        else
        {
            result[i++] = g_strdup(*p);
        }
    }

    result[i++] = g_strdup("COLORTERM=roxterm");
    result[i++] = g_strdup("TERM=xterm");
    result[i++] = g_strdup_printf("WINDOWID=%ld",
            GDK_WINDOW_XWINDOW(roxterm->widget->window));
    result[i++] = g_strdup_printf("DISPLAY=%s",
            gdk_display_get_name(gtk_widget_get_display(roxterm->widget)));
    result[i++] = g_strdup_printf("ROXTERM_ID=%p", roxterm);
    
    result[i] = NULL;
    
    return result;
}

static GtkWindow *roxterm_get_toplevel(ROXTermData *roxterm)
{
    if (roxterm && roxterm->win)
    {
        GtkWidget *tl = multi_win_get_widget(roxterm->win);
        if (tl && GTK_WIDGET_TOPLEVEL(tl))
            return GTK_WINDOW(tl);
    }
    return NULL;
}

static char *get_default_command(ROXTermData *roxterm)
{
    char *command = NULL;
    struct passwd *pinfo = getpwuid(getuid());

    if (pinfo)
    {
        command = g_strdup(pinfo->pw_shell);
    }
    else
    {
        dlg_warning(roxterm_get_toplevel(roxterm),
                _("Unable to look up login details: %s"), strerror(errno));
    }
    if (!command || !command[0])
        command = g_strdup("/bin/sh");
    return command;
}

static void roxterm_run_command(ROXTermData *roxterm, VteTerminal *vte)
{
    char **commandv = NULL;
    char *command = NULL;
    char *filename = NULL;
    gboolean special = FALSE;    /* special_command and -e override login_shell
                                 */
    gboolean login = FALSE;
    char **env = roxterm_get_environment(roxterm);

    if (roxterm->special_command)
    {
        /* Use special_command, currently single string */
        command = roxterm->special_command;
        special = TRUE;
    }
    else if (roxterm->commandv)
    {
        /* Use -e commandv */
        special = TRUE;
    }
    else
    {
        /* Either use custom command from option or default (as single string)
         */
        if (options_lookup_int_with_default(roxterm->profile,
                    "use_custom_command", FALSE));
        {
            command = options_lookup_string(roxterm->profile, "command");
        }
        if (!command || !command[0])
        {
            command = get_default_command(roxterm);
        }
        else
        {
            /* Using custom command disables login_shell */
            special = TRUE;
        }
    }
    if (roxterm->commandv && !roxterm->special_command)
    {
        /* We're using roxterm->commandv, point commandv to it */
        commandv = roxterm->commandv;
        filename = commandv[0];
    }
    else
    {
        /* Parse our own commandv from single string command */
        int argc;
        GError *error = NULL;

        commandv = NULL;
        if (!g_shell_parse_argv(command, &argc, &commandv, &error))
        {
            dlg_critical(roxterm_get_toplevel(roxterm),
                    _("Unable to parse command %s: %s"),
                    command, error ? error->message : _("unknown reason"));
            if (commandv)
                g_strfreev(commandv);
            commandv = g_new(char *, 2);
            commandv[0] = g_strdup(get_default_command(roxterm));
            commandv[1] = NULL;
        }
        filename = commandv[0];
        if (!special && options_lookup_int_with_default(roxterm->profile,
                "login_shell", 0))
        {
            /* If login_shell, make sure commandv[0] is a leafname and
             * prepend - */
            char *leaf = strrchr(commandv[0], G_DIR_SEPARATOR);

            if (leaf)
                ++leaf;
            else
                leaf = commandv[0];
            commandv[0] = g_strconcat("-", leaf, NULL);
            login = TRUE;
        }
    }

    if (commandv && commandv[0])
    {
        gboolean xtmplog = options_lookup_int_with_default(roxterm->profile,
                "update_records", 1);

        roxterm->pid = vte_terminal_fork_command(vte, filename,
            commandv, env, roxterm->directory, login, xtmplog, xtmplog);
        if (roxterm->pid == -1)
        {
            dlg_warning(roxterm_get_toplevel(roxterm),
                    _("The command for this terminal failed to run"));
        }
    }
    else
    {
        dlg_critical(roxterm_get_toplevel(roxterm),
                _("Trying to run NULL command in new terminal"));
    }

    roxterm->special_command = NULL;
    /* If special_command was set, command points to the same string */
    g_free(command);
    if (filename != commandv[0])
        g_free(filename);
    if (commandv && commandv != roxterm->commandv)
        g_strfreev(commandv);
    g_strfreev(env);
}

static char *roxterm_lookup_uri_handler(ROXTermData *roxterm, const char *tag)
{
    char *filename = options_lookup_string(roxterm->profile, tag);

    if (g_file_test(filename, G_FILE_TEST_IS_DIR))
    {
        char *apprun = g_build_filename(filename, "AppRun", NULL);

        if (g_file_test(apprun, G_FILE_TEST_IS_EXECUTABLE))
        {
            g_free(filename);
            filename = apprun;
        }
        else
        {
            g_free(apprun);
        }
    }
    return filename;
}

static void roxterm_launch_browser(ROXTermData *roxterm, const char *url)
{
    char *browser_o = roxterm_lookup_uri_handler(roxterm, "browser");
    char *browse = uri_get_browser_command(url, browser_o);

    if (browser_o)
        g_free(browser_o);
    if (browse)
    {
        roxterm_spawn(roxterm, browse, options_lookup_int_with_default
            (roxterm->profile, "browser_spawn_type", 0));
        g_free(browse);
    }
}

static void roxterm_launch_email(ROXTermData *roxterm, const char *uri)
{
    char *mailer_o = roxterm_lookup_uri_handler(roxterm, "mailer");
    char *mailer = uri_get_mailer_command(uri, mailer_o);

    if (mailer_o)
        g_free(mailer_o);
    if (mailer)
    {
        roxterm_spawn(roxterm, mailer, options_lookup_int_with_default
            (roxterm->profile, "mailer_spawn_type", 0));
        g_free(mailer);
    }
}

static void roxterm_launch_uri(ROXTermData *roxterm)
{
    if (roxterm->match_type == ROXTerm_Match_MailTo)
        roxterm_launch_email(roxterm, roxterm->matched_url);
    else
        roxterm_launch_browser(roxterm, roxterm->matched_url);
}

static void roxterm_data_delete(ROXTermData *roxterm)
{
    /* This doesn't delete widgets because they're deleted when removed from
     * the parent */
    g_return_if_fail(roxterm);
    
    if (roxterm->post_exit_tag)
        g_source_remove(roxterm->post_exit_tag);
    if (roxterm->colour_scheme)
    {
        UNREF_LOG(colour_scheme_unref(roxterm->colour_scheme));
    }
    if (roxterm->profile)
    {
        UNREF_LOG(dynamic_options_unref(roxterm_profiles,
                options_get_leafname(roxterm->profile)));
    }
    if (roxterm->im_submenu1)
    {
        UNREF_LOG(g_object_unref(roxterm->im_submenu1));
    }
    if (roxterm->im_submenu2)
    {
        UNREF_LOG(g_object_unref(roxterm->im_submenu2));
    }
    drag_receive_data_delete(roxterm->drd);
    if (roxterm->commandv)
        g_strfreev(roxterm->commandv);
    g_free(roxterm->directory);
    g_free(roxterm->encoding);
    g_free(roxterm);
}

typedef enum {
    ROXTerm_DontShowURIMenuItems,
    ROXTerm_ShowWebURIMenuItems,
    ROXTerm_ShowMailURIMenuItems
} ROXTerm_URIMenuItemsShowType;

static void
set_show_uri_menu_items(MenuTree *tree, ROXTerm_URIMenuItemsShowType show_type)
{
    g_return_if_fail(tree);
    menutree_set_show_item(tree, MENUTREE_OPEN_IN_BROWSER,
        show_type == ROXTerm_ShowWebURIMenuItems);
    menutree_set_show_item(tree, MENUTREE_OPEN_IN_MAILER,
        show_type == ROXTerm_ShowMailURIMenuItems);
    menutree_set_show_item(tree, MENUTREE_COPY_URI,
        show_type != ROXTerm_DontShowURIMenuItems);
    menutree_set_show_item(tree, MENUTREE_URI_SEPARATOR,
        show_type != ROXTerm_DontShowURIMenuItems);
}

static void
roxterm_set_show_uri_menu_items(ROXTermData * roxterm,
    ROXTerm_URIMenuItemsShowType show_type)
{
    set_show_uri_menu_items(multi_win_get_popup_menu(roxterm->win),
            show_type);
    set_show_uri_menu_items(multi_win_get_short_popup_menu(roxterm->win),
            show_type);
}

static void roxterm_update_background(ROXTermData * roxterm, VteTerminal * vte)
{
    char *background_file;
    double saturation = options_lookup_double(roxterm->profile, "saturation");
    gboolean true_trans = FALSE;

    if (saturation == -1)
    {
        saturation = 1;
    }
    else if (saturation < 0 || saturation > 1)
    {
        g_warning(_("Saturation option %f out of range"), saturation);
        saturation = CLAMP(saturation, 0, 1);
    }
    vte_terminal_set_background_tint_color(vte,
        colour_scheme_get_background_colour(roxterm->colour_scheme, TRUE));

    switch (options_lookup_int(roxterm->profile, "background_type"))
    {
        case 1:
            background_file = options_lookup_string(roxterm->profile,
                    "background_img");
            if (background_file &&
                    g_file_test(background_file, G_FILE_TEST_EXISTS))
            {
                vte_terminal_set_background_image_file(vte, background_file);
                vte_terminal_set_scroll_background(vte,
                    options_lookup_int(roxterm->profile, "scroll_background")
                    == 1);
                break;
            }
            else
            {
                vte_terminal_set_background_image(vte, NULL);
            }
            vte_terminal_set_background_transparent(vte, FALSE);
            if (background_file)
                g_free(background_file);
            break;
        case 2:
            vte_terminal_set_background_image(vte, NULL);
            true_trans = multi_win_composite(roxterm->win) &&
                    options_lookup_int_with_default(roxterm->profile,
                            "true_translucence", 1);
            vte_terminal_set_background_transparent(vte,
                    saturation < 1 && !true_trans);
            break;
        default:
            saturation = 1;
            vte_terminal_set_background_image(vte, NULL);
            vte_terminal_set_background_transparent(vte, FALSE);
    }
    vte_terminal_set_background_saturation(vte, saturation);
    vte_terminal_set_opacity(vte, true_trans ?
            (guint16) (0xffff * (1 - saturation)) : 0xffff);
}

/* use_default = pick a default colour if option not given, otherwise
 * don't make the call */
static void
roxterm_update_cursor_colour(ROXTermData * roxterm, VteTerminal * vte)
{
    const GdkColor *cursor_colour =
            colour_scheme_get_cursor_colour(roxterm->colour_scheme, TRUE);

    if (cursor_colour)
    {
        vte_terminal_set_color_cursor(vte, cursor_colour);
    }
    else
    {
        vte_terminal_set_color_cursor(vte, NULL);
    }
}

guint16 extrapolate_chroma(guint16 bg, guint16 fg, double factor)
{
    gint32 ext = (gint32) bg +
        (gint32) ((double) ((gint32) fg - (gint32) bg) * factor);

    return (guint16) CLAMP(ext, 0, 0xffff);
}

static const GdkColor *extrapolate_colours(const GdkColor *bg,
        const GdkColor *fg, double factor)
{
    static GdkColor ext;

#define EXTRAPOLATE(c) ext. c = \
    extrapolate_chroma(bg ? bg->  c : 0, fg ? fg->  c : 0xffff, factor)
    EXTRAPOLATE(red);
    EXTRAPOLATE(green);
    EXTRAPOLATE(blue);
    return &ext;
}

static void
roxterm_apply_colour_scheme(ROXTermData * roxterm, VteTerminal * vte)
{
    int ncolours = 0;
    const GdkColor *palette = NULL;
    const GdkColor *foreground =
            colour_scheme_get_foreground_colour(roxterm->colour_scheme, TRUE);
    const GdkColor *background =
            colour_scheme_get_background_colour(roxterm->colour_scheme, TRUE);

    vte_terminal_set_default_colors(vte);
    ncolours = colour_scheme_get_palette_size(roxterm->colour_scheme);
    if (ncolours)
        palette = colour_scheme_get_palette(roxterm->colour_scheme);
    vte_terminal_set_colors(vte, foreground, background, palette, ncolours);
    if (!ncolours && foreground && background)
    {
        vte_terminal_set_color_bold(vte,
                extrapolate_colours(background, foreground, 1.2));
        vte_terminal_set_color_dim(vte,
                extrapolate_colours(background, foreground, 0.7));
    }
    roxterm_update_cursor_colour(roxterm, vte);
    roxterm_force_redraw(roxterm);
}

static void roxterm_geometry_func(ROXTermData *roxterm,
        GdkGeometry *geom, GdkWindowHints *hints)
{
    VteTerminal *vte = VTE_TERMINAL(roxterm->widget);

    vte_terminal_get_padding(vte, &geom->base_width, &geom->base_height);
    geom->width_inc = vte_terminal_get_char_width(vte);
    geom->height_inc = vte_terminal_get_char_height(vte);
    geom->min_width = geom->base_width + 4 * geom->width_inc;
    geom->min_height = geom->base_height + 4 * geom->height_inc;
    *hints = GDK_HINT_RESIZE_INC | GDK_HINT_BASE_SIZE | GDK_HINT_MIN_SIZE;
}

static void
roxterm_cell_to_pixel_size(ROXTermData * roxterm, int *pwidth, int *pheight)
{
    int xpad, ypad;
    VteTerminal *vte = VTE_TERMINAL(roxterm->widget);

    vte_terminal_get_padding(vte, &xpad, &ypad);
    *pwidth = *pwidth * vte->char_width + xpad;
    *pheight = *pheight * vte->char_height + ypad;
}

static void roxterm_size_func(ROXTermData *roxterm, MultiWinSizeReason reason,
        int *pwidth, int *pheight)
{
    VteTerminal *vte = VTE_TERMINAL(roxterm->widget);

    if (reason == MultiWin_UpdateConfiguredSize)
    {
        roxterm_set_geometry(roxterm,
                *pwidth = vte->column_count, *pheight = vte->row_count);
    }
    else if (reason == MultiWin_GetActualSize)
    {
        *pwidth = vte->column_count;
        *pheight = vte->row_count;
    }
    else /* if (reason == MultiWin_GetTargetSize) */
    {
        *pwidth = roxterm->nonfs_width;
        *pheight = roxterm->nonfs_height;
    }
    roxterm_cell_to_pixel_size(roxterm, pwidth, pheight);
}

static void roxterm_update_size(ROXTermData * roxterm, VteTerminal * vte)
{
    if (multi_win_get_current_tab(roxterm->win) == roxterm->tab)
    {
        int width;
        int height;

        roxterm_set_geometry(roxterm,
            options_lookup_int_with_default(roxterm->profile, "width", 80),
            options_lookup_int_with_default(roxterm->profile, "height", 24));
        vte_terminal_set_size(vte, roxterm->geom_width, roxterm->geom_height);
        width = roxterm->geom_width;
        height = roxterm->geom_height;
        roxterm_cell_to_pixel_size(roxterm, &width, &height);
        multi_win_set_size(roxterm->win, roxterm->widget, width, height);
    }
}

static void roxterm_update_geometry(ROXTermData * roxterm, VteTerminal * vte)
{
    if (multi_win_get_current_tab(roxterm->win) == roxterm->tab)
    {
        GdkGeometry geom;
        GdkWindowHints hints;
        int width = roxterm->geom_width;
        int height = roxterm->geom_height;

        /*
           int width = vte->column_count;
           int height = vte->row_count;
         */

        roxterm_geometry_func(roxterm, &geom, &hints);
        multi_win_set_geometry_hints(roxterm->win, roxterm->widget,
            &geom, hints);
        roxterm_cell_to_pixel_size(roxterm, &width, &height);
        multi_win_set_size(roxterm->win, roxterm->widget, width, height);
    }
}

static void
roxterm_update_font(ROXTermData *roxterm, VteTerminal *vte,
    gboolean update_geometry)
{
    char *fdesc;
    PangoFontDescription *pango_desc = NULL;

    fdesc = options_lookup_string(roxterm->profile, "font");
    if (fdesc || roxterm->target_zoom_factor != roxterm->current_zoom_factor ||
            update_geometry)
    {
        if (fdesc && roxterm->target_zoom_factor == 1.0)
        {
            vte_terminal_set_font_from_string(vte, fdesc);
        }
        else
        {
            int size;
            gboolean abs = FALSE;
            double zf = roxterm->target_zoom_factor;

            if (fdesc)
            {
                pango_desc = pango_font_description_from_string(fdesc);
            }
            else
            {
                pango_desc = pango_font_description_copy(
                        vte_terminal_get_font(vte));
                zf /= roxterm->current_zoom_factor;
            }
            if (!pango_desc)
            {
                g_free(fdesc);
                g_return_if_fail(pango_desc != NULL);
            }
            size = pango_font_description_get_size(pango_desc);
            if (!size)
                size = 10 * PANGO_SCALE;
            else
                abs = pango_font_description_get_size_is_absolute(pango_desc);
            size = (int) ((double) size * zf);
            if (abs)
                pango_font_description_set_absolute_size(pango_desc, size);
            else
                pango_font_description_set_size(pango_desc, size);
            vte_terminal_set_font(vte, pango_desc);
            pango_font_description_free(pango_desc);
        }
        g_free(fdesc);
    }
    roxterm->current_zoom_factor = roxterm->target_zoom_factor;
    if (update_geometry)
        roxterm_update_geometry(roxterm, vte);
}

static void roxterm_set_zoom_factor(ROXTermData *roxterm, double factor,
        int index)
{
    roxterm->zoom_index = index;
    roxterm->target_zoom_factor = factor;
    roxterm_update_font(roxterm, VTE_TERMINAL(roxterm->widget), TRUE);
}

/* Change a terminal's font etc so it matches size with another */
static void roxterm_match_text_size(ROXTermData *roxterm, ROXTermData *other)
{
    VteTerminal *vte;
    const PangoFontDescription *fd;
    
    if (roxterm == other)
        return;
    vte = VTE_TERMINAL(roxterm->widget);
    fd = vte_terminal_get_font(VTE_TERMINAL(other->widget));
    roxterm->target_zoom_factor = other->current_zoom_factor;
    roxterm->current_zoom_factor = other->current_zoom_factor;
    roxterm->zoom_index = other->zoom_index;
    if (!pango_font_description_equal(
            vte_terminal_get_font(VTE_TERMINAL(roxterm->widget)), fd))
        vte_terminal_set_font(vte, fd);
    if (roxterm->geom_width != other->geom_width
            || roxterm->geom_height != other->geom_height)
    {

        roxterm_set_geometry(roxterm,
                other->geom_width, other->geom_height);
        vte_terminal_set_size(VTE_TERMINAL(roxterm->widget),
                roxterm->geom_width, roxterm->geom_height);
    }
}

static void roxterm_about_www_hook(GtkAboutDialog *about,
        const gchar *link, gpointer data)
{
    roxterm_launch_browser(data, link);
}

static void roxterm_about_email_hook(GtkAboutDialog *about,
        const gchar *link, gpointer data)
{
    roxterm_launch_email(data, link);
}

static gboolean roxterm_popup_handler(GtkWidget * widget, ROXTermData * roxterm)
{
    (void) widget;
    roxterm_set_show_uri_menu_items(roxterm, ROXTerm_DontShowURIMenuItems);
    multi_tab_popup_menu(roxterm->tab, 0, gtk_get_current_event_time());
    return TRUE;
}

/* Checks whether a button event position is over a matched expression; if so
 * roxterm->matched_url is set and the result is TRUE */
static gboolean roxterm_check_match(ROXTermData *roxterm, VteTerminal *vte,
        int event_x, int event_y)
{
    int xpad, ypad;
    int tag;

    vte_terminal_get_padding(vte, &xpad, &ypad);
    g_free(roxterm->matched_url);
    roxterm->matched_url = vte_terminal_match_check(vte,
        (event_x - ypad) / vte->char_width,
        (event_y - ypad) / vte->char_height, &tag);
    if (roxterm->matched_url)
        roxterm->match_type = roxterm_get_match_type(roxterm, tag);
    return roxterm->matched_url != NULL;
}

static void roxterm_clear_hold_over_uri(ROXTermData *roxterm)
{
    if (roxterm->hold_over_uri)
    {
        g_signal_handler_disconnect(roxterm->widget, roxterm->hold_handler_id);
        roxterm->hold_over_uri = FALSE;
    }
}

inline static void roxterm_clear_drag_url(ROXTermData *roxterm)
{
    roxterm_clear_hold_over_uri(roxterm);
}

static GtkTargetList *roxterm_get_uri_drag_target_list(void)
{
    static const GtkTargetEntry target_table[] = {
        { (char *) "text/uri-list", 0, ROXTERM_DRAG_TARGET_URI_LIST },
        { (char *) "UTF8_STRING", 0, ROXTERM_DRAG_TARGET_UTF8_STRING }
    };

    return gtk_target_list_new(target_table, G_N_ELEMENTS(target_table));
}

static void roxterm_uri_drag_ended(GtkWidget *widget,
        GdkDragContext *drag_context, ROXTermData *roxterm)
{
    roxterm_clear_drag_url(roxterm);
}

static void roxterm_uri_drag_data_get(GtkWidget *widget,
        GdkDragContext *drag_context, GtkSelectionData *selection,
        guint info, guint time, ROXTermData *roxterm)
{
    const char *encoding;
    char *uri_list[2];
    char *utf8 = NULL;

    g_return_if_fail(roxterm->matched_url);

    encoding = vte_terminal_get_encoding(VTE_TERMINAL(roxterm->widget));
    if (g_ascii_strcasecmp(encoding, "utf-8"))
    {
        GError *err = NULL;

        utf8 = g_convert(roxterm->matched_url, -1, "UTF-8", encoding,
                NULL, NULL, &err);
        if (err)
        {
            dlg_warning(roxterm_get_toplevel(roxterm),
                    _("Unable to convert dragged URI to UTF-8: %s"),
                    err->message);
            g_free(utf8);
            utf8 = NULL;
        }
    }
    if (!utf8)
        utf8 = g_strdup(roxterm->matched_url);

    if (info == ROXTERM_DRAG_TARGET_URI_LIST)
    {
        if (roxterm->match_type == ROXTerm_Match_WWW)
        {
            uri_list[0] = g_strdup_printf("http://%s", utf8);
        }
        else if (roxterm->match_type == ROXTerm_Match_FTP)
        {
            uri_list[0] = g_strdup_printf("ftp://%s", utf8);
        }
        else if (roxterm->match_type == ROXTerm_Match_MailTo
                && strncmp(utf8, "mailto:", 7))
        {
            uri_list[0] = g_strdup_printf("mailto:%s", utf8);
        }
        else
        {
            uri_list[0] = g_strdup(utf8);
        }
        uri_list[1] = NULL;
        gtk_selection_data_set_uris(selection, uri_list);
        g_free(uri_list[0]);
    }
    else
    {
        gtk_selection_data_set_text(selection, utf8, -1);
    }
    g_free(utf8);
}

/* Starts drag & drop process if user tries to drag a URL */
static gboolean roxterm_motion_handler(GtkWidget *widget, GdkEventButton *event,
        ROXTermData *roxterm)
{
    GtkTargetList *target_list;

    if (!gtk_drag_check_threshold(widget, roxterm->hold_x,
                roxterm->hold_y, event->x, event->y)
            || !roxterm->hold_over_uri)
    {
        return FALSE;
    }

    roxterm_clear_hold_over_uri(roxterm);
    g_return_val_if_fail(roxterm->matched_url, FALSE);


    target_list = roxterm_get_uri_drag_target_list();
    gtk_drag_begin(roxterm->widget, target_list, GDK_ACTION_COPY, 1,
            (GdkEvent *) event);
    
    return TRUE;
}

static gboolean roxterm_click_handler(GtkWidget *widget,
        GdkEventButton *event, ROXTermData *roxterm)
{
    VteTerminal *vte = VTE_TERMINAL(roxterm->widget);

    (void) widget;

    roxterm_clear_drag_url(roxterm);
    roxterm_check_match(roxterm, vte, event->x, event->y);
    if (event->button == 3 && !(event->state & (GDK_SHIFT_MASK |
                GDK_CONTROL_MASK | GDK_MOD1_MASK)))
    {
        ROXTerm_URIMenuItemsShowType show_type = ROXTerm_DontShowURIMenuItems;

        if (roxterm->matched_url)
        {
            if (roxterm->match_type == ROXTerm_Match_MailTo)
                show_type = ROXTerm_ShowMailURIMenuItems;
            else
                show_type = ROXTerm_ShowWebURIMenuItems;
        }
        roxterm_set_show_uri_menu_items(roxterm, show_type);
        multi_tab_popup_menu(roxterm->tab, event->button, event->time);
        return TRUE;
    }
    else if ((event->state & GDK_CONTROL_MASK) && event->button == 1
             && event->type == GDK_BUTTON_PRESS && roxterm->matched_url)
    {
        roxterm->hold_over_uri = TRUE;
        roxterm->hold_x = event->x;
        roxterm->hold_y = event->y;
        roxterm->hold_handler_id = g_signal_connect(roxterm->widget,
                "motion-notify-event",
                G_CALLBACK(roxterm_motion_handler), roxterm);
        return TRUE;
    }
    return FALSE;
}

static gboolean roxterm_release_handler(GtkWidget *widget,
        GdkEventButton *event, ROXTermData *roxterm)
{
    gboolean result = FALSE;

    if ((event->state & GDK_CONTROL_MASK) && event->button == 1
            && roxterm->hold_over_uri && roxterm->matched_url)
    {
        roxterm_launch_uri(roxterm);
        result = TRUE;
    }
    roxterm_clear_hold_over_uri(roxterm);
    return result;
}

static void roxterm_title_handler(VteTerminal * vte, ROXTermData * roxterm)
{
    multi_tab_set_title(roxterm->tab,
        vte->window_title ? vte->window_title : "roxterm");
}

inline static void
roxterm_update_geom_record(ROXTermData * roxterm, VteTerminal * vte)
{
    roxterm_set_geometry(roxterm, vte->column_count, vte->row_count);
}

/* data is cast to char const **pname - indirect pointer to name to check for -
 * if a match is found, *pname is set to NULL */
static void check_if_name_matches_property(GtkWidget *widget, gpointer data)
{
    const char *profile_name = g_object_get_data(G_OBJECT(widget),
            PROFILE_NAME_KEY);
    char const **pname = data;
    gboolean check = FALSE;

    g_return_if_fail(profile_name);
    /* pname may have been NULLed by an earlier iteration */
    if (!*pname)
        return;
    if (!strcmp(profile_name, *pname))
    {
        check = TRUE;
        *pname = NULL;
    }
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(widget), check);
}

static void check_preferences_submenu(MenuTree *tree, MenuTreeID id,
        const char *current_name)
{
    GtkMenu *submenu = menutree_submenu_from_id(tree, id);
    const char *name = current_name;

    gtk_container_foreach(GTK_CONTAINER(submenu),
            check_if_name_matches_property, &name);
    if (name)
    {
        /* name hasn't been NULLed, no match, try "ghost" */
        char *ghost_name = g_strdup_printf("(%s)", current_name);

        name = ghost_name;
        gtk_container_foreach(GTK_CONTAINER(submenu),
                check_if_name_matches_property, &name);
        g_free(ghost_name);
        if (name)
        {
            g_critical(_("No menu item matches profile/scheme '%s'"),
                    current_name);
        }
    }
}

static void check_preferences_submenu_pair(ROXTermData *roxterm, MenuTreeID id,
        const char *current_name)
{
    MenuTree *tree = multi_win_get_popup_menu(roxterm->win);

    multi_win_set_ignore_toggles(roxterm->win, TRUE);
    if (tree)
    {
        check_preferences_submenu(tree, id, current_name);
    }
    tree = multi_win_get_menu_bar(roxterm->win);
    if (tree)
    {
        check_preferences_submenu(tree, id, current_name);
    }
    multi_win_set_ignore_toggles(roxterm->win, FALSE);
}

static void create_im_submenus(ROXTermData *roxterm, VteTerminal *vte)
{
    /* Need to create extra refs for IM submenus because they get removed
     * from parent menus when switching tabs */
    if (!roxterm->im_submenu1)
    {
        roxterm->im_submenu1 = gtk_menu_new();
        g_object_ref(roxterm->im_submenu1);
        vte_terminal_im_append_menuitems(vte,
                GTK_MENU_SHELL(roxterm->im_submenu1));
    }
    if (!roxterm->im_submenu2)
    {
        roxterm->im_submenu2 = gtk_menu_new();
        g_object_ref(roxterm->im_submenu2);
        vte_terminal_im_append_menuitems(vte,
                GTK_MENU_SHELL(roxterm->im_submenu2));
    }
    if (!roxterm->im_submenu3)
    {
        roxterm->im_submenu3 = gtk_menu_new();
        g_object_ref(roxterm->im_submenu3);
        vte_terminal_im_append_menuitems(vte,
                GTK_MENU_SHELL(roxterm->im_submenu3));
    }
}

static void roxterm_tab_selection_handler(ROXTermData * roxterm, MultiTab * tab)
{
    VteTerminal *vte = VTE_TERMINAL(roxterm->widget);
    MenuTree *menu_bar = multi_win_get_menu_bar(roxterm->win);
    MenuTree *popup_menu = multi_win_get_popup_menu(roxterm->win);
    MenuTree *short_popup = multi_win_get_short_popup_menu(roxterm->win);

    check_preferences_submenu_pair(roxterm,
            MENUTREE_PREFERENCES_SELECT_PROFILE,
            options_get_leafname(roxterm->profile));
    check_preferences_submenu_pair(roxterm,
            MENUTREE_PREFERENCES_SELECT_COLOUR_SCHEME,
            options_get_leafname(roxterm->colour_scheme));
    check_preferences_submenu_pair(roxterm,
            MENUTREE_PREFERENCES_SELECT_SHORTCUTS,
            options_get_leafname(multi_win_get_shortcut_scheme(roxterm->win)));
    
    /* Creation of im submenus deferred to this point because vte
     * widget must be realized first */
    create_im_submenus(roxterm, vte);
    menutree_attach_im_submenu(popup_menu, roxterm->im_submenu1);
    menutree_attach_im_submenu(menu_bar, roxterm->im_submenu2);
    menutree_attach_im_submenu(short_popup, roxterm->im_submenu3);

    multi_win_set_ignore_toggles(roxterm->win, TRUE);
    if (!roxterm->running)
    {
        char **encodings = roxterm_list_encodings();

        menutree_build_encodings_menu(popup_menu,
                encodings, roxterm_encoding_toggled);
        menutree_build_encodings_menu(menu_bar,
                encodings, roxterm_encoding_toggled);
        menutree_build_encodings_menu(short_popup,
                encodings, roxterm_encoding_toggled);
        if (encodings)
            g_strfreev(encodings);
    }
    menutree_select_encoding(popup_menu, roxterm->encoding);
    menutree_select_encoding(menu_bar, roxterm->encoding);
    menutree_select_encoding(short_popup, roxterm->encoding);
    multi_win_set_ignore_toggles(roxterm->win, FALSE);

    if (!roxterm->running)
    {
        roxterm->running = TRUE;
        roxterm_run_command(roxterm, vte);
    }

    roxterm_update_geom_record(roxterm, vte);
}

static void roxterm_browser_action(MultiWin * win)
{
    ROXTermData *roxterm = multi_win_get_user_data_for_current_tab(win);

    g_return_if_fail(roxterm);
    if (roxterm->matched_url)
        roxterm_launch_browser(roxterm, roxterm->matched_url);
}

static void roxterm_mailto_action(MultiWin * win)
{
    ROXTermData *roxterm = multi_win_get_user_data_for_current_tab(win);

    g_return_if_fail(roxterm);
    if (roxterm->matched_url)
        roxterm_launch_email(roxterm, roxterm->matched_url);
}

static void roxterm_copy_url_action(MultiWin * win)
{
    ROXTermData *roxterm = multi_win_get_user_data_for_current_tab(win);

    g_return_if_fail(roxterm);
    if (roxterm->matched_url)
    {
        gtk_clipboard_set_text(gtk_clipboard_get(GDK_SELECTION_PRIMARY),
            roxterm->matched_url, -1);
        gtk_clipboard_set_text(gtk_clipboard_get(GDK_SELECTION_CLIPBOARD),
            roxterm->matched_url, -1);
    }
}

static void roxterm_copy_clipboard_action(MultiWin * win)
{
    ROXTermData *roxterm = multi_win_get_user_data_for_current_tab(win);

    g_return_if_fail(roxterm);
    vte_terminal_copy_clipboard(VTE_TERMINAL(roxterm->widget));
}

static void roxterm_paste_clipboard_action(MultiWin * win)
{
    ROXTermData *roxterm = multi_win_get_user_data_for_current_tab(win);

    g_return_if_fail(roxterm);
    vte_terminal_paste_clipboard(VTE_TERMINAL(roxterm->widget));
}

static void roxterm_copy_and_paste_action(MultiWin * win)
{
    ROXTermData *roxterm = multi_win_get_user_data_for_current_tab(win);

    g_return_if_fail(roxterm);
    vte_terminal_copy_clipboard(VTE_TERMINAL(roxterm->widget));
    vte_terminal_paste_clipboard(VTE_TERMINAL(roxterm->widget));
}

static void roxterm_reset_action(MultiWin * win)
{
    ROXTermData *roxterm = multi_win_get_user_data_for_current_tab(win);

    g_return_if_fail(roxterm);
    vte_terminal_reset(VTE_TERMINAL(roxterm->widget), TRUE, FALSE);
}

static void roxterm_reset_and_clear_action(MultiWin * win)
{
    ROXTermData *roxterm = multi_win_get_user_data_for_current_tab(win);

    g_return_if_fail(roxterm);
    vte_terminal_reset(VTE_TERMINAL(roxterm->widget), TRUE, TRUE);
}

static void roxterm_edit_profile_action(MultiWin * win)
{
    ROXTermData *roxterm = multi_win_get_user_data_for_current_tab(win);

    g_return_if_fail(roxterm);
    optsdbus_send_edit_profile_message(options_get_leafname(roxterm->profile));
}

static void roxterm_edit_colour_scheme_action(MultiWin * win)
{
    ROXTermData *roxterm = multi_win_get_user_data_for_current_tab(win);

    g_return_if_fail(roxterm);
    optsdbus_send_edit_colour_scheme_message(
            options_get_leafname(roxterm->colour_scheme));
}

static void roxterm_show_about(MultiWin * win)
{
    ROXTermData *roxterm = multi_win_get_user_data_for_current_tab(win);

    g_return_if_fail(roxterm);
    about_dialog_show(GTK_WINDOW(multi_win_get_widget(win)),
            roxterm_about_www_hook, roxterm_about_email_hook, roxterm);
}

static void roxterm_show_manual(MultiWin * win)
{
    ROXTermData *roxterm = multi_win_get_user_data_for_current_tab(win);
    char *filename = NULL;
    char *url;
    
    g_return_if_fail(roxterm);

    if (global_options_appdir)
    {
        filename = g_build_filename(global_options_appdir,
            "Help", "index.html#manual", NULL);
    }
    else
    {
        filename = g_build_filename(PKG_DOC_DIR, "index.html#manual", NULL);
    }
    url = g_strconcat("file://", filename, NULL);

    roxterm_launch_browser(roxterm, url);
    g_free(url);
    g_free(filename);
}

static void roxterm_open_config_manager(MultiWin * win)
{
    (void) win;
    optsdbus_send_edit_opts_message("Configlet", NULL);
}

static void
roxterm_style_change_handler(GtkWidget * widget, GtkStyle * previous_style,
    ROXTermData * roxterm)
{
    roxterm_update_geometry(roxterm, VTE_TERMINAL(roxterm->widget));
}

static void
roxterm_char_size_changed(GtkSettings * settings, guint arg1, guint arg2,
    ROXTermData * roxterm)
{
    roxterm_update_geometry(roxterm, VTE_TERMINAL(roxterm->widget));
}

typedef enum {
    roxterm_ChildExitClose,
    roxterm_ChildExitHold,
    roxterm_ChildExitRespawn,
    roxterm_ChildExitAsk
} roxterm_ChildExitAction;

static gboolean roxterm_post_child_exit(ROXTermData *roxterm)
{
    roxterm_ChildExitAction action =
        options_lookup_int_with_default(roxterm->profile, "exit_action",
                roxterm_ChildExitClose);

    if (action == roxterm_ChildExitAsk)
    {
        GtkWidget *dialog = gtk_message_dialog_new(
                roxterm_get_toplevel(roxterm),
                GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT |
                        GTK_DIALOG_NO_SEPARATOR,
                GTK_MESSAGE_QUESTION, GTK_BUTTONS_NONE,
                _("The command in this terminal has terminated. "
                "What do you want to do with the terminal now?"));
        GtkWidget *respawn;
        
        gtk_dialog_add_button(GTK_DIALOG(dialog),
                GTK_STOCK_CLOSE, roxterm_ChildExitClose);
        gtk_dialog_add_button(GTK_DIALOG(dialog),
                _("Leave open"), roxterm_ChildExitHold);
        respawn = gtk_dialog_add_button(GTK_DIALOG(dialog),
                _("Rerun command"), roxterm_ChildExitRespawn);
        if (roxterm->no_respawn)
            gtk_widget_set_sensitive(respawn, FALSE);
        gtk_widget_show_all(dialog);
        action = gtk_dialog_run(GTK_DIALOG(dialog));
        gtk_widget_destroy(dialog);
    }
    roxterm->post_exit_tag = 0;
    if (action == roxterm_ChildExitRespawn && roxterm->no_respawn)
        action = roxterm_ChildExitClose;
    switch (action)
    {
        case roxterm_ChildExitClose:
            multi_tab_delete(roxterm->tab);
            break;
        case roxterm_ChildExitHold:
            break;
        case roxterm_ChildExitRespawn:
            roxterm_run_command(roxterm, VTE_TERMINAL(roxterm->widget));
            break;
        default:
            break;
    }
    return FALSE;
}

static void roxterm_child_exited(VteTerminal *vte, ROXTermData *roxterm)
{
    double delay = 0;
    
    if ((options_lookup_int(roxterm->profile, "exit_action") !=
            roxterm_ChildExitAsk) &&
        ((delay = options_lookup_double(roxterm->profile, "exit_pause")) != 0))
    {
        vte_terminal_feed(vte, _("\n.\n"), -1);
        roxterm->post_exit_tag = g_timeout_add((guint) (delay * 1000.0),
                (GSourceFunc) roxterm_post_child_exit, roxterm);
    }
    else
    {
        roxterm_post_child_exit(roxterm);
    }
}

inline static ROXTermData *roxterm_from_menutree(MenuTree *mtree)
{
    return (ROXTermData *) multi_win_get_user_data_for_current_tab(
                     (MultiWin *) mtree->user_data);

}

static void roxterm_change_profile(ROXTermData *roxterm, Options *profile)
{
    if (roxterm->profile != profile)
    {
        if (roxterm->profile)
        {
            dynamic_options_unref(roxterm_profiles,
                    options_get_leafname(roxterm->profile));
        }
        roxterm->profile = profile;
        options_ref(roxterm->profile);
        roxterm_apply_profile(roxterm, VTE_TERMINAL(roxterm->widget));
        roxterm_update_size(roxterm, VTE_TERMINAL(roxterm->widget));
    }
}

static void roxterm_change_colour_scheme(ROXTermData *roxterm,
        Options *colour_scheme)
{
    if (roxterm->colour_scheme != colour_scheme)
    {
        if (roxterm->colour_scheme)
            colour_scheme_unref(roxterm->colour_scheme);
        roxterm->colour_scheme = colour_scheme;
        options_ref(colour_scheme);
        roxterm_apply_colour_scheme(roxterm, VTE_TERMINAL(roxterm->widget));
    }
}
            
static void match_text_size_foreach_tab(MultiTab *tab, void *data)
{
    ROXTermData *roxterm = multi_tab_get_user_data(tab);
    
    if (roxterm != data)
        roxterm_match_text_size(roxterm, data);
}

static void roxterm_profile_selected(GtkCheckMenuItem *mitem, MenuTree *mtree)
{
    ROXTermData *roxterm = roxterm_from_menutree(mtree);
    const char *profile_name;

    if (!roxterm || multi_win_get_ignore_toggles(roxterm->win))
        return;

    profile_name = g_object_get_data(G_OBJECT(mitem), PROFILE_NAME_KEY);
    if (!profile_name)
    {
        g_critical(_("Menu item has no '%s' data"), PROFILE_NAME_KEY);
        return;
    }
    if (!gtk_check_menu_item_get_active(mitem))
    {
        check_preferences_submenu_pair(roxterm,
                MENUTREE_PREFERENCES_SELECT_PROFILE, profile_name);
    }

    if (strcmp(profile_name, options_get_leafname(roxterm->profile)))
    {
        Options *profile = dynamic_options_lookup_and_ref(roxterm_profiles,
            profile_name, "roxterm profile");

        if (profile)
        {
            roxterm_change_profile(roxterm, profile);
            /* All tabs in the same window must have same size */
            multi_win_foreach_tab(roxterm->win, match_text_size_foreach_tab,
                    roxterm);
            /* Have one more ref than we need, so decrease it */
            options_unref(profile);
        }
        else
        {
            dlg_warning(roxterm_get_toplevel(roxterm),
                    _("Profile '%s' not found"), profile_name);
        }
    }
}

static void roxterm_colour_scheme_selected(GtkCheckMenuItem *mitem,
        MenuTree *mtree)
{
    ROXTermData *roxterm = roxterm_from_menutree(mtree);
    const char *scheme_name;

    if (!roxterm || multi_win_get_ignore_toggles(roxterm->win))
        return;

    scheme_name = g_object_get_data(G_OBJECT(mitem), PROFILE_NAME_KEY);
    if (!scheme_name)
    {
        g_critical(_("Menu item has no '%s' data"), PROFILE_NAME_KEY);
        return;
    }
    if (!gtk_check_menu_item_get_active(mitem))
    {
        check_preferences_submenu_pair(roxterm,
                MENUTREE_PREFERENCES_SELECT_COLOUR_SCHEME, scheme_name);
    }

    if (strcmp(scheme_name, options_get_leafname(roxterm->colour_scheme)))
    {
        Options *colour_scheme = colour_scheme_lookup_and_ref(scheme_name);

        if (colour_scheme)
        {
            roxterm_change_colour_scheme(roxterm, colour_scheme);
            options_unref(colour_scheme);
        }
        else
        {
            dlg_warning(roxterm_get_toplevel(roxterm),
                    _("Colour scheme '%s' not found"), scheme_name);
        }
    }
}

static void roxterm_shortcuts_selected(GtkCheckMenuItem *mitem,
        MenuTree *mtree)
{
    ROXTermData *roxterm = roxterm_from_menutree(mtree);
    char *scheme_name;
    Options *shortcuts;

    if (!roxterm || multi_win_get_ignore_toggles(roxterm->win))
        return;

    scheme_name = g_object_get_data(G_OBJECT(mitem), PROFILE_NAME_KEY);
    if (!scheme_name)
    {
        g_critical(_("Menu item has no '%s' data"), PROFILE_NAME_KEY);
        return;
    }
    if (!gtk_check_menu_item_get_active(mitem))
    {
        check_preferences_submenu_pair(roxterm,
                MENUTREE_PREFERENCES_SELECT_SHORTCUTS, scheme_name);
    }
    shortcuts = shortcuts_open(scheme_name);
    multi_win_set_shortcut_scheme(roxterm->win, shortcuts);
    shortcuts_unref(shortcuts);
    g_free(scheme_name);
}

static void roxterm_beep_handler(VteTerminal *vte, ROXTermData *roxterm)
{
    if (roxterm->tab != multi_win_get_current_tab(roxterm->win) &&
            options_lookup_int_with_default(roxterm->profile,
                    "bell_highlights_tab", TRUE))
    {
        multi_tab_draw_attention(roxterm->tab);
    }
}

static GtkWidget *create_radio_menu_item(MenuTree *mtree,
        const char *name, GSList **group, GCallback handler)
{
    GtkWidget *mitem;

    mitem = gtk_radio_menu_item_new_with_label(*group, name);
    gtk_widget_show(mitem);
    *group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(mitem));
    if (handler)
    {
        g_object_set_data_full(G_OBJECT(mitem), PROFILE_NAME_KEY,
                g_strdup(name), g_free);
        g_signal_connect(mitem, "activate", handler, mtree);
    }
    return mitem;
}

static GtkMenu *radio_menu_from_strv(char **items,
        GCallback handler, MenuTree *mtree)
{
    int n;
    GtkMenu *menu;
    GSList *group = NULL;

    if (!items || !items[0])
        return NULL;

    menu = GTK_MENU(gtk_menu_new());

    for (n = 0; items[n]; ++n)
    {
        GtkWidget *mitem = create_radio_menu_item(mtree, items[n],
                &group, handler);

        gtk_menu_shell_append(GTK_MENU_SHELL(menu), mitem);
        gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mitem),
                FALSE);
    }
    return menu;
}

static void roxterm_add_pair_of_pref_submenus(MultiWin *win,
        DynamicOptions *family, MenuTreeID id, GCallback handler)
{
    char **items = dynamic_options_list(family);
    GtkMenu *submenu;
    MenuTree *menutree;

    g_return_if_fail(items);
    multi_win_set_ignore_toggles(win, TRUE);
    menutree = multi_win_get_menu_bar(win);
    if (menutree)
    {
        submenu = radio_menu_from_strv(items, handler, menutree);
        gtk_menu_item_set_submenu(
                GTK_MENU_ITEM(menutree_get_widget_for_id(menutree, id)),
                GTK_WIDGET(submenu));
    }
    menutree = multi_win_get_popup_menu(win);
    if (menutree)
    {
        submenu = radio_menu_from_strv(items, handler, menutree);
        gtk_menu_item_set_submenu(
                GTK_MENU_ITEM(menutree_get_widget_for_id(menutree, id)),
                GTK_WIDGET(submenu));
    }
    multi_win_set_ignore_toggles(win, FALSE);
    g_strfreev(items);
}

static void roxterm_add_all_pref_submenus(MultiWin *win)
{
    roxterm_add_pair_of_pref_submenus(win, dynamic_options_get("Profiles"),
            MENUTREE_PREFERENCES_SELECT_PROFILE,
            G_CALLBACK(roxterm_profile_selected));
    roxterm_add_pair_of_pref_submenus(win, dynamic_options_get("Colours"),
            MENUTREE_PREFERENCES_SELECT_COLOUR_SCHEME,
            G_CALLBACK(roxterm_colour_scheme_selected));
    roxterm_add_pair_of_pref_submenus(win, dynamic_options_get("Shortcuts"),
            MENUTREE_PREFERENCES_SELECT_SHORTCUTS,
            G_CALLBACK(roxterm_shortcuts_selected));
}

static void roxterm_connect_menu_signals(MultiWin * win)
{
    multi_win_menu_connect_swapped(win, MENUTREE_EDIT_COPY,
        G_CALLBACK(roxterm_copy_clipboard_action), win, NULL, NULL, NULL);
    multi_win_menu_connect_swapped(win, MENUTREE_EDIT_PASTE,
        G_CALLBACK(roxterm_paste_clipboard_action), win, NULL, NULL, NULL);
    multi_win_menu_connect_swapped(win, MENUTREE_EDIT_COPY_AND_PASTE,
        G_CALLBACK(roxterm_copy_and_paste_action), win, NULL, NULL, NULL);
    multi_win_menu_connect_swapped(win, MENUTREE_EDIT_RESET,
        G_CALLBACK(roxterm_reset_action), win, NULL, NULL, NULL);
    multi_win_menu_connect_swapped(win, MENUTREE_EDIT_RESET_AND_CLEAR,
        G_CALLBACK(roxterm_reset_and_clear_action), win, NULL, NULL, NULL);

    multi_win_menu_connect_swapped(win,
            MENUTREE_PREFERENCES_EDIT_CURRENT_PROFILE,
        G_CALLBACK(roxterm_edit_profile_action), win, NULL, NULL, NULL);
    multi_win_menu_connect_swapped(win,
            MENUTREE_PREFERENCES_EDIT_CURRENT_COLOUR_SCHEME,
        G_CALLBACK(roxterm_edit_colour_scheme_action), win, NULL, NULL, NULL);

    multi_win_menu_connect_swapped(win, MENUTREE_HELP_ABOUT,
        G_CALLBACK(roxterm_show_about), win, NULL, NULL, NULL);
    multi_win_menu_connect_swapped(win, MENUTREE_HELP_SHOW_MANUAL,
        G_CALLBACK(roxterm_show_manual), win, NULL, NULL, NULL);

    multi_win_menu_connect_swapped(win, MENUTREE_PREFERENCES_CONFIG_MANAGER,
        G_CALLBACK(roxterm_open_config_manager), win, NULL, NULL, NULL);

    multi_win_menu_connect_swapped(win, MENUTREE_OPEN_IN_BROWSER,
        G_CALLBACK(roxterm_browser_action), win, NULL, NULL, NULL);
    multi_win_menu_connect_swapped(win, MENUTREE_OPEN_IN_MAILER,
        G_CALLBACK(roxterm_mailto_action), win, NULL, NULL, NULL);
    multi_win_menu_connect_swapped(win, MENUTREE_COPY_URI,
        G_CALLBACK(roxterm_copy_url_action), win, NULL, NULL, NULL);

    roxterm_add_all_pref_submenus(win);
}

static void roxterm_connect_misc_signals(ROXTermData * roxterm)
{
    g_signal_connect(roxterm->widget,
        "child-exited", G_CALLBACK(roxterm_child_exited), roxterm);
    g_signal_connect(roxterm->widget, "popup-menu",
            G_CALLBACK(roxterm_popup_handler), roxterm);
    g_signal_connect(roxterm->widget, "button-press-event",
            G_CALLBACK (roxterm_click_handler), roxterm);
    g_signal_connect(roxterm->widget, "button-release-event",
            G_CALLBACK (roxterm_release_handler), roxterm);
    g_signal_connect(roxterm->widget, "icon-title-changed",
        G_CALLBACK(roxterm_title_handler), roxterm);
    g_signal_connect(roxterm->widget, "window-title-changed",
        G_CALLBACK(roxterm_title_handler), roxterm);
    g_signal_connect(roxterm->widget, "style_set",
        G_CALLBACK(roxterm_style_change_handler), roxterm);
    g_signal_connect(roxterm->widget, "char-size-changed",
        G_CALLBACK(roxterm_char_size_changed), roxterm);
    g_signal_connect(roxterm->widget, "drag-end",
        G_CALLBACK(roxterm_uri_drag_ended), roxterm);
    g_signal_connect(roxterm->widget, "drag-data-get",
        G_CALLBACK(roxterm_uri_drag_data_get), roxterm);
    if (g_signal_lookup("beep",
            G_TYPE_FROM_INSTANCE(G_OBJECT(roxterm->widget))))
    {
        g_signal_connect(roxterm->widget, "beep",
                G_CALLBACK(roxterm_beep_handler), roxterm);
    }
}

inline static void
roxterm_set_select_by_word_chars(ROXTermData * roxterm, VteTerminal * vte)
{
    char *wchars = options_lookup_string_with_default
        (roxterm->profile, "sel_by_word", "-A-Za-z0-9,./?%&#_");

    vte_terminal_set_word_chars(vte, wchars);
    if (wchars)
        g_free(wchars);
}

inline static void
roxterm_update_audible_bell(ROXTermData * roxterm, VteTerminal * vte)
{
    vte_terminal_set_audible_bell(vte, options_lookup_int
        (roxterm->profile, "audible_bell") != 0);
}

inline static void
roxterm_update_visible_bell(ROXTermData * roxterm, VteTerminal * vte)
{
    vte_terminal_set_visible_bell(vte, options_lookup_int
        (roxterm->profile, "visible_bell") == 1);
}

inline static void
roxterm_update_allow_bold(ROXTermData * roxterm, VteTerminal * vte)
{
    vte_terminal_set_allow_bold(vte, options_lookup_int
        (roxterm->profile, "allow_bold") != 0);
}

#ifdef HAVE_VTE_TERMINAL_SET_CURSOR_BLINK_MODE
#define CURSOR_BLINK_OPTION "cursor_blink_mode"
#else
#define CURSOR_BLINK_OPTION "cursor_blinks"
inline
#endif
static void
roxterm_update_cursor_blink_mode(ROXTermData * roxterm, VteTerminal * vte)
{
#ifdef HAVE_VTE_TERMINAL_SET_CURSOR_BLINK_MODE
    int o = options_lookup_int(roxterm->profile, CURSOR_BLINK_OPTION);
    
    if (o == -1)
    {
        o = options_lookup_int(roxterm->profile, "cursor_blinks") + 1;
        if (o)
            o ^= 3;
    }
    vte_terminal_set_cursor_blink_mode(vte, o);
#else
    vte_terminal_set_cursor_blinks(vte, options_lookup_int
        (roxterm->profile, CURSOR_BLINK_OPTION) == 1);
#endif
}

#ifdef HAVE_VTE_TERMINAL_SET_CURSOR_SHAPE
inline static void
roxterm_update_cursor_shape(ROXTermData * roxterm, VteTerminal * vte)
{
    vte_terminal_set_cursor_shape(vte, options_lookup_int_with_default
        (roxterm->profile, "cursor_shape", 0));
}
#endif

inline static void
roxterm_update_mouse_autohide(ROXTermData * roxterm, VteTerminal * vte)
{
    vte_terminal_set_mouse_autohide(vte, options_lookup_int(roxterm->profile,
            "mouse_autohide") != 0);
}

static void roxterm_set_scrollback_lines(ROXTermData * roxterm,
        VteTerminal * vte)
{
    vte_terminal_set_scrollback_lines(vte, options_lookup_int_with_default
        (roxterm->profile, "scrollback_lines", 1000));
}

static void roxterm_set_scroll_on_output(ROXTermData * roxterm,
        VteTerminal * vte)
{
    vte_terminal_set_scroll_on_output(vte, options_lookup_int_with_default
        (roxterm->profile, "scroll_on_output", 0));
}

static void roxterm_set_scroll_on_keystroke(ROXTermData * roxterm,
        VteTerminal * vte)
{
    vte_terminal_set_scroll_on_keystroke(vte, options_lookup_int_with_default
        (roxterm->profile, "scroll_on_keystroke", 0));
}

static void roxterm_set_backspace_binding(ROXTermData * roxterm,
        VteTerminal * vte)
{
    vte_terminal_set_backspace_binding(vte, (VteTerminalEraseBinding)
        options_lookup_int_with_default(roxterm->profile,
            "backspace_binding", VTE_ERASE_AUTO));
}

static void roxterm_set_delete_binding(ROXTermData * roxterm,
        VteTerminal * vte)
{
    vte_terminal_set_delete_binding(vte, (VteTerminalEraseBinding)
        options_lookup_int_with_default(roxterm->profile,
            "delete_binding", VTE_ERASE_AUTO));
}

inline static void roxterm_apply_wrap_switch_tab(ROXTermData *roxterm)
{
    multi_win_set_wrap_switch_tab(roxterm->win,
        options_lookup_int_with_default(roxterm->profile,
            "wrap_switch_tab", FALSE));
}

static void roxterm_apply_disable_menu_access(ROXTermData *roxterm)
{
    static char *orig_menu_access = NULL;

    if (!orig_menu_access)
    {
        g_object_get(G_OBJECT(gtk_settings_get_default()),
                "gtk-menu-bar-accel", &orig_menu_access, NULL);
    }
    gtk_settings_set_string_property(gtk_settings_get_default(),
            "gtk-menu-bar-accel",
            options_lookup_int_with_default(roxterm->profile,
                    "disable_menu_access", FALSE) ?
                "<Shift><Control><Mod1><Mod2><Mod3><Mod4><Mod5>F10" :
                orig_menu_access,
            "roxterm");
}

static void roxterm_apply_title_template(ROXTermData *roxterm)
{
    char *tt = NULL;
    
    if (roxterm->title_template)
    {
        tt = g_strdup(roxterm->title_template);
    }
    else
    {
        tt = options_lookup_string_with_default(roxterm->profile,
            "title_string", "%s");
    }
    multi_win_set_title_template(roxterm->win, tt, TRUE);
    g_free(tt);
}
    
/*
static void roxterm_apply_emulation(ROXTermData *roxterm, VteTerminal *vte)
{
    char *emu = options_lookup_string(roxterm->profile, "emulation");

    vte_terminal_set_emulation(vte, emu);
    g_free(emu);
}
*/

static void roxterm_apply_profile(ROXTermData *roxterm, VteTerminal *vte)
{
    roxterm_set_select_by_word_chars(roxterm, vte);
    roxterm_update_audible_bell(roxterm, vte);
    roxterm_update_visible_bell(roxterm, vte);
    roxterm_update_allow_bold(roxterm, vte);
    roxterm_update_cursor_blink_mode(roxterm, vte);
#ifdef HAVE_VTE_TERMINAL_SET_CURSOR_SHAPE
    roxterm_update_cursor_shape(roxterm, vte);
#endif

    roxterm_apply_colour_scheme(roxterm, vte);
    roxterm_update_background(roxterm, vte);

    roxterm_update_font(roxterm, vte, FALSE);

    roxterm_set_scrollback_lines(roxterm, vte);
    roxterm_set_scroll_on_output(roxterm, vte);
    roxterm_set_scroll_on_keystroke(roxterm, vte);

    roxterm_set_backspace_binding(roxterm, vte);
    roxterm_set_delete_binding(roxterm, vte);

    roxterm_update_mouse_autohide(roxterm, vte);
    roxterm_apply_encoding(roxterm, vte);
    /*roxterm_apply_emulation(roxterm, vte);*/

    roxterm_apply_wrap_switch_tab(roxterm);

    roxterm_apply_disable_menu_access(roxterm);
    
    roxterm_apply_title_template(roxterm);
}

static gboolean
roxterm_drag_data_received(GtkWidget *widget,
        const char *text, gulong len, gpointer data)
{
    char *sep = strstr(text, "\r\n");
    char *rejoined = NULL;

    if (sep)
    {
        char **uris = g_strsplit(text, "\r\n", 0);
        int n;
        char *rejoined = NULL;
        char *tmp;

        for (n = 0; uris[n]; ++n)
        {
            /* Last uri is usually a blank line so skip */
            if (!uris[n][0])
            {
                continue;
            }
            /* escape ' chars */
            if (strchr(uris[n], '\''))
            {
                char **split = g_strsplit(uris[n], "'", 0);

                tmp = g_strjoinv("'\\''", split);
                g_free(uris[n]);
                uris[n] = tmp;
            }
            tmp = g_strdup_printf("%s%s%s", rejoined ? rejoined : "",
                    rejoined ? "' '" : "'", uris[n]);
            g_free(rejoined);
            rejoined = tmp;
        }
        if (rejoined && rejoined[0])
        {
            tmp = g_strdup_printf("%s' ", rejoined);
            g_free(rejoined);
            rejoined = tmp;
        }
        text = rejoined;
        if (text)
            len = strlen(text);
        g_strfreev(uris);
    }
    if (text)
        vte_terminal_feed_child(VTE_TERMINAL(widget), text, len);
    if (rejoined)
        g_free(rejoined);
    return TRUE;
}

static GtkWidget *roxterm_multi_tab_filler(MultiWin * win, MultiTab * tab,
    ROXTermData * roxterm_template, ROXTermData ** roxterm_out,
    char **title, GtkWidget ** vte_widget, GtkAdjustment **adjustment)
{
    VteTerminal *vte;
    const char *title_orig;
    ROXTermData *roxterm = roxterm_data_clone(roxterm_template);
    int hide_menu_bar = global_options_lookup_int("hide_menubar");
    MultiWinScrollBar_Position scrollbar_pos;
    char *tab_name;

    if ((!roxterm_template->win || roxterm_template->win == win)
        && roxterm_template->title_template)
    {
        roxterm->title_template = g_strdup(roxterm_template->title_template);
    }
    roxterm_terms = g_list_append(roxterm_terms, roxterm);

    if (hide_menu_bar == -1)
    {
        hide_menu_bar = options_lookup_int(roxterm_template->profile,
            "hide_menubar") == 1;
    }
    multi_win_set_show_menu_bar(win, !hide_menu_bar, FALSE);

    roxterm->win = win;
    roxterm->tab = tab;
    *roxterm_out = roxterm;

    roxterm->widget = vte_terminal_new();
    gtk_widget_grab_focus(roxterm->widget);
    vte = VTE_TERMINAL(roxterm->widget);
    if (vte_widget)
        *vte_widget = roxterm->widget;
    if (adjustment)
        *adjustment = vte->adjustment;

    scrollbar_pos = multi_win_set_scroll_bar_position(win,
        options_lookup_int_with_default(roxterm_template->profile,
            "scrollbar_pos", MultiWinScrollBar_Right));
    if (scrollbar_pos)
    {
        roxterm->hbox = gtk_hbox_new(FALSE, 0);
        roxterm->scrollbar = gtk_vscrollbar_new(vte->adjustment);
        if (scrollbar_pos == MultiWinScrollBar_Left)
        {
            gtk_box_pack_end_defaults(GTK_BOX(roxterm->hbox), roxterm->widget);
            gtk_box_pack_end(GTK_BOX(roxterm->hbox), roxterm->scrollbar,
                FALSE, FALSE, 0);
        }
        else
        {
            gtk_box_pack_start_defaults(GTK_BOX(roxterm->hbox),
                roxterm->widget);
            gtk_box_pack_start(GTK_BOX(roxterm->hbox), roxterm->scrollbar,
                FALSE, FALSE, 0);
        }
        gtk_widget_show_all(roxterm->hbox);
    }
    else
    {
        gtk_widget_show(roxterm->widget);
    }

    roxterm_add_matches(roxterm, vte);

    roxterm_apply_profile(roxterm, vte);
    tab_name = global_options_lookup_string("tab-name");
    if (tab_name)
    {
        multi_tab_set_name(tab, tab_name);
        options_set_string(global_options, "tab-name", NULL);
        g_free(tab_name);
    }

    /* Used to call vte_terminal_set_size() here, but doing so before parent
     * window is shown seems to mess up scrollbar in libvte9, and everything
     * works OK without */

    title_orig = vte_terminal_get_window_title(vte);
    *title = g_strdup(title_orig ? title_orig : _("ROXTerm"));

    roxterm_connect_misc_signals(roxterm);
    multi_tab_connect_tab_selection_handler(win,
        (MultiTabSelectionHandler) roxterm_tab_selection_handler);
    roxterm->drd = drag_receive_setup_dest_widget(roxterm->widget,
            roxterm_drag_data_received, roxterm);

    return scrollbar_pos ? roxterm->hbox : roxterm->widget;
}

static void roxterm_multi_tab_destructor(ROXTermData * roxterm)
{
    roxterm_terms = g_list_remove(roxterm_terms, roxterm);
    roxterm_data_delete(roxterm);
}

/* Returns FALSE if the value hasn't really changed */
static gboolean
roxterm_update_option(Options * opts, const char *key,
    OptsDBusOptType opt_type, OptsDBusValue val)
{
    gboolean result = TRUE;
    char *oldval;

    switch (opt_type)
    {
        case OptsDBus_StringOpt:
            oldval = options_lookup_string(opts, key);
            if (options_strcmp(oldval, val.s))
                options_set_string(opts, key, val.s);
            else
                result = FALSE;
            if (oldval)
                g_free(oldval);
            return result;
        case OptsDBus_IntOpt:
            if (options_lookup_int(opts, key) == val.i)
                return FALSE;
            options_set_int(opts, key, val.i);
            return TRUE;
        case OptsDBus_FloatOpt:
            if (options_lookup_double(opts, key) == val.f)
                return FALSE;
            options_set_double(opts, key, val.f);
            return TRUE;
        default:
            g_critical(_("Unknown option type (%d)"), opt_type);
    }
    return FALSE;
}

static void roxterm_reflect_profile_change(Options * profile, const char *key)
{
    GList *link;

    for (link = roxterm_terms; link; link = g_list_next(link))
    {
        ROXTermData *roxterm = link->data;
        VteTerminal *vte;
        gboolean apply_to_win = FALSE;

        if (roxterm->profile != profile || roxterm->profile->deleted)
            continue;

        vte = VTE_TERMINAL(roxterm->widget);
        if (!strcmp(key, "font"))
        {
            roxterm_update_font(roxterm, vte, TRUE);
            apply_to_win = TRUE;
        }
        else if (!strcmp(key, "allow_bold"))
        {
            roxterm_update_allow_bold(roxterm, vte);
        }
        else if (!strcmp(key, "hide_menubar") &&
            multi_win_get_current_tab(roxterm->win) == roxterm->tab)
        {
            multi_win_set_show_menu_bar(roxterm->win,
                !options_lookup_int(roxterm->profile, "hide_menubar"), TRUE);
        }
        else if (!strcmp(key, "audible_bell"))
        {
            roxterm_update_audible_bell(roxterm, vte);
        }
        else if (!strcmp(key, "visible_bell"))
        {
            roxterm_update_visible_bell(roxterm, vte);
        }
        else if (!strcmp(key, CURSOR_BLINK_OPTION))
        {
            roxterm_update_cursor_blink_mode(roxterm, vte);
        }
#ifdef HAVE_VTE_TERMINAL_SET_CURSOR_SHAPE
        else if (!strcmp(key, "cursor_shape"))
        {
            roxterm_update_cursor_shape(roxterm, vte);
        }
#endif
        else if (!strcmp(key, "mouse_autohide"))
        {
            roxterm_update_mouse_autohide(roxterm, vte);
        }
        else if (!strcmp(key, "sel_by_word"))
        {
            roxterm_set_select_by_word_chars(roxterm, vte);
        }
        /*
        else if (!strcmp(key, "emulation"))
        {
            roxterm_apply_emulation(roxterm, vte);
        }
        */
        else if (!strcmp(key, "width") || !strcmp(key, "height"))
        {
            roxterm_update_size(roxterm, vte);
            apply_to_win = TRUE;
        }
        else if (!strcmp(key, "background_img")
                || !strcmp(key, "background_type")
                || !strcmp(key, "scroll_background")
                || !strcmp(key, "saturation")
                || !strcmp(key, "true_translucence"))
        {
            roxterm_update_background(roxterm, vte);
        }
        else if (!strcmp(key, "scrollback_lines"))
        {
            roxterm_set_scrollback_lines(roxterm, vte);
        }
        else if (!strcmp(key, "scroll_on_output"))
        {
            roxterm_set_scroll_on_output(roxterm, vte);
        }
        else if (!strcmp(key, "scroll_on_keystroke"))
        {
            roxterm_set_scroll_on_keystroke(roxterm, vte);
        }
        else if (!strcmp(key, "backspace_binding"))
        {
            roxterm_set_backspace_binding(roxterm, vte);
        }
        else if (!strcmp(key, "delete_binding"))
        {
            roxterm_set_delete_binding(roxterm, vte);
        }
        else if (!strcmp(key, "encoding"))
        {
            char *encoding =
                options_lookup_string(roxterm->profile, "encoding");

            vte_terminal_set_encoding(vte, encoding);
            if (encoding)
                g_free(encoding);
        }
        else if (!strcmp(key, "wrap_switch_tab"))
        {
            roxterm_apply_wrap_switch_tab(roxterm);
        }
        else if (!strcmp(key, "disable_menu_access"))
        {
            roxterm_apply_disable_menu_access(roxterm);
        }
        else if (!strcmp(key, "disable_menu_shortcuts"))
        {
            gboolean disable = options_lookup_int(roxterm->profile,
                    "disable_menu_shortcuts");
            MenuTree *mtree = multi_win_get_popup_menu(roxterm->win);

            menutree_disable_shortcuts(mtree, disable);
            mtree = multi_win_get_menu_bar(roxterm->win);
            menutree_disable_shortcuts(mtree, disable);
            mtree = multi_win_get_short_popup_menu(roxterm->win);
            menutree_disable_shortcuts(mtree, disable);
        }
        else if (!strcmp(key, "disable_tab_menu_shortcuts"))
        {
            gboolean disable = options_lookup_int(roxterm->profile,
                    "disable_tab_menu_shortcuts");
            MenuTree *mtree = multi_win_get_popup_menu(roxterm->win);

            menutree_disable_tab_shortcuts(mtree, disable);
            mtree = multi_win_get_menu_bar(roxterm->win);
            menutree_disable_tab_shortcuts(mtree, disable);
        }
        else if (!strcmp(key, "title_string"))
        {
            g_free(roxterm->title_template);
            roxterm->title_template = NULL;
            roxterm_apply_title_template(roxterm);
        }
        if (apply_to_win)
        {
            multi_win_foreach_tab(roxterm->win,
                    match_text_size_foreach_tab, roxterm);
        }
    }
}

static gboolean roxterm_update_colour_option(Options *scheme, const char *key,
        const char *value)
{
    void (*setter)(Options *, const char *) = NULL;
    GdkColor *old_colour;
    GdkColor *pnew_colour = NULL;
    GdkColor  new_colour;

    if (value)
    {
        g_return_val_if_fail(gdk_color_parse(value, &new_colour), FALSE);
        pnew_colour = &new_colour;
    }
    if (!strcmp(key, "foreground"))
    {
        old_colour = colour_scheme_get_foreground_colour(scheme, TRUE);
        setter = colour_scheme_set_foreground_colour;
    }
    else if (!strcmp(key, "background"))
    {
        old_colour = colour_scheme_get_background_colour(scheme, TRUE);
        setter = colour_scheme_set_background_colour;
    }
    else if (!strcmp(key, "cursor"))
    {
        old_colour = colour_scheme_get_cursor_colour(scheme, TRUE);
        setter = colour_scheme_set_cursor_colour;
    }
    else
    {
        old_colour = colour_scheme_get_palette(scheme) + atoi(key);
    }
    if (!old_colour && !pnew_colour)
        return FALSE;
    if (old_colour && pnew_colour && gdk_color_equal(old_colour, pnew_colour))
        return FALSE;
    if (setter)
        setter(scheme, value);
    else
        colour_scheme_set_palette_entry(scheme, atoi(key), value);
    return TRUE;
}

static gboolean roxterm_update_palette_size(Options *scheme, int size)
{
    if (colour_scheme_get_palette_size(scheme) == size)
        return FALSE;
    colour_scheme_set_palette_size(scheme, size);
    return TRUE;
}

static void roxterm_reflect_colour_change(Options *scheme, const char *key)
{
    GList *link;

    for (link = roxterm_terms; link; link = g_list_next(link))
    {
        ROXTermData *roxterm = link->data;
        VteTerminal *vte;

        if (roxterm->colour_scheme != scheme || roxterm->colour_scheme->deleted)
            continue;

        vte = VTE_TERMINAL(roxterm->widget);

        if (!strcmp(key, "cursor"))
            roxterm_update_cursor_colour(roxterm, vte);
        else
            roxterm_apply_colour_scheme(roxterm, vte);
    }
}

static void
roxterm_opt_signal_handler(const char *profile_name, const char *key,
    OptsDBusOptType opt_type, OptsDBusValue val)
{
    if (!strncmp(profile_name, "Profiles/", 9))
    {
        const char *short_profile_name = profile_name + 9;
        Options *profile = dynamic_options_lookup_and_ref(roxterm_profiles,
            short_profile_name, "roxterm profile");

        if (roxterm_update_option(profile, key, opt_type, val))
            roxterm_reflect_profile_change(profile, key);
        dynamic_options_unref(roxterm_profiles, short_profile_name);
    }
    else if (!strncmp(profile_name, "Colours/", 8))
    {
        const char *scheme_name = profile_name + 8;
        Options *scheme = colour_scheme_lookup_and_ref(scheme_name);
        gboolean changed = FALSE;

        if (!strcmp(key, "palette_size"))
            changed = roxterm_update_palette_size(scheme, val.i);
        else
            changed = roxterm_update_colour_option(scheme, key, val.s);
        if (changed)
            roxterm_reflect_colour_change(scheme, key);
        colour_scheme_unref(scheme);
    }
    else if (!strcmp(profile_name, "Global") && !strcmp(key, "warn_close"))
    {
        options_set_int(global_options, "warn_close", val.i);
    }
    else
    {
        g_critical(_("Profile/key '%s/%s' not understood"), profile_name, key);
    }
}

/* data is cast to char const **pname; if the deleted item is currently
 * selected *pname is changed to NULL */
static void delete_name_from_menu(GtkWidget *widget, gpointer data)
{
    const char *profile_name = g_object_get_data(G_OBJECT(widget),
            PROFILE_NAME_KEY);
    char const **pname = data;

    g_return_if_fail(profile_name);
    /* pname may have been NULLed by an earlier call */
    if (!*pname)
        return;
    if (!strcmp(profile_name, *pname))
    {
        if (gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(widget)))
            *pname = NULL;
        gtk_widget_destroy(widget);
    }
}

/* Casts data to GtkWidget ** and sets target to found widget if it finds an
 * entry with no PROFILE_NAME_KEY */
static void find_unhandled_menu_entry(GtkWidget *widget, gpointer data)
{
    if (!g_object_get_data(G_OBJECT(widget), PROFILE_NAME_KEY))
        *((GtkWidget **) data) = widget;
}

static GtkWidget *append_name_to_menu(MenuTree *mtree, GtkMenu *menu,
        const char *name, GCallback handler)
{
    GSList *group = NULL;
    GtkWidget *mitem;

    gtk_container_foreach(GTK_CONTAINER(menu),
            menutree_find_radio_group, &group);
    mitem = create_radio_menu_item(mtree, name, &group, handler);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mitem);
    return mitem;
}

/* Sets *pname to NULL if it's currently selected */
inline static void remove_name_from_menu(GtkMenu *menu, char const **pname)
{
    gtk_container_foreach(GTK_CONTAINER(menu),
            delete_name_from_menu, (gpointer) pname);
}

static void stuff_changed_do_menu(MenuTree *mtree,
        const char *what_happened, const char *family_name,
        const char *current_name, const char *new_name)
{
    MenuTreeID id;
    GtkMenu *menu;
    GCallback handler = NULL;

    if (!strcmp(family_name, "Profiles"))
    {
        id = MENUTREE_PREFERENCES_SELECT_PROFILE;
        handler = G_CALLBACK(roxterm_profile_selected);
    }
    else if (!strcmp(family_name, "Colours"))
    {
        id = MENUTREE_PREFERENCES_SELECT_COLOUR_SCHEME;
        handler = G_CALLBACK(roxterm_colour_scheme_selected);
    }
    else if (!strcmp(family_name, "Shortcuts"))
    {
        id = MENUTREE_PREFERENCES_SELECT_SHORTCUTS;
        handler = G_CALLBACK(roxterm_shortcuts_selected);
    }
    else if (!strcmp(family_name, "encodings"))
    {
        roxterm_encodings_changed(mtree, what_happened, current_name, new_name);
        return;
    }
    else
    {
        g_warning(_("Changed options family '%s' not known"), family_name);
        return;
    }
    g_return_if_fail((menu = menutree_submenu_from_id(mtree, id)) != NULL);

    if (!strcmp(what_happened, OPTSDBUS_DELETED))
    {
        GtkWidget *unhandled_mitem = NULL;

        remove_name_from_menu(menu, &current_name);
        gtk_container_foreach(GTK_CONTAINER(menu),
                find_unhandled_menu_entry, &unhandled_mitem);
        if (!unhandled_mitem)
        {
            unhandled_mitem = append_name_to_menu(mtree, menu,
                    _("[Deleted]"), NULL);
            gtk_widget_set_sensitive(unhandled_mitem, FALSE);
        }
        /* If deleted item was selected, select unhandled item in its place
         */
        gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(unhandled_mitem),
                current_name == NULL);
    }
    else if (!strcmp(what_happened, OPTSDBUS_ADDED))
    {
        append_name_to_menu(mtree, menu, current_name, handler);
    }
    else if (!strcmp(what_happened, OPTSDBUS_RENAMED))
    {
        GtkWidget *mitem;

        remove_name_from_menu(menu, &current_name);
        mitem = append_name_to_menu(mtree, menu, new_name, handler);
        gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mitem),
                current_name == NULL);
    }
    else
    {
        g_warning(_("Options family change action '%s' not known"),
                what_happened);
    }
}

static void roxterm_stuff_changed_handler(const char *what_happened,
        const char *family_name, const char *current_name,
        const char *new_name)
{
    GList *link;
    DynamicOptions *dynopts = NULL;
    Options *options = NULL;

    if (strcmp(family_name, "encodings"))
    {
        dynopts = dynamic_options_get(family_name);
        options = dynamic_options_lookup(dynopts, current_name);
    }

    if (!strcmp(what_happened, OPTSDBUS_DELETED))
    {
        if (options)
        {
            dynamic_options_forget(dynopts, current_name);
            options->deleted = TRUE;
        }
    }
    else if (!strcmp(what_happened, OPTSDBUS_RENAMED))
    {
        if (options)
        {
            options_change_leafname(options, new_name);
            dynamic_options_rename(dynopts, current_name, new_name);
        }
    }

    for (link = multi_win_all; link; link = g_list_next(link))
    {
        MultiWin *win = (MultiWin *) link->data;
        MenuTree *mtree = multi_win_get_menu_bar(win);

        multi_win_set_ignore_toggles(win, TRUE);
        if (mtree)
        {
            stuff_changed_do_menu(mtree, what_happened, family_name,
                    current_name, new_name);
        }
        mtree = multi_win_get_popup_menu(win);
        if (mtree)
        {
            stuff_changed_do_menu(mtree, what_happened, family_name,
                    current_name, new_name);
        }
        multi_win_set_ignore_toggles(win, FALSE);
    }
}

static gboolean roxterm_verify_id(ROXTermData *roxterm)
{
    GList *link;
    
    for (link = roxterm_terms; link; link = g_list_next(link))
    {
        if (link->data == roxterm)
            return TRUE;
    }
    g_warning(_("Invalid ROXTERM_ID %p in D-Bus message "
            "(this is expected if you used roxterm's --separate option)"),
            roxterm);
    return FALSE;
}

static void roxterm_set_profile_handler(ROXTermData *roxterm, const char *name)
{
    if (!roxterm_verify_id(roxterm))
        return;
        
    Options *profile = dynamic_options_lookup_and_ref(roxterm_profiles, name,
            "roxterm profile");
    
    if (profile)
    {
        roxterm_change_profile(roxterm, profile);
        multi_win_foreach_tab(roxterm->win,
                match_text_size_foreach_tab, roxterm);
        options_unref(profile);
    }
    else
    {
        dlg_warning(roxterm_get_toplevel(roxterm),
                _("Unknown profile '%s' in D-Bus signal"), name);
    }
}

static void roxterm_set_colour_scheme_handler(ROXTermData *roxterm,
        const char *name)
{
    if (!roxterm_verify_id(roxterm))
        return;
        
    Options *scheme = colour_scheme_lookup_and_ref(name);
    
    if (scheme)
    {
        roxterm_change_colour_scheme(roxterm, scheme);
        colour_scheme_unref(scheme);
    }
    else
    {
        dlg_warning(roxterm_get_toplevel(roxterm),
                _("Unknown colour scheme '%s' in D-Bus signal"), name);
    }
}

static void roxterm_set_shortcut_scheme_handler(ROXTermData *roxterm,
        const char *name)
{
    if (!roxterm_verify_id(roxterm))
        return;
        
    Options *shortcuts = shortcuts_open(name);
    multi_win_set_shortcut_scheme(roxterm->win, shortcuts);
    shortcuts_unref(shortcuts);
}

/* Parses a number from a geometry string which must be terminated by NULL or
 * one of chars from trmntrs immediately after digits. Returns pointer to that
 * terminator or NULL if invalid. */
static char *parse_geom_n(char *g, int *n, const char *trmntrs)
{
    char *g2 = g;
    size_t l;

    if (*g == '-' || *g == '+')
        ++g2;
    if (!isdigit(*g2))
        return NULL;
    l = strspn(g2, "0123456789");
    /* Check we've reached end of string or one of trmntrs */
    if (g2[l] && (!trmntrs || !strchr(trmntrs, g2[l])))
        return NULL;
    if (n && sscanf(g, "%d", n) != 1)
        return NULL;
    return g2 + l;
}

/* This just checks whether the rest of the string is a valid pair of
 * geometry offsets */
static gboolean check_geom_offsets(char *g)
{
    if ((g = parse_geom_n(g, NULL, "+-")) == NULL)
    {
        return FALSE;
    }
    if ((*g != '+' && *g != '-')
        || (g = parse_geom_n(g, NULL, NULL)) == NULL)
    {
        return FALSE;
    }
    return TRUE;
}

static GtkPositionType roxterm_get_tab_pos(const ROXTermData *roxterm)
{
    switch (options_lookup_int_with_default(roxterm->profile, "tab_pos", 0))
    {
        case 0:
            return GTK_POS_TOP;
        case 1:
            return GTK_POS_BOTTOM;
        case 2:
            return GTK_POS_LEFT;
        case 3:
            return GTK_POS_RIGHT;
        case 4:
            return -1;
        default:
            return GTK_POS_TOP;
    }
}

static void roxterm_get_initial_tabs(ROXTermData *roxterm,
        GtkPositionType *p_pos, int *p_num)
{
    *p_pos = roxterm_get_tab_pos(roxterm);
    *p_num = options_lookup_int_with_default(roxterm->profile, "init_tabs", 1);
}

void roxterm_launch(void)
{
    ROXTermData *roxterm = g_new0(ROXTermData, 1);
    GtkPositionType tabpos;
    char *colour_scheme_name;
    char *shortcut_scheme;
    char *profile_name;
    Options *shortcuts;
    char *geom;
    gboolean size_on_cli = FALSE;
    int numtabs;

    roxterm->title_template = global_options_lookup_string("title");
    roxterm->target_zoom_factor = global_options_lookup_double("zoom");
    if (!roxterm->target_zoom_factor)
    {
        roxterm->target_zoom_factor = 1.0;
        roxterm->zoom_index = -1;
    }
    else
    {
        roxterm->zoom_index = multi_win_get_nearest_index_for_zoom(
                roxterm->target_zoom_factor);
    }
    if (global_options_commandv)
        roxterm->commandv = global_options_copy_strv(global_options_commandv);
    if (global_options_directory)
        roxterm->directory = g_strdup(global_options_directory);
    profile_name = global_options_lookup_string_with_default("profile",
            "Default");
    if (!roxterm_profiles)
        roxterm_profiles = dynamic_options_get("Profiles");
    roxterm->profile = dynamic_options_lookup_and_ref(roxterm_profiles,
        profile_name, "roxterm profile");

    roxterm_default_dimensions(roxterm);
    roxterm_get_initial_tabs(roxterm, &tabpos, &numtabs);
    geom = options_lookup_string(global_options, "geometry");
    if (geom)
    {
        /* Can't rely on gtk_window_parse_geometry alone because we want to
         * know width and height before creating the window */
        char *g = geom;

        if (*g == '+' || *g == '-')
        {
            /* Offsets first (and only); we don't actually want to know them
             * here; just check validity of string */
            if (!check_geom_offsets(g))
            {
                g_free(geom);
                geom = NULL;
            }
        }
        else
        {
            int width, height;

            g = parse_geom_n(g, &width, "x");
            if (g && *g == 'x')
            {
                ++g;
                g = parse_geom_n(g, &height, "+-");
                if (*g && !check_geom_offsets(g))
                {
                    g_free(geom);
                    geom = NULL;
                }
            }
            else
            {
                g_free(geom);
                geom = NULL;
            }
            if (geom)
            {
                roxterm_set_geometry(roxterm, width, height);
                size_on_cli = TRUE;
            }
        }
        if (!geom)
        {
            dlg_warning(roxterm_get_toplevel(roxterm),
                    _("Invalid geometry specification"));
        }
    }

    colour_scheme_name = global_options_lookup_string_with_default(
            "colour_scheme", "Default");
    if (colour_scheme_name)
    {
        roxterm->colour_scheme = colour_scheme_lookup_and_ref
            (colour_scheme_name);
    }

    shortcut_scheme = global_options_lookup_string_with_default
        ("shortcut_scheme", "Default");
    shortcuts = shortcuts_open(shortcut_scheme);

    roxterm->encoding = global_options_lookup_string("encoding");
    if (roxterm->encoding && !strcmp(roxterm->encoding, "Default"))
    {
        g_free(roxterm->encoding);
        roxterm->encoding = NULL;
    }

    roxterm->pid = -1;
    
    if (global_options_tab)
    {
        GList *link;
        
        /* global_options_tab is now used to indicate whether we found a
         * suitable window */
        global_options_tab = FALSE;
        for (link = multi_win_all; link; link = g_list_next(link))
        {
            MultiWin *win = link->data;
            ROXTermData *partner = multi_win_get_user_data_for_current_tab(win);
            
            if (partner->profile == roxterm->profile)
            {
                global_options_tab = TRUE;
                roxterm->target_zoom_factor = partner->target_zoom_factor;
                roxterm->zoom_index = partner->zoom_index;
                roxterm_set_geometry(roxterm,
                        partner->geom_width, partner->geom_height);
                multi_tab_new(win, roxterm);
                break;
            }
        }
    }

    if (global_options_tab)
    {
        global_options_tab = FALSE;
    }
    else if (global_options_fullscreen)
    {
        global_options_fullscreen = FALSE;
        multi_win_new_fullscreen(shortcuts, roxterm->zoom_index, roxterm,
                numtabs, tabpos);
    }
    else
    {
        if (!size_on_cli)
        {
            if (geom)
            {
                char *old_geom = geom;

                geom = g_strdup_printf("%dx%d%s",
                        roxterm->geom_width, roxterm->geom_height, old_geom);
                g_free(old_geom);
            }
            else
            {
                geom = g_strdup_printf("%dx%d",
                        roxterm->geom_width, roxterm->geom_height);
            }
        }
        multi_win_new_with_geom(shortcuts, roxterm->zoom_index, roxterm, geom,
                numtabs, tabpos);
    }
    g_free(geom);

    roxterm_data_delete(roxterm);
    g_free(shortcut_scheme);
    g_free(colour_scheme_name);
    g_free(profile_name);
}

static void roxterm_tab_to_new_window(MultiWin *win, MultiTab *tab)
{
    ROXTermData *roxterm;
    ROXTermData *match_roxterm;
    MultiTab *match_tab = multi_win_get_current_tab(win);

    roxterm = multi_tab_get_user_data(tab);
    roxterm->win = win;
    g_return_if_fail(match_tab);
    match_roxterm = multi_tab_get_user_data(match_tab);
    if (roxterm == match_roxterm)
    {
        return;
    }
    roxterm_match_text_size(roxterm, match_roxterm);
}

static void roxterm_get_disable_menu_shortcuts(ROXTermData *roxterm,
    gboolean *general, gboolean *tabs)
{
    if (general)
    {
        *general = options_lookup_int_with_default(roxterm->profile,
            "disable_menu_shortcuts", FALSE);
    }
    if (tabs)
    {
        *tabs = options_lookup_int_with_default(roxterm->profile,
            "disable_tab_menu_shortcuts", FALSE);
    }
}

typedef struct {
    guint ntabs;
    int warn;
} DontShowData;

static void dont_show_again_toggled(GtkToggleButton *button, DontShowData *d)
{
    int val = 0;
    
    if (gtk_toggle_button_get_active(button))
    {
        if (d->ntabs > 1)
            val = 0;
        else
            val = 1;
    }
    else
    {
        val = d->warn;
    }
    options_set_int(global_options, "warn_close", val);
    options_file_save(global_options->kf, "Global");
}

static gboolean roxterm_win_delete_handler(GtkWindow *gtkwin, GdkEvent *event,
        MultiWin * win)
{
    GtkWidget *dialog;
    GtkWidget *noshow;
    gboolean response;
    DontShowData d;
    
    (void) event;
    
    d.warn = options_lookup_int_with_default(global_options, "warn_close", 2);
    d.ntabs = multi_win_get_ntabs(win);
    if (!d.warn || (d.warn == 1 && d.ntabs <= 1))
        return FALSE;
        
    dialog = gtk_message_dialog_new(gtkwin,
            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
            GTK_MESSAGE_WARNING, GTK_BUTTONS_OK_CANCEL,
            d.ntabs <= 1 ?
                _("You are about to close a ROXTerm window; this may cause "
                    "loss of data. Are you sure you want to continue?") :
                _("You are about to close a window containing multiple "
                    "ROXTerm tabs; this may cause loss of data. Are you sure "
                    "you want to continue?"));
    gtk_window_set_title(GTK_WINDOW(dialog), _("ROXTerm: Confirm close"));
    noshow = gtk_check_button_new_with_mnemonic(_("_Don't show this again"));
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(noshow), FALSE);
    g_signal_connect(noshow, "toggled",
            G_CALLBACK(dont_show_again_toggled), &d);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), noshow,
            FALSE, FALSE, 0);
    gtk_widget_show(noshow);
    response = gtk_dialog_run(GTK_DIALOG(dialog)) != GTK_RESPONSE_OK;
    gtk_widget_destroy(dialog);
    return response;
}

void roxterm_init(void)
{
    char *logo_filename = logo_find();
    
    gtk_window_set_default_icon_from_file(logo_filename, NULL);
    g_free(logo_filename);

    roxterm_encodings = encodings_load();
    
    optsdbus_listen_for_opt_signals(roxterm_opt_signal_handler);
    optsdbus_listen_for_stuff_changed_signals(roxterm_stuff_changed_handler);
    optsdbus_listen_for_set_profile_signals(
            (OptsDBusSetProfileHandler) roxterm_set_profile_handler);
    optsdbus_listen_for_set_colour_scheme_signals(
            (OptsDBusSetProfileHandler) roxterm_set_colour_scheme_handler);
    optsdbus_listen_for_set_shortcut_scheme_signals(
            (OptsDBusSetProfileHandler) roxterm_set_shortcut_scheme_handler);

    multi_tab_init((MultiTabFiller) roxterm_multi_tab_filler,
        (MultiTabDestructor) roxterm_multi_tab_destructor,
        roxterm_connect_menu_signals,
        (MultiWinGeometryFunc) roxterm_geometry_func,
        (MultiWinSizeFunc) roxterm_size_func,
        roxterm_tab_to_new_window,
        (MultiWinZoomHandler) roxterm_set_zoom_factor,
        (MultiWinGetDisableMenuShortcuts) roxterm_get_disable_menu_shortcuts,
        (MultiWinInitialTabs) roxterm_get_initial_tabs,
        (MultiWinDeleteHandler) roxterm_win_delete_handler);
    roxterm_launch();
}

void roxterm_spawn(ROXTermData *roxterm, const char *command,
    ROXTerm_SpawnType spawn_type)
{
    GError *error = NULL;
    GtkPositionType tabpos;
    int numtabs;

    roxterm_get_initial_tabs(roxterm, &tabpos, &numtabs);
    switch (spawn_type)
    {
        case ROXTerm_SpawnNewWindow:
            roxterm->special_command = g_strdup(command);
            roxterm->no_respawn = TRUE;
            multi_win_new(multi_win_get_shortcut_scheme(roxterm->win),
                      roxterm->zoom_index, roxterm, numtabs, tabpos);
            break;
        case ROXTerm_SpawnNewTab:
            roxterm->special_command = g_strdup(command);
            roxterm->no_respawn = TRUE;
            multi_tab_new(roxterm->win, roxterm);
            break;
        default:
            g_spawn_command_line_async(command, &error);
            if (error)
            {
                dlg_warning(roxterm_get_toplevel(roxterm),
                        _("Unable to spawn command %s: %s"),
                    command, error->message);
                g_error_free(error);
            }
            break;
    }
}

/* vi:set sw=4 ts=4 noet cindent cino= */
