/* 
 * ------------------------------------------------------------------
 * Role PlayingDB V2.0 by Deepwoods Software
 * ------------------------------------------------------------------
 * Spells.h - Spell classes
 * Created by Robert Heller on Thu Jul 30 09:35:39 1998
 * ------------------------------------------------------------------
 * Modification History: 
 * $Log: Spells.h,v $
 * Revision 1.3  1999/07/13 00:29:40  heller
 * Documentation updates (spelling, punctionation, spelling, caps/nocaps).
 *
 * Revision 1.2  1998/12/27 21:48:19  heller
 * Spelling errors fixed.
 *
 * Revision 1.1  1998/08/04 21:17:00  heller
 * Initial revision
 *
 * ------------------------------------------------------------------
 * Contents:
 * ------------------------------------------------------------------
 *  
 *     Role Playing DB -- A database package that creates and maintains
 * 		       a database of RPG characters, monsters, treasures,
 * 		       spells, and playing environments.
 * 
 *     Copyright (C) 1995,1998  Robert Heller D/B/A Deepwoods Software
 * 			51 Locke Hill Road
 * 			Wendell, MA 01379-9728
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 *  
 */

#ifndef _SPELLS_H_
#define _SPELLS_H_

#include <Record.h>

#ifndef HAVE_BOOL
typedef enum bool {false=0, true=1};
#define HAVE_BOOL 1
#endif

/*@ManDoc:
  \typeout{Generated by $Id: Spells.h,v 1.3 1999/07/13 00:29:40 heller Rel1 $.}
  Basic spell class.  Contains all of the information needed to describe a
  spell.
 */

class Spell {
private:
	//@ManDoc: Spell class (type of character that can use it).
	const char *spclass;
	//@ManDoc: Spell name.
	const char *name;
	//@ManDoc: Spell type.
	const char *sptype;
	//@ManDoc: Descriptive text.
	const char *description;
	//@ManDoc: Area of effect.
	const char *area;
	//@ManDoc: Casting time.
	const char *casttime;
	//@ManDoc: Duration.
	const char *duration;
	//@ManDoc: Saving throw.
	const char *savethrow;
	//@ManDoc: Spell level.
	int level;
	//@ManDoc: Range.
	int range;
	//@ManDoc: Flag bit structure.
	struct FlagBits {
		//@Man: reversibile:1
		//@Type: unsigned
		//@Doc: Is spell reversible?
		unsigned reversible:1;
		//@Man: verbal:1
		//@Type: unsigned
		//@Doc: Does spell use a verbal component?
		unsigned verbal:1;
		//@Man: somatic:1
		//@Type: unsigned
		//@Doc: Does spell use a somatic component?
		unsigned somatic:1;
		//@Man: material:1
		//@Type: unsigned
		//@Doc: Does spell use a material component?
		unsigned material:1;
		//@Man: fill:4
		//@Type: unsigned
		//@Doc: Fill out the byte.
		unsigned fill:4;
	};
	//@ManDoc: Flag union.
	union FlagUnion {
		//@ManDoc: As bits.
		FlagBits bits;
		//@ManDoc: As a byte.
		unsigned char byte;
		//@ManDoc: Constructor.
		FlagUnion () {byte = 0;}
	};
	//@ManDoc: Flags -- components and reversibility.
	FlagUnion flags;
	//@ManDoc: Data record.
	Record rawData;
	//@ManDoc: Convert internal record to monster.
	void RecordToSpell ();
	//@ManDoc: Update internal record.
	void UpdateRecord();
public:	
	//@ManDoc: Update Spell from Record.
	void UpdateFromRecord (const Record &rec)
	{
		rawData = rec;
		RecordToSpell ();
	}
	//@ManDoc: Constructor.
	Spell (
		//@ManDoc: Spell class.
		const char *spc = "",
		//@ManDoc: Name.
		const char *nm = "",
		//@ManDoc: Type.
		const char *spt = "",
		//@ManDoc: Description.
		const char *desc = "",
		//@ManDoc: Effect area.
		const char *a = "",
		//@ManDoc: Casting time.
		const char *ct = "",
		//@ManDoc: Duration.
		const char *dt = "",
		//@ManDoc: Saving throw.
		const char *st = "",
		//@ManDoc: Level.
		int l = 0,
		//@ManDoc: Range.
		int r = 0,
		//@ManDoc: Reversible?
		bool revP = false,
		//@ManDoc: Verbal?
		bool verbP = false,
		//@ManDoc: Somatic?
		bool somP = false,
		//@ManDoc: Material?
		bool matP = false
	) {
		spclass = spc;
		name    = nm;
		sptype  = spt;
		description = desc;
		area    = a;
		casttime = ct;
		duration = dt;
		savethrow = st;
		level   = l;
		range   = r;
		flags.bits.reversible = (revP)?(1):(0);
		flags.bits.verbal     = (verbP)?(1):(0);
		flags.bits.somatic    = (somP)?(1):(0);
		flags.bits.material   = (matP)?(1):(0);
		UpdateRecord();
	}
	//@ManDoc: Copy constructor (from pointer).
	Spell (const Spell *that)
	{
		UpdateFromRecord((Record)(*that));
	}
	//@ManDoc: Copy constructor (from reference).
	Spell (const Spell &that)
	{
		UpdateFromRecord((Record)(that));
	}
	//@ManDoc: Type conversion constructor, from a Record.
	Spell (const Record *rec)
	{
		UpdateFromRecord(*rec);
	}
	//@ManDoc: Destructor.
	~Spell() {}
	//@ManDoc: Type conversion: convert to a Record.
	operator const Record  () const {return rawData;}
	//@ManDoc: Return the spell's class.
	const char *SpellClass() const {return spclass;}
	//@ManDoc: Return the spell's name.
	const char *Name() const {return name;}
	//@ManDoc: Return the spell's type.
	const char *SpellType() const {return sptype;}
	//@ManDoc: Return the spell's description.
	const char *Description() const {return description;}
	//@ManDoc: Return the spell's area of effect.
	const char *AreaOfEffect() const {return area;}
	//@ManDoc: Return the spell's casting time.
	const char *CastingTime() const {return casttime;}
	//@ManDoc: Return the spell's duration.
	const char *Duration() const {return duration;}
	//@ManDoc: Return the spell's Saving Throw.
	const char *SavingThrow() const {return savethrow;}
	//@ManDoc: Return the spell's level.
	int Level() const {return level;}
	//@ManDoc: Return the spell's range.
	int Range() const {return range;}
	//@ManDoc: Return spell's reversibility flag.
	bool ReversibleP() const {return flags.bits.reversible == 1;}
	//@ManDoc: Return spell's verbal component flag.
	bool VerbalP() const {return flags.bits.verbal == 1;}
	//@ManDoc: Return spell's somatic component flag.
	bool SomaticP() const {return flags.bits.somatic == 1;}
	//@ManDoc: Return spell's material component flag.
	bool MaterialP() const {return flags.bits.material == 1;}
};



#endif // _SPELLS_H_

