//ff_vclt.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2009-2010
 *
 *  This file is part of RoarAudio PlayList Daemon,
 *  a playlist management daemon for RoarAudio.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "rpld.h"

int ff_vclt_pl_export(struct fformat_handle * handle, struct roar_vio_calls * vio, struct rpld_playlist  * pl) {
 struct rpld_playlist_entry * cur = rpld_pl_get_first(pl);
#ifdef HAVE_LIB_UUID
 char uuid[37];
#endif

 (void)handle;

 while ( cur != NULL ) {
  if ( cur->io.filename[0] )
   roar_vio_printf(vio, "FILENAME=%s\n", cur->io.filename);

  if ( cur->meta.album[0] )
   roar_vio_printf(vio, "ALBUM=%s\n", cur->meta.album);
  if ( cur->meta.title[0] )
   roar_vio_printf(vio, "TITLE=%s\n", cur->meta.title);
  if ( cur->meta.artist[0] )
   roar_vio_printf(vio, "ARTIST=%s\n", cur->meta.artist);
  if ( cur->meta.performer[0] )
   roar_vio_printf(vio, "PERFORMER=%s\n", cur->meta.performer);
  if ( cur->meta.version[0] )
   roar_vio_printf(vio, "VERSION=%s\n", cur->meta.version);

  if ( cur->meta.genre != -1 )
   roar_vio_printf(vio, "GENRE=%s\n", roar_meta_strgenre(cur->meta.genre));

  if ( cur->meta.tracknum )
   roar_vio_printf(vio, "TRACKNUMBER=%i\n", cur->meta.tracknum);
  if ( cur->meta.discid )
   roar_vio_printf(vio, "DISCID=%.8x\n", cur->meta.discid);

  if ( cur->length )
   roar_vio_printf(vio, "LENGTH=%s\n", rpld_ple_time_hr(cur));

  if ( cur->codec != -1 )
   roar_vio_printf(vio, "SIGNALINFO=codec:%s\n", roar_codec2str(cur->codec));

#ifdef HAVE_LIB_UUID
  uuid_unparse(cur->uuid, uuid);
   roar_vio_printf(vio, "HASH={UUID}%s\n", uuid);
#endif

  roar_vio_printf(vio, "LIKENESS=%f\n", (double)cur->likeness);

  roar_vio_printf(vio, "==\n");
  cur = cur->list.next;
 }

 return 0;
}

int ff_vclt_pl_import(struct fformat_handle * handle, struct roar_vio_calls * vio, struct rpld_playlist  * pl) {
 struct rpld_playlist_entry * plent = NULL;
 struct roar_audio_info       info;
 char buf[1024];
 char * s, * delm;
 char * dst;
 size_t len;
 int    meta;
#ifdef HAVE_LIB_UUID
 uuid_t uuid;
#endif

 (void)handle;

 while ((s = roar_vio_fgets(vio, buf, 1024)) != NULL ) {
  len = strlen(s);

  for (s += len - 1; *s == '\r' || *s == '\n'; s--)
   *s = 0;

  if ( !*buf )
   continue;

  if ( !strcmp(buf, "==") ) {
   if ( plent != NULL )
    rpld_pl_push(pl, plent);

   plent = NULL;

   continue;
  }

  if ( (delm = strstr(buf, "=")) == NULL )
   continue;

  *delm = 0;
   delm++;

  if ( plent == NULL )
   if ( (plent = rpld_ple_new()) == NULL )
    return -1;

  meta = roar_meta_inttype(buf);

  switch (meta) {
   case ROAR_META_TYPE_FILENAME:
   case ROAR_META_TYPE_FILEURL:
   case ROAR_META_TYPE_STREAMURL:
     dst = plent->io.filename;
    break;
   case ROAR_META_TYPE_ALBUM:
     dst = plent->meta.album;
    break;
   case ROAR_META_TYPE_TITLE:
     dst = plent->meta.title;
    break;
   case ROAR_META_TYPE_ARTIST:
     dst = plent->meta.artist;
    break;
   case ROAR_META_TYPE_PERFORMER:
     dst = plent->meta.performer;
    break;
   case ROAR_META_TYPE_VERSION:
     dst = plent->meta.version;
    break;
   case ROAR_META_TYPE_LENGTH:
     rpld_ple_set_time_hr(plent, delm);
     continue;
    break;
   case ROAR_META_TYPE_AUDIOINFO:
      if ( roar_meta_parse_audioinfo(&info, delm) != -1 ) {
       plent->codec = info.codec;
      }
     continue;
    break;
   case ROAR_META_TYPE_TRACKNUMBER:
     plent->meta.tracknum = atoi(delm);
     continue;
    break;
   case ROAR_META_TYPE_GENRE:
     plent->meta.genre = roar_meta_intgenre(delm);
     continue;
    break;
   case ROAR_META_TYPE_DISCID:
     sscanf(delm, "%8x", &(plent->meta.discid));
     continue;
    break;
#ifdef HAVE_LIB_UUID
   case ROAR_META_TYPE_HASH:
     if ( !strncasecmp(delm, "{UUID}", 6) ) {
      if ( uuid_parse(delm+6, uuid) == 0 ) {
       uuid_copy(plent->uuid, uuid);
      }
     }
     continue;
    break;
#endif
#ifdef ROAR_META_TYPE_LIKENESS
   case ROAR_META_TYPE_LIKENESS:
     plent->likeness = atof(delm);
     continue;
    break;
#endif
   default:
     continue;
    break;
  }

  strncpy(dst, delm, RPLD_MAX_PLF_LEN);
  dst[RPLD_MAX_PLF_LEN-1] = 0;
 }

 if ( plent != NULL )
  rpld_pl_push(pl, plent);

 return 0;
}

//ll
