//filter_clip.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2008-2013
 *
 *  This file is part of libroardsp a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  libroardsp is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include "libroardsp.h"

int roardsp_clip_init  (struct roardsp_filter * filter, struct roar_stream * stream, int id) {
 (void)stream, (void)id;

 if ( (filter->inst = roar_mm_malloc(sizeof(struct roardsp_clip))) == NULL )
  return -1;

 return roardsp_filter_reset(filter, ROARDSP_RESET_FULL);
}

int roardsp_clip_uninit(struct roardsp_filter * filter) {
 roar_mm_free(filter->inst);
 return 0;
}

#define _clipX(bits) \
static inline int##bits##_t _clip##bits (int##bits##_t s, struct roardsp_clip * self) { \
 switch (self->mode) { \
  case ROARDSP_CLIP_MODE_LIMIT: \
    return s > 0 ? self->limit : -self->limit; \
   break; \
  case ROARDSP_CLIP_MODE_ZERO: \
    return 0; \
   break; \
  case ROARDSP_CLIP_MODE_WARP: \
    if ( s > 0 ) { \
     return s - 2*self->limit; \
    } else { \
     return s + 2*self->limit; \
    } \
   break; \
  case ROARDSP_CLIP_MODE_NOISE: \
    return (s > 0 ? 1 : -1) * (self->limit - (roar_random_uint16() & 0xFF)); \
   break; \
 } \
\
 ROAR_WARN("_clip16(s=%i, self=%p{.mode=%i, ...}) = 0 // ERROR: Bad mode", (int)s, self, (int)self->mode); \
 return 0; \
}

#define _calcX(bits) \
int roardsp_clip_calc##bits  (struct roardsp_filter * filter, void * data, size_t samples) { \
 struct roardsp_clip * self = filter->inst; \
 int##bits##_t * samp = (int##bits##_t *) data; \
 register int32_t s = self->limit; \
 size_t i; \
\
 for (i = 0; i < samples; i++) { \
  if ( samp[i] > s ) { \
   samp[i]  = _clip##bits (samp[i], self); \
  } else if ( -samp[i] > s ) { \
   samp[i]  = _clip##bits (samp[i], self); \
  } \
 } \
\
 ROAR_DBG("roardsp_quantify_calc16(*) = 0"); \
 return 0; \
}

#define _setX(bits) \
 _clipX(bits) \
 _calcX(bits)

_setX(8)
_setX(16)
_setX(32)

int roardsp_clip_ctl   (struct roardsp_filter * filter, int cmd, void * data) {
 struct roardsp_clip * self = filter->inst;
 int32_t old;

 switch (cmd) {
  case ROARDSP_FCTL_LIMIT:
    old = self->limit;
    self->limit = labs(*(int32_t*)data);
    *(int32_t*)data = old;
   break;
  case ROARDSP_FCTL_MODE:
    old = self->mode;
    self->mode = *(int32_t*)data;
    *(int32_t*)data = old;
   break;
  default:
    ROAR_DBG("roardsp_clip_ctl(*) = -1");
    return -1;
   break;
 }


 ROAR_DBG("roardsp_clip_ctl(*) = 0");
 return 0;
}

int roardsp_clip_reset (struct roardsp_filter * filter, int what) {
 int32_t mode = ROARDSP_CLIP_MODE_LIMIT;
 int32_t n;

 if ( filter == NULL )
  return -1;

 switch (filter->bits) {
  case  8: n =         64L; break;
  case 16: n =      16384L; break;
  case 32: n = 1073741824L; break;
  default:
    roar_err_set(ROAR_ERROR_NOTSUP);
    return -1;
   break;
 }

 switch (what) {
  case ROARDSP_RESET_NONE:
  case ROARDSP_RESET_STATE:
    return  0;
   break;
  case ROARDSP_RESET_FULL:
    roardsp_clip_ctl(filter, ROARDSP_FCTL_LIMIT, &n);
    roardsp_clip_ctl(filter, ROARDSP_FCTL_MODE, &mode);
    return  0;
   break;
  default:
    return -1;
 }

 return -1;
}

//ll
