//emul_gopher.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2010-2012
 *
 *  This file is part of roard a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include "roard.h"

#ifndef ROAR_WITHOUT_DCOMP_EMUL_GOPHER
#include <roaraudio/proto_gopher.h>

#define _INFO  ROAR_GOPHER_TYPE_INFO
#define _DIR   ROAR_GOPHER_TYPE_DIR
#define _FILE  ROAR_GOPHER_TYPE_FILE
#define _SOUND ROAR_GOPHER_TYPE_SOUND

struct item;

static int scb_status_txt (int client, struct roar_vio_calls * vio, char * selector, char ** text, struct item * sitem);
static int scb_test       (int client, struct roar_vio_calls * vio, char * selector, char ** text, struct item * sitem);
static int scb_clients    (int client, struct roar_vio_calls * vio, char * selector, char ** text, struct item * sitem);
static int scb_streams    (int client, struct roar_vio_calls * vio, char * selector, char ** text, struct item * sitem);
static int scb_client_info(int client, struct roar_vio_calls * vio, char * selector, char ** text, struct item * sitem);
static int scb_stream_info(int client, struct roar_vio_calls * vio, char * selector, char ** text, struct item * sitem);
static int scb_listen_menu(int client, struct roar_vio_calls * vio, char * selector, char ** text, struct item * sitem);
static int scb_listen     (int client, struct roar_vio_calls * vio, char * selector, char ** text, struct item * sitem);

static struct roar_gopher_menu_item g_gopher_root_menu[] = {
 {.type = _INFO, .name = "roard Root Menu"},
 {.type = _FILE, .name = "Server Info",   .selector = "/info.txt",   .host = NULL, .port = 0},
 {.type = _FILE, .name = "Server Status", .selector = "/status.txt", .host = NULL, .port = 0},
 {.type = _DIR,  .name = "Clients",       .selector = "/clients/",   .host = NULL, .port = 0},
 {.type = _DIR,  .name = "Streams",       .selector = "/streams/",   .host = NULL, .port = 0},
 {.type = _DIR,  .name = "Listen!",       .selector = "/listen/",    .host = NULL, .port = 0},
};

// need a true constant string.
#define info_text \
 "This server is RoarAudio server (roard) with a small status gopher server\n" \
 "integrated. (For version and stuff see /status.txt.)\n" \
 "RoarAudio is a modern, multi OS, networed sound system.\n" \
 "For more information see http://roaraudio.keep-cool.org/\n"

static struct item {
 const char * selector;
 char type;
 struct roar_gopher_menu menu;
 struct roar_audio_info  info;
 int dir;
 const char * text;
 int (*cb)(int client, struct roar_vio_calls * vio, char * selector, char ** text, struct item * sitem);
} g_gopher_items[] = {
 {.selector = "", .type = _DIR,
  .menu = {.items = g_gopher_root_menu, .items_len = sizeof(g_gopher_root_menu)/sizeof(*g_gopher_root_menu)},
  .cb = NULL
 },
 // and again as selector '/' as some clients seems to require it:
 {.selector = "/", .type = _DIR,
  .menu = {.items = g_gopher_root_menu, .items_len = sizeof(g_gopher_root_menu)/sizeof(*g_gopher_root_menu)},
  .cb = NULL
 },
 {.selector = "/info.txt",   .type = _FILE, .text = info_text, .cb = NULL},
 {.selector = "/status.txt", .type = _FILE, .cb = scb_status_txt},
 {.selector = "/test/*",     .type = _FILE, .cb = scb_test},
 {.selector = "/clients/",   .type = _DIR,  .cb = scb_clients},
 {.selector = "/streams/",   .type = _DIR,  .cb = scb_streams},
 {.selector = "/clients/*/", .type = _DIR,  .cb = scb_client_info},
 {.selector = "/streams/*/", .type = _DIR,  .cb = scb_stream_info},
 {.selector = "/listen/",    .type = _DIR,  .cb = scb_listen_menu},
 {.selector = "/listen/*/*/*/*/*", .type = _FILE,  .cb = scb_listen}
};

static int strselcmp(const char *s1, const char *s2);
static ssize_t strseltok(const char *s1, char *s2, char ** tok, size_t toks);

static char * _aprintf(size_t sizehint, const char * format, ...);

static int send_menu (int client, struct roar_gopher_menu * menu, struct roar_vio_calls * vio);
static int send_text (int client, const char * text, struct roar_vio_calls * vio);


// SCBs:
static int scb_status_txt (int client, struct roar_vio_calls * vio, char * selector, char ** text, struct item * sitem) {
 const size_t len = 1024;
 const char * server_version = NULL;

 if ( DISTRIBUTION_VERSION_STRING[0] == 0 ) {
  server_version = "roard/" PACKAGE_VERSION " <" DEVICE_VENDOR_STRING ">";
 } else {
  server_version = "roard/" PACKAGE_VERSION " <" DEVICE_VENDOR_STRING "> (" DISTRIBUTION_VERSION_STRING ")";
 }

 *text = roar_mm_malloc(len);
 if ( *text == NULL )
  return -1;

 **text = 0;

 snprintf(*text, len,
          "Server version:     %s\r\n"
          "Server location:    %s\r\n"
          "Server description: %s\r\n"
          "\r\n"
          "Counters current:   %llu clients, %llu streams\r\n"
          "Counters sum:       %llu clients, %llu streams\r\n",
          server_version,
          g_config->location,
          g_config->description,
          (long long unsigned int)g_counters.cur.clients,
          (long long unsigned int)g_counters.cur.streams,
          (long long unsigned int)g_counters.sum.clients,
          (long long unsigned int)g_counters.sum.streams
         );

 (*text)[len-1] = 0;
 return 0;
}

static int scb_test(int client, struct roar_vio_calls * vio, char * selector, char ** text, struct item * sitem) {
 ssize_t toks;
 char * tok;
 size_t len;

 toks = strseltok(sitem->selector, selector, &tok, 1);

 if ( toks == -1 )
  return -1;

 len  = strlen(tok);
 len += 64;

 *text = roar_mm_malloc(1024);
 if ( *text == NULL )
  return -1;

 **text = 0;

 snprintf(*text, len, "Your text was: %s", tok);

 (*text)[len-1] = 0;

 return 0;
}

static int scb_clients    (int client, struct roar_vio_calls * vio, char * selector, char ** text, struct item * sitem) {
 struct roar_gopher_menu_item items[ROAR_CLIENTS_MAX];
 struct roar_gopher_menu menu = {.flags = 0, .items = items, .items_len = 0};
 struct roar_gopher_menu_item * item;
 struct roar_client_server * cs;
 struct roar_client        * c;
 const size_t len = 80;
 char * d;
 size_t i;
 int ret;

 memset(items, 0, sizeof(items));

 for (i = 0; i < ROAR_CLIENTS_MAX; i++) {
  if ( (c = ROAR_CLIENT((cs = g_clients[i]))) != NULL ) {
   item = &(items[menu.items_len++]);
   item->type = _DIR;
   d = roar_mm_malloc(len);
   if ( d == NULL ) {
    menu.items_len--;
    continue;
   }
   if ( c->name != NULL && c->name[0] != 0 ) {
    snprintf(d, len, "Client %i: %s", (int)i, c->name);
   } else {
    snprintf(d, len, "Client %i", (int)i);
   }
   item->name = d;

   d = roar_mm_malloc(len);
   if ( d == NULL ) {
    if ( item->name != NULL )
     roar_mm_free((void*)item->name);
    menu.items_len--;
    continue;
   }

   snprintf(d, len, "/clients/%i/", (int)i);
   item->selector = d;
  }
 }

 ret = send_menu(client, &menu, vio);

 for (i = 0; i < menu.items_len; i++) {
  if ( items[i].name != NULL )
   roar_mm_free((void*)items[i].name);
  if ( items[i].selector != NULL )
   roar_mm_free((void*)items[i].selector);
 }

 return ret;
}

static int scb_streams    (int client, struct roar_vio_calls * vio, char * selector, char ** text, struct item * sitem) {
 struct roar_gopher_menu_item items[ROAR_STREAMS_MAX];
 struct roar_gopher_menu menu = {.flags = 0, .items = items, .items_len = 0};
 struct roar_gopher_menu_item * item;
 struct roar_stream_server * ss;
 struct roar_stream        * s;
 const size_t len = 80;
 char * d;
 size_t i;
 int ret;

 memset(items, 0, sizeof(items));

 for (i = 0; i < ROAR_STREAMS_MAX; i++) {
  if ( (s = ROAR_STREAM((ss = g_streams[i]))) != NULL ) {
   item = &(items[menu.items_len++]);
   item->type = _DIR;
   d = roar_mm_malloc(len);
   if ( d == NULL ) {
    menu.items_len--;
    continue;
   }
   if ( ss->name != NULL && ss->name[0] != 0 ) {
    snprintf(d, len, "Stream %i: %s", (int)i, ss->name);
   } else {
    snprintf(d, len, "Stream %i", (int)i);
   }
   item->name = d;

   d = roar_mm_malloc(len);
   if ( d == NULL ) {
    if ( item->name != NULL )
     roar_mm_free((void*)item->name);
    menu.items_len--;
    continue;
   }

   snprintf(d, len, "/streams/%i/", (int)i);
   item->selector = d;
  }
 }

 ret = send_menu(client, &menu, vio);

 for (i = 0; i < menu.items_len; i++) {
  if ( items[i].name != NULL )
   roar_mm_free((void*)items[i].name);
  if ( items[i].selector != NULL )
   roar_mm_free((void*)items[i].selector);
 }

 return ret;
}

static int scb_client_info(int client, struct roar_vio_calls * vio, char * selector, char ** text, struct item * sitem) {
#define _MAX_ITEMS (16 + ROAR_CLIENTS_MAX_STREAMS_PER_CLIENT)
 struct roar_gopher_menu_item items[_MAX_ITEMS];
 struct roar_gopher_menu menu = {.flags = 0, .items = items, .items_len = 0};
 struct roar_gopher_menu_item * item;
 struct roar_client_server * cs;
 struct roar_client        * c;
 size_t i;
 int ret;
 ssize_t toks;
 char * tok;
 int id;
 char tmp[80];

 memset(items, 0, sizeof(items));

 toks = strseltok(sitem->selector, selector, &tok, 1);
 if ( toks == -1 )
  return -1;

 id = atoi(tok);

 if ( clients_get_server(id, &cs) == -1 )
  return -1;

 c = ROAR_CLIENT(cs);

 item = &(items[menu.items_len++]);
 item->type = _INFO;
 if ( c->name != NULL && c->name[0] != 0 ) {
  item->name = _aprintf(64, "Client %i: %s", id, c->name);
 } else {
  item->name = _aprintf(64, "Client %i", id);
 }

 if ( roar_nnode_get_socktype(&(c->nnode)) != ROAR_SOCKET_TYPE_UNKNOWN ) {
  if ( roar_nnode_to_str(&(c->nnode), tmp, sizeof(tmp)) == 0 ) {
   item = &(items[menu.items_len++]);
   item->type = _INFO;
   item->name = _aprintf(64, "Network node: %s", tmp);
  }
 }

 item = &(items[menu.items_len++]);
 item->type = _INFO;
 item->name = _aprintf(64, "Protocol: %s", roar_proto2str(c->proto));

 if ( c->execed != -1 ) {
  item = &(items[menu.items_len++]);
  item->type = _DIR;
  item->name = _aprintf(64, "Exected Stream: %i", c->execed);
  item->selector = _aprintf(64, "/streams/%i/", c->execed);
 }

 for (i = 0; i < ROAR_CLIENTS_MAX_STREAMS_PER_CLIENT; i++) {
  if ( c->streams[i] != -1 ) {
   item = &(items[menu.items_len++]);
   item->type = _DIR;
   item->name = _aprintf(64, "Stream: %i", c->streams[i]);
   item->selector = _aprintf(64, "/streams/%i/", c->streams[i]);
  }
 }

 ret = send_menu(client, &menu, vio);

 for (i = 0; i < menu.items_len; i++) {
  if ( items[i].name != NULL )
   roar_mm_free((void*)items[i].name);
  if ( items[i].selector != NULL )
   roar_mm_free((void*)items[i].selector);
 }

#undef _MAX_ITEMS
 return ret;
}

static int scb_stream_info(int client, struct roar_vio_calls * vio, char * selector, char ** text, struct item * sitem) {
#define _MAX_ITEMS 12
 struct roar_gopher_menu_item items[_MAX_ITEMS];
 struct roar_gopher_menu menu = {.flags = 0, .items = items, .items_len = 0};
 struct roar_gopher_menu_item * item;
 struct roar_stream_server * ss;
 struct roar_stream        * s;
 size_t i;
 int ret;
 ssize_t toks;
 char * tok;
 int id;

 memset(items, 0, sizeof(items));

 toks = strseltok(sitem->selector, selector, &tok, 1);
 if ( toks == -1 )
  return -1;

 id = atoi(tok);

 if ( streams_get(id, &ss) == -1 )
  return -1;

 s = ROAR_STREAM(ss);


 item = &(items[menu.items_len++]);
 item->type = _INFO;
 if ( ss->name != NULL && ss->name[0] != 0 ) {
  item->name = _aprintf(64, "Stream %i: %s", id, ss->name);
 } else {
  item->name = _aprintf(64, "Stream %i", id);
 }

 item = &(items[menu.items_len++]);
 item->type = _INFO;
 item->name = _aprintf(64, "Stream state: %s", roar_streamstate2str(ss->state));

 item = &(items[menu.items_len++]);
 item->type = _INFO;
 item->name = _aprintf(64, "Stream direction: %s", roar_dir2str(s->dir));

 item = &(items[menu.items_len++]);
 item->type = _INFO;
 item->name = _aprintf(128, "Signal info: rate:%iHz bits:%i channels:%i codec:%s",
                            s->info.rate, s->info.bits, s->info.channels, roar_codec2str(s->info.codec));

 if ( ss->codec_orgi != -1 && ss->codec_orgi != s->info.codec ) {
  item = &(items[menu.items_len++]);
  item->type = _INFO;
  item->name = _aprintf(64, "Streamed codec: %s", roar_codec2str(ss->codec_orgi));
 }

 if ( ss->role != -1 ) {
  item = &(items[menu.items_len++]);
  item->type = _INFO;
  item->name = _aprintf(64, "Stream role: %s", roar_role2str(ss->role));
 }

 item = &(items[menu.items_len++]);
 item->type = _DIR;
 item->name = _aprintf(64, "Client: %i", ss->client);
 item->selector = _aprintf(64, "/clients/%i/", ss->client);

 ret = send_menu(client, &menu, vio);

 for (i = 0; i < menu.items_len; i++) {
  if ( items[i].name != NULL )
   roar_mm_free((void*)items[i].name);
  if ( items[i].selector != NULL )
   roar_mm_free((void*)items[i].selector);
 }

#undef _MAX_ITEMS
 return ret;
}

static int scb_listen_menu(int client, struct roar_vio_calls * vio, char * selector, char ** text, struct item * sitem) {
#define _MAX_ITEMS 12
 struct roar_gopher_menu_item items[_MAX_ITEMS];
 struct roar_gopher_menu menu = {.flags = 0, .items = items, .items_len = 0};
 struct roar_gopher_menu_item * item;
 const char * codec;
 int ret;
 int codecs[] = {ROAR_CODEC_DEFAULT, ROAR_CODEC_RIFF_WAVE, ROAR_CODEC_AU, ROAR_CODEC_OGG_VORBIS};
 size_t i;

 memset(items, 0, sizeof(items));

 item = &(items[menu.items_len++]);
 item->type = _INFO;
 item->name = roar_mm_strdup("Select a format you want to listen in:");

 item = &(items[menu.items_len++]);
 item->type = _INFO;
 item->name = NULL; // empty lion

 for (i = 0; i < (sizeof(codecs)/sizeof(*codecs)); i++) {
  item = &(items[menu.items_len++]);
  item->type = _SOUND;
  codec = roar_codec2str(codecs[i]);
  item->name = _aprintf(64, "%u channels with %u bits at %uHz, %s",
                            g_sa->channels, g_sa->bits, g_sa->rate, codec);
  item->selector = _aprintf(64, "/listen/monitor/%u/%u/%u/%s", g_sa->rate, g_sa->bits, g_sa->channels, codec);
 }

 ret = send_menu(client, &menu, vio);

 for (i = 0; i < menu.items_len; i++) {
  if ( items[i].name != NULL )
   roar_mm_free((void*)items[i].name);
  if ( items[i].selector != NULL )
   roar_mm_free((void*)items[i].selector);
 }

#undef _MAX_ITEMS
 return ret;
}

static int scb_listen     (int client, struct roar_vio_calls * vio, char * selector, char ** text, struct item * sitem) {
 struct roar_stream_server * ss;
 struct roar_stream        * s;
 struct roar_audio_info info;
 int dir = -1;
 ssize_t toks;
 char * tok[5];
 int stream = -1;

 toks = strseltok(sitem->selector, selector, tok, 5);

 if ( toks != 5 )
  return -1;

 memset(&info, 0, sizeof(info));

 if ( (dir = roar_str2dir(tok[0])) == -1 )
  return -1;

 // test for unsupported dirs:
 switch (dir) {
  case ROAR_DIR_THRU:
  case ROAR_DIR_RAW_IN:
  case ROAR_DIR_FILTER:
    return -1;
   break;
 }

 info.rate     = atoi(tok[1]);
 info.bits     = atoi(tok[2]);
 info.channels = atoi(tok[3]);
 info.codec    = roar_str2codec(tok[4]);

 if ( info.codec == -1 )
  return -1;

 if ((stream = streams_new()) == -1 )
  return -1;

 if ( streams_get(stream, &ss) == -1 ) {
  streams_delete(stream);
  return -1;
 }

 s = ROAR_STREAM(ss);

 if ( client_stream_add(client, stream) == -1 ) {
  streams_delete(stream);
  return -1;
 }

 memcpy(&(s->info), &info, sizeof(info));
 ss->codec_orgi = info.codec;

 if ( streams_set_dir(stream, dir, 1) == -1 ) {
  streams_delete(stream);
  return -1;
 }

 if ( client_stream_exec(client, stream) == -1 ) {
  streams_delete(stream);
  return -1;
 }

 return 0;
}

// other code:
static int strip_nl (char * str) {
 register char c;

 for (; (c = *str) != 0; str++) {
  if ( c == '\r' || c == '\n' ) {
   *str = 0;
   return 1;
  }
 }

 return 0;
}

static int strselcmp(const char *s1, const char *s2) {
 register char a, b;

 if ( s1 == s2 )
  return 0;

 if ( s1 == NULL || s2 == NULL )
  return -1;

 for (; ; s1++, s2++) {
  a = *s1;
  b = *s2;

  if ( a == '*' ) {
   s1++;
   a = *s1;
   if ( a == 0 ) {
    if ( b == 0 ) {
     return 1; // no match! ('*' does not mach no-chars)
    } else {
     return 0; // match! (string ends with '*' and something not EOS is in b)
    }
   } else {
    for (; *s2 != 0 && *s2 != a; s2++);
    if ( a != *s2 )
     return 1; // no match! (did not find correct char)
   }
  } else if ( a == 0 || b == 0 ) {
   if ( a == b ) {
    return 0; // match!
   } else {
    return 1; // no match! (dffrent length)
   }
  } else if ( a != b ) {
   return 1; // no match! (diffrent chars)
  }
 }

 return -1;
}

static ssize_t strseltok(const char *s1, char *s2, char ** tok, size_t toks) {
 register char a, b;
 size_t idx = 0;

 if ( s1 == NULL || s2 == NULL )
  return -1;

 for (; ; s1++, s2++) {
  a = *s1;
  b = *s2;

  if ( a == 0 || b == 0 ) {
   if ( a == b ) {
    return idx;
   } else {
    return -1;
   }
  } else if ( a == '*' ) {
   s1++;
   a = *s1;
   if ( idx == toks )
    return -1;

   tok[idx] = s2;
   idx++;

   for (; *s2 != 0 && *s2 != a; s2++);

   if ( a == 0 )
    return idx;

   if ( *s1 == 0 )
    return -1;

   *s2 = 0;
  }
 }

 return -1;
}

static char * _aprintf(size_t sizehint, const char * format, ...) {
 va_list ap;
 char * buf;
 int ret;

 sizehint += 128;

 if ( (buf = roar_mm_malloc(sizehint)) == NULL )
  return NULL;

 va_start(ap, format);
 ret = vsnprintf(buf, sizehint, format, ap);
 va_end(ap);

 buf[sizehint-1] = 0;

 return buf;
}

static int send_menu (int client, struct roar_gopher_menu * menu, struct roar_vio_calls * vio) {
 struct roar_buffer * buf;
 struct roar_gopher_menu_item * item;
 const size_t len = 256;
 size_t i;
 void * data;
 char * chardata;
 const char * host;
 unsigned int port;
 struct roar_sockname sockaddr;

 if ( roar_vio_ctl(vio, ROAR_VIO_CTL_GET_SOCKNAME, &sockaddr) == -1 ) {
  memset(&sockaddr, 0, sizeof(sockaddr));
 }

 for (i = 0; i < menu->items_len; i++) {
  item = &(menu->items[i]);
  if ( roar_buffer_new_data(&buf, len, &data) == -1 ) {
   if ( sockaddr.addr != NULL )
    roar_mm_free(sockaddr.addr);
   return -1;
  } 

  chardata = data;

  switch (item->type) {
   case _INFO:
     snprintf(data, len-1, "i%s\tfake\t(NULL)\t0\r\n", item->name == NULL ? "" : item->name);
    break;
   default:
     host = item->host == NULL ? sockaddr.addr : item->host;
     port = item->port ==    0 ? sockaddr.port : item->port;
     snprintf(data, len-1, "%c%s\t%s\t%s\t%u\r\n", item->type, item->name, item->selector, host, port);
    break;
  }

  chardata[len-1] = 0;

  if ( roar_buffer_set_len(buf, strlen(data)) == -1 ) {
   roar_buffer_free(buf);
   return -1;
  }

  clients_add_output(client, &buf);
 }

 if ( sockaddr.addr != NULL )
  roar_mm_free(sockaddr.addr);

 return 0;
}

static int send_text (int client, const char * text, struct roar_vio_calls * vio) {
 struct roar_buffer * buf;
 void * data;
 size_t len = strlen(text);

 if ( roar_buffer_new_data(&buf, len+6, &data) == -1 )
  return -1;

 memcpy(data, text, len);
 //memcpy(data+len, "\r\n.\r\n\0", 6);
 memcpy(data+len, "\0", 1);
 clients_add_output(client, &buf);

 return 0;
}

int emul_gopher_check_client(int client, struct roar_vio_calls * vio) {
 struct roar_client_server * cs;
 struct roar_vio_calls     rvio;
 struct item * c = NULL;
 char inbuf[1024];
 ssize_t ret;
 size_t i;
 int funcret = -1;
 size_t len = 0;
 void * data;
 char * text;

 ROAR_DBG("emul_gopher_check_client(client=%i, vio=%p) = ?", client, vio);

 if ( clients_get_server(client, &cs) == -1 ) {
  clients_delete(client);
  return -1;
 }

 ROAR_DBG("emul_gopher_check_client(client=%i, vio=%p) = ?", client, vio);

 if ( vio == NULL ) {
  vio = &rvio;
  roar_vio_open_fh_socket(vio, clients_get_fh(client));
 }

 ROAR_DBG("emul_gopher_check_client(client=%i, vio=%p) = ?", client, vio);

 if ( cs->inbuf != NULL ) {
  len = sizeof(inbuf)-1;
  if ( roar_buffer_shift_out(&(cs->inbuf), inbuf, &len) == -1 ) {
   clients_delete(client);
   return -1;
  }

  if ( cs->inbuf != NULL ) {
   roar_buffer_free(cs->inbuf);
   clients_delete(client);
   return -1;
  }

  // test if we have still buffer space left.
  if ( len == (sizeof(inbuf)-1) ) {
   clients_delete(client);
   return -1;
  }
 }

 ROAR_DBG("emul_gopher_check_client(client=%i, vio=%p) = ?", client, vio);

 ret = roar_vio_read(vio, inbuf+len, sizeof(inbuf)-len-1);
 if ( ret < 1 ) {
  clients_delete(client);
  ROAR_DBG("emul_gopher_check_client(client=%i, vio=%p) = -1", client, vio);
  return -1;
 }

 ret += len;

 inbuf[ret] = 0;

 ROAR_DBG("emul_gopher_check_client(client=%i, vio=%p) = ?", client, vio);

 if ( !strip_nl(inbuf) ) {
  if ( roar_buffer_new_data(&(cs->inbuf), ret, &data) == -1 ) {
   clients_delete(client);
   ROAR_DBG("emul_gopher_check_client(client=%i, vio=%p) = -1", client, vio);
   return -1;
  }

  memcpy(data, inbuf, ret);

  ROAR_DBG("emul_gopher_check_client(client=%i, vio=%p) = 0", client, vio);

  return 0;
 }

 ROAR_DBG("emul_gopher_check_client(client=%i, vio=%p) = ?", client, vio);

 for (i = 0; i < sizeof(g_gopher_items)/sizeof(*g_gopher_items); i++) {
//  if ( !strselcmp(g_gopher_items[i].selector, inbuf) ) {
  if ( !strselcmp(g_gopher_items[i].selector, inbuf) ) {
   c = &(g_gopher_items[i]);
   break;
  }
 }

 if ( c == NULL ) {
  clients_delete(client);
  ROAR_DBG("emul_gopher_check_client(client=%i, vio=%p) = -1", client, vio);
  return -1;
 }

 ROAR_DBG("emul_gopher_check_client(client=%i, vio=%p) = ?", client, vio);

 if ( c->cb != NULL ) {
  text = NULL;
  funcret = c->cb(client, vio, inbuf, &text, c);

  if ( funcret == 0 && text != NULL )
   funcret = send_text(client, text, vio);

  if ( text != NULL )
   roar_mm_free(text);
 } else {
  switch (c->type) {
   case _DIR:
     funcret = send_menu(client, &(c->menu), vio);
    break;
   case _FILE:
     funcret = send_text(client, c->text, vio);
    break;
   default:
     funcret = -1;
    break;
  }
 }

 ROAR_DBG("emul_gopher_check_client(client=%i, vio=%p) = ?", client, vio);

 if ( funcret == -1 ) {
  clients_delete(client);
  ROAR_DBG("emul_gopher_check_client(client=%i, vio=%p) = -1", client, vio);
  return -1;
 }

 ROAR_DBG("emul_gopher_check_client(client=%i, vio=%p) = 0", client, vio);

 return 0;
}

int emul_gopher_flushed_client(int client, struct roar_vio_calls * vio) {
 ROAR_DBG("emul_gopher_flushed_client(client=%i, vio=%p) = ?", client, vio);

 return clients_delete(client);
}

#endif

//ll
