//vio_proto.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2009-2011
 *
 *  This file is part of libroar a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  libroar is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 *  NOTE for everyone want's to change something and send patches:
 *  read README and HACKING! There a addition information on
 *  the license of this document you need to read before you send
 *  any patches.
 *
 *  NOTE for uses of non-GPL (LGPL,...) software using libesd, libartsc
 *  or libpulse*:
 *  The libs libroaresd, libroararts and libroarpulse link this lib
 *  and are therefore GPL. Because of this it may be illigal to use
 *  them with any software that uses libesd, libartsc or libpulse*.
 */

#include "libroar.h"

#ifndef ROAR_WITHOUT_VIO_PROTO
#include <roaraudio/proto_gopher.h>
#endif

int roar_vio_proto_init_def  (struct roar_vio_defaults * def, char * dstr, int proto, struct roar_vio_defaults * odef) {
#ifndef ROAR_WITHOUT_VIO_PROTO
 int                        port = 0;
 int                        ret;
 char                     * ed;
 char                     * tmp;

 if ( def == NULL )
  return -1;

 switch (proto) {
  case ROAR_VIO_PROTO_P_HTTP:    port =   80; break;
  case ROAR_VIO_PROTO_P_GOPHER:  port =   70; break;
  case ROAR_VIO_PROTO_P_ICY:     port = 8000; break;
  default:
    return -1;
 }

 if ( dstr == NULL )
  dstr = "//";

 if ( roar_vio_dstr_init_defaults(def, ROAR_VIO_DEF_TYPE_SOCKET, O_RDWR, 0644) == -1 )
  return -1;

 if ( roar_vio_socket_init_tcp4_def(def, "localhost", port) == -1 )
  return -1;

 if ( !strncmp(dstr, "//", 2) )
  dstr += 2;

 if ( (ed = strstr(dstr, "/")) != NULL )
  *ed = 0;

 if ( (tmp = strstr(dstr, "@")) != NULL )
  dstr = tmp + 1;

 ROAR_DBG("roar_vio_proto_init_def(*): def->o_flags=%i", def->o_flags);

 ret = roar_vio_socket_init_dstr_def(def, dstr, -1, SOCK_STREAM, def);

 ROAR_DBG("roar_vio_proto_init_def(*): def->o_flags=%i", def->o_flags);

 if ( ed != NULL )
  *ed = '/';

 ROAR_DBG("roar_vio_proto_init_def(*): dstr='%s'", dstr);

 return ret;
#else
 return -1;
#endif
}

int roar_vio_open_proto      (struct roar_vio_calls * calls, struct roar_vio_calls * dst,
                              char * dstr, int proto, struct roar_vio_defaults * odef) {
#ifndef ROAR_WITHOUT_VIO_PROTO
 struct roar_userpass userpass = {.subtype = -1, .user = NULL, .pass = NULL};
 struct roar_vio_proto * self;
 char * host;
 char * tmp;

 ROAR_DBG("roar_vio_open_proto(calls=%p, dst=%p, dstr='%s', proto=%i, odef=%p) = ?", calls, dst, dstr, proto, odef);

 if ( calls == NULL || dst == NULL || odef == NULL )
  return -1;

 ROAR_DBG("roar_vio_open_proto(*): odef->o_flags=%i", odef->o_flags);
 ROAR_DBG("roar_vio_open_proto(*) = ?");

 if ( (self = roar_mm_malloc(sizeof(struct roar_vio_proto))) == NULL )
  return -1;

 memset(self, 0, sizeof(struct roar_vio_proto));

 self->next      = dst;

 calls->inst     = self;

 calls->read     = roar_vio_proto_read;
 calls->write    = roar_vio_proto_write;
// calls->lseek    = roar_vio_proto_lseek; // TODO: this is currently not supported
 calls->nonblock = roar_vio_proto_nonblock;
 calls->sync     = roar_vio_proto_sync;
 calls->ctl      = roar_vio_proto_ctl;
 calls->close    = roar_vio_proto_close;

 ROAR_DBG("roar_vio_open_proto(*) = ?");

 if ( dstr != NULL ) {
  dstr += 2;
  host  = dstr;

  if ( (tmp = strstr(dstr, "/")) == NULL )
   return -1;

  *tmp++ = 0;
  dstr   = tmp;

  if ( (tmp = strstr(dstr, "#")) != NULL )
   *tmp = 0;
 } else {
  ROAR_DBG("roar_vio_open_proto(*): no dstr!, odef->type=%i", odef->type);
  if ( odef->type == ROAR_VIO_DEF_TYPE_FILE ) {
   dstr = odef->d.file;
   host = "localhost";

   for (; *dstr == '/'; dstr++);

  } else if ( odef->type == ROAR_VIO_DEF_TYPE_SOCKET ) {
   dstr = ""; // index document
   host = odef->d.socket.host;
  } else {
   return -1;
  }
 }

 if ( (tmp = strstr(host, "@")) != NULL ) {
  userpass.user = host;
  *tmp = 0;
  host = tmp + 1;
  if ( (tmp = strstr(userpass.user, ":")) != NULL ) {
   *tmp = 0;
   userpass.pass = tmp + 1;
  }
 }

 ROAR_DBG("roar_vio_open_proto(*) = ?");
 ROAR_DBG("roar_vio_open_proto(*): proto=%i, host='%s', file='%s', userpass={.user='%s', .pass='%s'}", proto, host, dstr, userpass.user, userpass.pass);

 self->proto = proto;

 switch (proto) {
  case ROAR_VIO_PROTO_P_HTTP:
  case ROAR_VIO_PROTO_P_ICY:
    return roar_vio_open_proto_http(calls, dst, host, dstr, userpass.user != NULL ? &userpass : NULL);
   break;
  case ROAR_VIO_PROTO_P_GOPHER:
    return roar_vio_open_proto_gopher(calls, dst, host, dstr);
   break;
 }

 ROAR_DBG("roar_vio_open_proto(*) = -1 // no matching protocol");
 return -1;
#else
 return -1;
#endif
}

#ifndef ROAR_WITHOUT_VIO_PROTO
ssize_t roar_vio_proto_read    (struct roar_vio_calls * vio, void *buf, size_t count) {
 struct roar_vio_proto * self = vio->inst;
 ssize_t ret;
 ssize_t have = 0;
 size_t  len;

 ROAR_DBG("roar_vio_proto_read(*): have=%lli, count=%lli", (long long int)have, (long long int)count);

 if ( self->reader.buffer != NULL ) {
  len = count;
  if ( roar_buffer_shift_out(&(self->reader.buffer), buf, &len) == -1 ) {
   // This is very bad.
   return -1;
  }

  if ( len ) {
   have   = len;
   buf   += len;
   count -= len;
  }
 }

 ROAR_DBG("roar_vio_proto_read(*): have=%lli, count=%lli", (long long int)have, (long long int)count);

 if ( count == 0 )
  return have;

 ROAR_DBG("roar_vio_proto_read(*): have=%lli, count=%lli", (long long int)have, (long long int)count);

 if ( (ret = roar_vio_read(self->next, buf, count)) == -1 )
  return ret;

 return have + ret;
}

ssize_t roar_vio_proto_write   (struct roar_vio_calls * vio, void *buf, size_t count) {
 struct roar_vio_proto * self = vio->inst;

 return roar_vio_write(self->next, buf, count);
}

// TODO: this is currently not implemented as this is hard to implement with buffers:
off_t   roar_vio_proto_lseek   (struct roar_vio_calls * vio, off_t offset, int whence);

int     roar_vio_proto_nonblock(struct roar_vio_calls * vio, int state) {
 struct roar_vio_proto * self = vio->inst;

 /* we can simply use the next layer's nonblock as all we do in addtion *
  * to call there functions are our buffers which do not block normaly  */

 return roar_vio_nonblock(self->next, state);
}

int     roar_vio_proto_sync    (struct roar_vio_calls * vio) {
 struct roar_vio_proto * self = vio->inst;

 return roar_vio_sync(self->next);
}

int     roar_vio_proto_ctl     (struct roar_vio_calls * vio, int cmd, void * data) {
 struct roar_vio_proto * self = vio->inst;

 if (vio == NULL || cmd == -1)
  return -1;

 ROAR_DBG("roar_vio_proto_ctl(vio=%p, cmd=0x%.8x, data=%p) = ?", vio, cmd, data);

 switch (cmd) {
  case ROAR_VIO_CTL_GET_NAME:
    if ( data == NULL )
     return -1;

    switch (self->proto) {
     case ROAR_VIO_PROTO_P_HTTP:
       *(char**)data = "http";
      break;
     case ROAR_VIO_PROTO_P_GOPHER:
       *(char**)data = "gopher";
      break;
     case ROAR_VIO_PROTO_P_ICY:
       *(char**)data = "icy";
      break;
     default:
       *(char**)data = "proto";
      break;
    }
    return 0;
   break;
  case ROAR_VIO_CTL_GET_NEXT:
    *(struct roar_vio_calls **)data = self->next;
    return 0;
   break;
  case ROAR_VIO_CTL_SET_NEXT:
    self->next = *(struct roar_vio_calls **)data;
    return 0;
   break;
  case ROAR_VIO_CTL_GET_MIMETYPE:
    if ( data == NULL )
     return -1;

    if ( self->content_type == NULL )
     return -1;

    *(char**)data = self->content_type;
    return 0;
   break;
  case ROAR_VIO_CTL_GET_FH:
  case ROAR_VIO_CTL_GET_SELECT_FH:
    if ( self->reader.buffer == NULL && self->writer.buffer == NULL )
     return roar_vio_ctl(self->next, cmd, data);
    return -1;
   break;
  case ROAR_VIO_CTL_GET_READ_FH:
  case ROAR_VIO_CTL_GET_SELECT_READ_FH:
    if ( self->reader.buffer == NULL )
     return roar_vio_ctl(self->next, cmd, data);
    return -1;
   break;
  case ROAR_VIO_CTL_GET_WRITE_FH:
  case ROAR_VIO_CTL_GET_SELECT_WRITE_FH:
    if ( self->writer.buffer == NULL )
     return roar_vio_ctl(self->next, cmd, data);
    return -1;
   break;
 }

 return roar_vio_ctl(self->next, cmd, data);
}

int     roar_vio_proto_close   (struct roar_vio_calls * vio) {
 struct roar_vio_proto * self = vio->inst;

 if ( roar_vio_close(self->next) == -1 )
  return -1;

 if ( self->content_type != NULL )
  roar_mm_free(self->content_type);

 roar_mm_free(self);

 return 0;
}

static int _parse_header(struct roar_keyval * kv, char ** buf, int * aligned, char * endofheader) {
 char * p = *buf;
 char   c = 0;

 if ( !(*aligned) ) {
  for (; *p != 0 && *p != '\r' && *p != '\n'; p++);
  p++;
  if ( *p == '\n' )
   p++;
 }

 if ( p >= endofheader )
  return -1;

/*
 if ( *p == '\r' || *p == '\n' )
  return 0;
 */

 kv->key = p;

 for (; *p != 0 && *p != '\r' && *p != '\n' && *p != ':'; p++);

 if ( *p == 0 )
  return -1;

 if ( p >= endofheader )
  return -1;

 c = *p;
 *p = 0;

 if ( c == '\r' && *(p+1) == '\n' )
  p++;

 p++;

 if ( c == '\r' || c == '\n' ) {
  if ( *(kv->key) == '\r' || *(kv->key) == '\n' )
   return 0;
//  printf("Key-only\n");
  kv->value = kv->key;
  kv->key   = NULL;
  *buf = p;
  return 1;
 }

 for (; *p == ' '; p++);

 if ( *p == 0 )
  return -1;

 kv->value = p;

 for (; *p != 0 && *p != '\r' && *p != '\n'; p++);

 if ( *p == 0 )
  return -1;

 if ( p >= endofheader )
  return -1;

 c = *p;
 *p = 0;

 if ( c == '\r' && *(p+1) == '\n' )
  p++;

 p++;

 *buf = p;

 if ( c == '\r' || c == '\n' ) {
//  printf("aligned\n");
  *aligned = 1;
  p++;
 } else {
//  printf("non-aligned(c=0x%x)\n", (int)c);
  *aligned = 0;
 }

 if ( *(kv->key) != 0 )
  return 1;

 return 0;
}

static void _handle_header (struct roar_vio_proto * self, struct roar_keyval * kv) {
 ROAR_DBG("_handle_header(*): Header: key='%s', value='%s'", kv->key, kv->value);

 if ( kv->key == NULL || kv->value == NULL )
  return;

 if ( !strcasecmp(kv->key, "Content-Type") ) {
  if ( self->content_type != NULL )
   roar_mm_free(self->content_type);

  self->content_type = roar_mm_strdup(kv->value);
 }
}

static inline char * _up2http_auth (struct roar_userpass * up) {
 char * inbuf, * outbuf;
 size_t inlen, outlen;
 ssize_t ret;
 //Authorization: Basic dXNlcjpwdw==

 if ( up == NULL )
  return NULL;

 if ( up->subtype != -1 )
  return NULL;

 if ( up->user == NULL || up->pass == NULL )
  return NULL;

 inlen = roar_mm_strlen(up->user) + roar_mm_strlen(up->pass) + 2;
 inbuf = roar_mm_malloc(inlen);
 if ( inbuf == NULL )
  return NULL;

 inbuf[0] = 0;
 roar_mm_strlcat(inbuf, up->user, inlen);
 roar_mm_strlcat(inbuf, ":", inlen);
 roar_mm_strlcat(inbuf, up->pass, inlen);

 outlen = ((inlen * 3) / 2) + 3 /* padding... */ + 6 /* 'Basic ' */;
 outbuf = roar_mm_malloc(outlen);
 if ( outbuf == NULL ) {
  roar_mm_free(inbuf);
  return NULL;
 }

 strncpy(outbuf, "Basic ", 7); 

 ROAR_DBG("_up2http_auth(up=%p{.subtype=%i, .user='%s', .pass='%s'): inbuf='%s', outbuf='%s'", up, up->subtype, up->user, up->pass, inbuf, outbuf);
 ret = roar_base64_encode(NULL, outbuf + 6, outlen - 6, inbuf, inlen - 1, NULL, 1);
 ROAR_DBG("_up2http_auth(up=%p{.subtype=%i, .user='%s', .pass='%s'): inbuf='%s', outbuf='%s'", up, up->subtype, up->user, up->pass, inbuf, outbuf);

 roar_mm_free(inbuf);

 if ( ret == -1 ) {
  roar_mm_free(outbuf);
  return NULL;
 }

 return outbuf;
}

int roar_vio_open_proto_http   (struct roar_vio_calls * calls, struct roar_vio_calls * dst, char * host, char * file, struct roar_userpass * up) {
 struct roar_keyval kv;
 struct roar_vio_proto * self;
 struct roar_buffer * bufbuf;
 void * vpbuf;
 char * authbuf;
 char * buf;
 char * endofheader = NULL;
 char * p;
 char b0[80], b1[80];
 int  status;
 int  len;
 int  oeflen = 4;
 int  aligned = 1;

 ROAR_DBG("roar_vio_open_proto_http(calls=%p, dst=%p, host='%s', file='%s') = ?", calls, dst, host, file);

 if ( calls == NULL || dst == NULL || host == NULL || file == NULL )
  return -1;

 self         = calls->inst;
 calls->write = NULL; // Disable write as we do not support this

 if ( roar_buffer_new_data(&bufbuf, 1024, &vpbuf) == -1 )
  return -1;

 buf = vpbuf;

 ROAR_DBG("roar_vio_open_proto_http(calls=%p, dst=%p, host='%s', file='%s') = ?", calls, dst, host, file);

 roar_vio_printf(dst, "GET /%s HTTP/1.1\r\n", file);
 roar_vio_printf(dst, "Host: %s\r\n", host);
 roar_vio_printf(dst, "User-Agent: roar_vio_open_proto_http() $Revision: 1.26 $\r\n");
 roar_vio_printf(dst, "Connection: close\r\n");
 if ( up != NULL ) {
  if ( (authbuf = _up2http_auth(up)) != NULL ) {
   roar_vio_printf(dst, "Authorization: %s\r\n", authbuf);
   roar_mm_free(authbuf);
  }
 }
 roar_vio_printf(dst, "\r\n");

 ROAR_DBG("roar_vio_open_proto_http(*) = ?");

 roar_vio_sync(dst);

 ROAR_DBG("roar_vio_open_proto_http(*) = ?");

 if ( (len = roar_vio_read(dst, buf, 1023)) < 1 ) {
  ROAR_DBG("roar_vio_open_proto_http(*) = -1");
  roar_buffer_free(bufbuf);
  return -1;
 }

 ROAR_DBG("roar_vio_open_proto_http(*): got %i bytes from server.", len);

 buf[len] = 0;

 ROAR_DBG("roar_vio_open_proto_http(*) = ?");

 if ( sscanf(buf, "%79s %i %79s\n", b0, &status, b1) != 3 ) {
  ROAR_DBG("roar_vio_open_proto_http(*) = -1");
  roar_buffer_free(bufbuf);
  return -1;
 }

 ROAR_DBG("roar_vio_open_proto_http(*): b0='%s'", b0);

 ROAR_DBG("roar_vio_open_proto_http(*) = ?");

 if ( status != 200 ) {
  ROAR_DBG("roar_vio_open_proto_http(*) = -1 // status=%i", status);
  roar_buffer_free(bufbuf);
  return -1;
 }

 ROAR_DBG("roar_vio_open_proto_http(*): status=%i", status);
// ROAR_WARN("roar_vio_open_proto_http(*): buf='%s'", buf);

 endofheader = strstr(buf, "\r\n\r\n");
 if ( endofheader == NULL ) {
  endofheader = strstr(buf, "\n\n");
  oeflen = 2;
 }

 ROAR_DBG("roar_vio_open_proto_http(*): endofheader=%p\n", endofheader);

 p = buf;
 while ( _parse_header(&kv, &p, &aligned, endofheader) > 0 )
  if ( aligned )
   _handle_header(self, &kv);

 while ( endofheader == NULL ) {
  if ( (len = roar_vio_read(dst, buf, 1023)) < 1 )
   return -1;

  buf[len] = 0;
  endofheader = strstr(buf, "\r\n\r\n");
  if ( endofheader == NULL ) {
   endofheader = strstr(buf, "\n\n");
   oeflen = 2;
  }

/* Doesn't work good.
  while ( _parse_header(&kv, &p, &aligned, endofheader) > 0 )
   if ( aligned )
    _handle_header(self, &kv);
*/

  p = buf;
  while ( _parse_header(&kv, &p, &aligned, endofheader) > 0 )
   if ( aligned )
    _handle_header(self, &kv);

  ROAR_DBG("roar_vio_open_proto_http(*): endofheader=%p\n", endofheader);
 }

 ROAR_DBG("roar_vio_open_proto_http(*): endofheader=%p\n", endofheader);
 ROAR_DBG("roar_vio_open_proto_http(*): buf=%p\n", buf);

 if ( (endofheader - buf) == (len - oeflen) ) {
  roar_buffer_free(bufbuf);
  bufbuf = NULL;
 }

 if ( bufbuf != NULL ) {
  roar_buffer_set_offset(bufbuf, endofheader - buf + oeflen);
  roar_buffer_set_len(bufbuf,    len - (endofheader - buf + oeflen) - 0 /* ??? */);
 }
 self->reader.buffer = bufbuf;

/*
 if ( !strcmp((buf+len)-4, "\r\n\r\n") )
  return 0;

 while (*buf != '\r' && *buf != '\n') {
  if ( (len = roar_vio_read(dst, buf, 1023)) < 1 )
   return -1;
 }
*/

 return 0;
}

int roar_vio_open_proto_gopher (struct roar_vio_calls * calls, struct roar_vio_calls * dst, char * host, char * file) {
 struct roar_vio_proto * self;
 char type;
 const char * mime = NULL;

 if ( calls == NULL || dst == NULL || host == NULL || file == NULL )
  return -1;

 self         = calls->inst;
 calls->write = NULL; // Disable write as we do not support this

 ROAR_DBG("roar_vio_open_proto_gopher(calls=%p, dst=%p, host='%s', file='%s') = ?", calls, dst, host, file);

 type = file[0];

 file++;

 ROAR_DBG("roar_vio_open_proto_gopher(*): type='%c'", type);

 switch (type) {
  case ROAR_GOPHER_TYPE_FILE:
    mime = "text/plain";
   break;
  case ROAR_GOPHER_TYPE_DIR:
    mime = "inode/directory";
   break;
  case ROAR_GOPHER_TYPE_BIN:
    mime = "application/octet-stream";
   break;
  case ROAR_GOPHER_TYPE_GIF:
    mime = "image/gif";
   break;
  case ROAR_GOPHER_TYPE_HTML:
    mime = "text/html";
   break;
 }

 if ( mime != NULL ) {
  self->content_type = roar_mm_strdup(mime);
 }

 roar_vio_printf(dst, "%s\r\n", file);

 roar_vio_sync(dst); // for encryption/compression layers

 return 0;
}
#endif

//ll
