//rlp_math.cpp, Copyright (c) 2004, 2005 R.Lackner
//
//    This file is part of RLPlot.
//
//    RLPlot is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    RLPlot is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with RLPlot; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
#include "rlplot.h"
#include <math.h>
#include <stdlib.h>

#define SWAP(a,b) {double temp=(a);(a)=(b);(b)=temp;}
#define _PREC 1.0e-12

static char *MRQ_error = 0L;

//---------------------------------------------------------------------------
//utilitity functions for memory allocation
double **dmatrix(int nrl, int nrh, int ncl, int nch)
{
	int i;
	double **m;

	m = (double **)malloc(nrh * sizeof(double*));
	//Allocate rows and set pointers to them
	for(i = 0; i < nrh; i++) {
		m[i] = (double *)malloc(nrh * sizeof(double));
		}
	return m;
}
void free_dmatrix(double **m, int nrl, int nrh, int ncl, int)
{
	int i;

	for(i = 0; i < nrh; i++) free(m[i]);
	free(m);
}

//---------------------------------------------------------------------------
//The routine gaussj solves linear equations by Gauss-Jordan elimination
bool gaussj(double **a, int n, double **b, int m)
{
	int *indxc, *indxr, *ipiv;
	int i, icol, irow, j, k, l, ll;
	double big, dum, pivinv;

	indxc = (int*)malloc(n*sizeof(int*));
	indxr = (int*)malloc(n*sizeof(int*));
	ipiv = (int*)malloc(n*sizeof(int*));
	for (j = 0; j < n; j++) ipiv[j] = 0;
	for (i = 0; i < n; i++) {				//This is the main loop over the
		big = 0.0;							//    columns to be reduced
		for(j = 0; j < n; j ++)				//This is the outer loop of the search
			if(ipiv[j] != 1)				//    for a pivot element
				for(k = 0; k < n; k ++) {
					if (ipiv[k] == 0) {
						if(fabs(a[j][k]) >= big) {
							big = fabs(a[j][k]);
							irow = j;				icol = k;
							}
						}
					else if(ipiv[k] > 1) {
						MRQ_error = "Singular Matrix (1)";
						free(ipiv);		free(indxr);	free(indxc);
						return false;
						}
				}
		++(ipiv[icol]);
		//We now have the pivot element, so we interchange rows, if needed,
		// to put the pivot element on the diagonal.
		if(irow != icol) {
			for(l = 0; l < n; l++) SWAP(a[irow][l], a[icol][l])
			for(l = 0; l < m; l++) SWAP(b[irow][l], b[icol][l])
			}
		indxr[i] = irow;		indxc[i] = icol;
		if(a[icol][icol] == 0.0) {
			MRQ_error = "Singular Matrix (2)";
			free(ipiv);		free(indxr);	free(indxc);
			return false;
			}
		pivinv = 1.0/a[icol][icol];
		a[icol][icol] = 1.0;
		for(l = 0; l < n; l++) a[icol][l] *= pivinv;
		for(l = 0; l < m; l++) b[icol][l] *= pivinv;
		for(ll = 0; ll <  n; ll++)
			if(ll != icol) { 							//Next, we reduce the rows
				dum = a[ll][icol];
				a[ll][icol] = 0.0;
				for(l = 0; l < n; l++) a[ll][l] -= a[icol][l]*dum;
				for(l = 0; l < m; l++) b[ll][l] -= b[icol][l]*dum;
				}
		}											// This is the end of the main loop
	for (l = n; l > 0; l--) {						//   over columns of the reduction.
		if(indxr[l] != indxc[l]) 					//   Unscramble the solution
			for(k = 0; k < n; k++) SWAP (a[k][indxr[l]], a[k][indxc[l]]);
		}											//And we are done.
	free(ipiv);		free(indxr);	free(indxc);
	return true;
}

//---------------------------------------------------------------------------
//The routine mrqcof is called by mrqmin to evaluate the linearized fitting
// matrix alpha and vector beta
void mrqcof(double x[], double y[], double z[], int ndata, double **a, int ma,
	int lista[], int mfit, double **alpha, double beta[], double *chisq,
	void (*funcs)(double, double, double **, double *, double *, int))
{
	int k, j, i;
	double ymod, wt, dy;
	double *dyda;

	dyda = (double*)malloc(ma*sizeof(double));
	for(j = 0; j < mfit; j++) {					//Initialize (symmetric) alpha, beta
		for(k = 0; k <= j; k++) alpha[j][k] = 0.0;
		beta[j] = 0.0;
		}
	*chisq = 0.0;
	for (i = 0; i < ndata; i++) {		 		//Summation loop over all data
		(*funcs)(x[i], z ? z[i] : 0.0, a, &ymod, dyda, ma);
		if(ymod != 0.0) dy = y[i]-ymod;			//functions = 0.0 if out of range
		else dy = 0.0;
		for(j = 0; j < mfit; j++) {
			wt = dyda[lista[j]];
			for (k = 0; k <= j; k++){
				alpha[j][k] += wt*dyda[lista[k]];
				}
			beta[j] += dy*wt;
			}
		(*chisq) += dy*dy; 							//And find X^2 if function o.k.
		}
	for(j = 0; j < mfit; j++)						//Fill the symmetric side
		for(k = 0; k <= j; k++) alpha[k][j]=alpha[j][k];
	free(dyda);
}

//---------------------------------------------------------------------------
//The routine mrqmin performs one iteration of Marquart's method for nonlinear
// parameter estimation
bool mrqmin(double *x, double *y, double *z, int ndata, double **a, int ma,
	int *lista, int mfit, double **covar, double **alpha, double *chisq,
	void (*funcs)(double, double, double **, double *, double *, int), double *alamda)
{
	int k, kk, j, ihit;
	static double *da, *atry, *beta, ochisq;
	static double **oneda, **atryref;

	if (*alamda < 0.0) {								//Initialization
		MRQ_error = 0L;
		oneda = dmatrix(1, mfit, 1, 1);
		atry = (double *)malloc(ma * sizeof(double));
		atryref = (double**)malloc(ma * sizeof(double*));
		for(j=0; j < ma; atryref[j++] = &atry[j]);
		da = (double*)malloc(ma *sizeof(double));
		beta = (double*)malloc(ma *sizeof(double));
		kk = mfit+1;
		for(j = 0; j < ma; j++) { 						//Does lista contain a proper
			ihit = 0;									//   permutation of the
			for(k = 0; k < mfit; k++)					//   coefficients ?
				if(lista[k] == j) ihit++;
			if(ihit == 0)
				lista[kk++] = j;
			else if (ihit >1) ErrorBox("Bad LISTA permutations in MRQMIN-1");
			}
		if(kk != ma+1) ErrorBox("Bad LISTA permutations in MRQMIN-2");
		*alamda = 0.001;
		mrqcof(x, y, z, ndata, a, ma, lista, mfit, alpha, beta, chisq, funcs);
		ochisq=(*chisq);
		}
	for (j = 0; j < mfit; j++) {						//Alter linearized fitting matrix
		for(k = 0; k < mfit; k++) covar[j][k] = alpha[j][k];	// by augmenting
		covar[j][j] = alpha[j][j]*(1.0+(*alamda));		// diagaonal elements
		oneda[j][0] = beta[j];
		}
	if (!gaussj(covar, mfit, oneda, 1)) return false;	//Matrix solution ?
	for(j = 0; j < mfit; j++) da[j] = oneda[j][0];
	if(*alamda == 0.0) {								//Once converged evaluate
														//  covariance matrix with
		free(beta);										//  alamda = 0.
		free(da);
		free(atry);
		free(atryref);
		free_dmatrix(oneda, 1, mfit, 1, 1);
		return true;
		}
	for(j = 0; j < ma; j++) atry[j] = *a[j];
	for(j = 0; j < mfit; j++)							//Did the trial succeed ?
		atry[lista[j]] = *a[lista[j]] + da[j];
	mrqcof(x, y, z, ndata, atryref, ma, lista, mfit, covar, da, chisq, funcs);
	if(*chisq < ochisq) {								//Success, accept the new solution
		*alamda *= 0.1;
		ochisq=(*chisq);
		for(j = 0; j < mfit; j++) {
			for(k = 0; k < mfit; k++) alpha[j][k] = covar[j][k];
			beta[j] = da[j];
			*a[lista[j]] = atry[lista[j]];
			}
		}
	else {												//Failure, increase almda and
		*alamda *= 10.0;								//    return.
		*chisq = ochisq;
		}
	return true;
}

bool Check_MRQerror()
{
	bool bRet;

	if(bRet = MRQ_error != 0L) ErrorBox(MRQ_error);
	MRQ_error = 0L;
	return bRet;
}

//---------------------------------------------------------------------------
//Use heap sort to sort elements of an float array
//W.H. pres, B.P. Flannery, S.A. Teukolsky, W.T. Vetterling (1988/1989)
//Numerical Recipes in C, Cambridge University Press, ISBN 0-521-35465-X
// p. 245
void SortArray(int n, double *vals)
{
	int l, j, ir, i;
	double rra, *ra = vals-1;

	if(n < 2 || !vals) return;
	l=(n >> 1) + 1;				ir = n;
	for( ; ; ) {
		if(l > 1) rra = ra[--l];
		else {
			rra = ra[ir];		ra[ir] = ra[1];
			if(--ir == 1) {
				ra[1] = rra;	return;
				}
			}
		i = l;					j = l << 1;
		while (j <= ir) {
			if (j < ir && ra[j] < ra[j+1]) ++j;
			if (rra < ra[j]) {
				ra[i] = ra[j];	j += (i=j);
				}
			else j = ir + 1;
			}
		ra[i] = rra;
		}
}

//sorts array v1 making the corresponding rearrangement of v2
void SortArray2(int n, double *v1, double *v2)
{
	int l, j, ir, i;
	double rra, rrb, *ra = v1-1, *rb = v2-1;

	if(n < 2 || !v1 || !v2) return;
	l=(n >> 1) + 1;				ir = n;
	for( ; ; ) {
		if(l > 1) {
			rra = ra[--l];		rrb = rb[l];
			}
		else {
			rra = ra[ir];		rrb = rb[ir];
			ra[ir] = ra[1];		rb[ir] = rb[1];
			if(--ir == 1) {
				ra[1] = rra;	rb[1] = rrb;
				return;
				}
			}
		i = l;					j = l << 1;
		while (j <= ir) {
			if (j < ir && ra[j] < ra[j+1]) ++j;
			if (rra < ra[j]) {
				ra[i] = ra[j];	rb[i] = rb[j];
				j += (i=j);
				}
			else j = ir + 1;
			}
		ra[i] = rra;			rb[i] = rrb;
		}
}

//Use heap sort to sort elements of an xy array
void SortFpArray(int n, lfPOINT *vals)
{
	int l, j, ir, i;
	lfPOINT rra, *ra = vals-1;

	if(n < 2) return;
	l=(n >> 1) + 1;					ir = n;
	for( ; ; ) {
		if(l > 1) {
			rra.fx = ra[--l].fx; rra.fy = ra[l].fy;
			}
		else {
			rra.fx = ra[ir].fx;		rra.fy = ra[ir].fy;
			ra[ir].fx = ra[1].fx;	ra[ir].fy = ra[1].fy;	
			if(--ir == 1) {
				ra[1].fx = rra.fx;	ra[1].fy = rra.fy;
				return;
				}
			}
		i = l;					j = l << 1;
		while (j <= ir) {
			if (j < ir && ra[j].fx < ra[j+1].fx) ++j;
			if (rra.fx < ra[j].fx) {
				ra[i].fx = ra[j].fx;	ra[i].fy = ra[j].fy;
				j += (i=j);
				}
			else j = ir + 1;
			}
		ra[i].fx = rra.fx;				ra[i].fy = rra.fy;
		}
}

//---------------------------------------------------------------------------
// Cubic Spline Interpolation
// Ref.: W.H. Press, B.P. Flannery, S.A. Teukolsky, W.T. Vetterling (1989), 
//    Numerical Rcipies in C. The Art of Scientific Computing, 
//    Cambridge University Press, ISBN 0-521-35465, pp. 96 ff.
void spline(lfPOINT *v, int n, double *y2)
{
	int i, k;
	double p, qn, sig, un, *u;

	u = (double *)malloc(n * sizeof(double));
	y2[0] = u[0] = 0.0;
	for(i = 1; i < (n-1); i++) {
		sig = (v[i].fx-v[i-1].fx)/(v[i+1].fx-v[i-1].fx);
		p = sig*y2[i-1]+2.0;			y2[i]=(sig-1.0)/p;
		u[i]=(v[i+1].fy-v[i].fy)/(v[i+1].fx-v[i].fx)-(v[i].fy-v[i-1].fy)/(v[i].fx-v[i-1].fx);
		u[i]=(6.0*u[i]/(v[i+1].fx-v[i-1].fx)-sig*u[i-1])/p;
		}
	qn = un = 0.0;
	y2[n-1] = (un - qn * u[n-2])/(qn*y2[n-2]+1.0);
	for(k = n-2; k >= 0; k--) {
		y2[k] = y2[k]*y2[k+1]+u[k];
		}
	free(u);
}

//---------------------------------------------------------------------------
// Special Functions
// Ref.: W.H. Press, B.P. Flannery, S.A. Teukolsky, W.T. Vetterling (1989), 
//    Numerical Rcipies in C. The Art of Scientific Computing, 
//    Cambridge University Press, ISBN 0-521-35465, pp. 166 ff.

// The Gamma Function: return the ln(G(xx)) for xx > 0
double gammln(double xx)
{
	double x, tmp, ser;
	static double cof[6] = {76.18009173, -86.50532033, 24.01409822,
		-1.231739516, 0.120858003e-2, -0.536382e-5};
	int j;
	
	if(xx < 0.0) return 0.0;
	x = xx-1;		tmp = x + 5.5;		tmp -= (x + 0.5)*log(tmp);
	for (j = 0, ser = 1.0; j <= 5; j++) {
		x += 1.0;	ser += cof[j]/x;
		}
	return -tmp + log(2.50662827465*ser);
}

//The Factorial Function: return n!
double factrl(int n)
{
	static int ntop = 4;
	static double a[33]={1.0, 1.0, 2.0, 6.0, 24.0};
	int j;

	if(n < 0) return 0.0;		//error: no factorial for negative numbers
	if(n > 32) return exp(gammln(n+1.0));
	while(ntop < n) {			//fill in table up to desired value
		j = ntop++;		a[ntop]=a[j] * ntop;
		}
	return a[n];
}

//returns the incomplete gamma function evaluated by its series representation
void gser(double *gamser, double a, double x, double *gln)
{
	int n;
	double sum, del, ap;

	*gln = gammln(a);
	if(x <= 0) {
		*gamser = 0.0;			return;
		}
	else {
		ap = a;					del = sum = 1.0/a;
		for(n = 1; n <= 100; n++) {
			ap += 1.0;			del *= x/ap;		sum += del;
			if(fabs(del) <= fabs(sum) * _PREC) {
				*gamser = sum * exp(-x + a * log(x)-(*gln));
				return;
				}
			}
		// maximum number of iterations exceeded
		*gamser = sum * exp(-x + a * log(x)-(*gln));
		}

}

//returns the incomplete gamma function evaluated by its continued fraction representation
void gcf(double *gammcf, double a, double x, double *gln)
{
	int n;
	double gold=0.0, g, fac=1.0, b1=1.0, b0=0.0, anf, ana, an, a1, a0=1.0;

	*gln=gammln(a);		a1=x;
	for(n=1; n <= 100; n++) {
		an = (double)n;			ana = an -a;		a0 = (a1 + a0 * ana) * fac;
		b0 = (b1 + b0 * ana) *fac;					anf = an * fac;
		a1 = x * a0 + anf * a1;						b1 = x * b0 + anf * b1;
		if(a1) {
			fac = 1.0 / a1;							g = b1 * fac;
			if(fabs((g-gold)/g) <= _PREC) {
				*gammcf = exp(-x + a * log(x) -(*gln)) * g;
				return;
				}
			gold = g;
			}
		}
	// maximum number of iterations exceeded
	*gammcf = exp(-x + a * log(x) -(*gln)) * gold;
}

//returns the incomplete gamma function P(a,x)
double gammp(double a, double x)
{
	double gamser, gammcf, gln;

	if(x < 0.0 || a <= 0.0) return 0.0;
	if(x < (a+1.0)) {
		gser(&gamser, a, x, &gln);			return gamser;
		}
	else {
		gcf(&gammcf, a, x, &gln);			return 1.0-gammcf;
		}
	return 0.0;
}

//returns the complementary incomplete gamma function Q(a,x)
double gammq(double a, double x)
{
	double gamser, gammcf, gln;

	if(x < 0.0 || a <= 0.0) return 0.0;
	if(x < (a+1.0)) {
		gser(&gamser, a, x, &gln);			return 1.0-gamser;
		}
	else {
		gcf(&gammcf, a, x, &gln);			return gammcf;
		}
	return 0.0;
}

//continued fraction for incomplete beta function, used by betai()
double betacf(double a, double b, double x)
{
	double qap, qam, qab, em, tem, d, bz, bm = 1.0, bp, bpp, az = 1.0, am = 1.0, ap, app, aold;
	int m;

	qab = a+b;		qap = a+1.0;		qam = a-1.0;	bz = 1.0-qab*x/qap;
	for(m = 1; m <= 100; m++) {
		em = (double)m;			tem = em+em;
		d = em*(b-em)*x/((qam+tem)*(a+tem));
		ap = az + d * am;		bp = bz + d *bm;
		d = -(a+em)*(qab+em)*x/((qap+tem)*(a+tem));
		app = ap + d * az;		bpp = bp + d * bz;
		aold = az;				am = ap/bpp;
		bm = bp/bpp;			az = app/bpp;
		bz = 1.0;
		if(fabs(az-aold) <= (_PREC * fabs(az))) return az;	//success: return
		}
	return az;												//fail: iterations exceeded
}

//The incomplete beta function Ix(a,b) for 0 <= x <= 1
double betai(double a, double b, double x)
{
	double bt;

	if(x < 0.0 || x > 1.0) return 0.0;		//range !
	if(x == 0.0 || x == 1.0) bt = 0.0;
	else
		bt = exp(gammln(a+b)-gammln(a)-gammln(b)+a*log(x)+b*log(1.0-x));
	if(x < (a+1.0)/(a+b+2.0)) return bt * betacf(a, b, x)/a;
	else return 1.0 - bt * betacf(b, a, 1.0 - x)/b;
}

//the binomial coefficient
double bincof(double n, double k)
{
	if(n<0 || k<0 || k > n) return 0.0;
	return exp(gammln(n+1.0) - gammln(k+1.0) - gammln(n-k+1.0));
}

//the cumulative binomial distribution
double binomdistf(double k, double n, double p)
{
	if(k > n || n < 0.0 || p < 0.0 || p >1.0) return 0.0;
	return betai(n-k, k+1, p);
}

//the beta function
double betaf(double z, double w)
{
	return exp(gammln(z)+gammln(w)-gammln(z+w));
}

//the error function: not all compilers have a built in erf()
double errf(double x)
{
	return x < 0.0 ? -gammp(0.5, x*x) : gammp(0.5, x*x);
}

//the complementary error function
double  errfc(double x)
{
	double t, z, ans;

	z = fabs(x);
	t = 1.0/(1.0+0.5*z);
	ans = t * exp(-z*z-1.26551223+t*(1.00002368+t*(0.37409196+t*(0.09678418+
		t*(-0.18628806+t*(0.27886807+t*(-1.13520398+t*(1.48851587+
		t*(-0.82215223+t*0.17087277)))))))));
	return x >= 0.0 ? ans : 2.0-ans;
}

//cumulative normal distribution
double norm_dist(double x, double m, double s)
{
	return 0.5 + errf((x - m)/(s * _SQRT2))/2.0;
}

//chi square distribution
double chi_dist(double x, double df, double)
{
	return gammq(df/2.0, x/2);
}

//t-distribution
double t_dist(double t, double df, double)
{
	return betai(df/2.0, 0.5, (df/(df+t*t)));
}

//poisson distribution
double pois_dist(double x, double m, double)
{
	return gammq(x+1.0, m);
}

//f-distribution
double f_dist(double f, double df1, double df2)
{
	return betai(df2/2.0, df1/2.0, df2/(df2+df1*f));
}

//---------------------------------------------------------------------------
// Inverse of statitistical functions:
//    Use a combination of the Newton-Raphson method and bisection
// Ref.: W.H. Press, B.P. Flannery, S.A. Teukolsky, W.T. Vetterling (1989), 
//    Numerical Rcipies in C. The Art of Scientific Computing, 
//    Cambridge University Press, ISBN 0-521-35465, pp. 273 ff.

// funcd supplies the function value fn and the derivative df of the function sf at x
void funcd(double x, double *fn, double *df, double (*sf)(double, double, double), 
		   double df1, double df2, double p)
{
	double y1, y2;

	*fn = (sf)(x, df1, df2);
	y1 = (sf)(x * 0.995, df1, df2);
	y2 = (sf)(x * 1.005, df1, df2);
	*df = (y2-y1)*100.0/x;
	*fn = *fn - p;
}

//distinv does actual bisection and Newton-Raphson root finding
double distinv(double (*sf)(double, double, double), double df1, double df2, double p, double x0)
{
	int j;
	double df, dx, dxold, f, fh, fl;
	double swap, temp, xh, xl, rts; 
	double x1 = 0.0001, x2 = 10000;
	char info[80];

	funcd(x1, &fl, &df, sf, df1, df2, p);
	funcd(x2, &fh, &df, sf, df1, df2, p);
	for(j = 0; fl*fh >= 0 && j < 10; j++) {
		x1 /= 2.0;		x2 *= 2.0;
		funcd(x1, &fl, &df, sf, df1, df2, p);
		funcd(x2, &fh, &df, sf, df1, df2, p);
		}
	if(fl*fh >= 0) {
		sprintf(info, "Value for inverse distribution\nmust be between %g and %g!", x1, x2);
		InfoBox(info);
		return 0.0;
		}
	if(fl < 0.0) {
		xl = x1;		xh = x2;
		}
	else {
		xh = x1;		xl = x2;
		swap = fl;		fl = fh;	fh = swap;
		}
	rts = x0;	dxold = fabs(x2-x1);	dx = dxold;
	funcd(rts, &f, &df, sf, df1, df2, p);
	for(j = 1; j <= 100; j++) {
        if((((rts-xh)*df-f)*((rts-xl)*df-f) >= 0.0) || (fabs(2.0*f) > fabs(dxold * df))) {
			dxold = dx;		dx = 0.5 * (xh-xl);		rts = xl + dx;
			if(xl == rts) return rts;
			}
		else {
			dxold = dx;		dx = f/df;		temp = rts;		rts -= dx;
			if(temp == rts) return rts;
			}
		if(fabs(dx) < _PREC) return rts;
		funcd(rts, &f, &df, sf, df1, df2, p);
		if(f < 0.0) {
			xl = rts;	fl = f;
			}
		else {
			xh = rts;	fh = f;
			}
		}
	return 0.0;
}

//---------------------------------------------------------------------------
//some statistical basics
//do quartiles, median of data
void d_quartile(int n, double *v, double *q1, double *q2, double *q3)
{
	int n2, n3;
	double f1, f2;

	if(!v || n<2) return;
	SortArray(n, v);			n2 = n >> 1;
	if(q1) {
		n3 = n2 >> 1;
		switch(n%4) {
		case 3:		n3 ++;		f1 = 2.0;		f2 = 2.0;		break;
		case 2:		n3 ++;		f1 = 3.0;		f2 = 1.0;		break;
		case 1:		n3 ++;		f1 = 4.0;		f2 = 0.0;		break;
		default:	f1 = 1.0;	f2 = 3.0;						break;
			}
		*q1 = (f1*v[n3-1] + f2*v[n3])/4.0;
		}
	if(q2) {
		if(n & 1) *q2 = v[n2];
		else *q2 = (v[n2-1] + v[n2])/2.0;
		}
	if(q3) {
		n3 = n2 >> 1;
		switch(n%4) {
		case 3:		n3++;		f1 = 2.0;		f2 = 2.0;	break;
		case 2:		f1 = 3.0;	f2 = 1.0;					break;
		case 1:		f1 = 4.0;	f2 = 0.0;					break;
		default:	f1 = 1.0;	f2 = 3.0;					break;
			}
		n3 += n2;
		*q3 = (f2*v[n3-1] + f1*v[n3])/4.0;
		}
}

//do arithmethic mean
double d_amean(int n, double *v)
{
	int i;
	double sum;

	for(i = 0, sum = 0.0; i < n; i++) {
		sum += (v[i]);
		}
	return (sum/n);
}


//do geometric mean
double d_gmean(int n, double *v)
{
	int i;
	double sum;

	for(i = 0, sum = 0.0; i < n; i++) {
		if(v[i] <= 0.0) return 0.0;
		sum += log(v[i]);
		}
	return exp(sum/n);
}

//do harmonic mean
double d_hmean(int n, double *v)
{
	int i;

	double sum;

	for(i = 0, sum = 0.0; i < n; i++) {
		if(v[i] == 0.0) return 0.0;
		sum += 1.0/(v[i]);
		}
	return (n/sum);
}

//---------------------------------------------------------------------------
// Pearsons linear correlation
// Ref.: W.H. Press, B.P. Flannery, S.A. Teukolsky, W.T. Vetterling (1989), 
//    Numerical Rcipies in C. The Art of Scientific Computing, 
//    Cambridge University Press, ISBN 0-521-35465, pp. 503 ff.
double d_pearson(double *x, double *y, int n, char *dest, DataObj *data)
{
	int j, r, c;
	double yt, xt, t, df, res[3];
	double syy=0.0, sxy=0.0, sxx=0.0, ay=0.0, ax=0.0;
	AccRange *rD;

	for(j = 0;	j < n; j++) {				// find means
		ax += x[j];			ay += y[j];
		}
	ax /= n;			ay /= n;
	for(j = 0; j < n; j++) {				// correlation
		xt = x[j] - ax;		yt = y[j] - ay;
		sxx += xt*xt;		syy += yt*yt;		sxy += xt * yt;
		}
	res[0] = sxy/sqrt(sxx*syy);		//pearsons r
	if(dest) {
		res[1] = 0.5 * log((1.0+res[0]+_PREC)/(1.0-res[0]+_PREC));	//Fishers z-transform
		df = n-2;
		t = res[0]*sqrt(df/((1.0-res[0]+_PREC)*(1.0+res[0]+_PREC)));	//Student's t
		res[2] = betai(0.5*df, 0.5, df/(df+t*t));					//probability
		}
	if((dest) && (data) && (rD = new AccRange(dest))) {
		rD->GetFirst(&c, &r);
		for(j = 0; j < 3 && rD->GetNext(&c, &r); j++) {
			data->SetValue(r, c, res[j]);
			}
		data->Command(CMD_UPDATE, 0L, 0L);
		delete rD;
		}
	return res[0];
}

//---------------------------------------------------------------------------
// Spearman rank-order correlation
// Ref.: W.H. Press, B.P. Flannery, S.A. Teukolsky, W.T. Vetterling (1989), 
//    Numerical Recipies in C. The Art of Scientific Computing, 
//    Cambridge University Press, ISBN 0-521-35465, pp. 507 ff.

//Given a sorted array w, crank replaces the elements by their rank
void crank(int n, double *w0, double *s)
{
	int j=1, ji, jt;
	double t, rank, *w = w0-1;

	*s = 0.0;
	while (j < n) {
		if(w[j+1] != w[j]) {
			w[j] = j;		++j;
			}
		else {
			for(jt = j+1; jt <= n; jt++)
				if(w[jt] != w[j]) break;
			rank = 0.5 * (j+jt-1);
			for(ji = j; ji <= (jt-1); ji++) w[ji] = rank;
			t = jt -j;
			*s += t*t*t -t;
			j = jt;
			}
		}
	if(j == n) w[n] = n;
}

//the actual rank correlation
double d_spearman(double *x, double *y, int n, char *dest, DataObj *data)
{
	int j, r, c;
	double vard, t, sg, sf, fac, en3n, en, df, aved, tmp;
	double res[5];
	AccRange *rD;

	SortArray2(n, x, y);		crank(n, x, &sf);
	SortArray2(n, y, x);		crank(n, y, &sg);
	for(j = 0, res[0] = 0.0; j < n; j++) res[0] += ((tmp = (x[j]-y[j]))*tmp);
	en = n;						en3n = en*en*en -en;
	aved = en3n/6.0 - (sf+sg)/12.0;
	fac = (1.0-sf/en3n)*(1.0-sg/en3n);
	vard = ((en-1.0)*en*en*((en+1.0)*(en+1.0))/36.0)*fac;
	vard = ((en-1.0)*en*en*((tmp = (en+1.0))*tmp)/36.0)*fac;
	res[1] = (res[0]-aved)/sqrt(vard);
	res[2] = errfc(fabs(res[1])/_SQRT2);
	res[3] = (1.0-(6.0/en3n)*(res[0]+0.5*(sf+sg)))/fac;
	t = res[3]*sqrt((en-2.0)/((res[3]+1.0)*(1.0-res[3])));
	df = en-2.0;
    res[4] = betai(0.5*df, 0.5, df/(df+t*t));
	if((dest) && (data) && (rD = new AccRange(dest))) {
		rD->GetFirst(&c, &r);
		for(j = 0; j < 5 && rD->GetNext(&c, &r); j++) {
			data->SetValue(r, c, res[j]);
			}
		data->Command(CMD_UPDATE, 0L, 0L);
		delete rD;
		}
	return res[3];
}

//linear regression
double d_regression(double *x, double *y, int n, char *dest, DataObj *data)
{
	double sx, sy, dx, dy, sxy, sxx, syy, sdy;
	double res[10];		// slope, intercept, mean x, mean y, SE of slope, 
						//   variance(x), variance(y), variance(fit), F of regression 
	int i, j, r, c;
	AccRange *rD;

	if(n < 2) return 0.0;
	for(i = 0, 	sx = sy = 0.0; i < n; i++) {
		sx += x[i];			sy += y[i];
		}
	res[2] = sx /n;			res[3] = sy/n;
	sxy = sxx = syy = 0.0;
	for(i = 0; i < n; i++) {
		dx = x[i]-res[2];	dy = y[i]-res[3];
		sxx += (dx*dx);		syy += (dy*dy);		sxy += (dx*dy);
		}
	res[0] = sxy / sxx;		res[1] = res[3] - res[0] * res[2];
	for(i = 0, sdy = 0.0; i < n; i++) {
		dy = y[i] - (res[1] + x[i] *res[0]);
		sdy += (dy * dy);
		}
	sdy = sdy/(n-2);		res[4] = sqrt(sdy/sxx);
	res[5] = sxx/(n-1);		res[6] = syy/(n-1);			res[7] = sdy;
	res[8] = sxy/sdy*sxy/sxx;
	if((dest) && (data) && (rD = new AccRange(dest))) {
		rD->GetFirst(&c, &r);
		for(j = 0; j < 9 && rD->GetNext(&c, &r); j++) {
			data->SetValue(r, c, res[j]);
			}
		data->Command(CMD_UPDATE, 0L, 0L);
		delete rD;
		}
	return n;
}

//t-test
double d_ttest(double *x, double *y, int n1, int n2, char *dest, DataObj *data)
{
	int i, r, c;
	double sx, sy, mx, my, d, df, p;
	double res[9];			// mean1, SD1, n1, mean2, SD2, n2, p if variances equal,
	AccRange *rD;			//    corrected df, corrected p

	for(i=0, sx = 0.0; i < n1; sx += x[i], i++);				mx = sx/n1;
	for(i=0, sy = 0.0; i < n2; sy += y[i], i++);				my = sy/n2;
	for(i=0, sx = 0.0; i < n1; sx += ((d=(x[i]-mx))*d), i++);
	for(i=0, sy = 0.0; i < n2; sy += ((d=(y[i]-my))*d), i++);
    d = ((sx+sy)/(n1+n2-2)) * ((double)(n1+n2)/(double)(n1*n2));
	d = (mx-my)/sqrt(d);										//Student's t

	//Welch's correction for differences in variance
	df = (sx/(double)n1)*(sx/(double)n1)/(double)(n1+1)+(sy/(double)n2)*(sy/(double)n2)/(double)(n2+1);
	df = (sx/(double)n1+sy/(double)n2)*(sx/(double)n1+sy/(double)n2)/df;
	df -= 2.0;
	p = betai(df/2.0, 0.5, (df/(df+d*d)));
	if((dest) && (data) && (rD = new AccRange(dest))) {
		res[0] = mx;	res[1] = sqrt(sx/(double)(n1-1));	res[2] = n1;
		res[3] = my;	res[4] = sqrt(sy/(double)(n2-1));	res[5] = n2;
		res[7] = df;	df = (n1-1) + (n2-1);	res[6] = betai(df/2.0, 0.5, (df/(df+d*d)));
		res[8] = p;
		rD->GetFirst(&c, &r);
		for(i = 0; i < 9 && rD->GetNext(&c, &r); i++) {
			data->SetValue(r, c, res[i]);
			}
		data->Command(CMD_UPDATE, 0L, 0L);
		delete rD;
		}
	return p;
}

//f-test
double d_ftest(double *x, double *y, int n1, int n2, char *dest, DataObj *data)
{
	int i, r, c;
	double sx, sy, mx, my, d, df1, df2, p;
	double res[6];			// mean1, SD1, n1, mean2, SD2, n2
	AccRange *rD;

	for(i=0, sx = 0.0; i < n1; sx += x[i], i++);				mx = sx/n1;
	for(i=0, sy = 0.0; i < n2; sy += y[i], i++);				my = sy/n2;
	for(i=0, sx = 0.0; i < n1; sx += ((d=(x[i]-mx))*d), i++);	sx /= (n1-1);
	for(i=0, sy = 0.0; i < n2; sy += ((d=(y[i]-my))*d), i++);	sy /= (n2-1);
	d = sx/sy;		df1 = n1-1;		df2 = n2-1;
	p= 2.0 * betai(df2/2.0, df1/2.0, df2/(df2+df1*d));
	if((dest) && (data) && (rD = new AccRange(dest))) {
		res[0] = mx;	res[1] = sqrt(sx);	res[2] = n1;
		res[3] = my;	res[4] = sqrt(sy);	res[5] = n2;
		rD->GetFirst(&c, &r);
		for(i = 0; i < 6 && rD->GetNext(&c, &r); i++) {
			data->SetValue(r, c, res[i]);
			}
		data->Command(CMD_UPDATE, 0L, 0L);
		delete rD;
		}
	return p;
}
