//UtilObj.cpp, (c)2000, 2001, 2002, 2003, 2004 by R. Lackner
//
//    This file is part of RLPlot.
//
//    RLPlot is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    RLPlot is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with RLPlot; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
#include "rlplot.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <fcntl.h>				//file open flags
#include <sys/stat.h>			//I/O flags
#ifdef _WINDOWS
	#include <io.h>					//for read/write
#else
	#define O_BINARY 0x0
	#include <unistd.h>
#endif

Default defs;

static LineDEF ETbgnn = {0.0f, 1.0f, 0x00d8d8d8L, 0L};
static LineDEF ETbgna = {0.0f, 1.0f, 0x00ffffffL, 0L};
static LineDEF ETbgmn = {0.0f, 1.0f, 0x00000000L, 0L};
static LineDEF ETbgma = {0.0f, 1.0f, 0x00ffff00L, 0L};
extern const LineDEF BlackLine = {0.0f, 1.0f, 0x00000000L, 0L};

static FillDEF ETfbnn = {FILL_NONE, 0x00d8d8d8L, 1.0f, NULL, 0x00ffffffL};
static FillDEF ETfbna = {FILL_NONE, 0x00ffffffL, 1.0f, NULL, 0x00ffffffL};
static FillDEF ETfbmn = {FILL_NONE, 0x00000000L, 1.0f, NULL, 0x00ffffffL};
static FillDEF ETfbma = {FILL_NONE, 0x00ffff00L, 1.0f, NULL, 0x00ffffffL};

extern char TmpTxt[500];
extern unsigned long cObsW;				//count objects written
extern GraphObj *CurrGO, *TrackGO;		//Selected Graphic Objects
extern dragHandle *CurrHandle;
extern UndoObj Undo;

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Process fields with user input text
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
EditText *CurrText = 0L;

EditText::EditText(void *par, POINT where, POINT right, char *msg)
{
	loc.x = where.x;				loc.y = where.y;
	crb.x = rb.x = right.x;			crb.y = rb.y = right.y;
	if(msg && msg[0]) text = strdup(msg);
	else text = 0L;						Value = 0.0;
	CursorPos = length = Align = 0;		type = ET_UNKNOWN;		TextCol=0x00000000L;
	bgLine = &ETbgnn;					bgFill = &ETfbnn;		parent = par;
	FindType();
	m1 = m2 = -1;						//cursor positions track marks
}

EditText::EditText(void *par, char *msg)
{
	loc.x = loc.y = crb.x = rb.x = crb.y = rb.y = 0;	Value = 0.0;
	if(msg && msg[0]) text = strdup(msg);
	else text = 0L;
	CursorPos = length = Align = 0;		type = ET_UNKNOWN;		TextCol=0x00000000L;
	bgLine = &ETbgnn;					bgFill = &ETfbnn;		parent = par;
	FindType();
	m1 = m2 = -1;						//cursor positions track marks
}

EditText::~EditText()
{
	if(CurrText == this)	CurrText = 0L;
	if(text) free(text);	text = 0L;
}

bool
EditText::AddChar(int ci, anyOutput *Out)
{
	char byte1, byte2, c, *tmp;
	POINT MyPos;
	int i;

	if(ci < 254 && ci > 31) c = (char)ci;
	else if(ci == 27) {						//Esc
		m1 = m2 = -1;		Redraw(Out, true);		return true;
		}
	else return false;
	if(text)length = strlen(text);
	else length = 0;
	if(text) tmp = (char *)realloc(text, length+2);
	else tmp = (char *)calloc(2, sizeof(char));
	if(!tmp) return false;
	text = tmp;
//	isModified = TRUE;
	byte1 = byte2 = 0;
	//replace mark by character if mark exists
	if(m1 > -1 && m2 > -1) Command(CMD_DELETE, (anyOutput *)0L, (DataObj *)0L);
	byte1 = text[CursorPos];
	i = CursorPos;
	text[i++] = c;
	while(byte1) {
		byte2 = byte1;			byte1 = text[i];			text[i++] = byte2;
		}
	text[i] = byte1;
	CursorPos++;
	Redraw(Out, true);
	MyPos.y = loc.y;
	MyPos.x = Align & TXA_HRIGHT ? crb.x - 2 : loc.x + 2;
	if(Out)Out->TextCursor(text, MyPos, (POINT *) NULL, &CursorPos);
	return true;
}

void
EditText::Update(int select, anyOutput *Out, POINT *MousePos)
{
	POINT MyPos;

	if(select != 1 && select != 5) m1 = m2 = -1;		//no mark;
	switch(select) {
		case 0:							//just redraw with current settings
			Redraw(Out, true);
			break;
		case 5:							//dialog control
			if(!text)Align = TXA_VTOP | TXA_HLEFT;
		case 1:							//active spread sheet cell with cursor
			if(!text && !(text = (char *) calloc(2, sizeof(char))))return;
			if(CursorPos > (int)strlen(text)) CursorPos = (int)strlen(text);
			bgLine = &ETbgna; bgFill = &ETfbna; TextCol = 0x00000000L;
			Redraw(Out, true);
			MyPos.y = loc.y;
			MyPos.x = Align & TXA_HRIGHT ? crb.x - 2 : loc.x + 2;
			if(MousePos && MousePos->x && MousePos->y) Out->TextCursor(text,MyPos,MousePos,&CursorPos);
			else if(select ==1) Out->TextCursor(text, MyPos, NULL, &CursorPos);
			break;
		case 2:							//inactive spreadsheet cell
			if(crb.x > rb.x) {
				crb.x = rb.x;	crb.y = rb.y;
				}
			if(CurrText == this) FindType();
			bgLine = &ETbgnn; bgFill = &ETfbnn; TextCol = 0x00000000L;
			Redraw(Out, true);
			break;
		case 10:						//value filled in by external app.
			if(text && text[0]) {
				type = ET_VALUE;
				Align = TXA_VTOP | TXA_HRIGHT;
				sscanf(text, "%lf", &Value);
				}
			break;
		case 20:						//update value only
			FindType();
			break;
		}
}

bool
EditText::Redraw(anyOutput *Out, bool display)
{
	RECT rc;
	POINT MyPos;
	char *txt, tmptxt[40];
	int w, h, o_crbx;

	o_crbx = crb.x;			crb.x = rb.x;				crb.y = rb.y;
	if (Out) {
		Out->TxtSet.Align = Align;
		Out->TxtSet.ColTxt = TextCol;
		Out->TxtSet.ColBg = bgLine->color;
		if(CurrText == this && text && text[0]) {
			Out->oGetTextExtent(text, strlen(text), &w, &h);
			while((crb.x - loc.x) < (w+(h>>1))) crb.x += (rb.x - loc.x);
			if(o_crbx > loc.x && o_crbx > crb.x && o_crbx < 4000) crb.x = o_crbx;
			}
		Out->SetFill(bgFill);
		Out->SetLine(bgLine);
		rc.left = loc.x;			rc.right = crb.x;
		rc.top = loc.y;				rc.bottom = crb.y;
		Out->oRectangle(loc.x, loc.y, crb.x-1, crb.y-1);
		MyPos.y = loc.y;
		if(Align & TXA_HRIGHT) {	//right justified text
			MyPos.x = crb.x-2;
			}
		else {						//left justified text
			MyPos.x = loc.x +2;
			}
		if(!text && (type & 0xff) == ET_VALUE){
			sprintf(tmptxt, "%g", Value);
			text = strdup(tmptxt);
			}
		if(text && text[0]){
			if((type & 0xff) == ET_FORMULA && (bgLine == &ETbgnn || bgLine == &ETbgmn)) {
				if(do_formula((DataObj*)parent, text+1, &Value)) sprintf(tmptxt, "%g", Value);
				else sprintf(tmptxt, "#ERROR");
				txt = tmptxt;
				}
			else txt = text;
			Out->oTextOut(MyPos.x, loc.y, txt, 0);
			}
		if(display) {
			if(!(Out->UpdateRect(&rc, false))) return false;
			if(m1 != m2 && m1 >=0 && m2 >=0) {
				if (m1 >m2) Swap(m1, m2);
				rc.left = mx1;		rc.right = mx2;
				Out->ShowMark(&rc, MRK_INVERT);
				Out->MrkMode = MRK_NONE;
				}
			}
		return true;
	}
	return false;
}

void
EditText::Mark(anyOutput *Out, int mark)
{
	LineDEF *ol = bgLine;
	FillDEF *of = bgFill;
	DWORD ocol = TextCol;

	m1 = m2 = -1;
	switch (mark){
	case 0:				//normal not active
		bgLine = &ETbgnn; bgFill = &ETfbnn; TextCol = 0x00000000L;
		break;
	case 1:				//normal active
		bgLine = &ETbgna; bgFill = &ETfbna; TextCol = 0x00000000L;
		break;
	case 2:				//mark not active
		bgLine = &ETbgmn; bgFill = &ETfbmn; TextCol = 0x00ffffffL;
		break;
	case 3:				//mark active
		bgLine = &ETbgma; bgFill = &ETfbma; TextCol = 0x00ff0000L;
		break;
		}
	Redraw(Out, true);
	bgLine = ol;	bgFill = of;	TextCol = ocol;
}

bool
EditText::Command(int cmd, anyOutput *Out, void *data_obj)
{
	int i, j, w, h;
	POINT MyPos;
	MouseEvent *mev;
	static RECT rMark;
	bool bRet;

	MyPos.y = loc.y;
	MyPos.x = Align & TXA_HRIGHT ? crb.x - 2 : loc.x + 2;
	if(!(text)) return false;
	switch(cmd) {
		case CMD_BACKSP:
			if(CursorPos <=0){
				Out->TextCursor(text, MyPos, (POINT *) NULL, &CursorPos);
				return false;
				}
			CursorPos--;						//continue as if delete
		case CMD_DELETE:
			bRet = false;
			if(m1 > -1 && m2 > -1) {			//delete marked part of text
				if (!text || !text[0]) return false;
				if(m1 > m2) Swap(m1, m2);
//         		isModified = TRUE;
				if(m2 >= (short int)strlen(text)) text[m1] = 0;
				else strcpy(text+m1, text+m2);
				CursorPos = m1;
				m1 = m2 = -1;
				if(Out) Redraw(Out, (bRet = true));
				}
			else if(text[CursorPos]) {
//         		isModified = TRUE;
				strcpy(text + CursorPos, text + CursorPos + 1);
				if(Out)Redraw(Out, (bRet = true));
				}
			if(Out)Out->TextCursor(text, MyPos, (POINT *) NULL, &CursorPos);
			return bRet;
		case CMD_SHIFTRIGHT:
			if(CursorPos == m1 && text[m1]) m1++;
			else if(CursorPos == m2 && text[m2]) m2++;
			else if(text[CursorPos]){
				m1 = CursorPos;	m2 = CursorPos+1;
				}
			if(text[CursorPos]) CursorPos++;
		case CMD_SHIFTLEFT:
			if(cmd == CMD_SHIFTLEFT) {
				if(CursorPos == m1 && m1 >0) m1--;
				else if(CursorPos == m2 && m2 >0) m2--;
				else if(CursorPos > 0){
					m1 = CursorPos;	m2 = CursorPos-1;
					}
				if(CursorPos >0) CursorPos--;
				}
			if(m1 >=0 && m2 >= 0 && m1 != m2 && Out) {
				if(m1 > m2) Swap(m1, m2);
				w = h = 0;
				if(Align & TXA_HRIGHT) {	//right justified text
					Out->oGetTextExtent(text, 0, &w, &h);
					mx1 = crb.x-2 - w;
					}
				else {						//left justified text
					mx1 = loc.x +2;
					}
				Out->oGetTextExtent(text, m1, &w, &h);
				mx1 += (m1 ? w : 0);
				Out->oGetTextExtent(text+m1, m2-m1, &w, &h);
				mx2 = mx1 + w;
				}
			HideTextCursor();		Redraw(Out, true);
			Out->TextCursor(text, MyPos, (POINT *) NULL, &CursorPos);
			break;
		case CMD_CURRLEFT:
			m1 = m2 = -1;
			if(CursorPos >0) {
				CursorPos--;
				if(Redraw(Out, true) && Out->TextCursor(text, MyPos, (POINT *) NULL,
					&CursorPos)) return true;
				else return false;
				}
			else if (data_obj) {
				MyPos.x = loc.x-2;			MyPos.y = (rb.y+loc.y)/2;
				if(((DataObj*)data_obj)->Select(&MyPos))return true;
				MyPos.x = loc.x+2;
				((DataObj*)data_obj)->Select(&MyPos);
				}
			return false;
		case CMD_CURRIGHT:
			m1 = m2 = -1;
			if(text[CursorPos]){
				CursorPos++;
				if(Redraw(Out, true) && Out->TextCursor(text, MyPos, (POINT *) NULL,
					&CursorPos)) return true;
				else return false;
				}
			else if (data_obj) {
				MyPos.x = rb.x+2;		MyPos.y = (rb.y+loc.y)/2;	crb.x = rb.x;
				if(((DataObj*)data_obj)->Select(&MyPos)) return true;
				MyPos.x = rb.x-2;
				((DataObj*)data_obj)->Select(&MyPos);
				}
			return false;
		case CMD_POS_FIRST:
		case CMD_POS_LAST:
			CursorPos = (cmd == CMD_POS_LAST && text && text[0]) ? strlen(text) : 0;
			m1 = m2 = -1;
			Redraw(Out, true);
			Out->TextCursor(text, MyPos, (POINT *) NULL, &CursorPos);
			return true;
		case CMD_CURRDOWN:		case CMD_CURRUP:
			if (data_obj) {
			//the following calculation of the cursor position is crude
            //it is based on a aspect of 2:1 for digits
				if(Align & TXA_HRIGHT)		//right justified text
					MyPos.x = rb.x-2-((rb.y-loc.y-4)*(strlen(text)-CursorPos))/2;
				else MyPos.x = loc.x+2+((rb.y-loc.y-4)*CursorPos)/2;
				MyPos.y = (cmd == CMD_CURRUP) ? loc.y-2 : rb.y+2;
				if(MyPos.x < loc.x) MyPos.x = loc.x +2;
				if(MyPos.x > rb.x) MyPos.x = rb.x -2;
				if(((DataObj*)data_obj)->Select(&MyPos))return true;
				MyPos.y = rb.y;
				((DataObj*)data_obj)->Select(&MyPos);
				}
			return false;
		case CMD_MOUSE_EVENT:					//track left mouse button
			mev = (MouseEvent*) data_obj;
			if(!text || !text[0]) return false;
			if(mev->x <loc.x || mev->x >crb.x || mev->y <loc.y || mev->y >rb.y)return false;
			if(mev->Action == MOUSE_LBDOWN) {
				m1 = m2 = -1;
				return true;
				}
			if(mev->Action == MOUSE_LBDOUBLECLICK) {
				rMark.top = loc.y;				rMark.bottom = rb.y;
				if(!Out->oGetTextExtent(text, strlen(text), &w, &h)) return false;
				m1 = 0;							m2 = strlen(text);
				if(Align & TXA_HRIGHT) {		//right justfied text
					rMark.right = crb.x -2;		rMark.left = crb.x - w - 2;
					}
				else {							//left justified text
					rMark.left = loc.x +2;		rMark.right = rMark.left +w;
					}
				Out->UpdateRect(&rMark,true);
				return true;
				}
			MyPos.x = Align & TXA_HRIGHT ? mev->x + 2 : mev->x - 2;
			MyPos.y = mev->y;
			Out->TxtSet.Align = Align;
			j = Out->CalcCursorPos(text, Align & TXA_HRIGHT ? crb :loc, &MyPos);
			if(j == m1 || j == m2) return true;
			if(Align & TXA_HRIGHT) {			//right justfied text
				if((i = strlen(text)-j)){
					if(!Out->oGetTextExtent(text+j, i, &w, &h)) return false;
					w = crb.x - w - 2;
					}
				else w = crb.x-1;
				}
			else {								//left justified text
				if(!j) w = 0;
				else if(!Out->oGetTextExtent(text, j, &w, &h))return false;
				w += (loc.x+2);
				}
			if(m1 == m2 && m1 == -1) {
				mx1 = (short)(rMark.left = w);
				m1 = j;
				}
			else if(j != m2){
				m2 = j;
				if(m2 >= 0)Out->UpdateRect(&rMark, false);
				mx2 = (short)(rMark.right = w);
				rMark.top = loc.y;
				rMark.bottom = rb.y;
				if(rMark.right < crb.x && rMark.right > loc.x &&
					rMark.left > loc.x && rMark.left < crb.x)
					Out->UpdateRect(&rMark,true);
				}
			if(m1 >= 0 && m2 >= 0 && m1 != m2 && parent)
				//remove range-mark of data 
				((DataObj*)parent)->Command(CMD_UNLOCK, 0L, 0L);
			return true;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// return the value (i.e. the floating point equivalent) of text
bool
EditText::GetValue(double *v)
{
	if(!text) {
		if((type & 0xff) == ET_VALUE) {
			*v = Value;		return true;
			}
		return false;
		}
	if(CurrText == this && !(type & ET_BUSY)) FindType();
	if((type & 0xff) == ET_VALUE){
		*v = Value;			return true;
		}
	if((type & 0xff) == ET_FORMULA && text && text[0]){
		if(!(type & ET_BUSY)){
			type |= ET_BUSY;
			if(do_formula((DataObj*)parent, text+1, &Value)) {
				*v = Value;		type &= ~ET_BUSY;
				return true;
				}
			type &= ~ET_BUSY;
			return false;
			}
		else if(parent) {
			((DataObj*)parent)->Command(CMD_ERROR, (void*)"circular reference in formula", 0L);
			}
		*v = Value;
		return true;
		}
	return false;
}

bool
EditText::GetText(char *t, int size)
{
	if(text) {
		if((int)strlen(text) < size) strcpy(t, text);
		else {
			memcpy(t, &text, size-1);			t[size-1] = 0;
			}
		return true;
		}
	if((type & 0xff) == ET_VALUE) {
		sprintf(TmpTxt, "%lf", Value);
		text = strdup(TmpTxt);
		return GetText(t, size);
		}
	return false;
}

bool
EditText::SetValue(double v)
{
	if(text){
		free(text);		text = 0L;
		}
	Value = v;	type = ET_VALUE;
	return true;
}

bool
EditText::SetText(char *t)
{
	if(text){
		free(text);		text = 0L;
		}
	Value = 0.0;	type = ET_UNKNOWN;
	if(t && t[0]){
		text = strdup(t);
		FindType();
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// return the ASCII reprentation of value of text if applicable. If not return
//    the text with double quotes: comma separated file syntax.
bool
EditText::GetItem(char *dest, int size)
{
	char tmp[50];
	int i;

	if(!text || !text[0] || !dest || size < 3) return false;
	if(CurrText == this && Txt2Flt(text, &Value)) type = ET_VALUE;
	//its a value
	if((type & 0xff) == ET_VALUE){
		if(text && strlen(text) < (sizeof(tmp)-1)){
			strcpy(tmp, text);
			if(defs.DecPoint[0] != '.') {
				for(i = 0; tmp[i]; i++) 
					if(tmp[i] == defs.DecPoint[0]) tmp[i]='.';
				}
			}
		else {
			sprintf(tmp, "%f", Value);			RmTrail(tmp);
			}
		for(i = 0; i < (size-1) && tmp[i]; i++) dest[i] = tmp[i];
		dest[i] = 0;
		}
	//else its a string
	else {
		dest[0] = '"';
		for(i = 1; i < (size-2) && text[i-1]; i++) dest[i] = text[i-1];
		dest[i++] = '"';
		dest[i] = 0;
		}
	return true;
}

void
EditText::SetRec(RECT *rc)
{
	loc.x = rc->left;				loc.y = rc->top;
	crb.x = rb.x = rc->right;		crb.y = rb.y = rc->bottom;
}

void
EditText::FindType()
{
	if(text && text[0] == '=') {
		Align = TXA_VTOP | TXA_HRIGHT;
		type = ET_FORMULA;
		}
	else if(text && (Txt2Flt(text, &Value))) {
		Align = TXA_VTOP | TXA_HRIGHT;
		type = ET_VALUE;
		}
	else if (text && text[0]) {
		Align = TXA_VTOP | TXA_HLEFT;
		type = ET_TEXT;
		}
	else {
		Align = TXA_VTOP | TXA_HRIGHT;
		type = ET_UNKNOWN;
		}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// the basic data object
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
DataObj::DataObj()
{
	cRows = cCols = 0;
	etRows = 0L;
}

DataObj::~DataObj()
{
	FlushData();
}

bool
DataObj::Init(int nR, int nC)
{
	int i, j;

	if(etRows)FlushData();
	if(!(etRows = (EditText ***)calloc (cRows = nR, sizeof(EditText **)))) return false;
	for(i = 0, cCols = nC; i < cRows; i++) {
		if(!(etRows[i] = (EditText **)calloc(cCols, sizeof(EditText *)))) {
			FlushData();	return false;
			}
		if(etRows[i]) for(j = 0; j < cCols; j++) {
			etRows[i][j] = new EditText(this, 0L);
			}
		}
	return true;
}

bool
DataObj::SetValue(int row, int col, double val)
{
	if(row < 0 || row >= cRows || col < 0 || col >= cCols) return false;
	if(etRows[row][col]) return etRows[row][col]->SetValue(val);
	return false;
}

bool
DataObj::SetText(int row, int col, char *txt)
{
	if(row < 0 || row >= cRows || col < 0 || col >= cCols) return false;
	if(etRows[row][col]) return etRows[row][col]->SetText(txt);
	return false;
}

bool
DataObj::GetValue(int row, int col, double *v)
{
	if(row < 0 || row >= cRows || col < 0 || col >= cCols) return false;
	if(etRows[row][col]) return etRows[row][col]->GetValue(v);
	return false;
}

bool
DataObj::GetText(int row, int col, char *txt, int len)
{
	if(!cRows || !cCols || !etRows) return false;
	if(row < 0 || row >= cRows || col < 0 || col >= cCols) return false;
	if(txt && etRows[row][col]) return etRows[row][col]->GetText(txt, len);
	return false;
}

bool
DataObj::GetSize(int *width, int *height)
{
	if(width)*width = cCols;		if(height)*height = cRows;
	return true;
}

void
DataObj::FlushData()
{
	int i, j;

	if(etRows){
		for(i = 0; i < cRows; i++) if(etRows[i]) {
			for (j = 0; j< cCols; j++) if(etRows[i][j]) delete etRows[i][j];
			free(etRows[i]);
			}
		free(etRows);
		}
	etRows = 0L;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// The notary class handles different types of supervision and indexing
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
notary::notary()
{
	gObs = 0L;
	goStack = 0L;
	NextPopGO = NextPushGO = NextRegGO = 0L;
}

notary::~notary()
{
	FreeStack();
}

unsigned long
notary::RegisterGO(GraphObj *go)
{
	int i, j;

	if(!go) return 0L;
	if(!gObs) {
		gObs = (GraphObj ***)calloc(0x2000L, sizeof(GraphObj **));
		gObs[0] = (GraphObj **)calloc(0x2000L, sizeof(GraphObj *));
		if(gObs && gObs[0]) {
			gObs[0][0] = go;
			return 1L;
			}
		return 0L;
		}
	i = (int)(NextRegGO >> 13);
	j = (int)(NextRegGO & 0x1fff)+1;
	if(j >=0x2000){
		i++;	j = 0;
		}
	if(gObs[i] && gObs[i][j] && gObs[i][j] == go) {
		NextRegGO = ((i << 13) | j);
		return (unsigned long)i*0x2000L+j+1;
		}
	if(gObs && gObs[0]) {
		for(i = 0; i < 0x2000; i++) {
			for(j = 0; j < 0x2000L; j++) {
				if(gObs[i][j] == go) {
					NextRegGO = ((i << 13) | j);
					return (unsigned long)i*0x2000L+j+1;
					}
				if(!gObs[i][j]) {
					gObs[i][j] = go;
					NextRegGO = ((i << 13) | j);
					return (unsigned long)i*0x2000L+j+1;
					}
				}
			if(i < 0x1fffL && !gObs[i+1])
				gObs[i+1] = (GraphObj **)calloc(0x2000L, sizeof(GraphObj *));
			if(i < 0x1fffL && !gObs[i+1]) return 0L;
			}
		}
	return 0L;
}

void
notary::AddRegGO(GraphObj *go)
{
	int i, j;

	if(!go) return;
	if(!gObs) {
		gObs = (GraphObj ***)calloc(0x2000L, sizeof(GraphObj **));
		gObs[0] = (GraphObj **)calloc(0x2000L, sizeof(GraphObj *));
		if(gObs && gObs[0]) {
			gObs[0][0] = go;
			return;
			}
		return;
		}
	i = (int)(NextRegGO >> 13);
	j = (int)(NextRegGO & 0x1fff)+1;
	if(j >=0x2000){
		i++;
		j = 0;
		}
	if(!gObs[i]) gObs[i] = (GraphObj **)calloc(0x2000L, sizeof(GraphObj *));
	if(gObs[i] && !gObs[i][j]) {
		gObs[i][j] = go;
		NextRegGO = ((i << 13) | j);
		}
	else RegisterGO(go);
}

bool
notary::PushGO(unsigned long id, GraphObj *go)
{
	int i, j;

	NextPopGO = 0L;
	if(!go) return true;
	go->Id = id;
	if(!goStack) {
		goStack = (GraphObj ***)calloc(8192, sizeof(GraphObj **));
		goStack[0] = (GraphObj **)calloc(8192, sizeof(GraphObj *));
		if(goStack && goStack[0]) {
			goStack[0][0] = go;
			return true;
			}
		return false;
		}
	i = (int)(NextPushGO >> 13);
	j = (int)(NextPushGO & 0x1fff)+1;
	if(j >=0x2000){
		i++;
		j = 0;
		}
	if(!goStack || !goStack[0]) return false;
	if(goStack[i] && !goStack[i][j]) {
		goStack[i][j] = go;
		NextPushGO = ((i << 13) | j);
		return true;
		}
	for(i = 0; i < 0x2000; i++) {
		for(j = 0; j < 0x2000; j++) {
			if(!goStack[i][j]) {
				goStack[i][j] = go;
				NextPushGO = ((i << 13) | j);
				return true;
				}
			}
		if(i < 0x1fff && !goStack[i+1] && !(goStack[i+1] =
			(GraphObj **)calloc(0x2000, sizeof(GraphObj *)))) return false;
		}
	return false;
}

GraphObj *
notary::PopGO(unsigned long id)
{
	int i, j;
	GraphObj *go;

	NextPushGO = 0L;
	if(!id || !goStack || !goStack[0]) return 0L;
	i = (int)(NextPopGO >> 13);
	j = (int)(NextPopGO & 0x1fff)+1;
	if(j >=0x2000){
		i++;
		j = 0;
		}
	if(goStack[i] && goStack[i][j] && goStack[i][j]->Id == id) {
		go = goStack[i][j];
		goStack[i][j] = 0L;
		go->Id = 0L;
		NextPopGO = ((i << 13) | j);
		return go;
		}
	for(i = 0; i < 0x2000; i++) {
		for(j = 0; j < 0x2000; j++) {
			if(goStack[i][j] && goStack[i][j]->Id == id) {
				go = goStack[i][j];
				goStack[i][j] = 0L;
				go->Id = 0L;
				NextPopGO = ((i << 13) | j);
				return go;
				}
			}
		if(i < 0x1fff && !goStack[i+1]) return 0L;
		}
	return 0L;
}

void
notary::FreeStack()
{
	int i, j, k;

	if(gObs) {
		for(i = 0; gObs[i] && i <8192; i++) free(gObs[i]);
		free(gObs);
		gObs = 0L;
		}
	if(goStack) {
		for(i = 0; goStack[i] && i <8192; i++){
			for(j = k = 0; j < 8192; j++){
				if(goStack[i][j]) {
					goStack[i][j]->Id = 0L;
					DeleteGO(goStack[i][j]);
					k++;
					}
				}
			free(goStack[i]);
			}
		free(goStack);
		if(k){
			sprintf(TmpTxt,"%d objects deleted\nby notary", k);
			ErrorBox(TmpTxt);
			}
		}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Calculate continuous index to a range given by an ASCII string
// string examples include  "a1:a12"  or  "a1:a4;a12:a24"
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
AccRange::AccRange(char *asc)
{
	int i, j;

	if(asc && *asc){
		i = strlen(asc)+2;
		txt = (char *)malloc(i);
		for(i = j = 0; i< (int)strlen(asc); i++)
			if(asc[i] > 32) txt[j++] = asc[i];
		txt[j] = 0;
		}
	else txt = 0L;
	x1 = y1 = x2 = y2 = 0;
}

AccRange::~AccRange()
{
	if(txt) free(txt);
}

int
AccRange::CountItems()
{
	int RetVal;

	RetVal = 0;
	if(txt && Reset())	do {
		RetVal += ((x2-x1+1)*(y2-y1+1));
		} while((curridx < (int)strlen(txt)) && Parse(curridx));
	return RetVal;
}

bool
AccRange::GetFirst(int *x, int *y)
{
	if(txt && Reset()) {
		if(x && y) {*x = x1; *y = y1;}
		return true;
		}
	return false;
}

bool
AccRange::GetNext(int *x, int *y)
{
	if(txt && x && y) {
		if(cx <= x2) {*x = cx; *y = cy; cx++; return true;}
		else {
			cx = x1; cy++;
			if(cy <= y2) return GetNext(x, y);
			else if(txt[curridx]){
				if(Parse(curridx)) return GetNext(x, y);
				return false;
				}
			}
		}
	return false;
}

bool
AccRange::IsInRange(int x, int y)
{
	if(txt && Reset())	do {
		if(x >= x1 && x <= x2 && y >= y1 && y <= y2) return true;
		} while((curridx < (int)strlen(txt)) && Parse(curridx));
	return false;
}

bool
AccRange::BoundRec(RECT *rec)
{
	if(txt && Reset()){
		SetMinMaxRect(rec, x1, y1, x2, y2);
		while((curridx < (int)strlen(txt)) && Parse(curridx)) {
			UpdateMinMaxRect(rec, x1, y1);	UpdateMinMaxRect(rec, x2, y2);
			}
		return true;
		}
	return false;
}

bool
AccRange::Reset()
{
	curridx = 0;
	return Parse(curridx);
}

bool
AccRange::Parse(int start)
{
	int i, step, *v;

	i = start;
	if(!txt) return false;
	if(txt[i] == ';') i++;
	if(!txt[i]) return false;
	step = x1 = y1 = x2 = y2 = 0;
	v = &x1;
	for ( ; i < (int)strlen(txt)+1; i++) {
		if(txt[i] == '$') i++;
		switch(step) {
		case 0:
		case 2:
			if((txt[i] >= 'a') && (txt[i] <= 'z')){
				*v *= 26;
				*v += (txt[i]-'a'+1);
				}
			else if((txt[i] >= 'A') && (txt[i] <= 'Z')){
				*v *= 26;
				*v += (txt[i]-'A'+1);
				}
			else if((txt[i] >= '0') && (txt[i] <= '9')){
				v = step == 0 ? &y1 : &y2;
				*v = txt[i]-'0';
				step++;
				}
			else return false;
			break;
		case 1:
		case 3:
			if((txt[i] >= '0') && (txt[i] <= '9')){
				*v *= 10;
				*v += (txt[i]-'0');
				}
			else if((txt[i] >= 'a') && (txt[i] <= 'z') ||
				(txt[i] >= 'A') && (txt[i] <= 'Z')){
				if(step == 1) v =  &x2;
				else return false;
				*v = txt[i] >='a' && txt[i] <= 'z' ? 
					txt[i]-'a' : txt[i]-'A';
				step++;
				}
			else if(step == 1 && (txt[i] == ':')) {
				v = &x2;
				step++;
				}
			else if((txt[i] == ';') || (txt[i] == ',') || (txt[i] == 0)) {
				if(step == 1) {		//one single cell selected
					x2 = x1;	y2 = y1;
					}
				if(x2<x1) Swap(x1,x2);		if(y2<y1) Swap(y1,y2);
				if(y1 >0) y1--;				if(y2 >0) y2--;
				if(x1 >0) x1--;				if(x2 >0) x2--;
				curridx = i;
				cx = x1; cy = y1;
				return true;
				}
			break;
			}
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Default data vault
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Default::Default()
{
	dUnits = cUnits = 0;
	strcpy(DecPoint, ".");		strcpy(ColSep, ",");
	Line_0.width = .4;		Line_1.width = .04,		Line_2.width = 0.016;
	Line_0.patlength = 6.0;	Line_1.patlength = 0.6;	Line_2.patlength = 0.24;
	Line_0.color = Line_1.color = Line_2.color = 0x00000000L;	//black
	Line_0.pattern = Line_1.pattern = Line_2.pattern = 0L;		//solid line
	FillLine_0.width = FillLine_1.width = FillLine_2.width = 0.0;
	FillLine_0.patlength = 6.0;	FillLine_1.patlength = 0.6;	FillLine_2.patlength = 0.24;
	Line_0.color = Line_1.color = Line_2.color = 0x00000000L;	//black
	Line_0.pattern = Line_1.pattern = Line_2.pattern = 0L;		//solid line
	Fill_0.type = Fill_1.type = Fill_2.type = FILL_NONE;
	Fill_0.color = Fill_1.color = Fill_2.color = 0x00ffffffL;	//white background
	Fill_0.scale = Fill_1.scale = Fill_2.scale = 1.0;			//size = 100%
	Fill_0.hatch = &FillLine_0;	Fill_1.hatch = &FillLine_1;	Fill_2.hatch = &FillLine_2;
	Fill_0.color2 = Fill_1.color2 = Fill_2.color2 = 0x00ffffffL;	//white background
	OutLine_0.width = .2;	OutLine_1.width = .1;	OutLine_2.width = 0.008;
	OutLine_0.patlength = 6.0;	OutLine_1.patlength = 0.6;	OutLine_2.patlength = 0.24;
	OutLine_0.color = OutLine_1.color = OutLine_2.color = 0x00000000L;
	OutLine_0.pattern = OutLine_1.pattern = OutLine_2.pattern = 0L;
	pl = pgl = 0L;	pg = 0L;	pg_fl = 0L;	rrect_rad = 0L;
	cdisp = 0L;		min4log = 0.000001;
	axis_color = 0x0L;
	svgAttr = svgScript = currPath = IniFile = 0L;
	File1 = File2 = File3 = File4 = File5 = File6 = 0L;
}

Default::~Default()
{
	if(svgAttr) free(svgAttr);		if(svgScript) free(svgScript);
	if(currPath) free(currPath);	if(IniFile) free(IniFile);
	svgAttr = svgScript = currPath = 0L;
	if(pl) free(pl);				if(pgl) free(pgl);
	if(pg_fl) free(pg_fl);			if(pg) free(pg);
	if(rrect_rad) free(rrect_rad);
	if(File1) free(File1);			if(File2) free(File2);
	if(File3) free(File3);			if(File4) free(File4);
	if(File5) free(File5);			if(File6) free(File6);
}

void
Default::SetDisp(anyOutput *o)
{
	if(o && o != cdisp) {
		Undo.SetDisp(o);
		cdisp = o;
		}
}

double
Default::GetSize(int select)
{
	double RetVal = 0.0;

	switch (select){
	case SIZE_SYMBOL:				RetVal = 3.0;		break;
	case SIZE_SYM_LINE:				RetVal = 0.2;		break;
	case SIZE_DATA_LINE:
		switch(cUnits) {
		case 1:		return Line_1.width;
		case 2:		return Line_2.width;
		default:	return Line_0.width;
		}
	case SIZE_TEXT:					RetVal = 4.0;		break;
	case SIZE_GRECT_TOP:			RetVal = 0.0;		break;
	case SIZE_GRECT_BOTTOM:			RetVal = 110.0;		break;
	case SIZE_GRECT_LEFT:			RetVal = 0.0;		break;
	case SIZE_GRECT_RIGHT:			RetVal = 160.0;		break;
	case SIZE_DRECT_TOP:			RetVal = 10.0;		break;
	case SIZE_DRECT_BOTTOM:			RetVal = 90.0;		break;
	case SIZE_DRECT_LEFT:			RetVal = 25.0;		break;
	case SIZE_DRECT_RIGHT:			RetVal = 140.0;		break;
	case SIZE_PATLENGTH:
	case SIZE_DATA_LINE_PAT:
		switch(cUnits) {
		case 1:		return Line_1.patlength;
		case 2:		return Line_2.patlength;
		default:	return Line_0.patlength;
		}
	case SIZE_AXIS_TICKS:			RetVal = 2.0;		break;
	case SIZE_TICK_LABELS:			RetVal = 4.0;		break;
	case SIZE_WHISKER:
	case SIZE_ERRBAR:				RetVal = 3.0;		break;
	case SIZE_WHISKER_LINE:
	case SIZE_AXIS_LINE:
	case SIZE_BAR_LINE:			
	case SIZE_ERRBAR_LINE:
		switch(cUnits) {
		case 1:		return OutLine_1.width;
		case 2:		return OutLine_2.width;
		default:	return OutLine_0.width;
		}
	case SIZE_BOX:
	case SIZE_BAR:					RetVal = 10.0;		break;
	case SIZE_BUBBLE_LINE:			RetVal = 0.4;		break;
	case SIZE_BUBBLE_HATCH_LINE:	RetVal = 0.1;		break;
	case SIZE_ARROW_LINE:			RetVal = 0.4;		break;
	case SIZE_ARROW_CAPWIDTH:		RetVal = 3.0;		break;
	case SIZE_ARROW_CAPLENGTH:		RetVal = 4.0;		break;
	case SIZE_HAIRLINE:				RetVal = 0.1;		break;
	case SIZE_SEGLINE:				RetVal = 0.4;		break;
	case SIZE_CELLWIDTH:			RetVal = 20.0;		break;
	case SIZE_CELLTEXT:				RetVal = 4.5;		break;
	case SIZE_RRECT_RAD:			
		return rrect_rad ? *rrect_rad : GetSize(SIZE_SYMBOL)/2.0;
	default:	return 0.0;
		}
	switch(cUnits) {
	case 1:	RetVal /= 10.0;	break;
	case 2:	RetVal = NiceValue(RetVal/25.4);	break;
		}
	return RetVal;
}

DWORD
Default::Color(int select)
{
	switch (select){
	case COL_ERROR_LINE:
	case COL_WHISKER:
	case COL_SYM_LINE:			return 0x00000000L;
	case COL_SYM_FILL:			return 0x00ffffffL;
	case COL_ARROW:
	case COL_DATA_LINE:			return Line_0.color;
	case COL_TEXT:				return 0x00000000L;
	case COL_BG:				return 0x00ffffffL;
	case COL_AXIS:				return axis_color;
	case COL_BAR_LINE:			return OutLine_0.color;
	case COL_BAR_FILL:			return 0x00ffffffL;
	case COL_BUBBLE_LINE:
	case COL_BUBBLE_FILLLINE:	return 0x00ff0000L;
	case COL_BUBBLE_FILL:		return 0x00ffc0c0L;
	case COL_DRECT:				return 0x00ffffffL;
	case COL_GRECT:				return 0x00ffffffL;
	case COL_GRECTLINE:			return 0x00e0e0e0L;
	case COL_POLYLINE:			return pl ? pl->color : OutLine_0.color;
	case COL_POLYGON:			return pgl ? pgl->color : OutLine_0.color;
	default:					return 0x00C0C0C0L;	//Error
	}
}

LineDEF *
Default::GetLine()
{
	switch (cUnits) {
	case 1:		return &Line_1;
	case 2:		return &Line_2;
	default:	return &Line_0;
		}
}

void
Default::SetLine(int u, LineDEF *l, int which)
{
	double lw, pl;
	LineDEF *l1, *l2, *l3;

	switch (which) {
	case 0:	l1 = &Line_0;		l2 = &Line_1;		l3 = &Line_2;		break;
	case 1:	l1 = &FillLine_0;	l2 = &FillLine_1;	l3 = &FillLine_2;	break;
	case 2:	l1 = &OutLine_0;	l2 = &OutLine_1;	l3 = &OutLine_2;	break;
	default: return;
		}
	l1->color = l2->color = l3->color = l->color;
	l1->pattern = l2->pattern = l3->pattern = l->pattern;
	switch (u) {
	case 1:
		lw = l->width*10.0;	pl = l->patlength*10.0;
		l1->width = lw;			l1->patlength = pl;
		l2->width = l->width;	l2->patlength = l->patlength;
		l3->patlength = NiceValue(pl/25.4);
		l3->patlength = NiceValue(pl/25.4);
		break;
	case 2:
		lw = NiceValue(l->width*25.4);	pl = NiceValue(l->patlength*25.4);
		l1->width = lw;			l1->patlength = pl;
		l2->width = lw/10.0;	l2->patlength = pl/10.0;
		l3->width = l->width;	l3->patlength = l->patlength;
		break;
	default:
		lw = l->width;		pl = l->patlength;
		l1->width = l->width;	l1->patlength = l->patlength;
		l2->width = lw/10.0;		l2->patlength = pl/10.0;
		l3->patlength = NiceValue(pl/25.4);
		l3->patlength = NiceValue(pl/25.4);
		break;
		}
}

FillDEF *
Default::GetFill()
{
	switch (cUnits) {
	case 1:		return &Fill_1;
	case 2:		return &Fill_2;
	default:	return &Fill_0;
		}
}

void
Default::SetFill(int u, FillDEF *fd)
{
	memcpy(&Fill_0, fd, sizeof(FillDEF));
	memcpy(&Fill_1, fd, sizeof(FillDEF));
	memcpy(&Fill_2, fd, sizeof(FillDEF));
	if(fd->hatch) SetLine(u, fd->hatch, 1);
	Fill_0.hatch = &FillLine_0;
	Fill_1.hatch = &FillLine_1;
	Fill_2.hatch = &FillLine_2;
}

LineDEF *
Default::GetOutLine()
{
	switch (cUnits) {
	case 1:		return &OutLine_1;
	case 2:		return &OutLine_2;
	default:	return &OutLine_0;
		}
}

LineDEF *
Default::plLineDEF(LineDEF *ld)
{
	if(ld) {
		if(pl) free(pl);
		if(pl = (LineDEF *)malloc(sizeof(LineDEF))) memcpy(pl, ld, sizeof(LineDEF));
		}
	if(pl) return pl;
	else return GetOutLine();
}

LineDEF *
Default::pgLineDEF(LineDEF *ol)
{
	if(ol) {
		if(pgl) free(pgl);
		if(pgl = (LineDEF *)malloc(sizeof(LineDEF))) memcpy(pgl, ol, sizeof(LineDEF));
		}
	if(pgl) return pgl;
	else return GetOutLine();
}

FillDEF *
Default::pgFillDEF(FillDEF *fd)
{
	if(fd) {
		if(pg) free(pg);
		if(pg = (FillDEF *)malloc(sizeof(FillDEF))){
			memcpy(pg, fd, sizeof(FillDEF));
			if(pg->hatch) {
				if(pg_fl) free(pg_fl);
				if(pg_fl = (LineDEF *)malloc(sizeof(LineDEF)))
					memcpy(pg_fl, pg->hatch, sizeof(LineDEF));
				pg->hatch = pg_fl;
				}
			}
		}
	if(pg) return pg;
	else return GetFill();
}

double
Default::rrectRad(double rad)
{
	if(!rrect_rad)rrect_rad=(double*)malloc(sizeof(double));
	if(rrect_rad) return (*rrect_rad = rad);
	return 0.0;
}

void
Default::FileHistory(char *path)
{
	char *tmp_path = 0L, *tmp;
	char **history[] = {&File1, &File2, &File3, &File4, &File5, &File6};
	int i;

	if(path && (tmp_path=strdup(path))){
		for(i=strlen(path); i > 0 && tmp_path[i] != '/' && tmp_path[i] != '\\'; i--);
		tmp_path[i] = 0;
		if(currPath) free(currPath);
		if(strlen(tmp_path)) currPath = strdup(tmp_path);
		else currPath = 0L;
		strcpy(tmp_path, path);
		if(File1 && strcmp(File1, tmp_path)) {
			for(i = 0; i < 6 && tmp_path; i++) {
				if(i && *history[i] && !strcmp(File1, *history[i])){
					free(*history[i]);
					*history[i] = tmp_path;
					tmp_path = 0L;
					break;
					}
				if(*history[i]) {
					if(strcmp(tmp_path, *history[i])){
						tmp = *history[i];
						*history[i] = tmp_path;
						tmp_path = tmp;
						}
					else { 
						free(tmp_path);
						tmp_path = 0L;
						}
					}
				else{
					tmp = *history[i];
					*history[i] = tmp_path;
					tmp_path = tmp;
					}
				}
			}
		if(!(*history[0])) {
			*history[0] = tmp_path;
			tmp_path = 0L;
			}
		}
	if(tmp_path) free(tmp_path);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Chache file input for read operations
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#define CharCacheSize 1024
ReadCache::ReadCache()
{
	Cache = 0L;
	idx = max = 0;
	eof = true;
}

ReadCache::~ReadCache()
{
	if(Cache) free(Cache);
	Cache = 0L;
}

bool
ReadCache::Open(char *name)
{
	idx = max = 0;
	eof = true;
	if(!name) iFile = 0;		//use stdin
	else if(-1 ==(iFile = open(name, O_BINARY))) return false;
	Cache = (unsigned char *)malloc((unsigned)(CharCacheSize + 1));
	if(Cache) return true;
	return false;
}

void
ReadCache::Close()
{
	close(iFile);
	if(Cache) free(Cache);
	Cache = 0L;
}

unsigned char
ReadCache::Getc()
{
	if(Cache){
		if(idx < max) return (last = Cache[idx++]);
		else {
			do {
				max = read(iFile, Cache, CharCacheSize);
				if(max <=0) {
					eof = true;
					return 0;
					}
				else eof = false;
				}while(max == 0);
			idx = 1;
			return(last = Cache[0]);
			}
		}
	return 0;
}

unsigned char *
ReadCache::GetField()
{
	int i;
	static unsigned char *ret;

	if(Cache && max) {
		while(idx < max && Cache[idx] < 43) idx++;
		if(idx == max){
			if(max == CharCacheSize) {
				max = read(iFile, Cache, CharCacheSize);
				idx = 0;
				return GetField();
				}
			else return 0L;
			}
		i = idx;
		while(i < max && Cache[i] > 32 && Cache[i] <= 'z') i++;
		if(i == max) {
			for(i = 0; (Line[i] = Getc()) >32 && Line[i] <= 'z' && i < 4096; i++);
			Line[i] = 0;
			return Line;
			}
		else {
			ret = Cache+idx;
			idx = i;
			return ret;
			}
		}
	return 0L;
}

void
ReadCache::ReadLine(char *dest, int size)
{
	int i=0;
	unsigned char c;

	dest[0] = 0;
	do {
		c =  Getc();
		if(c == 0x09) c = 0x32;			// tab to space
		if(c > 31) dest[i++] = (char)c;
		}while(c && c != 0x0a && i < size);
	dest[i] = 0;
}

bool
ReadCache::GetInt(long *in)
{
	unsigned char *field;

	field = GetField();
	if(field && field[0]) {
		*in = atol((char*)field);
		if(*in == 0 && field[0] == '}') return false;
		return true;
		}
	return false;
}

bool
ReadCache::GetFloat(double *fn)
{
	unsigned char *field;

	field = GetField();
	if(field && field[0]) {
		*fn = atof((char*)field);
		if(*fn == 0.0 && field[0] == '}') return false;
		return true;
		}
	return false;
}

unsigned char
ReadCache::Lastc()
{
	return last;
}

bool
ReadCache::IsEOF()
{
	return eof;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Process memory block as if file input
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
MemCache::MemCache(unsigned char *ptr)
:ReadCache()
{
	if(ptr) {
		Cache = (unsigned char*) strdup((char*)ptr);
		max = strlen((char*)Cache);
		eof = false;
		}
}

MemCache::~MemCache()
{
	if(Cache) free(Cache);
	Cache = 0L;
}

unsigned char
MemCache::Getc()
{
	if(Cache){
		if(idx < max) return (last = Cache[idx++]);
		else {
			eof = true;
			return 0;
			}
		}
	return 0;
}

unsigned char *
MemCache::GetField()
{
	int i;
	static unsigned char *ret;

	if(Cache && max) {
		while(idx < max && Cache[idx] < 43) idx++;
		i = idx;
		while(i < max && Cache[i] > 32 && Cache[i] <= 'z') i++;
		ret = Cache+idx;
		idx = i;
		return ret;
		}
	return 0L;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Process Undo 
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#define UNDO_RING_SIZE 0x100
#define UNDO_IDX_MASK 0xff
UndoObj::UndoObj()
{
	buff = (UndoInfo**)calloc(UNDO_RING_SIZE, sizeof(UndoInfo*));
	stub1 = ndisp = 0;
	pcb = &stub1;	cdisp = 0L;	buffers = 0L;
}

UndoObj::~UndoObj()
{
	Flush();
	free(buff);
}

void
UndoObj::Flush()
{
	int i, j;

	for(i = 0; i < UNDO_RING_SIZE; i++) if(buff[i]) FreeInfo(&buff[i]);
	*pcb = 0;

	if(!buffers) return;
	for(i = 0; i < ndisp; i++) if(buffers[i]) {
		if(buffers[i]->buff) for(j = 0; j < UNDO_RING_SIZE; j++) {
			if(buffers[i]->buff[j]) FreeInfo(&buffers[i]->buff[j]);
			}
		free(buffers[i]);
		}
	free(buffers);
	stub1 = ndisp = 0;
	pcb = &stub1;	cdisp = 0L;	buffers = 0L;
}
	
void
UndoObj::SetDisp(anyOutput *o)
{
	int i;
	void *ptmp;

	if(o && o != cdisp) {
		if(buffers) {
			for(i = 0; i < ndisp; i++) {
				if(buffers[i] && buffers[i]->disp == o){
					cdisp = o;
					buff = buffers[i]->buff;	pcb = &buffers[i]->count;
					return;
					}
				else if(!buffers[i] && (buffers[0] = (UndoBuff*)calloc(1, sizeof(UndoBuff)))) {
					buffers[i]->buff = (UndoInfo**)calloc(UNDO_RING_SIZE, sizeof(UndoInfo*));
					buffers[i]->disp = cdisp = o;
					buff = buffers[i]->buff;	pcb = &buffers[i]->count;
					return;
					}
				}
			if(ptmp = memdup(buffers, sizeof(UndoBuff*) *(ndisp+1), 0)){
				free(buffers);		buffers = (UndoBuff**)ptmp;
				if(buffers[ndisp] = (UndoBuff*)calloc(1, sizeof(UndoBuff))) {
					buffers[ndisp]->buff = (UndoInfo**)calloc(UNDO_RING_SIZE, sizeof(UndoInfo*));
					buffers[ndisp]->disp = cdisp = o;
					buff = buffers[ndisp]->buff;	pcb = &buffers[ndisp]->count;
					ndisp++;
					}
				}
			}
		else if(buffers = (UndoBuff**)calloc(1, sizeof(UndoBuff*))){
			if(buffers[0] = (UndoBuff*)calloc(1, sizeof(UndoBuff))) {
				buffers[0]->buff = (UndoInfo**)calloc(UNDO_RING_SIZE, sizeof(UndoInfo*));
				buffers[0]->disp = cdisp = o;
				ndisp = 1;
				buff = buffers[0]->buff;	pcb = &buffers[0]->count;
				}
			}
		}
}

void
UndoObj::InvalidGO(GraphObj *go)
{
	int i, i1, i2;

	if(*pcb >10) {
		if(*pcb < UNDO_RING_SIZE){
			i1 = 0;		i2 = *pcb;
			}
		else {
			i1 = ((*pcb) & UNDO_IDX_MASK);
			i2 = (((*pcb)-UNDO_RING_SIZE) & UNDO_IDX_MASK);
			if(i1 > i2) Swap(i1, i2);
			}
		}
	else {
		i1 = 0;		i2 = *pcb;
		}
	for(i = i1; i < i2; i++) {
		if(buff[i] && buff[i]->owner == go) FreeInfo(&buff[i]);
		if(buff[i]) switch(buff[i]->cmd) {
//		case UNDO_OBJCONF:
		case UNDO_OBJCONF_1:
			if(buff[i]->loc == (void**)go) FreeInfo(&buff[i]);
			break;
			}
		}
}

void
UndoObj::Pop()
{
	int idx;

	if(*pcb < 1) return;
	idx = ((*pcb-1) & UNDO_IDX_MASK);
	if(buff[idx]) FreeInfo(&buff[idx]);
	(*pcb)--;
}

void
UndoObj::Restore(bool redraw, anyOutput*o)
{
	int i, j, idx;
	DWORD flags;
	UndoList *ul;
	GraphObj **gol;
	unsigned char *savbuf, *target;

	if(o) SetDisp(o);
	else if(cdisp) o = cdisp;
	CurrGO = 0L;
	if(*pcb < 1){
		InfoBox("The UNDO-cache is empty");
		return;
		}
	do {
		idx = ((*pcb-1) & UNDO_IDX_MASK);
		if(!buff[idx] && *pcb > 0) (*pcb)--;
		} while(!buff[idx] && *pcb > 0);
	if(!buff[idx]) return;
	if(buff[idx]->zd.org.fx != cdisp->VPorg.fx ||
		buff[idx]->zd.org.fy != cdisp->VPorg.fy || buff[idx]->zd.scale != cdisp->VPscale){
		cdisp->VPorg.fx = buff[idx]->zd.org.fx;
		cdisp->VPorg.fy = buff[idx]->zd.org.fy;
		cdisp->VPscale = buff[idx]->zd.scale;
		if(cdisp->VPscale > 100.0) cdisp->VPscale = 100.0;
		if(cdisp->VPscale < 0.05) cdisp->VPscale = 0.05;
		if(buff[idx]->owner)
			if(buff[idx]->owner->Id < GO_PLOT) CurrGO = buff[idx]->owner;
			if(!((buff[idx]->owner)->Command(CMD_SETSCROLL, 0L, cdisp)))
				(buff[idx]->owner)->Command(CMD_REDRAW, 0L, cdisp);
		return;
		}
	(*pcb)--;
	if(buff[idx]) {
		flags = buff[idx]->flags;
		switch(buff[idx]->cmd) {
		case UNDO_MUTATE:
		case UNDO_DEL_GO:
			((GraphObj*)(buff[idx]->data))->parent = buff[idx]->owner;
			if(buff[idx]->cmd == UNDO_MUTATE && *(buff[idx]->loc))
				::DeleteGO((GraphObj*)*(buff[idx]->loc));
			else CurrGO = (GraphObj*) buff[idx]->data;
			*(buff[idx]->loc) = buff[idx]->data;
			(buff[idx]->owner)->Command(CMD_MRK_DIRTY, 0L, 0L);
			break;
		case UNDO_DROPGOLIST:
			if((ul = (UndoList *)(buff[idx]->data)) && (ul->array)){
				gol = (GraphObj**)(*(ul->loc_arr));
				if(gol) for (i = 0; i < *(ul->loc_count); i++) if(gol[i]) ::DeleteGO(gol[i]);
				*(ul->loc_count) = ul->count;				free(gol);
				*(ul->loc_arr) = ul->array;					free(ul);
				}
			break;
		case UNDO_GOLIST:
			if((ul = (UndoList *)(buff[idx]->data)) && (ul->array)){
				memcpy(*(ul->loc_arr), ul->array, ul->count * sizeof(GraphObj*));
				*(ul->loc_count) = ul->count;				free(ul->array);
				free(ul);
				}
			break;
		case UNDO_DROPMEM:
			*(buff[idx]->loc) = buff[idx]->data;				break;
		case UNDO_SAVVAR:
			if(!(savbuf = (unsigned char *)buff[idx]->data))break;
			for(i = 0; ; ) {
				memcpy(&target, savbuf+i, sizeof(unsigned char*));	i += sizeof(unsigned char*);
				memcpy(&j, savbuf+i, sizeof(int));					i += sizeof(int);
				if(!target) break;
				memcpy(target, savbuf+i, j);						i += j;
				}
			if(buff[idx]->owner)(buff[idx]->owner)->Command(CMD_MRK_DIRTY, 0L, 0L);
			free(savbuf);
			break;
		case UNDO_VALDWORD:
			*((DWORD*)(buff[idx]->loc)) = *((DWORD*)(buff[idx]->data));
			free(buff[idx]->data);								break;
		case UNDO_VALINT:
			*((int*)(buff[idx]->loc)) = *((int*)(buff[idx]->data));
			free(buff[idx]->data);								break;
		case UNDO_OBJCONF_1:			//single object restore
			UpdGOfromMem((GraphObj *)buff[idx]->loc, (unsigned char *)buff[idx]->data);
			free(buff[idx]->data);								break;
		case UNDO_OBJCONF:				//tree of objects to restore
			RestoreConf(buff[idx]);
			if(buff[idx] && buff[idx]->data) free(buff[idx]->data);								break;
		case UNDO_LFP:
			memcpy(buff[idx]->loc, buff[idx]->data, sizeof(lfPOINT));
			free(buff[idx]->data);								break;
		case UNDO_MOVE:
			(buff[idx]->owner)->Command(CMD_UNDO_MOVE, buff[idx]->data, 0L);
			free(buff[idx]->data);								break;
		case UNDO_RECT:
			memcpy(buff[idx]->loc, buff[idx]->data, sizeof(fRECT));
			free(buff[idx]->data);								break;
		case UNDO_STRING:
			if(*(buff[idx]->loc)) free(*(buff[idx]->loc));
			CurrGO = buff[idx]->owner;
			*(buff[idx]->loc) = buff[idx]->data;				break;
		case UNDO_ROTDEF:
			memcpy(*(buff[idx]->loc), buff[idx]->data, 6 * sizeof(double));
			free(buff[idx]->data);								break;
		case UNDO_SETGO:
			::DeleteGO(*((GraphObj**)(buff[idx]->loc)));
			*((GraphObj**)(buff[idx]->loc)) = 0L;				break;
		case UNDO_LINEDEF:
			memcpy(buff[idx]->loc, buff[idx]->data, sizeof(LineDEF));
			free(buff[idx]->data);								break;
		case UNDO_FILLDEF:
			memcpy(buff[idx]->loc, buff[idx]->data, sizeof(FillDEF) - (sizeof(LineDEF*) + sizeof(DWORD)));
			((FillDEF*)(buff[idx]->loc))->color2 = ((FillDEF*)(buff[idx]->data))->color2;
			free(buff[idx]->data);								break;
		case UNDO_AXISDEF:
			if(((AxisDEF*)(buff[idx]->loc))->breaks) free(((AxisDEF*)(buff[idx]->loc))->breaks);
			memcpy(buff[idx]->loc, buff[idx]->data, sizeof(AxisDEF));
			free(buff[idx]->data);								break;
		case UNDO_TEXTDEF:
			if(((TextDEF*)(buff[idx]->loc))->text) free(((TextDEF*)(buff[idx]->loc))->text);
			memcpy(buff[idx]->loc, buff[idx]->data, sizeof(TextDEF));
			free(buff[idx]->data);								break;
		case UNDO_LFP3D:
			memcpy(buff[idx]->loc, buff[idx]->data, sizeof(fPOINT3D));
			free(buff[idx]->data);								break;
		case UNDO_FLOAT:
			*((double*)(buff[idx]->loc)) = *((double*)(buff[idx]->data));
			free(buff[idx]->data);								break;
		case UNDO_MEM:
			if((ul = (UndoList *)(buff[idx]->data)) && (ul->array)){
				memcpy(*(ul->loc_arr), ul->array, ul->size);
				*(ul->loc_count) = ul->count;
				free(ul->array);
				if(buff[idx]->owner->Id < GO_PLOT) CurrGO = (GraphObj*) buff[idx]->owner;
				if(buff[idx]->owner)(buff[idx]->owner)->Command(CMD_MRK_DIRTY, 0L, 0L);
				}
			break;
			}
		if(flags & UNDO_CONTINUE){
			free(buff[idx]);	buff[idx] = 0L;
			Restore(redraw, cdisp);
			}
		else {
			if(o) o->MrkMode = MRK_NONE;
			if(redraw && buff[idx] && buff[idx]->owner){
				(buff[idx]->owner)->Command(CMD_MRK_DIRTY, 0L, 0L);
				(buff[idx]->owner)->Command(CMD_REDRAW, 0L, 0L);
				}
			if(buff[idx]) free(buff[idx]);	buff[idx] = 0L;
			}
		}
	else {
		InfoBox("The UNDO-cache is empty");
		}
}

void
UndoObj::ListGOmoved(GraphObj **oldlist, GraphObj **newlist, long size)
{
	long i;
	int c;

	if(!oldlist || !newlist || oldlist == newlist) return;
	for(i = 0; i < size; i++) if(oldlist[i] == newlist[i]) {
		for(c = 0; c < UNDO_RING_SIZE; c++) {
			if(buff[c]) switch(buff[c]->cmd) {
			case UNDO_DEL_GO:
			case UNDO_SETGO:
			case UNDO_OBJCONF_1:
			case UNDO_OBJCONF:
				if(buff[c]->loc == (void**) &oldlist[i]){
					buff[c]->loc = (void**) &newlist[i];
					}
				break;
				}
			}
		}
}

void
UndoObj::DeleteGO(GraphObj **go, DWORD flags, anyOutput *o)
{
	if(!go || !(*go)) return;
	if(o){
		SetDisp(o);					 o->HideMark();
		}
	if(CurrGO == *go) CurrGO = 0L;
	if((*go)->Id == GO_POLYLINE || (*go)->Id == GO_POLYGON){
		if(CurrHandle && CurrHandle->parent==*go) {
			if((*go)->Command(CMD_DELOBJ, CurrHandle, 0l)) return;
			}
		}
	NewItem(UNDO_DEL_GO, flags, (*(go))->parent, *(go), (void**)go);
	(*(go))->parent->Command(CMD_MRK_DIRTY, 0L, 0L);
	(*(go))->parent = 0L;
	*(go) = CurrGO = 0L;
}

void
UndoObj::MutateGO(GraphObj **old, GraphObj *repl, DWORD flags, anyOutput *o)
{
	if(!old || !(*old)) return;
	if(o){
		SetDisp(o);					 o->HideMark();
		}
	if(!(*old))return;	//HideMark might delete object: should never happen
	if(CurrGO == *old) CurrGO = 0L;
	NewItem(UNDO_MUTATE, flags, (*(old))->parent, *(old), (void**)old);
	repl->parent = (*(old))->parent;
	(*(old))->parent = 0L;
	*(old) = repl;
	repl->parent->Command(CMD_REDRAW, 0L, o);
}

void
UndoObj::StoreListGO(GraphObj *parent, GraphObj ***go, long *count, DWORD flags)
{
	UndoList *ul;

	if(ul = (UndoList *)malloc(sizeof(UndoList))) {
		if(ul->array = memdup(*go, *count * sizeof(GraphObj*), 0)){
			ul->loc_arr = (void **)go;
			ul->count = *count;
			ul->loc_count = count;
			if(0 > NewItem(UNDO_GOLIST, flags, parent, ul, 0L)) {
				free(ul->array);			free(ul);
				}
			}
		else free(ul);
		}
}

void
UndoObj::DropListGO(GraphObj *parent, GraphObj ***go, long *count, DWORD flags)
{
	UndoList *ul;

	if(ul = (UndoList *)malloc(sizeof(UndoList))) {
		if(ul->array = memdup(*go, *count * sizeof(GraphObj*), 0)){
			ul->loc_arr = (void **)go;		*go = 0L;
			ul->count = *count;				*count = 0;
			ul->loc_count = count;
			if(0 > NewItem(UNDO_DROPGOLIST, flags, parent, ul, 0L)) {
				free(ul->array);			free(ul);
				}
			}
		else free(ul);
		}
}

void
UndoObj::DropMemory(GraphObj *parent, void **mem, DWORD flags)
{
	NewItem(UNDO_DROPMEM, flags, parent, *(mem), mem);
	*mem = 0L;
}

void
UndoObj::SavVarBlock(GraphObj *parent, void **mem, DWORD flags)
{
	NewItem(UNDO_SAVVAR, flags, parent, *(mem), mem);
	*mem = 0L;
}

void
UndoObj::ValDword(GraphObj *parent, DWORD *val, DWORD flags)
{
	void *ptr;

	if(!(ptr = memdup(val, sizeof(DWORD), 0))) return;
	if(0 > NewItem(UNDO_VALDWORD, flags, parent, ptr, (void**)val)) free(ptr);
}

void
UndoObj::ValInt(GraphObj *parent, int *val, DWORD flags)
{
	void *ptr;

	if(!(ptr = memdup(val, sizeof(int), 0))) return;
	if(0 > NewItem(UNDO_VALINT, flags, parent, ptr, (void**)val)) free(ptr);
}

void
UndoObj::ObjConf(GraphObj *go, DWORD flags)
{
	long sz;
	int idx;
	
	InvalidGO(go);
	if(0<=(idx = NewItem(UNDO_OBJCONF, flags, go->parent, GraphToMem(go, &sz), (void**)go))){
		if(cObsW == 1) buff[idx]->cmd = UNDO_OBJCONF_1;
		(buff[idx]->owner)->Command(CMD_MRK_DIRTY, 0L, 0L);
		}
}

int
UndoObj::SaveLFP(GraphObj *go, lfPOINT *lfp, DWORD flags)
{
	int idx;
	void *ptr;

	if(!(ptr = memdup(lfp, sizeof(lfPOINT), 0))) return -1;
	if(0 > (idx = NewItem(UNDO_LFP, flags, go, ptr, (void**)lfp))) free(ptr);
	return idx;
}

void
UndoObj::MoveObj(GraphObj *go, lfPOINT *lfp, DWORD flags)
{
	int idx;
	lfPOINT dsp;

	if(!lfp) return;
	dsp.fx = -1.0 * lfp->fx;		dsp.fy = -1.0 * lfp->fy;
	if((idx = SaveLFP(go, &dsp, flags)) <0) return;
	buff[idx]->cmd = UNDO_MOVE;
}

void
UndoObj::ValRect(GraphObj *go, fRECT *rec, DWORD flags)
{
	void *ptr;

	if(!(ptr = memdup(rec, sizeof(fRECT), 0))) return;
	if(0 > NewItem(UNDO_RECT, flags, go, ptr, (void**)rec)) free(ptr);
}

void
UndoObj::String(GraphObj *go, char **s, DWORD flags)
{
	char *ptr;

	ptr = (s && *s &&  *(*s)) ? strdup(*(s)):0L;
	if(0 > NewItem(UNDO_STRING, flags, go, ptr, (void**)s)) if(ptr) free(ptr);
}

void
UndoObj::RotDef(GraphObj *go, double **d, DWORD flags)
{
	void *ptr;

	if(!(ptr = memdup(*d, 6 * sizeof(double), 0))) return;
	if(0 > NewItem(UNDO_ROTDEF, flags, go, ptr, (void**)d)) free(ptr);
}

void
UndoObj::SetGO(GraphObj *parent, GraphObj **where, GraphObj *go, DWORD flags)
{
	*where = go;
	NewItem(UNDO_SETGO, flags, parent, 0L, (void**)where);
}

void
UndoObj::Line(GraphObj *go, LineDEF *ld, DWORD flags)
{
	void *ptr;

	if(!(ptr = memdup(ld, sizeof(LineDEF), 0))) return;
	if(0 > NewItem(UNDO_LINEDEF, flags, go, ptr, (void**)ld)) free(ptr);
}

void
UndoObj::Fill(GraphObj *go, FillDEF *fd, DWORD flags)
{
	void *ptr;

	if(!(ptr = memdup(fd, sizeof(FillDEF), 0))) return;
	if(0 > NewItem(UNDO_FILLDEF, flags, go, ptr, (void**)fd)) free(ptr);
}

void
UndoObj::AxisDef(GraphObj *go, AxisDEF *ad, DWORD flags)
{
	AxisDEF *ptr;

	if(!(ptr = (AxisDEF*) memdup(ad, sizeof(AxisDEF), 0))) return;
	if(ptr->nBreaks && ptr->breaks) ptr->breaks = 
		(lfPOINT*)memdup(ad->breaks, ad->nBreaks * sizeof(lfPOINT), 0);
	if(0 > NewItem(UNDO_AXISDEF, flags, go, ptr, (void**)ad)) free(ptr);
}

void
UndoObj::TextDef(GraphObj *go, TextDEF *td, DWORD flags)
{
	TextDEF *ptr;

	if(!(ptr = (TextDEF*) memdup(td, sizeof(TextDEF), 0))) return;
	if(td->text) ptr->text = strdup(td->text);
	if(0 > NewItem(UNDO_TEXTDEF, flags, go, ptr, (void**)td)) free(ptr);
}

void
UndoObj::ValLFP3D(GraphObj *go, fPOINT3D *lfp, DWORD flags)
{
	void *ptr;

	if(!(ptr = memdup(lfp, sizeof(fPOINT3D), 0))) return;
	if(0 > NewItem(UNDO_LFP3D, flags, go, ptr, (void**)lfp)) free(ptr);
}

void
UndoObj::ValFloat(GraphObj *parent, double *val, DWORD flags)
{
	void *ptr;

	if(!(ptr = memdup(val, sizeof(double), 0))) return;
	if(0 > NewItem(UNDO_FLOAT, flags, parent, ptr, (void**)val)) free(ptr);
}

void
UndoObj::DataMem(GraphObj *go, void **mem, int size, long *count, DWORD flags)
{
	UndoList *ul;

	if(ul = (UndoList *)malloc(sizeof(UndoList))) {
		if(ul->array = memdup(*mem, size, 0)){
			ul->loc_arr = (void **)mem;
			ul->size = size;
			ul->count = *count;
			ul->loc_count = count;
			if(0 > NewItem(UNDO_MEM, flags, go, ul, 0L)) {
				free(ul->array);			free(ul);
				}
			}
		else free(ul);
		}
}

int
UndoObj::NewItem(int cmd, DWORD flags, GraphObj *owner, void *data, void **loc)
{
	UndoInfo *tmp;
	int idx;

	if(!buff || !cdisp) return -1;
	if(!(tmp = (UndoInfo *)malloc(sizeof(UndoInfo))))return -1;
	tmp->cmd = cmd;			tmp->flags = flags;
	tmp->owner = owner;		tmp->data = data;
	tmp->loc = loc;
	tmp->zd.org.fx = cdisp->VPorg.fx;
	tmp->zd.org.fy = cdisp->VPorg.fy;
	tmp->zd.scale = cdisp->VPscale;
	idx = (*pcb & UNDO_IDX_MASK);
	if(buff[idx]) FreeInfo(&buff[idx]);
	buff[idx] = tmp;
	(*pcb)++;
	return idx;
}

void
UndoObj::FreeInfo(UndoInfo** inf)
{
	int i;
	UndoList *ul;
	GraphObj *go, **gol;

	if(!inf || !(*inf)) return;
	switch((*inf)->cmd) {
	case UNDO_SETGO:
		break;
	case UNDO_MUTATE:
	case UNDO_DEL_GO:
		go = (GraphObj*)((*inf)->data);
		(*inf)->data = 0L;		::DeleteGO(go);
		break;
	case UNDO_DROPGOLIST:
		if((ul = (UndoList *)((*inf)->data)) && (ul->array)) {
			gol = (GraphObj**)(ul->array);
			for (i = 0; i < ul->count; i++) if(gol[i]) ::DeleteGO(gol[i]);
			free(ul->array);				free(ul);
			}
		break;
	case UNDO_GOLIST:	case UNDO_MEM:
		if((ul = (UndoList *)((*inf)->data)) && (ul->array)) {
			free(ul->array);				free(ul);
			}
		break;
	case UNDO_AXISDEF:
		if(((AxisDEF*)(*inf)->data)->breaks) free(((AxisDEF*)(*inf)->data)->breaks);
		free((*inf)->data);
		break;
	case UNDO_TEXTDEF:
		if(((TextDEF*)(*inf)->data)->text) free(((TextDEF*)(*inf)->data)->text);
		free((*inf)->data);
		break;
	case UNDO_DROPMEM:		case UNDO_VALDWORD:		case UNDO_VALINT:
	case UNDO_OBJCONF:		case UNDO_OBJCONF_1:	case UNDO_LFP:
	case UNDO_MOVE:			case UNDO_RECT:			case UNDO_STRING:
	case UNDO_ROTDEF:		case UNDO_LINEDEF:		case UNDO_FILLDEF:
	case UNDO_LFP3D:		case UNDO_FLOAT:		case UNDO_SAVVAR:
		free((*inf)->data);
		break;
		}
	free(*inf);
	*inf = 0L;
}

class UndoUtil:public GraphObj {
public:
	GraphObj *res;

	UndoUtil(GraphObj *p, GraphObj *old):GraphObj(0L, 0L){root = p; optr = old; res = 0L;};
	bool Command(int cmd, void *tmpl, anyOutput *o);

private:
	GraphObj *root, *optr;
};

bool
UndoUtil::Command(int cmd, void *tmpl, anyOutput *o)
{
	GraphObj *xch[2];

	switch(cmd){
	case CMD_DROP_GRAPH:
		//we come here if conversion of undo-information is
		//   successfully converted into a tree of graphic objects.
		//   Now ask the parent object to replace the modified
		//   object with a previous version (i.e. undo modifications).
		xch[0] = optr;
		xch[1] = res = (GraphObj*)tmpl;
		if(root) return root->Command(CMD_REPL_GO, xch, o);
		break;
		}
	return false;
}

void
UndoObj::RestoreConf(UndoInfo *inf)
{
	UndoUtil *proc;
	int i;

	//Create a message object which will accept the translated graphic
	//   object or tree of objects, and which will forward it to the parent
	//   of the tree finalizing undo.
	if(!inf->data) return;
	if(!(proc = new UndoUtil(inf->owner, (GraphObj*)inf->loc))) return; 
	OpenGraph(proc, 0L, (unsigned char *)inf->data);
	if(proc->res) for(i = 0; i < UNDO_RING_SIZE; i++) {
		if(buff[i] && buff[i]->owner == (GraphObj*)inf->loc) FreeInfo(&buff[i]);
		if(buff[i] && buff[i]->cmd == UNDO_OBJCONF){
			if(buff[i]->loc == inf->loc) buff[i]->loc = (void**)proc->res;
			}
		}
	delete proc;
}
#undef UNDO_RING_SIZE
#undef UNDO_IDX_MASK
