//PlotObs.cpp, Copyright (c) 2001, 2002, 2003, 2004 R.Lackner
//
//    This file is part of RLPlot.
//
//    RLPlot is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    RLPlot is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with RLPlot; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// This modules contains code for the differnt Plot objects. Plots are
// graphic objects containing more objects, which represent the data.
// Several Plots may be contained in a Graph: Plots are the different layers
// of a Graph.
// Most part of this module has been moved here from rlplot.cpp of
// earlier versions. 
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#include "rlplot.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

extern char TmpTxt[];
extern Default defs;
extern int cPlots;
extern GraphObj *CurrGO, *TrackGO;			//Selected Graphic Objects
extern Axis **CurrAxes;						//axes of current graph
extern UndoObj Undo;

int AxisTempl3D = 0;

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

Plot::Plot(GraphObj *par, DataObj *d):GraphObj(par, d)
{
	Id = GO_PLOT;
	Bounds.Xmin = Bounds.Ymin = 0.0;	Bounds.Xmax = Bounds.Ymax = 100.0;
	sprintf(TmpTxt, "Plot %d", ++cPlots);
	name = strdup(TmpTxt);
	use_xaxis = use_yaxis = 0;
	hidden = 0;
}

double
Plot::GetSize(int select)
{
	switch(select){
	case SIZE_MINE:				return 0.0;
	//The Bounds values must be returned by every plot:
	//   they are necessary for scaling !
	case SIZE_BOUNDS_XMIN:		
		return parent ? parent->GetSize(SIZE_BOUNDS_XMIN) : Bounds.Xmin;
	case SIZE_BOUNDS_XMAX:		
		return parent ? parent->GetSize(SIZE_BOUNDS_XMAX) : Bounds.Xmax;
	case SIZE_BOUNDS_YMIN:		
		return parent ? parent->GetSize(SIZE_BOUNDS_YMIN) : Bounds.Ymin;	
	case SIZE_BOUNDS_YMAX:		
		return parent ? parent->GetSize(SIZE_BOUNDS_YMAX) : Bounds.Ymax;
	case SIZE_BARMINX:
	case SIZE_BARMINY:
		return 1.0f;
	default:
		if(parent) return parent->GetSize(select);
		else return defs.GetSize(select);
		}
}

DWORD
Plot::GetColor(int select)
{
	if(parent) return parent->GetColor(select);
	else return defs.Color(select);
}

void
Plot::CheckBounds(double x, double y)
{
	if(x < Bounds.Xmin) Bounds.Xmin = x;	if(x > Bounds.Xmax) Bounds.Xmax = x;
	if(y < Bounds.Ymin) Bounds.Ymin = y;	if(y > Bounds.Ymax) Bounds.Ymax = y;
}

bool
Plot::UseAxis(int idx)
{
	double dx, dy;

	if(CurrAxes && CurrAxes[idx]) {
		dx = fabs(CurrAxes[idx]->GetSize(SIZE_XPOS+1) -
			CurrAxes[idx]->GetSize(SIZE_XPOS));
		dy = fabs(CurrAxes[idx]->GetSize(SIZE_YPOS+1) -
			CurrAxes[idx]->GetSize(SIZE_YPOS));
		if(dx > dy && idx != use_xaxis) {
			Undo.ValInt(parent, &use_xaxis, 0L);
			use_xaxis = idx;			return true;
			}
		else if(idx != use_yaxis) {
			Undo.ValInt(parent, &use_yaxis, 0L);
			use_yaxis = idx;			return true;
			}
		}
	return false;
}

void
Plot::ApplyAxes(anyOutput *o)
{
	AxisDEF x_axis, y_axis;
	double dsp;
	fRECT CurrRect;
	
	if(!o || !CurrAxes || !parent) return;
	memcpy(&x_axis, &o->xAxis, sizeof(AxisDEF));
	memcpy(&y_axis, &o->yAxis, sizeof(AxisDEF));
	CurrRect.Xmin = (dsp = parent->GetSize(SIZE_GRECT_LEFT)) +
		parent->GetSize(SIZE_DRECT_LEFT);
	CurrRect.Xmax = dsp + parent->GetSize(SIZE_DRECT_RIGHT);
	if(use_xaxis && CurrAxes[use_xaxis]) {
		memcpy(&x_axis, CurrAxes[use_xaxis]->axis, sizeof(AxisDEF));
		if(x_axis.flags & AXIS_DEFRECT) {
			x_axis.loc[0].fx += dsp;	x_axis.loc[1].fx += dsp;
			}
		}
	else use_xaxis = 0;
	CurrRect.Ymin = (dsp = parent->GetSize(SIZE_GRECT_TOP)) +
		parent->GetSize(SIZE_DRECT_BOTTOM);
	CurrRect.Ymax = dsp + parent->GetSize(SIZE_DRECT_TOP);
	if(use_yaxis && CurrAxes[use_yaxis]) {
		memcpy(&y_axis, CurrAxes[use_yaxis]->axis, sizeof(AxisDEF));
		if(y_axis.flags & AXIS_DEFRECT) {
			y_axis.loc[0].fy += dsp;	y_axis.loc[1].fy += dsp;
			}
		}
	else use_yaxis = 0;
	o->SetRect(CurrRect, o->units, &x_axis, &y_axis);
}

void
Plot::CheckBounds3D(double x, double y, double z)
{
	if(x < xBounds.fx) xBounds.fx = x;	if(x > xBounds.fy) xBounds.fy = x;
	if(y < yBounds.fx) yBounds.fx = y;	if(y > yBounds.fy) yBounds.fy = y;
	if(z < zBounds.fx) zBounds.fx = z;	if(z > zBounds.fy) zBounds.fy = z;
	CheckBounds(x, y);
}

bool
Plot::SavVarObs(GraphObj **gol, long ngo, DWORD flags)
{
	int i;
	void *ptr;

	if(!gol || !ngo) return false;
	SavVarInit();
	for(i = 0; i < ngo; i++) 
		if(gol[i]) gol[i]->FileIO(SAVE_VARS);
	ptr = SavVarFetch();
	Undo.SavVarBlock(this, &ptr, flags);
	return true;
}

DataObj *
Plot::CreaCumData(char *xr, char *yr, int mode, double base)
{
	char **yranges;
	int i, j, nc, nr, ir, ic;
	double value, old_val;
	DataObj *CumData = 0L;
	AccRange *ax = 0L, *ay = 0L;

	if(!xr || !yr || !mode || !data) return 0L;
	if(!(CumData = new DataObj()))return 0L;
	if(!(ax = new AccRange(xr))) {
		delete CumData;		return 0L;
		}
	nr = ax->CountItems();
	if(!(yranges = split(yr, '&', &nc))){
		delete CumData;		delete ax;		return 0L;
		}
	if(CumData->Init(mode == 1 || mode == 2 ? nr : nr * 2, nc+2)){
		// set x values as first column
		for(i = 0, ax->GetFirst(&ic, &ir); ax->GetNext(&ic, &ir); i++) {
			if(data->GetValue(ir, ic, &value)) CumData->SetValue(i, 0, value);
			CumData->SetValue(i, 1, base);
			}
		if(mode == 3 || mode == 4) for(i = 1; i <= nr; i++) {	//complete polygon data
			if(CumData->GetValue(nr-i, 0, &value)) CumData->SetValue(i-1+nr, 0, value);
			}
		//process all y-ranges
		for (j = 2; j <= (nc+1); j++) if(ay = new AccRange(yranges[j-2])){
			for(i = 0; i < nr; i++) {
				if(CumData->GetValue(i, j-1, &value)) CumData->SetValue(i, j, value);
				}
			for(i = 0, ay->GetFirst(&ic, &ir); ay->GetNext(&ic, &ir) && i < nr; i++) {
				if(data->GetValue(ir, ic, &value) && CumData->GetValue(i, j, &old_val)){
					switch (mode) {
					case 1:	case 3:	value += old_val;			break;
					case 2:	case 4: value = old_val -value;		break;
						}
					CumData->SetValue(i, j, value);
					}
				}
			if(mode == 3 || mode == 4) for(i = 1; i <= nr; i++) {
				//complete polygon data
				if(CumData->GetValue(nr-i, j-1, &value)) CumData->SetValue(i-1+nr, j, value);
				}
			delete ay;		ay = 0L;
			}
		}



	for(i = 0; i < nc; i++) if(yranges[i]) free(yranges[i]);
	if(ax) delete ax;	if(ay) delete ay;
	free(yranges);
	return CumData;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// PlotScatt handles most XY-Plots: its a Plot-Class
PlotScatt::PlotScatt(GraphObj *par, DataObj *d, DWORD presel):Plot(par, d)
{
	FileIO(INIT_VARS);
	DefSel = presel;
	Id = GO_PLOTSCATT;
	if (!d) {
		if(parent && parent->Command(CMD_DELOBJ, this, NULL)) return;
		ErrorBox("Attempt to create plot\nwithout any data.");
		return;
		}
}

PlotScatt::PlotScatt(GraphObj *par, DataObj *d, int cBars, Bar **bars):Plot(par, d)
{
	int i;

	FileIO(INIT_VARS);
	if(cBars && bars) {
		if((Bars = (Bar**)calloc(cBars, sizeof(Bar*)))) {
			nPoints = cBars;
			for(i = 0; i < cBars; i++) {
				if((Bars[i] = bars[i])) Bars[i]->parent = this;
				bars[i] = 0L;
				}
			}
		}
	Id = GO_PLOTSCATT;
}

PlotScatt::PlotScatt(GraphObj *par, DataObj *d, int nPts, Symbol **sym, DataLine *lin):
	Plot(par, d)
{
	int i;

	FileIO(INIT_VARS);
	nPoints = nPts;
	if(Symbols = sym) for(i = 0; i < nPts; i++) if(Symbols[i]) Symbols[i]->parent = this;
	if(TheLine = lin) TheLine->parent = this;
	Id = GO_PLOTSCATT;
}

PlotScatt::PlotScatt(int src):Plot(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		//now set parent in all children
		ForEach(FE_PARENT, this, 0L);
		}
}

PlotScatt::~PlotScatt()
{
	ForEach(FE_FLUSH, 0L, 0L);
	Undo.InvalidGO(this);
}

double
PlotScatt::GetSize(int select)
{
	int i;
	double ft1, ft2;

	switch(select){
	case SIZE_BARMINX:
		if(BarDist.fx >= 0.0001) return BarDist.fx;
		if((!Bars) | (nPoints < 2)) return 1.0f;
		ft1 = fabs(Bars[1]->GetSize(SIZE_XPOS) - Bars[0]->GetSize(SIZE_XPOS));
		for(i = 2; i < nPoints; i++) {
			if(Bars[i] && Bars[i-1]) {
				ft2 = fabs(Bars[i]->GetSize(SIZE_XPOS) - Bars[i-1]->GetSize(SIZE_XPOS));
				if(ft2 < ft1 || ft1 < 0.0001f) ft1 = ft2;
				}
			}
		return BarDist.fx = ft1 > 0.0001 ? ft1 : 1.0;
	case SIZE_BARMINY:
		if(BarDist.fy >= 0.0001) return BarDist.fy;
		if((!Bars) | (nPoints < 2)) return 1.0f;
		ft1 = fabs(Bars[1]->GetSize(SIZE_YPOS) - Bars[0]->GetSize(SIZE_YPOS));
		for(i = 2; i < nPoints; i++) {
			if(Bars[i] && Bars[i-1]) {
				ft2 = fabs(Bars[i]->GetSize(SIZE_YPOS) - Bars[i-1]->GetSize(SIZE_YPOS));
				if(ft2 < ft1 || ft1 < 0.0001f) ft1 = ft2;
				}
			}
		return BarDist.fy = ft1 > 0.0001 ? ft1 : 1.0;
	default:
		return Plot::GetSize(select);
		}
}

bool
PlotScatt::SetSize(int select, double value)
{
	int i;

	switch(select & 0xfff){
	case SIZE_BARMINX:
		BarDist.fx = value;
		return true;
	case SIZE_BARMINY:
		BarDist.fy = value;
		return true;
	case SIZE_SYMBOL:
	case SIZE_SYM_LINE:
		if(Symbols)	for(i = 0; i < nPoints; i++) 
			if(Symbols[i]) Symbols[i]->SetSize(select, value);
		return true;
	case SIZE_ERRBAR:
	case SIZE_ERRBAR_LINE:
		if(Errors)	for(i = 0; i < nPoints; i++) 
			if(Errors[i]) Errors[i]->SetSize(select, value);
		return true;
	case SIZE_BAR_LINE:
	case SIZE_BAR:
	case SIZE_XBASE:
	case SIZE_YBASE:
		if(Bars) for(i = 0; i < nPoints; i++) 
			if(Bars[i]) Bars[i]->SetSize(select, value);
		return true;
	case SIZE_ARROW_LINE:
	case SIZE_ARROW_CAPWIDTH:
	case SIZE_ARROW_CAPLENGTH:
		if(Arrows) for(i = 0; i < nPoints; i++)
			if(Arrows[i]) Arrows[i]->SetSize(select, value);
		return true;
	}
	return false;
}

bool
PlotScatt::SetColor(int select, DWORD col)
{
	int i;
	GraphObj **go = 0L;

	switch(select) {
	case COL_SYM_LINE:
	case COL_SYM_FILL:		go = (GraphObj**)Symbols;		break;
	case COL_ERROR_LINE:	go = (GraphObj**)Errors;		break;
	case COL_BAR_LINE:
	case COL_BAR_FILL:		go = (GraphObj**)Bars;			break;
	case COL_ARROW:			go = (GraphObj**)Arrows;		break;
	default:				return false;
		}
	if(go) for(i = 0; i < nPoints; i++)
		if(go[i]) go[i]->SetColor(select, col);
	return true;
}

void
PlotScatt::DoPlot(anyOutput *o)
{
	if(!parent) return;
	parent->Command(CMD_REG_AXISPLOT, (void*)this, o);
	if(use_xaxis || use_yaxis) {
		ApplyAxes(o);
		ForEach(FE_PLOT, 0L, o);
		parent->Command(CMD_AXIS, 0L, o);
		}
	else {
		ForEach(FE_PLOT, 0L, o);
		}
	dirty = false;
}

bool
PlotScatt::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		if(!CurrGO && ((MouseEvent*)tmpl)->Action == MOUSE_LBUP)
			return ForEach(cmd, tmpl, o);
		return false;
	case CMD_LEGEND:
		if(((GraphObj*)tmpl)->Id != GO_LEGEND) return false;
		if(Bars) for (i = 0; i < nPoints; i++)
			if(Bars[i]) Bars[i]->Command(cmd, tmpl, o);
		if(Symbols) {
			if(TheLine && TheLine->Id == GO_DATALINE) {
				for (i = 0; i < nPoints && i < 100; i++)
					if(Symbols[i]) ((Legend*)tmpl)->HasSym(&TheLine->LineDef, Symbols[i]);
				}
			else {
				for (i = 0; i < nPoints && i < 100; i++)
					if(Symbols[i]) ((Legend*)tmpl)->HasSym(0L, Symbols[i]);
				}
			if(TheLine && TheLine->Id == GO_DATAPOLYGON) TheLine->Command(cmd, tmpl, o);
			}
		else if(TheLine) TheLine->Command(cmd, tmpl, o);
		break;
	case CMD_MRK_DIRTY:
		dirty = true;
	case CMD_SETSCROLL:
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_USEAXIS:
		return UseAxis(*((int*)tmpl));
	case CMD_FLUSH:
		return ForEach(FE_FLUSH, 0L, 0L);
	case CMD_AUTOSCALE:
		if(dirty){
			Bounds.Xmin = Bounds.Ymin = HUGE_VAL;
			Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
			}
		else return true;
		dirty = false;
	case CMD_UPDATE:
		if(cmd == CMD_UPDATE){
			Undo.ObjConf(this, UNDO_CONTINUE);
			dirty = true;
			}
	case CMD_SET_DATAOBJ:
		if(cmd == CMD_SET_DATAOBJ) {
			Id = GO_PLOTSCATT;
			data = (DataObj *)tmpl;	
			}
		ForEach(cmd, tmpl, o);
		return true;
	case CMD_HIDE_MARK:
		return ForEach(cmd, tmpl, o);
	case CMD_MUTATE:		case CMD_REPL_GO:
		dirty = true;
		return ForEach(cmd == CMD_REPL_GO ? FE_REPLGO : FE_MUTATE, tmpl, o);
	case CMD_REG_GO:
		return ForEach(FE_REGGO, tmpl, o);
	case CMD_SYMTEXT:		case CMD_SYMTEXT_UNDO:	case CMD_SYM_RANGETEXT:
	case CMD_SYMTEXTDEF:	case CMD_SYM_TYPE:
		if(Symbols) for(i = 0; i < nPoints; i++)
			if(Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_DL_LINE:		case CMD_DL_TYPE:
		if(DropLines) for(i = 0; i < nPoints; i++)
			if(DropLines[i]) DropLines[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_ERR_TYPE:
		if(Errors) for(i = 0; i < nPoints; i++) 
			if(Errors[i]) Errors[i]->type = *((int*)tmpl);
		return true;
	case CMD_BAR_TYPE:		case CMD_BAR_FILL:
		if(Bars) for(i = 0; i < nPoints; i++) {
			if(Bars[i]) Bars[i]->Command(cmd, tmpl, o);
			}
		return true;
	case CMD_ARROW_TYPE:	case CMD_ARROW_ORG:
		if(Arrows) for(i = 0; i < nPoints; i++) {
			if(Arrows[i]) Arrows[i]->Command(cmd, tmpl, o);
			}
		return true;
	case CMD_DELOBJ:
		dirty = true;
		if(parent && tmpl && o) return ForEach(FE_DELOBJ, tmpl, o);
		break;
	case CMD_SAVE_SYMBOLS:
		return SavVarObs((GraphObj **)Symbols, nPoints, 0L);
	case CMD_SAVE_BARS:
		return SavVarObs((GraphObj **)Bars, nPoints, 0L);
	case CMD_SAVE_ERRS:
		return SavVarObs((GraphObj **)Errors, nPoints, 0L);
	case CMD_SAVE_ARROWS:
		return SavVarObs((GraphObj **)Arrows, nPoints, 0L);
	case CMD_SAVE_DROPLINES:
		return SavVarObs((GraphObj **)DropLines, nPoints, 0L);
		}
	return false;
}

bool
PlotScatt::ForEach(int cmd, void *tmp, anyOutput *o)
{
	int i, j;
	GraphObj **obs[] = {(GraphObj**)Symbols, (GraphObj**)Errors, (GraphObj**)Arrows,
		(GraphObj**)DropLines, (GraphObj**)Labels, (GraphObj**)Bars};
	GraphObj ***go = 0L;
	GraphObj **tmpPlots;
	bool bRedraw, bFound;

	switch(cmd) {
	case FE_MUTATE:
	case FE_REPLGO:
		if((tmpPlots = (GraphObj **)tmp) && tmpPlots[0] && tmpPlots[1]) {
			for(j = 0; j < 6; j++){
				if(obs[j]) for(i = 0; i < nPoints; i++){
					if(obs[j][i] && obs[j][i] == tmpPlots[0]) {
						if(cmd == FE_REPLGO) return ReplaceGO(&obs[j][i], tmpPlots);
						else {
							Undo.MutateGO(&obs[j][i], tmpPlots[1], 0L, o);
							return true;
							}
						}
					}
				}
			if(TheLine == tmpPlots[0]){
				if(cmd == FE_REPLGO) return ReplaceGO((GraphObj**)&TheLine, tmpPlots);
				else {
					Undo.MutateGO((GraphObj**)&TheLine, tmpPlots[1], 0L, o);
					return true;
					}
				}
			}
		return false;
	case FE_PARENT:
		for(j = 0; j < 6; j++){
			if(obs[j]) for(i = 0; i < nPoints; i++){
				if(obs[j][i]) obs[j][i]->parent = this;
				}
			}
		if(TheLine) TheLine->parent = this;
		return true;
	case FE_REGGO:
		ForEach(CMD_REG_GO, tmp, o);
		((notary*)tmp)->AddRegGO(this);
		return true;
	case CMD_REG_GO:
	case CMD_UPDATE:
	case CMD_SET_DATAOBJ:
	case CMD_AUTOSCALE:
		for(j = 0; j < 6; j++){
			if(obs[j]) for(i = 0; i < nPoints; i++){
				if(obs[j][i]) obs[j][i]->Command(cmd, tmp, o);
				}
			}
		if(TheLine) TheLine->Command(cmd, tmp, o);
		return true;
	case FE_PLOT:
		if(TheLine) TheLine->DoPlot(o);
		for(j = 5; j >= 0; j--){
			if(obs[j]) for(i = 0; i < nPoints; i++){
				if(obs[j][i]) obs[j][i]->DoPlot(o);
				}
			}
		return true;
	case FE_FLUSH:
		for(j = 0; j < 6; j++){
			if(obs[j]) {
				for(i = 0; i < nPoints; i++)
					if(obs[j][i]) DeleteGO(obs[j][i]);
				free(obs[j]);
				}
			}
		if(ErrRange) free(ErrRange);	if(yRange) free(yRange);
		if(xRange) free(xRange);		if(LbRange) free(LbRange);
		ErrRange = yRange = xRange = LbRange = 0L;
		if(TheLine) DeleteGO(TheLine);
		Bars = 0L;		Symbols = 0L;		Errors = 0L; 
		Arrows = 0L;	DropLines = 0L;		Labels = 0L;	TheLine = 0L;
		return true;
	case FE_DELOBJ:
		for(j = 0, bRedraw = false, go = 0L; j < 6 && !bRedraw; j++) {
			if(obs[j]) for(i = 0; i < nPoints; i++){
				if(obs[j][i]){
					if(tmp == (void*)obs[j][i]) {
						o->HideMark();
						if(o) o->MouseCursor(MC_WAIT, true);
						Undo.DeleteGO(&obs[j][i], 0L, o);
						switch(j) {
						case 0: go = (GraphObj***)&Symbols;		break;
						case 1: go = (GraphObj***)&Errors;		break;
						case 2: go = (GraphObj***)&Arrows;		break;
						case 3: go = (GraphObj***)&DropLines;	break;
						case 4: go = (GraphObj***)&Labels;		break;
						case 5: go = (GraphObj***)&Bars;		break;
							}
						bRedraw = true;
						break;
						}
					}
				}
			}
		if(!bRedraw && TheLine && tmp == (void *) TheLine) {
			o->HideMark();
			Undo.DeleteGO((GraphObj**)(&TheLine), 0L, o);
			bRedraw = true;
			}
		if(bRedraw && go) for(i = j = 0; i < nPoints; i++) if(go[0][i]) j++;
		if(!j) Undo.DropMemory(this, (void**)go, UNDO_CONTINUE);
		if(bRedraw && dirty) Command(CMD_AUTOSCALE, 0L, o); 
		if(!Bars && !Symbols && !Errors && !Arrows && !TheLine && !DropLines
			&& !Labels) parent->Command(CMD_DELOBJ_CONT, this, o);
		else if(bRedraw) parent->Command(CMD_REDRAW, NULL, o);
		return bRedraw;
	case FE_WRITE:
		if(TheLine) TheLine->FileIO(FILE_WRITE);
		for(j = 0; j < 6; j++){
			if(obs[j]) for(i = 0; i < nPoints; i++){
				if(obs[j][i]) obs[j][i]->FileIO(FILE_WRITE);
				}
			}
		return true;
	case CMD_HIDE_MARK:
		if(!o || !tmp) return false;
		if(bFound =(tmp == (void*)TheLine)) TheLine->DoMark(o, false);
		else for(j = 5; j >= 0 && !bFound; j--){
			if(obs[j]) for(i = 0; i < nPoints && !bFound; i++){
				if(bFound = (tmp == (void*)obs[j][i])) obs[j][i]->DoMark(o, false);
				else if(obs[j][i] && obs[j][i]->Id == GO_MLABEL &&
					obs[j][i]->Command(cmd, tmp, o)) return true;
				}
			}
		return bFound;
	default:							//pass command to all objects
		for(j = 0; j < 6; j++){
			if(obs[j]) for(i = 0; i < nPoints; i++){
				if(obs[j][i]) if(obs[j][i]->Command(cmd, tmp, o)) return true;
				}
			}
		if(TheLine) return (TheLine->Command(cmd, tmp, o));
		return false;
		}
	return false;
}


//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// BarChart is based on scatterplot
BarChart::BarChart(GraphObj *par, DataObj *d):PlotScatt(par, d, 0L)
{
	Id = GO_BARCHART;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Regression line and symbols
Regression::Regression(GraphObj *par, DataObj *d):Plot(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_REGRESSION;
}

Regression::Regression(int src):Plot(0L, 0L)
{
	int i;

	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		//now set parent in all children
		if(rLine) rLine->parent = this;
		if(sde) sde->parent = this;
		if(Symbols)
			for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->parent = this;
		}
}

Regression::~Regression()
{
	Command(CMD_FLUSH, 0L, 0L);
	Undo.InvalidGO(this);
}

double
Regression::GetSize(int select)
{
	return Plot::GetSize(select);
}

bool
Regression::SetSize(int select, double value)
{
	int i;

	switch(select & 0xfff){
	case SIZE_SYMBOL:
	case SIZE_SYM_LINE:
		if(Symbols)	for(i = 0; i < nPoints; i++) 
			if(Symbols[i]) Symbols[i]->SetSize(select, value);
		return true;
	}
	return false;
}

bool
Regression::SetColor(int select, DWORD col)
{
	int i;
	switch(select) {
	case COL_SYM_LINE:
	case COL_SYM_FILL:
		if(Symbols) for(i = 0; i < nPoints; i++) 
			if(Symbols[i]) Symbols[i]->SetColor(select, col);
		return true;
	}
	return false;
}

void
Regression::DoPlot(anyOutput *o)
{
	int i;
	

	parent->Command(CMD_REG_AXISPLOT, (void*)this, o);
	if(use_xaxis || use_yaxis) ApplyAxes(o);
	if(sde){
		if(rLine && sde->Command(CMD_DROP_OBJECT, rLine, o)) rLine = 0L;
		sde->DoPlot(o);
		}
	if(rLine) rLine->DoPlot(o);
	if(Symbols)	for(i = 0; i < nPoints; i++) 
		if(Symbols[i]) Symbols[i]->DoPlot(o);
	if(use_xaxis || use_yaxis) parent->Command(CMD_AXIS, 0L, o);
	dirty = false;
}

bool
Regression::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i, j;
	static MouseEvent *mev;
	bool bEmpty,bRedraw = false;
	LineDEF *ld;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch(mev->Action) {
		case MOUSE_LBUP:
			if(Symbols) for (i = nPoints-1; i >= 0; i--)
				if(Symbols[i] && Symbols[i]->Command(cmd, tmpl, o))return true;
			if(rLine && rLine->Command(cmd, tmpl, o)) return true;
			if(sde && sde->Command(cmd, tmpl, o)) return true;
			break;
			}
		break;
	case CMD_LEGEND:
		if(tmpl && ((GraphObj*)tmpl)->Id == GO_LEGEND && rLine && rLine->Id == GO_REGLINE) {
			ld = rLine->GetLine();
			if(Symbols) {
				for (i = 0; i < nPoints && i < 100; i++)
					if(Symbols[i]) ((Legend*)tmpl)->HasSym(ld, Symbols[i]);
				}
			else ((Legend*)tmpl)->HasFill(ld, 0L);
			return true;
			}
		return false;
	case CMD_MRK_DIRTY:
		if(rLine || sde) Recalc();
		dirty = true;
	case CMD_SETSCROLL:
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_USEAXIS:
		UseAxis(*((int*)tmpl));
		return true;
	case CMD_DROP_OBJECT:
		if(tmpl && ((GraphObj*)tmpl)->Id == GO_REGLINE && !rLine) {
			rLine = (RegLine *)tmpl;
			rLine->parent = this;
			return true;
			}
		break;
	case CMD_FLUSH:
		if(yRange) free(yRange);		if(xRange) free(xRange);
		yRange = xRange = 0L;
		if(rLine) DeleteGO(rLine);
		if(sde) DeleteGO(sde);
		rLine = 0L;		sde = 0L;
		if(Symbols) for (i = nPoints-1; i >= 0; i--)
			if(Symbols[i]) DeleteGO(Symbols[i]);
		if(Symbols) free(Symbols);
		Symbols = 0L;
		return true;
	case CMD_AUTOSCALE:
		if(dirty){
			Bounds.Xmin = Bounds.Ymin = HUGE_VAL;
			Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
			}
		else return true;
		dirty = false;
	case CMD_SET_DATAOBJ:
		if(cmd == CMD_SET_DATAOBJ) {
			Id = GO_REGRESSION;
			data = (DataObj *)tmpl;	
			}
	case CMD_REG_GO:
		if(rLine) rLine->Command(cmd, tmpl, o);
		if(Symbols) for (i = 0; i < nPoints; i++)
			if(Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
		if(cmd == CMD_REG_GO) ((notary*)tmpl)->AddRegGO(this);
		return true;
	case CMD_SYMTEXT:		case CMD_SYM_RANGETEXT:
	case CMD_SYMTEXTDEF:	case CMD_SYM_TYPE:
		if(Symbols) for(i = 0; i < nPoints; i++)
			if(Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_HIDE_MARK:
		return false;
	case CMD_SAVE_SYMBOLS:
		return SavVarObs((GraphObj **)Symbols, nPoints, 0L);
	case CMD_UPDATE:
		if(Symbols) {
			Undo.ObjConf(this, UNDO_CONTINUE);
			for(i = 0; i < nPoints; i++)
				if(Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
			if(rLine || sde) Recalc();
			return true;
			}
		return false;
	case CMD_DELOBJ:
		if(!parent || !o) return false;
		dirty = bEmpty = bRedraw = false;
		if(Symbols) for(i = 0; i < nPoints; i++)
			if(Symbols[i] && (void*)Symbols[i] == tmpl) {
				bRedraw = true;
				o->HideMark();
				Undo.DeleteGO((GraphObj**)(&Symbols[i]), 0L, o);
				for(j = 0, bEmpty = true; j < nPoints; j++) {
					if(Symbols[j]) {
						bEmpty = false;
						break;
						}
					}
				if(!bEmpty && dirty) Command(CMD_AUTOSCALE, 0L, o);
				break;
				}
		if(rLine && (void*)rLine == tmpl) {
			Undo.DeleteGO((GraphObj**)(&rLine), 0L, o);
			if(!Symbols && !sde) parent->Command(CMD_DELOBJ_CONT, this, o);
			else bRedraw = true;
			}
		if(sde && (void*)sde == tmpl) {
			sde->Command(CMD_RMU, 0L, 0L);
			Undo.DeleteGO((GraphObj**)(&sde), 0L, o);
			if(!Symbols && !rLine) parent->Command(CMD_DELOBJ_CONT, this, o);
			else bRedraw = true;
			}
		if(bEmpty && Symbols) {
			Undo.DropMemory(this, (void**)(&Symbols), UNDO_CONTINUE);
			bRedraw = false;
			if(!rLine && !sde) parent->Command(CMD_DELOBJ_CONT, this, o);
			else bRedraw = true;
			}
		if(bRedraw)parent->Command(CMD_REDRAW, 0L, o);
		return bRedraw;
		}
	return false;
}

void
Regression::Recalc()
{
	int i, j;
	long n;
	bool dValid;
	lfPOINT *val;

	if(nPoints <2 || !Symbols || 
		!(val = (lfPOINT*)calloc(nPoints, sizeof(lfPOINT)))) return;
	for(i = 0, n = 0; i < nPoints; i++){
		if(Symbols[i] && Symbols[i]->Id == GO_SYMBOL) {
			val[j = (int)n].fx = Symbols[i]->GetSize(SIZE_XPOS);
			val[j].fy = Symbols[i]->GetSize(SIZE_YPOS);
			dValid = true;
			switch(type & 0x700) {
			case 0x100:					//logarithmic x
				if(dValid = val[j].fx > defs.min4log) val[j].fx = log10(val[j].fx);
				break;
			case 0x200:					//reciprocal x
				if(dValid = fabs(val[j].fx) >defs.min4log) val[j].fx = 1.0/val[j].fx;
				break;
			case 0x300:					//square root x
				if(dValid = fabs(val[j].fx) >defs.min4log) val[j].fx = sqrt(val[j].fx);
				break;
				}
			if(dValid) switch(type & 0x7000) {
			case 0x1000:				//logarithmic y
				if(dValid = val[j].fy > defs.min4log) val[j].fy = log10(val[j].fy);
				break;
			case 0x2000:				//reciprocal y
				if(dValid = fabs(val[j].fy) >defs.min4log) val[j].fy = 1.0/val[j].fy;
				break;
			case 0x3000:				//square root y
				if(dValid = fabs(val[j].fy) >defs.min4log) val[j].fy = sqrt(val[j].fy);
				break;
				}
			if(dValid) n++;
			}
		}
	if(sde && sde->Id == GO_SDELLIPSE) sde->Recalc(val, n);
	if(rLine && rLine->Id == GO_REGLINE) rLine->Recalc(val, n);
	free(val);
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// BubblePlot is a Plot-Class
BubblePlot::BubblePlot(GraphObj *par, DataObj *d):Plot(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_BUBBLEPLOT;
	if (!d && parent) parent->Command(CMD_DELOBJ, this, NULL);
}

BubblePlot::BubblePlot(int src):Plot(0L, 0L)
{
	long i;

	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		BubbleFill.hatch = &BubbleFillLine;
		if(Bubbles)for(i = 0; i< nPoints; i++) {
			if(Bubbles[i])Bubbles[i]->parent = this;
			}
		}
}

BubblePlot::~BubblePlot()
{
	int i;

	if(Bubbles) {
		for(i = 0; i < nPoints; i++) if(Bubbles[i]) DeleteGO(Bubbles[i]);
		free (Bubbles);
		}
	Undo.InvalidGO(this);
}

DWORD
BubblePlot::GetColor(int select)
{
	switch(select) {
	case COL_BUBBLE_FILL:			return BubbleFill.color;
	case COL_BUBBLE_LINE:			return BubbleLine.color;
	case COL_BUBBLE_FILLLINE:	   	return BubbleFillLine.color;
	default:
		return Plot::GetColor(select);
	}
}

void
BubblePlot::DoPlot(anyOutput *o)
{
	int i;

	parent->Command(CMD_REG_AXISPLOT, (void*)this, o);
	if(use_xaxis || use_yaxis) {
		ApplyAxes(o);
		if(Bubbles) for(i = 0; i < nPoints; i++) 
			if(Bubbles[i]) Bubbles[i]->DoPlot(o);
		parent->Command(CMD_AXIS, 0L, o);
		}
	else {
		if(Bubbles) for(i = 0; i < nPoints; i++) 
			if(Bubbles[i]) Bubbles[i]->DoPlot(o);
		}
	dirty = false;
}

bool
BubblePlot::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;
	static MouseEvent *mev;
	GraphObj **tmpPlots;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch(mev->Action) {
		case MOUSE_LBUP:
			//select objects invers to plot order
			if(Bubbles && !CurrGO) for(i = nPoints-1; i >=0; i--)
				if(Bubbles[i]) if(Bubbles[i]->Command(cmd, tmpl, o))break;
			break;
			}
		break;
	case CMD_REPL_GO:
		if((tmpPlots = (GraphObj **)tmpl) && tmpPlots[0] && tmpPlots[1] && Bubbles) {
			for(i = 0; i < nPoints; i++) if(Bubbles[i] && Bubbles[i] == tmpPlots[0]) { 
				return ReplaceGO((GraphObj**)&Bubbles[i], tmpPlots);
				}
			}
		return false;
	case CMD_MRK_DIRTY:
		dirty = true;
	case CMD_SETSCROLL:
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_USEAXIS:
		UseAxis(*((int*)tmpl));
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_BUBBLEPLOT;
		data = (DataObj *)tmpl;
	case CMD_UPDATE:
		if(cmd == CMD_UPDATE) Undo.ObjConf(this, UNDO_CONTINUE);
	case CMD_BUBBLE_ATTRIB:
	case CMD_BUBBLE_TYPE:
	case CMD_BUBBLE_FILL:
	case CMD_BUBBLE_LINE:
		if(Bubbles) for(i = 0; i < nPoints; i++)
			if(Bubbles[i]) Bubbles[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_DELOBJ:
		if(Bubbles && parent) for(i = 0; i < nPoints; i++) {
			o->HideMark();
			if(Bubbles[i] && tmpl == (void *)Bubbles[i]) {
				Undo.DeleteGO((GraphObj**)(&Bubbles[i]), 0L, o);
				parent->Command(CMD_REDRAW, 0L, o);
				return true;
				}
			}
		break;
	case CMD_SAVE_SYMBOLS:
		return SavVarObs((GraphObj **)Bubbles, nPoints, 0L);
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// PolarPlot is a Plot-Class
PolarPlot::PolarPlot(GraphObj *par, DataObj *d):Plot(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_POLARPLOT;
	if (!d && parent) parent->Command(CMD_DELOBJ, this, NULL);
}

PolarPlot::PolarPlot(int src):Plot(0L, 0L)
{
	int i;

	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		Fill.hatch = &FillLine;
		//now set parent in all children
		if(Plots) 
			for(i = 0; i < nPlots; i++) if(Plots[i]) Plots[i]->parent = this;
		if(Axes) 
			for(i = 0; i < nAxes; i++) if(Axes[i]) Axes[i]->parent = this;
		}
}

PolarPlot::~PolarPlot()
{
	int i;

	if(Plots){
		for(i = 0; i < nPlots; i++) if(Plots[i]) DeleteGO(Plots[i]);
		free(Plots);		Plots = 0L;
		}
	if(Axes){
		for(i = 0; i < nAxes; i++) if(Axes[i]) DeleteGO(Axes[i]);
		free(Axes);			Axes = 0L;
		}
	Undo.InvalidGO(this);
}

double
PolarPlot::GetSize(int select)
{
	switch(select) {
	case SIZE_BOUNDS_XMIN:		return Bounds.Xmin;
	case SIZE_BOUNDS_XMAX:		return Bounds.Xmax;
	case SIZE_BOUNDS_YMIN:		return Bounds.Ymin;
	case SIZE_BOUNDS_YMAX:		return Bounds.Ymax;
	case SIZE_BOUNDS_LEFT:		return (Axes && Axes[0])?(((Axis*)Axes[0])->GetAxis())->min:0.0;
	case SIZE_BOUNDS_RIGHT:		return (Axes && Axes[0])?(((Axis*)Axes[0])->GetAxis())->max:0.0;
	case SIZE_BOUNDS_TOP:		return (Axes && Axes[1])?(((Axis*)Axes[1])->GetAxis())->max:0.0;
	case SIZE_BOUNDS_BOTTOM:	return (Axes && Axes[1])?(((Axis*)Axes[1])->GetAxis())->min:0.0;
	case SIZE_YAXISX:
		if(!CurrDisp) return 0.0;
		if((((Axis*)Axes[1])->GetAxis())->flags & AXIS_X_DATA) 
			return CurrDisp->fx2fix((((Axis*)Axes[1])->GetAxis())->loc[0].fx);
		else return CurrDisp->co2fix((((Axis*)Axes[1])->GetAxis())->loc[0].fx);
	case SIZE_XAXISY:
		if(!CurrDisp) return 0.0;
		if((((Axis*)Axes[0])->GetAxis())->flags & AXIS_Y_DATA) 
			return CurrDisp->fy2fiy((((Axis*)Axes[0])->GetAxis())->loc[0].fy);
		else return CurrDisp->co2fiy((((Axis*)Axes[0])->GetAxis())->loc[0].fy);
	case SIZE_XCENTER:			return (((Axis*)Axes[0])->GetAxis())->Center.fx;
	case SIZE_YCENTER:			return (((Axis*)Axes[0])->GetAxis())->Center.fy;
	default:
		if(parent) return parent->GetSize(select);
		}
	return defs.GetSize(select);
}

void
PolarPlot::DoPlot(anyOutput *o)
{
	int i;

	if(o) CurrDisp = o;
	else return;
	if(!parent) return;
	CurrRect.Xmin = CurrRect.Xmax = (((Axis*)Axes[1])->GetAxis())->Center.fx + 
		parent->GetSize(SIZE_GRECT_LEFT);
	CurrRect.Xmin -= (((Axis*)Axes[0])->GetAxis())->Radius;	
	CurrRect.Xmax += (((Axis*)Axes[0])->GetAxis())->Radius;
	CurrRect.Ymin = CurrRect.Ymax = (((Axis*)Axes[0])->GetAxis())->Center.fy +
		parent->GetSize(SIZE_GRECT_TOP);
	CurrRect.Ymin -= (((Axis*)Axes[0])->GetAxis())->Radius;
	CurrRect.Ymax += (((Axis*)Axes[0])->GetAxis())->Radius;
	(((Axis*)Axes[0])->GetAxis())->Start = ((((Axis*)Axes[0])->GetAxis())->flags & AXIS_INVERT) ? -offs : offs;
	o->SetRect(CurrRect, defs.cUnits, ((Axis*)Axes[0])->GetAxis(), ((Axis*)Axes[1])->GetAxis());
	o->SetFill(&Fill);
	if(Axes) for(i = 0; i < nAxes; i++) {
		if(i == 1) {
			if(!(type & 0x01) && Axes[i]) Axes[i]->DoPlot(o);
			}
		else if(Axes[i]) Axes[i]->DoPlot(o);
		}
	if(Plots) for(i = 0; i < nPlots; i++) if(Plots[i]) {
		if(Plots[i]->Id >= GO_PLOT && Plots[i]->Id < GO_GRAPH) {
			if(((Plot*)Plots[i])->hidden == 0) Plots[i]->DoPlot(o);
			}
		else Plots[i]->DoPlot(o);
		}
	rDims.left = o->co2ix(CurrRect.Xmin);	rDims.right = o->co2ix(CurrRect.Xmax);
	rDims.top = o->co2iy(CurrRect.Ymin);	rDims.bottom = o->co2iy(CurrRect.Ymax);
	if(parent) parent->Command(CMD_AXIS, 0L, o);
}

bool
PolarPlot::Command(int cmd, void *tmpl, anyOutput *o)
{
	GraphObj **tmpPlots;
	int i;
	AxisDEF *ad0, *ad1;
	double tmp;

	switch (cmd) {
	case CMD_SET_DATAOBJ:
		Id = GO_POLARPLOT;
		data = (DataObj *)tmpl;	
	case CMD_UPDATE:
		if(Axes) for(i = 0; i< nAxes; i++) if(Axes[i]) Axes[i]->Command(cmd, tmpl, o);
	case CMD_LEGEND:
		if(Plots) for(i = 0; i < nPlots; i++) if(Plots[i]) Plots[i]->Command(cmd, tmpl, o);
		return false;
	case CMD_OBJTREE:
		if(!tmpl) return false;
		if(Plots) for(i = 0; i < nPlots; i++) if(Plots[i]) 
			((ObjTree*)tmpl)->Command(CMD_REG_GO, Plots[i], 0L);
		return true;
	case CMD_HIDE_MARK:
		if(!tmpl || !o) return false;
		//do all axes
		if(Axes)for(i = nAxes-1; i>=0; i--) {
			if(tmpl == (void*)Axes[i]){
				Axes[i]->DoMark(CurrDisp, false);		return true;
				}
			else if(Axes[i]->Id == GO_AXIS) {
				if(Axes[i]->Command(cmd, tmpl, o))		return true;
				}
			}
		//do all plots
		if(Plots)for(i = nPlots-1; i>=0; i--) {
			if(tmpl == (void*)Plots[i]){
				Plots[i]->DoMark(CurrDisp, false);		return true;
				}
			else if(Plots[i] && (Plots[i]->Id == GO_MLABEL || Plots[i]->Id == GO_LEGEND || 
				(Plots[i]->Id >= GO_PLOT && Plots[i]->Id < GO_GRAPH))) {
				if(Plots[i]->Command(cmd, tmpl, o))		return true;
				}
			}
		return false;
	case CMD_MOUSE_EVENT:
		if(o) switch(((MouseEvent*)tmpl)->Action) {
		case MOUSE_LBUP:
			if(Axes) for(i = 0; i< nAxes; i++) if(Axes[i])
				if(Axes[i]->Command(cmd, tmpl, o))return true;
			if(Plots) for(i = 0; i < nPlots; i++) if(Plots[i]) 
				Plots[i]->Command(cmd, tmpl, o);
			if(!CurrGO && IsInRect(&rDims, ((MouseEvent*)tmpl)->x, ((MouseEvent*)tmpl)->y)){
				CurrGO = this;
				o->ShowMark(&rDims, MRK_INVERT);
				return true;
				}
			break;
			}
		return false;
	case CMD_REPL_GO:
		if(!(tmpPlots = (GraphObj **)tmpl) || !tmpPlots[0] || !tmpPlots[1]) return false;
		if(Axes) for(i = 0; i < nAxes; i++) if(Axes[i] && Axes[i] == tmpPlots[0]){
			return ReplaceGO(&Axes[i], tmpPlots);
			}
		break;
	case CMD_SETSCROLL:
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_DELOBJ_CONT:
	case CMD_DELOBJ:
		if(Plots && nPlots) for(i = 0; i < nPlots; i++) if(tmpl == (void*)Plots[i]) {
			Undo.DeleteGO((GraphObj**)(&Plots[i]), cmd == CMD_DELOBJ_CONT ? UNDO_CONTINUE : 0L, o);
			if(parent)parent->Command(CMD_REDRAW, NULL, o);
			return true;
			}
		if(Axes && nAxes > 1 && (tmpl == (void*)Axes[0] || tmpl == (void*)Axes[1]))
			InfoBox("Axes required for scaling\ncannot be deleted.");
		break;
	case CMD_AXIS:		//axis changed: reconstruct corresponding axis
		if(Axes && nAxes >1 && tmpl && Axes[0] && Axes[1]) {
			ad0 = ((Axis*)Axes[0])->GetAxis();		ad1 = ((Axis*)Axes[1])->GetAxis();
			if(tmpl == Axes[0]) {
				CheckNewFloat(&ad1->loc[1].fy, ad1->loc[1].fy, ad0->Center.fy, this, UNDO_CONTINUE);
				tmp = ad1->loc[1].fy - ad0->Radius;
				CheckNewFloat(&ad1->loc[0].fy, ad1->loc[0].fy, tmp, this, UNDO_CONTINUE);
				}
			if(tmpl == Axes[1]) {
				CheckNewFloat(&ad0->Center.fy, ad0->Center.fy, ad1->loc[1].fy, this, UNDO_CONTINUE);
				tmp = fabs(ad1->loc[1].fy - ad1->loc[0].fy);
				CheckNewFloat(&ad0->Radius, ad0->Radius, tmp, this, UNDO_CONTINUE);
				}
			}
		break;	
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// BoxPlot is a Plot-Class
BoxPlot::BoxPlot(GraphObj *par, DataObj *d):Plot(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_BOXPLOT;
	if (!d && parent) parent->Command(CMD_DELOBJ, this, NULL);
}

BoxPlot::BoxPlot(int src):Plot(0L, 0L)
{
	int i;

	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		//now set parent in all children
		if(Boxes) 
			for(i = 0; i < nPoints; i++) if(Boxes[i]) Boxes[i]->parent = this;
		if(Whiskers) 
			for(i = 0; i < nPoints; i++) if(Whiskers[i]) Whiskers[i]->parent = this;
		if(Symbols) 
			for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->parent = this;
		if(TheLine) TheLine->parent = this;
		}
}

BoxPlot::BoxPlot(GraphObj *par, DataObj *dt, int mode, int c1, int c2, int c3):Plot(par, dt)
{
	int i, nr;
	lfPOINT fp;

	FileIO(INIT_VARS);		Id = GO_BOXPLOT;	fp.fx = fp.fy = 0.0;
	if(data && data->GetSize(&i, &nr)) {
		nPoints = nr;
		Bounds.Xmin = Bounds.Ymin = HUGE_VAL;
		Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
		if(Boxes = (Box**)calloc(nr, sizeof(Box*))) for(i = 0; i < nr; i++) {
			if(mode == 1) Boxes[i] = new Box(this, data, fp, fp, 0, c1, i, c2, i, c1, i, c3, i);
			else Boxes[i] = new Box(this, data, fp, fp, 0, c2, i, c1, i, c3, i, c1, i);
			if(Boxes[i]) Boxes[i]->Command(CMD_AUTOSCALE, 0L, 0L);
			}
		}
	BoxDist.fx = BoxDist.fy = GetSize(mode == 2  ? SIZE_BOXMINY : SIZE_BOXMINX);
}

BoxPlot::~BoxPlot()
{
	int i;

	if(Whiskers) {
		for(i = 0; i < nPoints; i++) if(Whiskers[i]) delete(Whiskers[i]);
		free (Whiskers);
		}
	if(Boxes) {
		for(i = 0; i < nPoints; i++) if(Boxes[i]) delete(Boxes[i]);
		free (Boxes);
		}
	if(Symbols) {
		for(i = 0; i < nPoints; i++) if(Symbols[i]) delete(Symbols[i]);
		free (Symbols);
		}
	if(TheLine) delete(TheLine);
	Undo.InvalidGO(this);
}

double
BoxPlot::GetSize(int select)
{
	int i;
	double ft1, ft2;

	switch(select){
	case SIZE_BOXMINX:
		if(BoxDist.fx >= 0.0001) return BoxDist.fx;
		if((!Boxes) | (nPoints < 2)) return 1.0;
		ft1 = ft2 = 1.0;
		if(Boxes[0] && Boxes[1])  ft1 = fabs(Boxes[1]->GetSize(SIZE_XPOS) - 
			Boxes[0]->GetSize(SIZE_XPOS));
		else return 1.0;
		for(i = 2; i < nPoints; i++) {
			if(Boxes[i] && Boxes[i-1]) ft2 = fabs(Boxes[i]->
				GetSize(SIZE_XPOS) - Boxes[i-1]->GetSize(SIZE_XPOS));
			if(ft2 < ft1 || ft1 < 0.0001) ft1 = ft2;
			}
		return BoxDist.fx = ft1 > 0.0001 ? ft1 : 1.0;
	case SIZE_BOXMINY:
		if(BoxDist.fy >= 0.0001) return BoxDist.fy;
		if((!Boxes) | (nPoints < 2)) return 1.0f;
		ft1 = ft2 = 1.0f;
		if(Boxes[0] && Boxes[1]) ft1 = fabs(Boxes[1]->GetSize(SIZE_YPOS) -
			Boxes[0]->GetSize(SIZE_YPOS));
		else return 1.0f;
		for(i = 2; i < nPoints; i++) {
			if(Boxes[i] && Boxes[i-1]) ft2 = fabs(Boxes[i]->
				GetSize(SIZE_YPOS) - Boxes[i-1]->GetSize(SIZE_YPOS));
			if(ft2 < ft1 || ft1 < 0.0001f) ft1 = ft2;
			}
		return BoxDist.fy = ft1 > 0.0001 ? ft1 : 1.0f;
	default:
		return Plot::GetSize(select);
		}
}

bool
BoxPlot::SetSize(int select, double value)
{
	int i;

	switch(select & 0xfff){
	case SIZE_SYMBOL:
	case SIZE_SYM_LINE:
		if(Symbols) for(i = 0; i < nPoints; i++) 
			if(Symbols[i]) Symbols[i]->SetSize(select, value);
		return true;
	case SIZE_WHISKER:
	case SIZE_WHISKER_LINE:
		if(Whiskers) for(i = 0; i < nPoints; i++) 
			if(Whiskers[i]) Whiskers[i]->SetSize(select, value);
		return true;
	case SIZE_BOX:
	case SIZE_BOX_LINE:
		if(Boxes) for(i = 0; i < nPoints; i++) 
			if(Boxes[i]) Boxes[i]->SetSize(select, value);
		return true;
	}
	return false;
}

bool
BoxPlot::SetColor(int select, DWORD col)
{
	int i;

	switch(select) {
	case COL_SYM_LINE:
	case COL_SYM_FILL:
		if(Symbols)	for(i = 0; i < nPoints; i++) 
			if(Symbols[i]) Symbols[i]->SetColor(select, col);
		return true;
	case COL_WHISKER:
		if(Whiskers) for(i = 0; i < nPoints; i++)
			if(Whiskers[i]) Whiskers[i]->SetColor(select, col);
		return true;
	case COL_BOX_LINE:
		if(Boxes) for(i = 0; i < nPoints; i++)
			if(Boxes[i]) Boxes[i]->SetColor(select, col);
		return true;
	default:
		return false;
		}
}

void
BoxPlot::DoPlot(anyOutput *o)
{
	int i;

	if(!parent) return;
	parent->Command(CMD_REG_AXISPLOT, (void*)this, o);
	if(use_xaxis || use_yaxis) ApplyAxes(o);
	if(TheLine) TheLine->DoPlot(o);
	if(Whiskers) 
		for(i = 0; i < nPoints; i++) if(Whiskers[i]) Whiskers[i]->DoPlot(o);
	if(Boxes) 
		for(i = 0; i < nPoints; i++) if(Boxes[i]) Boxes[i]->DoPlot(o);
	if(Symbols)
		for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->DoPlot(o);
	dirty = false;
	if(use_xaxis || use_yaxis)parent->Command(CMD_AXIS, 0L, o);
}

bool
BoxPlot::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i, j;
	bool bRedraw;
	MouseEvent *mev;
	GraphObj **obs[] = {(GraphObj**)Boxes, (GraphObj**)Whiskers, (GraphObj**)Symbols};
	GraphObj ***go;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch(mev->Action) {
		case MOUSE_LBUP:
			for(j = 2; j >= 0 && !CurrGO; j--) {	//invers to plot order
				if(obs[j]) for (i = nPoints-1; i >= 0; i--)
					if(obs[j][i]) if(obs[j][i]->Command(cmd, tmpl, o))break;
				}
			if(TheLine && !CurrGO) TheLine->Command(cmd, tmpl, o);
			break;
			}
		break;
	case CMD_LEGEND:
		if(Boxes) for (i = 0; i < nPoints; i++)
			if(Boxes[i]) Boxes[i]->Command(cmd, tmpl, o);
		break;
	case CMD_SET_DATAOBJ:
		Id = GO_BOXPLOT;		data = (DataObj *)tmpl;		dirty = true;
	case CMD_AUTOSCALE:
		if(cmd == CMD_AUTOSCALE){
			if(!dirty) return true;
			Bounds.Xmin = Bounds.Ymin = HUGE_VAL;
			Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
			}
	case CMD_UPDATE:
		if(cmd == CMD_UPDATE) Undo.ObjConf(this, UNDO_CONTINUE);
		for(j = 0; j < 3; j++) {
			if(obs[j]) for (i = 0; i < nPoints; i++)
				if(obs[j][i]) obs[j][i]->Command(cmd, tmpl, o);
			}
		if(TheLine) TheLine->Command(cmd, tmpl, o);
		if(cmd == CMD_AUTOSCALE && parent && parent->Id > GO_PLOT && parent->Id < GO_GRAPH){
			((Plot*)parent)->CheckBounds(Bounds.Xmin, Bounds.Ymin);
			((Plot*)parent)->CheckBounds(Bounds.Xmax, Bounds.Ymax);
			}
		return true;
	case CMD_USEAXIS:
		UseAxis(*((int*)tmpl));
		return true;
	case CMD_MRK_DIRTY:
		dirty = true;
	case CMD_SETSCROLL:
	case CMD_REDRAW:
		if(parent)return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_DELOBJ:
		if(!parent || !o) return false;
		for(j = 0, bRedraw = false, go = 0L; j < 3 && !bRedraw; j++) {
			if(obs[j]) for(i = 0; i < nPoints; i++)
				if(obs[j][i] && tmpl == (void*)obs[j][i]) {
					o->HideMark();
					Undo.DeleteGO(&obs[j][i], 0L, o);
					switch(j) {
					case 0: go = (GraphObj***)&Boxes;		break;
					case 1: go = (GraphObj***)&Whiskers;	break;
					case 2: go = (GraphObj***)&Symbols;		break;
						}
					bRedraw = true;
					break;
					}
			}
		if(!bRedraw && TheLine && tmpl == (void *) TheLine) {
			o->HideMark();
			Undo.DeleteGO((GraphObj**)(&TheLine), 0L, o);
			bRedraw = true;
			}
		if(bRedraw && go) for(i = j = 0; i < nPoints; i++) if(go[0][i]) j++;
			if(!j) Undo.DropMemory(this, (void**)go, UNDO_CONTINUE);
		if(!Boxes && !Whiskers && !Symbols && !TheLine) 
			parent->Command(CMD_DELOBJ_CONT, this, o);
		else if(bRedraw) parent->Command(CMD_REDRAW, NULL, o);
		return bRedraw;
	case CMD_SYMTEXT:
	case CMD_SYM_RANGETEXT:
	case CMD_SYMTEXTDEF:
	case CMD_SYM_TYPE:
		if(Symbols) for(i = 0; i < nPoints; i++)
			if(Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_SAVE_SYMBOLS:
		return SavVarObs((GraphObj **)Symbols, nPoints, 0L);
	case CMD_SAVE_BARS:
		return SavVarObs((GraphObj **)Boxes, nPoints, 0L);
	case CMD_SAVE_BARS_CONT:
		return SavVarObs((GraphObj **)Boxes, nPoints, UNDO_CONTINUE);
	case CMD_SAVE_ERRS:
		return SavVarObs((GraphObj **)Whiskers, nPoints, 0L);
	case CMD_BOX_TYPE:
		BoxDist.fy = BoxDist.fx = 0.0;
	case CMD_BOX_FILL:
		if(Boxes) for (i = 0; i < nPoints; i++)
			if(Boxes[i]) Boxes[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_WHISKER_STYLE:
		if(Whiskers) for (i = 0; i < nPoints; i++)
			if(Whiskers[i]) Whiskers[i]->Command(cmd, tmpl, o);
		return true;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Density distribution plot
DensDisp::DensDisp(GraphObj *par, DataObj *d):Plot(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_DENSDISP;
	if (!d && parent) parent->Command(CMD_DELOBJ, this, NULL);
}

DensDisp::DensDisp(int src):Plot(0L, 0L)
{
	int i;

	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		//now set parent in all children
		if(Boxes) 
			for(i = 0; i < nPoints; i++) if(Boxes[i]) Boxes[i]->parent = this;
		}
}

DensDisp::~DensDisp()
{
	int i;

	if(Boxes) {
		for(i = 0; i < nPoints; i++) if(Boxes[i]) DeleteGO(Boxes[i]);
		free (Boxes);
		}
	if(yRange) free(yRange);		if(xRange) free(xRange);
	yRange = xRange = 0L;
	Undo.InvalidGO(this);
}

bool
DensDisp::SetSize(int select, double value)
{
	int i;

	switch(select & 0xfff){
	case SIZE_BOX_LINE:
		if(Boxes) for(i = 0; i < nPoints; i++) 
			if(Boxes[i]) Boxes[i]->SetSize(select, value);
		return true;
	}
	return false;
}

bool
DensDisp::SetColor(int select, DWORD col)
{
	int i;

	switch(select) {
	case COL_BOX_LINE:
		if(Boxes) for(i = 0; i < nPoints; i++)
			if(Boxes[i]) Boxes[i]->SetColor(select, col);
		return true;
	default:
		return false;
		}
}

void
DensDisp::DoPlot(anyOutput *o)
{
	int i;

	if(!parent) return;
	parent->Command(CMD_REG_AXISPLOT, (void*)this, o);
	if(use_xaxis || use_yaxis) {
		ApplyAxes(o);
		if(Boxes) for(i = 0; i < nPoints; i++) 
			if(Boxes[i]) Boxes[i]->DoPlot(o);
		parent->Command(CMD_AXIS, 0L, o);
		}
	else {
		if(Boxes) for(i = 0; i < nPoints; i++) 
			if(Boxes[i]) Boxes[i]->DoPlot(o);
		}
	dirty = false;
}

bool
DensDisp::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;
	MouseEvent *mev;

	switch (cmd) {
	case CMD_MRK_DIRTY:
		dirty = true;
	case CMD_SETSCROLL:
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_USEAXIS:
		UseAxis(*((int*)tmpl));
		return true;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch(mev->Action) {
		case MOUSE_LBUP:
			for(i = nPoints-1; i >= 0 && !CurrGO; i--) {
				if(Boxes[i] && Boxes[i]->Command(cmd, tmpl, o))break;
				}
			break;
			}
		break;
	case CMD_SET_DATAOBJ:
		for(i = 0; i < nPoints; i++) if(Boxes[i]) Boxes[i]->Command(cmd, tmpl, o);
		Id = GO_DENSDISP;
		data = (DataObj *)tmpl;
		return true;
	case CMD_DELOBJ:
		if(!parent || !o) return false;
		for(i = 0; i < nPoints; i++) {
			if(Boxes[i] && tmpl == (void*)Boxes[i]){
				o->HideMark();
				Undo.DeleteGO((GraphObj**)(&Boxes[i]), 0L, o);
				return parent->Command(CMD_REDRAW, NULL, o);
				}
			}
		break;
	case CMD_AUTOSCALE:
		if(dirty){
			Bounds.Xmin = Bounds.Ymin = HUGE_VAL;
			Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
			}
		else return true;
		dirty = false;
	case CMD_BOX_TYPE:
	case CMD_BOX_FILL:
		if(Boxes) for (i = 0; i < nPoints; i++)
			if(Boxes[i]) Boxes[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_UPDATE:
		Undo.ObjConf(this, UNDO_CONTINUE);
		DoUpdate();
		return true;
		}
	return false;
}

void
DensDisp::DoUpdate()
{
	AccRange *rX, *rY;
	int i, j, k, l, ic, n;
	double v, w;
	lfPOINT fp1, fp2;

	if(xRange && yRange && (rX = new AccRange(xRange)) && (rY = new AccRange(yRange))) {
		if((n=rX->CountItems()) == rY->CountItems()) {
			if(Boxes) {
				for(i = 0; i < nPoints; i++) if(Boxes[i]) DeleteGO(Boxes[i]);
				free (Boxes);
				}
			Bounds.Xmin = Bounds.Ymin = HUGE_VAL;		Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
			Boxes = (Box**)calloc(nPoints = n, sizeof(Box*));
			rX->GetFirst(&i, &j);	rY->GetFirst(&k, &l);
			rX->GetNext(&i, &j);	rY->GetNext(&k, &l);
			for(ic = 0; ic < n && !data->GetValue(j, i, &v); ic++) {
				rX->GetNext(&i, &j);	rY->GetNext(&k, &l);
				}
			rX->GetNext(&i, &j);	rY->GetNext(&k, &l);
			if(type & 0x10){			//vertical ?
				fp2.fx = 0;		fp2.fy = v;
				}
			else {
				fp2.fx = v;		fp2.fy = 0.0;
				}
			ic = 0;
			do {
				if(data->GetValue(j, i, &v) && data->GetValue(l, k, &w)){
					memcpy(&fp1, &fp2, sizeof(lfPOINT));
					if(type & 0x10) {
						CheckBounds(w, fp1.fy);			CheckBounds(-w, v);
						fp2.fy = v;		
						switch(type & 0x3) {
						case 1:		fp1.fx = fp2.fx = w/2.0;		break;
						case 2:		fp1.fx = fp2.fx = -w/2.0;		break;
						default:	fp2.fx = 0.0;					break;
							}
						}
					else {
						CheckBounds(fp1.fx, w);			CheckBounds(v, -w);
						fp2.fx = v;
						switch(type & 0x3) {
						case 1:		fp1.fy = fp2.fy = w/2.0;		break;
						case 2:		fp1.fy = fp2.fy = -w/2.0;		break;
						default:	fp2.fy = 0.0;					break;
							}
						}
					if(Boxes[ic] = new Box(this, data, fp1, fp2, BAR_WIDTHDATA))
						Boxes[ic]->SetSize(SIZE_BOX, (type &0x03) ? w/2.0 : w);
					}
				ic++;
				}while(rX->GetNext(&i, &j) && rY->GetNext(&k, &l));
			SetSize(SIZE_BOX_LINE, DefLine.width);
			SetColor(COL_BOX_LINE, DefLine.color);
			Command(CMD_BOX_FILL, (void*)&DefFill, 0L);
			}
		else InfoBox("Error updating Plot!");
		delete(rX);		delete(rY);
		}
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Stacked bars consist of several box-plots
StackBar::StackBar(GraphObj *par, DataObj *d):Plot(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_STACKBAR;
	if (!d && parent) parent->Command(CMD_DELOBJ, this, NULL);
}

StackBar::StackBar(int src):Plot(0L, 0L)
{
	int i;

	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		//now set parent in all children
		if(Boxes) for(i = 0; i < numPlots; i++)
			if(Boxes[i]) Boxes[i]->parent = this;
		if(xyPlots) for(i = 0; i < numXY; i++)
			if(xyPlots[i]) xyPlots[i]->parent = this;
		if(Polygons) for(i = 0; i < numPG; i++)
			if(Polygons[i]) Polygons[i]->parent = this;
		if(Lines) for(i = 0; i < numPL; i++)
			if(Lines[i]) Lines[i]->parent = this;
		}
}

StackBar::~StackBar()
{
	int i;

	if(Boxes){
		for(i = 0; i < numPlots; i++) if(Boxes[i]) DeleteGO(Boxes[i]);
		free(Boxes);
		}
	if(xyPlots){
		for(i = 0; i < numXY; i++) if(xyPlots[i]) DeleteGO(xyPlots[i]);
		free(xyPlots);
		}
	if(Polygons) {
		for(i = 0; i < numPG; i++) if(Polygons[i]) DeleteGO(Polygons[i]);
		free(Polygons);
		}
	if(Lines) {
		for(i = 0; i < numPL; i++) if(Lines[i]) DeleteGO(Lines[i]);
		free(Lines);
		}
	if(ssXrange) free(ssXrange);	if(ssYrange) free(ssYrange);
	if(CumData) delete CumData;		CumData = 0L;
	Undo.InvalidGO(this);
}

bool
StackBar::SetSize(int select, double value)
{
	int i;

	switch(select & 0xfff){
	case SIZE_BAR:
		if(xyPlots) for(i = 0; i < numXY; i++)
			if(xyPlots[i]) xyPlots[i]->SetSize(select, value);
		return true;
	case SIZE_BOX:
	case SIZE_BOX_LINE:
		if(Boxes) for(i = 0; i < numPlots; i++) 
			if(Boxes[i]) Boxes[i]->SetSize(select, value);
		return true;
	}
	return false;
}

void
StackBar::DoPlot(anyOutput *o)
{
	int i;
	double dx, dy;
	fRECT oldREC;

	if(!o || !parent) return;
	parent->Command(CMD_REG_AXISPLOT, (void*)this, o);
	if(use_xaxis || use_yaxis) ApplyAxes(o);
	dx = o->un2fix(dspm.fx);		dy = o->un2fiy(dspm.fy);
	memcpy(&oldREC, &o->Box1, sizeof(fRECT));
	if(Boxes) for(i = 0; i < numPlots; i++) if(Boxes[i]) {
		if(Boxes[i]->Id >= GO_PLOT && Boxes[i]->Id < GO_GRAPH) {
			if(((Plot*)Boxes[i])->hidden == 0) Boxes[i]->DoPlot(o);
			}
		else Boxes[i]->DoPlot(o);
		}
	if(xyPlots) for(i = 0; i < numXY; i++) if(xyPlots[i]) {
		if(xyPlots[i]->Id >= GO_PLOT && xyPlots[i]->Id < GO_GRAPH) {
			if(((Plot*)xyPlots[i])->hidden == 0) xyPlots[i]->DoPlot(o);
			}
		else xyPlots[i]->DoPlot(o);
		}
	if(Polygons) for(i = 0; i < numPG; i++)
		if(Polygons[i]) Polygons[i]->DoPlot(o);
	if(Lines) for(i = numPL-1; i >= 0; i--){
		o->Box1.Xmin = oldREC.Xmin + dx*i;
		o->Box1.Ymin = oldREC.Ymin + dy*i;
		o->Box1.Xmax = oldREC.Xmax + dx*i;
		o->Box1.Ymax = oldREC.Ymax + dy*i;
		if(Lines[i]) Lines[i]->DoPlot(o);
		}
	dirty = false;
	memcpy(&o->Box1, &oldREC, sizeof(fRECT));
	if(use_xaxis || use_yaxis) parent->Command(CMD_AXIS, 0L, o);
}

bool
StackBar::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;
	static MouseEvent *mev;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch(mev->Action) {
		case MOUSE_LBUP:
			if(Boxes && !CurrGO) for(i = 0; i < numPlots; i++)
				if(Boxes[i]) if(Boxes[i]->Command(cmd, tmpl, o))return true;
			if(xyPlots && !CurrGO) for(i = 0; i < numXY; i++)
				if(xyPlots[i]) if(xyPlots[i]->Command(cmd, tmpl, o))return true;
			if(Polygons && !CurrGO) for(i = 0; i < numPG; i++)
				if(Polygons[i]) if(Polygons[i]->Command(cmd, tmpl, o))return true;
			if(Lines && !CurrGO) for(i = 0; i < numPL; i++)
				if(Lines[i]) if(Lines[i]->Command(cmd, tmpl, o))return true;
			break;
			}
		break;
	case CMD_OBJTREE:
		if(!tmpl) return false;
		if(Boxes) for(i = 0; i < numPlots; i++) if(Boxes[i]) 
			((ObjTree*)tmpl)->Command(CMD_REG_GO, Boxes[i], 0L);
		if(xyPlots) for(i = 0; i < numXY; i++) if(xyPlots[i]) 
			((ObjTree*)tmpl)->Command(CMD_REG_GO, xyPlots[i], 0L);
		return true;
	case CMD_LEGEND:
		if(Boxes) for (i = 0; i < numPlots; i++)
			if(Boxes[i]) Boxes[i]->Command(cmd, tmpl, o);
		if(Polygons) for (i = numPG-1; i >= 0; i--)
			if(Polygons[i]) Polygons[i]->Command(cmd, tmpl, o);
		break;
	case CMD_USEAXIS:
		UseAxis(*((int*)tmpl));
		return true;
	case CMD_SETSCROLL:		case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_BAR_TYPE:
		if(xyPlots) for(i = 0; i < numXY; i++)
			if(xyPlots[i]) xyPlots[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_SAVE_BARS:		case CMD_SAVE_BARS_CONT:
		if(Boxes) for(i = 0; i < numPlots; i++)
			if(Boxes[i])Boxes[i]->Command(CMD_SAVE_BARS_CONT, tmpl, o);
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_STACKBAR;
		if(data == tmpl) return true;
		data = (DataObj *)tmpl;
	case CMD_AUTOSCALE:		case CMD_UPDATE:
		if(cmd == CMD_UPDATE){
			Undo.ObjConf(this, UNDO_CONTINUE);
			Bounds.Xmin = Bounds.Ymin = HUGE_VAL;
			Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
			}
		if(cum_data_mode){
			if(CumData) delete CumData;
			CumData = CreaCumData(ssXrange, ssYrange, cum_data_mode, StartVal);
			if(cmd == CMD_SET_DATAOBJ) tmpl = CumData;
			if(cmd == CMD_UPDATE && CumData) {
				if(Polygons) for(i = 0; i < numPG; i++)
					if(Polygons[i]) Polygons[i]->Command(CMD_SET_DATAOBJ, CumData, o);
				if(Lines) for(i = 0; i < numPL; i++)
					if(Lines[i]) Lines[i]->Command(CMD_SET_DATAOBJ, CumData, o);
				if(xyPlots) for(i = 0; i < numXY; i++)
					if(xyPlots[i]) xyPlots[i]->Command(CMD_SET_DATAOBJ, CumData, o);
				if(Boxes) for (i = 0; i < numPlots; i++) 
					if(Boxes[i]) Boxes[i]->Command(CMD_SET_DATAOBJ, CumData, o);
				}
			}
		if(Polygons) for(i = 0; i < numPG; i++)
			if(Polygons[i]) Polygons[i]->Command(cmd, tmpl, o);
		if(Lines) for(i = 0; i < numPL; i++)
			if(Lines[i]) Lines[i]->Command(cmd, tmpl, o);
		if(xyPlots) for(i = 0; i < numXY; i++)
			if(xyPlots[i]) xyPlots[i]->Command(cmd, tmpl, o);
	case CMD_BOX_TYPE:
		if(Boxes) for (i = 0; i < numPlots; i++) 
			if(Boxes[i]) Boxes[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_DELOBJ:
		if(o) o->HideMark();
		if(!tmpl || !parent) return false;
		if(Polygons) for(i = 0; i < numPG; i++) if(Polygons[i] == tmpl) {
			Undo.DeleteGO((GraphObj**)(&Polygons[i]), 0L, o);
			return parent->Command(CMD_REDRAW, 0L, o);
			}
		if(Lines) for(i = 0; i < numPL; i++) if(Lines[i] == tmpl) {
			Undo.DeleteGO((GraphObj**)(&Lines[i]), 0L, o);
			return parent->Command(CMD_REDRAW, 0L, o);
			}
		if(xyPlots) for(i = 0; i < numXY; i++){
			if(xyPlots[i] ==  tmpl) {
				Undo.DeleteGO((GraphObj**)(xyPlots[i]), 0L, o);
				return parent->Command(CMD_REDRAW, 0L, o);
				}
			else if(xyPlots[i] && xyPlots[i]->Command(cmd, tmpl, o)) return true;
			}
		if(Boxes) for(i = 0; i < numPlots; i++){
			if(Boxes[i] ==  tmpl) {
				Undo.DeleteGO((GraphObj**)(&Boxes[i]), 0L, o);
				return parent->Command(CMD_REDRAW, 0L, o);
				}
			else if(Boxes[i] && Boxes[i]->Command(cmd, tmpl, o)) return true;
			}
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Stacked polygons is based on stacked bar
StackPG::StackPG(GraphObj *par, DataObj *d):StackBar(par, d)
{
	Id = GO_STACKPG;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// waterfall plot is based on stacked bar
Waterfall::Waterfall(GraphObj *par, DataObj *d):StackBar(par, d)
{
	Id = GO_WATERFALL;
	dspm.fx = dspm.fy = 0.0;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// simple pie chart
PieChart::PieChart(GraphObj *par, DataObj *d):Plot(par, d)
{
	FileIO(INIT_VARS);
	Id = GO_PIECHART;
	if (!d && parent) parent->Command(CMD_DELOBJ, this, NULL);
}

PieChart::PieChart(int src):Plot(0L, 0L)
{
	int i;
	
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		//now set parent in  children
		if(Segments) 
			for(i = 0; i < nPts; i++) if(Segments[i]) Segments[i]->parent = this;
		}
}

PieChart::~PieChart()
{
	int i;

	if(Segments) {
		for(i = 0; i < nPts; i++) if(Segments[i]) DeleteGO(Segments[i]);
		free(Segments);		Segments = 0L;
		}
	if(ssRefA) free(ssRefA);	if(ssRefR) free(ssRefR);
	ssRefA = ssRefR = 0L;
	Undo.InvalidGO(this);
}

bool 
PieChart::SetSize(int select, double value)
{
	int i;

	switch(select & 0xfff) {
	case SIZE_XPOS:
	case SIZE_YPOS:
	case SIZE_RADIUS1:
	case SIZE_RADIUS2:
		if(Segments) for(i = 0; i < nPts; i++) {
			if(Segments[i]) Segments[i]->SetSize(select, value);
			}
		return true;
	default:
		return false;
		}
	return true;
}

void
PieChart::DoPlot(anyOutput *o)
{
	int i;

	if(Segments) for(i = 0; i < nPts; i++) if(Segments[i]) Segments[i]->DoPlot(o);
}

bool
PieChart::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;
	MouseEvent *mev;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch(mev->Action) {
		case MOUSE_LBUP:
			//select objects invers to plot order
			if(Segments && !CurrGO) for(i = nPts-1; i>=0; i--)
				if(Segments[i]) if(Segments[i]->Command(cmd, tmpl, o))break;
			break;
			}
		break;
	case CMD_SETSCROLL:		case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		return false;
	case CMD_UPDATE:
		DoUpdate();
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_PIECHART;
		data = (DataObj *)tmpl;
	case CMD_SHIFT_OUT:		case CMD_SEG_FILL:		case CMD_SEG_LINE:
	case CMD_SEG_MOVEABLE:	case CMD_LEGEND:
		if(Segments) for(i = 0; i < nPts; i++)
			if(Segments[i]) Segments[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_DELOBJ:
		o->HideMark();
		if(Segments && parent) for(i = 0; i < nPts; i++) {
			if(Segments[i] && tmpl == (void *)Segments[i]) {
				Undo.DeleteGO((GraphObj**)(&Segments[i]), 0L, o);
				parent->Command(CMD_REDRAW, NULL, o);
				return true;
				}
			}
		break;
	case CMD_SAVE_SYMBOLS:
		return SavVarObs((GraphObj **)Segments, nPts, 0L);
		}
	return false;
}

void
PieChart::DoUpdate()
{
	AccRange *rY = 0L, *rR = 0L;
	double sum, fv, dang1, dang2;
	int i, ix, iy, rix, riy;

	if(ssRefA && (rY = new AccRange(ssRefA))) {
		Undo.ObjConf(this, UNDO_CONTINUE);
		if(ssRefR) rR = new AccRange(ssRefR);
		rY->GetFirst(&ix, &iy);				rY->GetNext(&ix, &iy);
		for(i = 0, sum = 0.0; i < nPts; i++){
			if(data->GetValue(iy, ix, &fv)) sum += fv;
			rY->GetNext(&ix, &iy);
			}
		sum /= CtDef.fy;
		dang1 = dang2 = CtDef.fx;
		rY->GetFirst(&ix, &iy);				rY->GetNext(&ix, &iy);
		if(rR) {
			rR->GetFirst(&rix, &riy);		rR->GetNext(&rix, &riy);
			}
		for(i = 0; i < nPts; i++){
			if(data->GetValue(iy, ix, &fv)) {
				dang2 -= (double)fv / sum;
				if(dang2 < 0.0) dang2 += 360.0;
				if(Segments[i]) {
					Segments[i]->SetSize(SIZE_ANGLE1, dang1);
					Segments[i]->SetSize(SIZE_ANGLE2, dang2);
					if(rR && data->GetValue(riy, rix, &fv)){
						fv *= FacRad;
						Segments[i]->SetSize(SIZE_RADIUS2, fv);
						}
					}
				dang1 = dang2;
				}
			rY->GetNext(&ix, &iy);
			if(rR) rR->GetNext(&rix, &riy);
			}
		}
	if(rY) delete rY;		if(rR) delete rR;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// ring chart is based on piechart
RingChart::RingChart(GraphObj *par, DataObj *d):PieChart(par, d)
{
	Id = GO_RINGCHART;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// a GoGroup contains objects NOT referring to data (e.g. drawing objects)
GoGroup::GoGroup(GraphObj *par, DataObj *d):Plot(par, d)
{
	Objects = 0L;
	nObs = 0;
	fPos.fx = fPos.fy = 0.0;
	Id = GO_GROUP;
}

GoGroup::GoGroup(int src):Plot(0L, 0L)
{
	int i;
	
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		//now set parent in  children
		if(Objects) 
			for(i = 0; i < nObs; i++) if(Objects[i]) Objects[i]->parent = this;
		}
}

GoGroup::~GoGroup()
{
	int i;

	if(Objects && nObs) {
		for(i = 0; i < nObs; i++) if(Objects[i]) DeleteGO(Objects[i]);
		free(Objects);
		}
	Undo.InvalidGO(this);
}

double
GoGroup::GetSize(int select)
{
	if(parent) switch(select){
	case SIZE_GRECT_TOP:
	case SIZE_GRECT_BOTTOM:
		return parent->GetSize(select)-fPos.fy;
	case SIZE_GRECT_LEFT:
	case SIZE_GRECT_RIGHT:
		return parent->GetSize(select)-fPos.fx;
	case SIZE_XPOS:
		return fPos.fx;
	case SIZE_YPOS:
		return fPos.fy;
		}
	return 0.0f;
}

void 
GoGroup::DoPlot(anyOutput *o)
{
	int i;
	double dx, dy;
	
	dx = o->un2fix(fPos.fx + (parent ? parent->GetSize(SIZE_GRECT_LEFT) : 0.0));
	dy = o->un2fiy(fPos.fy + (parent ? parent->GetSize(SIZE_GRECT_TOP) : 0.0));
	o->VPorg.fx += dx;				o->VPorg.fy += dy;
	for(i = 0; i < nObs; i++) if(Objects[i]) Objects[i]->DoPlot(o);
	o->VPorg.fx -= dx;				o->VPorg.fy -= dy;
}

bool
GoGroup::Command(int cmd, void *tmpl, anyOutput *o)
{
	GraphObj **tmp_go;
	MouseEvent *mev;
	int i;

	switch(cmd) {
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch(mev->Action) {
		case MOUSE_LBUP:
			//select objects invers to plot order
			if(Objects && !CurrGO) for(i = nObs-1; i>=0; i--)
				if(Objects[i]) if(Objects[i]->Command(cmd, tmpl, o))break;
			break;
			}
		break;
	case CMD_SET_DATAOBJ:
		Id = GO_GROUP;
		if(Objects) for(i = 0; i < nObs; i++)
			if(Objects[i]) Objects[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_DROP_OBJECT:
		if(!Objects || nObs<1) {
			if((Objects = (GraphObj **)calloc(1, sizeof(GraphObj*)))){
				Objects[0] = (GraphObj *)tmpl;
				nObs = 1;
				return true;
				}
			}
		else if((tmp_go = (GraphObj **)realloc(Objects, (nObs+1)*sizeof(GraphObj*)))) {
			Objects = tmp_go;
			Objects[nObs++] = (GraphObj *)tmpl;
			return true;
			}
		break;
	case CMD_SETSCROLL:
	case CMD_REDRAW:
		if(parent) return parent->Command(CMD_REDRAW, tmpl, o);
		return false;
	case CMD_DELOBJ:
		if(Objects && parent) for(i = 0; i < nObs; i++) {
			o->HideMark();
			if(Objects[i] && tmpl == (void *)Objects[i]) {
				Undo.DeleteGO((GraphObj**)(&Objects[i]), 0L, o);
				parent->Command(CMD_REDRAW, NULL, o);
				return true;
				}
			}
		break;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// star chart
StarChart::StarChart(GraphObj *par, DataObj *d):GoGroup(par, d)
{
	Id = GO_STARCHART;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// three dimensional scatterplot
Scatt3D::Scatt3D(GraphObj *par, DataObj *d, DWORD flags):Plot(par, d)
{
	FileIO(INIT_VARS);
	c_flags = flags;
	Id = GO_SCATT3D;
}

Scatt3D::Scatt3D(GraphObj *par, DataObj *d, Brick **cols, long nob):Plot(par, d)
{
	int i;

	FileIO(INIT_VARS);
	c_flags = 0L;		Id = GO_SCATT3D;
	Columns = cols;		nColumns = nob;
	if(Columns) for(i = 0; i < nColumns; i++) if(Columns[i]) Columns[i]->parent=this;
}

Scatt3D::Scatt3D(GraphObj *par, DataObj *d, Sphere **ba, long nob):Plot(par, d)
{
	int i;

	FileIO(INIT_VARS);
	c_flags = 0L;		Id = GO_SCATT3D;
	Balls = ba;			nBalls = nob;
	if(Balls) for(i = 0; i < nBalls; i++) if(Balls[i]) Balls[i]->parent=this;
}

Scatt3D::Scatt3D(int src):Plot(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

Scatt3D::~Scatt3D()
{
	long i;

	if(ssRefX) free(ssRefX);	if(ssRefY) free(ssRefY);
	if(ssRefZ) free(ssRefZ);	ssRefX = ssRefY = ssRefZ = 0L;
	Undo.InvalidGO(this);
	if(Balls) {
		for(i = 0; i < nBalls; i++) if(Balls[i]) DeleteGO(Balls[i]);
		free(Balls);		Balls = 0L;
		}
	if(Columns) {
		for(i = 0; i < nColumns; i++) if(Columns[i]) DeleteGO(Columns[i]);
		free(Columns);		Columns = 0L;
		}
	if(DropLines) {
		for(i = 0; i < nDropLines; i++) if(DropLines[i]) DeleteGO(DropLines[i]);
		free(DropLines);	DropLines = 0L;
		}
	if(Arrows) {
		for(i = 0; i < nArrows; i++) if(Arrows[i]) DeleteGO(Arrows[i]);
		free(Arrows);		Arrows = 0L;
		}
	if(Line) {
		DeleteGO(Line);		Line = 0L;
		}
}

double
Scatt3D::GetSize(int select)
{
	if(parent) return parent->GetSize(select);
	return 0.0;
}

bool
Scatt3D::SetSize(int select, double value)
{
	int i;

	switch(select & 0xfff) {
	case SIZE_SYM_LINE:
	case SIZE_SYMBOL:
		if(Balls) for (i=0; i < nBalls; i++)
			if(Balls[i]) Balls[i]->SetSize(select, value);
		return true;
	case SIZE_BAR_BASE:		case SIZE_BAR_LINE:		case SIZE_BAR:
	case SIZE_BAR_DEPTH:
		if(Columns) for (i=0; i < nColumns; i++)
			if(Columns[i]) Columns[i]->SetSize(select, value);
		return true;
	case SIZE_ARROW_LINE:	case SIZE_ARROW_CAPWIDTH:
	case SIZE_ARROW_CAPLENGTH:
		if(Arrows) for (i=0; i < nArrows; i++)
			if(Arrows[i]) Arrows[i]->SetSize(select, value);
		return true;
		}
	return false;
}

bool
Scatt3D::SetColor(int select, DWORD col)
{
	int i;

	switch(select) {
	case COL_SYM_LINE:			case COL_SYM_FILL:
		if(Balls) for (i=0; i < nBalls; i++)
			if(Balls[i]) Balls[i]->SetColor(select, col);
		return true;
	case COL_BAR_LINE:			case COL_BAR_FILL:
		if(Columns) for (i=0; i < nColumns; i++)
			if(Columns[i]) Columns[i]->SetColor(select, col);
		return true;
	case COL_ARROW:
		if(Arrows) for (i=0; i < nArrows; i++)
			if(Arrows[i]) Arrows[i]->SetColor(select, col);
		return true;
		}
	return false;
}

void
Scatt3D::DoPlot(anyOutput *o)
{
	long i;
	RECT rc;

	if(!o) return;
	o->GetSize(&rc);
	rDims.left = rc.right;			rDims.right = rc.left;
	rDims.top = rc.bottom;			rDims.bottom = rc.top;
	if(Balls) {
		for(i = 0; i < nBalls; i++){
			if(Balls[i]){
				Balls[i]->DoPlot(o);
				UpdateMinMaxRect(&rDims, Balls[i]->rDims.right, Balls[i]->rDims.top);
				UpdateMinMaxRect(&rDims, Balls[i]->rDims.left, Balls[i]->rDims.bottom);
				}
			}
		}
	if(Columns) {
		for(i = 0; i < nColumns; i++){
			if(Columns[i]){
				Columns[i]->DoPlot(o);
				UpdateMinMaxRect(&rDims, Columns[i]->rDims.right, Columns[i]->rDims.top);
				UpdateMinMaxRect(&rDims, Columns[i]->rDims.left, Columns[i]->rDims.bottom);
				}
			}
		}
	if(DropLines) {
		for(i = 0; i < nDropLines; i++){
			if(DropLines[i]){
				DropLines[i]->DoPlot(o);
				UpdateMinMaxRect(&rDims, DropLines[i]->rDims.right, DropLines[i]->rDims.top);
				UpdateMinMaxRect(&rDims, DropLines[i]->rDims.left, DropLines[i]->rDims.bottom);
				}
			}
		}
	if(Arrows) {
		for(i = 0; i < nArrows; i++){
			if(Arrows[i]){
				Arrows[i]->DoPlot(o);
				UpdateMinMaxRect(&rDims, Arrows[i]->rDims.right, Arrows[i]->rDims.top);
				UpdateMinMaxRect(&rDims, Arrows[i]->rDims.left, Arrows[i]->rDims.bottom);
				}
			}
		}
	if(Line) {
		Line->DoPlot(o);
		UpdateMinMaxRect(&rDims, Line->rDims.right, Line->rDims.top);
		UpdateMinMaxRect(&rDims, Line->rDims.left, Line->rDims.bottom);
		}
	dirty = false;
}

bool
Scatt3D::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;
	MouseEvent *mev;

	switch (cmd) {
	case CMD_HIDE_MARK:
		if(!o || !tmpl) return false;
		if(Line && (void*) Line == tmpl){
			Line->DoMark(o, false);					return true;
			}
		if(Columns) for(i = 0; i < nColumns; i++) {
			if(Columns[i] && (void*)Columns[i] == tmpl) {
				Columns[i]->DoMark(o, false);		return true;
				}
			}
		if(DropLines) for(i = 0; i < nDropLines; i++) {
			if(DropLines[i] && (void*)DropLines[i] == tmpl) {
				DropLines[i]->DoMark(o, false);		return true;
				}
			}
		return false;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch(mev->Action) {
		case MOUSE_LBUP:
			if(Balls && !CurrGO) for(i = 0; i < nBalls; i++)
				if(Balls[i]) if(Balls[i]->Command(cmd, tmpl, o))return true;
			if(Columns && !CurrGO) for(i = 0; i < nColumns; i++)
				if(Columns[i]) if(Columns[i]->Command(cmd, tmpl, o))return true;
			if(DropLines && !CurrGO) for(i = 0; i < nDropLines; i++)
				if(DropLines[i]) if(DropLines[i]->Command(cmd, tmpl, o))return true;
			if(Arrows && !CurrGO) for(i = 0; i < nArrows; i++)
				if(Arrows[i]) if(Arrows[i]->Command(cmd, tmpl, o))return true;
			if(Line && !CurrGO) if(Line->Command(cmd, tmpl, o)) return true;
			break;
			}
		break;
	case CMD_MRK_DIRTY:
		dirty = true;
	case CMD_SET_GO3D:
	case CMD_SETSCROLL:
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_DL_LINE:
	case CMD_DL_TYPE:
		if(DropLines) for(i = 0; i < nDropLines; i++)
			if(DropLines[i]) DropLines[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_ARROW_TYPE:
	case CMD_ARROW_ORG3D:
		if(Arrows) for(i = 0; i < nArrows; i++)
			if(Arrows[i]) Arrows[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_SET_DATAOBJ:
		Id = GO_SCATT3D;
		data = (DataObj *)tmpl;
	case CMD_UPDATE:
		if(cmd == CMD_UPDATE) Undo.ObjConf(this, UNDO_CONTINUE);
		if(Balls) for(i = 0; i < nBalls; i++)
			if(Balls[i]) Balls[i]->Command(cmd, tmpl, o);
		if(Columns) for(i = 0; i < nColumns; i++)
			if(Columns[i]) Columns[i]->Command(cmd, tmpl, o);
		if(DropLines) for(i = 0; i < nDropLines; i++)
			if(DropLines[i]) DropLines[i]->Command(cmd, tmpl, o);
		if(Arrows) for(i = 0; i < nArrows; i++)
			if(Arrows[i]) Arrows[i]->Command(cmd, tmpl, o);
		if(Line) Line->Command(cmd, tmpl, o);
		return true;
	case CMD_AUTOSCALE:
		if(dirty) {
			Bounds.Xmin = Bounds.Ymin = HUGE_VAL;	Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
			xBounds.fx = yBounds.fx = zBounds.fx = HUGE_VAL;
			xBounds.fy = yBounds.fy = zBounds.fy = -HUGE_VAL;
			if(Balls) for(i = 0; i < nBalls; i++)
				if(Balls[i]) Balls[i]->Command(cmd, tmpl, o);
			if(Columns) for(i = 0; i < nColumns; i++)
				if(Columns[i]) Columns[i]->Command(cmd, tmpl, o);
			if(DropLines) for(i = 0; i < nDropLines; i++)
				if(DropLines[i]) DropLines[i]->Command(cmd, tmpl, o);
			if(Arrows) for(i = 0; i < nArrows; i++)
				if(Arrows[i]) Arrows[i]->Command(cmd, tmpl, o);
			if(Line) Line->Command(cmd, tmpl, o);
			}
		if(parent && parent->Id > GO_PLOT && parent->Id < GO_GRAPH &&
			xBounds.fx <= xBounds.fy && yBounds.fx <= yBounds.fy && zBounds.fx <= zBounds.fy){
			((Plot*)parent)->CheckBounds3D(xBounds.fx, yBounds.fx, zBounds.fx);
			((Plot*)parent)->CheckBounds3D(xBounds.fy, yBounds.fy, zBounds.fy);
			}
		return true;
	case CMD_DELOBJ:
		if(o) o->HideMark();
		if(!tmpl || !parent) return false;
		if(Balls) for(i = 0; i < nBalls; i++) if(Balls[i] == tmpl) {
			Undo.DeleteGO((GraphObj**)(&Balls[i]), 0L, o);
			return parent->Command(CMD_REDRAW, 0L, o);
			}
		if(Columns) for(i = 0; i < nColumns; i++) if(Columns[i] == tmpl) {
			Undo.DeleteGO((GraphObj**)(&Columns[i]), 0L, o);
			return parent->Command(CMD_REDRAW, 0L, o);
			}
		if(DropLines) for(i = 0; i < nDropLines; i++) if(DropLines[i] == tmpl) {
			Undo.DeleteGO((GraphObj**)(&DropLines[i]), 0L, o);
			return parent->Command(CMD_REDRAW, 0L, o);
			}
		if(Arrows) for(i = 0; i < nArrows; i++) if(Arrows[i] == tmpl) {
			Undo.DeleteGO((GraphObj**)(&Arrows[i]), 0L, o);
			return parent->Command(CMD_REDRAW, 0L, o);
			}
		if(Line && Line == tmpl) {
			Undo.DeleteGO((GraphObj**)(&Line), 0L, o);
			return parent->Command(CMD_REDRAW, 0L, o);
			}
	case CMD_SYM_FILL:
		if(Balls) for(i= 0; i < nBalls; i++) if(Balls[i])Balls[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_BAR_FILL:
		if(Columns) for(i= 0; i < nColumns; i++) if(Columns[i])Columns[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_SAVE_SYMBOLS:
		return SavVarObs((GraphObj **)Balls, nBalls, 0L);
	case CMD_SAVE_BARS:
		return SavVarObs((GraphObj **)Columns, nColumns, 0L);
	case CMD_SAVE_ARROWS:
		return SavVarObs((GraphObj **)Arrows, nArrows, 0L);
	case CMD_SAVE_DROPLINES:
		return SavVarObs((GraphObj **)DropLines, nDropLines, 0L);
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// three dimensional ribbon based on list of Plane3D objects
Ribbon::Ribbon(GraphObj *par, DataObj *d, double z, double width, char *xr, char *yr)
	:Plot(par, d)
{
	FileIO(INIT_VARS);		Id = GO_RIBBON;		type = 1;
	if(xr) ssRefX = strdup(xr);		if(yr) ssRefY = strdup(yr);
	z_value = z;	z_width = width;
}

Ribbon::Ribbon(int src):Plot(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

Ribbon::~Ribbon()
{
	int i;

	if(ssRefX) free(ssRefX);	if(ssRefY) free(ssRefY);
	if(ssRefZ) free(ssRefZ);	ssRefX = ssRefY = ssRefZ = 0L;
	Undo.InvalidGO(this);
	if(planes) {
		for(i = 0; i < nPlanes; i++) if(planes[i]) DeleteGO(planes[i]);
		free(planes);		planes = 0L;
		}
	if(values) free(values);	values = 0L;	nVal = 0;
}

double
Ribbon::GetSize(int select)
{
	switch(select) {
	case SIZE_CELLWIDTH:	return relwidth;
	case SIZE_ZPOS:			return z_value;
		}
	return 0.0;
}

bool
Ribbon::SetSize(int select, double value)
{
	int i;

	switch(select) {
	case SIZE_SYM_LINE:
		if(planes) for (i=0; i < nPlanes; i++)
			if(planes[i]) planes[i]->SetSize(select, value);
		return true;
	case SIZE_CELLWIDTH:
		if(value != relwidth) {
			//assume planes saved already by CMD_SAVE_SYMBOLS
			Undo.ValFloat(this, &relwidth, UNDO_CONTINUE);
			relwidth = value;
			if(planes) UpdateObs(false);
			}
		return true;
	case SIZE_ZPOS:
		if(value != z_value) {
			//assume planes saved already by CMD_SAVE_SYMBOLS
			Undo.ValFloat(this, &z_value, UNDO_CONTINUE);
			z_value = value;
			if(planes) UpdateObs(false);
			}
		return true;
		}
	return false;
}

bool
Ribbon::SetColor(int select, DWORD col)
{
	int i;

	switch(select) {
	case COL_POLYLINE:
		Line.color = col;
	case COL_POLYGON:
		if(select == COL_POLYGON) Fill.color = col;
		if(planes) for (i=0; i < nPlanes; i++)
			if(planes[i]) planes[i]->SetColor(select, col);
		return true;
		}
	return false;
}

void
Ribbon::DoPlot(anyOutput *o)
{
	int i;

	if(!planes) CreateObs();
	if(planes) for(i = 0; i < nPlanes; i++) if(planes[i]) planes[i]->DoPlot(o);
	dirty = false;
}

bool
Ribbon::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;
	MouseEvent *mev;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch(mev->Action) {
		case MOUSE_LBUP:
			if(planes && !CurrGO) for(i = 0; i < nPlanes; i++)
				if(planes[i]) if(planes[i]->Command(cmd, tmpl, o)) return true;
			break;
			}
		break;
	case CMD_MRK_DIRTY:
		dirty = true;
	case CMD_SET_GO3D:
	case CMD_SETSCROLL:
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_SET_DATAOBJ:
		Id = GO_RIBBON;
		data = (DataObj *)tmpl;
		if(planes) for(i = 0; i < nPlanes; i++)
			if(planes[i]) planes[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_UPDATE:
		SavVarObs((GraphObj **)planes, nPlanes, UNDO_CONTINUE);
		Undo.DataMem(this, (void**)&values, nVal * sizeof(fPOINT3D), &nVal, UNDO_CONTINUE);
		UpdateObs(dirty = true);
		if(parent) parent->Command(CMD_MRK_DIRTY, tmpl, o);
		return true;
	case CMD_AUTOSCALE:
		if(!planes) CreateObs();
		if(dirty) {
			Bounds.Xmin = Bounds.Ymin = HUGE_VAL;	Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
			xBounds.fx = yBounds.fx = zBounds.fx = HUGE_VAL;
			xBounds.fy = yBounds.fy = zBounds.fy = -HUGE_VAL;
			if(planes) for(i = 0; i < nPlanes; i++)
				if(planes[i]) planes[i]->Command(cmd, tmpl, o);
			}
		if(parent && parent->Id > GO_PLOT && parent->Id < GO_GRAPH &&
			xBounds.fx <= xBounds.fy && yBounds.fx <= yBounds.fy && zBounds.fx <= zBounds.fy){
			((Plot*)parent)->CheckBounds3D(xBounds.fx, yBounds.fx, zBounds.fx);
			((Plot*)parent)->CheckBounds3D(xBounds.fy, yBounds.fy, zBounds.fy);
			}
		return true;
	case CMD_DELOBJ:
		if(!tmpl || !parent) return false;
		if(planes) for(i = 0; i < nPlanes; i++) if(planes[i] == tmpl) {
			Undo.DeleteGO((GraphObj**)(&planes[i]), 0L, o);
			return parent->Command(CMD_REDRAW, 0L, o);
			}
	case CMD_SYM_FILL:
		if(planes) for(i = 0; i < nPlanes; i++) if(planes[i]) planes[i]->Command(cmd, tmpl, o); 
		return true;
	case CMD_SAVE_SYMBOLS:
		return SavVarObs((GraphObj **)planes, nPlanes, 0L);
		}
	return false;
}

void
Ribbon::CreateObs()
{
	int i, rx, cx, ry, cy;
	double fx, fy, tmp;
	fPOINT3D pg[5];
	AccRange *rX, *rY;

	if(planes) return;
	rX = rY = 0L;
	switch(type) {
	case 1:
		if(!ssRefX || !ssRefY || !data) return;
		if(z_width == 0.0) z_width = 1.0;	if(relwidth == 0.0) relwidth = 0.6;
		if((rX = new AccRange(ssRefX)) && (rY = new AccRange(ssRefY))) {
			tmp = relwidth*z_width/2.0;
			if(!(values = (fPOINT3D*)calloc(i = rX->CountItems(), sizeof(fPOINT3D)))){
				delete rX;	delete rY;	return;
				}
			if(!(planes = (Plane3D**)calloc(i-1, sizeof(Plane3D*)))){
				free(values);	values = 0L;	delete rX;	delete rY;	return;
				}
			for(i = 0, rX->GetFirst(&cx, &rx), rY->GetFirst(&cy, &ry);
				rX->GetNext(&cx, &rx) && rY->GetNext(&cy, &ry); i++) {
				if(data->GetValue(rx, cx, &fx) && data->GetValue(ry, cy, &fy)) {
					values[i].fx = fx;	values[i].fy = fy;	values[i].fz = z_value;
					pg[3].fx = pg[2].fx = fx;	pg[3].fy = pg[2].fy = fy;
					pg[2].fz = z_value - tmp;	pg[3].fz = z_value +tmp;
					if(i) {
						pg[4].fx = pg[0].fx;	pg[4].fy = pg[0].fy; pg[4].fz = pg[0].fz;
						planes[i-1] = new Plane3D(this, data, pg, 5);
						if(planes[i-1]) planes[i-1]->Command(CMD_PG_FILL, &Fill, 0L);
						}
					pg[0].fx = pg[3].fx;	pg[0].fy = pg[3].fy; pg[0].fz = pg[3].fz;
					pg[1].fx = pg[2].fx;	pg[1].fy = pg[2].fy; pg[1].fz = pg[2].fz;
					}
				}
			nPlanes = i-1;		nVal = i;
			}
		break;
		}
	if(rX) delete rX;	if(rY) delete rY;
}

void
Ribbon::UpdateObs(bool bNewData)
{
	int i, j, k, rx, cx, ry, cy;
	double fx, fy, tmp, da1, da2;
	AccRange *rX, *rY;
	int sel_id[] = {SIZE_XPOS, SIZE_YPOS, SIZE_ZPOS};

	if(!planes || !values) return;
	rX = rY = 0L;
	switch(type) {
	case 1:
		if(!ssRefX || !ssRefY || !data) return;
		if(z_width == 0.0) z_width = 1.0;	if(relwidth == 0.0) relwidth = 0.6;
		if((rX = new AccRange(ssRefX)) && (rY = new AccRange(ssRefY))) {
			tmp = relwidth*z_width/2.0;
			for(i = 0, rX->GetFirst(&cx, &rx), rY->GetFirst(&cy, &ry);
				rX->GetNext(&cx, &rx) && rY->GetNext(&cy, &ry) && i < nVal; i++) {
				if(data->GetValue(rx, cx, &fx) && data->GetValue(ry, cy, &fy)) {
					if(bNewData) {
						values[i].fx = fx;	values[i].fy = fy;	values[i].fz = z_value;
						}
					else {
						fx = values[i].fx;	fy = values[i].fy;	values[i].fz = z_value;
						}
					if(i && planes[i-1]) {
						for(j = 0; j < 3; j++){
							for(k = 0; k <5; k++) {
								switch (j) {
								case 0:	
									da1 = values[i-1].fx;		da2 = values[i].fx;
									break;
								case 1:
									da1 = values[i-1].fy;		da2 = values[i].fy;
									break;
								case 2:
									if(k != 1 && k != 2) da1 = da2 = (values[i].fz + tmp);
									else da1 = da2 = (values[i].fz - tmp);
									break;
									}
								planes[i-1]->SetSize(sel_id[j]+k, (k != 2 && k != 3) ? da1 : da2);
								}
							}
						}
					}
				}
			}
		break;
		}
	if(rX) delete rX;	if(rY) delete rY;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// define minima and maxima rectangle to be used by graph
Limits::Limits(int src):Plot(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

Limits::~Limits()
{
}

double
Limits::GetSize(int select)
{
	return 0.0;
}

bool
Limits::Command(int cmd, void *tmpl, anyOutput *o)
{
	switch (cmd) {
	case CMD_SET_DATAOBJ:
		Id = GO_LIMITS;
		data = (DataObj *)tmpl;	
		return true;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Calculate and display a user defined function
Function::Function(GraphObj *par, DataObj *d):Plot(par, d)
{
	FileIO(INIT_VARS);
	if(parent && parent->Id == GO_POLARPLOT) {
		x1 = 0.0;			x2 = 360.0;			xstep = 0.5;
		cmdxy = strdup("sin(pi*x/30)+1.1");
		}
	else {
		x1 = 0.0;			x2 = 100.0;			xstep = 0.5;
		cmdxy = strdup("sin(x)/x");
		}
	Id = GO_FUNCTION;
}

Function::Function(int src):Plot(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}

Function::~Function()
{
	if(cmdxy) free(cmdxy);		cmdxy = 0L;
	if(param) free(param);		param = 0L;
	if(dl) DeleteGO(dl);		dl = 0L;
}

bool
Function::SetSize(int select, double value)
{
	switch(select & 0xfff){
	case SIZE_MIN_X:	x1 = value;		return true;
	case SIZE_MAX_X:	x2 = value;		return true;
	case SIZE_XSTEP:	xstep=value;	return true;
		}
	return false;
}

void
Function::DoPlot(anyOutput *o)
{
	if(!dl && cmdxy && cmdxy[0]) Update(o, 0);
	dirty = false;
	if(dl && o) {
		dl->Command(CMD_SET_LINE, &Line, o);
		dl->DoPlot(o);
		}
}

bool
Function::Command(int cmd, void *tmpl, anyOutput *o)
{
	switch (cmd) {
	case CMD_LEGEND:	case CMD_MOUSE_EVENT:
		if(dl) return dl->Command(cmd, tmpl, o);
		break;
	case CMD_REG_GO:
		((notary*)tmpl)->AddRegGO(this);
		return true;
	case CMD_HIDE_MARK:
		if(dl && tmpl == dl) {
			dl->DoMark(o, false);
			return true;
			}
		return false;
	case CMD_DELOBJ:
		if(parent && tmpl && tmpl == dl) return parent->Command(CMD_DELOBJ, this, o);
		break;
	case CMD_MRK_DIRTY:
		if(parent) parent->Command(cmd, tmpl, o);
		return dirty = true;
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_SET_LINE:
		if(tmpl) memcpy(&Line, tmpl, sizeof(LineDEF));
		break;
	case CMD_SET_DATAOBJ:
		if(dl) dl->Command(cmd, tmpl, o);
		Id = GO_FUNCTION;
		data = (DataObj *)tmpl;
		return true;
	case CMD_SETPARAM:
		if(tmpl) {
			if(param) free(param);
			param = strdup((char*)tmpl);
			}
		dirty = true;
		return true;
	case CMD_SETFUNC:
		if(tmpl) {
			if(cmdxy) free(cmdxy);
			cmdxy = strdup((char*)tmpl);
			}
		dirty = true;
		return true;
	case CMD_UPDATE:
		return Update(o, UNDO_CONTINUE);
	case CMD_AUTOSCALE:
		if(!dl) return Update(o, 0L);
		if(dirty) {
			Bounds.Xmin = Bounds.Ymin = HUGE_VAL;
			Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
			xBounds.fx = yBounds.fx = zBounds.fx = HUGE_VAL;
			xBounds.fy = yBounds.fy = zBounds.fy = -HUGE_VAL;
			if(dl) dl->Command(cmd, tmpl, o);
			dirty = false;
			}
		return true;
		}
	return false;
}

bool
Function::Update(anyOutput *o, DWORD flags)
{
	lfPOINT *xydata;
	long ndata;

	if(!parent || !cmdxy) return false;
	if(parent->Id != GO_FITFUNC && dl) Undo.DeleteGO((GraphObj**)&dl, flags, o);
	do_xyfunc(data, x1, x2, xstep, cmdxy, &xydata, &ndata, param);
	if(xydata && ndata >1) {
		if(!dl) dl = new DataLine(this, data, xydata, ndata);
		else dl->LineData(xydata, ndata);
		dirty = true;
		Command(CMD_AUTOSCALE, 0L, 0L);
		return true;
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Calculate and display a user defined function
FitFunc::FitFunc(GraphObj *par, DataObj *d):Plot(par, d)
{
	int width, height;

	FileIO(INIT_VARS);
	x1 = 0.0;			x2 = 100.0;			xstep = 0.5;	dl = 0L;
	cmdxy = strdup("a+b*x^c");
	parxy = strdup("a=1; b=1; c=0.1;");
	data->GetSize(&width, &height);
	if(!ssXref) {
		sprintf(TmpTxt, "a1:a%d", height);
		ssXref = strdup(TmpTxt);
		}
	if(!ssYref) {
		sprintf(TmpTxt, "b1:b%d", height);
		ssYref = strdup(TmpTxt);
		}
	Id = GO_FITFUNC;
}


FitFunc::FitFunc(int src):Plot(0L, 0L)
{
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		}
}


FitFunc::~FitFunc()
{
	int i;

	if(Symbols) {
		for(i = 0; i< nPoints; i++) if(Symbols[i]) DeleteGO(Symbols[i]);
		free(Symbols);
		}
	if(cmdxy) free(cmdxy);		cmdxy = 0L;
	if(parxy) free(parxy);		parxy = 0L;
	if(ssXref) free(ssXref);	ssXref = 0L;
	if(ssYref) free(ssYref);	ssYref = 0L;
	if(dl) DeleteGO(dl);		dl = 0L;
	Undo.InvalidGO(this);
}

bool
FitFunc::SetSize(int select, double value)
{
	int i;

	switch(select & 0xfff){
	case SIZE_SYMBOL:
	case SIZE_SYM_LINE:
		if(Symbols)	for(i = 0; i < nPoints; i++) 
			if(Symbols[i]) Symbols[i]->SetSize(select, value);
		return true;
		}
	return false;
}

bool
FitFunc::SetColor(int select, DWORD col)
{
	int i;

	switch(select) {
	case COL_SYM_LINE:
	case COL_SYM_FILL:
		if(Symbols) for(i = 0; i < nPoints; i++)
			if(Symbols[i]) Symbols[i]->SetColor(select, col);
		return true;
		}
	return false;
}

void
FitFunc::DoPlot(anyOutput *o)
{
	int i;

	if(!data || x1 >= x2) return;
	dirty = false;
	if(xstep <= 0.0) xstep = (x2-x1)/100.0;
	if(!dl) dl = new Function(this, data);
	if(dl && o){
		dl->SetSize(SIZE_MIN_X, x1);			dl->SetSize(SIZE_MAX_X, x2);
		dl->SetSize(SIZE_XSTEP, xstep);			dl->Command(CMD_SETFUNC, cmdxy, 0L);
		dl->Command(CMD_SETPARAM, parxy, 0L);	dl->Command(CMD_SET_LINE, &Line, 0L);
		dl->Update(o, 0L);						dl->DoPlot(o);
		}
	if(Symbols) for(i = 0; i < nPoints; i++) if(Symbols[i]) Symbols[i]->DoPlot(o);
}

bool 
FitFunc::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;
	MouseEvent *mev;
	LineDEF *ld;

	switch(cmd) {
	case CMD_REG_GO:
		if(Symbols) for(i = 0; i < nPoints; i++)
				if(Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
		((notary*)tmpl)->AddRegGO(this);
		return true;
	case CMD_LEGEND:
		if(tmpl && ((GraphObj*)tmpl)->Id == GO_LEGEND && dl && dl->Id == GO_FUNCTION) {
			ld = dl->GetLine();
			if(Symbols) {
				for (i = 0; i < nPoints && i < 100; i++)
					if(Symbols[i]) ((Legend*)tmpl)->HasSym(ld, Symbols[i]);
				}
			else ((Legend*)tmpl)->HasFill(ld, 0L);
			return true;
			}
		return false;
	case CMD_HIDE_MARK:
		if(dl) return dl->Command(cmd, tmpl, o);
		return false;
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch(mev->Action) {
		case MOUSE_LBUP:
			//select objects invers to plot order
			if(Symbols && !CurrGO) for(i = nPoints-1; i >=0; i--)
				if(Symbols[i] && Symbols[i]->Command(cmd, tmpl, o))return true;
			break;
			}
		if(dl) return dl->Command(cmd, tmpl, o);
		return false;
	case CMD_AUTOSCALE:
		if(dirty) {
			if(!dl) if(!(dl = new Function(this, data))) return false;
			dl->SetSize(SIZE_MIN_X, x1);			dl->SetSize(SIZE_MAX_X, x2);
			dl->SetSize(SIZE_XSTEP, xstep);			dl->Command(CMD_SETFUNC, cmdxy, 0L);
			dl->Command(CMD_SETPARAM, parxy, 0L);	dl->Update(o, 0L);
			memcpy(&Bounds, &dl->Bounds, sizeof(fRECT));
			if(Symbols) for(i = 0; i < nPoints; i++) if(Symbols[i])Symbols[i]->Command(cmd, tmpl, o);
			dirty = false;
			}
		return true;
	case CMD_UPDATE:
		Undo.ObjConf(this, UNDO_CONTINUE);
		if(Symbols) for(i = 0; i < nPoints; i++)
			if(Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
		do_fitfunc(data, ssXref, ssYref, 0L, &parxy, cmdxy, conv, maxiter);
		dirty = true;
		if(parent) parent->Command(CMD_MRK_DIRTY, 0L, o);
		return true;
	case CMD_DELOBJ:
		if(parent && tmpl && tmpl == dl) return parent->Command(CMD_DELOBJ, this, o);
		else if(dl) return dl->Command(cmd, tmpl, o);
		break;
	case CMD_MRK_DIRTY:
		dirty = true;
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_SET_DATAOBJ:
		if(dl) dl->Command(cmd, tmpl, o);
		if(Symbols) for(i = 0; i < nPoints; i++)
			if(Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
		Id = GO_FITFUNC;
		data = (DataObj *)tmpl;
		return true;
	case CMD_SYMTEXT:		case CMD_SYMTEXT_UNDO:	case CMD_SYM_RANGETEXT:
	case CMD_SYMTEXTDEF:	case CMD_SYM_TYPE:
		if(Symbols) for(i = 0; i < nPoints; i++)
			if(Symbols[i]) Symbols[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_SAVE_SYMBOLS:
		return SavVarObs((GraphObj **)Symbols, nPoints, 0L);
		}
	return false;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// three dimensional graph
Plot3D::Plot3D(GraphObj *par, DataObj *d, DWORD flags):Plot(par, d)
{
	RotDef = (double*)malloc(6 *sizeof(double));
	FileIO(INIT_VARS);
	Id = GO_PLOT3D;
	crea_flags = flags;
	xBounds.fx = yBounds.fx = zBounds.fx = Bounds.Xmin = Bounds.Ymin = HUGE_VAL;
	xBounds.fy = yBounds.fy = zBounds.fy = Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
}

Plot3D::Plot3D(int src):Plot(0L, 0L)
{
	int i;

	RotDef = (double*)malloc(6 *sizeof(double));
	FileIO(INIT_VARS);
	if(src == FILE_READ) {
		FileIO(FILE_READ);
		//now set parent in all children
		if(Axes) for(i = 0; i < nAxes; i++)
			if(Axes[i]) Axes[i]->parent = this;
		if(plots) for(i = 0; i < nPlots; i++)
			if(plots[i]) plots[i]->parent = this;
		}
}

Plot3D::~Plot3D()
{
	int i;

	if(plots) {
		for(i = 0; i < nPlots; i++) if(plots[i]) DeleteGO(plots[i]);
		free(plots);
		}
	if(Axes) {
		for(i = 0; i < nAxes; i++) if(Axes[i]) DeleteGO(Axes[i]);
		free(Axes);
		}
	plots = 0L;		nPlots = nAxes = 0;
	if(drag) DeleteGO(drag);	drag = 0L;
	if(dispObs) free(dispObs);	dispObs = 0L;
	free(RotDef);
	Undo.InvalidGO(this);
}

double
Plot3D::GetSize(int select)
{
	AxisDEF *ax;

	switch(select){
	//The Bounds values must be returned by every plot:
	//   they are necessary for scaling !
	case SIZE_BOUNDS_XMIN:
		if(Axes && nAxes >2 && Axes[0] && (ax = Axes[0]->GetAxis()))
			return (ax->flags & AXIS_INVERT) ? ax->max : ax->min;
		return 0.0;
	case SIZE_BOUNDS_XMAX:
		if(Axes && nAxes >2 && Axes[0] && (ax = Axes[0]->GetAxis()))
			return (ax->flags & AXIS_INVERT) ? ax->min : ax->max;
		return 0.0;
	case SIZE_BOUNDS_YMIN:
		if(Axes && nAxes >2 && Axes[1] && (ax = Axes[1]->GetAxis())) 
			return (ax->flags & AXIS_INVERT) ? ax->max : ax->min;
		return 0.0;
	case SIZE_BOUNDS_YMAX:
		if(Axes && nAxes >2 && Axes[1] && (ax = Axes[1]->GetAxis())) 
			return (ax->flags & AXIS_INVERT) ? ax->min : ax->max;
		return 0.0;
	case SIZE_BOUNDS_ZMIN:
		if(Axes && nAxes >2 && Axes[2] && (ax = Axes[2]->GetAxis())) 
			return (ax->flags & AXIS_INVERT) ? ax->max : ax->min;
		return 0.0;
	case SIZE_BOUNDS_ZMAX:
		if(Axes && nAxes >2 && Axes[2] && (ax = Axes[2]->GetAxis())) 
			return (ax->flags & AXIS_INVERT) ? ax->min : ax->max;
		return 0.0;
	case SIZE_XPOS:		case SIZE_XPOS+4:		return cu1.fx;
	case SIZE_XPOS+1:	case SIZE_XPOS+5:		return cu2.fx;
	case SIZE_XPOS+2:	case SIZE_XPOS+6:		return cu2.fx;
	case SIZE_XPOS+3:	case SIZE_XPOS+7:		return cu1.fx;
	case SIZE_YPOS:		case SIZE_YPOS+1:		case SIZE_YPOS+2:
	case SIZE_YPOS+3:			return cu1.fy;
	case SIZE_YPOS+4:	case SIZE_YPOS+5:		case SIZE_YPOS+6:
	case SIZE_YPOS+7:			return cu2.fy;
	case SIZE_ZPOS:		case SIZE_ZPOS+1:		case SIZE_ZPOS+4:
	case SIZE_ZPOS+5:			return cu1.fz;
	case SIZE_ZPOS+2:	case SIZE_ZPOS+3:		case SIZE_ZPOS+6:
	case SIZE_ZPOS+7:			return cu2.fz;
	case SIZE_XCENTER:	return rotC.fx;
	case SIZE_YCENTER:	return rotC.fy;
	case SIZE_ZCENTER:	return rotC.fz;
	default:
		if(parent) return parent->GetSize(select);
		else return defs.GetSize(select);
		}
}

bool
Plot3D::SetColor(int select, DWORD col)
{
	int i;

	switch(select & 0xfff) {
	case COL_AXIS:
		if(Axes) for(i = 0; i< nAxes; i++) if(Axes[i]) Axes[i]->SetColor(select, col);
		return true;
		}
	return false;
}

void
Plot3D::DoPlot(anyOutput *o)
{
	long i, j, fo;
	double dx, dy;

	nObs = 0;
	if(!parent || !o) return;
	o->MouseCursor(MC_WAIT, true);
	if(dirty) DoAutoscale();
	o->LightSource(8.0, -16.0);
	dx = parent->GetSize(SIZE_GRECT_LEFT);	dy = parent->GetSize(SIZE_GRECT_TOP);
	cu1.fx = cub1.fx + dx;	cu1.fy = cub1.fy + dy;	cu1.fz = cub1.fz;
	cu2.fx = cub2.fx + dx;	cu2.fy = cub2.fy + dy;	cu2.fz = cub2.fz;
	rc.fx = rotC.fx + dx;	rc.fy = rotC.fy + dy;	rc.fz = rotC.fz;
	if(Axes && nAxes >2) {
		o->SetSpace(&cu1, &cu2, defs.cUnits, RotDef, &rc, Axes[0]->GetAxis(),
			Axes[1]->GetAxis(), Axes[2]->GetAxis());
		for(i = 0; i< nAxes; i++) if(Axes[i]) Axes[i]->DoPlot(o);
		if(plots) for(i = 0; i < nPlots; i++) if(plots[i]){
			if(plots[i]->Id >= GO_PLOT && plots[i]->Id < GO_GRAPH) {
				if(((Plot*)plots[i])->hidden == 0) plots[i]->DoPlot(o);
				}
			else plots[i]->DoPlot(o);
			if(i) {
				UpdateMinMaxRect(&rDims, plots[i]->rDims.right, plots[i]->rDims.top);
				UpdateMinMaxRect(&rDims, plots[i]->rDims.left, plots[i]->rDims.bottom);
				}
			else memcpy(&rDims, &plots[i]->rDims, sizeof(RECT));
			}
		for(i = 0; i< nAxes; i++) if(Axes[i]){
			UpdateMinMaxRect(&rDims, Axes[i]->rDims.right, Axes[i]->rDims.top);
			UpdateMinMaxRect(&rDims, Axes[i]->rDims.left, Axes[i]->rDims.bottom);
			}
		}
	for(i = j = 1; i < nObs; i++) if(dispObs[i] && dispObs[i]->go) 
		dispObs[j++] = dispObs[i];
	nObs = j;
	SortObj();
	if(nObs && dispObs){
		for (i = fo = 1; i <= nObs; i++){
			while(dispObs[fo]->Zmax < dispObs[i]->Zmin && fo < (nObs-1)) fo++;
			for(j = fo; j <= nObs; j++) {
				if(dispObs[j]->go->Id != GO_LINESEG && i != j) switch(dispObs[i]->go->Id) {
				case GO_LINESEG:
				case GO_SPHERE:
				case GO_PLANE:
					dispObs[i]->go->Command(CMD_CLIP, dispObs[j]->go, o);
					break;
					}
				//the following line, if included, reduces time but clipping 
				//   is not complete because of CMD_DRAW_LATER
//				if(dispObs[j]->Zmin > dispObs[i]->Zmax) break;
				}
			}
		for (i = 1; i <= nObs; i++)	dispObs[i]->go->Command(CMD_REDRAW, 0L, o);
		}
	dirty = false;
	o->MouseCursor(MC_ARROW, true);
}

void
Plot3D::DoMark(anyOutput *o, bool mark)
{
	RECT upd;

	if(!drag) drag = new Drag3D(this);
	if(mark && drag) drag->DoPlot(o);
	else {
		memcpy(&upd, &rDims, sizeof(RECT));
		IncrementMinMaxRect(&upd, 6);
		o->UpdateRect(&upd, false);
		}
}

bool
Plot3D::Command(int cmd, void *tmpl, anyOutput *o)
{
	int i;
	MouseEvent *mev;
	GraphObj **tmpPlots;

	switch (cmd) {
	case CMD_MOUSE_EVENT:
		mev = (MouseEvent *) tmpl;
		switch(mev->Action) {
		case MOUSE_LBUP:
			if(CurrGO) return false;
			if(Axes) for (i = 0; i < nAxes; i++)
				if(Axes[i] && Axes[i]->Command(cmd, tmpl, o)) return true;
			if(plots) for(i = nPlots-1; i >= 0; i--)
				if(plots[i] && plots[i]->Command(cmd, tmpl, o)) return true;
			if(IsInRect(&rDims, mev->x, mev->y)) {
				o->ShowMark(CurrGO = this, MRK_GODRAW);
				return true;
				}
			break;
			}
		break;
	case CMD_HIDE_MARK:
		if(!tmpl) return false;
		//do all axes
		if(Axes)for(i = nAxes-1; i>=0; i--) {
			if(tmpl == (void*)Axes[i]){
				Axes[i]->DoMark(o, false);
				return true;
				}
			else if(Axes[i]->Id == GO_AXIS) {
				if(Axes[i]->Command(cmd, tmpl, o)) return true;
				}
			}
		//do all plots
		if(plots)for(i = nPlots-1; i>=0; i--) {
			if(tmpl == (void*)plots[i]){
				plots[i]->DoMark(o, false);
				return true;
				}
			else if(plots[i]->Id == GO_MLABEL || (plots[i]->Id >= GO_PLOT && plots[i]->Id < GO_GRAPH)) {
				if(plots[i]->Command(cmd, tmpl, o)) return true;
				}
			}
		return false;
	case CMD_OBJTREE:
		if(!tmpl || !plots) return false;
		for(i = 0; i < nPlots; i++) if(plots[i]) 
			((ObjTree*)tmpl)->Command(CMD_REG_GO, plots[i], 0L);
		return true;
	case CMD_REPL_GO:
		if(!(tmpPlots = (GraphObj **)tmpl) || !tmpPlots[0] || !tmpPlots[1]) return false;
		if(plots) for(i = 0; i < nPlots; i++) if(plots[i] && plots[i] == tmpPlots[0]){
			return dirty = ReplaceGO((GraphObj**)&plots[i], tmpPlots);
			}
		return false;
	case CMD_MRK_DIRTY:
		return dirty = true;
	case CMD_ADDAXIS:
		InfoBox("Add axis to 3D graph:\n\nThis feature is not yet implemented!");
		return false;
	case CMD_SET_GO3D:
		return AcceptObj((GraphObj *)tmpl);
	case CMD_SETSCROLL:
	case CMD_REDRAW:
		if(parent) return parent->Command(cmd, tmpl, o);
		break;
	case CMD_SHIFTLEFT:
		return Rotate(-0.017452406, 0.0, 0.0, o, true);
	case CMD_SHIFTRIGHT:
		return Rotate(0.017452406, 0.0, 0.0, o, true);
	case CMD_SHIFTUP:
		return Rotate(0.0, 0.017452406, 0.0, o, true);
	case CMD_SHIFTDOWN:
		return Rotate(0.0, -0.017452406, 0.0, o, true);
	case CMD_CURRIGHT:
		return Rotate(0.087155742, 0.0, 0.0, o, true);
	case CMD_CURRLEFT:
		return Rotate(-0.087155742, 0.0, 0.0, o, true);
	case CMD_CURRUP:
		return Rotate(0.0, 0.087155742, 0.0, o, true);
	case CMD_CURRDOWN:
		return Rotate(0.0, -0.087155742, 0.0, o, true);
	case CMD_ADDCHAR:
		if(tmpl && *((int*)tmpl) == 'r') return Rotate(0.0, 0.0, 0.087155742, o, true);
		if(tmpl && *((int*)tmpl) == 'l') return Rotate(0.0, 0.0, -0.087155742, o, true);
		if(tmpl && *((int*)tmpl) == 'R') return Rotate(0.0, 0.0, 0.017452406, o, true);
		if(tmpl && *((int*)tmpl) == 'L') return Rotate(0.0, 0.0, -0.017452406, o, true);
		return false;
	case CMD_SET_DATAOBJ:
		Id = GO_PLOT3D;
		data = (DataObj *)tmpl;
	case CMD_UPDATE:
		if(plots) for(i = 0; i < nPlots; i++)
			if(plots[i]) plots[i]->Command(cmd, tmpl, o);
		if(Axes) for(i = 0; i < nAxes; i++)
			if(Axes[i]) Axes[i]->Command(cmd, tmpl, o);
		return true;
	case CMD_DELOBJ:
		if(o) o->HideMark();
		if(!tmpl || !parent) return false;
		if(plots) for(i = 0; i < nPlots; i++) if(plots[i] == tmpl) {
			Undo.DeleteGO((GraphObj**)(&plots[i]), 0L, o);
			return parent->Command(CMD_REDRAW, 0L, o);
			}
	case CMD_MOVE:
		if(CurrGO && CurrGO->Id == GO_DRAGHANDLE) {
			CalcRotation(((lfPOINT*)tmpl)[0].fx, ((lfPOINT*)tmpl)[0].fy, o, true);
			if(parent) return parent->Command(CMD_REDRAW, 0L, 0L);
			}
		return true;
	case CMD_AUTOSCALE:
		if(dirty) {
			DoAutoscale();
			dirty = false;
			}
		return true;
	case CMD_DROP_PLOT:
		if(!parent || !tmpl || ((GraphObj*)tmpl)->Id < GO_PLOT) return false;
		if(!nPlots) {
			plots = (GraphObj**)calloc(2, sizeof(GraphObj*));
			if(plots) {
				nPlots = 1;					plots[0] = (Plot *)tmpl;
				plots[0]->parent = this;	CreateAxes();
				return dirty = parent->Command(CMD_REDRAW, 0L, 0L);
				}
			}
		else {
			((Plot *)tmpl)->parent = this;
			tmpPlots = (GraphObj**)memdup(plots, sizeof(GraphObj*) * (nPlots+2), 0);
			Undo.ListGOmoved(plots, tmpPlots, nPlots);
			Undo.SetGO(this, &tmpPlots[nPlots++], (Plot *)tmpl, 0L);
			free(plots);			plots = tmpPlots;
			return dirty = parent->Command(CMD_REDRAW, 0L, 0L);
			}
		return false;
	case CMD_ADDPLOT:
		return AddPlot(0x0);
		}
	return false;
}

void *
Plot3D::ObjThere(int x, int y)
{
	if(drag) return drag->ObjThere(x, y);
	return 0L;
}

void
Plot3D::Track(POINT *p, anyOutput *o)
{
	fPOINT3D v, iv;
	POINT pts[5];
	RECT upd_rc;

	CalcRotation(((lfPOINT*)p)->fx, ((lfPOINT*)p)->fy, o, false);
	memcpy(&upd_rc, &rDims, sizeof(RECT));
	IncrementMinMaxRect(&rDims, 3);
	o->UpdateRect(&upd_rc, false);
	memcpy(&v, &cu2, sizeof(fPOINT3D));
	o->cvec2ivec(&v, &iv);
	pts[0].x = iround(iv.fx);		pts[0].y = iround(iv.fy);
	UpdateMinMaxRect(&rDims, pts[0].x, pts[0].y);
	v.fx = cu1.fx;					o->cvec2ivec(&v, &iv); 
	pts[1].x = iround(iv.fx);		pts[1].y = iround(iv.fy);
	UpdateMinMaxRect(&rDims, pts[1].x, pts[1].y);
	v.fy = cu1.fy;					o->cvec2ivec(&v, &iv); 
	pts[2].x = iround(iv.fx);		pts[2].y = iround(iv.fy);
	UpdateMinMaxRect(&rDims, pts[2].x, pts[2].y);
	v.fz = cu1.fz;					o->cvec2ivec(&v, &iv); 
	pts[3].x = iround(iv.fx);		pts[3].y = iround(iv.fy);
	UpdateMinMaxRect(&rDims, pts[3].x, pts[3].y);
	v.fy = cu2.fy;					o->cvec2ivec(&v, &iv); 
	pts[4].x = iround(iv.fx);		pts[4].y = iround(iv.fy);
	UpdateMinMaxRect(&rDims, pts[4].x, pts[4].y);
	o->ShowLine(pts, 5, 0x000000ff);
	v.fz = cu2.fz;					o->cvec2ivec(&v, &iv);
	pts[0].x = iround(iv.fx);		pts[0].y = iround(iv.fy);
	v.fz = cu1.fz;					o->cvec2ivec(&v, &iv); 
	pts[1].x = iround(iv.fx);		pts[1].y = iround(iv.fy);
	v.fx = cu2.fx;					o->cvec2ivec(&v, &iv);
	pts[2].x = iround(iv.fx);		pts[2].y = iround(iv.fy);
	v.fz = cu2.fz;					o->cvec2ivec(&v, &iv);
	pts[3].x = iround(iv.fx);		pts[3].y = iround(iv.fy);
	v.fy = cu1.fy;					o->cvec2ivec(&v, &iv);
	pts[4].x = iround(iv.fx);		pts[4].y = iround(iv.fy);
	o->ShowLine(pts, 5, 0x000000ff);
	v.fy = cu2.fy;	v.fz = cu1.fz;		o->cvec2ivec(&v, &iv);
	pts[0].x = iround(iv.fx);		pts[0].y = iround(iv.fy);
	v.fy = cu1.fy;					o->cvec2ivec(&v, &iv);
	pts[1].x = iround(iv.fx);		pts[1].y = iround(iv.fy);
	v.fx = cu1.fx;					o->cvec2ivec(&v, &iv);
	pts[2].x = iround(iv.fx);		pts[2].y = iround(iv.fy);
	o->ShowLine(pts, 3, 0x000000ff);
	v.fz = cu2.fz;					o->cvec2ivec(&v, &iv);
	pts[0].x = iround(iv.fx);		pts[0].y = iround(iv.fy);
	v.fx = cu2.fx;					o->cvec2ivec(&v, &iv);
	pts[1].x = iround(iv.fx);		pts[1].y = iround(iv.fy);
	v.fz = cu1.fz;					o->cvec2ivec(&v, &iv);
	pts[2].x = iround(iv.fx);		pts[2].y = iround(iv.fy);
	o->ShowLine(pts, 3, 0x000000ff);
}

void
Plot3D::CreateAxes()
{
	typedef struct {
		double x1, y1, z1, x2, y2, z2;
		DWORD flags;
		int a_type, t_type;
		double lb_x, lb_y, tlb_x, tlb_y;
		int txa;
		}Axis3Ddef;
	AxisDEF tmp_axis;
	double ts = defs.GetSize(SIZE_AXIS_TICKS);
	int i;
	if(Axes || !parent)return;
	TextDEF tlbdef = {parent->GetColor(COL_AXIS), 0x00ffffffL, defs.GetSize(SIZE_TICK_LABELS),
		0.0, 0.0, 0, TXA_HLEFT | TXA_VCENTER, TXM_TRANSPARENT, TXS_NORMAL, FONT_HELVETICA, 0L};
	Axis3Ddef *at = 0L;
	Axis3Ddef at1[] = {
		{cub1.fx, cub1.fy, 0.0, cub2.fx, cub1.fy, 0.0,
			AXIS_3D | AXIS_DEFRECT | AXIS_AUTOTICK | AXIS_AUTOSCALE | AXIS_NEGTICKS, 1, 3,
			0.0, NiceValue((ts+defs.GetSize(SIZE_AXIS_TICKS))*2.0), 0.0,
			NiceValue(ts * 2.0), TXA_HCENTER | TXA_VTOP},
		{cub1.fx, cub1.fy, 0.0, cub1.fx, cub2.fy, 0.0,
			AXIS_3D | AXIS_DEFRECT | AXIS_AUTOTICK | AXIS_AUTOSCALE | AXIS_NEGTICKS, 2, 2,
			-NiceValue((ts+defs.GetSize(SIZE_AXIS_TICKS))*3.0), 0.0,
			-NiceValue(ts * 2.0), 0.0, TXA_HRIGHT | TXA_VCENTER},
		{cub1.fx, cub1.fy, 0.0, cub1.fx, cub1.fy, cub2.fz,
			AXIS_3D | AXIS_DEFRECT | AXIS_AUTOTICK | AXIS_AUTOSCALE | AXIS_NEGTICKS, 3, 2,
			-NiceValue((ts+defs.GetSize(SIZE_AXIS_TICKS))*3.0), 0.0,
			-NiceValue(ts * 2.0), 0.0, TXA_HRIGHT | TXA_VCENTER}};
	Axis3Ddef at2[] = {
		{at1[0].x1, at1[0].y1, at1[2].z2, at1[0].x2, at1[0].y2, at1[2].z2,
			AXIS_3D | AXIS_DEFRECT | AXIS_AUTOTICK | AXIS_AUTOSCALE | AXIS_NEGTICKS, 1, 3,
			0.0, at1[0].lb_y, 0.0,	at1[0].tlb_y, TXA_HCENTER | TXA_VTOP},
		{at1[0].x2, at1[1].y1, 0.0, at1[0].x2, at1[1].y2, 0.0,
			AXIS_3D | AXIS_DEFRECT | AXIS_AUTOTICK | AXIS_AUTOSCALE | AXIS_POSTICKS, 2, 2,
			-at1[1].lb_x, 0.0, -at1[1].tlb_x, 0.0, TXA_HLEFT | TXA_VCENTER},
		{at1[0].x2, at1[0].y1, 0.0, at1[0].x2, at1[0].y1, at1[2].z2,
			AXIS_3D | AXIS_DEFRECT | AXIS_AUTOTICK | AXIS_AUTOSCALE | AXIS_POSTICKS, 3, 2,
			-at1[2].lb_x, 0.0, -at1[2].tlb_x, 0.0, TXA_HLEFT | TXA_VCENTER},
		{at1[0].x1, at1[0].y1, 0.0, at1[0].x2, at1[0].y2, 0.0,
			AXIS_3D, 1, 3, 0.0, at1[0].lb_y, 0.0, at1[0].tlb_y, TXA_HCENTER | TXA_VCENTER},
		{at1[0].x1, at1[1].y2, 0.0, at1[0].x2, at1[1].y2, 0.0,
			AXIS_3D, 1, 3, 0.0, at1[0].lb_y, 0.0, at1[0].tlb_y, TXA_HCENTER | TXA_VCENTER},
		{at1[0].x1, at1[1].y1, 0.0, at1[0].x1, at1[1].y2, 0.0,
			AXIS_3D, 2, 2, at1[1].lb_x, 0.0, at1[1].tlb_x, 0.0, TXA_HCENTER | TXA_VCENTER},
		{at1[0].x1, at1[1].y1, at1[2].z2, at1[0].x1, at1[1].y2, at1[2].z2,
			AXIS_3D, 2, 2, at1[1].lb_x, 0.0, at1[1].tlb_x, 0.0, TXA_HCENTER | TXA_VCENTER},
		{at1[0].x1, at1[0].y1, 0.0, at1[0].x1, at1[0].y1, at1[2].z2,
			AXIS_3D, 3, 2, at1[2].lb_x, 0.0, at1[2].tlb_x, 0.0, TXA_HCENTER | TXA_VCENTER},
		{at1[0].x1, at1[1].y2, 0.0, at1[0].x1, at1[1].y2, at1[2].z2,
			AXIS_3D, 3, 2, at1[2].lb_x, 0.0, at1[2].tlb_x, 0.0}, TXA_HCENTER | TXA_VCENTER};
	Axis3Ddef at3[] = {
		{at1[0].x1, (at1[1].y1+at1[1].y2)/2.0, at1[2].z2/2.0, at1[0].x2, 
			(at1[1].y1+at1[1].y2)/2.0, at1[2].z2/2.0,
			AXIS_3D | AXIS_AUTOTICK | AXIS_AUTOSCALE | AXIS_NEGTICKS, 1, 3,
			0.0, at1[0].lb_y, 0.0,	at1[0].tlb_y, TXA_HCENTER | TXA_VTOP},
		{(at1[0].x1 + at1[0].x2)/2.0, at1[1].y1, at1[2].z2/2.0, 
			(at1[0].x1 + at1[0].x2)/2.0, at1[1].y2, at1[2].z2/2.0,
			AXIS_3D | AXIS_AUTOTICK | AXIS_AUTOSCALE | AXIS_POSTICKS, 2, 2,
			-at1[1].lb_x, 0.0, -at1[1].tlb_x, 0.0, TXA_HLEFT | TXA_VCENTER},
		{(at1[0].x1 + at1[0].x2)/2.0, (at1[1].y1+at1[1].y2)/2.0, 0.0,
			(at1[0].x1 + at1[0].x2)/2.0, (at1[1].y1+at1[1].y2)/2.0, at1[2].z2,
			AXIS_3D | AXIS_AUTOTICK | AXIS_AUTOSCALE | AXIS_POSTICKS, 3, 2,
			-at1[2].lb_x, 0.0, -at1[2].tlb_x, 0.0, TXA_HLEFT | TXA_VCENTER}};

	tmp_axis.min = 0.0;			tmp_axis.max = 100.0;
	tmp_axis.Start = 0.0;		tmp_axis.Step = 20.0;
	tmp_axis.Center.fx = tmp_axis.Center.fy = 0.0;
	tmp_axis.Radius = 0.0;		tmp_axis.nBreaks = 0;
	tmp_axis.breaks = 0L;		tmp_axis.owner = 0L;
	switch(AxisTempl3D){
	case 0:		at = at1;		nAxes = 3;		break;
	case 1:		at = at2;		nAxes = 9;		break;
	case 2:		at = at3;		nAxes = 3;		break;
		}
	if(!(Axes = (Axis**)calloc(nAxes, sizeof(Axis *))))return;
	if(at && nAxes) for(i = 0; i < nAxes; i++) {
		tmp_axis.loc[0].fx = at[i].x1;		tmp_axis.loc[0].fy = at[i].y1;
		tmp_axis.loc[0].fz = at[i].z1;		tmp_axis.loc[1].fx = at[i].x2;
		tmp_axis.loc[1].fy = at[i].y2;		tmp_axis.loc[1].fz = at[i].z2;
		tlbdef.Align = at[i].txa;
		if((Axes[i] = new Axis(this, data, &tmp_axis, at[i].flags))){
			Axes[i]->type = at[i].a_type;
			Axes[i]->SetSize(SIZE_LB_YDIST, at[i].lb_y);
			Axes[i]->SetSize(SIZE_LB_XDIST, at[i].lb_x);
			Axes[i]->SetSize(SIZE_TLB_YDIST, at[i].tlb_y);
			Axes[i]->SetSize(SIZE_TLB_XDIST, at[i].tlb_x);
			Axes[i]->Command(CMD_TICK_TYPE, &at[i].t_type, 0L);
			Axes[i]->Command(CMD_TLB_TXTDEF, (void*)&tlbdef, 0L);
			}
		}
}

void
Plot3D::DoAutoscale()
{
	int i;
	AxisDEF *ad;

	if(!plots) return;
	Bounds.Xmin = Bounds.Ymin = HUGE_VAL;	Bounds.Xmax = Bounds.Ymax = -HUGE_VAL;
	xBounds.fx = yBounds.fx = zBounds.fx = HUGE_VAL;
	xBounds.fy = yBounds.fy = zBounds.fy = -HUGE_VAL;
	for(i = 0; i < nPlots; i++) if(plots[i]) plots[i]->Command(CMD_AUTOSCALE, 0L, 0L);
	if(xBounds.fx <= xBounds.fy && yBounds.fx <= yBounds.fy && zBounds.fx <= zBounds.fy){
		if(Axes)for(i = 0; i < 3; i++) if(Axes[i]){
			ad = Axes[i]->axis;
			if(ad->flags & AXIS_AUTOSCALE) {
				switch(i) {
				case 0:
					if(xBounds.fx == xBounds.fy) {
						xBounds.fx -= 1.0;	xBounds.fy += 1.0;
						}
					ad->min = xBounds.fx;	ad->max = xBounds.fy;	break;
				case 1:
					if(yBounds.fx == yBounds.fy) {
						yBounds.fx -= 1.0;	yBounds.fy += 1.0;
						}
					ad->min = yBounds.fx;	ad->max = yBounds.fy;	break;
				case 2:
					if(zBounds.fx == zBounds.fy) {
						zBounds.fx -= 1.0;	zBounds.fy += 1.0;
						}
					ad->min = zBounds.fx;	ad->max = zBounds.fy;	break;
					}
				NiceAxis(ad, 4);
				if(ad->min <= 0.0 && ((ad->flags & 0xf000) == AXIS_LOG ||
					(ad->flags & 0xf000) == AXIS_RECI)) {
					ad->min = base4log(ad, i);
					}
				Axes[i]->Command(CMD_AUTOSCALE, ad, 0L);
				}
			}
		}
}

//Implement some kind of virtual trackball
//see: J. Hultquist: A Virtual Trackball
//Graphic Gems, A.S. Glassner ed.; Academic Press Inc.
//ISBN 0-12-286165-5, pp. 462-463
void
Plot3D::CalcRotation(double dx, double dy, anyOutput *o, bool accept)
{
	fPOINT3D V0, V1, A;
	double R, R2, si, dp, NewRot[6];
	double rotM[3][3], newM[3][3];			//rotation matrices

	if(!CurrGO || CurrGO->Id != GO_DRAGHANDLE || CurrGO->type < DH_18 ||
		CurrGO->type > DH_88) return;
	//get coordinates for last accepted rotation
	V0.fx = GetSize(SIZE_XPOS + CurrGO->type - DH_18);
	V0.fy = GetSize(SIZE_YPOS + CurrGO->type - DH_18);
	V0.fz = GetSize(SIZE_ZPOS + CurrGO->type - DH_18);
	//revert to last matrix	
	o->SetSpace(&cu1, &cu2, defs.cUnits, RotDef, &rc, Axes[0]->GetAxis(),
		Axes[1]->GetAxis(), Axes[2]->GetAxis());
	o->cvec2ivec(&V0, &V1);
	memcpy(&V0, &V1, sizeof(fPOINT3D));
	V1.fx += o->un2fix(dx);		V1.fy += o->un2fiy(dy);
	V0.fx -= o->rotC.fx;		V0.fy -= o->rotC.fy;		V0.fz -= o->rotC.fz;
	V1.fx -= o->rotC.fx;		V1.fy -= o->rotC.fy;		V1.fz -= o->rotC.fz;
	R = sqrt(R2 = V0.fx * V0.fx + V0.fy * V0.fy + V0.fz * V0.fz);
	R2 -= (V1.fx * V1.fx + V1.fy * V1.fy);
	if (R2 <= 1.0) return;
	V1.fz = V1.fz > 0.0 ? sqrt(R2) : -sqrt(R2);
	V0.fx /= R;			V0.fy /= R;			V0.fz /= R;
	V1.fx /= R;			V1.fy /= R;			V1.fz /= R;
	A.fx = (V1.fy * V0.fz) - (V1.fz * V0.fy);
	A.fy = (V1.fz * V0.fx) - (V1.fx * V0.fz);
	A.fz = (V1.fx * V0.fy) - (V1.fy * V0.fx);

	si = sqrt(A.fx * A.fx + A.fy * A.fy + A.fz * A.fz);
	if(si > 0.001) {
		NewRot[0] = A.fx;	NewRot[1] = A.fy;	NewRot[2] = A.fz;
		NewRot[3] = si;		NewRot[4] = sqrt(1.0-si*si);	NewRot[5] = 1.0-NewRot[4];
		//normalize vector part of NewRot
		dp = sqrt(NewRot[0]*NewRot[0] + NewRot[1]*NewRot[1] + NewRot[2]*NewRot[2]);
		NewRot[0] /= dp;	NewRot[1] /= dp;	NewRot[2] /= dp;
		//set up rotation matrix from quaternion
		//see: Graphic Gems, A.S. Glassner ed.; Academic Press Inc.
		//M.E. Pique: Rotation Tools
		// ISBN 0-12-286165-5, p.466
		rotM[0][0] = NewRot[5]*NewRot[0]*NewRot[0] + NewRot[4];
		rotM[0][1] = NewRot[5]*NewRot[0]*NewRot[1] + NewRot[3]*NewRot[2];
		rotM[0][2] = NewRot[5]*NewRot[0]*NewRot[2] - NewRot[3]*NewRot[1];
		rotM[1][0] = NewRot[5]*NewRot[0]*NewRot[1] - NewRot[3]*NewRot[2];
		rotM[1][1] = NewRot[5]*NewRot[1]*NewRot[1] + NewRot[4];
		rotM[1][2] = NewRot[5]*NewRot[1]*NewRot[2] + NewRot[3]*NewRot[0];
		rotM[2][0] = NewRot[5]*NewRot[0]*NewRot[2] + NewRot[3]*NewRot[1];
		rotM[2][1] = NewRot[5]*NewRot[1]*NewRot[2] - NewRot[3]*NewRot[0];
		rotM[2][2] = NewRot[5]*NewRot[2]*NewRot[2] + NewRot[4];
		//rotate rotation matrix
		if(MatMul(o->rotM, rotM, newM)) memcpy(&o->rotM, &newM, sizeof(newM));
		}
	if(accept) {
		//create new quaternion in RotDef from rotation matrix of output class
		Undo.RotDef(this, &RotDef, 0L);
		RotDef[4] = (o->rotM[0][0] + o->rotM[1][1] + o->rotM[2][2] -1)/2.0;
		RotDef[3] = sqrt(1.0-RotDef[4]*RotDef[4]);
		RotDef[0] = (o->rotM[1][2] - o->rotM[2][1])/(2.0 * RotDef[3]);
		RotDef[1] = (o->rotM[2][0] - o->rotM[0][2])/(2.0 * RotDef[3]);
		RotDef[2] = (o->rotM[0][1] - o->rotM[1][0])/(2.0 * RotDef[3]);
		RotDef[5] = 1.0-RotDef[4];
		}
}

bool
Plot3D::AcceptObj(GraphObj *go)
{
	if(!dispObs && !(dispObs = (obj_desc**)
		calloc(nmaxObs = 1024, sizeof(obj_desc*))))return false;
	else if((nObs+1) >= nmaxObs) dispObs = (obj_desc**)
		realloc(dispObs, (nmaxObs = nmaxObs +1024) * sizeof(obj_desc*));
	if(dispObs[++nObs] = (obj_desc*)calloc(1, sizeof(obj_desc))){
		dispObs[nObs]->Zmin = go->GetSize(SIZE_MIN_Z);
		dispObs[nObs]->Zmax = go->GetSize(SIZE_MAX_Z);
		dispObs[nObs]->go = go;
		}
	return true;
}

//Execute a heap sort before drawing the objects
//W.H. pres, B.P. Flannery, S.A. Teukolsky, W.T. Vetterling (1988/1989)
//Numerical Recipes in C, Cambridge University Press, ISBN 0-521-35465-X
// p. 245
void
Plot3D::SortObj()
{
	int l, j, ir, i;
	obj_desc *rra;

	if(nObs < 2) return;
	l=(nObs >> 1)+1;			ir = nObs;
	for( ; ; ){
		if(l > 1) rra = dispObs[--l];
		else {
			rra = dispObs[ir];
			dispObs[ir] = dispObs[1];
			if(--ir == 1) {
				dispObs[1] = rra;
				return;
				}
			}
		i = l;					j = l << 1;
		while(j <= ir) {
			if(j < ir && dispObs[j]->Zmin < dispObs[j+1]->Zmin) ++j;
			if(rra->Zmin < dispObs[j]->Zmin) {
				dispObs[i] = dispObs[j];
				j += (i = j);
				}
			else j = ir + 1;
			}
		dispObs[i] = rra;
		}
}

bool
Plot3D::Rotate(double dx, double dy, double dz, anyOutput *o, bool accept)
{
	int i;
	double si, NewRot[6];
	double rotM[3][3], newM[3][3];			//rotation matrices
	bool bRet = true;

	for(i = 0; i < 3; i++) {
		switch (i){
		case 0:
			if(dx > 0.0) {
				NewRot[0] = -o->rotM[1][0];			NewRot[1] = -o->rotM[1][1];
				NewRot[2] = -o->rotM[1][2];
				NewRot[3] = si = dx;
				}
			else {
				NewRot[0] = o->rotM[1][0];			NewRot[1] = o->rotM[1][1];
				NewRot[2] = o->rotM[1][2];
				NewRot[3] = si = -dx;
				}
			break;
		case 1:
			if(dy > 0.0) {
				NewRot[0] = -o->rotM[0][0];			NewRot[1] = -o->rotM[0][1];
				NewRot[2] = -o->rotM[0][2];
				NewRot[3] = si = dy;
				}
			else {
				NewRot[0] = o->rotM[0][0];			NewRot[1] = o->rotM[0][1];
				NewRot[2] = o->rotM[0][2];
				NewRot[3] = si = -dy;
				}
			break;
		case 2:
			if(dz > 0.0) {
				NewRot[0] = -o->rotM[2][0];			NewRot[1] = -o->rotM[2][1];
				NewRot[2] = -o->rotM[2][2];
				NewRot[3] = si = dz;
				}
			else {
				NewRot[0] = o->rotM[2][0];			NewRot[1] = o->rotM[2][1];
				NewRot[2] = o->rotM[2][2];
				NewRot[3] = si = -dz;
				}
			break;
			}
		if(si > 0.0) {
			NewRot[4] = sqrt(1.0-si*si);	NewRot[5] = 1.0-NewRot[4];
			//set up rotation matrix from quaternion
			//see: Graphic Gems, A.S. Glassner ed.; Academic Press Inc.
			//M.E. Pique: Rotation Tools
			// ISBN 0-12-286165-5, p.466
			rotM[0][0] = NewRot[5]*NewRot[0]*NewRot[0] + NewRot[4];
			rotM[0][1] = NewRot[5]*NewRot[0]*NewRot[1] + NewRot[3]*NewRot[2];
			rotM[0][2] = NewRot[5]*NewRot[0]*NewRot[2] - NewRot[3]*NewRot[1];
			rotM[1][0] = NewRot[5]*NewRot[0]*NewRot[1] - NewRot[3]*NewRot[2];
			rotM[1][1] = NewRot[5]*NewRot[1]*NewRot[1] + NewRot[4];
			rotM[1][2] = NewRot[5]*NewRot[1]*NewRot[2] + NewRot[3]*NewRot[0];
			rotM[2][0] = NewRot[5]*NewRot[0]*NewRot[2] + NewRot[3]*NewRot[1];
			rotM[2][1] = NewRot[5]*NewRot[1]*NewRot[2] - NewRot[3]*NewRot[0];
			rotM[2][2] = NewRot[5]*NewRot[2]*NewRot[2] + NewRot[4];
			if(MatMul(o->rotM, rotM, newM))	memcpy(&o->rotM, &newM, sizeof(newM));
			else accept = bRet = false;
			}
		}
	if(accept && bRet) {
		//create new quaternion in RotDef from rotation matrix of output class
		Undo.RotDef(this, &RotDef, 0L);
		RotDef[4] = (o->rotM[0][0] + o->rotM[1][1] + o->rotM[2][2] -1)/2.0;
		RotDef[3] = sqrt(1.0-RotDef[4]*RotDef[4]);
		RotDef[0] = (o->rotM[1][2] - o->rotM[2][1])/(2.0 * RotDef[3]);
		RotDef[1] = (o->rotM[2][0] - o->rotM[0][2])/(2.0 * RotDef[3]);
		RotDef[2] = (o->rotM[0][1] - o->rotM[1][0])/(2.0 * RotDef[3]);
		RotDef[5] = 1.0-RotDef[4];
		if(parent)return parent->Command(CMD_REDRAW, 0L, o);
		}
	return bRet;
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// use Plot3D to create a 2.5 dimensional chart
Chart25D::Chart25D(GraphObj *par, DataObj *d, DWORD flags)
	:Plot3D(par, d, flags)
{
	dspm.fx = dspm.fy = dspm.fz = 1.0;
}

Chart25D::~Chart25D()
{
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// use Plot3D to create a 2.5 dimensional ribbon chart
Ribbon25D::Ribbon25D(GraphObj *par, DataObj *d, DWORD flags)
	:Plot3D(par, d, flags)
{
	dspm.fx = dspm.fy = dspm.fz = 1.0;
}

Ribbon25D::~Ribbon25D()
{
}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// use Plot3D to create a 3 dimensional bubble plot
BubblePlot3D::BubblePlot3D(GraphObj *par, DataObj *d)
	:Plot3D(par, d, 0x0L)
{
}

BubblePlot3D::~BubblePlot3D()
{
}
