# ifndef _SKIT_INHB_AUX_H
# define _SKIT_INHB_AUX_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
//
//       auxilliary functios for HB i/O
//
# include "rheolef/compiler.h"
namespace rheolef { 

# define hb_id(file_name) \
        (file_name ? file_name : "line")

# define hb_error(line_no, file_name, message) \
        (std::cerr << "error: "   << hb_id(file_name) << "(" << line_no << "): " << message << ".\n", exit(1), false)

# define hb_warning(line_no, file_name, message) \
        (std::cerr << "warning: " << hb_id(file_name) << "(" << line_no << "): " << message << ".\n", true)

//
// eat until eol
//
template <class Size>
bool
endofline (std::istream& s, Size nmax)
{
	for (Size i = 0; i < nmax && s.peek() != '\n'; i++, s.get()) 
	    ;
	if (s.peek() != '\n') {
	    return false;
	}
	s.get();
	return true;
}
template <class Size>
inline
bool
skip_eoln (std::istream& s, const char *file_name, Size line_no)
{
	if (endofline(s,Size(256))) 
	    return true;
	
	if (s.eof())
	    return hb_error (line_no, file_name, "unexpected end of file");
	if (s.fail())
	    return hb_error (line_no, file_name, "input stream failed");
	if (s.bad())
	    return hb_error (line_no, file_name, "bad input stream");
	return hb_error (line_no, file_name, "unexpected input");
}
//
// if no integer read, set it to 0
//
template <class IteratorSize>
bool
getIoptional (std::istream& s, IteratorSize pi, int width = 14, bool is_required = false)
{
	char I_buf [256];
	if (is_required) {
	    s.get (I_buf, width+1);
	} else {
	    if (s.peek() == '\n') {
	        // nothing to read: set default value = 0
	        *pi = 0;
		return false;
	    }
	    // something to read...
	    s.get (I_buf, width+1);
	}
	if (!s) {
	    if (is_required) {
	        // s.state = fail or bad
	        return s.good();
	    } else {
	        // no integer read: set default value = 0
	        *pi = 0;
	        return s.good();
	    }
	}
	int tmp;
	int n_read = sscanf (I_buf, "%d", &tmp);
	warning_macro ("string("<<I_buf<<") -> int("<<tmp<<")");
	*pi = tmp;
	if (n_read != 1) {
#ifdef _RHEOLEF_HAVE_IOS_SETSTATE
	    if (is_required) {
	        // state = fail or bad
	        s.setstate(std::ios::failbit);
	    }
#endif // _RHEOLEF_HAVE_IOS_SETSTATE
	    if (!is_required) {
	        // no integer read: set default value = 0
	        *pi = 0;
	    }
	}
	return s.good();
}
//
// if no integer read, set bad/fail flag to istream
//
template <class IteratorSize>
bool
getI (std::istream& s, IteratorSize pi, int width = 14)
{
	return getIoptional (s, pi, width, true);
}
//
//  Converts a line with floats data in 'D' or 'd' format
//  to 'E' or 'e' format by simply setting to 'E'
//
inline
void
convertDtoE (char* line)
{
    for (int i = 0; line[i] != 0; i++)

       if (line[i] == 'D' || line[i] == 'd')
	   
	       line[i] = 'E';
}
inline
void
convertFDtoE (char* line)
{
    for (int i = 0; line[i] != 0; i++)

       if (line[i] == 'D' || line[i] == 'd' ||
           line[i] == 'F' || line[i] == 'f' ||
                             line[i] == 'e'  )

	       line[i] = 'E';
}
//
//  Return the index of the first occurrence of C in S
//  Return -1 if C does not occur in S
//
inline
int
getindex (const char* S, char C)
{
	for (int i = 0; S [i] != 0; i++) {
            if (S[i] == C) 
	        return i;
        }
	return -1;
}
//
// convert S to uppercase
//
inline
void
upcase (char* S)
{
	for (int i = 0; S[i] != 0; i++)
            if ('a' <= S[i]  && S[i] <= 'z' ) 
	        S[i] = S[i] + 'A' - 'a';
}
// 
// read a positive integer in S after char C
// or -1 if error
//
inline
int
read_integer_after (const char* S, char C, char ALT_C = 0)
{
	int pos = getindex (S, C);
	if ((pos < 0 || S[pos+1] == 0) && ALT_C) getindex(S, ALT_C);
	if  (pos < 0 || S[pos+1] == 0) return -1;
	int val = 0;
	for (int i = pos+1; S[i] != 0 && isdigit(S[i]); i++)
	    val = 10*val + S[i] - '0';
	return val;
}
// 
// read a positive integer in S before char C
// or -1 if error
//
inline
int
read_integer_before  (const char* S, char C, char ALT_C = 0)
{
	int pos = getindex (S, C);
	if (pos < 0 && ALT_C) getindex (S, ALT_C);
	if (pos <= 0) return -1;
	
	int val = 0;
	int P = 1;
	for (int i = pos-1; i >= 0 && isdigit(S[i]); i--) {
	    val = P*(S[i] - '0') + val;
	    P *= 10;
	}
	return val;
}
//
//  parse an INTEGER format field to determine
//  width and number of elements per line.
//
template <class Size>
bool
parse_ifmt (const char *fmt, Size line_no, const char *file_name, int *nperline, int *width)
{
	char FMT [256];
	strcpy (FMT, fmt);
	upcase(FMT);

	*nperline = read_integer_before (FMT, 'I', 'i');
	*width    = read_integer_after  (FMT, 'I', 'i');
	return (*nperline > 0 && *width > 0);
}
//
//  Parse a REAL format field to determine
//  width and number of elements per line.
//  Also sets flag indicating 'E' 'F' 'P' or 'D'
//  format.
//
template <class Size>
bool
parse_rfmt (const char *fmt, Size line_no, const char *file_name, int *nperline, int *width, bool *hasD)
{
	// set flag: will need to put a E instead of a D in each input line !
	*hasD = ((getindex (fmt, 'D') >= 0 || getindex (fmt, 'd') > 0));

	char FMT [256];
    	strcpy (FMT, fmt);
	convertFDtoE (FMT);
	
	*nperline = read_integer_before (FMT, 'E');
        *width    = read_integer_after  (FMT, 'E');
	return (*nperline > 0 && *width > 0);
}
}// namespace rheolef
# endif // _SKIT_INHB_AUX_H
