#ifndef _RHEOLEF_FORM_VF_EXPR_H
#define _RHEOLEF_FORM_VF_EXPR_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// form_vf_expr: used for expressions in variationnal formulations
//
#include "rheolef/vf_tag.h"
#include <boost/numeric/ublas/matrix.hpp> // ublas::matrix

namespace rheolef {
namespace ublas = boost::numeric::ublas;

// ---------------------------------------------------------------------------
// wrapper
// ---------------------------------------------------------------------------
template<class RawExpr>
class form_vf_expr {
public:
// typedefs:

  typedef typename RawExpr::size_type            size_type;
  typedef typename RawExpr::memory_type          memory_type;
  typedef typename RawExpr::value_type           value_type;
  typedef typename RawExpr::scalar_type          scalar_type;
  typedef typename RawExpr::float_type           float_type;
  typedef typename RawExpr::space_type           space_type;
  typedef typename RawExpr::vf_tag_type          vf_tag_type;
  typedef typename RawExpr::vf_dual_tag_type     vf_dual_tag_type;
  typedef form_vf_expr<RawExpr>                  self_type;
  typedef form_vf_expr<typename RawExpr::dual_self_type> 
                                                 dual_self_type;
  typedef typename RawExpr::maybe_symmetric 	 maybe_symmetric;

// alocators:

  form_vf_expr (const RawExpr& raw_expr) 
    : _raw_expr(raw_expr) {}

// accessors:

  const space_type& get_test_space() const  { return _raw_expr.get_test_space(); }
  const space_type& get_trial_space() const { return _raw_expr.get_trial_space(); }
  size_type n_derivative() const            { return _raw_expr.n_derivative(); }

// mutable modifiers:
  void initialize (const geo_basic<float_type,memory_type>& dom, const quadrature<float_type>& quad, bool ignore_sys_coord) const {
    _raw_expr.initialize (dom, quad, ignore_sys_coord);
  }
  void initialize (const band_basic<float_type,memory_type>& gh, const quadrature<float_type>& quad, bool ignore_sys_coord) const {  
    _raw_expr.initialize (gh, quad, ignore_sys_coord);
  }
  void element_initialize (const geo_element& K) const {
    _raw_expr.element_initialize (K);
  }
  void basis_evaluate (const reference_element& hat_K, size_type q, ublas::matrix<scalar_type>& value) const {
    _raw_expr.basis_evaluate (hat_K, q, value);
  }
  template<class ValueType>
  void valued_check() const {
    static const bool status = details::is_equal<ValueType,scalar_type>::value;
    check_macro (status, "unexpected result_type"); 
    _raw_expr.valued_check<ValueType>();
  }

protected:
// data:
  RawExpr   _raw_expr;
};
// ---------------------------------------------------------------------------
// binary function call: (f field_expr1 field_expr2) -> form_expr
// example: integrate(u*v)
// ---------------------------------------------------------------------------
template<class BinaryFunction, class Expr1, class Expr2>
class form_vf_expr_bf_field {
public:
// typedefs:

  typedef geo_element::size_type                   	size_type;
  typedef typename promote_memory<typename Expr1::memory_type,typename Expr2::memory_type>::type 
 				                   	memory_type;
  typedef typename details::generic_binary_traits<BinaryFunction>::template result_hint<
          typename Expr1::value_type
         ,typename Expr2::value_type>::type             result_hint;
  typedef typename details::generic_binary_traits<BinaryFunction>::template hint<
	  typename Expr1::value_type
	 ,typename Expr2::value_type
	 ,result_hint>::result_type                     value_type;
  typedef typename scalar_traits<value_type>::type  	scalar_type;
  typedef typename  float_traits<value_type>::type 	float_type;
  typedef space_basic<scalar_type,memory_type>		space_type; // TODO: deduce from Exprs
  typedef typename details::bf_vf_tag<BinaryFunction,
	typename Expr1::vf_tag_type,
	typename Expr2::vf_tag_type>::type              vf_tag_type;
  typedef typename details::dual_vf_tag<vf_tag_type>::type
                                                        vf_dual_tag_type;
  typedef form_vf_expr_bf_field<BinaryFunction,Expr1,Expr2>   self_type;
  typedef form_vf_expr_bf_field<BinaryFunction,typename Expr1::dual_self_type,typename Expr2::dual_self_type>
                                                        dual_self_type;
  typedef typename mpl::and_<typename details::generic_binary_traits<BinaryFunction>::is_symmetric::type,
		             typename details::is_equal<
				Expr1,
				typename Expr2::dual_self_type>::type>::type 
	                                                maybe_symmetric;

// alocators:

  form_vf_expr_bf_field (const BinaryFunction& f, 
		    const Expr1&    expr1,
                    const Expr2&    expr2)
    : _f(f), _expr1(expr1), _expr2(expr2) {}

// accessors:

  const space_type&  get_test_space()  const { 
    return (details::is_equal<typename Expr1::vf_tag_type, details::vf_tag_01>::value) ?
	_expr1.get_vf_space() : _expr2.get_vf_space();
  }
  const space_type&  get_trial_space() const {
    return (details::is_equal<typename Expr1::vf_tag_type, details::vf_tag_10>::value) ?
	_expr1.get_vf_space() : _expr2.get_vf_space();
  }

  size_type n_derivative() const             { return _expr1.n_derivative() + _expr2.n_derivative(); }

// mutable modifiers:

  void initialize (const geo_basic<float_type,memory_type>& dom, const quadrature<float_type>& quad, bool ignore_sys_coord) const { 
    _expr1.initialize (dom, quad, ignore_sys_coord);
    _expr2.initialize (dom, quad, ignore_sys_coord);
  }
  void initialize (const band_basic<float_type,memory_type>& gh, const quadrature<float_type>& quad, bool ignore_sys_coord) const {  
    _expr1.initialize (gh, quad, ignore_sys_coord);
    _expr2.initialize (gh, quad, ignore_sys_coord);
  }
  void element_initialize (const geo_element& K) const {
    _expr1.element_initialize (K);
    _expr2.element_initialize (K);
  }
  template<class ValueType, class Arg1, class Arg2>
  void evaluate_call (const reference_element& hat_K, size_type q, ublas::matrix<ValueType>& value) const {
    if (details::is_equal<typename Expr2::vf_tag_type, details::vf_tag_01>::value) {
      // expr1 is trial and expr2 is test
      std::vector<Arg2> v_test  (value.size1()); _expr2.basis_evaluate (hat_K, q, v_test);
      std::vector<Arg1> u_trial (value.size2()); _expr1.basis_evaluate (hat_K, q, u_trial);
      for (size_type i = 0, ni = value.size1(); i < ni; ++i) {
      for (size_type j = 0, nj = value.size2(); j < nj; ++j) {
        value(i,j) = _f (u_trial[j], v_test[i]);
      }}
    } else {
      // expr2 is trial and expr1 is test
      std::vector<Arg2> v_test  (value.size1()); _expr1.basis_evaluate (hat_K, q, v_test);
      std::vector<Arg1> u_trial (value.size2()); _expr2.basis_evaluate (hat_K, q, u_trial);
      for (size_type i = 0, ni = value.size1(); i < ni; ++i) {
      for (size_type j = 0, nj = value.size2(); j < nj; ++j) {
        value(i,j) = _f (u_trial[j], v_test[i]);
      }}
    }
  }
  // when both args are defined at compile time:
  template<class This, class ValueType,
        class Arg1,        space_constant::valued_type Arg1Tag,
        class Arg2,        space_constant::valued_type Arg2Tag>
  struct evaluate_switch {
    void operator() (const This& obj, const reference_element& hat_K, size_type q, ublas::matrix<ValueType>& value) const {
      obj.template evaluate_call<ValueType, Arg1, Arg2> (hat_K, q, value);
    }
  };
  template<class ValueType>
  void basis_evaluate (const reference_element& hat_K, size_type q, ublas::matrix<ValueType>& value) const {
    typedef typename details::generic_binary_traits<BinaryFunction>::template hint<
	  typename Expr1::value_type
	 ,typename Expr2::value_type
	 ,ValueType>::first_argument_type   first_argument_type;
    typedef typename details::generic_binary_traits<BinaryFunction>::template hint<
	  typename Expr1::value_type
	 ,typename Expr2::value_type
	 ,ValueType>::second_argument_type second_argument_type;
    static const space_constant::valued_type  first_argument_tag = space_constant::valued_tag_traits<first_argument_type>::value;
    static const space_constant::valued_type second_argument_tag = space_constant::valued_tag_traits<second_argument_type>::value;
    evaluate_switch <self_type, ValueType,
        first_argument_type,   first_argument_tag,
        second_argument_type, second_argument_tag>   eval;
    eval (*this, hat_K, q, value);
  }
  template<class ValueType>
  void valued_check() const {
    typedef typename details::generic_binary_traits<BinaryFunction>::template hint<
	  typename Expr1::value_type
	 ,typename Expr2::value_type
	 ,ValueType>::first_argument_type   A1;
    typedef typename details::generic_binary_traits<BinaryFunction>::template hint<
	  typename Expr1::value_type
	 ,typename Expr2::value_type
	 ,ValueType>::second_argument_type A2;
    if (! is_undeterminated<A1>::value) _expr1.valued_check<A1>();
    if (! is_undeterminated<A2>::value) _expr2.valued_check<A2>();
  }
protected:
// data:
  BinaryFunction  _f;
  Expr1           _expr1;
  Expr2           _expr2;
};
// ---------------------------------------------------------------------------
// unary function call: (f form_expr) -> form_expr
// example: -(u*v), 2*(u*v), (u*v)/2
// ---------------------------------------------------------------------------
template<class UnaryFunction, class Expr>
class form_vf_expr_uf {
public:
// typedefs:

  typedef geo_element::size_type                   	size_type;
  typedef typename Expr::memory_type                    memory_type;
  typedef typename details::generic_unary_traits<UnaryFunction>::template result_hint<
          typename Expr::value_type>::type              result_hint;
  typedef typename details::generic_unary_traits<UnaryFunction>::template hint<
	  typename Expr::value_type
	 ,result_hint>::result_type                     value_type;
  typedef typename scalar_traits<value_type>::type  	scalar_type;
  typedef typename  float_traits<value_type>::type 	float_type;
  typedef space_basic<scalar_type,memory_type>		space_type;
  typedef typename Expr::vf_tag_type                    vf_tag_type;
  typedef typename details::dual_vf_tag<vf_tag_type>::type
                                                        vf_dual_tag_type;
  typedef form_vf_expr_uf<UnaryFunction,Expr>           self_type;
  typedef form_vf_expr_uf<UnaryFunction, typename Expr::dual_self_type>
                                                        dual_self_type;
  typedef typename Expr::maybe_symmetric::type          maybe_symmetric;

// alocators:

  form_vf_expr_uf (const UnaryFunction& f, const Expr& expr)
    : _f(f), _expr(expr) {}

// accessors:

  const space_type&  get_trial_space() const { return _expr.get_trial_space(); }
  const space_type&  get_test_space()  const { return _expr.get_test_space(); }
  size_type n_derivative() const             { return _expr.n_derivative(); }

// mutable modifiers:

  void initialize (const geo_basic<float_type,memory_type>& dom, const quadrature<float_type>& quad, bool ignore_sys_coord) const { 
    _expr.initialize (dom, quad, ignore_sys_coord);
  }
  void initialize (const band_basic<float_type,memory_type>& gh, const quadrature<float_type>& quad, bool ignore_sys_coord) const {  
    _expr.initialize (gh, quad, ignore_sys_coord);
  }
  void element_initialize (const geo_element& K) const {
    _expr.element_initialize (K);
  }
  template<class ValueType>
  void basis_evaluate (const reference_element& hat_K, size_type q, ublas::matrix<ValueType>& value) const {
    // ValueType is float_type in general: elementary matrix
    typedef ValueType A1;
    ublas::matrix<A1> value1 (value.size1(), value.size2());
    _expr.basis_evaluate (hat_K, q, value1);
    for (size_type i = 0, ni = value.size1(); i < ni; ++i) {
    for (size_type j = 0, nj = value.size2(); j < nj; ++j) {
      value(i,j) = _f (value1(i,j));
    }}
  }
  template<class ValueType>
  void valued_check() const {
    typedef ValueType A1;
    if (! is_undeterminated<A1>::value) _expr.valued_check<A1>();
  }
protected:
// data:
  UnaryFunction  _f;
  Expr           _expr;
};
// ---------------------------------------------------------------------------
// binary function call: (f form_expr1 form_expr2) -> form_expr
// example: operator+ between two forms as in
//	    (u*v) + dot(grad(u),grad(v))
// ---------------------------------------------------------------------------
template<class BinaryFunction, class Expr1, class Expr2>
class form_vf_expr_bf {
public:
// typedefs:

  typedef geo_element::size_type                   	size_type;
  typedef typename promote_memory<typename Expr1::memory_type,typename Expr2::memory_type>::type 
 				                   	memory_type;
  typedef typename details::generic_binary_traits<BinaryFunction>::template result_hint<
          typename Expr1::value_type
         ,typename Expr2::value_type>::type             result_hint;
  typedef typename details::generic_binary_traits<BinaryFunction>::template hint<
	  typename Expr1::value_type
	 ,typename Expr2::value_type
	 ,result_hint>::result_type                     value_type;
  typedef typename scalar_traits<value_type>::type  	scalar_type;
  typedef typename  float_traits<value_type>::type 	float_type;
  typedef space_basic<scalar_type,memory_type>		space_type; // TODO: deduce from Exprs
  typedef typename details::bf_vf_tag<BinaryFunction,
	typename Expr1::vf_tag_type,
	typename Expr2::vf_tag_type>::type              vf_tag_type;
  typedef typename details::dual_vf_tag<vf_tag_type>::type
                                                        vf_dual_tag_type;
  typedef form_vf_expr_bf<BinaryFunction,Expr1,Expr2>         self_type;
  typedef form_vf_expr_bf<BinaryFunction,typename Expr1::dual_self_type,
                                        typename Expr2::dual_self_type>
                                                        dual_self_type;
  typedef typename mpl::and_<typename Expr1::maybe_symmetric::type,
		             typename Expr2::maybe_symmetric::type>::type
	                                                maybe_symmetric;

// alocators:

  form_vf_expr_bf (const BinaryFunction& f, 
		    const Expr1&    expr1,
                    const Expr2&    expr2)
    : _f(f), _expr1(expr1), _expr2(expr2) {}

// accessors:

  const space_type&  get_trial_space() const { return _expr1.get_trial_space(); }
  const space_type&  get_test_space()  const { return _expr1.get_test_space(); }
  size_type n_derivative() const             { return std::max(_expr1.n_derivative(), _expr2.n_derivative()); }

// mutable modifiers:

  // TODO: at init, check that exp1 & expr2 has the same test & trial spaces
  void initialize (const geo_basic<float_type,memory_type>& dom, const quadrature<float_type>& quad, bool ignore_sys_coord) const { 
    _expr1.initialize (dom, quad, ignore_sys_coord);
    _expr2.initialize (dom, quad, ignore_sys_coord);
  }
  void initialize (const band_basic<float_type,memory_type>& gh, const quadrature<float_type>& quad, bool ignore_sys_coord) const {  
    _expr1.initialize (gh, quad, ignore_sys_coord);
    _expr2.initialize (gh, quad, ignore_sys_coord);
  }
  void element_initialize (const geo_element& K) const {
    _expr1.element_initialize (K);
    _expr2.element_initialize (K);
  }
  // ValueType is float_type in general: elementary matrix
  template<class ValueType>
  void basis_evaluate (const reference_element& hat_K, size_type q, ublas::matrix<ValueType>& value) const {
    // for f=operator+ => sum of two elementary matrix of the same type
    // TODO: otherwise ValueType1 and 2 could be obtained from the hint<> helper
    typedef ValueType A1;
    typedef ValueType A2;
    ublas::matrix<A1> value1 (value.size1(), value.size2());
    ublas::matrix<A2> value2 (value.size1(), value.size2());
    _expr1.basis_evaluate (hat_K, q, value1);
    _expr2.basis_evaluate (hat_K, q, value2);
    for (size_type i = 0, ni = value.size1(); i < ni; ++i) {
    for (size_type j = 0, nj = value.size2(); j < nj; ++j) {
      value(i,j) = _f (value1(i,j), value2(i,j));
    }}
  }
  template<class ValueType>
  void valued_check() const {
    typedef ValueType A1;
    typedef ValueType A2;
    if (! is_undeterminated<A1>::value) _expr1.valued_check<A1>();
    if (! is_undeterminated<A2>::value) _expr2.valued_check<A2>();
  }
protected:
// data:
  BinaryFunction  _f;
  Expr1           _expr1;
  Expr2           _expr2;
};
// ---------------------------------------------------------------------------
// binary function call: (f nl_field_expr vf_form_expr) -> form_expr
// example: integrate(eta_h*(u*v))
// ---------------------------------------------------------------------------
template<class BinaryFunction, class NLExpr, class VFExpr>
class form_vf_expr_binded_bf {
public:
// typedefs:

  typedef geo_element::size_type                   	size_type;
  typedef typename promote_memory<typename NLExpr::memory_type,typename VFExpr::memory_type>::type 
 				                   	memory_type;
  typedef typename details::generic_binary_traits<BinaryFunction>::template result_hint<
          typename NLExpr::value_type
         ,typename VFExpr::value_type>::type             result_hint;
  typedef typename details::generic_binary_traits<BinaryFunction>::template hint<
	  typename NLExpr::value_type
	 ,typename VFExpr::value_type
	 ,result_hint>::result_type                     value_type;
  typedef typename scalar_traits<value_type>::type  	scalar_type;
  typedef typename  float_traits<value_type>::type 	float_type;
  typedef space_basic<scalar_type,memory_type>		space_type; // TODO: deduce from Exprs
  typedef typename details::bf_vf_tag<BinaryFunction,
	details::vf_tag_00,
	typename VFExpr::vf_tag_type>::type             vf_tag_type;
  typedef typename details::dual_vf_tag<vf_tag_type>::type
                                                        vf_dual_tag_type;
  typedef form_vf_expr_binded_bf<BinaryFunction,NLExpr,VFExpr>         self_type;
  typedef form_vf_expr_binded_bf<BinaryFunction,NLExpr,typename VFExpr::dual_self_type>
                                                        dual_self_type;
  typedef typename VFExpr::maybe_symmetric::type	maybe_symmetric;
  // TODO: symmetry: works only when eta_h is scalar
  // TODO: problem when ddot(eta_h,otimes(u,v)) when eta_h is unsymmetric tensor 
  // and "unsymmetric tensor" is not known at compile time

// alocators:

  form_vf_expr_binded_bf (const BinaryFunction& f, 
		    const NLExpr&    nl_expr,
                    const VFExpr&    vf_expr)
    : _f(f), 
      _nl_expr(nl_expr),
      _vf_expr(vf_expr),
      _scalar_nl_value_quad(),
      _vector_nl_value_quad(),
      _tensor_nl_value_quad()
    {}

// accessors:

  const space_type&  get_trial_space() const { return _vf_expr.get_trial_space(); }
  const space_type&  get_test_space()  const { return _vf_expr.get_test_space(); }
  size_type n_derivative() const             { return _vf_expr.n_derivative(); }

// mutable modifiers:

  void initialize (const geo_basic<float_type,memory_type>& dom, const quadrature<float_type>& quad, bool ignore_sys_coord) const { 
    _nl_expr.initialize (dom, quad);
    _vf_expr.initialize (dom, quad, ignore_sys_coord);
  }
  void initialize (const band_basic<float_type,memory_type>& gh, const quadrature<float_type>& quad, bool ignore_sys_coord) const {  
    _nl_expr.initialize (gh.level_set(), quad);
    _vf_expr.initialize (gh,             quad, ignore_sys_coord);
  }
  // ---------------------------------------------
  // element initialize: evaluate nl_expr
  // ---------------------------------------------
  void element_initialize (const geo_element& K) const {
    typedef typename details::generic_binary_traits<BinaryFunction>::template hint<
          typename NLExpr::value_type
         ,typename VFExpr::value_type
         ,value_type>::first_argument_type   first_argument_type;
    nl_switch<self_type,first_argument_type> nl_helper;
    nl_helper.element_initialize (*this, K);
    _vf_expr.element_initialize (K);
  }
  template<class ValueType>
  void basis_evaluate (const reference_element& hat_K, size_type q, ublas::matrix<ValueType>& value) const {
    typedef ValueType Arg1; // TODO: switch
    typedef ValueType Arg2; // TODO: switch ; is float_type in general, as elementary matrix
    nl_switch<self_type,Arg1> nl_helper;
    const Arg1& value1 = nl_helper.get_nl_value (*this, q);
    ublas::matrix<Arg2> value2 (value.size1(), value.size2());
    _vf_expr.basis_evaluate (hat_K, q, value2);
    for (size_type i = 0, ni = value.size1(); i < ni; ++i) {
    for (size_type j = 0, nj = value.size2(); j < nj; ++j) {
      value(i,j) = _f (value1, value2(i,j));
    }}
  }
  template<class ValueType>
  void valued_check() const {
    typedef ValueType A1;
    typedef ValueType A2;
    if (! is_undeterminated<A1>::value) _nl_expr.valued_check<A1>();
    if (! is_undeterminated<A2>::value) _vf_expr.valued_check<A2>();
  }
//protected:
// data:
  BinaryFunction  _f;
  NLExpr          _nl_expr;
  VFExpr          _vf_expr;
  mutable std::vector<scalar_type>                 _scalar_nl_value_quad;
  mutable std::vector<point_basic<scalar_type> >   _vector_nl_value_quad;
  mutable std::vector<tensor_basic<scalar_type> >  _tensor_nl_value_quad;
};
#ifndef TODO
#endif // TODO

} // namespace rheolef
#endif // _RHEOLEF_FIELD_VF_EXPR_H
