///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
// 
// check the "2Ds_Ds" form on some simple geometries
// 
// authors: Mahamar Dicko, Jocelyn Etienne, Pierre Saramito
//
// usage:
//	prog mesh.geo Pk qorder tol u0 u1 u2 v0 v1 v2 n0 n1 n2
//
// examples:
//
//	mkgeo_ball -s -t 10 > ball-P1.geo
// 	./form_2Ds_Ds_tst ball-P1.geo P1 18 0.050 z x^2 'x*y' 'x*y' x^2 z x y z
//
//	mkgeo_ball -s -t 10 -order 2 > ball-P2.geo
// 	./form_2Ds_Ds_tst ball-P2.geo P2 18 0.002 z x^2 'x*y' 'x*y' x^2 z x y z
//
//	mkgeo_grid -T 10 > cube-10.geo
//	./geo_domain_tst cube-10 right > cube-10.right.geo
// 	./form_2Ds_Ds_tst cube-10.right.geo P1 18 1e-14 z x^2 'x*y' 'x*y' x^2 z 0 1 0
// 	./form_2Ds_Ds_tst cube-10.right.geo P2 18 1e-14 z x^2 'x*y' 'x*y' x^2 z 0 1 0
//
// note: the banded level set is also supported: see the form_2Ds_Ds_tst.sh for invocations.
//
#include <iostream>
#include <vector>
#include <ginac/ginac.h>
#include "rheolef.h"
using namespace rheolef;
using namespace GiNaC;
using namespace std;

// --------------------------------------------------------------
// scalar-valued class-function:
// constructor from a string or a ginac expression
// --------------------------------------------------------------
class scalar_function: public unary_function<point, Float> {
public:

// allocators:

  scalar_function (
    const ex&                  f,
    const point_basic<symbol>& x = point_basic<symbol>(symbol("x"),symbol("y"),symbol("z")))
   : _x(x), _f(f), _m() {}

  scalar_function (const string& f_expr)
   : _x(symbol("x"),symbol("y"),symbol("z")), _f(), _m()
  {
    symtab table;
    table["x"] = _x[0];
    table["y"] = _x[1];
    table["z"] = _x[2];
    parser reader (table);
    _f = reader (f_expr);
  }

// accessor:

  Float operator() (const point& x) const {
     _m[_x[0]] = x[0];
     _m[_x[1]] = x[1];
     _m[_x[2]] = x[2];
     return Float (ex_to<numeric>(_f.subs(_m).evalm()).to_double());
  }
// data:
protected:
  point_basic<symbol>  _x; 
  ex                   _f;
  mutable exmap        _m;
};
// --------------------------------------------------------------
// vector-valued class-function:
// constructor from a string or a ginac expression
// --------------------------------------------------------------
class vector_function: public unary_function<point, point> {
public:

// allocators:

  vector_function (
    const point_basic<ex>&     f, 
    const point_basic<symbol>& x = point_basic<symbol>(symbol("x"),symbol("y"),symbol("z")))
   : _x(x), _f(f), _m() {}

  vector_function (const point_basic<string>& f_expr)
   : _x(symbol("x"),symbol("y"),symbol("z")), _f(), _m() {
    symtab table;
    table["x"] = _x[0];
    table["y"] = _x[1];
    table["z"] = _x[2];
    parser reader (table);
    for (size_t i = 0; i < 3; i++) {
      _f[i] = reader (f_expr[i]);
    }
  }

// accesors:

  point operator() (const point& x) const {
     _m[_x[0]] = x[0];
     _m[_x[1]] = x[1];
     _m[_x[2]] = x[2];
     return point(ex_to<numeric>(_f[0].subs(_m).evalm()).to_double(),
		  ex_to<numeric>(_f[1].subs(_m).evalm()).to_double(),
		  ex_to<numeric>(_f[2].subs(_m).evalm()).to_double());
  }
// data:
protected:
  point_basic<symbol>  _x; 
  point_basic<ex>      _f;
  mutable exmap        _m;
};
// --------------------------------------------------------------
// symbolic computation of the form 2Ds_Ds(u,v) with given n(x)
// --------------------------------------------------------------
// call with ginac::ex arguments
ex
form_2Ds_Ds (
  const point_basic<ex>&     u,
  const point_basic<ex>&     v,
  const point_basic<ex>&     n,
  const point_basic<symbol>& x)
{
  // compute Du
  ex Du00 =  diff(u[0],x[0]);
  ex Du01 = (diff(u[0],x[1]) + diff(u[1],x[0]))/2;
  ex Du02 = (diff(u[0],x[2]) + diff(u[2],x[0]))/2;
  ex Du11 =  diff(u[1],x[1]);
  ex Du12 = (diff(u[1],x[2]) + diff(u[2],x[1]))/2;
  ex Du22 =  diff(u[2],x[2]);
  matrix Du(3,3);
  Du = Du00, Du01, Du02,
       Du01, Du11, Du12,
       Du02, Du12, Du22;
         
  // compute Dv
  ex Dv00 =  diff(v[0],x[0]);
  ex Dv01 = (diff(v[0],x[1]) + diff(v[1],x[0]))/2;
  ex Dv02 = (diff(v[0],x[2]) + diff(v[2],x[0]))/2;
  ex Dv11 =  diff(v[1],x[1]);
  ex Dv12 = (diff(v[1],x[2]) + diff(v[2],x[1]))/2;
  ex Dv22 =  diff(v[2],x[2]);      
  matrix Dv(3,3);
  Dv = Dv00, Dv01, Dv02,
       Dv01, Dv11, Dv12,
       Dv02, Dv12, Dv22;

  // compute P 
  matrix P(3,3);
  P = 1-n[0]*n[0],   -n[0]*n[1],  -n[0]*n[2],
       -n[0]*n[1] , 1-n[1]*n[1],  -n[1]*n[2],
       -n[0]*n[2] ,  -n[1]*n[2], 1-n[2]*n[2];
             
  // compute D_su and D_sv
  matrix D_su = P.mul(Du.mul(P)); 
  matrix D_sv = P.mul(Dv.mul(P)); 
   
  // compute 2D_su:D_sv
  ex D_su_D_sv = 0;
  for (size_t i = 0; i < 3; i++){
    for (size_t j = 0; j < 3; j++){
      D_su_D_sv += D_su(i,j)*D_sv(i,j);
    }
  }     
  ex expr = 2*D_su_D_sv;
  derr << "2Ds(u):Ds(v) = " << expr << endl;
  return expr;
}
// call with string arguments
scalar_function
form_2Ds_Ds (
  const point_basic<string>& u_expr,
  const point_basic<string>& v_expr,
  const point_basic<string>& n_expr)
{
    point_basic<symbol> x (symbol("x"),symbol("y"),symbol("z"));
    symtab table;
    table["x"] = x[0];
    table["y"] = x[1];
    table["z"] = x[2];
    parser reader(table);
    point_basic<ex> u, v, n;
    for (size_t i = 0; i < 3; i++) {
      u[i] = reader(u_expr[i]);
      v[i] = reader(v_expr[i]);
      n[i] = reader(n_expr[i]);
    }
    return scalar_function (form_2Ds_Ds (u,v,n,x), x);
}
// --------------------------------------------------------------
// compare two evaluations of 2Ds(u):Ds(v) :
//  symbolic : integrate_omega f(x) dx
//  		with f = 2Ds(u):Ds(v) and using a quadrature
//  numeric :  form 2Ds_Ds (uh,vh)
// arguments are sring expressions for u,v and n
// --------------------------------------------------------------
Float
compute_error_s (
  const geo&                      omega,
  const string&                   approx,
  const point_basic<std::string>& u_expr,
  const point_basic<std::string>& v_expr,
  const point_basic<std::string>& n_expr,
  const quadrature_option_type&   qopt)
{
  // symbolic computation:
  Float int_f = integrate (omega, form_2Ds_Ds (u_expr,v_expr,n_expr), qopt);

  // numeric computation:
  space Xh (omega, approx, "vector");
  field uh = interpolate (Xh, vector_function (u_expr));
  field vh = interpolate (Xh, vector_function (v_expr));
  form a (Xh, Xh, "2D_D", qopt);
  Float int_fh = a (uh, vh);

  derr << setprecision(16)
       << "int_h 2Ds(u):Ds(v)   = " << int_f  << endl
       << "int_h 2Ds(uh):Ds(vh) = " << int_fh << endl;
    
  return fabs(int_f - int_fh);
}
// the same with the banded level set method
// assume the unit sphere:
Float phi (const point& x) { return norm(x) - 1; }

Float
compute_error_band (
  const geo&                      lambda,
  const string&                   approx,
  const point_basic<std::string>& u_expr,
  const point_basic<std::string>& v_expr,
  const point_basic<std::string>& n_expr,
  const quadrature_option_type&   qopt)
{
  space Xh  (lambda, approx);
  field phi_h = interpolate(Xh, phi);
  band gh (phi_h); // TODO: option with quadrangles

  // symbolic computation:
  Float int_f = integrate (gh.level_set(), form_2Ds_Ds (u_expr,v_expr,n_expr), qopt);

  // numeric computation:
  space Bh  (gh.band(), approx, "vector");
  field uh = interpolate (Bh, vector_function (u_expr));
  field vh = interpolate (Bh, vector_function (v_expr));
  form a (Bh, Bh, "2D_D", gh, qopt);
  Float int_fh = a (uh, vh);

  derr << setprecision(16)
       << "int_h 2Ds(u):Ds(v)   = " << int_f  << endl
       << "int_h 2Ds(uh):Ds(vh) = " << int_fh << endl;
    
  return fabs(int_f - int_fh);
}
// --------------------------------------------------------------
// usage: prog mesh.geo Pk qorder tol u0 u1 u2 v0 v1 v2 n0 n1 n2
// --------------------------------------------------------------
int main(int argc, char**argv) {
    environment rheolef (argc,argv);
    geo omega (argv[1]);
    string approx = argv[2];
    quadrature_option_type qopt;
    qopt.set_order  (atoi(argv[3]));
    qopt.set_family (quadrature_option_type::gauss);
    Float tol = atof (argv[4]);

    size_t d = omega.dimension();
    point_basic<std::string> u_expr ("0", "0", "0");
    point_basic<std::string> v_expr ("0", "0", "0");
    point_basic<std::string> n_expr ("0", "0", "0");
    size_t next = 5;
    for (size_t i = 0; i < d; ++i) {
      u_expr [i] = (next+i < size_t(argc)) ? argv[next+i] : "0";
    }
    next += d;
    for (size_t i = 0; i < d; ++i) {
      v_expr [i] = (next+i < size_t(argc)) ? argv[next+i] : "0";
    }
    next += d;
    for (size_t i = 0; i < d; ++i) {
      n_expr [i] = (next+i < size_t(argc)) ? argv[next+i] : "0";
    }
    derr << "approx = "<<approx<<endl
         << "qorder = "<<qopt.get_order()<<endl
         << "u = "<<u_expr<<endl
         << "v = "<<v_expr<<endl
         << "n = "<<n_expr<<endl;
    Float err = (omega.map_dimension() < omega.dimension()) ? 
        compute_error_s    (omega, approx, u_expr, v_expr, n_expr, qopt) :
        compute_error_band (omega, approx, u_expr, v_expr, n_expr, qopt);

    dout << setprecision(16)
         << "# nelt err" << endl
         << omega.dis_size() << " " << err << endl;
    return (err < tol) ? 0 : 1;
}
