#ifndef _RHEOLEF_INV_MASS_H
#define _RHEOLEF_INV_MASS_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/form_element.h"

namespace rheolef { 

/*Class:inv_mass
NAME: @code{inv_mass} -- invert of L2 scalar product
@bfindex inv_mass
@cindex mass matrix inversion
@cindex discontinuous approximation
@clindex form
@apindex P0
@apindex P1d
SYNOPSIS:
  @example
  	form(const space& V, const space& V, "inv_mass");
  @end example
DESCRIPTION:       
  @noindent
  Assembly the invert of the matrix associated to the
  L2 scalar product of the finite element space V:
@iftex
@tex
  $$
   	m(u,v) = \int_\Omega  u v \, dx
  $$
@end tex
@end iftex
@ifnottex
  @example
                 /
                 |
   	m(u,v) = | u v dx
                 |
                 / Omega
  @end example
@end ifnottex
  @noindent
  The V space may be either a 
  @code{P0} or
  @code{P1d}
  discontinuous finite 
  element spaces
  @pxref{form class}.

EXAMPLE:
  @noindent
  The following piece of code build the invert of
  the mass matrix
  associated to the @code{P1d}  approximation:
  @example
	geo omega_h ("square");
   	space Vh (omega_h , "P1d");
   	form im (Vh, Vh, "inv_mass");
  @end example
AUTHOR: 
    LMC-IMAG, 38041 Grenoble cedex 9, France
    | Pierre.Saramito@imag.fr
DATE:
    27 november 1997
End:
*/
template<class T, class M>
class inv_mass: public form_element_rep<T,M> {
public:
    typedef form_element_rep<T,M>       base;
    typedef typename base::size_type    size_type;
    typedef typename base::geo_type     geo_type;
    typedef typename base::space_type   space_type;

    inv_mass (const space_type& X, const space_type& Y, const geo_type& omega,
        const quadrature_option_type& qopt)
	: form_element_rep<T,M> (X,Y,omega,qopt) { base::initialize_all(); }

    void operator() (const geo_element& K, ublas::matrix<T>& m) const;
    void initialize () const;
    bool is_symmetric () const { return true; }
};

}// namespace rheolef
#endif // _RHEOLEF_INV_MASS_H
