
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// TODO: filter options
// ------------
//  -extract N
// ------------
//  -mask  u
//  -unmask  u
//  -mask-all
//  -unmask-all
//
//  -select u <==> -mask-all -unmask u
//
//  -show-names
// ------------
//  -norm {-l2|-linf|-h1}
//	-> normes in direct via gnuplot via branch (geo with dim=0)
//
//  Q. how to combine and synchronize images u(t) and |u(t)| ?
//     -> field n(t,x), mesh with dim=0, one point p0, value n(t,p0) = |u(t)| 
// ------------
// -energy-norm
//  sqrt(|u^{n+1}-u^n|/(t^{n+1}-t^n)) : mesure a(u,u), l'energie (norme l2 pour u)
// ------------
// graphic:
//   -height      : with -topography, interprets scalar as height 
//   -yield value : mask all scalars lesser than
// menu:
//   stop/start
//   start number
//   transparency : of the height
//   edit color map for height : can put uniform color
//   stop/start

/*Prog:branch
NAME: @code{branch} -- handle a family of fields

@cindex animation
@cindex continuation methods
@cindex time-dependent problems
@pindex branch
@cindex RHEOPATH environment variable
@fiindex @file{.branch} family of fields 

SYNOPSIS:
  @example
  	branch [@var{options}] @var{filename}
  @end example

EXAMPLE:
  Generates vtk file colection for visualization with paraview:
  @example
	branch output.branch -paraview
  @end example
DESCRIPTION:      
  Read and output a branch of finite element fields from file,
  in field text file format.
INPUT FILE SPECIFICATION:
    @table @code

    @itemx -I@var{dir}
	add @var{dir} to the RHEOPATH search path.
    	See also @ref{geo class} for RHEOPATH mechanism. 

    @itemx @var{filename}
	specifies the name of the file containing
      	the input field.

    @itemx -
	read field on standard input instead on a file.

    @itemx -ndigit @var{int}
        Number of digits used to print floating point values
        when using the @samp{-geo} option.
        Default depends upon the machine precision associated to the
        @code{Float} type.

    @end table

OUTPUT AND RENDER SPECIFICATION:
    @table @code

    @itemx -extract @var{int}
	Extract the i-th record in the file. The output
	is a field or multi-field file format.

    @itemx -branch
        Output on stdout in @file{.branch} format.
	This is the default.

@toindex gnuplot
@toindex plotmtv
@toindex paraview
@toindex vtk

    @itemx -paraview
        Generate a collection of @code{vtk} files
        for using @code{paraview}.

    @itemx -vtk
        Generate a single @code{vtk} file
        with numbered fields.

    @itemx -gnuplot
        Run 1d animation using @code{gnuplot}.

    @itemx -plotmtv
	This driver is unsupported for animations.

    @end table

OTHER OPTIONS:
    @table @code

    @itemx -umin @var{float}
    @itemx -umax @var{float}
	set the solution range for the @code{gnuplot} driver.
 	By default this range is computed from the first field
        of the branch, and this could be problematic when this
        field is initialy zero.

@cindex topography
    @itemx -topography @var{filename}[.field[.gz]]
        performs a tridimensionnal elevation view based
	on the topographic data.

@cindex projection
@apindex P0
@apindex P1
    @itemx -proj
	performs a @code{P1} projection on the fly.
	This option is useful when rendering @code{P0} data
	while @code{vtk} render requieres @code{P1} description.

@cindex elevation
    @itemx -elevation
        For  two  dimensional  field, represent values as elevation in the third dimension.
	This is the default.

    @itemx -noelevation
        Prevent from the elevation representation. 

    @itemx -scale @var{float}
        applies a multiplicative factor to the field.
        This is useful e.g. in conjonction with the @code{elevation} option.
        The default value is 1.

    @itemx -verbose
        print messages related to graphic files created and
       command system calls (this is the default).

    @itemx -noverbose
        does not print previous messages.

    @itemx -clean
        clear temporary graphic files (this is the default).
   
    @itemx -noclean
        does not clear temporary graphic files.

    @itemx -execute
        execute graphic command (this is the default).
   
    @itemx -noexecute
        does not execute graphic command. Generates only
        graphic files. This is usefull in conjuction with the
        @code{-noclean} command.

    @end table

BRANCH FILE FORMAT:

@pindex field
@pindex mfield
@fiindex @file{.field} field 

   The @file{.branch} file format bases on the @file{.field} one:
   @example
       EXAMPLE          GENERAL FORM

        #!branch        #!branch
        branch          branch
          1 1 11        <version> <nfield=1> <nvalue=N>
          time u        <key> <field name>

          #time 3.14    #<key> <key value 1>
	  #u		#<field name>
          field         <field 1>
          .....         ....

          .....         ....
          #time 6.28    #<key> <key value N>
	  #u		#<field name>
          field         <field N>
          .....         ....
   @end example
   The key name is here @code{time}, but could be any string (without spaces).
   The previous example contains one @code{field} at each time step.
   Labels appears all along the file to facilitate direct jumps and field and step skips.

   The format supports several fields, such as (t,u(t),p(t)), where u could
   be a multi-component (e.g. a vector) field:
   @example
        #!branch
        branch
          1 2 11
          time u p

          #time 3.14
	  #u
          mfield
	  1 2
	  #u0
          field
	  ...
	  #u1
          field
	  ...
	  #p

          #time 6.28
	  ...
   @end example

END:*/
// -------------------------------------------------------------
// program
// -------------------------------------------------------------
#include <rheolef.h>
#include <rheolef/iofem.h>
using namespace rheolef;
using namespace std;

void usage()
{
      cerr << "branch: usage: branch"
	   << " [-ndigit int]"
	   << " {-Igeodir}*"
	   << " [-extract int]"
	   << " [-[no]elevation]"
           << " [-scale float]"
           << " [-umin float]"
           << " [-umax float]"
	   << " [-topography filename]"
	   << " [-proj]"
	   << " [-paraview|-vtk|-gnuplot|-plotmtv]"
	   << " [-[no]verbose|-[no]clean|-[no]execute]"
	   << " -|filename"
	   << endl;
      exit (1);
}
typedef field::size_type size_type;

void
extract (idiststream& in, odiststream& out, bool do_proj, size_type extract_id,const Float& scale_value)
{
    branch_basic<Float,sequential> event;
    in  >> event.header();
    out << event.header();
    size_t i = 0;
    do {
        in >> catchmark (event.parameter_name());
	i++;
    } while (i <= extract_id);
    if (!in) return;

    Float t;
    in >> t;
    if (!in) return;
    event.set_parameter (t);
    for (size_t i = 0; in && i < event.size(); i++) {
       in >> catchmark (event[i].first) >> event[i].second;
    }
    field_basic<Float,sequential> u = event[0].second;
    if (do_proj) {
	    fatal_macro ("-proj: not yet -- sorry");
#ifdef TODO
    	    space_basic<Float,sequential> V = u.get_space();
    	    space_basic<Float,sequential> V1 (u.get_geo(), "P1");
    	    field_basic<Float,sequential> one (V1, 1);
    	    form_basic<Float,sequential> proj (V, V1, "mass");
    	    form_diag_basic<Float,sequential> m (V1, "mass");
    	    field_basic<Float,sequential> u1(V1);
    	    u1.u = (1./m.uu)*(proj.uu*u.u);
	    u = u1;
#endif // TODO
    }
    if (scale_value != Float(1)) {
      u *= scale_value;
    }
    out << event(t,u);
    out << event.finalize();
}
void
put (
	idiststream&                  in,
	odiststream&                  out,
	bool                          do_proj,
	size_type                     extract_id,
	const Float&                  scale_value,
	const std::pair<Float,Float>& u_range)
{
    if (extract_id != numeric_limits<size_type>::max()) {
      extract (in, out, do_proj, extract_id, scale_value);
      return;
    }
    Float t = 0;
    field_basic<Float,sequential> u;
    branch_basic<Float,sequential> event;
    event.set_range (u_range);
    size_type n = 0;
    in  >> event.header();
    out << event.header();
    while (in >> event) {
      for (size_t i = 0; i < event.size(); i++) {
        u = event[i].second;
	if (do_proj) {
	    fatal_macro ("-proj: not yet -- sorry");
#ifdef TODO
    	    space_basic<Float,sequential> V = u.get_space();
    	    size_t n_comp = u.n_component();
    	    space_basic<Float,sequential> V_new_i (u.get_geo(), "P1");
	    space_basic<Float,sequential> V_new   = pow(V_new_i, n_comp);
    	    field_basic<Float,sequential> u_new(V_new);
    	    field_basic<Float,sequential> one (V_new_i, 1);
	    for (size_t i = 0; i < n_comp; i++) {
    	      space_basic<Float,sequential> Vi = V[i];
              form_basic<Float,sequential> proj (Vi, V_new_i, "mass");
    	      form_diag_basic<Float,sequential> m (V_new_i, "mass");
    	      field_basic<Float,sequential> u_new_i(V_new_i);
    	      u_new_i.u = (1./m.uu)*(proj.uu*u[i].u);
	      u_new[i] = u_new_i;
	    }
	    u = u_new;
#endif // TODO
	}
        if (scale_value != Float(1)) {
          u *= scale_value;
        }
	event[i].second = u;
      }
      out << event;
      n++;
    }
    out << event.finalize();
}
int main(int argc, char**argv)
{
    environment distributed(argc, argv);
    if (argc <= 1) usage();
    clog << verbose;
    dout.os() << elevation;
    bool on_stdin = false;
    bool do_proj  = false;
    int digits10 = numeric_limits<Float>::digits10;
    typedef enum { text_render, paraview_render, vtk_render, plotmtv_render, gnuplot_render } render_type;
    render_type render = text_render;
    size_type extract_id = numeric_limits<size_type>::max();
    Float scale_value = 1;
    string file_name;
    std::pair<Float,Float> u_range;
    u_range.first  = std::numeric_limits<Float>::min();
    u_range.second = std::numeric_limits<Float>::max();

    for (int i = 1; i < argc; i++) {

        if      (strcmp (argv[i], "-ndigit") == 0)    { digits10 = atoi(argv[++i]); }
        else if (strcmp (argv[i], "-extract") == 0)   { extract_id = atoi(argv[++i]); }
        else if (strcmp (argv[i], "-branch") == 0)    { render = text_render; }
        else if (strcmp (argv[i], "-paraview") == 0)  { render = paraview_render; dout.os() << paraview; }
        else if (strcmp (argv[i], "-vtk") == 0)       { render = vtk_render; dout.os() << vtk; }
        else if (strcmp (argv[i], "-plotmtv") == 0)   { render = plotmtv_render; dout.os() << plotmtv; }
        else if (strcmp (argv[i], "-gnuplot") == 0)   { render = gnuplot_render; dout.os() << gnuplot; }
        else if (strcmp (argv[i], "-proj") == 0)      { do_proj = true; }
        else if (strcmp (argv[i], "-elevation") == 0) { dout.os() << elevation; }
        else if (strcmp (argv[i], "-noelevation") == 0) { dout.os() << noelevation; }
        else if (strcmp (argv[i], "-umin") == 0)   {
            if (i+1 == argc || !is_float(argv[i+1])) usage();
            u_range.first = to_float (argv[++i]);
        } else if (strcmp (argv[i], "-umax") == 0)   {
            if (i+1 == argc || !is_float(argv[i+1])) usage();
            u_range.second = to_float (argv[++i]);
        } else if (strcmp (argv[i], "-scale") == 0)   {
            if (i+1 == argc || !is_float(argv[i+1])) usage();
            scale_value = to_float (argv[++i]);
        } else if (strcmp (argv[i], "-topography") == 0)   {

            if (i+1 == argc) usage();
	    idiststream zin (argv[++i]);
            field_basic<Float,sequential> z;
            zin >> z;
            dout.os() << settopography(z);
	}
        else if (strcmp (argv[i], "-noclean") == 0)   clog << noclean;
        else if (strcmp (argv[i], "-clean") == 0)     clog << clean;
        else if (strcmp (argv[i], "-noexecute") == 0) clog << noexecute;
        else if (strcmp (argv[i], "-execute") == 0)   clog << execute;
        else if (strcmp (argv[i], "-verbose") == 0)   clog << verbose;
        else if (strcmp (argv[i], "-noverbose") == 0) clog << noverbose;

	else if (argv [i][0] == '-' && argv [i][1] == 'I') {

	    append_dir_to_rheo_path (argv[i]+2);
        }
	else if (strcmp (argv [i], "-") == 0) {
	    
	    on_stdin = true;
            dout.os() << setbasename("output") << reader_on_stdin;
	    file_name = "output";
	}
	else {

            // input on file
            string dir_name = get_dirname(argv[i]);
	    prepend_dir_to_rheo_path (dir_name);
            file_name = get_basename(delete_suffix (delete_suffix (argv[i], "gz"), "branch"));
        }
    }
    if (!on_stdin && file_name == "") {
	cerr << "branch: no input specified" << endl;
	usage();
    }
    dout.os() << setbasename(file_name)
              << setprecision(digits10);

    if (on_stdin) {
        put(din,dout, do_proj, extract_id, scale_value, u_range);
    } else {
        idiststream in (file_name, "branch");
        check_macro(in.good(), "\"" << file_name << "[.branch[.gz]]\" not found.");
        put(in, dout, do_proj, extract_id, scale_value, u_range);
    }
}
