#ifndef _RHEOLEF_GEO_H
#define _RHEOLEF_GEO_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================

#include "rheolef/array.h"
#include "rheolef/polymorphic_array.h"
#include "rheolef/geo_element.h"
#include "rheolef/basic_point.h"
#include "rheolef/point_serialize.h"
#include "rheolef/domain_indirect.h"

namespace rheolef {

// =========================================================================
// point io helpers
// =========================================================================
/// @brief point input helper
template <class T>
struct _point_get {
  typedef typename basic_point<T>::size_type size_type;
  _point_get (size_type d1) : d(d1) {}
  size_type d;
  std::istream& operator() (std::istream& is, basic_point<T>& x) { return x.get (is, d); }
}; 
/// @brief point output helper
template <class T>
struct _point_put {
  typedef typename basic_point<T>::size_type size_type;
  _point_put (size_type d1) : d(d1) {}
  size_type d;
  std::ostream& operator() (std::ostream& os, const basic_point<T>& x) { return x.put (os, d); }
};

// forward declaration:
template <class T, class M> class geo_basic;
template <class T, class M> class geo_domain_indirect_rep;

// =========================================================================
/// @brief abstract interface class shared by geo_base_rep and domain_indirect_rep classes and used by space_rep
// =========================================================================
template <class T>
class geo_abstract_base_rep {
public:

// typedefs

    typedef enum { 
	geo                 = 0, 
	geo_domain          = 1, 
	geo_domain_indirect = 2, 
	max_variant         = 3 
    } variant_type; 

    typedef typename polymorphic_array<geo_element,sequential>::size_type       size_type;
    typedef typename polymorphic_array<geo_element,sequential>::iterator        iterator;
    typedef typename polymorphic_array<geo_element,sequential>::const_iterator  const_iterator;
    typedef typename polymorphic_array<geo_element,sequential>::reference       reference;
    typedef typename polymorphic_array<geo_element,sequential>::const_reference const_reference;
    typedef basic_point<T>                                                      vertex_type;

// allocators:

    geo_abstract_base_rep () {}
    virtual ~geo_abstract_base_rep () {}

// abstract accessors:

    virtual size_type variant() const = 0;
    virtual std::string name() const = 0;
    virtual size_type dimension() const = 0;
    virtual size_type map_dimension() const = 0;
    virtual const size_type* size_by_dimension() const = 0;
    virtual const size_type* size_by_variant() const = 0;
    virtual const size_type* dis_size_by_dimension() const = 0;
    virtual const size_type* dis_size_by_variant() const = 0;
    virtual const size_type* ios_size_by_dimension() const = 0;
    virtual const size_type* ios_size_by_variant() const = 0;

    virtual const distributor& geo_element_ownership (size_type dim) const = 0;
    virtual const_reference get_geo_element (size_type dim, size_type ige) const = 0;
    virtual const_iterator begin (size_type dim) const = 0;
    virtual const_iterator end   (size_type dim) const = 0;

    virtual const basic_point<T>&  vertex  (size_type iv) const = 0;
    virtual size_type n_domain_indirect () const = 0;
    
// abstract i/o:

    virtual odiststream& put (odiststream& ops) const = 0;

// deduced comparator:

    bool operator== (const geo_abstract_base_rep<T>& omega2) const {
	return name() == omega2.name(); }
};
template <class T, class M>
class geo_abstract_rep {};

template <class T>
class geo_abstract_rep<T,sequential> : public geo_abstract_base_rep<T> {
public:

// typedefs

    typedef geo_abstract_base_rep<T>       base;
    typedef typename base::size_type       size_type;

// allocators:

    geo_abstract_rep () {}
    virtual ~geo_abstract_rep () {}

// abstract accessors:

    virtual const domain_indirect_basic<sequential>& get_domain_indirect (size_type i) const = 0;
    virtual const domain_indirect_basic<sequential>& get_domain_indirect (const std::string& name) const = 0;
};
#ifdef _RHEOLEF_HAVE_MPI
template <class T>
class geo_abstract_rep<T,distributed> : public geo_abstract_base_rep<T> {
public:

// typedefs

    typedef geo_abstract_base_rep<T>       base;
    typedef typename base::size_type       size_type;
    typedef typename base::vertex_type     vertex_type;
    typedef std::map <size_type, vertex_type, std::less<size_type>,
            heap_allocator<std::pair<size_type,vertex_type> > >    vertex_map_type;

// allocators:

    geo_abstract_rep () {}
    virtual ~geo_abstract_rep () {}

// abstract accessors:

    virtual const vertex_type& dis_vertex (size_type dis_iv) const = 0;

    virtual distributor geo_element_ios_ownership (size_type dim) const = 0;
    virtual size_type             ige2ios_dis_ige (size_type dim, size_type     ige) const = 0;
    virtual size_type         dis_ige2ios_dis_ige (size_type dim, size_type dis_ige) const = 0;
    virtual size_type             ios_ige2dis_ige (size_type dim, size_type ios_ige) const = 0;

    virtual const domain_indirect_basic<distributed>& get_domain_indirect (size_type i) const = 0;
    virtual const domain_indirect_basic<distributed>& get_domain_indirect (const std::string& name) const = 0;
};
#endif // _RHEOLEF_HAVE_MPI
// =========================================================================
/// @brief base class for M=sequential or distributed meshes representations
// =========================================================================
// NOTE: since geo_rep<seq> contains sequential arrays for vertices and elts,
//  the geo_rep<mpi> cannot derive from geo_rep<seq>. The solution is to
//  derive both geo_rep<seq> and geo_rep<mpi> classes from a generic base class
//  named geo_base_rep that takes the memory model (seq or mpi) as template argument.
template <class T, class M>
class geo_base_rep : public geo_abstract_rep<T,M> {
public:
// typedefs:

    typedef geo_abstract_rep<T,M>          base;
    typedef typename base::size_type       size_type;
    typedef typename base::vertex_type     vertex_type;
    typedef typename base::const_iterator  const_iterator;
    typedef typename base::reference       reference;
    typedef typename base::const_reference const_reference;

// allocators:

    geo_base_rep ();
    geo_base_rep (const geo_base_rep<T,M>&);
    ~geo_base_rep ();

// abstract accessors defined:
  
    size_type variant() const { return geo_abstract_base_rep<T>::geo; }
    std::string name() const { return _name; }
    size_type     dimension() const { return _dimension; }
    size_type map_dimension() const { return _map_dimension; }
    const size_type* size_by_dimension()     const { return _size_by_dimension; }
    const size_type* size_by_variant()       const { return _size_by_variant; }
    const size_type* dis_size_by_dimension() const { return _size_by_dimension; }
    const size_type* dis_size_by_variant()   const { return _size_by_variant; }
    const size_type* ios_size_by_dimension() const { return _size_by_dimension; }
    const size_type* ios_size_by_variant()   const { return _size_by_variant; }
    const distributor& geo_element_ownership (size_type dim) const { return _geo_element[dim].ownership(); }
    const_reference get_geo_element (size_type dim, size_type ige) const { return _geo_element[dim][ige]; }
    const_iterator begin (size_type dim) const { return _geo_element[dim].begin(); }
    const_iterator end   (size_type dim) const { return _geo_element[dim].end(); }
    const basic_point<T>&  vertex  (size_type iv) const { return _vertex[iv]; }
    size_type n_domain_indirect () const { return _domains.size(); }
    const domain_indirect_basic<M>& get_domain_indirect (size_type i) const { return _domains[i]; }
    const domain_indirect_basic<M>& get_domain_indirect (const std::string& name) const;

// additional accessors & modifier:

    reference get_geo_element (size_type dim, size_type ige) { return _geo_element[dim][ige]; }

// deduced accessors:

    size_type     size(size_type dim) const { return geo_element_ownership (dim).size(); }
    size_type dis_size(size_type dim) const { return geo_element_ownership (dim).dis_size(); }
    const distributor&        ownership() const { return geo_element_ownership (map_dimension()); }
    const distributor& vertex_ownership() const { return geo_element_ownership (0); }
    const communicator&            comm() const { return ownership().comm(); }

    size_type      n_vertex() const { return     size (0); }
    size_type          size() const { return     size (map_dimension()); }
    size_type  dis_n_vertex() const { return dis_size (0); }
    size_type    dis_n_edge() const { return dis_size (1); }
    size_type    dis_n_face() const { return dis_size (2); }
    size_type      dis_size() const { return dis_size (map_dimension()); }

    const_reference operator[] (size_type ie) const { return get_geo_element (map_dimension(), ie); }
    reference  	    operator[] (size_type ie)	    { return get_geo_element (map_dimension(), ie); }

    const_iterator begin_edge() const { return begin (1); }
    const_iterator end_edge()   const { return end   (1); }
    const_iterator begin_face() const { return begin (2); }
    const_iterator end_face()   const { return end   (2); }
    const_iterator begin()      const { return begin (map_dimension()); }
    const_iterator end()        const { return end   (map_dimension()); }

protected:
// internal:
    void reset_size_by();
// data:
    polymorphic_array<geo_element,M>         _geo_element [4];
    array<vertex_type, M>                    _vertex;
    std::vector<domain_indirect_basic<M> >   _domains;
    std::string                              _name;
    size_type                                _version;
    size_type                                _dimension;
    size_type                                _map_dimension;
    size_type                                _size_by_dimension [4]; // counters by geo_element dimension: 0,1,2,3
    size_type                                _size_by_variant   [reference_element::max_size]; // by type: t,q...
};
template <class T, class M>
inline
geo_base_rep<T,M>::geo_base_rep ()
 : geo_abstract_rep<T,M>(),
   _geo_element(),
   _vertex(),
   _domains(),
   _name("*nogeo*"),
   _version(0),
   _dimension(0),
   _map_dimension(0),
   _size_by_dimension(),
   _size_by_variant()
{
}
// =========================================================================
/// @brief sequential mesh representation
// =========================================================================
template <class T, class M> class geo_rep {};

template <class T>
class geo_rep<T,sequential> : public geo_base_rep<T,sequential> {
public:
// typedefs:

    typedef geo_base_rep<T,sequential>     base;
    typedef typename base::size_type       size_type;
    typedef typename base::vertex_type     vertex_type;
    typedef typename base::reference       reference;
    typedef typename base::const_reference const_reference;
    typedef typename base::const_iterator  const_iterator;

// allocators:

    geo_rep();
    void build_from_domain (const domain_indirect_rep<sequential>& indirect, const geo_abstract_rep<T,sequential>& omega);

// herited accessors:

    size_type map_dimension ()     const { return base::map_dimension(); }
    const distributor& geo_element_ownership(size_type dim) const { return base::geo_element_ownership (dim); }
    const_reference get_geo_element (size_type dim, size_type ige) const { return base::get_geo_element (dim, ige); }
    reference       get_geo_element (size_type dim, size_type ige)       { return base::get_geo_element (dim, ige); }
    const_iterator begin (size_type dim) const { return base::begin(dim); }
    const_iterator end   (size_type dim) const { return base::end  (dim); }
    const vertex_type& dis_vertex (size_type dis_iv) const { return base::vertex (dis_iv); }
    const domain_indirect_basic<sequential>& get_domain_indirect (const std::string& name) const { return base::get_domain_indirect (name); }

// deduced accessors:

    const distributor& vertex_ownership()        const { return geo_element_ownership(0); }
    const_reference operator[] (size_type ie) const { return get_geo_element (map_dimension(), ie); }
    reference       operator[] (size_type ie)       { return get_geo_element (map_dimension(), ie); }

// i/o:

    idiststream& get (idiststream&);
    odiststream& put (odiststream&) const;
    void dump (std::string name) const;
    void load (std::string name);
};
template <class T>
inline
geo_rep<T,sequential>::geo_rep ()
 : geo_base_rep<T,sequential>()
{
}

#ifdef _RHEOLEF_HAVE_MPI
// =========================================================================
/// @brief distributed mesh representation
// =========================================================================
template <class T>
class geo_rep<T,distributed> : public geo_base_rep<T,distributed> {
public:
// typedefs:

    typedef geo_base_rep<T,distributed>    base;
    typedef typename base::size_type       size_type;
    typedef typename base::vertex_type     vertex_type;
    typedef typename base::vertex_map_type vertex_map_type;
    typedef typename base::reference       reference;
    typedef typename base::const_reference const_reference;
    typedef typename base::const_iterator  const_iterator;

// allocators:

    geo_rep ();
    void build_from_domain (const domain_indirect_rep<distributed>& indirect, const geo_abstract_rep<T,distributed>& omega);

// abstract accessors defined:

    const vertex_type& dis_vertex (size_type dis_iv) const;
    distributor geo_element_ios_ownership (size_type dim) const;
    size_type             ige2ios_dis_ige (size_type dim, size_type ige) const;
    size_type         dis_ige2ios_dis_ige (size_type dim, size_type dis_ige) const;
    size_type             ios_ige2dis_ige (size_type dim, size_type ios_ige) const;
    const size_type* dis_size_by_dimension() const { return _dis_size_by_dimension; }
    const size_type* dis_size_by_variant()   const { return _dis_size_by_variant; }
    const size_type* ios_size_by_dimension() const { return _ios_size_by_dimension; }
    const size_type* ios_size_by_variant()   const { return _ios_size_by_variant; }

// herited accessors:

    size_type map_dimension ()     const { return base::map_dimension(); }
    size_type size (size_type dim) const { return base::size(dim); }
    const distributor& geo_element_ownership(size_type dim) const { return base::geo_element_ownership (dim); }
    const_reference get_geo_element (size_type dim, size_type ige) const { return base::get_geo_element (dim, ige); }
    reference       get_geo_element (size_type dim, size_type ige)       { return base::get_geo_element (dim, ige); }
    const_iterator begin (size_type dim) const { return base::begin(dim); }
    const_iterator end   (size_type dim) const { return base::end  (dim); }
    const domain_indirect_basic<distributed>& get_domain_indirect (const std::string& name) const { return base::get_domain_indirect (name); }

// deduced accessors:

    size_type          size ()                   const { return size (map_dimension()); }
    const distributor& vertex_ownership()        const { return geo_element_ownership(0); }
    const_reference operator[] (size_type ie) const { return get_geo_element (map_dimension(), ie); }
	  reference operator[] (size_type ie)       { return get_geo_element (map_dimension(), ie); }

// i/o:

    idiststream& get (idiststream&);
    odiststream& put (odiststream&) const;
    void dump (std::string name) const;
    void load (std::string name, const communicator& comm);

protected:
// internal:
    void build_external_entities ();
    void reset_size_by();
    void set_element_edge_index();
    void set_element_face_index();
// data:
    array<size_type>             _ios_ige2dis_ige[4]; // reverse permutation for geo_element[dim]
    polymorphic_map<geo_element> _ext_geo_element_0; // external geo_element[0], at part bdry
    vertex_map_type              _ext_vertex;        // external vertex, at partition bdry
    size_type                    _dis_size_by_dimension [4]; // geo_elements global counters by dimension: 0,1,2,3
    size_type                    _dis_size_by_variant [reference_element::max_size]; // by type: t,q...
    size_type                    _ios_size_by_dimension [4]; // i/o geo_elements local counters by dimension: 0,1,2,3
    size_type                    _ios_size_by_variant [reference_element::max_size]; // by type: t,q...
};
template <class T>
inline
geo_rep<T,distributed>::geo_rep ()
 : geo_base_rep<T,distributed>(),
   _ios_ige2dis_ige(),
   _ext_geo_element_0(),
   _ext_vertex(),
   _dis_size_by_dimension(),
   _dis_size_by_variant(),
   _ios_size_by_dimension(),
   _ios_size_by_variant()
{
}
#endif // _RHEOLEF_HAVE_MPI

// =========================================================================
/// @brief generic mesh with rerefence counting
// =========================================================================
template <class T, class M = rheo_default_memory_model>
class geo_basic {
public:
    typedef M memory_type;
};
/*Class:geo
NAME:  geo - finite element mesh (@PACKAGE@-@VERSION@)
SYNOPSYS:       
 Distributed finite element mesh.
SEE ALSO: "geo_element"(3)
AUTHORS: Pierre.Saramito@imag.fr
DATE:   10 december 2010
End:
*/
// =========================================================================
/// @brief sequential mesh with reference counting
// =========================================================================
//<verbatim:
template <class T>
class geo_basic<T,sequential> : public smart_pointer<geo_abstract_rep<T,sequential> > {
public:

// typedefs:

    typedef sequential                     memory_type;
    typedef geo_abstract_rep<T,sequential> rep;
    typedef smart_pointer<rep>             base;
    typedef typename rep::size_type        size_type;
    typedef typename rep::vertex_type      vertex_type;
    typedef typename rep::reference        reference;
    typedef typename rep::const_reference  const_reference;
    typedef typename rep::const_iterator   const_iterator;

// allocators:

    geo_basic ();
    geo_basic (std::string name, const communicator& comm = communicator());
    void load (std::string name, const communicator& comm = communicator());
    geo_basic (const domain_indirect_basic<sequential>& dom, const geo_basic<T,sequential>& omega);

// accessors:

    std::string                name() const { return base::data().name(); }
    size_type             dimension() const { return base::data().dimension(); }
    size_type         map_dimension() const { return base::data().map_dimension(); }
    const distributor& geo_element_ownership(size_type dim) const { return base::data().geo_element_ownership(dim); }
    const size_type* size_by_dimension()     const { return base::data().size_by_dimension(); }
    const size_type* size_by_variant()       const { return base::data().size_by_variant(); }
    const size_type* dis_size_by_dimension() const { return base::data().dis_size_by_dimension(); }
    const size_type* dis_size_by_variant()   const { return base::data().dis_size_by_variant(); }
    const size_type* ios_size_by_dimension() const { return base::data().ios_size_by_dimension(); }
    const size_type* ios_size_by_variant()   const { return base::data().ios_size_by_variant(); }
    const_reference get_geo_element (size_type dim, size_type ige) const { return base::data().get_geo_element (dim, ige); }
          reference get_geo_element (size_type dim, size_type ige)       { return base::data().get_geo_element (dim, ige); }
    const vertex_type&     vertex (size_type iv) const { return base::data().vertex(iv); }
    const vertex_type& dis_vertex (size_type iv) const { return vertex(iv); }

    size_type n_domain_indirect () const { return base::data().n_domain_indirect (); }
    const domain_indirect_basic<sequential>& get_domain_indirect (size_type i) const {
	  return base::data().get_domain_indirect (i); }
    const domain_indirect_basic<sequential>& get_domain_indirect (const std::string& name) const {
	  return base::data().get_domain_indirect (name); }

    size_type n_domain () const { return base::data().n_domain_indirect (); }
    geo_basic<T,sequential> get_domain (size_type i) const;
    geo_basic<T,sequential> operator[] (const std::string& name) const;

// extended accessors:

    const communicator& comm()        const { return geo_element_ownership (0).comm(); }
    size_type     size(size_type dim) const { return base::data().geo_element_ownership(dim).size(); }
    size_type dis_size(size_type dim) const { return base::data().geo_element_ownership(dim).dis_size(); }
    size_type     size()              const { return size     (map_dimension()); }
    size_type dis_size()              const { return dis_size (map_dimension()); }
    size_type     n_vertex()          const { return size     (0); }
    size_type dis_n_vertex()          const { return dis_size (0); }
    const_reference operator[] (size_type ie) const { return get_geo_element (map_dimension(), ie); }
	  reference operator[] (size_type ie)       { return get_geo_element (map_dimension(), ie); }
    const_iterator begin (size_type dim) const { return base::data().begin(dim); }
    const_iterator end   (size_type dim) const { return base::data().end  (dim); }
    const_iterator begin () const { return begin(map_dimension()); }
    const_iterator end   () const { return end  (map_dimension()); }
    size_type dis_ige2ios_dis_ige (size_type dim, size_type dis_ige) const { return dis_ige; }

    const geo_basic<T,sequential>& get_background_geo() const; // code in geo_domain.h
          geo_basic<T,sequential>  get_background_domain() const;

// comparator:

    bool operator== (const geo_basic<T,sequential>& omega2) const { return base::data().operator== (omega2.data()); }

// i/o:

    idiststream& get (idiststream& ips);
    odiststream& put (odiststream& ops) const { return base::data().put (ops); }
    void dump (std::string name) const      {        base::data().dump (name); }
};
//>verbatim:
template <class T>
inline
geo_basic<T,sequential>::geo_basic()
  : base (new_macro((geo_rep<T,sequential>)))
{
}
template <class T>
inline
geo_basic<T,sequential>::geo_basic (std::string name, const communicator& comm)
  : base (0)
{
    // allocate a new geo_rep<mpi> object ; TODO geo_rep(name,comm)
    geo_rep<T,sequential>* ptr = new_macro((geo_rep<T,sequential>));
    ptr->load (name);
    base::operator= (ptr);
}
template <class T>
inline
void
geo_basic<T,sequential>::load (std::string name, const communicator& comm)
{
    // allocate a new geo_rep<mpi> object ; TODO geo_rep(name,comm)
    geo_rep<T,sequential>* ptr = new_macro((geo_rep<T,sequential>));
    ptr->load (name);
    base::operator= (ptr);
}
template <class T>
inline
idiststream&
geo_basic<T,sequential>::get (idiststream& ips)
{
    // allocate a new geo_rep object (TODO: do a dynamic_cast ?)
    geo_rep<T,sequential>* ptr = new_macro((geo_rep<T,sequential>));
    ptr->get (ips);
    base::operator= (ptr);
    return ips;
}
#ifdef _RHEOLEF_HAVE_MPI
// =========================================================================
/// @brief distributed mesh with rerefence counting
// =========================================================================
//<verbatim:
template <class T>
class geo_basic<T,distributed> : public smart_pointer<geo_abstract_rep<T,distributed> > {
public:

// typedefs:

    typedef distributed                             memory_type;
    typedef geo_abstract_rep<T,distributed>         rep;
    typedef smart_pointer<rep>                      base;
    typedef typename rep::size_type                 size_type;
    typedef typename rep::vertex_type               vertex_type;
    typedef typename rep::vertex_map_type           vertex_map_type;
    typedef typename rep::reference                 reference;
    typedef typename rep::const_reference           const_reference;
    typedef typename rep::const_iterator            const_iterator;

// allocators:

    geo_basic ();
    geo_basic (std::string name, const communicator& comm = communicator());
    void load (std::string name, const communicator& comm = communicator());
    geo_basic (const domain_indirect_basic<distributed>& dom, const geo_basic<T,distributed>& omega);

// accessors:

    std::string                name() const { return base::data().name(); }
    size_type             dimension() const { return base::data().dimension(); }
    size_type         map_dimension() const { return base::data().map_dimension(); }
    const distributor& geo_element_ownership(size_type dim) const
			{ return base::data().geo_element_ownership (dim); }
    const size_type* size_by_dimension()     const { return base::data().size_by_dimension(); }
    const size_type* size_by_variant()       const { return base::data().size_by_variant(); }
    const size_type* dis_size_by_dimension() const { return base::data().dis_size_by_dimension(); }
    const size_type* dis_size_by_variant()   const { return base::data().dis_size_by_variant(); }
    const size_type* ios_size_by_dimension() const { return base::data().ios_size_by_dimension(); }
    const size_type* ios_size_by_variant()   const { return base::data().ios_size_by_variant(); }
    const_reference get_geo_element (size_type dim, size_type ige) const
		{ return base::data().get_geo_element (dim, ige); }
    distributor geo_element_ios_ownership (size_type dim) const {
	return base::data().geo_element_ios_ownership (dim); }
    size_type ige2ios_dis_ige (size_type dim, size_type ige) const {
	return base::data().ige2ios_dis_ige (dim,ige); }
    size_type dis_ige2ios_dis_ige (size_type dim, size_type dis_ige) const {
	return base::data().dis_ige2ios_dis_ige (dim,dis_ige); }
    size_type ios_ige2dis_ige (size_type dim, size_type ios_ige) const {
    	return base::data().ios_ige2dis_ige (dim, ios_ige); }
    const vertex_type&      vertex(size_type     iv) const { return base::data().vertex(iv); }
    const vertex_type& dis_vertex (size_type dis_iv) const { return base::data().dis_vertex(dis_iv); }

    size_type n_domain_indirect () const { return base::data().n_domain_indirect (); }
    const domain_indirect_basic<distributed>& get_domain_indirect (size_type i) const {
	  return base::data().get_domain_indirect (i); }
    const domain_indirect_basic<distributed>& get_domain_indirect (const std::string& name) const {
	  return base::data().get_domain_indirect (name); }

    size_type n_domain () const { return base::data().n_domain_indirect (); }
    geo_basic<T,distributed> get_domain (size_type i) const;
    geo_basic<T,distributed> operator[] (const std::string& name) const;

// extended accessors:

    size_type     size(size_type dim) const { return base::data().geo_element_ownership(dim).size(); }
    size_type dis_size(size_type dim) const { return base::data().geo_element_ownership(dim).dis_size(); }
    const communicator& comm()        const { return geo_element_ownership (0).comm(); }
    size_type     size()              const { return size     (map_dimension()); }
    size_type dis_size()              const { return dis_size (map_dimension()); }
    const_reference operator[] (size_type ie) const	
		{ return get_geo_element (map_dimension(), ie); }
    const_iterator begin (size_type dim) const { return base::data().begin(dim); }
    const_iterator end   (size_type dim) const { return base::data().end  (dim); }
    const_iterator begin () const { return begin(map_dimension()); }
    const_iterator end   () const { return end  (map_dimension()); }

    const geo_basic<T,distributed>& get_background_geo() const; // code in geo_domain.h
          geo_basic<T,distributed>  get_background_domain() const;

// comparator:

    bool operator== (const geo_basic<T,distributed>& omega2) const { return base::data().operator== (omega2.data()); }

// i/o:

    odiststream& put (odiststream& ops) const { return base::data().put (ops); }
    idiststream& get (idiststream& ips);
};
//>verbatim:
#endif // _RHEOLEF_HAVE_MPI

/// @brief geo - the default mesh class
typedef geo_basic<Float,rheo_default_memory_model> geo;

// ==============================================================================
// inlined: geo<T,distributed>
// ==============================================================================
#ifdef _RHEOLEF_HAVE_MPI
template <class T>
inline
geo_basic<T,distributed>::geo_basic()
  : base (new_macro((geo_rep<T,distributed>)))
{
}
template <class T>
inline
geo_basic<T,distributed>::geo_basic (std::string name, const communicator& comm)
  : base (0)
{
    // allocate a new geo_rep<mpi> object ; TODO geo_rep(name,comm)
    geo_rep<T,distributed>* ptr = new_macro((geo_rep<T,distributed>));
    ptr->load (name, comm);
    base::operator= (ptr);
}
template <class T>
inline
void
geo_basic<T,distributed>::load (std::string name, const communicator& comm)
{
    // allocate a new geo_rep<mpi> object ; TODO geo_rep(name,comm)
    geo_rep<T,distributed>* ptr = new_macro((geo_rep<T,distributed>));
    ptr->load (name, comm);
    base::operator= (ptr);
}
template <class T>
inline
idiststream&
geo_basic<T,distributed>::get (idiststream& ips)
{
    // allocate a new geo_rep object (TODO: do a dynamic_cast ?)
    geo_rep<T,distributed>* ptr = new_macro((geo_rep<T,distributed>));
    ptr->get (ips);
    base::operator= (ptr);
    return ips;
}
#endif // _RHEOLEF_HAVE_MPI

// ==============================================================================
// inlined: geo<T,M>
// ==============================================================================
template <class T, class M>
inline 
idiststream&
operator>> (idiststream& ips, geo_basic<T,M>& omega)
{
    return omega.get (ips);
}
template <class T, class M>
inline 
odiststream&
operator<< (odiststream& ops, const geo_basic<T,M>& omega)
{
    return omega.put (ops);
}

} // namespace rheolef
#endif // _RHEOLEF_GEO_H
