///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
// 
// Subdivision tree on a mesh with grid stump 
// and octree-like embranching. 
//
// author: jocelyn.etienne@imag.fr
//
// date: 15 Feb 2002
// 
#include "rheolef/hazel.h"
using namespace rheolef;
using namespace std;

//--- hazel_stump implementation ------------------------------------

hazel_stump::hazel_stump (int level, int* i_stump, int* i_bough)
 {
     level_ =level;
    for (int i=0; i<3; i++)
     {
         i_stump_[i] = i_stump[i]; 
        i_bough_[i] = i_bough[i]; 
     }
     n_interior_elements_=0;
     n_boundary_elements_=0;
 };

hazel_stump::~hazel_stump ()
 {
 };
    
void 
hazel_stump::add_interior_element (size_type element_index)
 {
    interior_element_list.insert(interior_element_list.end(), element_index);
    n_interior_elements_++;
 };

void 
hazel_stump::add_boundary_element (size_type element_index)
 {
    boundary_element_list.insert(boundary_element_list.end(), element_index);
    n_boundary_elements_++;
 };

hazel_stump::size_type
hazel_stump::pop_interior_element ()
 {
    n_interior_elements_--;
    size_type e = *interior_element_list.begin();
    interior_element_list.pop_front();
    return e;
 };


hazel_stump::size_type
hazel_stump::pop_boundary_element ()
 {
    n_boundary_elements_--;
    size_type e = *boundary_element_list.begin();
    boundary_element_list.pop_front();
    return e;
 };


//--- hazel_bough implementation ------------------------------------


void
hazel_bough::set_neighbours (hazel_node* raw_neighbours[6], 
                const int squirrel_leap[3])
 {
    int i;
    for (i=0; i<6; i++) neighbour_[i] =raw_neighbours[i];
    for (i=0; i<3; i++) squirrel_leap_[i] =squirrel_leap[i];
 };
    
void
hazel_bough::identify_neighbours ()
 {
    error_macro ("hazel: identify_neighbours not implemented yet.");
 };

hazel_node**
hazel_bough::get_neighbours ()
 {
    return neighbour_; 
 }

//--- hazel implementation ------------------------------------------

hazel::hazel ()
 { grown =false; };

void
hazel::grow     
     (
        vector<geo_element>::const_iterator begin_element,
        vector<geo_element>::const_iterator end_element,
        const vector<point>::const_iterator begin_node,
        const vector<point>::const_iterator end_node,
        vector<geo_element>::const_iterator begin_boundary,
        vector<geo_element>::const_iterator end_boundary,
        const int mesh_size,
        const int space_dimension,
        const point& x_min, 
        const point& x_max,
        Float tolerance,
        const size_type elements_per_leaf, 
        const size_type max_depth
     )
 {
     if (!grown)
    {
    tol = (tolerance < Float(0)) ? ::sqrt(numeric_limits<Float>::epsilon()) : tolerance;
    dimension = space_dimension;

    xmin =x_min -point(.1, .1, (dimension>=3)?.1:0 );
    xmax =x_max +point(.1, .1, (dimension>=3)?.1:0 );
    begin_element_    =    begin_element; 
    end_element_    =    end_element;
    begin_boundary_    =    begin_boundary;
    end_boundary_    =    end_boundary; 
    
    begin_node_        =    begin_node;    

    // Determine the number of stumps (if not given)
    size_type grid_size;
    int el =elements_per_leaf;
    if (elements_per_leaf<1)
        el = int(pow(float(4),dimension));
    grid_size = int(ceil(Float(mesh_size /el
        *(1+Float(1<<dimension)/pow(el,1./dimension)))));

    // Determine the number of stumps in each direction of the grid
    int i;
    point omega_size =xmax-xmin;
    Float omega_volume =1.;
    for (i=0; i<dimension; i++) omega_volume *=omega_size[i]; 
    Float alpha = pow (Float(grid_size)/omega_volume, Float(1)/dimension);
    for (i=0; i<dimension; i++) 
     {
        n_stump[i] = int(ceil(alpha*omega_size[i]));
         h_stump[i] =(xmax[i]-xmin[i]) /n_stump[i];
     }
    for (i=dimension; i<3; i++)
     {
        n_stump[i] =1;
        h_stump[i] =0;
     }
        
    // Allocate stump boughs
    resize(n_stump[0]*n_stump[1]*n_stump[2]);
    int i_stump[3];
    int i_zero[3]={0,0,0};
    Vector<hazel_node*>::iterator ivec =begin();

    for (i_stump[2]=0; i_stump[2]<n_stump[2]; i_stump[2]++)
     for (i_stump[1]=0; i_stump[1]<n_stump[1]; i_stump[1]++)
      for (i_stump[0]=0; i_stump[0]<n_stump[0]; i_stump[0]++)
         {
             (*ivec) = new hazel_node (0, i_stump, i_zero );
            ivec++;
         }

    // ...prepare neighbours
    ivec = begin();
    int ix,iy,iz;
    hazel_node* neighbours[6]={NULL,NULL,NULL,NULL,NULL,NULL};
    for (iz=0; iz<n_stump[2]; iz++)
     for (iy=0; iy<n_stump[1]; iy++)
      for (ix=0; ix<n_stump[0]; ix++)
         {
            size_type pos =index(ix,iy,iz);
            size_type offset =1;
            for (i=0; i<dimension; i++)
             {
                neighbours[2*i]   =(pos -offset>=0) ?
                                (begin()[pos-offset]) : NULL;
                neighbours[2*i+1] =(pos +offset<size()) ?
                                (begin()[pos+offset]) : NULL;
                offset*=n_stump[i];
             };
            if (ix==0) neighbours[0]=NULL;
            if (ix==n_stump[0]-1) neighbours[1]=NULL;
            if (iy==0) neighbours[2]=NULL;
            if (iy==n_stump[1]-1) neighbours[3]=NULL;
            if (iz==0) neighbours[4]=NULL;
            if (iz==n_stump[2]-1) neighbours[5]=NULL;

            (**ivec).set_neighbours(neighbours, i_zero);
            ivec++;
         };
    
    // Fill in stump boughs interior element lists
    int jmin[3], jmax[3];
    int j, j0, j1, j2;
    size_type s;
    Vector<geo_element>::const_iterator e =begin_element_;
    while (e !=end_element_)
     {
        for (i=0; i<3; i++)
            { jmin[i] =n_stump[i]-1; jmax[i] =0; };

        // Find all intersecting stump-buckets
         for (s=0; s<(*e).size(); s++)
             for (i=0; i<dimension; i++)
             {
                j = int(floor(
                    (begin_node_[(*e)[s]][i] -xmin[i]) 
                        /omega_size[i] *n_stump[i] -tol));     

                // Fix the problem at the boundaries of box
                if (j==-1) j++;
                if (j==n_stump[i]) j--;

                 jmin[i] =(jmin[i]<j)? jmin[i] : j ;

                if (tol != Float(0)) 
                    j = int(floor(
                        (begin_node_[(*e)[s]][i] -xmin[i]) 
                            /omega_size[i] *n_stump[i] +tol));

                // Fix the problem at the boundaries of box
                if (j==-1) j++;
                if (j==n_stump[i]) j--;

                jmax[i] =(jmax[i]>j)? jmax[i] : j ;
             };

        // Add the element to all of them
        for (j2=jmin[2]; j2<=jmax[2]; j2++)
         for (j1=jmin[1]; j1<=jmax[1]; j1++)
          for (j0=jmin[0]; j0<=jmax[0]; j0++)
            {
                 at(index(j0,j1,j2))->add_interior_element ((*e).index());
            }
        e++;
     };

    // Grow hazel in dense areas
    // TO BE MODIFIED : insert here descent method
    
    // Insert boundary elements to the lists
    e = begin_boundary_;
    size_type e_index =0;
    while (e!=end_boundary_)
     {
        for (i=0; i<3; i++)
            { jmin[i] =n_stump[i]-1; jmax[i] =0; };

        // Find all intersecting stump-buckets
         for (s=0; s<(*e).size(); s++)
             for (i=0; i<dimension; i++)
             {
                j = int(floor(
                    (begin_node_[(*e)[s]][i] -xmin[i]) 
                        /omega_size[i] *n_stump[i] -tol));     

                // Fix the problem at the boundaries of box
                if (j==-1) j++;
                if (j==n_stump[i]) j--;

                 jmin[i] =(jmin[i]<j)? jmin[i] : j ;

                if (tol != Float(0)) 
                    j = int(floor(
                        (begin_node_[(*e)[s]][i] -xmin[i]) 
                            /omega_size[i] *n_stump[i] +tol));

                // Fix the problem at the boundaries of box
                if (j==-1) j++;
                if (j==n_stump[i]) j--;

                jmax[i] =(jmax[i]>j)? jmax[i] : j ;
             };

        // Add the element to all of them
        for (j2=jmin[2]; j2<=jmax[2]; j2++)
         for (j1=jmin[1]; j1<=jmax[1]; j1++)
          for (j0=jmin[0]; j0<=jmax[0]; j0++)
            {
                // TO BE MODIFIED : insert here descent method
                 at(index(j0,j1,j2))->add_boundary_element (e_index);
            }
        e++; e_index++;
     }

    grown =true;
 }
 else error_macro ( "Trying to overgrow hazel tree.\n" );
 }

int
hazel::avg_list_size ()
 {
    int n_incl=0;
    int n_nonempty=0;
    for (int i=0; i<n_stump[0]*n_stump[1]*n_stump[2]; i++)
        {
            n_incl+=at(i)->n_interior_elements();
            if (at(i)->n_interior_elements()!=0) n_nonempty++;
        }
    return n_incl/n_nonempty;
 }

hazel::~hazel    ()
 {}

bool
hazel::in_bbox_element (const geo_element& e, const point& x) const
 {
    switch(e.type())
     {
        case geo_element::p:
         {
            error_macro("Wrong element dimension");
         }
        case geo_element::e:
         {
            #ifdef PARANO
            if (dimension!=1) error_macro("Wrong dimension for having 1-d interior elements");
            #endif //PARANO
            if ((begin_node_[e[1]][0] <=x[0]) && (x[0] <=begin_node_[e[1]][0]))
                return true;
            else
                return false;
         }
        case geo_element::t:
         {
            point a =begin_node_[e[0]];
            point b =begin_node_[e[1]];
            point c =begin_node_[e[2]];
            if (x[0]<a[0]-tol && x[0]<b[0]-tol && x[0]<c[0]-tol) return false; 
            if (x[0]>a[0]+tol && x[0]>b[0]+tol && x[0]>c[0]+tol) return false; 
            if (x[1]<a[1]-tol && x[1]<b[1]-tol && x[1]<c[1]-tol) return false; 
            if (x[1]>a[1]+tol && x[1]>b[1]+tol && x[1]>c[1]+tol) return false; 
            return true;
         }
        default:
         error_macro("Not yet implemented");
     }
 }
bool
hazel::in_interior_element (const geo_element& e, const point& x) const
 {
   bool not_too_far;
   return in_interior_element (e,x,not_too_far);
 }

static
bool
in_triangle (const point& a, const point& b, const point& c,
    const point& x, const Float& tol, bool& not_too_far)
{
      not_too_far = false;
      point e_min = point
    (min(min(a[0],b[0]),c[0])-tol,
     min(min(a[1],b[1]),c[1])-tol,    
     0);
      point e_max = point
    (max(max(a[0],b[0]),c[0])+tol,
     max(max(a[1],b[1]),c[1])+tol,    
     0);
      if (e_min[0]<x[0] && x[0]<e_max[0]
    && e_min[1]<x[1] && x[1]<e_max[1])
        not_too_far=true;
      Float t9 = 1/(-b[0]*c[1]+b[0]*a[1]+a[0]*c[1]+c[0]*b[1]-c[0]*a[1]-a[0]*b[1]);
      Float t11 = -a[0]+x[0];
      Float t15 = -a[1]+x[1];
      point xhat((-c[1]+a[1])*t9*t11-(-c[0]+a[0])*t9*t15,
             (b[1]-a[1])*t9*t11-(b[0]-a[0])*t9*t15,
                 0);
      if (xhat[0]<-tol) return false;
      if (xhat[1]<-tol) return false;
      if (xhat[0]+xhat[1]>1+tol) return false;
      return true;
} 
bool
hazel::in_interior_element (const geo_element& e, const point& x, bool& not_too_far) const
{
  not_too_far = false;
  switch(e.type()) {
    case geo_element::p: {
      error_macro("Wrong element dimension");
    }
    case geo_element::e: {
#ifdef PARANO
      if (dimension != 1) 
        error_macro("Wrong dimension for having 1-d interior elements");
#endif //PARANO
      if ((begin_node_[e[1]][0] <= x[0]) && (x[0] <= begin_node_[e[1]][0]))
    return true;
      else
    return false;
    }
    case geo_element::t: {
      const point& a = begin_node_[e[0]];
      const point& b = begin_node_[e[1]];
      const point& c = begin_node_[e[2]];
      return in_triangle (a, b, c, x, tol, not_too_far);
    }
    case geo_element::q: {
      const point& a = begin_node_[e[0]];
      const point& b = begin_node_[e[1]];
      const point& c = begin_node_[e[2]];
      if (in_triangle (a, b, c, x, tol, not_too_far)) return true;
      bool not_too_far2;
      const point& d = begin_node_[e[3]];
      if (in_triangle (c, d, a, x, tol, not_too_far2)) return true;
      not_too_far |= not_too_far2;
      return false;
    }
    default:
      error_macro("element not yet implemented");
  }
}

bool
hazel::hit_boundary_element (const geo_element& e, 
                            const point& x0, const point& v, 
                            point& x, Float& t, bool& inward) const
 {
    point x1 = x0 +v;
    switch(e.type())
     {
        case geo_element::p:
         {
             error_macro("Unable to evaluate inward/outward in 1D, not implemented");
            #ifdef PARANO
            if (dimension!=1) error_macro("Wrong dimension for having 0-d boundaries");
            #endif //PARANO
            x =begin_node_[e[0]];
            point w         =x-x0;
            point w_dash     =x1-x;
            t =dot(v,w);
            if ( (Float(0) < t) && (Float(0) <= dot(v,w_dash)) )
             {
                 t /=dot(v,v);
                return true;
             }
            else
            if (t == Float(0)) return (w==v);
            else return false;
         }
        case geo_element::e:
         {
            #ifdef PARANO
            if (dimension!=2) error_macro("Wrong dimension for having 1-d boundaries");
            #endif //PARANO

            point u = begin_node_[e[1]] -begin_node_[e[0]];
            point w = x0 - begin_node_[e[0]];
            point w_dash = begin_node_[e[1]] - x0;
        
            // Check colinearity
            Float D = vect2d(u,v);
            inward = (D >= Float(0));
        
            if (fabs(D)<tol) //==0) 
             {  // Then check alignment
                 
                return false;
                // We want to slide !!
                
                if ( fabs(vect2d(w,u)) <tol)
                    // Then check overlapping : 
                    // if x0 is in e0e1, it's x0.
                    // if not, it may be e0 (or e1) iff x1 is on the
                    // opposite side of e0 (or e1).
                    if ( (Float(0) <=dot(w,u)) && (dot(w,u) <=dot(u,u)) )
                     {
                        x =x0;
                        t =0;
                        return true;
                     }
                    if (dot(w,w+v) < Float(0)) 
                     {
                         x = begin_node_[e[0]];
                        t = -dot(w,v)/dot(v,v);
                        return true;
                     }
                    if (dot(w_dash,w_dash-v) <= Float(0))
                     {
                        x = begin_node_[e[1]];
                        t = dot(w_dash,v)/dot(v,v);
                        return true;
                     }
             }
            else
            // If not, check that intersection occurs within [s0_s1_] :
                if ( (-tol <=vect2d(w,v)/D) && (-tol <=vect2d(w_dash,v)/D) )
                 {
                    t=vect2d(w,u)/D;
                    x =begin_node_[e[0]] +u *vect2d(w,v)/D;
                    // And that intersection occurs within [x1x0] :
                    if ( Float(0) <=t && t-1 <= Float(0))
                    return true;
                 }
            return false;
          }
        default: error_macro("Not yet implemented");
     }
 }

bool
hazel::hit_boundingbox (const point& x0, const point& v, 
                        const point& xmin, const point& xmax, 
                        Float& t, int& side, int& dir) const
 {
    #define    MILIEU -1 
    #define GAUCHE 0
    #define DROITE 1
    point xext;
     vector<int> position(dimension);

     int i;
    side=-1;
    dir =-1;
    t   =0.; 

    bool b=true;
    for (i=0; i<dimension; i++) 
        position[i]    = (x0[i]<xmin[i]-tol)    ? b=false, xext[i]=xmin[i], GAUCHE 
                    : (x0[i]>xmax[i]+tol)    ? b=false, xext[i]=xmax[i], DROITE
                                        : MILIEU;
    if (b) return true;

    for (i=0; i<dimension; i++) 
    {
        if ( (position[i]!=MILIEU) && (v[i] != Float(0)) )
             t = (             (xext[i]-x0[i])/(v[i]) > t ) 
                ? side=i,    (xext[i]-x0[i])/(v[i]) : t ;
    }
    b=true;
    for (i=0; i<dimension; i++)
        if ( (i!=side)
            && ( (x0[i] +t*v[i] <xmin[i] - tol)
                 || (xmax[i] + tol <x0[i] +t*v[i]) )
            )
            b=false;
    
    dir =position[side];
    return b;
 }

void 
hazel::list_leaf (std::ofstream& os, hazel_node& leaf) const
{
        slist<size_type>::const_iterator ie =(leaf).begin_interior();
        slist<size_type>::const_iterator last =(leaf).end_interior();
        while (ie!=last)
        {
                geo_element e=begin_element_[(*ie)];
                point mc=(begin_node_[e[0]]+begin_node_[e[1]]+begin_node_[e[2]])/3;
                os << "@point x1=" << mc[0] << " y1="<< mc[1] << " mt=3" <<endl;
                ie++;
        }    
        ie =(leaf).begin_boundary();
        last =(leaf).end_boundary();
        while (ie!=last)
        {
                geo_element e=begin_boundary_[(*ie)];
                point mc=(begin_node_[e[0]]+begin_node_[e[1]])/2;
                os << "@point x1=" << mc[0] << " y1="<< mc[1] << " mt=4" <<endl;
                ie++;
        }    
}

bool 
hazel::localize_in_leaf (const point& x, hazel_node& leaf, size_type& element) const
{
        slist<size_type>::const_iterator ie =leaf.begin_interior();
        slist<size_type>::const_iterator last =leaf.end_interior();
        while (ie!=last)
        {
                // TO BE MODIFIED: Test on hat coordinates ??
                if (in_interior_element(begin_element_[(*ie)],x))
                {
                        element =(*ie);
                        return true;
                }
                ie++;
        }    
        return false;
}

bool 
hazel::try_hard_in_leaf (const point& x, hazel_node& leaf, size_type& element) const
    //! localize in leaf with much higher tolerance -- used for debugging
{
        slist<size_type>::const_iterator ie =leaf.begin_interior();
        slist<size_type>::const_iterator last =leaf.end_interior();
        while (ie!=last)
        {
                // TO BE MODIFIED: Test on hat coordinates ??
                if (in_bbox_element(begin_element_[(*ie)],x))
                {
                        element =(*ie);
                        return true;
                }
                ie++;
        }    

        return false;
}

bool
hazel::localize_nearest_in_leaf (const point& x, point& y, const hazel_node& leaf) const
{
        // Localizes the nearest point y of the intersection of the boundary
        // and the given leaf.

        if (dimension!=2)
                error_macro("Geomap between approximate meshes not implemented for this dimension");

        point P0,P1,z, P0P1,P0x;
        Float distmin=numeric_limits<Float>::max();
            // min squared distance achieved yet
        Float c1,c2;

        slist<size_type>::const_iterator ie;
        slist<size_type>::const_iterator last;
        ie =leaf.begin_boundary();
        last =leaf.end_boundary();
        while (ie!=last)
        {
            geo_element element =begin_boundary_[(*ie)];
            // Source: http://geometryalgorithms.com/Archive/algorithm_0102/algorithm_0102.htm#Distance%20to%20Ray%20or%20Segment
            P0=begin_node_[element[0]];
            P1=begin_node_[element[1]];
            P0P1 =P1-P0;
            P0x  =x-P0;
            c1 = dot(P0x, P0P1);
            if (c1 <= Float(0)) // P0 is nearest to x in P0P1
             {
                 if (dist2(x,P0)<distmin) 
                 { distmin=dist2(x,P0); y=P0; }
             }
            else
             {
                 c2 =dot(P0P1,P0P1);
                if (c2 <= c1) // P1 is nearest
                 {
                    if (dist2(x,P1)<distmin) 
                     { distmin=dist2(x,P1); y=P1; }
                 }
                else 
                 {    
                    z=P0 + (c1/c2)*P0P1;
                    if (dist2(x,z)<distmin)
                     { distmin=dist2(x,z); y=z; }
                 }
             }
            ie++;
        }

        if (distmin==numeric_limits<Float>::max())
                // No boundary elements in this leaf !
                return false;

        return true;
}

void
hazel::localize_nearest (const point& x, point& y, geo_element::size_type& element) const
{
        if (!grown) error_macro("Hazel tree must be grown before it is used.");

        point z;
        Float xy; // distance y-x
        int i;
        int i_stump[3] = {0, 0, 0};
        for (i=0; i<dimension; i++)
        {
                i_stump[i] =(int)floor((x[i]-xmin[i])/(xmax[i]-xmin[i])*n_stump[i]);
                if (i_stump[i]>=n_stump[i]) i_stump[i]=n_stump[i]-1;
                if (i_stump[i]<0) i_stump[i]=0;
        }

        hazel_node* leaf =at(index(i_stump));

        hazel_node* leaf_y =at(index(i_stump));
        // the leaf of the nearest mesh-point may be different

        if (localize_in_leaf(x, *leaf, element))
          {
          y=x;
          return;
          }
        else 
        {
        std::ofstream file("hazel.mtv",ios::out|ios::trunc);
        file << "# Coordinates of the point and elements considered in failed localization"
            << endl;
        file << "@point x1=" << x[0] << " y1=" << x[1] << endl; 
        file << "@rectangle x1=" << (xmax-xmin)[0]/n_stump[0]*i_stump[0]+xmin[0]
             << " y1=" << (xmax-xmin)[1]/n_stump[1]*i_stump[1]+xmin[1]
             << " x2=" << (xmax-xmin)[0]/n_stump[0]*(i_stump[0]+1) +xmin[0]
             << " y2=" << (xmax-xmin)[1]/n_stump[1]*(i_stump[1]+1) +xmin[1]
             << endl;
        if (localize_nearest_in_leaf (x, y, *leaf))
         {
          file << "#Nearest:" << endl;
          file << "@point x1=" << y[0] << " y1=" << y[1] << endl;
         }
        list_leaf(file,*leaf_y);
        file.close();
        }
        /* 
        warning_macro("The original point was x="<< x);
        warning_macro("I thought I had found a valid element bucket for "
            << y << " in " << i_stump[0] <<"," << i_stump[1]<<","<<i_stump[2] 
            << " x " << (xmax-xmin)[0]/n_stump[0] <<"," 
            << (xmax-xmin)[1]/n_stump[1] <<"," );
        */

     // So x is not in the mesh :

        if (localize_nearest_in_leaf (x, y, *leaf))
          xy =dist2(x,y);
        else
          xy =-1;

    for (i=0; i<dimension; i++)
         {
                if ((leaf->get_neighbours()[2*i])
                   && ( (sqr(box_min(leaf)[i]-x[i]) <xy) || (xy == Float(-1)) ) )
                 {
                        localize_nearest_in_leaf(x,z,
                                *(leaf->get_neighbours()[2*i]));
                        if (dist2(x,z)<xy)
                         {
                                xy=dist2(x,z);
                                y=z;
                                leaf_y=leaf->get_neighbours()[2*i];
                         }
                 }
                if ((leaf->get_neighbours()[2*i+1])
                   && ( (sqr(box_max(leaf)[i]-x[i]) <xy) || (xy == Float(-1)) ) )
                 {
                        localize_nearest_in_leaf(x,z,
                                *(leaf->get_neighbours()[2*i+1]));
                        if (dist2(x,z)<xy)
                         {
                                xy=dist2(x,z);
                                y=z;
                                leaf_y=leaf->get_neighbours()[2*i+1];
                         }
                 }
         }
        if (localize_in_leaf(y, *leaf_y, element)) return;
        else if (try_hard_in_leaf(y, *leaf_y, element))
            {
                warning_macro("I had to use bounding box:" 
                    << begin_node_[begin_element_[element][0]] << " " 
                    << begin_node_[begin_element_[element][1]] << " " 
                    << begin_node_[begin_element_[element][2]] << " for localization of" 
            << y << " in " << i_stump[0] <<"," << i_stump[1]<<","<<i_stump[2] 
            << " x " << (xmax-xmin)[0]/n_stump[0] <<"," 
            << (xmax-xmin)[1]/n_stump[1] <<"," 
            << ", there must be a tolerance pb."
                    );
                warning_macro("The original point was x="<< x);
                return;
            }
        else
        {
                warning_macro("The original point was x="<< x);
        error_macro("I thought I had found a valid element bucket for "
            << y << " in " << i_stump[0] <<"," << i_stump[1]<<","<<i_stump[2] 
            << " x " << (xmax-xmin)[0]/n_stump[0] <<"," 
            << (xmax-xmin)[1]/n_stump[1] <<"," 
            << ", there must be a tolerance pb.");
        }
 }

bool
hazel::localize    (const point& x, geo_element::size_type& element) const
 {
    if (!grown) error_macro("Hazel tree must be grown before it is used.");
    if (!in_boundingbox(x, xmin, xmax)) return false;

    int i;
    int i_stump[3] = {0, 0, 0};
    for (i=0; i<dimension; i++)
     {
        i_stump[i] = int(floor((x[i]-xmin[i])/(xmax[i]-xmin[i])*n_stump[i]));
        if (i_stump[i]>=n_stump[i]) i_stump[i]=n_stump[i]-1;
        if (i_stump[i]<0) i_stump[i]=0;
     };
    hazel_node* leaf =at(index(i_stump));
    
    // TO BE MODIFIED:
    // Insert here descent method

    return localize_in_leaf(x, *leaf, element);
 }

bool
hazel::trace (const point& x0, const point& v, point& x, Float& t, size_type& element) const
 {
     if (!grown) error_macro("Hazel tree must be grown before it is used.");
    int    side, dir;
    bool  got_one =false;
    bool  inward;
    point boxmin;
    point boxmax;
    point y;

    Float ti;

    x=x0; t=0;
    if (!hit_boundingbox (x0, v, xmin, xmax, ti, side, dir)) 
        return false;
    
    point xi =x0 +ti*v;

    slist<size_type>::const_iterator ie; 
    slist<size_type>::const_iterator last;

    int i;
    int i_stump[3] ={0,0,0};

    for (i=0; i<dimension; i++)
     {
        i_stump[i] = int(floor((xi[i]-xmin[i])/h_stump[i]));
        if (i_stump[i]==n_stump[i]) i_stump[i]--;
        if (i_stump[i] <0) i_stump[i]=0;
     }

    hazel_node* leaf =at(index(i_stump));

    while (leaf) 
     {
        // TO BE MODIFIED: insert here descent method.

        // Test for hits on the bdry in this leaf. If we get some, the best is among them !
        t=1;
        ie =leaf->begin_boundary();
        last =leaf->end_boundary();
        while (ie!=last)
          {
            //TO BE MODIFIED:
            // Insert here bbox filter

            // Do we get a better hit than before ?
            if (hit_boundary_element(begin_boundary_[(*ie)], x0,v, xi, ti, inward) 
                && ti<=t) 
                {
                if ( (ti != Float(0)) || (!inward) )
                // It is the best so far !
                  {
                    t =ti; x=xi;
                    got_one =true;
                 } 
                }
            ie++;
         }    

        // Did we get one at all ?
        if (got_one) 
         {
             // Then find its interior element (and project if ever it was slightly
            // out of the mesh)
            if (!localize_in_leaf(x, *leaf, element))
             {
                localize_nearest (x, y, element) ;
                x=y;
             }
            return true;
         }

        // Is there a way out of this box ?
        boxmin =box_min(leaf);
        boxmax =box_max(leaf);
        // TO BE MODIFIED: tolerance
        if ( (!hit_boundingbox(x0+v, -v, boxmin, boxmax, t, side, dir)) )
         {
             Float tol2=::sqrt(tol);
            if (!hit_boundingbox(x0+v, -v, 
                boxmin-point(tol2,tol2,tol2), boxmax+point(tol2,tol2,tol2),
                t, side, dir)) 
                    error_macro("Something weird happened in " << boxmin << " " << boxmax 
                    << " for x0="<<x0<<" x1="<<x0+v<<" v="<<v<<" tol="<<tol
                    <<" and side, dir "<< side << " " <<dir);
         }

        // If it stops in the box :
        if (dir==-1) 
         {
            // Then find its interior element, if it's in the mesh
            if (localize_in_leaf (x0+v, *leaf, element))
             {
                 x =x0+v;
                 t=1; 
                return true;
             }
            else
                return false;
         }

        // OK, try go this way then
        leaf =leaf->get_neighbours()[2*side+dir];
     } 

    // we crossed the bounding box of omega in vain...
    return false;
 }
