%option noyywrap
%{
// This file is part of Rheolef.
//
// Copyright (C) 2000-2009 Pierre Saramito 
//
// Rheolef is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Rheolef is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Rheolef; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/* compile commands:

 	flex -+ -i doc2texi.l 
 	g++ -I/usr/local/gnu/include lex.yy.c -o doc2texi
*/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif 

#ifdef _RHEOLEF_HAVE_CSTDIO
#include <cstdio> // sscanf()
#else
#include <stdio.h>
#endif

#ifdef _RHEOLEF_HAVE_CCTYPE
#include <cctype> // tolower()
#else
#include <ctype.h>
#endif

#include <iostream>
#include <string>
#ifdef _RHEOLEF_HAVE_CSTRING
#include <cstring>
#endif // _RHEOLEF_HAVE_CSTRING



using namespace std;

void usage() 
{
  cerr << "usage:\n"
       << " doc2texi "
       << "[-latex] "
       << "[-chapter] "
       << "[-file string] "
       << "[-node string] "
       << "< file.c > file.texi"
       << endl;
}
static int line_no = 0;
string file_name;
string node_line;
static bool in_example = false;
static  char  string_buf [1000]; // be carreful to LONG string !
static  char *string_buf_ptr;
bool use_chapter = false; // true => do NAME title a chapter title

typedef enum {texinfo=0, latex=1} mode_type;
static mode_type mode = texinfo;

static const char* chapter_hierarchy[] = { "chapter", "section", "subsection"};
static const char* texinfo_section_hierarchy[] =
		{ "section", "subheading", "subsubheading"};
static const char* latex_section_hierarchy[] = 
		{ "section", "subsection", "subsubsection"};
static const char** hierarchy = texinfo_section_hierarchy;

static const char texinfo_special = '@';
static const char latex_special = '\\';
static char special = texinfo_special;

static const char* begin_example[] = { "@example", "\\begin{verbatim}"};
static const char* end_example[]   = { "@end example", "\\end{verbatim}"};

static const char* begin_title[]   = { " ", "{"};
static const char* end_title[]     = { "",  "}"};

#define error_macro(msg) {               \
	cerr << "error("<<line_no<<"): " \
	     << msg                      \
	     << endl;                    \
	exit(1);                         \
    }
void yyerror (const char* msg) { error_macro(msg); }
%}
ID		[a-zA-Z_][a-zA-Z_\-0-9]*
SPECIALS	[@{}]
SKIP            "AUTHOR"("S"?)|"DATE"|"METHOD"("S"?)|"SEE ALSO"
SUBJECT         "P"|"PROG"|"S"|"SYSTEM"|"F"|"Func"|"Class"|"E"|"D"|"G"|"M"|"A"

%x body section skip string_state verbatim after
%%
%{
// --------------------------------------------------------------
//  		start and finish body
//		  /*Prog:hb
//                   body....
//                End:
//                */
//
//                //<vec:
//                   verbatim
//                //>vec:
// --------------------------------------------------------------
%}
<INITIAL,after>^(("/*")?){SUBJECT}":"[^:\n]*\n {
			  // cout << "@c label:" << yytext_ptr;
			  line_no++;
			  BEGIN(body);
			}
<after>("//<"|"/*<"){ID}":"[ \t]*("*/")?[ \t]*\n        {
			  cout << special 
                               << hierarchy[1]
			       << begin_title[mode] 
                               << " Implementation" 
			       << end_title[mode]
                               << endl;
			  cout << begin_example[mode] << endl;
			  line_no++;
			  BEGIN(verbatim);
			}
<INITIAL,after>[^\n]*\n	{
			  // skip
			  line_no++;
			  ;
			}
<body,section,skip>^("END")":"[^:\n]*\n {
			  if (in_example) {
			        cout << end_example[mode] << endl;
				in_example = false;
			  }
			  // cout << "@c END" << endl;
			  BEGIN(after);
			  line_no++;
			}
<verbatim>"/*!"		{
			  // doxygen escapes
			  cout << "/* ";
			}
<verbatim>"//!"		{
			  // doxygen escapes
			  cout << "// ";
			}
<verbatim>("//>"|"/*>"){ID}":"[ \t]*("*/")?[ \t]*\n     {
			  cout << end_example[mode] << endl;
			  BEGIN(after);
			  line_no++;
			}
%{
// --------------------------------------------------------------
//  		special sections
// --------------------------------------------------------------
%}
<body,section,skip>^{SKIP}":" {
			  if (in_example) {
			        cout << end_example[mode] << endl;
				in_example = false;
			  }
			  // cout << "@c skip start:" << yytext_ptr << endl;
			  BEGIN(skip);
			}
%{
// --------------------------------------------------------------
//  		name section, with one-line title
// --------------------------------------------------------------
%}
<body,section,skip>^("NAME")":"[^\n]*\n	{
			  unsigned int i1 = 0;
			  for (;i1 < strlen(yytext_ptr); i1++) {
			      if (yytext_ptr[i1] == ':') break;
			  }
			  unsigned int i2 = strlen(yytext_ptr)-1;
			  for (;i2 > i1; i2--) {
			      if (yytext_ptr[i2] == '(') break;
			  }
			  if (i2 != i1) {
			      yytext_ptr[i2] = 0;
			  }
			  if (mode != latex && node_line.length() != 0) {
			      cout << "@node " << node_line << endl;
			  }
			  cout << special 
                               << hierarchy[0] 
			       << begin_title[mode] 
                               << yytext_ptr+i1+1 
			       << end_title[mode] 
                               << endl;
			  if (file_name.length() != 0) {
			      cout << "(Source file: " 
				   << special << "file{" << file_name 
                                   << "})" << endl;
			  }
			  BEGIN(section);
			  line_no++;
			}
<body,section,skip>^[a-zA-Z][a-zA-Z_\- ]*":" {
			  if (in_example) {
			        cout << end_example[mode] << endl;
				in_example = false;
			  }
			  // make title
			  yytext_ptr[0] = toupper(yytext_ptr[0]);
			  for (unsigned int i = 1; i < strlen(yytext_ptr); i++) {
			      yytext_ptr[i] = tolower(yytext_ptr[i]);
			  }

			  yytext_ptr[strlen(yytext_ptr)-1] = 0;
			  cout << special 
                               << hierarchy[1] 
			       << begin_title[mode] 
			       << yytext_ptr 
			       << end_title[mode] 
			       << endl;
			  BEGIN(section);
			}
<skip>[ \t]+[^\n]*\n	{
			  // skip
			  line_no++;
			  ;
			}
%{
// --------------------------------------------------------------
//  		strings
// --------------------------------------------------------------
%}
<section>\"		{
                            string_buf_ptr = string_buf;
                            BEGIN(string_state);
                        }
<string_state>\"			{ // saw closing quote - all done
			    BEGIN(section);
			    *string_buf_ptr = '\0';
			    // TOCLEAN:
			    // cout << special << "code{" << string_buf << "}"; 
			    cout << '\"' << string_buf << '\"'; 
			}
<string_state>{SPECIALS}		{	
			   // specials in string
			   if (mode != latex) {
				*string_buf_ptr++ = special;
			   }
			   *string_buf_ptr++ = *yytext;
                        }
<string_state>\\[0-7]{1,3} 	{ // octal escape sequence
			int result;
			sscanf (yytext_ptr + 1, "%o", &result );
			if ( result > 0xff ) {
		    	    error_macro("octal escape sequence `" 
                                 << yytext_ptr << "' in string: constant is out-of-bounds"); 
                        }  
			*string_buf_ptr++ = result;
			}
<string_state>\\[0-9]+ 		{
                           error_macro("bad escape sequence `"
                                << yytext_ptr <<"' in string");
                        }  
<string_state>\n			{ 
                           error_macro("unterminated string constant"); 
			   line_no++;
                        }
<string_state>\\n		*string_buf_ptr++ = '\n';
<string_state>\\t		*string_buf_ptr++ = '\t';
<string_state>\\r		*string_buf_ptr++ = '\r';
<string_state>\\b		*string_buf_ptr++ = '\b';
<string_state>\\f		*string_buf_ptr++ = '\f';
<string_state>\\(.|\n)		*string_buf_ptr++ = yytext[1]; 
<string_state>[^@{}\\\n\"]+	{ for (register char *p = yytext_ptr; *p; *string_buf_ptr++ = *p++); }

%{ 
// --------------------------------------------------------------
//  		specials
// --------------------------------------------------------------
%}
<verbatim>{SPECIALS}	{
			   if (mode != latex) {
				cout << special;
			   }
			   cout << yytext[0];
                        }
<section>{SPECIALS}	{
			   cout << yytext[0];
                        }
%{ 
// --------------------------------------------------------------
//  		others
// --------------------------------------------------------------
%}
<section>\n		{ cout << endl;  line_no++; }
<section>.		{ cout << *yytext ;  /* Le reste */ }
%%

int main(int argc, char** argv) {
// --------------------------------------------------------------
//  		command line options
// --------------------------------------------------------------
    hierarchy = texinfo_section_hierarchy;
    use_chapter = false;
    for (int i = 1; i < argc; i++) {
	if (strcmp(argv[i], "-chapter") == 0) {
    	    hierarchy = chapter_hierarchy;
	    use_chapter = true;
	} else if (strcmp(argv[i], "-file") == 0) {
	    char *p = argv[++i];
	    while (*p && (*p == '.' || *p == '/')) p++;
	    file_name = string(p);
	} else if (strcmp(argv[i], "-node") == 0) {
	    node_line = string(argv[++i]);
	} else if (strcmp(argv[i], "-latex") == 0) {
	    mode      = latex;
	    special   = latex_special;
	    hierarchy = latex_section_hierarchy;
	} else {
	    cerr << "doc2tex: invalid option `" 
	         << argv[i] << "'" << endl;
	    usage();
	    exit(0);
	}
    }
// --------------------------------------------------------------
//  		run lexer
// --------------------------------------------------------------
    yyFlexLexer x;
    line_no = 1;
    in_example = false;
    return x.yylex();
}
