///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
/*Prog:field_gen
NAME: "field_gen" - output a P1-continuous field 
SYNOPSIS:
   "field_gen" {-I'geodir'} 'filename'[".geo"]
DESCRIPTION:       
  Generate a P1 finite element field as the interpolation
  of a given function (furnished at compile time).
EXAMPLE: 
    enter command as:

    |  field_gen square.geo
OPTIONS:
    -I'geodir' add 'geodir' to the GEOPATH search path.
    See also "geo"(3) for GEOPATH mechanism. 

    'filename' specifies the name of the file containing
      the input mesh. The ".geo" extension is assumed.

    "-" read mesh on standard input instead on a file.
AUTHOR: 
    LMC-IMAG, 38041 Grenoble cedex 9, France
    | Nicolas.Roquet@imag.fr
    | Pierre.Saramito@imag.fr
SEE ALSO:
    class "field(3)", "space"(3), class "geo"(3), class "base"(3)
DATE:
    7 july 1997
End:
*/
#include "rheolef/rheolef.h"
using namespace rheolef;
using namespace std;

Float f_2d (const point& x)
{
    Float val = (x[0]*(1-x[0]) + x[1]*(1-x[1]))/4;
    return val;
}
Float f_3d (const point& x)
{
    return (x[0]*(1-x[0]) + x[1]*(1-x[1]) + x[2]*(1-x[2]))/6;
}
void usage()
{
      cerr << "field_gen: usage: field_gen "
	   << "{-Igeodir}*"
	   << "-|mesh[.geo]"
	   << endl;
      exit (1);
}

int main(int argc, char**argv)
{
    bool verbose = true;
    if (argc <= 1) usage();
    //
    // load geometry
    //
    geo g;  
    int io = 1; 
    while (argv [io][0] == '-' && argv [io][1] == 'I') {
	append_dir_to_rheo_path (argv[io]+2);
	io++;
    }
    if (strcmp (argv[io], "-") == 0) {
	// input geo on standard input
        cerr << "! field_gen: geo on stdin\n";
	cin >> g;
      
    } else {
    
	// input geo on file
  	g = geo(argv[io]);
    }
    space V (g, "P1");
    //
    // build and output field
    //
    field u;
    if (g.dimension() == 2) {
      u = interpolate(V, f_2d);
    } else {
      u = interpolate(V, f_3d);
    }
    // force conversion to double for hight prec classes
    // for non-regression test purpose
    numeric_flags<Float>::output_as_double(true);

    cout << rheo << u;
    return 0;
}
