#ifndef _RHEOLEF_DOMAIN_H
#define _RHEOLEF_DOMAIN_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================

#include "rheolef/array.h"
#include "rheolef/geo_element.h"

namespace rheolef {

// =====================================================================
// domain data = pair (sign orientation ; size_t element_index)
// =====================================================================
// TODO: compact the sign bit and size_t in a long int
struct domain_pair_type {

// typedefs:

    typedef array<size_t>::size_type      size_type;
    typedef geo_element::orientation_type orientation_type;

// allocators:

    domain_pair_type () : _pos(true), _isg (std::numeric_limits<size_type>::max()) {}
    domain_pair_type (orientation_type orient, size_type isg) : _pos(orient > 0), _isg(isg) {}

// accessors:

    orientation_type orientation() const { return (_pos ?  1 : - 1); }
    size_type        index()       const { return _isg; }

// modifiers:

    void set_orientation (orientation_type orient) { _pos = (orient > 0); }
    void set_index       (size_type isg)           { _isg = isg; }
    void set (orientation_type orient, size_type isg) {
        set_orientation (orient);
        set_index       (isg);
    }

// i/o:

    friend std::istream&  operator>> (std::istream& is, domain_pair_type& x);
    friend std::ostream&  operator<< (std::ostream& os, const domain_pair_type& x);
    template<class Archive>
    void serialize (Archive& ar, const unsigned int version) { ar & _pos; ar & _isg; }

// data:
protected:
    bool      _pos;
    size_type _isg;
};

} // namespace rheolef

#ifdef _RHEOLEF_HAVE_MPI
// =====================================================================
// Some serializable types, have a fixed amount of data stored at fixed field positions.
// When this is the case, boost::mpi can optimize their serialization and transmission to avoid extraneous 
// copy operations.
// To enable this optimization, we specialize the type trait is_mpi_datatype, e.g.:
namespace boost {
 namespace mpi {
  template <> struct is_mpi_datatype<rheolef::domain_pair_type> : mpl::true_ { };
 } // namespace mpi
} // namespace boost
#endif // _RHEOLEF_HAVE_MPI

namespace rheolef {

// foward declarations:
template <class T> class geo_seq_rep;
template <class T> class geo_mpi_rep;
template <class T, class M> class geo_basic;

// =====================================================================
// 1) representation: domain_rep, domain_seq_rep and domain_mpi_rep
// =====================================================================
template <class T, class M>
class domain_rep : public array<domain_pair_type,M> {
public:

// typedefs:

    typedef array<domain_pair_type,M> base;
    typedef typename domain_pair_type::size_type        size_type;
    typedef typename domain_pair_type::orientation_type orientation_type;
    typedef typename base::iterator                     iterator_ioisg;
    typedef typename base::const_iterator               const_iterator_ioisg;

// allocators:

    domain_rep();
    void resize (size_type n) { base::resize (n); }

// accessors & modifiers:

    size_type             size()     const { return base::size(); }
    size_type             par_size() const { return base::par_size(); }
    const geo_basic<T,M>& get_geo()  const;

    const_iterator_ioisg begin_ioisg() const { return base::begin(); }
    const_iterator_ioisg end_ioisg()   const { return base::end(); }
          iterator_ioisg begin_ioisg()       { return base::begin(); }
          iterator_ioisg end_ioisg()         { return base::end(); }

    const domain_pair_type& oisg (size_type ioisg) const {
	return base::operator[] (ioisg); }

    std::string name ()    const { return _name; }
    size_type dimension () const { return _dim; }
    void set_name (std::string name) { _name = name; }
    void set_dimension (size_type dim) { _dim = dim; }

// data:
protected:
    std::string           _name;
    size_type             _dim;
    const geo_basic<T,M>* _ptr_omega;
};
template <class T, class M>
inline
domain_rep<T,M>::domain_rep()
 : array<domain_pair_type,M>(),
   _name(),
   _dim(0),
   _ptr_omega(0)
{
}
template <class T, class M>
inline
const geo_basic<T,M>&
domain_rep<T,M>::get_geo() const
{
    check_macro (_ptr_omega != 0, "uninitialized domain: no geometry available");
    return *_ptr_omega;
}
// ---------------------------------------------------------------------
template <class T>
class domain_seq_rep : public domain_rep<T,sequential> {
public:

// typedefs:

    typedef domain_rep<T,sequential>              base;
    typedef typename base::size_type              size_type;
    typedef typename base::iterator_ioisg         iterator_ioisg;
    typedef typename base::const_iterator_ioisg   const_iterator_ioisg;
    typedef typename base::orientation_type       orientation_type;

// allocators:

    domain_seq_rep () : domain_rep<T,sequential>() {}
    void resize (size_type n) { base::resize (n); }

// accessors:

    size_type size()       const { return base::size(); }
    size_type par_size()   const { return base::par_size(); }

    const_iterator_ioisg begin_ioisg() const { return base::begin_ioisg(); }
    const_iterator_ioisg end_ioisg()   const { return base::end_ioisg(); }
          iterator_ioisg begin_ioisg()       { return base::begin_ioisg(); }
          iterator_ioisg end_ioisg()         { return base::end_ioisg(); }

    const domain_pair_type& oisg (size_type ioisg) const {
	return base::oisg (ioisg); }

    void set_name (std::string name)   { base::set_name(name); }
    void set_dimension (size_type dim) { base::set_dimension(dim); }
    std::string name ()    const { return base::name(); }
    size_type dimension () const { return base::dimension(); }

// i/o:

#ifdef TO_CLEAN
    iparstream& get (iparstream& ips);
#endif // TO_CLEAN

    oparstream& put (oparstream& ops) const;
    iparstream& get (iparstream& ips, const geo_basic<T,sequential>& smart_ptr_omega,
    	std::vector<std::set<size_type> > ball[4]);
};
// ---------------------------------------------------------------------
#ifdef _RHEOLEF_HAVE_MPI
template <class T>
class domain_mpi_rep : public domain_rep<T,distributed> {
public:

// typedefs:

    typedef domain_rep<T,distributed>             base;
    typedef typename base::size_type              size_type;
    typedef typename base::iterator_ioisg         iterator_ioisg;
    typedef typename base::const_iterator_ioisg   const_iterator_ioisg;
    typedef typename base::orientation_type       orientation_type;

// allocators:

    domain_mpi_rep();

// accessors & modifiers:

    size_type size()       const { return base::size(); }
    size_type par_size()   const { return base::par_size(); }

    const domain_pair_type& oisg (size_type ioisg) const {
	return base::oisg (ioisg); }

    void set_name (std::string name)   { base::set_name(name); }
    void set_dimension (size_type dim) { base::set_dimension(dim); }
    std::string name ()    const       { return base::name(); }
    size_type dimension () const       { return base::dimension(); }

// i/o:

    iparstream& get (iparstream& ips, const geo_basic<T,distributed>& omega);
    oparstream& put (oparstream& ops, const geo_mpi_rep<T>& omega) const;

// data:
protected:
    array<domain_pair_type,distributed> _ini_oisg;
    array<size_type,distributed>        _ioisg2par_ini_ioisg;
    array<size_type,distributed>        _ini_ioisg2par_ioisg;
};
template<class T>
inline
domain_mpi_rep<T>::domain_mpi_rep()
  : domain_rep<T,distributed>(),
    _ini_oisg(),
    _ioisg2par_ini_ioisg(),
    _ini_ioisg2par_ioisg()
{
}
#endif // _RHEOLEF_HAVE_MPI
// ====================================================================
// 2) wrapper class: seq & mpi specializations
// ====================================================================
/*Class:domain
NAME: @code{domain} - part of a finite element mesh
@cindex  mesh boundary
@clindex domain
DESCRIPTION:
  @noindent
  The @code{domain} class defines a container for a part of a
  finite element mesh.
  This describes the connectivity of edges or faces.
  This class is usefull for boundary condition setting.
DATE: 12 may 1997
End:
*/
/// @brief the finite element boundary domain
template <class T, class M = rheo_default_memory_model>
class domain_basic {
public:
};
typedef domain_basic<Float> domain;
// ---------------------------------------------------------------------
//<verbatim:
template <class T>
class domain_basic<T,sequential> : public smart_pointer<domain_seq_rep<T> > {
public:

// typedefs:

    typedef domain_seq_rep<T>                  rep;
    typedef smart_pointer<rep>                 base;
    typedef typename rep::size_type            size_type;
    typedef typename rep::iterator_ioisg       iterator_ioisg;
    typedef typename rep::const_iterator_ioisg const_iterator_ioisg;

// allocators:

    domain_basic ();
    void resize (size_type n);

// accessors:

    size_type size()     const;
    size_type par_size() const;

    const_iterator_ioisg begin_ioisg() const;
    const_iterator_ioisg end_ioisg()   const;
          iterator_ioisg begin_ioisg();
          iterator_ioisg end_ioisg();

    const domain_pair_type& oisg (size_type ioisg) const;

    void set_name (std::string name);
    void set_dimension (size_type dim);
    std::string name ()    const;
    size_type dimension () const;

// i/o:

    oparstream& put (oparstream&) const;
    iparstream& get (iparstream&);
    iparstream& get (iparstream& ips, const geo_basic<T,sequential>& omega,
    	std::vector<std::set<size_type> > ball[4]);
};
//>verbatim:
template<class T>
inline
domain_basic<T,sequential>::domain_basic ()
 : smart_pointer<domain_seq_rep<T> > (new_macro(domain_seq_rep<T>))
{
}
template<class T>
inline
void
domain_basic<T,sequential>::resize (size_type n)
{
  return base::data().resize (n);
}
template<class T>
inline
typename domain_basic<T,sequential>::size_type
domain_basic<T,sequential>::size() const
{
  return base::data().size();
}
template<class T>
inline
typename domain_basic<T,sequential>::size_type
domain_basic<T,sequential>::par_size() const
{
  return base::data().par_size();
}
template<class T>
inline
const domain_pair_type&
domain_basic<T,sequential>::oisg (size_type ioisg) const
{
  return base::data().oisg (ioisg);
}
template<class T>
inline
typename domain_basic<T,sequential>::const_iterator_ioisg
domain_basic<T,sequential>::begin_ioisg() const
{
  return base::data().begin_ioisg();
}
template<class T>
inline
typename domain_basic<T,sequential>::const_iterator_ioisg
domain_basic<T,sequential>::end_ioisg() const
{
  return base::data().end_ioisg();
}
template<class T>
inline
typename domain_basic<T,sequential>::iterator_ioisg
domain_basic<T,sequential>::begin_ioisg()
{
  return base::data().begin_ioisg();
}
template<class T>
inline
typename domain_basic<T,sequential>::iterator_ioisg
domain_basic<T,sequential>::end_ioisg()
{
  return base::data().end_ioisg();
}
template<class T>
inline
std::string
domain_basic<T,sequential>::name() const
{
  return base::data().name();
}
template<class T>
inline
typename domain_basic<T,sequential>::size_type
domain_basic<T,sequential>::dimension() const
{
  return base::data().dimension();
}
template<class T>
inline
void
domain_basic<T,sequential>::set_name(std::string name)
{
  return base::data().set_name (name);
}
template<class T>
inline
void
domain_basic<T,sequential>::set_dimension (size_type dim)
{
  return base::data().set_dimension (dim);
}
template<class T>
inline
oparstream&
domain_basic<T,sequential>::put (oparstream& ops) const
{
  return base::data().put (ops);
}
template<class T>
inline
iparstream&
domain_basic<T,sequential>::get (iparstream& ips)
{
  return base::data().get (ips);
}
template<class T>
inline
iparstream&
domain_basic<T,sequential>::get (
    iparstream&                       ips,
    const geo_basic<T,sequential>&    omega,
    std::vector<std::set<size_type> > ball[4])
{
  return base::data().template get (ips, omega, ball);
}
// ---------------------------------------------------------------------
#ifdef _RHEOLEF_HAVE_MPI
//<verbatim:
template <class T>
class domain_basic<T,distributed> : public smart_pointer<domain_mpi_rep<T> > {
public:

// typedefs:

    typedef domain_mpi_rep<T>        rep;
    typedef smart_pointer<rep>       base;
    typedef typename rep::size_type  size_type;

// allocators:

    domain_basic ();

// accessors/modifiers:

    size_type size()     const;
    size_type par_size() const;

    const domain_pair_type& oisg (size_type ioisg) const;

    void set_name (std::string name);
    void set_dimension (size_type dim);
    std::string name () const;
    size_type dimension () const;

// i/o:

    iparstream& get (iparstream& ips, const geo_basic<T,distributed>& omega);
    oparstream& put (oparstream& ops, const geo_mpi_rep<T>& omega) const;
};
//>verbatim:

template<class T>
inline
domain_basic<T,distributed>::domain_basic ()
 : smart_pointer<rep> (new_macro(rep))
{
}
template<class T>
inline
typename domain_basic<T,distributed>::size_type
domain_basic<T,distributed>::size() const
{
  return base::data().size();
}
template<class T>
inline
typename domain_basic<T,distributed>::size_type
domain_basic<T,distributed>::par_size() const
{
  return base::data().par_size();
}
template<class T>
inline
iparstream&
domain_basic<T,distributed>::get (
    iparstream&                     ips,
    const geo_basic<T,distributed>& omega)
{
  return base::data().template get (ips, omega);
}
template<class T>
inline
oparstream&
domain_basic<T,distributed>::put (
    oparstream& ops,
    const geo_mpi_rep<T>& omega) const
{
  return base::data().template put (ops, omega);
}
template<class T>
inline
std::string
domain_basic<T,distributed>::name() const
{
  return base::data().name();
}
template<class T>
inline
typename domain_basic<T,distributed>::size_type
domain_basic<T,distributed>::dimension() const
{
  return base::data().dimension();
}
template<class T>
inline
void
domain_basic<T,distributed>::set_name(std::string name)
{
  return base::data().set_name (name);
}
template<class T>
inline
void
domain_basic<T,distributed>::set_dimension (size_type dim)
{
  return base::data().set_dimension (dim);
}
template<class T>
inline
const domain_pair_type&
domain_basic<T,distributed>::oisg (size_type ioisg) const
{
  return base::data().oisg (ioisg);
}

#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
#endif // _RHEOLEF_DOMAIN_H
