///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// visualisation of geometries using mayavi
//
// authors:
//    	Pierre.Saramito@imag.fr
//
// date: 22 january 2006
//
// TODO:
//   - a mayavi Module ClipPolyData with a customizale cut plane
//   - how to cut all datas (boundfarie, ect) with the same plane in mayavi ??
//
#include "rheolef/georep.h"
#include "rheolef/geo-visu.h"
#include "rheolef/geo-visu-aux.h"
#include <sstream>
using namespace std;
namespace rheolef { 

// define when using old mayavi python library:
#undef _RHEOLEF_USE_MAYAVI_1

int
georep::visu_mayavi (const geo_visu_option_type& opt) const
{
  //
  // output geometry for vtkUnstructuredGridReader
  //
  string vtk_name = opt.basename + ".vtk";
  ofstream vtk (vtk_name.c_str());
  if (opt.verbose) clog << "! file \"" << vtk_name << "\" created." << endl;
  int digits10 = numeric_limits<Float>::digits10;
  vtk << setprecision(digits10);
  put_vtk (vtk);
  vtk.close();
  //
  // plot domains
  //
  domlist_type::const_iterator iter = _domlist.begin();
  domlist_type::const_iterator last = _domlist.end();
  for (; iter != last; ++iter) {
      const domain& dom = *iter;
      string dom_name = opt.basename + "." + dom.name() + ".vtk";
      vtk.open (dom_name.c_str());
      if (opt.verbose) clog << "! file \"" << dom_name << "\" created." << endl;
      dom.put_vtk (vtk, begin_node(), end_node());
      vtk.close();
  }
  //
  // output python script for mayavi/vtk executable
  //
  string py_name = opt.basename + ".py";
  ofstream py (py_name.c_str());
  if (opt.verbose) clog << "! file \"" << py_name << "\" created." << endl;
 
  py << "#!/usr/bin/env mayavi2" << endl
     << "# This is a mayavi script for the visualization of " << opt.basename << ".vtk" << endl
     << "# automatically generated by rheolef."  << endl
     << endl
     ;
#ifdef _RHEOLEF_USE_MAYAVI_1
  // old mayavi python library
  py << "import mayavi" << endl
     << "from mayavi_geo import *" << endl
     << endl
     ;
#else
  // new mayavi2 python library
  py << "from enthought.mayavi.scripts import mayavi2" << endl
     << "mayavi2.standalone(globals())" << endl
     << "from mayavi2_rheolef import *" << endl	
     << endl
     ;
#endif
  iter = _domlist.begin();
  last = _domlist.end();
  py << "d = [ '" << opt.basename << "'";
  for (; iter != last; ++iter) {
      const domain& dom = *iter;
      py << ", '" << dom.name() << "'";
  }
  py << "]" << endl;

  size_t d =  dimension();
  bool has_origin = (opt.origin[0] != numeric_limits<Float>::max());
  py << "opt = {			\\" << endl
     << "    'label'   : 1," << endl
     << "    'axis'    : 1," << endl
     << "    'color'   : '" << (opt.color ? "color" : (opt.gray ? "gray" : "black_and_white")) << "'," << endl
     << "    'stereo'  : " << opt.stereo << ",		\\" << endl
     << "    'ball'    : " << opt.ball << ",		\\" << endl
     << "    'cut'     : " << opt.cut  << ", 		\\" << endl
     << "    'fill'    : " << opt.fill << ",		\\" << endl
     << "    'full'    : " << opt.full << ",		\\" << endl
     << "    'lattice' : " << opt.lattice << ", 		\\" << endl
     << "    'shrink'  : " << opt.shrink << ", 		\\" << endl
     << "    'tube'    : " << opt.tube << ",		\\" << endl
     << "    'has_origin' : " << has_origin  << ", 		\\" << endl
     << "    'origin'     : " << python(opt.origin) << ", 		\\" << endl
     << "    'normal'     : " << python(-opt.normal) << "  		\\" << endl
     << "  }" << endl
     << endl
     ;
#ifdef _RHEOLEF_USE_MAYAVI_1
  py << "v = mayavi.mayavi()" << endl
     << "mayavi_geo(v, opt, d)" << endl
     << "v.Render()" << endl
     << "v.master.wait_window()" << endl
     << endl
     ;
#else
  py << "mayavi2_geo(mayavi, opt, d)" << endl
     << endl
     ;
#endif
  //
  // run pyton
  //
  int status = 0;
  string command;
  if (opt.execute) {
      command = "LANG=C PYTHONPATH=" + string(_RHEOLEF_PKGDATADIR) + " mayavi2 " + py_name;
      if (opt.verbose) clog << "! " << command << endl;
      status = system (command.c_str());
  }
  //
  // clear vtk data
  //
  if (opt.clean) {
      command = "/bin/rm -f " + opt.basename + ".vtk " + opt.basename + ".py";
      if (opt.verbose) clog << "! " << command << endl;
      status = system (command.c_str());
      iter = _domlist.begin();
      last = _domlist.end();
      for (; iter != last; ++iter) {
          const domain& dom = *iter;
          command = "/bin/rm -f " + opt.basename + "." + dom.name() + ".vtk";
          if (opt.verbose) clog << "! " << command << endl;
          status = system (command.c_str());
      }
  }
  return status;
}
}// namespace rheolef
