# Copyright (c) 2002 Red Hat, Inc. All rights reserved.
#
# This software may be freely redistributed under the terms of the
# GNU General Public License.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# Handles PostgreSQL style identifiers
#
# Written for Red Hat Inc. by Fernando Nasser <fnasser@redhat.com>
#
# Component of: Red Hat Database GUI Administration tool

package provide identifier 1.0

# validateIdentifier - Returns an empty string if identifier is valid,
# otherwise returns a string stating what is wrong with it.

proc validateIdentifier { newId} {
    global setting_autoQuoteIdentifiers 
    global MAXDATALEN
        
    # auto quoting is always on for now
    
    set setting_autoQuoteIdentifiers 1
    
    # if auto quote is on, anything can go, except for a double quote itself
    # otherwise check for valid PostgreSQL identifiers or delimited identifiers
    
    if {$setting_autoQuoteIdentifiers} {

	# can't have more than MAXDATALEN characters
		
	if {[string length $newId] > $MAXDATALEN} {
	    return "Object names must not be longer than $MAXDATALEN characters"
	}
	
	# can't have double quotes (yet)
	
	if {[string first {"} $newId] > 0} {
	    return "Double quotes are not allowed in object names"
	}
    	
    } else {
    
        # trim leading and trailing spaces as this is what we will do later
	
        set tempId [string trim $newId]

	if {[string index $tempId 0] == {"} } {
	
	    # a quoted identifier, we suppose
	    
	    set idLength [string lenght $tempId]
	    
	    # last character must also be a quote
	    
	    if {[string last {"} $tempId] != [expr $idLength - 1]} {
	        return "Unmatched quotes or extra characters after closing quotes in object name"
	    }
	    
	    # can't have double quotes (yet)
	    
	    if {[string first {"} $tempId 1] != [expr $idLength - 1]} {
	        return "Double quotes not allowed in object names, even on delimited ones"
	    }
	    
	    # can't have more than MAXDATALEN characters 
	    # not counting the quotes
	    
	    if {[string length $tempId] > [expr $MAXDATALEN + 2]} {
	        return "Object names must not be longer than MAXDATALEN characters"
	    }

	} else {
	
	    # This must be a plain PostgreSQL identifier
	    
            if {[regexp -nocase {^[a-z][a-z0-9_]*$} $tempId] == 0} {
                return "Invalid characters in object name without quotes"
            }
	
	    # can't have more than MAXDATALEN characters 
	    	    
	    if {[string length $tempId] > $MAXDATALEN} {
	        return "Object names must not be longer than MAXDATALEN characters"
	    }
	}
    }

    return ""
}

# validateOperatorName - Returns an empty string if newId is a valid
# operator name, otherwise returns a string stating what is wrong with it.

proc validateOperatorName { newId } {

    # operator name can only consist of +-*/<>=~!@#%^&|'?$
    if {[regexp -- {^[\+\-\*\/\<\>\=\~\!\@\#\%\^\&\|\'\?\$]+$} $newId] != 1} {
	return "Operator name can only consist of \+\-\*\/\<\>\=\~\!\@\#\%\^\&\|\'\?\$"
    }

    # "$" cannot be a single-character operator 
    if {$newId == "\$"} {
	return "Operator name must not be \$"
    }
    
    # "--" & "/*" will be taken as the start of a comment
    if {[string first -- $newId] != -1} {
	return "\-\- is not allowed as part of the name of an operator"
    }
    
    if {[string first /* $newId] != -1} {
	return "\/\* is not allowed as part of the name of an operator"
    }
    
    # multi-character operator name cannot end in "+" or "-" unless it
    # also conatins at least one of ~!@#%^&|'?$

    if {[string length $newId] > 1} {
	set lastChar [string range $newId end end]
	if {$lastChar == "+" || $lastChar == "\-"} {
	    foreach op {"\~" "\!" "\@" "\#" "\%" "\^" "\&" "\|" "\'" "\?" "\$"} {
		# if any of these characters is in the name, then it's valid.
		if {[string first $op $newId] != -1} {return ""}
	    }
	    # if none of the above character is in the name then it's invalid.
	    return "multi-character operator name cannot end in \+ or \- unless it also contains at least one of \~\"\!\@\#\%\^\&\|\'\?\$"
	}
    }
    
    return ""
}

# trimQuoteIdentifier - returns the identifier properly trimmed and/or quoted 
# so it can be sent to the database backend.
# Assumes validateIdentifier was called to validate the identifier

proc trimQuoteIdentifier { inputId } {
    global setting_autoQuoteIdentifiers 

    # auto quoting is always on for now
    
    set setting_autoQuoteIdentifiers 1
    
    # trim spaces
    
    set outputId [string trim $inputId]
    
    # if auto quote is on, check if quotes are needed and add them.
    # Capital letters cause quoting so that case is preserved
    
    if {$setting_autoQuoteIdentifiers} {
        if {[regexp {^[a-z][a-z0-9_]*$} $inputId] == 0} {
            set outputId \"$outputId\"
        }
    }
        
    return $outputId
}

# quoteIdentifier - returns the identifier properly quoted if quotes 
# are required to send it to the backend (used for display)

proc quoteIdentifier { inputId } {
    
    # check if quotes are needed and add them.
    # Capital letters cause quoting so that case is preserved
    
    if {[regexp {^[a-z][a-z0-9_]*$} $inputId] == 0} {
        set outputId \"$inputId\"
    } else {
        set outputId $inputId
    }
        
    return $outputId
}

# escapeIdentifier - returns the identifier with special characters
# properly escaped so that both Tcl and the backend are happy 
# (used to assemble Tcl commands)

proc escapeIdentifier { inputId } {
    
    # escape any quotes
        
    regsub -all {"} $inputId {\"} outputId
        
    return $outputId
}

# End of file
