# Copyright (c) 2002 Red Hat, Inc. All rights reserved.
#
# This software may be freely redistributed under the terms of the
# GNU General Public License.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# File to handle delete actions for various objects
# Written for Red Hat Inc. by Deepak Bhole <dbhole@redhat.com>
# Component of the RedHat database administration tool

package require Tk
package require Itcl
package require Itk
package require Iwidgets
package require cluster
package require aggregateHandle
package require columnHandle
package require constraintHandle
package require databaseHandle
package require dataTypeHandle
package require functionHandle
package require groupHandle
package require indexHandle
package require ruleHandle
package require sequenceHandle
package require tableHandle
package require triggerHandle
package require userHandle
package require viewHandle
package provide destructiveOperationHandler 1.0


itcl::class DeleteOperationsHandler {

    constructor {args} {

	# A list of what is acceptable

        set expectedArgs \
	    [list \
	     -clusterCollectionName\
	     -clusterName\
	     -uniqueConstraintIdentifyingName\
	     -databaseName\
	     -tableName\
	     -viewName\
	     -deleteName \
	     -objectType \
	     -objectHandle \
	     -refreshUpperLevelCommand \
	     -refreshAnotherLevelCommand\
	     -destructiveAction \
	     -optionalIdentificationData \
	    ]

	# Set the array to hold various options that we need

	foreach option $expectedArgs {
	    set options($option) ""
	}

	# Make sure we proceed only if we know how to perform the required 
	# action

	foreach { option value } $args {
	    if {[lsearch -exact $expectedArgs $option] == -1} {
		error "Unexpected option \"$option\" - only expect $expectedArgs"
	    }
	    set options($option) "$value"
	}
	
	# Get the cluster name, no matter what object we delete, we will need 
	# this so get it beforehand! [Note: In some cases, we have direct 
	# access to the object handle. Don't set clusterHandle in such cases.

	if {$options(-clusterCollectionName) != ""} {
	    set clusterHandle [$options(-clusterCollectionName) \
				   getClusterObject $options(-clusterName)]
	}

    }

    # Free the handles

    destructor {
        if {$options(-clusterCollectionName) != ""} {
	    itcl::delete object $clusterHandle
	}
    }
  
    
    # List of functions that do exactly what the name says!

    public method dropAggregate {} {

        set databaseHandle [$clusterHandle getDatabaseHandle \
				$options(-databaseName)]
	set aggregateHandle [$databaseHandle getAggregateHandle \
				$options(-deleteName)]

	set errorText [eval $aggregateHandle drop]
	itcl::delete object $databaseHandle
	itcl::delete object $aggregateHandle
	return $errorText
    }
    
    # Method drop comment drops the comment on a column. It receives the 
    # handle directly.

    public method dropComment {} {
        return [$options(-objectHandle) commentDrop]
    }

    public method dropConstraint {} {
	set databaseHandle [$clusterHandle getDatabaseHandle \
				$options(-databaseName)]
	set tableHandle [$databaseHandle getTableHandle \
			     $options(-tableName)]
	set constraintHandle [$tableHandle getConstraintHandle \
				  "$options(-uniqueConstraintIdentifyingName)" $options(-optionalIdentificationData)]

	# In case of constraints, dropping them drops some other
	# elements from the database. We need to make sure that
	# these elements are appropriately refreshed when a 
	# constraint is dropped. Before removing the constraint, we
	# gather the names of these elements.

	set constraintType [$constraintHandle getType]

	# Initialize the node list

	set additionalNodesToRefresh {}

	# For FK constraints

	if {[string compare $constraintType  "FK"] == 0} {

	    # First is the triggers on the current table itself

	    set additionalNodesToRefresh "{-objectType triggersHeader -clusterName {[$clusterHandle getName]} -databaseName {[$databaseHandle getName]} -tableName {[$tableHandle getName]}}"

	    # Next, get the name of the reference table.

	    lappend additionalNodesToRefresh "-objectType triggersHeader -clusterName {[$clusterHandle getName]} -databaseName {[$databaseHandle getName]} -tableName {[$constraintHandle getReferenceTable]}"

	    # set the triggers, we need to make sure there are 3, because only then will
	    # deletion take place.

	    set triggerList [$constraintHandle getFKTriggerInfo]
	}

	# For primary and unique constraints

	if {[string compare $constraintType  "UNIQUE"] == 0 || [string compare $constraintType  "PK"] == 0} {
	    
	    # In this case, we only need to refresh the indexes

	    set additionalNodesToRefresh "{-objectType indexesHeader -clusterName {[$clusterHandle getName]} -databaseName {[$databaseHandle getName]} -tableName {[$tableHandle getName]}}"

	}

	set errorText [eval $constraintHandle drop]

	itcl::delete object $databaseHandle
	itcl::delete object $tableHandle
	itcl::delete object $constraintHandle

	# Since we can return only once value (error message in this case), we
	# are unable to pass on the additional node names to the calling
	# function. Therefore, we need to initiate a refresh in here

	# If this was an FK constraint, make sure we had all the triggers

	if {[expr [string compare $constraintType "FK"] == 0] && [llength $triggerList] == 3} {
	    refreshAdditionalNodes "$additionalNodesToRefresh" "$errorText"
	} elseif [expr [string compare $constraintType "FK"] != 0] {
	    # Else just call the function
	    refreshAdditionalNodes "$additionalNodesToRefresh" "$errorText"
	}

	return $errorText
    }

    public method dropDatabase {} {

        if {$options(-deleteName) != "template1" } {
	    set databaseHandle [$clusterHandle getDatabaseHandle \
		$options(-deleteName)]
	    set interactorArgs [$databaseHandle getInteractorArgs]
	    itcl::delete object $databaseHandle
	    set result [eval SQL_pool closeAllConnectionsOfClass \
		\"$interactorArgs\"]
	    set databaseHandle [$clusterHandle getDatabaseHandle \
				    "template1"]
	    set errorText [eval $databaseHandle drop \
			       \"$options(-deleteName)\"]
	    itcl::delete object $databaseHandle
	    return $errorText
	} else {
	    # template1 can't be dropped, backend will complain
	    # it's a template
	    return "ERROR:  DROP DATABASE: database is \
                           marked as a template"
	}
    }

    public method dropDataType {} {

        set databaseHandle [$clusterHandle getDatabaseHandle \
				$options(-databaseName)]
	set dataTypeHandle [$databaseHandle getDataTypeHandle \
				$options(-deleteName)]
	set errorText [eval $dataTypeHandle drop]
	itcl::delete object $databaseHandle
	itcl::delete object $dataTypeHandle
	return $errorText
    }

    public method dropFunction {} {
		    
        set databaseHandle [$clusterHandle getDatabaseHandle \
				$options(-databaseName)]
	set functionHandle [$databaseHandle getFunctionHandle \
				$options(-deleteName)]
	set errorText [eval $functionHandle drop]
	itcl::delete object $databaseHandle
	itcl::delete object $functionHandle
	return $errorText
    }

    public method dropGroup {} {

        set groupHandle [$clusterHandle getGroupHandle \
			     $options(-deleteName)]
	set errorText [eval $groupHandle drop]
	itcl::delete object $groupHandle
	return $errorText
    }

    public method dropIndex {} {

        set databaseHandle [$clusterHandle getDatabaseHandle \
				$options(-databaseName)]
	set tableHandle [$databaseHandle getTableHandle \
			     $options(-deleteName)]
	set indexHandle [$tableHandle getIndexHandle \
			     $options(-deleteName)]

	# When an index is dropped, constraints may change.
	# Check if this index is a type of constraint

	set additionalNodesToRefresh {}

	if {[string compare [$indexHandle isPrimaryIndex] "t"] == 0 || [string compare [$indexHandle isUniqueIndex] "t"] == 0 } {
	    set additionalNodesToRefresh "{-objectType constraintsHeader -clusterName {[$clusterHandle getName]} -databaseName {[$databaseHandle getName]} -tableName {[$indexHandle getParentTable]}}"
	}

	set errorText [eval $indexHandle drop]
	itcl::delete object $databaseHandle
	itcl::delete object $tableHandle
	itcl::delete object $indexHandle 

	# Call refresh. Don't worry if the list is empty. The
	# refreshAdditionalNodes() function will just ignore it in 
	# that case.

	refreshAdditionalNodes "$additionalNodesToRefresh" "$errorText"
	
	return $errorText
    }

    public method dropLanguage {} {

        set databaseHandle [$clusterHandle getDatabaseHandle \
				$options(-databaseName)]
	set languageHandle [$databaseHandle getLanguageHandle \
				$options(-deleteName)]
	set errorText [eval $languageHandle drop]
	itcl::delete object $databaseHandle
	itcl::delete object $languageHandle
	return $errorText
    }
 
    public method dropOperator {} {

         set databaseHandle [$clusterHandle getDatabaseHandle \
				 $options(-databaseName)]
	 set operatorHandle [$databaseHandle getOperatorHandle \
				 $options(-deleteName)]
	 set errorText [eval $operatorHandle drop]
	 itcl::delete object $databaseHandle
	 itcl::delete object $operatorHandle
	 return $errorText
     }
 
    public method dropRule {} {

        set databaseHandle [$clusterHandle getDatabaseHandle \
				$options(-databaseName)]
	# determine if we are deleting a rule under a table/view and
	# create the handles accordingly
	if {$options(-tableName) != ""} {
	    set tableHandle [$databaseHandle getTableHandle \
			     $options(-tableName)]
	    set ruleHandle [$tableHandle getRuleHandle \
			    $options(-deleteName)]
	} else {
	    set viewHandle [$databaseHandle getViewHandle \
			     $options(-viewName)]
	    set ruleHandle [$viewHandle getRuleHandle \
			    $options(-deleteName)]
	}
	set errorText [eval $ruleHandle drop]
	itcl::delete object $databaseHandle
	# determine if we are deleting a rule under a table/view and
	# delete the handles accordingly
	if {$options(-tableName) != ""} {
	    itcl::delete object $tableHandle
	} else {
	    itcl::delete object $viewHandle
	}
	itcl::delete object $ruleHandle 
	return $errorText
    }

    public method dropSequence {} {

        set databaseHandle [$clusterHandle getDatabaseHandle \
				$options(-databaseName)]
	set sequenceHandle [$databaseHandle getSequenceHandle \
				$options(-deleteName)]
	set errorText [eval $sequenceHandle drop]
	itcl::delete object $databaseHandle
	itcl::delete object $sequenceHandle
	return $errorText
    }

    public method dropTable {} {

         set databaseHandle [$clusterHandle getDatabaseHandle \
				 $options(-databaseName)]
	 set tableHandle [$databaseHandle getTableHandle \
			      $options(-deleteName)]
	 set errorText [eval $tableHandle drop]
	 itcl::delete object $databaseHandle
	 itcl::delete object $tableHandle
	 return $errorText
     }
    

    public method dropTrigger {} {

	set databaseHandle [$clusterHandle getDatabaseHandle \
	    $options(-databaseName)]
	set tableHandle [$databaseHandle getTableHandle \
	    $options(-tableName)]
	set triggerHandle [$tableHandle getTriggerHandle \
	    $options(-deleteName)]

	# If this is an RI ins trigger, we need to refresh the 
	# constraints as well

	set additionalNodesToRefresh {}

	if {[string compare RI_FKey_check_ins() [$triggerHandle getProcedureSignature]] == 0 && [string compare [$triggerHandle isConstraint] "t"] == 0} {
	    set additionalNodesToRefresh "{-objectType constraintsHeader -clusterName {[$clusterHandle getName]} -databaseName {[$databaseHandle getName]} -tableName {[$triggerHandle getParentTable]}}"
	}

	set errorText [eval $triggerHandle drop \"$options(-tableName)\"]
	itcl::delete object $databaseHandle
	itcl::delete object $tableHandle
	itcl::delete object $triggerHandle

	# Call refresh. Don't worry if the list is empty. The
	# refreshAdditionalNodes() function will just ignore it in 
	# that case.

	refreshAdditionalNodes "$additionalNodesToRefresh" "$errorText"

	return $errorText
     }

    public method dropUser {} {

         set userHandle [$clusterHandle getUserHandle \
			     $options(-deleteName)]
	 set errorText [eval $userHandle drop]
	 itcl::delete object $userHandle
	 return $errorText
     }

    public method dropView {} {

         set databaseHandle [$clusterHandle getDatabaseHandle \
				 $options(-databaseName)]
	 set viewHandle [$databaseHandle getViewHandle \
			     $options(-deleteName)]
	 set errorText [eval $viewHandle drop]
	 itcl::delete object $databaseHandle
	 itcl::delete object $viewHandle
	 return $errorText
     }

    public method refreshAdditionalNodes {nodesToRefresh errorMsg} {
	
	# If there was an error, return

	if {[string compare $errorMsg ""] != 0} {
	    return
	}

	# Else continue

	for { set i 0 } { $i < [llength $nodesToRefresh] } { incr i } {
	    #puts "refreshing: [lindex $nodesToRefresh $i]"
	    eval eval $options(-refreshAnotherLevelCommand) \"{[lindex $nodesToRefresh $i]}\"
	}
    }

    private variable clusterHandle
    private variable errorText
    private variable options

}
