# Copyright (c) 2002 Red Hat, Inc. All rights reserved.
#
# This software may be freely redistributed under the terms of the
# GNU General Public License.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# Handler for handling the context menu driven COMMENT ON operations
#
# Written for Red Hat Inc. by Benjamin Mar Kuck <bmarkuck@redhat.com>
#
# Component of: Red Hat Database GUI Administration tool

package require getComment
package provide commentOnContextMenuHandler 1.0

proc commentOnHandler { args } {

    set expectedArgs \
        [list \
        -clusterCollectionName\
        -clusterName\
        -databaseName\
        -tableName\
        -indexName \
        -dataTypeName \
        -ruleName \
        -viewName \
        -triggerName \
        -sequenceName \
        -columnName \
        -functionName \
        -aggregateName \
        -operatorName \
        -constraintName \
        -uniqueConstraintIdentifyingName \
        -optionalIdentificationData \
        ]

    foreach option $expectedArgs {
        set options($option) ""
    }

    foreach { option value } $args {
        if {[lsearch -exact $expectedArgs $option] == -1} {
            error "Unexpected option \"$option\" - only expect $expectedArgs"
        }
        set options($option) "$value"
    }

    # Get the cluster name and database name, since this is needed no matter what object we want to
    # comment on

    set clusterHandle [$options(-clusterCollectionName) getClusterObject $options(-clusterName)]
    set databaseHandle [$clusterHandle getDatabaseHandle $options(-databaseName)]

    # Catch any error that may occur when creating handles

    set errorLevel [catch {

	# If a table name is specified, the comment is either on a table, column, index, rule, or trigger

	if {$options(-tableName) != ""} {
	    set tableHandle [$databaseHandle getTableHandle $options(-tableName)]

	    # Check whether the comment is on a column, index, rule, or trigger; Otherwise the comment is
	    # simply on the table itself
	
	    if {$options(-columnName) != ""} {
		set columnHandle [$tableHandle getColumnHandle $options(-columnName)]

		# When setting the title, the name should not be escaped. Since we
		# pass names around in escaped form, we need to unescape it here.

		set unescapedName [join [split $options(-columnName) \\] ""]

		getComment "COMMENT ON COLUMN $unescapedName" "column" "Comment:" $columnHandle

		itcl::delete object $columnHandle
	    } elseif {$options(-indexName) != ""} {
		set indexHandle [$tableHandle getIndexHandle $options(-indexName)]

		set unescapedName [join [split $options(-indexName) \\] ""]

		getComment "COMMENT ON INDEX $unescapedName" "index" "Comment:" $indexHandle

		itcl::delete object $indexHandle
	    } elseif {$options(-ruleName) != ""} {
		set ruleHandle [$tableHandle getRuleHandle $options(-ruleName)]

		set unescapedName [join [split $options(-ruleName) \\] ""]

		getComment "COMMENT ON RULE $unescapedName" "rule" "Comment:" $ruleHandle

		itcl::delete object $ruleHandle
	    } elseif {$options(-triggerName) != ""} {
		set triggerHandle [$tableHandle getTriggerHandle $options(-triggerName)]

		set unescapedName [join [split $options(-triggerName) \\] ""]

		getComment "COMMENT ON TRIGGER $unescapedName" "trigger" "Comment:" $triggerHandle

		itcl::delete object $triggerHandle
	    } elseif {$options(-constraintName) != ""} {
		set constraintHandle [$tableHandle getConstraintHandle \
					  "$options(-uniqueConstraintIdentifyingName)" $options(-optionalIdentificationData)]

		set unescapedName [join [split $options(-constraintName) \\] ""]

		getComment "COMMENT ON CONSTRAINT $unescapedName" "constraint" "Comment:" $constraintHandle
	    } else {

		set unescapedName [join [split $options(-tableName) \\] ""]

		getComment "COMMENT ON TABLE $unescapedName" "table" "Comment:" $tableHandle
	    }

	    itcl::delete object $tableHandle
	

	# If a table name is not specified, then the comment is either on a data type, view, sequence,
	# function, aggregate, operator, database, or a rule on a view

	} elseif {$options(-dataTypeName) != ""} {
	    set dataTypeHandle [$databaseHandle getDataTypeHandle $options(-dataTypeName)]

	    set unescapedName [join [split $options(-dataTypeName) \\] ""]

	    getComment "COMMENT ON TYPE $unescapedName" "dataType" "Comment:" $dataTypeHandle

	    itcl::delete object $dataTypeHandle
	} elseif {$options(-viewName) != "" && $options(-ruleName) == ""} {
	    set viewHandle [$databaseHandle getViewHandle $options(-viewName)]

	    set unescapedName [join [split $options(-viewName) \\] ""]

	    getComment "COMMENT ON VIEW $unescapedName" "view" "Comment:" $viewHandle

	    itcl::delete object $viewHandle
	} elseif {$options(-viewName) != "" && $options(-ruleName) != ""} {
	    set viewHandle [$databaseHandle getViewHandle $options(-viewName)]
	    set ruleHandle [$viewHandle getRuleHandle $options(-ruleName)]

	    set unescapedName [join [split $options(-ruleName) \\] ""]

	    getComment "COMMENT ON RULE $unescapedName" "ruleOnView" "Comment:" $ruleHandle

	    itcl::delete object $viewHandle
	    itcl::delete object $ruleHandle
	} elseif {$options(-sequenceName) != ""} {
	    set sequenceHandle [$databaseHandle getSequenceHandle $options(-sequenceName)]

	    set unescapedName [join [split $options(-sequenceName) \\] ""]

	    getComment "COMMENT ON SEQUENCE $unescapedName" "sequence" "Comment:" $sequenceHandle

	    itcl::delete object $sequenceHandle
	} elseif {$options(-functionName) != ""} {
	    set functionHandle [$databaseHandle getFunctionHandle $options(-functionName)]

	    set unescapedName [join [split $options(-functionName) \\] ""]

	    getComment "COMMENT ON FUNCTION $unescapedName" "function" "Comment:" $functionHandle

	    itcl::delete object $functionHandle
	} elseif {$options(-aggregateName) != ""} {
	    set aggregateHandle [$databaseHandle getAggregateHandle $options(-aggregateName)]

	    set unescapedName [join [split $options(-aggregateName) \\] ""]

	    getComment "COMMENT ON AGGREGATE $unescapedName" "aggregate" "Comment:" $aggregateHandle

	    itcl::delete object $aggregateHandle
	} elseif {$options(-operatorName) != ""} {
	    set operatorHandle [$databaseHandle getOperatorHandle $options(-operatorName)]

	    set unescapedName [join [split $options(-operatorName) \\] ""]

	    getComment "COMMENT ON OPERATOR $unescapedName" "operator" "Comment:" $operatorHandle

	    itcl::delete object $operatorHandle
	} else {

	    set unescapedName [join [split $options(-databaseName) \\] ""]

	    getComment "COMMENT ON DATABASE $unescapedName" "database" "Comment:" $databaseHandle
	}

	itcl::delete object $clusterHandle
	itcl::delete object $databaseHandle

    } errorText]

    if {$errorLevel != 0} {
	tk_messageBox -title Error -icon error -message "An error has occurred when preforming this operation: $errorText\nCommenting on this object is not possible."
    }

}

# End of file.
