# Copyright (c) 2002 Red Hat, Inc. All rights reserved.
#
# This software may be freely redistributed under the terms of the
# GNU General Public License.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# TCL code to raise a dialog box containing an ALTER / CREATE dialog
#
# Written for Red Hat Inc. by Neil Padgett <npadgett@redhat.com>
#
# Component of: Red Hat Database GUI Administration tool

package require Itcl
package require Itk
package require Iwidgets
package require megaAlterCreateDialog
package require identifier
package provide alterCreateDialogRaiser 1.0

proc alterCreateDialogRaise { windowTitle dialogPrompt args } {
    global setting_autoQuoteIdentifiers

    set errorLevel 0
    
    # This release always auto quote identifiers
    
    set setting_autoQuoteIdentifiers 1

    # Create the dialog window

    toplevel .dialog

    # Withdraw the window while we pack it

    wm withdraw .dialog

    # Disable the main program window

    wm transient .dialog .
    catch {grab .dialog}
   
    # Create and pack an entry widget if mode is CREATE (i.e., dialogPrompt != "")

    if {$dialogPrompt != ""} {
    
        # Set an input mask accordingly to preferences and type of object
	
        if {$setting_autoQuoteIdentifiers} {
            set inputMask ".*"
        } else {

	    # This is an ugly hack

	    if {[string first "CREATE OPERATOR" $windowTitle] > 0} {
	        set inputMask "{+-*/<>=~!@\#%^&|\'?\$}"
	    } else {
                set inputMask "{^\[a-zA-Z\"\]\$}"
	    }
        }
	
	iwidgets::regexpfield .dialog.entryBox -labeltext $dialogPrompt \
	    -textbackground white -width 33 -fixed 31 -regexp $inputMask \
	    -focuscommand "alterCreateDialog_edit {}" \
	    -command "alterCreateDialog_edit {}"
	pack .dialog.entryBox -side top -fill x -pady 5
	
	# When the box appears to enter the name, immediately give focus to the
	# text box.

	set comp [.dialog.entryBox component entry]
	focus $comp
	
	# Bind some events to keep object name (and window title) in synch

	bind $comp <KeyRelease> "+ alterCreateDialog_edit {}"
	bind $comp <Leave> "+ alterCreateDialog_edit {}"
	bind $comp <Enter> "+ alterCreateDialog_edit {}"

	# For some events we don't get the current value with %P,
	# so we would have to get it from the widget ourselves.
	# But this would not include the current keystroke as it
	# has not been validated yet.  So we ask the interpreter
	# to run the alterCreateDialog_edit routine at later time.

	bind $comp <ButtonRelease-2> "+ after idle [list alterCreateDialog_edit \{\}]"
	bind $comp <ButtonRelease-2> "+ after 200 [list alterCreateDialog_edit \{\}]"
	bind $comp <<Paste>> "+ after 200 [list alterCreateDialog_edit \{\}]"
	bind $comp <BackSpace> "+ after idle [list alterCreateDialog_edit \{\}]"
	bind $comp <Delete> "+ after idle [list alterCreateDialog_edit \{\}]"
    }

    # Create a frame to hold the megaAlterCreateDialog
    
    frame .dialog.hostedDialog -relief groove -borderwidth 1

    # Place the megaAlterCreateDialog in the window inside the frame

    if { [catch {eval MegaAlterCreateDialog .dialog.hostedDialog.megaDialog -parentDialog .dialog $args } errorText] } {
	set errorLevel 1
    }

    # Pack the frame

    if { $errorLevel == 0 } {
	pack .dialog.hostedDialog.megaDialog -fill both -expand y
	pack .dialog.hostedDialog -side top -fill both -expand y
    }

    # Create and pack OK and Cancel Buttons
    # On CREATE (i.e., dialogPrompt != "") we have some more work to do.

    if {$dialogPrompt != ""} {
        button .dialog.okButton -text "OK" -underline 0 -command alterCreateDialog_create_ok
    } else {
	button .dialog.okButton -text "OK" -underline 0 -command alterCreateDialog_ok
    }
    
    button .dialog.cancelButton -text "Cancel" -underline 0 -command alterCreateDialog_cancel
    pack .dialog.cancelButton .dialog.okButton -side right -pady 5

    bind .dialog <Alt-o> {alterCreateDialog_ok; break}
    bind .dialog <Alt-c> {alterCreateDialog_cancel; break}
    bind .dialog.okButton <Return> {alterCreateDialog_ok; break}
    bind .dialog.cancelButton <Return> {alterCreateDialog_cancel; break}
    bind .dialog <Escape> {alterCreateDialog_cancel; break}

    if { $errorLevel != 0 } {
	    
	# We need to contact the SQL pool to request that it
	# close all database connections so that we can recover from
	# the connection error.
	#
	# Note the SQL pool name is hard coded here.
	    
	SQL_pool closeAllConnections

	# Close the dialog

	alterCreateDialog_cancel

	# Report the error
	
	tk_messageBox -icon error -message "An error occurred while raising the dialog box:\n $errorText\nThe operation was aborted." -title "Error"
    } else {

	# Set the window title
	
	wm title .dialog $windowTitle

	wm deiconify .dialog
    }
}

proc alterCreateDialog_edit { objName } {
    global setting_autoQuoteIdentifiers

    # For some events we don't get the current value, so we
    # have to get it from the widget ourselves.
    
    if {$objName == ""} {
        set objName [.dialog.entryBox get]
    } else {

	# Come back again, after the new value has settled down
	# We need this due to an error in the value passed by %P 
	# when somebody selects the test and types one letter
	# that replaces the selected text
	after idle [list alterCreateDialog_edit {}]
    }
    
    # Interrogate the megaAlterCreateDialog for the window title
    
    set windowTitle [.dialog.hostedDialog.megaDialog getTitle]

    # Check for delimited identifiers if not auto quoting
    # and we are not creating operators

    if {(!$setting_autoQuoteIdentifiers) &&
	([string first "CREATE OPERATOR" $windowTitle] < 0)} {
        if {$objName != ""} {
            if {[string index $objName 0] == {"}} {
                .dialog.entryBox configure -regexp {^["][^"]*["]*$} -fixed 33
            } else {
                .dialog.entryBox configure -regexp {^[a-zA-Z][a-zA-Z0-9_]*$} -fixed 31
            }
        } else {
             .dialog.entryBox configure -regexp {^[a-zA-Z"]$}
        }
    }

    # pass down new name
    
    .dialog.hostedDialog.megaDialog configure -objectName $objName

    # Set the window title to the new name
    
    wm title .dialog [.dialog.hostedDialog.megaDialog getTitle]

    return true
}    

proc alterCreateDialog_create_ok {} {

    # Get the latest and greatest from the entry widget
    # if we are in CREATE mode (i.e., we have a .dialog.entryBox)
    
    if {[winfo exists .dialog.entryBox]} {
        alterCreateDialog_edit {}
    }
    
    # First, validate the identifier
    
    set objName [.dialog.entryBox get]
    set windowTitle [.dialog.hostedDialog.megaDialog getTitle]
    
    if {[string first "CREATE OPERATOR" $windowTitle] >= 0} {
        set errorText [validateOperatorName $objName]
    } else {
        set errorText [validateIdentifier $objName]
    }

    if {$errorText != "" } {

	# No, so display error message

	tk_messageBox -icon error -title "Error" \
		-message "$errorText"
	
	return
    }
    
    # Then, validate the form and apply if possible

    alterCreateDialog_ok
}

proc alterCreateDialog_ok {} {

    # Get the latest and greatest from the entry widget
    # if we are in CREATE mode (i.e., we have a .dialog.entryBox)
    
    if {[winfo exists .dialog.entryBox]} {
        alterCreateDialog_edit {}
    }
    
    # First, validate the form

    set errorText [.dialog.hostedDialog.megaDialog validate]

    if {$errorText != "" } {

	# No, so display error message

	tk_messageBox -icon error -title "Error" \
		-message "$errorText"
	
	return
    }

    # Now try and execute the action (apply the form)

    # Start the busy indicator

    status loadNotify 1

    # Do the deed

    if {[catch {set errorText [.dialog.hostedDialog.megaDialog apply]} errorText2]} {

	# Stop the busy indicator

	status loadNotify 0

	tk_messageBox -icon error -title "Error" \
	    -message "An error occurred while processing this directive:\n$errorText2\nThe operation was aborted."
	status loadNotify 0
	return
    }

    # Stop the busy indicator

    status loadNotify 0

    # Check if everything went cleanly

    if {$errorText != "" } {

	# No, so display error message

	tk_messageBox -icon error -title "Error" \
		-message "The backend returned the error:\n $errorText"
    } else {
	
	# Yes, so lower the dialog (just use the cancel procedure)

	alterCreateDialog_cancel

	# Things have probably changed, so issue a refresh
	# Note that this hardcodes the hyperlink dispatcher / hyperlink
	# watcher names

	hyperlinker selectorMoveRequest [hyperlinkWatcher getCurrentLocation]
    }
    
}

proc alterCreateDialog_cancel {} {
    destroy .dialog
}

# End of file
