/*
 * Copyright (c) 1998, 1999, 2000 Phil Thompson (phil@river-bank.demon.co.uk)
 *
 * The main header file for SIP.
 */


#ifndef SIP_H
#define SIP_H

#include <stdio.h>
#include <sys/types.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif


#ifdef TRUE
#undef TRUE
#endif

#ifdef FALSE
#undef FALSE
#endif

#define	TRUE	1
#define	FALSE	0


#define	MAX_NR_ARGS		20	/* Max. nr. args. to a function. */


/* For convenience. */

#define	classVersName(cvd)	((cvd) -> cd -> name -> text)


/* Handle section flags. */

#define	SECT_IS_PUBLIC		0x0001	/* It is public. */
#define	SECT_IS_PROT		0x0002	/* It is protected. */
#define	SECT_IS_PRIVATE		0x0004	/* It is private. */
#define	SECT_IS_SLOT		0x0008	/* It is a slot. */
#define	SECT_IS_SIGNAL		0x0010	/* It is a signal. */
#define	SECT_IS_PYMETHOD	0x0020	/* It is a Python method. */
#define	SECT_IS_PYNUMMETHOD	0x0040	/* It is a Python number method. */
#define	SECT_IS_PYSEQMETHOD	0x0080	/* It is a Python sequence method. */
#define	SECT_IS_PYMAPMETHOD	0x0100	/* It is a Python sequence method. */
#define	SECT_IS_PYCMPMETHOD	0x0200	/* It is a Python compare method. */


/* Handle class version flags. */

#define	CLASSVERS_IS_MAPPED	0x001	/* It is mapped. */
#define	CLASSVERS_HAS_SIGSLOTS	0x002	/* It has signals or slots. */
#define	CLASSVERS_CANNOT_CREATE	0x004	/* It can't be created by Python. */
#define	CLASSVERS_IS_COMPLEX	0x008	/* It is complex. */
#define	CLASSVERS_NO_PUB_DTOR	0x010	/* No public dtor. */
#define	CLASSVERS_IS_DTOR	0x020	/* There is an explicit dtor. */
#define	CLASSVERS_IS_OPAQUE	0x040	/* It is opaque. */

#define	isMapped(cv)		((cv) -> classversflags & CLASSVERS_IS_MAPPED)
#define	setIsMapped(cv)		((cv) -> classversflags |= CLASSVERS_IS_MAPPED)
#define	hasSigSlots(cv)		((cv) -> classversflags & CLASSVERS_HAS_SIGSLOTS)
#define	setHasSigSlots(cv)	((cv) -> classversflags |= CLASSVERS_HAS_SIGSLOTS)
#define	cannotCreate(cv)	((cv) -> classversflags & CLASSVERS_CANNOT_CREATE)
#define	setCannotCreate(cv)	((cv) -> classversflags |= CLASSVERS_CANNOT_CREATE)
#define	resetCannotCreate(cv)	((cv) -> classversflags &= ~CLASSVERS_CANNOT_CREATE)
#define	isComplex(cv)		((cv) -> classversflags & CLASSVERS_IS_COMPLEX)
#define	setIsComplex(cv)	((cv) -> classversflags |= CLASSVERS_IS_COMPLEX)
#define	noPublicDtor(cv)	((cv) -> classversflags & CLASSVERS_NO_PUB_DTOR)
#define	setNoPublicDtor(cv)	((cv) -> classversflags |= CLASSVERS_NO_PUB_DTOR)
#define	isDtor(cv)		((cv) -> classversflags & CLASSVERS_IS_DTOR)
#define	setIsDtor(cv)		((cv) -> classversflags |= CLASSVERS_IS_DTOR)
#define	isOpaque(cv)		((cv) -> classversflags & CLASSVERS_IS_OPAQUE)
#define	setIsOpaque(cv)		((cv) -> classversflags |= CLASSVERS_IS_OPAQUE)


/* Handle class flags. */

#define	CLASS_IS_USED		0x01	/* It is used. */
#define	CLASS_IS_STARTED	0x02	/* Class has been started in a walk. */
#define	CLASS_IS_DONE		0x04	/* Class has been done in a walk. */

#define	isUsed(c)		((c) -> classflags & CLASS_IS_USED)
#define	setIsUsed(c)		((c) -> classflags |= CLASS_IS_USED)
#define	isWalkStarted(c)	((c) -> classflags & CLASS_IS_STARTED)
#define	isWalkDone(c)		((c) -> classflags & CLASS_IS_DONE)
#define	setWalkStarted(c)	((c) -> classflags |= CLASS_IS_STARTED)
#define	setWalkDone(c)		((c) -> classflags |= CLASS_IS_DONE)
#define	resetWalk(c)		((c) -> classflags &= ~(CLASS_IS_STARTED|CLASS_IS_DONE))


/* Handle ctor flags. */

#define	isPublicCtor(c)		((c) -> ctorflags & SECT_IS_PUBLIC)
#define	setIsPublicCtor(c)	((c) -> ctorflags |= SECT_IS_PUBLIC)
#define	isProtectedCtor(c)	((c) -> ctorflags & SECT_IS_PROT)
#define	setIsProtectedCtor(c)	((c) -> ctorflags |= SECT_IS_PROT)
#define	isPrivateCtor(c)	((c) -> ctorflags & SECT_IS_PRIVATE)
#define	setIsPrivateCtor(c)	((c) -> ctorflags |= SECT_IS_PRIVATE)


/* Handle enum flags. */

#define	isProtectedEnum(e)	((e) -> enumflags & SECT_IS_PROT)


/* Handle overload flags. */

#define	OVER_IS_VIRTUAL		0x00010000	/* It is virtual. */
#define	OVER_IS_ABSTRACT	0x00020000	/* It is abstract. */
#define	OVER_IS_CONST		0x00040000	/* It is a const function. */
#define	OVER_IS_STATIC		0x00080000	/* It is a static function. */
#define	OVER_IS_BLOCKING	0x00100000	/* It is a blocking function. */

#define	isPublic(o)		((o) -> overflags & SECT_IS_PUBLIC)
#define	setIsPublic(o)		((o) -> overflags |= SECT_IS_PUBLIC)
#define	isProtected(o)		((o) -> overflags & SECT_IS_PROT)
#define	setIsProtected(o)	((o) -> overflags |= SECT_IS_PROT)
#define	isPrivate(o)		((o) -> overflags & SECT_IS_PRIVATE)
#define	setIsPrivate(o)		((o) -> overflags |= SECT_IS_PRIVATE)
#define	isSlot(o)		((o) -> overflags & SECT_IS_SLOT)
#define	setIsSlot(o)		((o) -> overflags |= SECT_IS_SLOT)
#define	isSignal(o)		((o) -> overflags & SECT_IS_SIGNAL)
#define	setIsSignal(o)		((o) -> overflags |= SECT_IS_SIGNAL)

#define	isVirtual(o)		((o) -> overflags & OVER_IS_VIRTUAL)
#define	setIsVirtual(o)		((o) -> overflags |= OVER_IS_VIRTUAL)
#define	isAbstract(o)		((o) -> overflags & OVER_IS_ABSTRACT)
#define	setIsAbstract(o)	((o) -> overflags |= OVER_IS_ABSTRACT)
#define	isConst(o)		((o) -> overflags & OVER_IS_CONST)
#define	setIsConst(o)		((o) -> overflags |= OVER_IS_CONST)
#define	isStatic(o)		((o) -> overflags & OVER_IS_STATIC)
#define	setIsStatic(o)		((o) -> overflags |= OVER_IS_STATIC)
#define	isBlocking(o)		((o) -> overflags & OVER_IS_BLOCKING)
#define	setIsBlocking(o)	((o) -> overflags |= OVER_IS_BLOCKING)


/* Handle variable flags. */

#define	VAR_IS_STATIC		0x01	/* It is a static variable. */

#define	isStaticVar(v)		((v) -> varflags & VAR_IS_STATIC)
#define	setIsStaticVar(v)	((v) -> varflags |= VAR_IS_STATIC)


/* Handle argument flags. */

#define	ARG_IS_REF		0x01	/* It is a reference. */
#define	ARG_IS_CONST		0x02	/* It is a const. */
#define	ARG_XFERRED		0x04	/* Ownership is transferred. */
#define	ARG_THIS_XFERRED	0x08	/* Ownership of this is transferred. */
#define	ARG_XFERRED_BACK	0x10	/* Ownership is transferred back. */

#define	isReference(a)		((a) -> argflags & ARG_IS_REF)
#define	setIsReference(a)	((a) -> argflags |= ARG_IS_REF)
#define	isConstArg(a)		((a) -> argflags & ARG_IS_CONST)
#define	setIsConstArg(a)	((a) -> argflags |= ARG_IS_CONST)
#define	isTransferred(a)	((a) -> argflags & ARG_XFERRED)
#define	setIsTransferred(a)	((a) -> argflags |= ARG_XFERRED)
#define	isThisTransferred(a)	((a) -> argflags & ARG_THIS_XFERRED)
#define	setIsThisTransferred(a)	((a) -> argflags |= ARG_THIS_XFERRED)
#define	isTransferredBack(a)	((a) -> argflags & ARG_XFERRED_BACK)
#define	setIsTransferredBack(a)	((a) -> argflags |= ARG_XFERRED_BACK)


/* Special Python method IDs. */

#define	SPM_METH_REPR		0x00000001
#define	SPM_METH_STR		0x00000002
#define	SPM_METH_CMP		0x00000004
#define	SPM_METH_HASH		0x00000008
#define	SPM_METH_CALL		0x00000010
#define	SPM_METH_RICH		0x80000000	/* Must be different from SPM_SEQ_*. */

#define	SPM_NUM_ADD		0x00000001
#define	SPM_NUM_SUB		0x00000002
#define	SPM_NUM_MUL		0x00000004
#define	SPM_NUM_DIV		0x00000008
#define	SPM_NUM_MOD		0x00000010
#define	SPM_NUM_DIVMOD		0x00000020
#define	SPM_NUM_POW		0x00000040
#define	SPM_NUM_NEG		0x00000080
#define	SPM_NUM_POS		0x00000100
#define	SPM_NUM_ABS		0x00000200
#define	SPM_NUM_NONZERO		0x00000400
#define	SPM_NUM_INVERT		0x00000800
#define	SPM_NUM_LSHIFT		0x00001000
#define	SPM_NUM_RSHIFT		0x00002000
#define	SPM_NUM_AND		0x00004000
#define	SPM_NUM_XOR		0x00008000
#define	SPM_NUM_OR		0x00010000
#define	SPM_NUM_COERCE		0x00020000	/* Must be different from SPM_SEQ_*. */
#define	SPM_NUM_INT		0x00040000
#define	SPM_NUM_LONG		0x00080000
#define	SPM_NUM_FLOAT		0x00100000
#define	SPM_NUM_OCT		0x00200000
#define	SPM_NUM_HEX		0x00400000
#define	SPM_NUM_IADD		0x00800000
#define	SPM_NUM_ISUB		0x01000000
#define	SPM_NUM_IMUL		0x02000000
#define	SPM_NUM_IDIV		0x04000000
#define	SPM_NUM_IMOD		0x08000000
#define	SPM_NUM_IPOW		0x10000000
#define	SPM_NUM_ILSHIFT		0x20000000
#define	SPM_NUM_IRSHIFT		0x40000000
#define	SPM_NUM_IAND		0x80000000
#define	SPM_NUM_IXOR		0x00000001	/* Use the overspill. */
#define	SPM_NUM_IOR		0x00000002

#define	SPM_SEQ_LEN		0x00000001
#define	SPM_SEQ_CONCAT		0x00000002
#define	SPM_SEQ_REPEAT		0x00000004
#define	SPM_SEQ_GETITEM		0x00000008
#define	SPM_SEQ_SETITEM		0x00000010
#define	SPM_SEQ_GETSLICE	0x00000020
#define	SPM_SEQ_SETSLICE	0x00000040
#define	SPM_SEQ_CONTAINS	0x00000080
#define	SPM_SEQ_ICONCAT		0x00000100
#define	SPM_SEQ_IREPEAT		0x00000200

#define	SPM_MAP_LEN		0x00000001
#define	SPM_MAP_GETITEM		0x00000002
#define	SPM_MAP_SETITEM		0x00000004


/* Argument types. */

typedef enum {
	no_type,
	defined_type,
	class_type,
	struct_type,
	voidptr_type,
	enum_type,
	template_type,
	rxcon_type,
	rxdis_type,
	slotcon_type,
	slotdis_type,
	ustring_type,
	string_type,
	short_type,
	ushort_type,
	int_type,
	uint_type,
	long_type,
	ulong_type,
	float_type,
	double_type,
	bool_type,
	array_type,
	uarray_type,
	arraysize_type,
	arrayusize_type
} argType;


/* Value types. */

typedef enum {
	qchar_value,
	string_value,
	numeric_value,
	real_value,
	scoped_value,
	fcall_value
} valueType;


/* An exposed function. */

typedef struct _expFuncDef {
	char			*name;		/* The function name. */
	struct _expFuncDef	*next;		/* Next in the list. */
} expFuncDef;


/* A version qualifier. */

typedef struct _versionQual {
	char			*name;		/* The version name. */
	int			order;		/* Order if it is a primary. */
	char			*vqcpp;		/* The C++ qualifier. */
	char			*vqpy;		/* The Python qualifier. */
	struct _versionQual	*next;		/* Next in the list. */
} versionQual;


/*
 * An anded version qualifier of an optional primary range and an optional
 * secondary.
 */

typedef struct {
	int			lowidx;		/* The primary lower index. */
	int			uppidx;		/* The primary upper index. */
	versionQual		*secondary;	/* The secondary. */
} versionAnded;


/* A list of ored version and lists. */

typedef struct _versionOrList {
	versionAnded		va;		/* The version and. */
	struct _versionOrList	*next;		/* Next in the list. */
} versionOrList;


/* A name. */

typedef struct _nameDef {
	char			*text;		/* The text of the name. */
	struct _moduleDef	*module;	/* The defining module. */
	struct _nameDef		*next;		/* Next in the list. */
} nameDef;


/* A module definition. */

typedef struct _moduleDef {
	nameDef			*name;		/* The module name. */
	char			*file;		/* The filename. */
	int			depth;		/* Depth in the hierachy. */
	struct _moduleDef	*next;		/* Next in the list. */
} moduleDef;


/* A literal code block. */

typedef struct _codeBlock {
	char			*frag;		/* The code itself. */
	char			*filename;	/* The original file. */
	int			linenr;		/* The line in the file. */
	versionAnded		version;	/* The version qualifier. */
	struct _codeBlock	*next;		/* Next in the list. */
} codeBlock;


/* A Makefile template definition. */

typedef struct _mkTemplateDef {
	char			*name;		/* The template name. */
	char			*objext;	/* The object file extension. */
	codeBlock		*templ;		/* The template itself. */
	struct _mkTemplateDef	*next;		/* Next in the list. */
} mkTemplateDef;


/* A special Python method. */

typedef struct {
	char		*name;			/* The method name. */
	int		id;			/* The method ID. */
	int		idxtra;			/* The extra method ID. */
	char		*rettype;		/* The return type. */
	int		nrargs;			/* The number of arguments. */
	unsigned	pyvers;			/* The Python version. */
} specialPyMethod;


/* An entry in a list of defined special Python methods. */

typedef struct _spmDef {
	specialPyMethod	*spm;			/* The actual method detail. */
	codeBlock	*code;			/* The implementation code. */
	struct _spmDef	*next;			/* Next in the list. */
} spmDef;


/* A scoped name. */

typedef struct _scopedNameDef {
	char			*name;		/* The name. */
	struct _scopedNameDef	*next;		/* Next in the scope list. */
} scopedNameDef;


/* A function call. */

typedef struct _fcallDef {
	scopedNameDef		*name;		/* The name. */
	int			nrArgs;		/* The number of arguments. */
	struct _valueDef	*args[MAX_NR_ARGS];	/* The arguments. */
} fcallDef;


/* A value. */

typedef struct _valueDef {
	valueType		vtype;		/* The type. */
	char			vunop;		/* Any unary operator. */
	char			vbinop;		/* Any binary operator. */
	union {
		char		vqchar;		/* Quoted character value. */
		long		vnum;		/* Numeric value. */
		double		vreal;		/* Real value. */
		nameDef		*vstr;		/* String value. */
		scopedNameDef	*vscp;		/* Scoped value. */
		fcallDef	*fcd;		/* Function call. */
	} u;
	struct _valueDef	*next;		/* Next in the expression. */
} valueDef;


/* A member function argument (or result). */

typedef struct {
	argType			atype;		/* The type. */
	int			argflags;	/* The argument flags. */
	int			nrderefs;	/* Nr. of dereferences. */
	valueDef		*defval;	/* The default value. */
	union {
		struct _funcArgs	*sa;	/* If it is a slot. */
		struct _templateDef	*td;	/* If it is a template. */
		struct _scopedNameDef	*snd;	/* If it is a defined type. */
		struct _classVersDef	*cvd;	/* If it is a class. */
		struct _enumDef		*ed;	/* If it is an enum. */
		char			*sname;	/* If it is a struct. */
	} u;
} argDef;


/* A template type. */

typedef struct _templateDef {
	char			*name;		/* The name. */
	argDef			type;		/* The type argument. */
} templateDef;


/* The arguments to a member function. */

typedef struct _funcArgs {
	int			nrArgs;		/* The number of arguments. */
	argDef			args[MAX_NR_ARGS];	/* The arguments. */
} funcArgs;


/* A list of function arguments. */

typedef struct _funcArgsList {
	funcArgs		*fa;		/* The arguments. */
	versionOrList		*vol;		/* The version qualifier. */
	struct _funcArgsList	*next;		/* Next in the list. */
} funcArgsList;


/* A typedef definition. */

typedef struct _typedefDef {
	char			*name;		/* The name. */
	argDef			type;		/* The actual type. */
	versionAnded		version;	/* The version qualifier. */
	struct _typedefDef	*next;		/* Next in the list. */
} typedefDef;


/* A variable definition. */

typedef struct _varDef {
	nameDef			*name;		/* The variable name. */
	moduleDef		*module;	/* The owning module. */
	int			varflags;	/* The variable flags. */
	argDef			type;		/* The actual type. */
	versionAnded		version;	/* The version qualifier. */
	codeBlock		*accessfunc;	/* The access function. */
	struct _varDef		*next;		/* Next in the list. */
} varDef;


/* An overloaded member function definition. */

typedef struct _overDef {
	char			*cppname;	/* The C++ name. */
	int			overflags;	/* The overload flags. */
	struct _memberDef	*common;	/* Common parts. */
	argDef			*result;	/* The result type. */
	funcArgs		args;		/* The arguments. */
	codeBlock		*cppcode;	/* Code in the specification. */
	codeBlock		*virtcode;	/* Virtual handler code. */
	versionAnded		version;	/* The version qualifier. */
	char			*prehook;	/* The pre-hook name. */
	char			*posthook;	/* The post-hook name. */
	struct _overDef		*next;		/* Next in the list. */
} overDef;


/* An overloaded constructor definition. */

typedef struct _ctorDef {
	int			ctorflags;	/* The ctor flags. */
	funcArgs		args;		/* The arguments. */
	codeBlock		*cppcode;	/* Code in the specification. */
	versionAnded		version;	/* The version qualifier. */
	struct _ctorDef		*next;		/* Next in the list. */
} ctorDef;


/* An enumerated type value definition. */

typedef struct _enumValueDef {
	nameDef			*name;		/* The value name. */
	struct _enumValueDef	*next;		/* Next in the list. */
} enumValueDef;


/* An enumerated type definition. */

typedef struct _enumDef {
	int			enumflags;	/* The enum flags. */
	char			*name;		/* The name. */
	struct _classVersDef	*cvd;		/* The class scope. */
	moduleDef		*module;	/* The owning module. */
	enumValueDef		*values;	/* The list of values. */
	versionAnded		version;	/* The version qualifier. */
	struct _enumDef		*next;		/* Next in the list. */
} enumDef;


/* An member function definition. */

typedef struct _memberDef {
	nameDef			*pyname;	/* The Python name. */
	moduleDef		*module;	/* The owning module. */
	struct _memberDef	*next;		/* Next in the list. */
} memberDef;


/* A list of visible member functions. */

typedef struct _visibleList {
	memberDef		*m;		/* The member definition. */
	struct _classVersDef	*cv;		/* The class version. */
	struct _visibleList	*next;		/* Next in the list. */
} visibleList;


/* An entry in a linked class version list. */

typedef struct _classVersList {
	struct _classVersDef	*cv;		/* The class version itself. */
	struct _classVersList	*next;		/* Next in the list. */
} classVersList;


/* An entry in a linked class list. */

typedef struct _classList {
	struct _classDef	*c;		/* The class itself. */
	struct _classList	*next;		/* Next in the list. */
} classList;


/* A virtual overload definition. */

typedef struct _virtOverDef {
	overDef			*o;		/* The overload. */
	struct _classVersDef	*nearc;		/* The nearest class. */
	struct _classVersDef	*farc;		/* The farthest class. */
	struct _virtOverDef	*next;		/* Next in the list. */
} virtOverDef;


/* A list of virtuals for a particular version. */

typedef struct _virtVersDef {
	versionAnded		version;	/* The version. */
	virtOverDef		*vo;		/* The overloads. */
	struct _virtVersDef	*next;		/* Next in the list. */
} virtVersDef;


/* A versioned class definition. */

typedef struct _classVersDef {
	int			classversflags;	/* The class version flags. */
	struct _classDef	*cd;		/* Back up to the class. */
	classList		*supers;	/* The parent classes. */
	classVersList		*hierachy;	/* The super-class hierachy. */
	versionAnded		version;	/* The version qualifier. */
	enumDef			*enums;		/* List of class enums. */
	typedefDef		*typedefs;	/* List of class typedefs. */
	ctorDef			*ctors;		/* The constructors. */
	codeBlock		*dtorcode;	/* Handwritten dtor code. */
	memberDef		*members;	/* The member functions. */
	overDef			*overs;		/* The overloads. */
	virtVersDef		*vmembers;	/* The virtual members. */
	visibleList		*visible;	/* The visible members. */
	varDef			*vars;		/* The class variables. */
	codeBlock		*cppcode;	/* Class C++ code. */
	codeBlock		*hdrcode;	/* Class header code. */
	codeBlock		*convfromcode;	/* Convert from C++ code. */
	codeBlock		*convtocode;	/* Convert to C++ code. */
	codeBlock		*convtosubcode;	/* Convert to sub C++ code. */
	codeBlock		*canconvtocode;	/* Can convert to C++ code. */
	spmDef			*spms;		/* Special Python methods. */
	int			combmeth;	/* Combined methods. */
	int			combmethx;	/* Combined methods. */
	int			combnummeth;	/* Combined numeric methods. */
	int			combnummethx;	/* Combined numeric methods. */
	int			combseqmeth;	/* Combined sequence methods. */
	int			combseqmethx;	/* Combined sequence methods. */
	int			combmapmeth;	/* Combined mapping methods. */
	int			combmapmethx;	/* Combined mapping methods. */
	int			combcmpmeth;	/* Combined compare methods. */
	int			combcmpmethx;	/* Combined compare methods. */
	struct _classVersDef	*next;		/* Next in the list. */
} classVersDef;


/* A class definition. */

typedef struct _classDef {
	nameDef			*name;		/* The class name. */
	moduleDef		*module;	/* The owning module. */
	int			classflags;	/* The class flags. */
	int			classnr;	/* The class number. */
	classList		*used;		/* Classes used. */
	classVersDef		*cvds;		/* The versioned definitions. */
	struct _classDef	*next;		/* Next in the list. */
} classDef;


/* The parse tree corresponding to the specification file. */

typedef struct {
	moduleDef		*moduleList;	/* The module list. */
	moduleDef		*module;	/* This module. */
	char			*cppmname;	/* The C++ module name. */
	nameDef			*namecache;	/* The name cache. */
	versionQual		*versions;	/* The version tags. */
	int			latestversion;	/* The latest version index. */
	int			nrsecs;		/* The number of secondaries. */
	classDef		*classes;	/* The list of classes. */
	classList		*used;		/* Classes used by main module. */
	int			qobjclass;	/* QObject class, -1 if none. */
	expFuncDef		*exposed;	/* Exposed functions. */
	enumDef			*enums;		/* List of global enums. */
	varDef			*globals;	/* List of globals. */
	memberDef		*othfuncs;	/* List of other functions. */
	overDef			*overs;		/* Global overloads. */
	typedefDef		*typedefs;	/* List of global typedefs. */
	codeBlock		*copying;	/* Software license. */
	codeBlock		*hdrcode;	/* Header code. */
	codeBlock		*versioncode;	/* Version code. */
	codeBlock		*exphdrcode;	/* Exported header code. */
	codeBlock		*cppcode;	/* Global C++ code. */
	codeBlock		*prepycode;	/* Global pre-Python code. */
	codeBlock		*pycode;	/* Global Python code. */
	codeBlock		*docs;		/* Documentation. */
	mkTemplateDef		*mktemplates;	/* The Makefile templates. */
	funcArgsList		*sigargs;	/* The list of signal arguments. */
	int			sigslots;	/* Set if signals or slots are used. */
} sipSpec;


/* A list of strings. */

typedef struct _stringList {
	char			*s;		/* The string. */
	struct _stringList	*next;		/* The next in the list. */
} stringList;


/* File specific context information for the parser. */

typedef struct _parserContext {
	int			ifdepth;	/* The depth of nested if's. */
	moduleDef		*prevmod;	/* The previous module. */
} parserContext;


char *sipVersion;		/* The version of SIP. */
stringList *includeDirList;	/* The include directory list for SIP files. */


void parse(sipSpec *,FILE *,char *,char *);
void parserEOF(char *,parserContext *);
void transform(sipSpec *);
void generateCode(sipSpec *,char *,stringList *,char *,char *);
void fatal(char *,...);
void fatalVersion(sipSpec *,versionAnded *);
void setInputFile(FILE *,char *,parserContext *);
void *sipMalloc(size_t);
char *sipStrdup(char *);
char *concat(char *,...);
void addToUsedList(classList **,classDef *);
void freeClassList(classList *);
void reduceOredVersion(sipSpec *,versionOrList **);
versionOrList *orVersionQual(sipSpec *,versionOrList *,versionAnded *);
versionQual *getVersionQual(sipSpec *,int);
int versionIsSubset(versionAnded *,versionAnded *);
int versionsOverlap(versionAnded *,versionAnded *);
int sameOverload(overDef *,overDef *);
int sameFuncArgs(funcArgs *,funcArgs *);


/* These are only here because bison publically references them. */

/* Represent a set of option flags. */

#define	MAX_NR_FLAGS	5

typedef enum {
	bool_flag,
	string_flag,
	name_flag
} flagType;

typedef struct {
	char			*fname;		/* The flag name. */
	flagType		ftype;		/* The flag type. */
	union {					/* The flag value. */
		char		*sval;		/* A string value. */
	} fvalue;
} optFlag;

typedef struct {
	int		nrFlags;		/* The number of flags. */
	optFlag		flags[MAX_NR_FLAGS];	/* Each flag. */
} optFlags;

#endif
