/*
 * Copyright (c) 1998,1999 Phil Thompson (phil@river-bank.demon.co.uk)
 *
 * The main module for SIP.
 */


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>

#include "sip.h"


static char *sipPackage = PACKAGE;


static void appendString(stringList **,char *);
static void help(void);
static void version(void);
static void usage(void);


int main(int argc,char **argv)
{
	char *filename, *docFile, *codeDir, *cppSuffix, *cppMName;
	int arg;
	FILE *file;
	classList *cl;
	sipSpec spec;
	stringList *makefiles;

	/* Initialise. */

	sipVersion = VERSION;
	includeDirList = NULL;
	makefiles = NULL;
	codeDir = NULL;
	docFile = NULL;
	cppSuffix = ".cpp";
	cppMName = NULL;

	/* Parse the command line. */

	opterr = 0;

	while ((arg = getopt(argc,argv,"hVm:c:d:I:s:p:")) != EOF)
		switch (arg)
		{
		case 'c':
			/* Where to generate the code. */

			codeDir = optarg;
			break;

		case 'm':
			/* Where to generate a Makefile. */

			appendString(&makefiles,optarg);
			break;

		case 'd':
			/* Where to generate the documentation. */

			docFile = optarg;
			break;

		case 'I':
			/* Where to get included files from. */

			appendString(&includeDirList,optarg);
			break;

		case 's':
			/* The suffix to use for C++ source files. */

			cppSuffix = optarg;
			break;

		case 'p':
			/* The name of the generated C++ module. */

			cppMName = optarg;
			break;

		case 'h':
			/* Help message. */

			help();
			break;

		case 'V':
			/* Display the version number. */

			version();
			break;

		default:
			usage();
		}

	if (optind < argc)
	{
		file = NULL;
		filename = argv[optind++];

		if (optind < argc)
			usage();
	}
	else
	{
		file = stdin;
		filename = "stdin";
	}

	/* Parse the input file. */

	parse(&spec,file,filename,cppMName);

	/* Verify and transform the parse tree. */

	transform(&spec);

	/* Generate code. */

	generateCode(&spec,codeDir,makefiles,docFile,cppSuffix);

	/* All done. */

	exit(0);
}


/*
 * Append a string to a list of them.
 */

static void appendString(stringList **headp,char *s)
{
	stringList *sl;

	/* Create the new entry. */

	sl = sipMalloc(sizeof (stringList));

	sl -> s = s;
	sl -> next = NULL;

	/* Append it to the list. */

	while (*headp != NULL)
		headp = &(*headp) -> next;

	*headp = sl;
}


/*
 * Display all or part of a one line error message describing a fatal error.
 * If the message is complete (it has a newline) then the program exits.
 */

void fatal(char *fmt,...)
{
	static int start = TRUE;

	va_list ap;

	if (start)
	{
		fprintf(stderr,"%s: ",sipPackage);
		start = FALSE;
	}

	va_start(ap,fmt);
	vfprintf(stderr,fmt,ap);
	va_end(ap);

	if (strchr(fmt,'\n') != NULL)
		exit(1);
}


/*
 * Display the SIP version number on stdout and exit with zero exit status.
 */

static void version(void)
{
	printf("%s\n",sipVersion);
	exit(0);
}


/*
 * Display the help message on stdout and exit with zero exit status.
 */

static void help(void)
{
	printf(
"Usage:\n"
"    %s [-h] [-V] [-c dir] [-d file] [-m file] [-I dir] [-s suffix] [-p module] [file]\n"
"where:\n"
"    -h         display this help message\n"
"    -V         display the %s version number\n"
"    -c dir     the name of the code directory [default not generated]\n"
"    -d file    the name of the documentation file [default not generated]\n"
"    -m file    the name of the Makefile [default none generated]\n"
"    -I dir     look in this directory when including files\n"
"    -s suffix  the suffix to use for C++ source files [default \".cpp\"]\n"
"    -p module  the name of the generated C++ module [default %%Module]\n"
"    file       the name of the specification file [default stdin]\n"
		,sipPackage,sipPackage);

	exit(0);
}


/*
 * Display the usage message.
 */

static void usage(void)
{
	fatal("Usage: %s [-h] [-V] [-c dir] [-d file] [-m file] [-I dir] [-s suffix] [-p module] [file]\n",sipPackage);
}
