/***************************************************************************
    file	         : kb_tablelist.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include 	<stdlib.h>
#include 	<errno.h>

#include 	<qcursor.h>
#include 	<qpopupmenu.h>
#include 	<qtextstream.h>
#include 	<qapplication.h>

#include	"kb_classes.h"
#include	"kb_location.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_attr.h"
#include	"kb_attrdict.h"
#include	"kb_prompt.h"

#include	"kb_dbinfo.h"
#include	"kb_dblink.h"
#include	"kb_viewer.h"
#include	"kb_serverinfo.h"
#include	"kb_notifier.h"
#include	"kb_objbase.h"

#include	"kb_tableinfodlg.h"
#include	"kb_listitem.h"
#include	"kb_filelist.h"
#include	"kb_tableviewer.h"

#ifndef		_WIN32
#include	"kb_tablelist.moc"
#else
#include	"kb_tablelist.h"
#endif

#include	"kb_callback.h"
#include	"kb_appptr.h"

#include	"tk_icons.h"
#include	"tk_filedialog.h"
#include	"tk_messagebox.h"



static	QString	lHead	("0_")	;
static	QString	lSort	("1_")	;
static	QString	lTail	("2_")	;
	


/*  KBTableItem								*/
/*  KBTableItem	: Constructor for table-list table item			*/
/*  svItem	: KBServerItem *  : Parent server item			*/
/*  order	: const QString & : Ordering value			*/
/*  tabList	: KBTableList *	  : Parent table list			*/
/*  tabName	: const QString & : Table name				*/
/*  tabType	: const QString & : Table type text			*/
/*  (returns)	: KBTableItem	  :					*/

KBTableItem::KBTableItem
	(	KBServerItem	*svItem,
		const QString	&order,
		KBTableList	*tabList,
		const QString	&tabName,
		const QString	&tabType
	)
	:
	KBObjectItem	(svItem, order, tabName, tabType),
	m_tabList	(tabList)
{
	setExpandable	(order != lHead) ;
	setPixmap	(0, getSmallIcon("table")) ;
}

/*  KBTableItem								*/
/*  setOpen	: Open/close table-list server item			*/
/*  open	: bool		: Opening				*/
/*  (returns)	: void		:					*/

void	KBTableItem::setOpen
	(	bool	open
	)
{
	const QString	&svName	 = m_parent->text (0) ;
	const QString	&tabName = text (0) ;
	KBDBLink	dbLink	 ;
	QListViewItem	*child	 ;

	/* If we are in a double click, rather then a click in the	*/
	/* decoration, ignore the open/close. This make show/hide	*/
	/* table summary, and show table in data view, independant.	*/
	if (m_tabList->inDblClick()) return ;


	if (!open)
	{
		QListViewItem::setOpen (false) ;
		return	;
	}

	while ((child = firstChild()) != 0)
		delete child ;

	if (!dbLink.connect (m_tabList->getDBInfo(), svName))
	{	dbLink.lastError().DISPLAY() ;
		return	;
	}

	KBTableSpec	tabSpec (tabName) ;
	if (!dbLink.listFields (tabSpec))
	{	dbLink.lastError().DISPLAY() ;
		return	;
	}

	for (uint idx = 0 ; idx < tabSpec.m_fldList.count() ; idx += 1)
	{
		KBFieldSpec  *field = tabSpec.m_fldList.at(idx) ;
		bool	     pk	    = (field->m_flags & KBFieldSpec::Primary ) != 0 ;
		bool	     nn	    = (field->m_flags & KBFieldSpec::NotNull ) != 0 ;
		bool	     un	    = (field->m_flags & KBFieldSpec::Unique  ) != 0 ;
		bool	     ix	    = (field->m_flags & KBFieldSpec::Indexed ) != 0 ;
		bool	     ai	    = (field->m_flags & KBFieldSpec::Serial  ) != 0 ;
		bool	     ro	    = (field->m_flags & KBFieldSpec::ReadOnly) != 0 ;

		QString	     size   = QString("%1").arg(field->m_length) ;
		if (field->m_prec > 0) size += QString(",%1").arg(field->m_prec) ;

		QListViewItem *item = new KBListItem
		(
			this,
			QString().sprintf("%05d_", idx),
			field->m_name,
			field->m_typeName,
			size,
			QString("%1%2%3%4%5%6")
				.arg(pk ? "PK " : "")
				.arg(nn ? "NN " : "")
				.arg(un ? "UN " : "")
				.arg(ix ? "IX " : "")
				.arg(ai ? "AI " : "")
				.arg(ro ? "RO " : "")
		)	;
		item->setPixmap (0, getSmallIcon("form")) ;
	}

	QListViewItem::setOpen (open) ;
}



/*  ------------------------------------------------------------------  */

/*  KBTableList								*/
/*  KBTableList	: Database/table list widget constructor		*/
/*  parent	: QWidget *	: Parent widget				*/
/*  dbInfo	: KBDBInfo *	: Database information			*/
/*  (returns)	: KBTableList	:					*/

KBTableList::KBTableList
	(	QWidget		*parent,
		KBDBInfo	*dbInfo
	)
	:
	KBFileList
	(	parent,
		dbInfo,
		"",
		"",
		"table",
		0
	)
{
	dblClick = 0 ;

	addColumn	   (TR("Server/Table/Field"), 150) ;
	addColumn	   (TR("Type"),		   70) ;
	addColumn	   (TR("Size"),		   80) ;
	addColumn	   (TR("Info"),		  100) ;
	setRootIsDecorated (true) ;
	setSorting	   (0) ;

	connect (KBNotifier::self(),
		 SIGNAL(sServerChanged(const KBLocation &)),
		 this,
		 SLOT  (serverChanged (const KBLocation &))) ;
	connect (KBNotifier::self(),
		 SIGNAL(sTablesChanged(const KBLocation &)),
		 this,
		 SLOT  (tablesChanged (const KBLocation &))) ;

}

/*  KBTableList								*/
/*  ~KBTableList: Database/table list widget destructor			*/
/*  (returns)	:		:					*/

KBTableList::~KBTableList ()
{
}

/*  KBTableList								*/
/*  loadServer	: Load/reload table list of specified server		*/
/*  server	: KBServerItem * : Item for server			*/
/*  (returns)	: void		 :					*/

void	KBTableList::reloadServer
	(	KBServerItem	*server
	)
{
	KBTableDetailsList	tabList	;
	KBDBLink		dbLink	;
	KBListItem		*child	;
	const QString		&svName	= server  ->text       (0) ;
	KBServerInfo		*svInfo	= m_dbInfo->findServer (svName) ;

	if ((svInfo != 0) && svInfo->dbType().isEmpty())
		return	;

	while ((child = (KBListItem *)server->firstChild()) != 0)
		delete child ;

	if (!dbLink.connect (m_dbInfo, svName))
	{	dbLink.lastError().DISPLAY() ;
		return	;
	}

#if	! __KB_RUNTIME
	child = new KBTableItem (server, lHead, this, "Create new table") ;
	child->setPixmap (0, getSmallIcon("filenew")) ;
	child->setType	 (KBListItem::Create) ;
#endif
	dbLink.flushTableCache () ;

	if (!dbLink.listTables (tabList))
	{	dbLink.lastError().DISPLAY() ;
		return	;
	}

	for (uint idx = 0 ; idx < tabList.count() ; idx += 1)
		new KBTableItem
		(	server,
			lSort,
			this,
			tabList[idx].m_name,
			tabList[idx].typeText()
		)	;
}

/*  KBTableList								*/
/*  serverChanged: Called when server details have changed		*/
/*  location	 : const KBLocation & : Location			*/
/*  (returns)	 : void		      :					*/

void	KBTableList::serverChanged
	(	const KBLocation &location
	)
{
	if (location.docLocn == KBLocation::m_pFile)
		return	;

	/* Scan looking for the corresponding entry. When found, update	*/
	/* the name (may or may not have changed), and drop all the	*/
	/* children tables (the server may now point somewhere else)	*/
	for (QListViewItem *item = firstChild () ; item != 0 ; item = item->nextSibling())
		if (item->text (0) == location.docLocn)
		{	QListViewItem *child  ;
			item->setText (0, location.docName) ;
			item->setOpen (false) ;
			while ((child = item->firstChild()) != 0) delete child ;
			return	;
		}

	/* Should only get here for new servers ....			*/
	new KBServerItem (this, "", location.docLocn) ;
}

/*  KBTableList								*/
/*  tablesChanged: Called when tables have changed on server		*/
/*  location	 : const KBLocation & : Location			*/
/*  (returns)	 : void		      :					*/

void	KBTableList::tablesChanged
	(	const KBLocation &location
	)
{
	for (QListViewItem *item = firstChild () ; item != 0 ; item = item->nextSibling())
		if (item->text (0) == location.docLocn)
		{	reloadServer ((KBServerItem *)item) ;
			return ;
		}

	new KBServerItem (this, "", location.docLocn) ;
}

/*  KBTableList								*/
/*  showObjectAs: Show a table in specified mode			*/
/*  item	: KBListItem *	: Selected list item			*/
/*  showAs	: KB::ShowAs	: Mode					*/
/*  (returns)	: void		:					*/

void	KBTableList::showObjectAs
	(	KBListItem	*item,
		KB::ShowAs	showAs
	)
{
	const QString &server  = item->parent()->text(0) ;
	const QString &table   = item          ->text(0) ;
	KBLocation    location (m_dbInfo, "table", server, table)   ;
	KBCallback    *cb      = KBAppPtr::getCallback() ;

	KBObjBase     *obj     = cb->showingObj (location) ;

	if (obj != 0)
	{
		KBError	  error	;
		if (obj->show (showAs, KBAttrDict(), 0, error) != KB::ShowRCOK)
			error.DISPLAY() ;
		return ;
	}

	KBError		error	;
	QDict<QString>	dummy	;

	if (cb->openObject (location, showAs, dummy, error) == KB::ShowRCError)
		error.DISPLAY () ;
}

/*  KBTableList								*/
/*  createTable	: Create a new table in a server			*/
/*  item	: KBServerItem *: Selected server item			*/
/*  (returns)	: void		:					*/

void	KBTableList::createTable
	(	KBServerItem	*server
	)
{
	QString	table ("NewTable") ;

	if (doPrompt ("Create table",
		      "Enter name for the new table",
		      table
		     ))
	{
		KBCallback *cb      = KBAppPtr::getCallback() ;
		KBLocation location (m_dbInfo, "table", server->text(0), table) ;
		KBError	   error    ;

		if (!cb->newObject (location, error))
			error.DISPLAY () ;
	}
}

/*  KBTableList								*/
/*  showDetault	: Show table request					*/
/*  item	: QListViewItem * : Selected list item			*/
/*  (returns)	: void		  :					*/

void	KBTableList::showDefault
	(	QListViewItem	*_item
	)
{
	KBListItem *item = (KBListItem *)_item ;

	switch (item->type())
	{
		case KBListItem::Create  :
			createTable  ((KBServerItem *)item->parent()) ;
			return	;

		case KBListItem::Object :
			showObjectAs (item, KB::ShowAsData)  ;
			return	;

		default	:
			break	;
	}
}

/*  KBTableList								*/
/*  showServerMenu : Show popup menu for server entries			*/
/*  (returns)	   : void	:					*/

void	KBTableList::showServerMenu ()
{
	QPopupMenu pop	;
	pop.insertItem (TR("Cancel")) ;
	pop.insertItem (TR("&Reload table list"),	this, SLOT(reloadServer   ())) ;
#if	! __KB_RUNTIME
	pop.insertItem (TR("&New table"),		this, SLOT(createTable    ())) ;
	pop.insertItem (TR("E&xport definitions"),	this, SLOT(exportAllTables())) ;
	pop.insertItem (TR("&Import definitions"), 	this, SLOT(importTables   ())) ;
#endif
#if	0
	pop.insertItem (TR("Show table &information"),	this, SLOT(showTableInfo  ())) ;
#endif
	pop.exec       (QCursor::pos()) ;
}

/*  KBTableList								*/
/*  showObjectMenu : Show popup menu for server entries			*/
/*  (returns)	   : void	:					*/

void	KBTableList::showObjectMenu ()
{
	QPopupMenu pop ;
	pop.insertItem (TR("Cancel")) ;
	pop.insertItem (TR("&Data view"),	  this, SLOT(showAsData  ())) ;
#if	! __KB_RUNTIME
	pop.insertItem (TR("D&esign view"),	  this, SLOT(showAsDesign())) ;
	pop.insertItem (TR("&Rename table"),	  this, SLOT(renameTable ())) ;
	pop.insertItem (TR("De&lete table"), 	  this, SLOT(deleteTable ())) ;
	pop.insertItem (TR("E&xport definition"), this, SLOT(exportTable ())) ;
#endif
	pop.exec       (QCursor::pos()) ;
}

/*  KBTableList								*/
/*  createTable	: Create a new table in a server			*/
/*  (returns)	: void		:					*/

void	KBTableList::createTable ()
{
	createTable ((KBServerItem*)m_curItem) ;
}


/*  KBTableList								*/
/*  renameTable	: Rename a table					*/
/*  (returns)	: void		:					*/

void	KBTableList::renameTable ()
{
	KBServerItem  *parent  = (KBServerItem *)m_curItem->parent()  ;
	const QString &server  = parent   ->text  (0) ;
	const QString &table   = m_curItem->text  (0) ;
	QString	      newName  = table ;
	KBCallback    *cb      = KBAppPtr::getCallback() ;

	KBLocation    location (m_dbInfo, "table", server, table) ;

	if (cb->showingObj (location) != 0)
	{
		TKMessageBox::sorry
		(	0,
			QString (TR("Table %1/%2 is currently open")).arg(server).arg(table),
			"Unable to rename table"
		)	;
		return	;
	}

	if (!doPrompt
		(	TR("Rename table"),
			TR("Enter new name for the table"),
			newName
		)
	   )	return ;

	/* Connect to the database, rename the table, and the reload	*/
	/* the table list to reflect the change. Note that we reload	*/
	/* after an apparent rename failure so that we definitely have	*/
	/* a correct list.						*/
	KBDBLink dbLink ;
	if (!dbLink.connect (m_dbInfo, server))
	{	dbLink.lastError().DISPLAY() ;
		return	;
	}
	if (!dbLink.renameTable (table, newName, true))
	{	dbLink.lastError().DISPLAY() ;
		reloadServer (parent) ;
		return	;
	}

	/* Rename the table in the table information. This will also	*/
	/* rename the associated document.				*/
	m_dbInfo->findTableInfoSet(server)->renameTable (table, newName) ;
	reloadServer (parent) ;
}

/*  KBTableList								*/
/*  getExportFile: Get file for table definition export			*/
/*  eFile	 : QFile &	   : Export file			*/
/*  eName	 : const QString & : Default name			*/
/*  (returns)	 : bool		   : SUccess				*/

bool	KBTableList::getExportFile
	(	QFile		&eFile,
		const QString	&eName
	)
{
	KBFileDialog	fDlg
			(	".",
				"*.tab|Table definition",
				qApp->activeWindow(),
				"savetable",
				true
			)	;

#if	! __KB_EMBEDDED
	fDlg.setSelection (eName)			;
#endif
	fDlg.setMode	  (KBFileDialog::AnyFile)	;
	fDlg.setCaption   (TR("Save definition ...."))	;

	if (!fDlg.exec ()) return false	;

	QString	name	= fDlg.selectedFile () ;
        if (name.findRev (".tab", -1, false) < 0) name += ".tab" ;

	eFile.setName	(name) ;

	if (QFileInfo(eFile).exists())
		if (TKMessageBox::questionYesNo
		   	(	0,
				QString (TR("%1 already exists: overwrite?")).arg(name),
				TR("Export definition ....")
			)
			!= TKMessageBox::Yes) return false ;

	if (!eFile.open (IO_WriteOnly|IO_Truncate))
	{
		KBError::EError
		(	QString	(TR("Cannot open \"%1\"")).arg(name),
			strerror(errno),
			__ERRLOCN
		)	;
		return	false	;
	}

	return	true	;
}


/*  KBTableList								*/
/*  getTableDef	: Get table definition					*/
/*  dbLink	: KBDBLink &	  : Database link			*/
/*  table	: const QString & : Table name				*/
/*  elem	: QDomElement &	  : Definition element			*/
/*  (returns)	: bool		  : Success				*/

bool	KBTableList::getTableDef
	(	KBDBLink	&dbLink,
		const QString	&table,
		QDomElement	&elem
	)
{
	KBTableSpec	tabSpec (table) ;

	if (!dbLink.listFields (tabSpec))
	{	dbLink.lastError().DISPLAY() ;
		return	false ;
	}

	tabSpec.toXML	(elem, 0) ;
	return	true	;
}

/*  KBTableList								*/
/*  exportTable	: Export a table definition				*/
/*  (returns)	: void		:					*/

void	KBTableList::exportTable ()
{
	QListViewItem *parent = m_curItem->parent()  ;
	const QString &server = parent   ->text  (0) ;
	const QString &table  = m_curItem->text  (0) ;

	QFile	eFile	;
	if (!getExportFile (eFile, table)) return  ;

	KBDBLink dbLink ;
	if (!dbLink.connect (m_dbInfo, server))
	{	dbLink.lastError().DISPLAY() ;
		return	;
	}

	QDomDocument	doc ("tablelist") ;
	doc.appendChild
	(	doc.createProcessingInstruction
		(	"xml",
			"version=\"1.0\" encoding=\"UTF=8\""
	)	)	;

	QDomElement	docElem	= doc.createElement ("tablelist") ;
	QDomElement	tabElem	= doc.createElement ("table"    ) ;

	doc    .appendChild (docElem) ;
	docElem.appendChild (tabElem) ;

	if (!getTableDef (dbLink, table, tabElem)) return ;

	QTextStream(&eFile) << doc.toString() ;
}	

/*  KBTableList								*/
/*  exportAllTables							*/
/*		: Export all table definitions				*/
/*  (returns)	: void		:					*/

void	KBTableList::exportAllTables ()
{
	QFile	eFile	;
	if (!getExportFile (eFile, "allTables")) return ;

	const QString &server = m_curItem->text (0) ;

	KBDBLink dbLink ;
	if (!dbLink.connect (m_dbInfo, server))
	{	dbLink.lastError().DISPLAY() ;
		return	;
	}

	KBTableDetailsList tabList	;
	if (!dbLink.listTables (tabList))
	{	dbLink.lastError().DISPLAY() ;
		return	;
	}

	QDomDocument	doc ("tablelist") ;
	doc.appendChild
	(	doc.createProcessingInstruction
		(	"xml",
			"version=\"1.0\" encoding=\"UTF=8\""
	)	)	;

	QDomElement	docElem	= doc.createElement ("tablelist") ;
	doc.appendChild (docElem) ;

	for (uint idx = 0 ; idx < tabList.count() ; idx += 1)
	{
		QDomElement tabElem = doc.createElement ("table")  ;
		docElem.appendChild (tabElem) ;

		if (!getTableDef (dbLink, tabList[idx].m_name, tabElem))
			return ;
	}

	QTextStream(&eFile) << doc.toString() ;
}

/*  KBTableList								*/
/*  importTables: Import table definitions				*/
/*  (returns)	: void		:					*/

void	KBTableList::importTables ()
{
	KBServerItem	*svItem	= (KBServerItem *)m_curItem ;
	const QString	&server = svItem->text (0) ;

	KBFileDialog	fDlg
			(	".",
				"*.tab|Table definition",
				qApp->activeWindow(),
				"loadtable",
				true
			)	;

	fDlg.setMode	  (KBFileDialog::ExistingFile )	;
	fDlg.setCaption   (TR("Load definitions ...."))	;

	if (!fDlg.exec ()) return ;

	QString	eName	= fDlg.selectedFile () ;
        if (eName.findRev (".tab", -1, false) < 0) eName += ".tab" ;

	QFile	eFile	(eName) ;

	if (!eFile.open (IO_ReadOnly))
	{
		KBError::EError
		(	QString	(TR("Cannot open \"%1\"")).arg(eName),
			strerror(errno),
			__ERRLOCN
		)	;
		return	;
	}

	QDomDocument	doc	;
	if (!doc.setContent (&eFile))
	{
		KBError::EError
		(	QString	(TR("Cannot parse \"%1\"")).arg(eName),
			strerror(errno),
			__ERRLOCN
		)	;
		return	;
	}

	KBDBLink dbLink ;
	if (!dbLink.connect (m_dbInfo, server))
	{	dbLink.lastError().DISPLAY() ;
		return	;
	}

	QDomElement	docElem	= doc.documentElement() ;
	QDomNode	node	= docElem.firstChild () ;

	while (!node.isNull())
	{
		QDomElement tabElem = node.toElement () ;
		KBTableSpec tabSpec (tabElem) ;

		if (!dbLink.createTable (tabSpec, true))
		{	dbLink.lastError().DISPLAY() ;
			reloadServer (svItem) ;
			return	 ;
		}

		node	= node.nextSibling() ;
	}

	reloadServer (svItem) ;
}

void	KBTableList::showTableInfo ()
{
}

/*  KBTableList								*/
/*  deleteTable	: Delete a table					*/
/*  (returns)	: void		:					*/

void	KBTableList::deleteTable ()
{
	KBServerItem  *parent  = (KBServerItem *)m_curItem->parent()  ;
	const QString &server  = parent   ->text (0) ;
	const QString &table   = m_curItem->text (0) ;
	KBLocation    location (m_dbInfo, "table", server, table) ;
	KBCallback    *cb      = KBAppPtr::getCallback() ;

	if (cb->showingObj (location) != 0)
	{
		TKMessageBox::sorry
		(	0,
			QString (TR("Table %1/%2 is currently open")).arg(server).arg(table),
			TR("Unable to delete table")
		) ;
		return	;
	}

	if (TKMessageBox::questionYesNo
	   	(	0,
			QString (TR("Definitely delete table %1/%2")).arg(server).arg(table),
			TR("Delete table")
		)
		!= TKMessageBox::Yes) return ;

	/* Connect to the database, delete the table, and the reload	*/
	/* the table list to reflect the change. Note that we reload	*/
	/* after an apparent delete failure so that we definitely have	*/
	/* a correct list.						*/
	KBDBLink dbLink ;
	if (!dbLink.connect (m_dbInfo, server))
	{	dbLink.lastError().DISPLAY() ;
		return	;
	}
	if (!dbLink.dropTable (table, true))
	{	dbLink.lastError().DISPLAY() ;
		reloadServer (parent) ;
		return	;
	}

	/* Drop the table from the table information, which will also	*/
	/* drop thr associated document.				*/
	m_dbInfo->findTableInfoSet(server)->dropTable (table) ;
	reloadServer (parent) ;
}

/*  KBTableList								*/
/*  contents...	: Trap double click					*/
/*  e		: QMouseEvent *	: Mouse event				*/
/*  (returns)	: void		:					*/

void	KBTableList::contentsMouseDoubleClickEvent
	(	QMouseEvent	*e
	)
{
	/* Since double-click on a table is used to open the table in	*/
	/* data view, we don't want it to expand the branch. There	*/
	/* seems not to be a way of doing this direcly in Qt, so set	*/
	/* a flag which will bracket double clicks.			*/
	dblClick += 1 ;
	QListView::contentsMouseDoubleClickEvent (e) ;
	dblClick -= 1 ;
}

/*  KBTableList								*/
/*  getObjectNames							*/
/*		: Get list of all objects on specified server		*/
/*  server	: KBServerInfo * : Server				*/
/*  (returns)	: QStringList	 : List of names			*/

QStringList
	KBTableList::getObjectNames
	(	KBServerInfo	*server
	)
{
	KBTableDetailsList	tabList	;
	QStringList		result	;
	KBDBLink		dbLink	;

	if (server->serverName() == KBLocation::m_pFile)
		return	result	;

	if (!dbLink.connect (m_dbInfo, server->serverName()))
	{	dbLink.lastError().DISPLAY() ;
		return	result	;
	}

	if (!dbLink.listTables (tabList))
	{	dbLink.lastError().DISPLAY() ;
		return	result	;
	}

	for (uint idx = 0 ; idx < tabList.count() ; idx += 1)
		result.append (tabList[idx].m_name) ;

	return	result	;
}

