/***************************************************************************
    file	         : kb_display.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<qpainter.h>

#include	"kb_formblock.h"

#include	"kb_display.h"
#include	"kb_options.h"
#include	"kb_layout.h"
#include	"kb_dispwidget.h"
#include	"kb_dispscroller.h"
#include	"kb_report.h"
#include	"kb_writer.h"



/*  snapPosition: Snap position to grid					*/
/*  pos		: int		: Position to be snapped		*/
/*  near	: bool		: True to snap to nearest		*/
/*  x		: bool		: True for X, else Y			*/
/*  (returns)	: int		: Position snapped to grid		*/

int	snapPosition
	(	int	pos,
		bool	bNear,
		bool	x
	)
{
	uint	step	;
	uint	delta	;

	/* Get the appropriate grid step. If zero (only if the user	*/
	/* edits the configuration file) or one then the position is	*/
	/* left unchanged (ie., treat zero as one).			*/
	step	= x ? KBOptions::getGridXStep() : KBOptions::getGridYStep() ;
	if (step <= 1) return pos ;

	/* To avoid round-towards-zero problems (ie, to get round-down	*/
	/* behaviour) we use an offset. THis must be a multiple of the	*/
	/* grid step and larger than any reasonable coordinate.		*/
	delta	= step	;
	while (delta < 100000) delta *= 100 ;

	/* If snapping to nearest rather than left/up, adjust the	*/
	/* position by half the grid step.				*/
	if (bNear) pos += step / 2 ;

	/* Rock and roll. Remember, sex-and-drugs-and-rock-and-roll are	*/
	/* only substitutes for caving.					*/
	return	((pos + delta) / step) * step - delta ;
}

/*  snapRect	: Snap rectangle to nearest grid points			*/
/*  x, y, w, h	: int &, ...	: Rectangle position and size		*/
/*  (returns)	: void		:					*/

void	snapRect
	(	int	&x,
		int	&y,
		int	&w,
		int	&h
	)
{
	w	= snapPosition (x + w, true, true ) ;
	h	= snapPosition (y + h, true, false) ;
	x	= snapPosition (x,     true, true ) ;
	y	= snapPosition (y,     true, false) ;
	w	= w - x ;
	h	= h - y ;
}


/*  ------------------------------------------------------------------  */



/*  KBDisplay								*/
/*  KBDisplay	: Constructor for root display window			*/
/*  _parent	: QWidget *	: Parent widget				*/
/*  _owner	: KBObject	: Owning object				*/
/*  showBar	: uint		: Show row scroller bar/mini-nav	*/
/*  stretch	: bool		: Form in stretch mode			*/
/*  showRuler	: bool		: Show design ruler			*/
/*  (returns)	: KBDisplay	:					*/

KBDisplay::KBDisplay
	(	QWidget		*_parent,
		KBObject	*_owner,
		uint		showBar,
		bool		stretch,
		bool		showRuler
	)
	:
	parent	(0),
	owner	(_owner)
{
	marking	 = false ; 
	scroller = new KBDispScroller (_parent, this, showBar, stretch, showRuler) ;
	widget	 = 0	 ;
	canvas	 = scroller->viewport () ;
	shown	 = false ;

	scroller->hide() ;
}



/*  KBDisplay								*/
/*  KBDisplay	: Constructor for display window			*/
/*  _parent	: KBDisplay *	: Parent display			*/
/*  _owner	: KBObject	: Owning object				*/
/*  showBar	: uint		: Show row scroller bar/mini-nav	*/
/*  (returns)	: KNDisplay	:					*/

KBDisplay::KBDisplay
	(	KBDisplay	*_parent,
		KBObject	*_owner,
		uint		showBar
	)
	:
	parent	(_parent),
	owner	(_owner)
{
	marking	 = false ;
	scroller = 0	 ;
	widget	 = new KBDispWidget
		       (	parent->getDisplayWidget(),
				this,
				showBar
		       )	;
	canvas	 = widget;
	shown	 = false ;

	widget->hide ()  ;
}

/*  KBDisplay								*/
/*  KBDisplay	: Destructor for display window				*/
/*  (returns)	:		:					*/

KBDisplay::~KBDisplay ()
{
	if 	(scroller) delete scroller ;
	else if (widget  ) delete widget   ;
}

#if	! __KB_RUNTIME

/*  KBDisplay								*/
/*  markStart	: Start marking movement for area selection		*/
/*  e		: QMouseEvent *	: Originating mouse event		*/
/*  (returns)	: bool		: True to start marking			*/

bool	KBDisplay::markStart
	(	QMouseEvent	*e
	)
{
	if (!marking)
	{
		m_left	= 0	 ;
		m_top	= 0	 ;

		/* If the owner is a report then get the left and top	*/
		/* margins and adjust the position accordingly; if not	*/
		/* then the effective margin is zero.			*/
		KBReport *report ;
		if ((report = owner->isReport()) != 0)
		{
			report->margins (m_left, m_top) ;

			m_left	= (int)(m_left * pixelsPerMM()) ;
			m_top	= (int)(m_top  * pixelsPerMM()) ;
		}

		int x	= e->x() - m_left ;
		int y	= e->y() - m_top  ;

		/* At the top level, check that we are inside the block	*/
		/* area. In either case, set the marking area bounds.	*/
		if (scroller)
		{
			if ((x >= topSize.width()) || (y >= topSize.height()))
				return	false	;

			bounds	= topSize	;
		}
		else	bounds	= canvas->size();

		if ((x < 0) || (y < 0))
			return	false	;

		marking	= true	  	;

		startMX	= markX = x	;
		startMY	= markY = y	;
		markW	= markH	= 0	;

		canvas->grabMouse ()  	;
		return	 true		;
	}

	return	false ;
}

/*  KBDisplay								*/
/*  markMoveEvent							*/
/*		: Handle mouse movement during marking			*/
/*  widget	: QWidget *	: Widget originating event		*/
/*  event	: QMouseEvent *	: Mouse event				*/
/*  (returns)	: void		:					*/

void	KBDisplay::markMoveEvent
	(	QWidget		*,
		QMouseEvent 	*event
	)
{
	/* If marking then sort out the new marked region (so that up-	*/
	/* and left-sweeps work), the get it drawn out.			*/
	if (marking)
	{
		int	x = event->x() ;
		int	y = event->y() ;

		/* Zaurus fix: We seem to get a (-2001,-2001) as the	*/
		/* button is released ...				*/
		if ((x < 0) || (y < 0)) return ;

		/* Adjust by the margins if any, and then bound to the	*/
		/* current bounding rectangle.				*/
		x	-= m_left ;
		y	-= m_top  ;

		if (x >= bounds.width ()) x = bounds.width () - 1 ;
		if (y >= bounds.height()) y = bounds.height() - 1 ;

		if (x < 0) x = 0 ;
		if (y < 0) y = 0 ;


		if (x >= startMX)
			markX = startMX, markW = x - startMX ;
		else	markX = x,	 markW = startMX - x ;
		if (y >= startMY)
			markY = startMY, markH = y - startMY ;
		else	markY = y,	 markH = startMY - y ;

		canvas->repaint () ;
	}
}

/*  KBDisplay								*/
/*  mouseEndEvent							*/
/*		: Handle mouse release to end marking			*/
/*  widget	: QWidget *	: Originating widget			*/
/*  event	: QMouseEvent *	: Mouse event				*/
/*  (returns)	: bool		: Event consumed			*/

bool	KBDisplay::markEndEvent
	(	QWidget		*,
		QMouseEvent	*event
	)
{
	/* If marking, then there are two possibilities, marking out to	*/
	/* create a new object, and marking out to select existing	*/
	/* objects.							*/
	if (marking)
	{
		canvas->releaseMouse ()	;

		if ((event->state() & Qt::ShiftButton) == 0)
		{
			if (!owner->checkOverlap (markX, markY, markW, markH))
				if (!owner->addNewObject (markX, markY, markW, markH))
					owner->getLayout()->addSizer (owner->getSizer(), false) ;
		}
		else	owner->selectOverlap (markX, markY, markW, markH) ;

		marking = false    ;
		canvas->repaint () ;
		return	true	   ;
	}

	return	false	;
}

/*  KBDisplay								*/
/*  drawDisplay	: Handle painting					*/
/*  p		: QPainter *	: Painter				*/
/*  clip	: QRect		: Area to paint				*/
/*  (returns)	: void		:					*/

void	KBDisplay::drawDisplay
	(	QPainter *p,
		QRect	
	)
{
	KBReport *report = owner->isReport() ;
	double	  ppm	 = pixelsPerMM    () ;

	int	lMargin  = 0 ;
	int	rMargin  = 0 ;
	int	tMargin  = 0 ;
	int	bMargin  = 0 ;

	p->setPen(Qt::black) ;

	if (report != 0)
	{
		report->margins (lMargin, rMargin, tMargin, bMargin) ;

		lMargin = (int)(lMargin * ppm) ;
		rMargin = (int)(rMargin * ppm) ;
		tMargin = (int)(tMargin * ppm) ;
		bMargin = (int)(bMargin * ppm) ;
	}

	if (marking)
	{
		int	x = markX + lMargin ;
		int	y = markY + tMargin ;

		p->drawRect (x, y, markW, markH) ;
	}

	int	gx	= (int)KBOptions::getGridXStep() ;
	int	gy	= (int)KBOptions::getGridXStep() ;

	QSize	size	= scroller ? topSize : canvas->geometry().size() ;

	int	dw	= size.width  () + lMargin ;
	int	dh	= size.height () + tMargin ;

	for (int px = lMargin ; px <= dw ; px += gx)
		for (int py = tMargin ; py <= dh ; py += gy)
			p->drawPoint (px, py) ;

	p->drawRect (lMargin, tMargin, dw - lMargin, dh - tMargin) ;

	if (report != 0)
	{
		int	width	;
		int	height	;

		getPixelPageSize("A4", width, height) ;

		QBrush	brush	(QColor(0xe0, 0xe0, 0xe0)) ;

		p->fillRect (0, 0, width, tMargin, brush) ;
		p->fillRect (0, height - bMargin, width, bMargin, brush) ;
		p->fillRect (0, tMargin, lMargin, height - tMargin - bMargin, brush) ;
		p->fillRect (width - rMargin, tMargin, rMargin, height - tMargin - bMargin, brush) ;

		p->drawLine (lMargin, 0, lMargin, height) ;
		p->drawLine (width - rMargin,  0, width - rMargin, height) ;
		p->drawLine (width,   0, width,   height) ;

		p->drawLine (0, tMargin, width,   tMargin) ;
		p->drawLine (0, height - bMargin, width, height - bMargin) ;
		p->drawLine (0, height,  width,   height) ;
	}
}

#endif

/*  KBDisplay								*/
/*  getDisplayWidget							*/
/*		: Get widget used for display				*/
/*  (returns)	: QWidget *	: Display widget			*/

QWidget	*KBDisplay::getDisplayWidget ()
{
	return	canvas	;
}

/*  KBDisplay								*/
/*  getTopWidget: Get top-most widget					*/
/*  (returns)	: QWidget *	: Top-most widget			*/

QWidget	*KBDisplay::getTopWidget ()
{
	return	scroller ? (QWidget *)scroller : (QWidget *)widget ;
}

/*  KBDisplay								*/
/*  addChild	: Add child widget					*/
/*  x		: int		: X-position				*/
/*  y		: int		: Y-position				*/
/*  (returns)	: void		:					*/

void	KBDisplay::addChild
	(	QWidget	*child,
		int	x,
		int	y
	)
{
	/* We only need do anything if we are showing the child		*/
	/* directly inside a scroll view.				*/
	if (scroller) scroller->addChild (child, x, y) ;
}

/*  KBDisplay								*/
/*  moveChild	: Move child widget					*/
/*  x		: int		: X-position				*/
/*  y		: int		: Y-position				*/
/*  (returns)	: void		:					*/

void	KBDisplay::moveChild
	(	QWidget	*child,
		int	x,
		int	y
	)
{
	if (scroller)
		scroller->moveChild (child, x, y) ;
	else	child   ->move      (x, y) ;
}

/*  KBDisplay								*/
/*  move	: Move display						*/
/*  x		: int		: X position				*/
/*  y		: int		: Y position				*/
/*  (returns)	: void		:					*/

void	KBDisplay::move
	(	int	x,
		int	y
	)
{
	/* If there is the top-most display, then movement is		*/
	/* meaningless. Only move for nested displays.			*/
	if (parent == 0) return ;

	parent->moveChild (widget, x, y) ;

	KBDisplay *pp = parent      ;
	uint	  ty  = widget->y() ;

	while (pp->getParent() != 0)
	{
		ty += pp->getTopWidget()->y() ;
		pp  = pp->getParent   ()      ;
	}

	pp->scroller->moveTags (widget, ty) ;
}

/*  KBDisplay								*/
/*  resize	: Resize display					*/
/*  w		: int		: Width					*/
/*  h		: int		: Height				*/
/*  (returns)	: void		:					*/

void	KBDisplay::resize
	(	int	w,
		int	h
	)
{
	/* If this is the top-most display then the resize is notional	*/
	/* and is handled by drawing appropriately on the scroller.	*/
	/* in this case, and there is no widget to actually alter.	*/
	if (parent == 0)
	{
		int	uw = topSize.width () ;
		int	uh = topSize.height() ;

		topSize	= QSize  (w, h) ;

		if (w > uw) uw = w ;
		if (h > uh) uh = h ;
		scroller->sizeAdjusted	 () ;
		scroller->updateContents (0, 0, uw + 1, uh +1) ;
		return	;
	}

	/* Not at the top level, so resize the actual display widget	*/
	widget->resize    (w, h) ;
}

/*  KBDisplay								*/
/*  width	: Get display width					*/
/*  (returns)	: int		: Width					*/

int	KBDisplay::width ()
{
	/* This is the width of the display widget if this is not a	*/
	/* scroller, or of the scroller if there is one.		*/
	return	scroller ? scroller->width() : widget->width() ;
}

/*  KBDisplay								*/
/*  height	: Get display height					*/
/*  (returns)	: int		: Height					*/

int	KBDisplay::height ()
{
	/* See width comments above ...					*/
	return	scroller ? scroller->height() : widget->height() ;
}

/*  KBDisplay								*/
/*  setPalette	: Set pallette						*/
/*  pal		: const QPalette *: Palette in question			*/
/*  (returns)	:		  : void				*/

void	KBDisplay::setPalette
	(	const QPalette *pal
	)
{
	getDisplayWidget()->setPalette    (*pal) ;
#if	QT_VERSION >= 300
	getDisplayWidget()->setEraseColor (pal->active().background()) ;
#endif
}

/*  KBDisplay								*/
/*  setFont	: Set font						*/
/*  font	: const QFont *	: Font in question			*/
/*  (returns)	:		: void					*/

void	KBDisplay::setFont
	(	const QFont *font
	)
{
	getDisplayWidget()->setFont (*font) ;
}

/*  KBDisplay								*/
/*  getPalette	: Get pallette						*/
/*  (returns)	: const QPalette: The palette				*/

const	QPalette KBDisplay::palette ()
{
	return	getDisplayWidget()->palette () ;
}

/*  KBDisplay								*/
/*  getFont	: Get font						*/
/*  (returns)	: const QFont	: The font				*/

const	QFont	KBDisplay::font ()
{
	return	getDisplayWidget()->font () ;
}

/*  KBDisplay								*/
/*  show	: Show widget						*/
/*  (returns)	: void		:					*/

void	KBDisplay::show ()
{
	if (!shown)
	{
		if (scroller)
			scroller->show () ;
		else	widget  ->show () ;

		shown	= true	;
	}
}

#if	! __KB_RUNTIME

/*  KBDisplay								*/
/*  contextEvent: Handler for mouse context menu events			*/
/*  widget	: QWidget *	: Widget originating event		*/
/*  e		: QMouseEvent *	: Mouse event				*/
/*  (returns)	: bool		: Event consumed			*/

bool	KBDisplay::contextEvent
	(	QWidget		*,
		QMouseEvent	*e
	)
{
	if (owner->showingDesign ())
	{	owner->designPopup (e, 0) ;
		return	true ;
	}

	return	false	;
}

/*  KBDisplay								*/
/*  markStartEvent							*/
/*		: Handler for mouse area mark start			*/
/*  widget	: QWidget *	: Widget originating event		*/
/*  e		: QMouseEvent *	: Mouse event				*/
/*  (returns)	: bool		: Event consumed			*/

bool	KBDisplay::markStartEvent
	(	QWidget		*,
		QMouseEvent	*e
	)
{
	if (owner->showingDesign ())
	{	markStart (e)	;
		return	  true	;
	}

	return	false	;
}
#endif

/*  KBDisplay								*/
/*  displayIsGone: Notification that display widget is gone		*/
/*  (returns)	 : void		:					*/

void	KBDisplay::displayIsGone ()
{
	/* This notification is used to simplify the problem of		*/
	/* ensuring that all widgets are cleaned up, however they are	*/
	/* destroyed, without having lots of order-of-deletion problems	*/ 
	scroller = 0 ;
	widget   = 0 ;
	canvas	 = 0 ;
}

/*  KBDisplay								*/
/*  getSizeNeeded: Set display size needed for specified size		*/
/*  size	 : const QSize &: Specified size			*/
/*  (returns)	 : QSize	: Required size				*/

QSize	KBDisplay::getSizeNeeded
	(	const	QSize	&size
	)
{
	/* If this is not a scroller then return just the size, else	*/
	/* add the frame width.						*/
	if (!scroller) return size ;

	return	QSize	(size.width () + 2 * scroller->frameWidth(),
			 size.height() + 2 * scroller->frameWidth()) ;
}

/*  KBDisplay								*/
/*  effectiveSize: Get effective display size				*/
/*  (returns)	 : QSize	: Effective size			*/

QSize	KBDisplay::effectiveSize ()
{
	if (scroller)
		return	scroller->effectiveSize (topSize) ;

	if (widget  )
		return	widget  ->effectiveSize () ;

	return	QSize(0,0) ;
}

/*  KBDisplay								*/
/*  makeVisible	: Try to ensure that area is visible			*/
/*  rCtrl	: const QRect &		: Area in question		*/
/*  reason	: QFocusEvent::Reason	: Focus reason			*/
/*  (returns)	: void		:					*/

void	KBDisplay::makeVisible
	(	const QRect		&rCtrl,
		QFocusEvent::Reason	reason
	)
{
	/* If the display is not a scroller then there is nothing to do	*/
	/* For a scroller, try to move the specified area into view,	*/
	/* plus some space round it.					*/
	if (scroller && scroller->isVisible())
	{
//		fprintf
//		(	stderr,
//			"KBDisplay::makeVisible: (%d,%d)(%d)\n",
//			rCtrl.x(),
//			rCtrl.y(),
//			(int)reason
//		)	;

		scroller->makeVisible (rCtrl, reason) ;
	}
}

/*  KBDisplay								*/
/*  cvtViewToCtrl: Convert geometry from viewport to contents		*/
/*  cRect	 : QRect &	: Position-and-size			*/
/*  (returns)	 : void		:					*/

void	KBDisplay::cvtViewToCtrl
	(	QRect	&cRect
	)
{
	/* If this is a scroll view then map the control geometry back	*/
	/* to contents coordinates; if not then it is OK already.	*/
	if (scroller)
		cRect	= QRect (scroller->viewportToContents (cRect.topLeft    ()),
				 scroller->viewportToContents (cRect.bottomRight())) ;
}

/*  KBDisplay								*/
/*  cvtCtrlToView: Convert geometry from contents to viewport		*/
/*  cRect	 : QRect &	: Position-and-size			*/
/*  (returns)	 : void		:					*/

void	KBDisplay::cvtCtrlToView
	(	QRect	&cRect
	)
{
	/* If this is a scroll view then map the control geometry to	*/
	/* viewport coordinates; if not then it is OK already.	*/
	if (scroller)
		cRect	= QRect (scroller->contentsToViewport (cRect.topLeft    ()),
				 scroller->contentsToViewport (cRect.bottomRight())) ;
}

/*  KBDisplay								*/
/*  getScroller	: Get scroller widget if any				*/
/*  (returns)	: QScrollView *	: Scroll view or null			*/

QScrollView *KBDisplay::getScroller ()
{
	return	scroller ? scroller->getScroller () : 0 ;
}

/*  KBDisplay								*/
/*  setRowRange	: Set range of rows being displayed			*/
/*  totalRows	: uint		: Total number of rows			*/
/*  extra	: uint		: Extra insertion rows			*/
/*  curQRow	: uint		: Current query row			*/
/*  curDRow	: uint		: Index of topmost displayed row	*/
/*  numRows	: uint		: Number of rows actually visible	*/
/*  (returns)	: void		:					*/

void	KBDisplay::setRowRange
	(	uint	totalRows,
		uint	extra,
		uint	curQRow,
		uint	curDRow,
		uint	numRows
	)
{
	if (scroller)
		scroller->setRowRange (totalRows, extra, curQRow, curDRow, numRows) ;
	else	widget  ->setRowRange (totalRows, extra, curQRow, curDRow, numRows) ;
}

/*  KBDisplay								*/
/*  scrollToRow	: Pass row scroll up to owning form block if any	*/
/*  row		: uint		: Row number				*/
/*  (returns)	: void		:					*/

void	KBDisplay::scrollToRow
	(	uint	row
	)
{
	KBFormBlock *block = owner->isFormBlock () ;

	if (block == 0) block = owner->getFormBlock () ;
	if (block != 0) block->scrollToRow (row) ;
}

/*  KBDisplay								*/
/*  scrollBy	: Scroll by number of rows				*/
/*  delta	: int		: Number of rows			*/
/*  (returns)	: void		:					*/

void	KBDisplay::scrollBy
	(	int	delta
	)
{
	KBFormBlock *block = owner->isFormBlock () ;

	if (block == 0) block = owner->getFormBlock () ;
	if (block != 0) block->scrollBy (delta) ;
}

/*  KBDisplay								*/
/*  doAction	: Pass action though to block				*/
/*  action	: KB::Action	: Action in question			*/
/*  (returns)	: void		:					*/

void	KBDisplay::doAction
	(	KB::Action	action
	)
{
	KBFormBlock *block = owner->isFormBlock () ;

	if (block == 0) block = owner->getFormBlock () ;
	if (block != 0) block->doAction (action) ;
}

/*  KBDisplay								*/
/*  keyStroke	: Handle keystroke					*/
/*  key		: QKeyEvent *	: Key event				*/
/*  (returns)	: bool		: Event consumed			*/

bool	KBDisplay::keyStroke
	(	QKeyEvent	*key
	)
{
	KBFormBlock *block = owner->isFormBlock () ;
	return	block == 0 ? false : block->KBNavigator::keyStroke (0, key) ;
}

/*  KBDisplay								*/
/*  resizeEvent	: Pass size change up to owning form block if any	*/
/*  width	: uint		: New width				*/
/*  height	: uint		: New height				*/
/*  (returns)	: void		:					*/

void	KBDisplay::resizeEvent
	(	uint	width,
		uint	height
	)
{
	KBFormBlock *block ;
	if ((block = owner->isFormBlock ()) != 0)
		block->resize (width, height) ;
}

/*  KBDisplay								*/
/*  showAs	: Set or clear design mode				*/
/*  mode	: KB::ShowAs	: Design mode				*/
/*  (returns)	: void		:					*/

void	KBDisplay::showAs
	(	KB::ShowAs	mode
	)
{
	showing	= mode ;
	if (scroller)
		scroller->showAs (mode) ;
	else	widget  ->showAs (mode) ;
}

/*  KBDisplay								*/
/*  setStretchable							*/
/*		: Set or clear stretch mode				*/
/*  stretch	: bool		: Can stretch				*/
/*  (returns)	: void		:					*/

void	KBDisplay::setStretchable
	(	bool	stretch
	)
{
	if (scroller)
		scroller->setStretchable (stretch) ;
}

/*  KBDisplay								*/
/*  setShowbar	: Set or clear scrollbar mode				*/
/*  showBar	: uint		: Show row scroller bar/mini-nav	*/
/*  (returns)	: void		:					*/

void	KBDisplay::setShowbar
	(	uint	showbar
	)
{
	if (scroller)
		scroller->setShowbar (showbar) ;
	else	widget  ->setShowbar (showbar) ;
}

/*  KBDisplay								*/
/*  setTitle	: Set "group box" title					*/
/*  title	: const QString & : Title text				*/
/*  (returns)	: void		  :					*/

void	KBDisplay::setTitle
	(	const QString	&title
	)
{
	if (widget) widget->setTitle (title) ;
}

/*  KBDisplay								*/
/*  setFrame	: Set frame atyle					*/
/*  title	: const QString & : Title text				*/
/*  (returns)	: void		  :					*/

void	KBDisplay::setFrame
	(	int	style,
		int	width
	)
{
	if (widget) widget->setFrame (style, width) ;
}

/*  KBDisplay								*/
/*  getBaseSize	: Get base size of owner				*/
/*  (returns)	: QSize		: Base size				*/

QSize	KBDisplay::getBaseSize ()
{
	return	owner->getBaseSize () ;
}

/*  KBDisplay								*/
/*  addMorph	: Add a morphing control				*/
/*  morph	: KBItem *	: Control in question			*/
/*  (returns)	: void		:					*/

void	KBDisplay::addMorph
	(	KBItem		*morph
	)
{
	if (scroller)
		scroller->addMorph (morph) ;
	else	widget  ->addMorph (morph) ;
}

/*  KBDisplay								*/
/*  remMorph	: Remove a morphing control				*/
/*  morph	: KBItem *	: Control in question			*/
/*  (returns)	: void		:					*/

void	KBDisplay::remMorph
	(	KBItem		*morph
	)
{
	if (scroller)
		scroller->remMorph (morph) ;
	else	widget  ->remMorph (morph) ;
}

/*  KBDisplay								*/
/*  updateMorph	: Update a morphing control				*/
/*  morph	: KBItem *	: Control in question			*/
/*  mRect	: const QRect &	: Area occupied by control		*/
/*  (returns)	: void		:					*/

void	KBDisplay::updateMorph
	(	KBItem		*morph,
		const QRect	&mRect
	)
{
	if (scroller)
	{	QRect	vRect	= mRect	;
		cvtCtrlToView (vRect)	;
		scroller->updateMorph (morph, vRect) ;
	}
	else	widget  ->updateMorph (morph, mRect) ;
}

/*  KBDisplay								*/
/*  updateMorph	: Update a morphing control				*/
/*  morph	: KBItem *	: Control in question			*/
/*  drow	: uint		: Row affected				*/
/*  (returns)	: void		:					*/

void	KBDisplay::updateMorph
	(	KBItem		*morph,
		uint		drow
	)
{
	if (scroller)
		scroller->updateMorph (morph, drow) ;
	else	widget  ->updateMorph (morph, drow) ;
}

/*  KBDisplay								*/
/*  setTagLabel	: Set design mode tag label text			*/
/*  text	: const QString & : Text				*/
/*  (returns)	: void		  :					*/

void	KBDisplay::setTagLabel
	(	const QString	&text
	)
{
	if (widget)
		widget->setTagLabel (text) ;
}
