/***************************************************************************
    file	         : kb_queryset.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<time.h>

#include	<qlist.h>

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_database.h"
#include	"kb_item.h"
#include	"kb_queryset.h"
#include	"kb_qrylevel.h"

#include	"tk_messagebox.h"

/*  KBRowSetValue							*/
/*  -------------							*/
/*  This structure hold a value in a row in a query set. The main value	*/
/*  is set from the database. If the user updates the field then the	*/
/*  update value is set; when (and if) the row is synchronised, then	*/
/*  the update value is copied back and cleared.			*/

class	KBRowSetValue 
{
	KBValue		m_value		;
	KBValue		*m_update	;

public	:

	inline	KBRowSetValue ()
	{
		m_update = 0 ;
	}
	inline	~KBRowSetValue ()
	{
		if (m_update != 0) delete m_update ;
	}

	/* Get the current value. This will be the updated value if it	*/
	/* is set, unless the "initial" flag is set.			*/
	inline	const	KBValue	&getValue
		(	bool		&dirty,
			bool		initial	= false
		)
	{
		dirty	= m_update != 0 ;
		if (!dirty || initial) return m_value ;
		return	*m_update ;
	}

	/* Set the current value. If the initial flag is set the set	*/
	/* the main value and clear any updated value, otherwise create	*/
	/* an updated value if needed and save the value.		*/
	inline	void	setValue
		(	const KBValue	&value,
			bool		initial = false
		)
	{
		if (initial)
		{	m_value	=   value  ;
			DELOBJ	(m_update) ;
			return	;	   ;
		}
		if (m_update != 0)
		{	*m_update = value  ;
			return	;
		}
		m_update = new KBValue (value) ;
	}

	/* Reset the value to the initial state if it has been updated	*/
	/* at some time.						*/
	inline	void	reset	()
	{
		DELOBJ	(m_update) ;
	}

	/* Get the value type. We assume that types no not change in	*/
	/* the lifetime of a rowset value, so return the type of the	*/
	/* main value whether we have been updated or not.		*/
	inline	KBType	*getType ()
	{
		return	m_value.getType() ;
	}
}	;


/*  ------------------------------------------------------------------  */

/*  KBRowSet								*/
/*  KBRowSet	: Row set constructor					*/
/*  _size	: uint		: Number of values (fields) in row	*/
/*  (returns)	: KBRowSet	:					*/

KBRowSet::KBRowSet
	(	uint	size
	)
{
	m_values	= new KBRowSetValue[m_size = size] ;
	m_subset	= 0		;
	m_state		= KB::RSInSync	;
	m_dirty		= true		;
	m_marked	= false		;
}

/*  KBRowSet								*/
/*  ~KBRowSet	: Row set destructor					*/
/*  (returns)	:		:					*/

KBRowSet::~KBRowSet ()
{
	delete	[] m_values ;
	if (m_subset != 0) delete m_subset ;
}


/*  KBQuerySetSortList							*/
/*  compareItems: Compare two items for ordering			*/
/*  item1	: QCollection::Item	: First item			*/
/*  item2	: QCollection::Item	: Second item			*/
/*  (returns)	: int			: Ordering value		*/

int	KBQuerySetSortList::compareItems
	(	QCollection::Item	item1,
		QCollection::Item	item2
	)
{
	KBRowSet 	*row1	= (KBRowSet *)item1 ;
	KBRowSet 	*row2	= (KBRowSet *)item2 ;
	const QString	&text1	= *row1->m_order ;
	const QString	&text2	= *row2->m_order ;
	int	 	order	;

	switch (m_sortType)
	{
		case KB::ITFixed	:
			order	= text1.toInt() - text2.toInt() ;
			break	;

		case KB::ITFloat	:
		case KB::ITDecimal	:
			{
			double	d = text1.toDouble() - text2.toDouble() ;
			order	= d < 0.0 ? -1 : d > 0.0 ? +1 : 0 ;
			break	;
			}

		default	:
			order	= text1.compare (text2) ;
			break	;
	}

	return	 m_sortAsc ? order : -order ;
}


/*  ------------------------------------------------------------------  */

/*  KBQuerySet								*/
/*  KBQuerySet	: Constructor for the query result set object		*/
/*  nFields	: uint		: Total number of fields		*/
/*  (returns)	: KBQuerySet	:					*/

KBQuerySet::KBQuerySet
	(	uint		nFields
	)
	:
	m_nFields	  (nFields)
{
	/* Set the query row list to auto-delete so that we don't have	*/
	/* to bother cleaning up.					*/
	m_rowSets.setAutoDelete (true) ;

	m_totalRows  = 0  ;
	m_lastMarked = -1 ;

	m_widths     = new uint		  [m_nFields] ;
	m_vtrans     = new KBValue::VTrans[m_nFields] ;

	for (uint f = 0 ; f < nFields ; f += 1)
	{
		m_widths[f] = 0 ;
		m_vtrans[f] = KBValue::VDefault ;
	}
}

/*  KBQuerySet								*/
/*  ~KBQuerySet	: Destructor for the query result set object		*/
/*  (returns)	:		:					*/

KBQuerySet::~KBQuerySet ()
{
	delete	[]m_widths  ;
	delete	[]m_vtrans  ;
}

/*  KBQuerySet								*/
/*  clear	: Clear all contents					*/
/*  (returns)	: void		:					*/

void	KBQuerySet::clear ()
{
	m_totalRows  = 0  ;
	m_lastMarked = -1 ;

	m_rowSets.clear() ;

	memset	(m_widths, 0, sizeof(uint) * m_nFields) ;
}

/*  KBQuerySet								*/
/*  getWidth	: Get nominal maximum column width			*/
/*  qcol	: uint		: Column number				*/
/*  (returns)	: uint		: Width					*/

uint	KBQuerySet::getWidth
	(	uint		qcol
	)
{
	return	qcol >= m_nFields ? 0 : m_widths[qcol] ;
}

/*  KBQuerySet								*/
/*  setVTrans	: Set value translation for column			*/
/*  qcol	: uint		  : Column number				*/
/*  vtrans	: KBValue::VTrans : Translation mode			*/
/*  (returns)	: void		  :					*/

void	KBQuerySet::setVTrans
	(	uint		qcol,
		KBValue::VTrans	vtrans
	)
{
	if (qcol < m_nFields) m_vtrans[qcol] = vtrans ;
}

/*  KBQuerySet								*/
/*  getVTrans	: Get value translation for column			*/
/*  qcol	: uint		  : Column number			*/
/*  (returns)	: KBValue::VTrans : Translation mode			*/

KBValue::VTrans	KBQuerySet::getVTrans
	(	uint		qcol
	)
{
	return	qcol >= m_nFields ? KBValue::VDefault : m_vtrans[qcol] ;
}


/*  KBQuerySet								*/
/*  insertRow	: Insert a new row at a specified point			*/
/*  qrow	: uint		: Position to insert			*/
/*  (returns)	: void		:					*/

void	KBQuerySet::insertRow
	(	uint	qrow
	)
{
	m_rowSets.insert (qrow, new KBRowSet (m_nFields)) ;
	for ( ; qrow < m_rowSets.count() ; qrow += 1)
		m_rowSets.at(qrow)->m_dirty = true ;
}

/*  KBQuerySet								*/
/*  deleteRow	: Delete a row set					*/
/*  qrow	: uint		: Row to be deleted			*/
/*  (returns)	: void		:					*/

void	KBQuerySet::deleteRow
	(	uint	qrow
	)
{
	m_rowSets.remove (qrow) ;
	for ( ; qrow < m_rowSets.count() ; qrow += 1)
		m_rowSets.at(qrow)->m_dirty = true ;
}

/*  KBQuerySet								*/
/*  setField	: Set a field value					*/
/*  qrow	: uint		 : Row number				*/
/*  qcol	: uint		 : Column number			*/
/*  value	: const KBValue &: Value to be set			*/
/*  initial	: bool		 : Set as initial value			*/
/*  ckc		: bool		 : Check and return changed		*/
/*  (returns)	: void		 :					*/

bool	KBQuerySet::setField
	(	uint		qrow,
		uint		qcol,
		const KBValue	&value,
		bool		initial
	)
{
	KBRowSet *rowSet ;
	bool	 changed = false ;

//	fprintf
//	(	stderr,
//		"KBQuerySet::setField: qrow=%d qcol=%d ini=%d [%s]\n",
//		qrow,
//		qcol,
//		initial,
//		(cchar *)value.getRawText()
//	)	;

	/* Check that we are at most extending the query set by one	*/
	/* row, and that we are in range of the number of fields. Treat	*/
	/* an error in either case as fatal.				*/
	if (qrow >= m_rowSets.count() + 1)
		KBError::EFatal
		(
			QString	(TR("KBQuerySet::setField(%1,%2) with %3 rows"))
				 .arg(qrow)
				 .arg(qcol)
				 .arg(m_rowSets.count()),
			QString::null,
			__ERRLOCN
		)	;

	if (qcol >= m_nFields)
		KBError::EFatal
		(
			QString	(TR("KBQuerySet::setField(%1,%2) with %3 fields"))
				 .arg(qrow)
				 .arg(qcol)
				 .arg(m_nFields),
			QString::null,
			__ERRLOCN
		)	;


	/* If extending then add a new row set and set its state to	*/
	/* "inserted", otherwise get the row which will be changed and	*/
	/* mark it as updated if it is currently in sync (this avoids	*/
	/* marking a newly "inserted" row as "changed".			*/
	if (qrow == m_rowSets.count())
	{
		rowSet = new KBRowSet (m_nFields) ;
		m_rowSets.append (rowSet)	  ;
		rowSet->m_state = KB::RSInserted  ;
		changed		= true		  ;
	}
	else
	{
		bool	dummy	;
		rowSet  = m_rowSets.at (qrow)	  ;
		changed = rowSet->m_values[qcol].getValue(dummy) != value ;

		if (rowSet->m_state == KB::RSInSync)
			if (changed)
				rowSet->m_state = KB::RSChanged ;
	}

	rowSet->m_values[qcol].setValue (value, initial) ;
	rowSet->m_dirty = true ;

	uint	w = value.dataLength() ;
	if (w > m_widths[qcol]) m_widths[qcol] = w ;

	return	changed	;
}

/*  KBQuerySet								*/
/*  getField	: Get a field value including dirty flag		*/
/*  qrow	: uint		  : Row number				*/
/*  qcol	: uint		  : Column number			*/
/*  dirty	: bool &	  : Value dirty flag			*/
/*  initial	: bool		  : Get initial value			*/
/*  (returns)	: const KBValue	& :					*/

const KBValue
	&KBQuerySet::getField
	(	uint	qrow,
		uint	qcol,
		bool	&dirty,
		bool	initial
	)
{
	dirty	= false	;
	if (qrow >= m_rowSets.count()) return m_empty ;
	if (qcol >= m_nFields        ) return m_empty ;

	return	m_rowSets.at(qrow)->m_values[qcol].getValue (dirty, initial) ;
}

/*  KBQuerySet								*/
/*  rowIsDirty	: Check whether row is dirty wrt. the display		*/
/*  qrow	: uint		: Query row				*/
/*  reset	: bool		: True to reset dirty flag		*/
/*  (returns)	: bool		: Dirty					*/

bool	KBQuerySet::rowIsDirty
	(	uint	qrow,
		bool	reset
	)
{
	/* Check that we are at most extending the query set by one	*/
	if (qrow < m_rowSets.count ())
	{
		KBRowSet *rowSet = m_rowSets.at(qrow) ;
		bool	 rc	 = rowSet->m_dirty    ;

		if (reset) rowSet->m_dirty = false    ;

//		fprintf
//		(	stderr,
//			"KBQuerySet::rowIsDirty: qrow=%d d=%d\n",
//			qrow,
//			rc
//		)	;

		return	rc ;
	}

	return	true	;
}

/*  KBQuerySet								*/
/*  markAllDirty: Mark all rows as dirty				*/
/*  (returns)	: void		:					*/

void	KBQuerySet::markAllDirty ()
{
	LITER
	(	KBRowSet,
		m_rowSets,
		rs,
		rs->m_dirty = true
	)
}

/*  KBQuerySet								*/
/*  setRowState	: Set state on a specifed row				*/
/*  qrow	: uint		: Row number				*/
/*  state	: KB::RState	: State					*/
/*  (returns)	: void		:					*/

void	KBQuerySet::setRowState
	(	uint		qrow,
		KB::RState	state
	)
{
	if (qrow >= m_rowSets.count())
		return ;

	if ((state == KB::RSDeleted) && (m_rowSets.at(qrow)->m_state == KB::RSInserted))
	{
		deleteRow (qrow) ;
		return	  ;
	}

	m_rowSets.at(qrow)->m_state = state ;
	m_rowSets.at(qrow)->m_dirty = true  ;
}

/*  KBQuerySet								*/
/*  getRowFlags	: Get flags on a specifed row				*/
/*  qrow	: uint		: Row number				*/
/*  def		; KB::RState	: Default state				*/
/*  (returns)	: uint		: Flags					*/

KB::RState KBQuerySet::getRowState
	(	uint		qrow,
		KB::RState	def
	)
{
	return	qrow < m_rowSets.count() ? m_rowSets.at(qrow)->m_state : def ;
}


/*  KBQuerySet								*/
/*  clearAllMarked							*/
/*		: Clear marks on all rows				*/
/*  (returns)	: void		:					*/

void	KBQuerySet::clearAllMarked ()
{
	for (KBRowSet *rs = m_rowSets.first() ; rs != 0 ; rs = m_rowSets.next())
		rs->m_marked = false ;
}

/*  KBQuerySet								*/
/*  setRowMarked: Set marked on a specifed row				*/
/*  qrow	: uint		: Row number				*/
/*  op		: StateOpMark	: Mark operation			*/
/*  (returns)	: void		:					*/

void	KBQuerySet::setRowMarked
	(	uint		qrow,
		KB::MarkOp	op
	)
{
	KBRowSet *rowSet = qrow < m_rowSets.count() ? m_rowSets.at(qrow) : 0 ;

	switch (op)
	{
		case KB::MarkOpSet    :
			/* Mark just the specified row. Clear current	*/
			/* marks, then set provided the row is in range	*/
			/* The row is noted for subsequent range	*/
			/* marking.					*/
			clearAllMarked () ;

			if (rowSet != 0)
			{	rowSet->m_marked= true	;
				m_lastMarked 	= qrow	;
			}
			else	m_lastMarked	= -1	;

			break	;

		case KB::MarkOpToggle :
			/* Toggle the mark state if the row is in range	*/
			/* and note the row for subsequent range	*/
			/* marking.					*/
			if (rowSet != 0)
			{	rowSet->m_marked= !rowSet->m_marked ;
				m_lastMarked	= qrow	;
			}
			else	m_lastMarked	= -1	;

			break	;

		case KB::MarkOpRange  :
			/* Mark the range from the specified row to the	*/
			/* last marked row. There are various checks to	*/
			/* ensure we don't overrun.			*/
			clearAllMarked () ;

			if ((rowSet != 0) && (m_lastMarked >= 0))
			{
				uint	rf = (uint)m_lastMarked < qrow ? m_lastMarked : qrow ;
				uint	rt = (uint)m_lastMarked > qrow ? m_lastMarked : qrow ;

				rt += 1 ;
				if (rt > m_rowSets.count()) rt = m_rowSets.count() ;

				while (rf < rt)
				{	m_rowSets.at(rf)->m_marked = true ;
					rf += 1 ;
				}
			}

			break	;

		case KB::MarkOpSetAll :
			/* Opening and closing braces added to		*/
			/* suppress warning message - JD during		*/
			/* Windows build				*/
			{
			/* Set mark on all rows.												*/
			for (KBRowSet *rs = m_rowSets.first() ; rs != 0 ; rs = m_rowSets.next())
				rs->m_marked = true ;
			}
			break	;

		case KB::MarkOpClear  :
			/* Clear all marks. The last marked row is also	*/
			/* cleared.					*/
			clearAllMarked () ;
			m_lastMarked = -1 ;
			break	;

		default	:
			break	;
	}
}

/*  KBQuerySet								*/
/*  getRowMarked: Return whether a row is marked or not			*/
/*  qrow	: uint		: Row number				*/
/*  (returns)	: bool		: True if marked			*/

bool	KBQuerySet::getRowMarked
	(	uint	qrow
	)
{
	return	qrow < m_rowSets.count() ? m_rowSets.at(qrow)->m_marked : false ;
}



/*  KBQuerySet								*/
/*  setSubset	: Get query subset for specified row			*/
/*  qrow	: uint		: Row number				*/
/*  width	: uint		: Query set width required		*/
/*  (returns)	: KBQuerySet *	: Query set				*/

KBQuerySet *KBQuerySet::getSubset
	(	uint	qrow,
		uint	width
	)
{
	if (qrow >= m_rowSets.count())
		KBError::EFatal
		(
			QString	(TR("KBQuerySet::setGetsubset(%1) with %2 rows"))
				 .arg(qrow)
				 .arg(m_rowSets.count()),
			QString::null,
			__ERRLOCN
		)	;

	KBQuerySet *subset = m_rowSets.at(qrow)->m_subset ;

	if (subset == 0)
	{
		subset = new KBQuerySet (width) ;
		m_rowSets.at(qrow)->m_subset = subset ;
	}

	return	subset	;
}

/*  KBQuerySet								*/
/*  sortByColumn: Sort query set by column				*/
/*  qcol	: uint		: Column number				*/
/*  asc		: bool		: Ascending				*/
/*  item	: KBItem	: Item associated with column		*/
/*  (returns)	: void		:					*/

void	KBQuerySet::sortByColumn
	(	uint	qcol,
		bool	asc,
		KBItem	*item
	)
{
	if ((qcol >= m_nFields) || (m_rowSets.count() <= 1)) return ;


	m_rowSets.m_sortCol  = qcol	;
	m_rowSets.m_sortAsc  = asc	;
	m_rowSets.m_sortItem = item	;
	m_rowSets.m_sortType = item->getOrderType() ;

	/* Speed hack, extract all the ordering values first, then run	*/
	/* the sort. This gives O(n) time to get the values rather	*/
	/* rather than the O(sort steps) for this part. Sort speed is	*/
	/* about doubled on a 30000-row table.				*/
	bool dirty ;
	for (uint idx1 = 0 ; idx1 < m_rowSets.count() ; idx1 += 1)
		m_rowSets.at(idx1)->m_order = new QString(item->getOrderValue(m_rowSets.at(idx1)->m_values[qcol].getValue(dirty))) ;

//	fprintf	(stderr, "start: %d\n", time(0)) ;
	m_rowSets.sort () ;
//	fprintf	(stderr, "end  : %d\n", time(0)) ;

	for (uint idx2 = 0 ; idx2 < m_rowSets.count() ; idx2 += 1)
		delete m_rowSets.at(idx2)->m_order ;
}

/*  KBQuerySet								*/
/*  deleteAllMarked							*/
/*		: Delete all marked rows				*/
/*  nrows	: uint &	: Return number of rows deleted		*/
/*  parent	: KBNode *	: Parent node to query level		*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: bool		: Success				*/

bool	KBQuerySet::deleteAllMarked
	(	uint		&nrows,
		KBNode		*parent,
		KBError		&pError
	)
{

	if (KBOptions::getVerDelete() != KBOptions::VerifyNever)
	{
		uint	found	= 0 ;

		for (KBRowSet *rs = m_rowSets.first() ; rs != 0 ; rs = m_rowSets.next())
			if (rs->m_marked)
			{	found  += 1 ;
				if (found > 1) break ;
			}
				
		if (found > 1)
		{
			QString	errText	;

			if (parent->isItem ()) errText = parent->isItem()->getErrText() ;
			if (errText.isEmpty()) errText = TR("record") ;

			if (TKMessageBox::questionYesNo
				(	0,
					QString (TR("You are about to delete more than one %2: proceed?"))
						.arg(errText),
					TR("Delete marked records")
				)
				!= TKMessageBox::Yes)
			{
				pError	= KBError
					  (	KBError::None,
						TR("User cancelled delete"),
						QString::null,
						__ERRLOCN
					  )	;
				return	false	;
			}
		}
	}

	nrows	= 0	;
	for (KBRowSet *rs = m_rowSets.first() ; rs != 0 ; rs = m_rowSets.next())
		if (rs->m_marked)
		{
			rs->m_state = KB::RSDeleted ;
			rs->m_dirty = true  ;
			nrows	   += 1 ;
		}

	return	true	;
}

void	KBQuerySet::resetData
	(	uint	qrow
	)
{
	if (qrow < m_rowSets.count())
	{
		KBRowSet *rowSet = m_rowSets.at(qrow) ;

		for (uint col = 0 ; col < m_nFields ; col += 1)
			rowSet->m_values[col].reset() ;

		rowSet->m_dirty = true ;
	}
}
