/***************************************************************************
    file	         : kb_primarydlg.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<qlist.h>

#include	"kb_classes.h"
#include	"kb_attr.h"
#include	"kb_database.h"
#include	"kb_dialog.h"

#ifndef		_WIN32
#include	"kb_primarydlg.moc"
#else
#include	"kb_primarydlg.h"
#endif



struct	UniqueTypeMap
{	KBTable::UniqueType	m_type		;
	bool			m_justOne	;
	bool			m_withExpr	;
	cchar			*m_descr	;
}	;

static	UniqueTypeMap	uMap[] =
{
{	KBTable::Auto,		false,	false,	"Auto"			},
{	KBTable::PrimaryKey,	true,	false,	"Primary key only"	},
{	KBTable::AnyUnique,	true,	false,	"Any unique column"	},
{	KBTable::AnySingle,	true,	false,	"Any single column"	},
{	KBTable::PreExpression,	true,	true,	"Column/Pre-Expression"	},
{	KBTable::PostExpression,true,	true,	"Column/Post-Expression"},
#ifdef	__NOTYET
{	KBTable::Multiple,	false,	false,	"Multiple column"	},
{	KBTable::TableSetup,	false,	false,	"Uses table setup"	}
#endif
}	;


/*  KBPrimaryDlg							*/
/*  KBPrimaryDlg: Constructor for primary (unique column) selector	*/
/*  parent	: QWidget *	: Parent widget				*/
/*  tabSpec	: KBTableSpec & : Table information object		*/
/*  justOne	: bool		: Only allow single column cases	*/
/*  withExpr	: bool		: Allow options with expressions	*/
/*  (returns)	: KBPrimaryDlg	:					*/

KBPrimaryDlg::KBPrimaryDlg
	(	QWidget		*parent,
		KBTableSpec	&tabSpec,
		bool		justOne,
		bool		withExpr
	)
	:
	QWidget		(parent),
	m_tabSpec	(tabSpec),
	m_mode		(this),
	m_singleColumn	(this),
	m_multiColumns	(this),
	m_expression	(this)
{
	QVBoxLayout *m_layMain = new QVBoxLayout (this) ;

	m_layMain->addWidget (&m_mode) ;
	m_layMain->addWidget (&m_singleColumn) ;
	m_layMain->addWidget (&m_multiColumns) ;
	m_layMain->addWidget (&m_expression  ) ;
	m_layMain->addStretch() ;

	m_layMain->setStretchFactor(&m_mode, 	     0) ;
	m_layMain->setStretchFactor(&m_singleColumn, 0) ;
	m_layMain->setStretchFactor(&m_multiColumns, 1) ;
	m_layMain->setStretchFactor(&m_expression,   0) ;

	for (uint idx = 0 ; idx < sizeof(uMap)/sizeof(UniqueTypeMap) ; idx += 1)
	{
		if (justOne   && !uMap[idx].m_justOne ) continue ;
		if (!withExpr &&  uMap[idx].m_withExpr) continue ;

		m_mode	    .insertItem (TR(uMap[idx].m_descr)) ;
		m_itemToType.append     (uMap[idx].m_type) ;
	}

	connect	(&m_mode, SIGNAL(activated(int)), SLOT(modeChanged())) ;

	m_multiColumns.setMultiSelection (true) ;

	_KBDialog::setupLayout (m_layMain, -1, -1) ;
}

/*  KBPrimaryDlg							*/
/*  loadAllKeys	: Load all columns as possible keys			*/
/*  (returns)	: void		:					*/

void	KBPrimaryDlg::loadAllKeys ()
{
	m_singleColumn.clear () ;

	LITER
	(	KBFieldSpec,
		m_tabSpec.m_fldList,
		fSpec,

		m_singleColumn.insertItem (fSpec->m_name) ;
	)
}

/*  KBPrimaryDlg							*/
/*  loadPrimaryKey: Load just the primary key as a possibility		*/
/*  (returns)	  : void		:				*/

void	KBPrimaryDlg::loadPrimaryKey ()
{
	m_singleColumn.clear () ;

	KBFieldSpec *fSpec = m_tabSpec.findPrimary() ;
	if (fSpec == 0)
	{
		KBError::EError
		(	QString(TR("Table does not have a primary key")),
			QString::null,
			__ERRLOCN
		)	;
		return	;
	}

	m_singleColumn.insertItem (fSpec->m_name) ;
}

/*  KBPrimaryDlg							*/
/*  loadUniqueKeys: Load just the unique keys as possibilities		*/
/*  (returns)	  : void		:				*/

void	KBPrimaryDlg::loadUniqueKeys ()
{
	m_singleColumn.clear () ;

	LITER
	(	KBFieldSpec,
		m_tabSpec.m_fldList,
		fSpec,

		if ((fSpec->m_flags & KBFieldSpec::Unique) != 0)
			m_singleColumn.insertItem (fSpec->m_name) ;
	)

	if (m_singleColumn.count() == 0)
		KBError::EError
		(	QString(TR("Table does not have any unique columns")),
			QString::null,
			__ERRLOCN
		)	;
}

/*  KBPrimaryDlg							*/
/*  loadMultiKeys: Load all columns for multi-column key selection	*/
/*  (returns)	 : void		:					*/

void	KBPrimaryDlg::loadMultiKeys ()
{
	m_multiColumns.clear () ;

	LITER
	(	KBFieldSpec,
		m_tabSpec.m_fldList,
		fSpec,

		m_multiColumns.insertItem (fSpec->m_name) ;
	)
}

/*  KBPrimaryDlg							*/
/*  modeChanged	: User changes column selection mode			*/
/*  (returns)	: void		:					*/

void	KBPrimaryDlg::modeChanged ()
{
	bool	single = false	;
	bool	multi  = false	;
	bool	expr   = false	;

	switch (m_itemToType[m_mode.currentItem()])
	{
		case KBTable::Auto	  :
			break	;

		case KBTable::PrimaryKey  :
			single	= true	  ;
			loadPrimaryKey () ;
			break	;

		case KBTable::AnyUnique	  :
			single	= true	  ;
			loadUniqueKeys () ;
			break	;

		case KBTable::AnySingle   :
			single	= true	  ;
			loadAllKeys    () ;
			break	;

		case KBTable::PreExpression 	:
		case KBTable::PostExpression 	:
			single	= true    ;
			expr	= true	  ;
			loadAllKeys    () ;
			break	;

#ifdef	__NOTYET
		case KBTable::Multiple    :
			multi	= true	  ;
			break	;

		case KBTable::TableSetup :
			break	;
#endif
		default	:
			break	;
	}

	if (single)
		m_singleColumn.show() ;
	else	m_singleColumn.hide() ;

	if (multi)
		m_multiColumns.show() ;
	else	m_multiColumns.hide() ;

	if (expr)
		m_expression  .show() ;
	else	m_expression  .hide() ;
}

/*  KBPrimaryDlg							*/
/*  set		: Set current selection					*/
/*  names	: QStringList & : Column names				*/
/*  utype	: UniqueType	: Uniqueness type			*/
/*  (returns)	: void		:					*/

void	KBPrimaryDlg::set
	(	QStringList		&names,
		KBTable::UniqueType	utype,
		const QString		&expr
	)
{
	/* Treat the Legacy100 setting as AnySingle so that the column	*/
	/* will be displayed.						*/
	if (utype == KBTable::Legacy100)
		utype	= KBTable::AnySingle	;


	/* Scan the acceptable types and set the combo box as		*/
	/* approriate. We *should* always find the required value.	*/
	for (uint idx = 0 ; idx < m_itemToType.count() ; idx += 1)
		if (m_itemToType[idx] == utype)
		{	m_mode.setCurrentItem (idx) ;
			break ;
		}


	m_expression.setText (expr) ;

	/* Load all possible keys into the multi-column selection list	*/
	/* and then get the correct mode displayed.			*/
	loadMultiKeys	() ;
	modeChanged	() ;


	/* Now set the current selection (or selections) in each of the	*/
	/* single-column and multi-column lists.			*/
	for (int  idx0 = 0 ; idx0 < m_singleColumn.count() ; idx0 += 1)
		if (names.findIndex(m_singleColumn.text(idx0)) >= 0)
		{
			m_singleColumn.setCurrentItem (idx0) ;
			break	;
		}

	for (uint idx1 = 0 ; idx1 < m_multiColumns.count() ; idx1 += 1)
		if (names.findIndex(m_multiColumns.text(idx1)) >= 0)
		{
			m_multiColumns.setSelected(idx1, true) ;
		}
}

/*  KBPrimaryDlg							*/
/*  set		: Set current selection					*/
/*  name	: const QString & : Current selection			*/
/*  utype	: UniqueType	: Uniqueness type			*/
/*  (returns)	: void		  :					*/

void	KBPrimaryDlg::set
	(	const QString		&name,
		KBTable::UniqueType	utype,
		const QString		&expr
	)
{
	QStringList  names  ;
	names.append (name) ;

	set (names, utype, expr)  ;
}

/*  KBPrimaryDlg							*/
/*  retrieve	: Retrieve selection					*/
/*  columns	: QString &	: Return columns			*/
/*  (returns	: UniqueType	: Uniqueness type			*/

KBTable::UniqueType
	KBPrimaryDlg::retrieve
	(	QString		&columns,
		QString		&expr
	)
{
	QStringList	    list ;
	KBTable::UniqueType type = m_itemToType[m_mode.currentItem()] ;

	switch (type)
	{
		case KBTable::Auto	 :
			columns	= QString::null ;
			break	;

		case KBTable::PrimaryKey :
		case KBTable::AnyUnique	 :
		case KBTable::AnySingle	 :
			columns	= m_singleColumn.currentText() ;
			break	;

		case KBTable::PreExpression 	:
		case KBTable::PostExpression 	:
			columns	= m_singleColumn.currentText() ;
			break	;

#ifdef	__NOTYET
		case KBTable::Multiple	 :
			for (uint idx = 0 ; idx < m_multiColumns.count() ; idx += 1)
				if (m_multiColumns.isSelected (idx))
					list.append (m_multiColumns.text(idx)) ;
			columns	= list.join (",") ;
			break	;

		case KBTable::TableSetup :
			columns	= QString::null ;
			break	;
#endif
		default	:
			columns	= QString::null ;
			break	;
	}

	expr	= m_expression.text() ;

	return	type	;
}

/*  ------------------------------------------------------------------  */

/*  KBQryPrimaryDlg							*/
/*  KBQryPrimaryDlg							*/
/*		: Constructor for key selection dialog			*/
/*  tabSpec	: KBTableSpec &	  : Table specification			*/
/*  names	: QStringList &	  : Current key columns			*/
/*  utype	: UniqueType	  : Current uniqueness type		*/
/*  (returns)	: KBQryPrimaryDlg :					*/

KBQryPrimaryDlg::KBQryPrimaryDlg
	(	KBTableSpec		&tabSpec,
		QStringList		&names,
		KBTable::UniqueType	utype,
		const QString		&expr
	)
	:
	_KBDialog	("Primary Key", true),
	m_primaryDlg	(this, tabSpec, true, false),
	m_bOK		(this),
	m_bCancel	(this)
{
	QVBoxLayout	*layMain = new QVBoxLayout(this) ;
	layMain->addWidget (&m_primaryDlg) ;

	QHBoxLayout	*layButt = new QHBoxLayout(layMain) ;
	layButt->addStretch () ;
	layButt->addWidget  (&m_bOK    ) ;
	layButt->addWidget  (&m_bCancel) ;

	m_bOK    .setText   (TR("OK"    )) ;
	m_bCancel.setText   (TR("Cancel")) ;

	connect	(&m_bOK,     SIGNAL(clicked()), SLOT(clickOK    ())) ;
	connect	(&m_bCancel, SIGNAL(clicked()), SLOT(clickCancel())) ;

	m_primaryDlg.set (names, utype, expr) ;

	/* Note that all the work is done in the KBPrimaryDlg object.	*/
	/* This class is just a wrapper.				*/
}

/*  KBQryPrimaryDlg							*/
/*  ~KBQryPrimaryDlg							*/
/*		: Destructor for key selection dialog			*/
/*  (returns)	:		:					*/

KBQryPrimaryDlg::~KBQryPrimaryDlg ()
{
}


/*  KBQryPrimaryDlg							*/
/*  clickOK	: User clicks OK					*/
/*  (returns)	: void		:					*/

void	KBQryPrimaryDlg::clickOK ()
{
	done	(1) ;
}

/*  KBQryPrimaryDlg							*/
/*  clickCancel	: User clicks cancel					*/
/*  (returns)	: void		:					*/

void	KBQryPrimaryDlg::clickCancel ()
{
	done	(0) ;
}
