/***************************************************************************
    file	         : syn.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>
#include	<stdarg.h>
#include	<string.h>
#include	<setjmp.h>
#include	<ctype.h>
#include	<std.h>

#include	"eli.h"
#include	"interp.h"
#include	"syn.h"

#define	ALLOC(T)	(T *)_el_alloc(sizeof(T))

#if	0
#define	ERRNULL()	if(_el_error)return NULL;
#define	ERRVOID()	if(_el_error)return;
#endif

#define	ERRNULL()
#define	ERRVOID()

typedef	struct	_block	BLOCK	;

struct	_block
{	BLOCK	*next	;
}	;

LVAR	STMT	*cblk	;	/* Current block			*/
LVAR	STMT	*citer	;	/* Current iterator			*/
LVAR	STMT	*cswit	;	/* Current switch			*/
LVAR	BLOCK	*blocks	;

/*G _el_alloc	: Allocate space					*/
/*  size	: int		: Space required in bytes		*/
/*  (returns)	: void *	: Pointer at allocated space		*/

GFUNC	void	*_el_alloc
	(	int	size
	)
{
	BLOCK	*b	= (BLOCK *) calloc (1, sizeof(BLOCK) + size) ;
	if (b == NULL) errorE ("el: out of memory\n") ;

	b->next	= blocks ;
	blocks	= b	 ;

	return	(void *)((char *)b + sizeof(b->next)) ;
}

GFUNC	void	el_syn_clean ()
{
	while (blocks != NULL)
	{	BLOCK	*next = blocks->next ;
		free	((void *)blocks) ;
		blocks	= next  ;
	}

	cblk	= 0	;
	citer	= 0	;
	cswit	= 0	;
}


/*L lookup	: Look for name on list					*/
/*  name	: char *	: Name to be found			*/
/*  list	: NAME *	: List to be searched			*/
/*  (returns)	: NAME *	: Name or NULL if not found		*/

LFUNC	NAME	*lookup
	(	const char	*name,
		NAME		*list
	)
{
	for ( ; list != NULL ; list = list->next)
		if (strcmp (name, list->name) == 0)
			return list ;
	return	NULL ;
}

/*G _el_name	: Register a new name					*/
/*  type	: int		: Type					*/
/*  value	: int		: Value					*/
/*  name	: char *	: Name string				*/
/*  (returns)	: NAME *	: Name definition entry			*/

GFUNC	NAME	*_el_name
	(	int		type,
		int		value,
		const char	*name
	)
{
	NAME	*nlp	;

	ERRNULL()	;

	if ((nlp = lookup (name, _el_nlist)) != NULL)
	{	
		/* This name already exists. This is an error if both	*/
		/* are fuction definitions, or if either is a constant.	*/
		if ( ((type	 & N_TYPE) == N_FUNC)  &&
		     ((nlp->type & N_TYPE) == N_FUNC) )
		{
			el_yyerror ("duplicate function : %s", name) ;
			return	NULL	;
		}
		if ( ((type      & N_TYPE) == N_CONST) ||
		     ((nlp->type & N_TYPE) == N_CONST) )
		{
			el_yyerror ("conflicting constant : %s", name) ;
			return	NULL	;
		}		   

		/* If the new type is a function then reset the type	*/
		/* as it may have been a variable.			*/
		if ((type & N_TYPE) == N_FUNC) nlp->type = type ;

		return	nlp ;
	}

	nlp		= ALLOC(NAME)	;
	nlp->type	= type		;
	nlp->value	= value		;
	nlp->name	= name		;
	nlp->next	= _el_nlist	;
	_el_nlist	= nlp		;

	return	nlp	;
}

/*G _el_newblk	: Construct new block statement node			*/
/*  (returns)	: void		:					*/

GFUNC	void	_el_newblk ()
{
	STMT	*stmt	;

	ERRVOID	()	;

	stmt		= ALLOC(STMT)	;
	stmt->lno	= el_lineno	;
	stmt->parent	= cblk		;
	stmt->tag	= S_BLOCK	;
	stmt->val.block.vars	= cblk == NULL ? NULL : cblk->val.block.vars ;
	cblk		= stmt		;
}

/*G _el_popblk	: Pop back a block					*/
/*  code	: STMT *	: Code in block				*/
/*  (returns)	: STMT *	: Current block when called		*/

GFUNC	STMT	*_el_popblk
	(	STMT	*code
	)
{
	STMT	*blk	;

	ERRNULL	()	;

	blk			= cblk ;
	blk->val.block.stmt	= code ;
	cblk			= blk->parent ;
	return	blk ;
}

/*G _el_funcd	: Define a function					*/
/*  type	: int		: Function type				*/
/*  name	: NAME *	: Function name				*/
/*  body	: STMT *	: Function body				*/
/*  (returns)	: void		:					*/

GFUNC	void	_el_funcd
	(	int	,
		NAME	*name,
		STMT	*body
	)
{
	FUNC	*func	;

	ERRVOID	()	;

	func		= ALLOC(FUNC)		;
	func->name	= name			;
	func->body	= body			;
	func->next	= _el_flist		;
	func->pars	= cblk->val.block.vars	;
	func->size	= _el_maxblk + 1      	;
	_el_flist	= func			;
	name->ref	= 1			;

	_el_popblk (NULL) ;
}

/*G _el_newvdef	: Define new parameter or block variable		*/
/*  value	: int		: Variable value			*/
/*  name	: char *	: Variable name				*/
/*  init	: ENODE *	: Initialiser				*/
/*  (returns)	: void		:					*/

GFUNC	void	_el_newvdef
	(	int		value,
		const char	*name,
		ENODE		*init
	)
{
	NAME	*var	;

	ERRVOID	()	;

	var		= ALLOC(NAME)	;
	var ->name	= name		;
	var ->val	= init		;
	var ->blk	= cblk		;
	var ->type	= N_BLOCK|N_VAR	;
	var ->value	= value		;
	var ->next	= cblk->val.block.vars ;
	var ->idx	= var->next == NULL ? 0 : var->next->idx + 1 ;
	cblk->val.block.vars = var	;

	/* Maintain the maximum block variable index so that we can	*/
	/* determin how much stack to allocate.				*/
	if (var->idx > _el_maxblk) _el_maxblk = var->idx ;
}

/*G _el_newcall	: Construct a new expression node for a function call	*/
/*  func	: ENODE *	: Function				*/
/*  alist	: ENODE *	: Argument list				*/
/*  (returns)	: ENODE *	: New node				*/

GFUNC	ENODE	*_el_newcall
	(	ENODE	*func,
		ENODE	*alist
	)
{
	ENODE	*node	;

	ERRNULL	()	;

	node		= ALLOC(ENODE)	;
	node->lno	= el_lineno	;
	node->tag	= E_CALL	;
	node->val.call.func	= func	;
	node->val.call.alist	= alist	;
	return	 node	;
}

/*G _el_newmethod: Construct a new expression node for a method app	*/
/*  object	 : ENODE *	: Object				*/
/*  method	 : const char *	: Method name				*/
/*  alist	 : ENODE *	: Argument list				*/
/*  (returns)	 : ENODE *	: New node				*/

GFUNC	ENODE	*_el_newmethod
	(	ENODE		*object,
		const char	*method,
		ENODE		*alist
	)
{
	ENODE	*node	;
	NAME	*nlp	;

	ERRNULL	()	;

	/* Look to see if there is already an instance of the string.	*/
	/* If not then add it to the string list.			*/
	if ((nlp = lookup (method, _el_slist)) == NULL)
	{
		nlp		= ALLOC(NAME)	  ;
		nlp->type	= N_STRING	  ;
		nlp->name	= strdup (method) ;
		nlp->next	= _el_slist	  ;
		nlp->ref	= 1		  ;
		_el_slist	= nlp		  ;
	}


	node		= ALLOC(ENODE)	 ;
	node->lno	= el_lineno	 ;
	node->tag	= E_METHOD	 ;
	node->val.method.object	= object ;
	node->val.method.method	= nlp	 ;
	node->val.method.alist	= alist  ;
	return	 node	;
}

/*G _el_newconst: Define a new constant					*/
/*  name	: char *	: Constant name				*/
/*  value	: ENODE *	: Constant value			*/
/*  (returns)	: void		:					*/

GFUNC	void	_el_newconst
	(	const char	*name,
		ENODE		*value
	)
{
	NAME	*nlp	;
	int	valt	= V_UNDEF ;

	ERRVOID	()	;

	switch (value->tag)
	{	case E_NUMB   : valt = V_NUM ; break ;
		case E_STRING : valt = V_STR ; break ;
		default	      : errorE ("elc: constant type error\n") ;
	}

	if ((nlp = _el_name (N_CONST, valt, name)) != NULL)
		nlp->val = value ;
}

/*G _el_newnumb	: Construct an expression node containing a number	*/
/*  numb	: int		: Number				*/
/*  (returns)	: ENODE *	: New node				*/

GFUNC	ENODE	*_el_newnumb
	(	int	numb
	)
{
	ENODE	*node	;

	ERRNULL	()	;

	node		= ALLOC(ENODE)	;
	node->lno	= el_lineno	;
	node->tag	= E_NUMB	;
	node->val.numb	= numb		;
	return	 node	;
}

/*G _el_newdbl	: Construct an expression node containing a double	*/
/*  dbl		: double	: Number				*/
/*  (returns)	: ENODE *	: New node				*/

GFUNC	ENODE	*_el_newdbl
	(	double	dbl
	)
{
	ENODE	*node	;

	ERRNULL	()	;

	node		= ALLOC(ENODE)	;
	node->lno	= el_lineno	;
	node->tag	= E_DBL		;
	node->val.dbl	= dbl		;
	return	 node	;
}

/*G _el_newstr	: Construct expression node for string			*/
/*  str		: char *	: String				*/
/*  (returns)	: ENODE *	: New node				*/

GFUNC	ENODE	*_el_newstr
	(	const char	*str
	)
{
	char		tmp[256]	;
	char		*tptr	= tmp	;
	const char	*sptr	= str	;
	NAME		*nlp		;
	ENODE		*node	;

	ERRNULL	()	;

	node	= ALLOC(ENODE) ;

	/* First process the string to handle any escaped characters.	*/
	while (*sptr)
		if (*sptr == '\\')
			sptr    = _el_escape (++sptr, tptr++, 0) ;
		else	*tptr++	= *sptr++ ;

	*tptr	= 0 ;

	/* Look to see if there is already an instance of the string.	*/
	/* If not then add it to the string list.			*/
	if ((nlp = lookup (tmp, _el_slist)) == NULL)
	{
		nlp		= ALLOC(NAME)	;
		nlp->type	= N_STRING	;
		nlp->name	= strdup (tmp)	;
		nlp->next	= _el_slist	;
		nlp->ref	= 1		;
		_el_slist	= nlp		;
	}

	node->tag	= E_STRING  ;	/* Mark node as string		*/
	node->lno	= el_lineno ;
	node->val.str	= nlp	    ;	/* Save pointer at list entry	*/

	return	node ;
}

/*G _el_newchar	: Construct expression node for character		*/
/*  str		: char *	: String				*/
/*  (returns)	: ENODE *	: New node				*/

GFUNC	ENODE	*_el_newchar
	(	const char	*str
	)
{
	char		tmp[256]	;
	char		*tptr	= tmp	;
	const char	*sptr	= str	;

	ERRNULL	()	;

	/* First process the string to handle any escaped characters.	*/
	while (*sptr)
		if (*sptr == '\\')
			sptr    = _el_escape (++sptr, tptr++, 0) ;
		else	*tptr++	= *sptr++ ;

	*tptr	= 0 ;

	if (strlen (tmp) != 1) el_yyerror ("bad character constant") ;

	return	_el_newnumb (tmp[0]) ;
}

/*G _el_newexpr	: Construct a new expression node for an operator	*/
/*  op		: int		: Operator				*/
/*  left	: ENODE *	: First operand				*/
/*  right	: ENODE *	: Second operand			*/
/*  (returns)	: ENODE *	: New node				*/

GFUNC	ENODE	*_el_newexpr
	(	int	op,
		ENODE	*left,
		ENODE	*right
	)
{
	ENODE	*node	;

	ERRNULL	()	;

	node		= ALLOC(ENODE)	;
	node->tag	= E_EXPR	;
	node->lno	= el_lineno	;
	node->val.expr.op	= op	;
	node->val.expr.left	= left	;
	node->val.expr.right	= right	;
	return	 node	;
}

/*G _el_newif	: Construct new if-then statement node			*/
/*  cond	: ENODE *	: Conditional expression		*/
/*  thenp	: STMT *	: Condition true part			*/
/*  elsep	: STMT *	: Condition false part			*/
/*  (returns)	: STMT		: New node				*/

GFUNC	STMT	*_el_newif
	(	ENODE	*cond,
		STMT	*thenp,
		STMT	*elsep
	)
{
	STMT	*node	;

	ERRNULL	()	;

	node		= ALLOC(STMT)	;
	node->parent	= cblk		;
	node->tag	= S_IFTHEN	;
	node->lno	= el_lineno	;
	node->val.ifthen.cond	= cond	;
	node->val.ifthen.thenp	= thenp	;
	node->val.ifthen.elsep	= elsep	;
	return	 node	;
}

/*G _el_newiter	: Construct new iteration statement node		*/
/*  init	: ENODE *	: Initialisation expression		*/
/*  cond	: ENODE *	: Conditional expression		*/
/*  iter	: ENODE *	: Iteration expression			*/
/*  (returns)	: void		:					*/

GFUNC	void	_el_newiter
	(	ENODE	*init,
		ENODE	*cond,
		ENODE	*iter
	)
{
	STMT	*node	;

	ERRVOID	()	;

	node		= ALLOC(STMT)	;
	node->parent	= cblk		;
	node->tag	= S_ITER	;
	node->lno	= el_lineno	;
	node->val.iter.init	= init	;
	node->val.iter.cond	= cond	;
	node->val.iter.iter	= iter	;
	node->val.iter.outer	= citer	;
	citer			= node	;
}

/*G _el_enditer	: End iterator construction				*/
/*  body	: STMT *	: Code body of iterator			*/
/*  (returns)	: STMT *	: Constructed iterator			*/

GFUNC	STMT	*_el_enditer
	(	STMT	*body
	)
{
	STMT	*iter	;

	ERRNULL	()	;
	iter	= citer			;
	citer->val.iter.body	= body	;
	citer	= citer->val.iter.outer ;
	return	iter ;
}

/*G _el_newbc	: Construct new break or continue statement node	*/
/*  tag		: int		: Statement type tag			*/
/*  (returns)	: STMT		: New node				*/

GFUNC	STMT	*_el_newbc
	(	int	tag
	)
{
	STMT	*node	;

	ERRNULL	()	;

	node	  = ALLOC(STMT)	;
	node->lno = el_lineno	;

	if (citer == NULL)
	{	el_yyerror ("break outside iterator") ;
		_el_error	= 1	  ;
		node->tag	= S_NULLS ;
	}
	else
	{	node->parent	= cblk	;
		node->tag	= tag	;
		node->val.iterp	= citer	;
	}

	return	 node	;
}

/*G _el_newretn	: Construct new return statement node			*/
/*  rete	: ENODE *	: Return value expression		*/
/*  (returns)	: STMT		: New node				*/

GFUNC	STMT	*_el_newretn
	(	ENODE	*rete
	)
{
	STMT	*node	;

	ERRNULL	()	;

	node		= ALLOC(STMT)	;
	node->lno	= el_lineno	;
	node->parent	= cblk		;
	node->tag	= S_RETURN	;
	node->val.rete	= rete		;
	return	 node	;
}

/*G _el_newstand: Construct new standalone expression node		*/
/*  expr	: ENODE *	: Expression				*/
/*  (returns)	: STMT		: New node				*/

GFUNC	STMT	*_el_newstand
	(	ENODE	*expr
	)
{
	STMT	*node	;

	ERRNULL	()	;

	node		= ALLOC(STMT)	;
	node->lno	= el_lineno	;
	node->parent	= cblk		;
	node->tag	= S_STAND	;
	node->val.expr	= expr		;
	return	 node	;
}

/*G _el_newnulls: Construct new null statement node			*/
/*  (returns)	: STMT		: New node				*/

GFUNC	STMT	*_el_newnulls ()
{
	STMT	*node	;

	ERRNULL	()	;

	node		= ALLOC(STMT)	;
	node->lno	= el_lineno	;
	node->parent	= cblk		;
	node->tag	= S_NULLS	;
	return	 node	;
}

/*G _el_newvar	: Locate variable for expression reference		*/
/*  name	: char *	: Variable name				*/
/*  (returns)	: ENODE *	: New expression node			*/

GFUNC	ENODE	*_el_newvar
	(	const char	*name
	)
{
	ENODE	*node	;
	NAME	*nptr	;

	ERRNULL	()	;

	/* Look for the variable, first amongst the current block	*/
	/* variables, then amongst the outer variables. If it cannot be	*/
	/* found then complain and return a number node to keep things	*/
	/* happy.							*/
	if ((nptr = lookup (name, cblk->val.block.vars)) == NULL)
		if ((nptr = lookup (name, _el_nlist)) == NULL)
		{	el_yyerror ("name not found: %s", name) ;
			nptr = _el_nlist ;
		}

	/* If the name found is a constant then just return the		*/
	/* expression node containing the value. Otherwise, construct	*/
	/* a new expression node for the value.				*/
	if ((nptr->type & N_TYPE) == N_CONST)
		return nptr->val ;

	node		= ALLOC(ENODE)	;
	node->lno	= el_lineno	;
	node->tag	= E_VAR	;
	node->val.var	= nptr	;
	nptr->ref	= 1	;

	return	node	;
}

/*G _el_newcase	: New case statement					*/
/*  expr	: ENODE *	: Case constant expression		*/
/*  stmt	: STMT *	: Case statement			*/
/*  (returns)	: CASE *	: Case construct			*/

GFUNC	CASE	*_el_newcase
	(	ENODE	*value,
		STMT	*stmt
	)
{
	CASE	*node	;

	ERRNULL	()	;

	node		= ALLOC(CASE) ;
	node->value	= value	;
	node->stmt	= stmt	;
	return	node	;
}

/*G _el_newswitch: New switch statement					*/
/*  value	 : ENODE *	: Switch expression			*/
/*  (returns)	 : void		:					*/

GFUNC	void	_el_newswitch
	(	ENODE	*value
	)
{
	STMT	*node	;

	ERRVOID	()	;

	node			= ALLOC(STMT)	;
	node->tag		= S_SWITCH	;
	node->lno		= el_lineno	;
	node->parent		= cblk	;
	node->val.swcase.value	= value	;
	node->val.swcase.outer	= cswit	;

	cswit	= node	;
}

/*G el_endswitch: End switch statement					*/
/*  cases	: CASE *	: List of cases				*/
/*  (returns)	: STMT *	: Switch statement			*/

GFUNC	STMT	*_el_endswitch
	(	CASE	*cases
	)
{
	int	deflt	= 0 ;
	STMT	*swit	= cswit	;

	ERRNULL	()	;

	cswit->val.swcase.cases	= cases	;
	cswit			= cswit->val.swcase.outer ;

	for ( ; cases != NULL ; cases = cases->next)
		if (cases->value == NULL)
			if (deflt)
				el_yyerror ("multiple defaults in switch") ;
			else	deflt	= 1 ;

	return	swit	;
}

/*G _el_newendc	: Endcase statement					*/
/*  (returns)	: STMT *	: Statement				*/

GFUNC	STMT	*_el_newendc ()
{
	STMT	*node	;

	ERRNULL	()	;

	if (cswit == NULL)
	{	el_yyerror ("endcase outside switch") ;
		return	NULL ;
	}

	node			= ALLOC(STMT)	;
	node->lno		= el_lineno	;
	node->tag		= S_ENDC	;
	node->val.endc.swit	= cswit		;

	return	node	;
}
