/***************************************************************************
    file	         : interp.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>
#include	<stdarg.h>
#include	<string.h>
#include	<ctype.h>
#include	<fcntl.h>
#include	<setjmp.h>
#include	<std.h>
#include	<except.h>



#ifdef		_WIN32
#include	<io.h>
#else
#include	<unistd.h>
#include	<regex.h>
#endif

#include	"eli.h"
#include	"interp.h"
#include	"syn.h"
#include	"code.h"

#define		DEBUG	0



 
LVAR	VALUE	*stack	      ;	/* Pointer to base of stack		*/
LVAR	VALUE	*block	      ;	/* Function block pointer		*/
LVAR	VALUE	*sttop	      ;	/* Top of stack				*/
LVAR	int	fnidx	      ;	/* Probable call index			*/

LVAR	long	icnt[32]      ;	/* Instruction execution count		*/
LVAR	long	ocnt[O_MAX+1] ;	/* Operation execution count		*/

GFUNC	const char *opToStr
	(	int	op
	)
{
	switch (op)
	{
		case O_PLUS	: return "+"  ;
		case O_MINUS	: return "-"  ;
		case O_MULT	: return "*"  ;
		case O_DIV	: return "/"  ;
		case O_REM	: return "%"  ;
		case O_AND	: return "&"  ;
		case O_OR	: return "|"  ;
		case O_XOR	: return "^"  ;
		case O_APLUS	: return "+=" ;
		case O_AMINUS	: return "-=" ;
		case O_AMULT	: return "*=" ;
		case O_ADIV	: return "/=" ;
		case O_AREM	: return "%=" ;
		case O_AAND	: return "&=" ;
		case O_AOR	: return "|=" ;
		case O_AXOR	: return "^=" ;
		case O_NOT	: return "!"  ;
		case O_NEQ	: return "!=" ;
		case O_ANDIF	: return "&&" ;
		case O_ORIF	: return "||" ;
		case O_COMMA	: return ","  ;
		case O_ASSIGN	: return "="  ;
		case O_LT	: return "<"  ;
		case O_LTEQ	: return "<=" ;
		case O_GT	: return ">"  ;
		case O_GTEQ	: return ">=" ;
		case O_EQ	: return "==" ;
		case O_SHL	: return "<<" ;
		case O_SHR	: return ">>" ;
		case O_COMP	: return "~"  ;
		case O_SUBS	: return "[]" ; 
		case O_SUBL	: return "[]&";
		case O_PREI	: 
		case O_POSTI	: return "++" ;
		case O_PRED	: 
		case O_POSTD	: return "--" ;
		case O_QUERY	: return "?"  ;

		case O_VEC	: return "vec"	;
		case O_HASH	: return "hash"	;
		case O_MATCH	: return "=~"	;

		default		: break ;
	}

	static	char	_b[80]	;
	sprintf	(_b, "unknown operator %d", op) ;
	return	_b ;
}

GFUNC	void	el_popstk
	(	int		cnt,
		const char	*
	)
{
	while (cnt > 0)
	{	POP	 ;
		cnt -= 1 ;
	}
}

/*G el_fexec	: Execute function					*/
/*  idx		: int		: Master table index			*/
/*  (returns)	: VALUE		: Result				*/

GFUNC	VALUE	el_fexec
	(	int	idx
	)
{
	int	errc	;
	VALUE	resval	;
	VALUE	*stos	= TOS	;

	TRAPE(errc)
	{	/* On error pop the stack back. The result is is the	*/
		/* error code.						*/
		while (TOS > stos) POP ;
		return	VALUE (errc, &tagERR) ;
	}
	WITHIN
	{
		/* Check that the index is in the correct range fo	*/
		/* the master table and that it does indeed refer to a	*/
		/* public function.					*/
		if ( (idx <         0) ||
		     (idx >= _el_mcnt) || (_el_master[idx].value.tag != &tagPUB))
			el_error ("Invalid call to el_fexec") ;

		/* The caller should have set up any arguments and the	*/
		/* number thereof, so execute it directly, and returns	*/
		/* its result.						*/
		return	_el_execute (_el_master[idx].value)  ;
	}
	ENDTRAP
}

/*G el_vexec	: Execute named function with arguments			*/
/*  name	: char *	: Module name				*/
/*  func	: char *	: Function name				*/
/*  argc	: uint		: Argument count			*/
/*  argv	: VALUE *	: Argument vector			*/
/*  (returns)	: VALUE		: Function result			*/

GFUNC	VALUE	el_vexec
	(	const char	*name,
		const char	*func,
		unsigned int	argc,
		VALUE		*argv
	)
{
	unsigned int	idx	;
	int		elc	;

	TRAP
	{
		return	VALUE (0, &tagERR) ;
	}
	WITHIN
	{
		char	fn1[256] ;
		char	fn2[256] ;

		_el_at = 0 ;
		sprintf	(fn1, "%s::%s", name, func) ;
		sprintf	(fn2,   "::%s",	      func) ; 

#if	0
		fprintf	(stderr, "--->[%s][%s]\n", fn1, fn2) ;
#endif
		/* Locate the function to be executed, and complain if	*/
		/* it cannot be found.					*/
		if (((elc = el_ffunc (fn1)) < 0) && ((elc = el_ffunc (fn2)) < 0))
			el_error    ("EL function \"%s\" not found for modules \"%s\"",
						func, name) ;

		for (idx = 0 ; idx < argc ; idx += 1)
			PUSH (argv[idx]) ;

		PUSH ((int)argc) ;
	}
	ENDTRAP

	/* All the work is now done, so go ahead and execute the	*/
	/* function. Its result is returned as the result.		*/
	return	el_fexec (elc) ;
}

/*  numArgs	: Count the number of arguments expected		*/
/*  aspec	: ELTAG **	: Argument specification vector		*/
/*  (returns)	: int		: Number of arguments			*/

LFUNC	int	numArgs
	(	ELTAG	**aspec
	)
{
	int	nargs	= 0 ;

	while (*aspec != 0)
	{	aspec	+= 1 ;
		nargs	+= 1 ;
	}

	return	nargs	;
}

/*G el_chkargs	: Check arguments					*/
/*  aspec	: ELTAG **	: Argument specification		*/
/*  name	: char *	: Name of called routine		*/
/*  (returns)	: VALUE *	: Pointer at first argument		*/

LFUNC	VALUE	*el_chkargs
	(	ELTAG		**aspec,
		const char	*name
	)
{
	VALUE	*argv	;
	int	iidx	= 0 ;
	int	nargs	= numArgs (aspec) ;

	/* The topmost stack value should be a number which specifies	*/
	/* the number of arguments on the stack. Check that this is the	*/
	/* number required. If OK then set the argument pointer and pop	*/
	/* the number.							*/
	if ((_el_tos->tag != &tagNUM) || (_el_tos->val.num != nargs))
		el_error ("Argument count error in %s (%d but needs %d)",
						name, (int)_el_tos->val.num, nargs) ;

	argv	= &_el_tos[-_el_tos->val.num] ;
	POP	;

	/* Check the argument types. They must match unless anything is	*/
	/* allowed.							*/
	while (iidx < nargs)
	{
		ELTAG	*spec	= aspec[iidx] ;

		if ((spec->tag != V_ANY) && (spec != argv[iidx].tag))
			el_error ("Argument type error in %s (arg %d: %s but needs %s)",
				   name, iidx, argv[iidx].tag->ident, aspec[iidx]->ident) ;

		iidx += 1 ;
	}

	return	argv	;
}

/*G el_chkargs	: Check arguments					*/
/*  aspec	: ELTAG **	: Argument specification		*/
/*  name	: char *	: Name of called routine		*/
/*  types	: char *	: First argument types for methods	*/
/*  (returns)	: VALUE *	: Pointer at first argument		*/

LFUNC	VALUE	*el_chkargs
	(	ELTAG		**aspec,
		const char	*name,
		ELTAG		**types
	)
{
	VALUE	*argv	;
	int	iidx	= 0 ;
	int	nargs	= numArgs (aspec) ;
	ELTAG	**tp	;

	/* The topmost stack value should be a number which specifies	*/
	/* the number of arguments on the stack. Check that this is the	*/
	/* number required. If OK then set the argument pointer and pop	*/
	/* the number.							*/
	if ((_el_tos->tag != &tagNUM) || (_el_tos->val.num != nargs + 1))
		el_error ("Argument count error in %s (%d but needs %d)",
						name, (int)_el_tos->val.num - 1, nargs) ;

	argv	= &_el_tos[-_el_tos->val.num] ;
	POP	;

	/* Check that the object is appropriate to the method. This	*/
	/* should be OK; an error here indicates a mismatch between the	*/
	/* method set and the object, and is an inernal error.		*/
	for (tp = types ; *tp != 0 ; tp += 1)
		if (argv[0].tag == *tp)
			break	;
	
	if (tp == 0)
		el_error ("Object type error in %s (got %s)",
			   name, argv[0].tag->ident) ;

	/* Check the argument types. They must match unless anything is	*/
	/* allowed.							*/
	while (iidx < nargs)
	{
		ELTAG	*spec	= aspec[iidx] ;

		if ((spec->tag != V_ANY) && (spec != argv[iidx+1].tag))
			el_error ("Argument type error in %s (arg %d: %s but needs %c)",
				   name, iidx, argv[iidx+1].tag->ident, aspec[iidx]) ;

		iidx += 1 ;
	}

	return	argv	;
}

/*L do_chksub	: Check subscript operands				*/
/*  op1		: VALUE		: Putative vector			*/
/*  op2		: VALUE		: Putative index			*/
/*  name	: char *	: Name of current routine		*/
/*  (returns)	: void		:					*/

LFUNC	void	do_chksub
	(	VALUE		op1,
		VALUE		op2,
		const char	*name
	)
{
	int	op2n	;

	if (op1.tag == &tagHASH) return ;

	if (op2.tag != &tagNUM)
		el_error ("Indexing with a non-number (%s) in %s",
				op2.tag->ident, name) ;
	op2n	= op2.val.num ;

	if	(op1.tag == &tagVEC)
	{	/* For a vector, check that the subscript refers to a	*/
		/* valid offset.					*/
		if ((op2n < 0) || (op2n >= op1.val.vec->size))
			el_error ("Invalid vector subscript %d (0 ... %d) in %s",
					op2n,
					op1.val.vec->size - 1,
					name) ;
	}
	else if (op1.tag == &tagSTR)
	{	/* For a string, allow any of the characters in the	*/
		/* string and the trailing null.			*/
		if ((op2n < 0) || (op2n > (int)strlen (op1.val.str->text)))
			el_error ("Envalid string subscript %d (0 ... %d) in %s",
					op2n,
					strlen (op1.val.str->text),
					name) ;

	}
	else	el_error ("Indexing non-vector/string %c in %s",
					name, op1.tag) ;
}

/*L do_oper	: Handle operator					*/
/*  op		: int		: Operator				*/
/*  name	: char *	: Name of current routine		*/
/*  (returns)	: void		:					*/

LFUNC	void	do_oper
	(	int		op,
		const char	*name
	)
{
	VALUE	&op2	= TOS[ 0] ;
	VALUE	&op1	= TOS[-1] ;
	int	numnum	= 0	  ;
	int	dbldbl	= 0	  ;

	/* Increment the count for this operation. Check that the	*/
	/* opcode is valid so we don't crash the interpreter if it is	*/
	/* invalid.							*/
	if ((op >= 0) && (op < O_MAX+1)) ocnt[op] += 1 ;


#define	NUM1	(op1.val.num)
#define	NUM2	(op2.val.num)
#define	DBL1	(op1.val.dbl)
#define	DBL2	(op2.val.dbl)
#define	STR1	(op1.val.str->text)
#define	STR2	(op2.val.str->text)

	/* Unary operators first ....					*/
	switch (op)
	{
		case O_VEC  :
			/* Allocate a new vector of the specified size.	*/
			if (op2.tag != &tagNUM)
				el_error ("Allocating vector with size type %s in %s",
						op2.tag->ident, name) ;
			op2	= new VEC (op2.val.num) ;
			return   ;

		case O_HASH :
			/* Allocate a new hash table with the specified	*/
			/* default value.				*/
			op2	= new HASH (*_el_tos) ;
			return	;

		case O_NOT  :
			/* Logical not. The number zero is false, and	*/
			/* anything else is true. Note that "popstk"	*/
			/* leaves the result value on the stack as	*/
			/* a number.					*/
			op2	= (op2.tag == &tagNUM) ? NUM2 != 0   :
			          (op2.tag == &tagDBL) ? DBL2 != 0.0 : true ;
			return	;

		case O_COMP :
			/* Error if the stack top is not a number, if	*/
			/* it is then just complement it.		*/
			if (op2.tag != &tagNUM)
				el_error ("Using operator ~ on type %s in %s",
						op2.tag->ident, name) ;
			op2	= ~op2.val.num ;
			return	;

	}

	/* Subscripting R-value. There are three cases, vector, hash	*/
	/* and string The "do_chksub" routine ensures that the		*/
	/* arguments are sensible.					*/
	if (op == O_SUBS)
	{
		do_chksub (op1, op2, name) ;

		switch (op1.tag->tag)
		{
			case V_VEC  :
				op1	= op1.val.vec->vals[op2.val.num] ;
				break	;

			case V_HASH :
				op1	= *op1.val.hash->entry (op2, 0) ;
				break	;
				
			default	:
				op1	= op1.val.str->text[op2.val.num] ;
				break	;
		}

		POP	;
		return	;
	}

	/* Subscripting L-value. This only works for vectors and hash	*/
	/* tables. The result is a pointer which will be used by the	*/
	/* following assignment.					*/
	if (op == O_SUBL)
	{
		do_chksub (op1, op2, name) ;

		switch (op1.tag->tag)
		{
			case V_VEC  :
				op1	= &op1.val.vec->vals[op2.val.num] ;
				break	;

			case V_HASH :
				op1	= op1.val.hash->entry (op2, 1) ;
				break	;

			default	:
				el_error ("Subscript update of string in %s", name) ;
				break	;
		}

		POP	;
		return	;
	}

	/* If one operand is a double and the other is a number then	*/
	/* convert the number to a double ....				*/
	if ((op1.tag == &tagNUM) && (op2.tag == &tagDBL))
		op1	= (double)op1.val.num ;
	if ((op1.tag == &tagDBL) && (op2.tag == &tagNUM))
		op2	= (double)op2.val.num ;


	if ((op == O_NEQ) || (op == O_EQ))
	{
		int	r = op1 == op2  ;
		if (op == O_NEQ) r = !r ;
		op1	= r ;
		POP	;
		return	;
	}

	/* From now on both operands must be numbers, or both must be	*/
	/* strings, or both must be doubles. Anything else is an error.	*/
	if (op1.tag == op2.tag)
	{
		if	(op1.tag == &tagNUM) numnum = 1 ;
		else if	(op1.tag == &tagDBL) dbldbl = 1 ;
		else if (op1.tag != &tagSTR)
			el_error ("Operand type error in %s: %s %s %s",
				  name,
				  op1.tag->ident,
				  opToStr (op),
				  op2.tag->ident) ;
	}
	else
		el_error ("Operand type error in %s: %s %s %s",
			  name,
			  op1.tag->ident,
			  opToStr (op),
			  op2.tag->ident) ;

	switch (op)
	{
		case O_LT   :
		case O_LTEQ :
		case O_GT   :
		case O_GTEQ :
		case O_PLUS :
		case O_MATCH:
			break	;
				
		case O_MINUS :
		case O_MULT  :
		case O_DIV   :
		case O_REM   :
		case O_AND   :
		case O_OR    :
		case O_XOR   :
		case O_SHL   :
		case O_SHR   :
			/* For these operators, both arguments must be	*/
			/* numbers. Also, for divide and remainder, the	*/
			/* second operand must be non-zero.		*/
			if (!numnum && !dbldbl)
				el_error ("Operands are not numbers for %s %s %s in %s",
						op1.tag->ident,
						opToStr (op),
						op2.tag->ident,
						name) ;
			if ((op == O_DIV) || (op == O_REM))
				if (dbldbl ? (DBL2 == 0.0) : (NUM2 == 0))
					el_error ("Divide (or rem) by zero in %s", name) ;

			break	;

		default :
			el_error ("EL internal error: unexpected operator (%s) in %s",
					opToStr (op), name) ;
	}

	if (numnum)
	{	/* If both operands are numbers and the result is a	*/
		/* number then pop one value of the stack and just	*/
		/* update the other one.				*/
		int	n1	= op1.val.num ;
		int	n2	= op2.val.num ;
		int	r	= 0	      ;

		switch (op)
		{	case O_PLUS  : r = n1 +  n2 ; break ;
			case O_MINUS : r = n1 -  n2 ; break ;
			case O_MULT  : r = n1 *  n2 ; break ;
			case O_DIV   : r = n1 /  n2 ; break ;
			case O_REM   : r = n1 %  n2 ; break ;
			case O_AND   : r = n1 &  n2 ; break ;
			case O_OR    : r = n1 |  n2 ; break ;
			case O_XOR   : r = n1 ^  n2 ; break ;
			case O_SHL   : r = n1 << n2 ; break ;
			case O_SHR   : r = n1 >> n2 ; break ;
			case O_LT    : r = n1 <  n2 ; break ;
			case O_LTEQ  : r = n1 <= n2 ; break ;
			case O_GT    : r = n1 >  n2 ; break ;
			case O_GTEQ  : r = n1 >= n2 ; break ;
			default	     :
				el_error ("EL internal error: unexpected operation (%s %s %s) in %s",
					   	op1.tag->ident,
					   	opToStr  (op),
					   	op2.tag->ident,
						name) ;
		}

		op1	= r ;
		POP	;
		return	;
	}

	if (dbldbl)
	{	/* If both operands are doubles and the result is a	*/
		/* number then pop one value of the stack and just	*/
		/* update the other one: this will already be correctly	*/
		/* tagged.						*/
		ELTAG	*tag	= 0 ;
		double	d1	= op1.val.dbl ;
		double	d2	= op2.val.dbl ;
		int	rn	= 0 ;
		double	rd	= 0 ;

		switch (op)
		{	case O_PLUS  : tag = &tagDBL ; rd = d1 +  d2 ; break ;
			case O_MINUS : tag = &tagDBL ; rd = d1 -  d2 ; break ;
			case O_MULT  : tag = &tagDBL ; rd = d1 *  d2 ; break ;
			case O_DIV   : tag = &tagDBL ; rd = d1 /  d2 ; break ;
			case O_LT    : tag = &tagNUM ; rn = d1 <  d2 ; break ;
			case O_LTEQ  : tag = &tagNUM ; rn = d1 <= d2 ; break ;
			case O_GT    : tag = &tagNUM ; rn = d1 >  d2 ; break ;
			case O_GTEQ  : tag = &tagNUM ; rn = d1 >= d2 ; break ;
			default	     :
				el_error ("EL internal error: unexpected operation (%s %s %s) in %s",
					   	op1.tag->ident,
					   	opToStr  (op),
					   	op2.tag->ident,
					   	name) ;
		}
	
		if (tag == &tagNUM)
			op1	= rn ;
		else	op1	= rd ;

		POP	;
		return	;
	}

	/* Remaining cases are string operations. Plus is a special	*/
	/* case as this is string concatenation.			*/
	if (op == O_PLUS)
	{
		STRING	*str = new STRING(strlen(STR1) + strlen(STR2) + 1) ;

		strcpy (str->text, STR1)  ;
		strcat (str->text, STR2)  ;

		op1	= str	;
		POP	;
		return	;
	}


	/* Next check for pattern matching. This is _not_ efficient, as	*/
	/* the pattern is compiled every time, regardless of whether it	*/
	/* has changed. We need a regular-expression object to allow	*/
	/* efficiency where needed.					*/
	if (op == O_MATCH)
	{
#ifdef	_WIN32
		el_error ("No regular expression matching in Win32") ;
#else
		regex_t		regex		;
		regmatch_t	regmat	[16]	;
		int		rc	;

		if ((rc = regcomp (&regex, op2.val.str->text, REG_EXTENDED)) != 0)
		{
			char	errb[256] ;
			regerror (rc, &regex, errb, sizeof(errb)) ;
			el_error ("Regular expression \"%s\" error: %s\n",
					   op2.val.str->text,
					   errb) ;
			return	 ;
		}

		/* Do the match. If this fails then free the expression	*/
		/* structure and return zero.				*/
		if (regexec (&regex, op1.val.str->text,
				        16, &regmat[0], REG_NOTBOL|REG_NOTEOL) != 0)
		{
			op1	= 0 ;
			POP	;
			regfree	(&regex) ;
			return	;
		}

		/* Success! A match! Allocate a new vector and push	*/
		/* onto it the main match plus any submatches.		*/
		VEC	*vec	= new VEC (0) ;

		for (int idx = 0 ; idx < 16 ; idx += 1)
			if (regmat[idx].rm_so >= 0)
			{
				int	len	= regmat[idx].rm_eo - regmat[idx].rm_so ;
				STRING	*str	= new STRING (len + 1) ;
				memcpy	(str->text, &op1.val.str->text[regmat[idx].rm_so], len) ;
				vec->push (VALUE (str)) ;
			}
			else	break	;

		op1	= vec	 ;
		POP		 ;
		regfree	(&regex) ;
#endif
		return	;
	}

	int	cmp	= strcmp (STR1, STR2) ;
	switch (op)
	{	case O_LT   : cmp = cmp <  0 ; break ;
		case O_LTEQ : cmp = cmp <= 0 ; break ;
		case O_GT   : cmp = cmp >  0 ; break ;
		case O_GTEQ : cmp = cmp >= 0 ; break ;
		default	    : el_error ("EL internal error: unexpected operator (%s) in %s",
					 opToStr (op), name) ;
	}

	op1	= cmp	;
	POP	;
	return	;
}

/*L do_call	: Handle routine call					*/
/*  func	: VALUE		: Putative function			*/
/*  cname	: char *	: Name of calling routine		*/
/*  (returns)	: void		:					*/

LFUNC	void	do_call
	(	const VALUE	&func,
		const char	*cname
	)
{
	VALUE		retval	;
	VALUE		*argv	;
	VALUE		*oblk	= block	;
	const char	*fname	;

	/* Execute either a machine code or an EL function, and push	*/
	/* the result. Anything else is an error.			*/
	switch (func.tag->tag)
	{	case V_FN  :
			argv	= el_chkargs (func.val.mc->args, func.val.mc->name) ;
			retval	= (*func.val.mc->fn) (argv) ;
			el_popstk (numArgs(func.val.mc->args), func.val.mc->name) ;
			break	;

		case V_ELC :
		case V_PUB :
			retval	= _el_execute (func) ;
			break	;

		default	   :
			if ((fname =_el_master[fnidx].name) == NULL)
				fname = "unknown" ;
			el_error ("Calling non-function (%s?) from %s",
				  fname, cname) ;
	}

	PUSH	(retval) ;
	block	= oblk	 ;
}

/*L do_method	: Execute method invocation				*/
/*  argc	: int		: Argument count			*/
/*  (returns)	: void		:					*/

LFUNC	void	do_method
	(	int	argc
	)
{
	/* Note the method name, which is at the top of the stack, and	*/
	/* get a pointer at the object, which is the first argument.	*/
	/* From the latter pick up a pointer at the method table.	*/
	char	*meth	= _el_tos->val.str->text ;
	VALUE	*obj	= &_el_tos[- argc - 1]	 ;
	METHSET	*mset	;
	MC	*mc	= 0 ;
	VALUE	retval	;
	VALUE	*argv	;

	/* If the method set pointer is null then this is not a	*/
	/* method, otherwise scan thr table for the requested method.	*/
	if (obj->tag->mset == NULL)
		el_error ("Requesting method %s on non-object type %s",
				meth, obj->tag->ident) ;

	for (mset = obj->tag->mset ; mset != NULL ; mset = mset->next)
		for (mc = mset->mtab ; mc->name != NULL ; mc += 1)
			if (strcmp (mc->name, meth) == 0)
				goto found ;

	el_error ("No method %s on object type %s",
				meth, obj->tag->ident) ;

	found :

	/* Pop the method name off the stack, then push the argument	*/
	/* count, incremented by one to account for the object pointer.	*/
	POP	;
	PUSH	(argc + 1) ;

	/* The rest is just a function call ....			*/
	argv	= el_chkargs (mc->args, mc->name, mset->types) ;
	retval	= (*mc->fn) (argv) ;
	while (argc > 0)
	{	POP	  ;
		argc -= 1 ;
	}
	POP	;
	*++_el_tos = retval ;
}

LFUNC	double	_el_getdbl
	(	int	*code
	)
{
	double	dbl	;
	int	*dp	= (int *)&dbl	;
	int	idx	= sizeof(double)/sizeof(int) ;

	while (idx > 0)
	{	*dp	= *code	;
		code   += 1 ;
		dp     += 1 ;
		idx    -= 1 ;
	}

	return	dbl	;
}

/*G _el_execute	: Execute an EL function				*/
/*  func	: VALUE		: Function definition			*/
/*  (returns)	: VALUE		: Value of function			*/

GFUNC	VALUE	_el_execute
	(	const VALUE	&func
	)
{
	ELF		*elf	= func.val.elf	;
	int		*code	= elf->code	;
	const char	*name	= elf->name	;
	
	int	pc	= 0 ;


	for (;;)
	{
		/* Get the opcode and operand. If the latter is the	*/
		/* escape value then the actual value is in the next	*/
		/* instruction.						*/
		int	op	= (code[pc] & OPBITS) >> OPSHIFT ;
		int	arg	= (code[pc] & ARBITS) ;

		if (arg == ARBITS) arg = code[++pc] ;
		pc += 1 ;

		/* Increment the execution count for this instruction,	*/
		/* whether or not logging is on, as testing the latter	*/
		/* would take longer.					*/
		icnt[op] += 1 ;
#if	DEBUG
	{	int	idx	;

		printf	("%5s %6d: ",
			 op == C_OPER  ? "oper"  :
			 op == C_BRA   ? "bra "  :
			 op == C_BRAT  ? "brat"  :
			 op == C_BRAF  ? "braf"  :
			 op == C_LOUT  ? "lout"  :
			 op == C_SOUT  ? "sout"  :
			 op == C_SOUTP ? "soutp" :
			 op == C_LBLK  ? "lblk"  :
			 op == C_SBLK  ? "sblk"  :
			 op == C_SBLKP ? "sblkp" :
			 op == C_LNUM  ? "lnum"  :
			 op == C_LSTR  ? "lstr"  :
			 op == C_POP   ? "pop "  :
			 op == C_CALL  ? "call"  :
			 op == C_CALLV ? "callv" :
			 op == C_METH  ? "meth"  :
			 op == C_RET   ? "ret "  :
			 op == C_RETN  ? "retn"  :
			 op == C_INCR  ? "incr"  :
			 op == C_DECR  ? "decr"  :
			 op == C_SBLKS ? "sblks" :
			 op == C_SBLKN ? "sblkn" :
			 op == C_LBLKS ? "lblks" :
			 op == C_LBLKN ? "lblkn" :
			 op == C_ARGC  ? "argc"  :
			 op == C_BLKS  ? "blks"  :
			 op == C_SIND  ? "sind"  :
			 op == C_AT    ? "at"	 : "????", arg)  ;

		if (op == C_OPER)
			printf	(" %3s ", opToStr(arg)) ;
		else	printf	("     ") ;

		for (idx = 0 ; (idx > -8) && (&_el_tos[idx] >= stack) ; idx -= 1)
			switch (_el_tos[idx].tag->tag)
			{	case V_NUM :
					printf	(" <%ld>", _el_tos[idx].val.num) ;
					break	;
				case V_STR :
					printf	(" <%s>",  _el_tos[idx].val.str->text) ;
					break	;
				default	   :
					printf	(" [%s]",  _el_tos[idx].tag->ident) ;
					break	;
			}
		printf	("\n") ;
		fflush	(stdout) ;
	}
#endif	/* DEBUG */
		switch (op)
		{
			case C_AT   :
				/* Execution position. Just note it in	*/
				/* case it is needed.			*/
				_el_at	= arg	;
				break	;

			case C_OPER :
				/* Unary and binary operators. These	*/
				/* are handled by a separate routine.	*/
				do_oper	(arg, name) ;
				break ;

			case C_INCR :
				/* To increment, push the number and	*/
				/* then apply the plus operator.	*/
				PUSH	(arg) ;
				do_oper	(O_PLUS, name) ;
				break ;

			case C_DECR :
				/* Similarly for decrement ...		*/
				PUSH	(arg) ;
				do_oper	(O_MINUS, name) ;
				break ;

			case C_BRA  :
				/* Unconditional branch, nice and easy.	*/
				pc	= arg ;
				break	;

			case C_BRAT :
			case C_BRAF :
			{	/* True is any non-number or non-zero	*/
				/* number.				*/
				int t = (_el_tos->tag     != &tagNUM) ||
					(_el_tos->val.num != 0) ;

				if ((op == C_BRAT) == t) pc = arg ;
				POP	;
				break	;
			}

			case C_LOUT :
				/* Push an outer value; note the index	*/
				/* in case the next instruction is a	*/
				/* call.				*/
				PUSH	(_el_master[arg].value) ;
				fnidx	= arg ;
				break	;

			case C_SOUT :
				/* Similarly, store a value. Note that	*/
				/* the stack is _not_ popped (so that	*/
				/* a = (b = c) is easily implemented).	*/
				_el_master[arg].value = *_el_tos ;
				break	;

			case C_SOUTP:
				/* Store to a block variable. Here the	*/
				/* stack is popped.			*/
				_el_master[arg].value = *_el_tos ;
				POP	;
				break	;

			case C_LBLK :
				/* Push the value of a block variable	*/
				/* onto the stack.			*/
				PUSH	(block[arg])  ;
				break	;

			case C_SBLK :
				/* Store to a block variable. As for	*/
				/* S_OUT, the stack is _not_ popped.	*/
				block[arg] = *_el_tos ;
				break	;

			case C_SBLKP:
				/* Store to a block variable. Here the	*/
				/* stack is popped.			*/
				block[arg] = *_el_tos ;
				POP	;
				break	;

			case C_LNUM :
				/* Load a number onto the stack.	*/
				PUSH	(arg) ;
				break	;

			case C_LSTR :
				/* Similarly load a string.		*/
				PUSH	(new STRING(_el_master[arg].name)) ;
				break	;

			case C_LDBL :
				PUSH	(_el_getdbl (&code[pc])) ;
				pc += sizeof(double)/sizeof(int) ;
				break	;

			case C_POP  :
				/* Pop the specified number of items	*/
				/* off the stack. The routines handles	*/
				/* freeing of space allocated to	*/
				/* strings.				*/
				while (arg > 0)
				{	POP	 ;
					arg -= 1 ;
				}
				break	;

			case C_CALL :
			case C_CALLV:
			{	/* The top of the stack is:		*/
				/* _el_tos	-> function		*/
				/*		   argN			*/
				/*		   ....			*/
				/*		   arg0			*/
				VALUE	func  = *_el_tos  ;
				*_el_tos = arg	     ;
				do_call (func, name) ;
				if (op == C_CALLV) POP ;
				break	;
			}

			case C_METH :
				/* Method call ....			*/
				do_method  (arg) ;
				break	;

			case C_RET  :
			{	/* The topmost item on the stack is	*/
				/* saved for return, then pop back the	*/
				/* block variables before returning.	*/
				VALUE	retval	 (*_el_tos) ;
				el_popstk  (_el_tos - block + 1, name) ;
				return	retval	 ;
			}

			case C_RETN :
			{	/* The result is a number embedded in	*/
				/* the instruction.			*/
				VALUE	retval	(arg) ;
				el_popstk  (_el_tos - block + 1, name) ;
				return	retval	;
			}

			case C_ARGC :
				/* Check that the correct number of	*/
				/* arguments are present. If OK then	*/
				/* set the block pointer and pop the	*/
				/* count.				*/
				if ((_el_tos->tag != &tagNUM) || (_el_tos->val.num != arg))
					el_error ("Argument count error in %s (%d but needs %d)",
						name, _el_tos->val.num, arg) ;
				block	= &_el_tos[-_el_tos->val.num] ;
				POP	;
				break	;

			case C_BLKS :
				/* Set the block size. This puts the	*/
				/* stack top above any block variables.	*/
				if (&block[arg - 1] >= _el_tos)
					_el_tos	= &block[arg - 1] ;
				else	el_popstk (_el_tos - &block[arg - 1], name) ;
				break	;

			case C_SIND :
				/* Store indirect to value and pop	*/
				/* pointer.				*/
				*_el_tos->val.ptr = TOS[-1] ;
				POP	;
				break	;

			case C_LIND :
				/* Load indirect replacing pointer	*/
				*_el_tos  = *_el_tos->val.ptr ;
				break	;

			case C_DUP  :
				/* Duplicate top of stack value.	*/
				_el_tos  += 1 ;
				_el_tos[0] = _el_tos[-1] ;
				break	;

			case C_SWAP :
				/* Swap top two stack values		*/
				{	VALUE	tmp	;
					tmp		= _el_tos[ 0] ;
					_el_tos[ 0]	= _el_tos[-1] ;
					_el_tos[-1]	= tmp ;
				}
				break	;

			default	    :
				/* Anything else is an unrecognised	*/
				/* opcode.				*/
				el_error ("Unexpected opcode %d in %s", op, name) ;
				break	;
		}

		/* Check for stack overflow. There is actually a small	*/
		/* safety margin, so it can be done just once here.	*/
		if (_el_tos >= sttop) el_error ("EL stack overflow in %s", name) ;
	}
}

/*G el_init	: Initialise interpreter				*/
/*  csize	: uint		: Size for allocated code space		*/
/*  ssize	: uint		: Size for stack			*/
/*  (returns)	: void		:					*/

GFUNC	int	el_init
	(	unsigned int	,
		unsigned int	ssize
	)
{
	int		lfd	;
	char		*ellog	= getenv ("ELLOGFILE") ;

	/* Initialise the root error environment.			*/
	EEINT	;

	/* Allocate space for the interpreter stack and ensure that	*/
	/* it is forthcoming. It is then cleared to all zeros to make	*/
	/* sure that nothing untoward will get accidentally freed.	*/
	stack	= new VALUE[ssize] ;
	_el_tos	= stack	;
	sttop	= &stack[ssize - 1] ;

	_el_master  = (ITEM *)calloc(MSIZE, sizeof(ITEM)) ;

	/* Insert a global reference entry for the "start" routine.	*/
	/* Since this is the first ever call to "insname", "start" will	*/
	/* always appear in entry zero.					*/
	_el_insname ("start", 1) ;

	/* If the log file environment variable is set and the file can	*/
	/* be opened, then load the current counts.			*/
	if ((ellog != NULL) && ((lfd = open (ellog, O_RDONLY)) >= 0))
	{	read	(lfd, icnt, sizeof(icnt)) ;
		read	(lfd, ocnt, sizeof(ocnt)) ;
		close	(lfd) ;
	}

	return	1 ;
}

/*G el_term	: Clean up for termination				*/
/*  (returns)	: void		:					*/

GFUNC	void	el_term ()
{
	int	lfd	;
	char	*ellog	= getenv ("ELLOGFILE") ;

	if ((ellog != NULL) && ((lfd = open (ellog, O_WRONLY|O_CREAT, 0666)) >= 0))
	{	write	(lfd, icnt, sizeof(icnt)) ;
		write	(lfd, ocnt, sizeof(ocnt)) ;
		close	(lfd) ;
	}
}

