/***************************************************************************
    file	         : input.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>
#include	<stdarg.h>
#include	<std.h>
#include	<string.h>
#include	<fcntl.h>
#include	<ctype.h>

#ifdef		_WIN32
#include	<io.h>
#else
#include	<unistd.h>
#endif

#define	LSIZE	132
#define	FSIZE	20
#define	SSIZE	32
#define	DSIZE	64
#define	ISIZE	16

#define	SEXEC	0
#define	SSKIP	1
#define	SSCAN	2

typedef	struct
{	const char	*name		;
	int		lno		;
	FILE		*fid		;
	const char	*buff		;
	const char	*bp		;
}	SOURCE	;

LVAR	int	lpend		;
LVAR	char	line  [LSIZE]	;
LVAR	SOURCE	source[FSIZE]	;
LVAR	int	fdp		;
LVAR	int	flineno		;

LVAR	char	buff[1024]	;
LVAR	int	bptr		;
LVAR	int	blen		;


typedef	struct
{	int	state	;
	char	*sym	;
}	SKIP	;

LVAR	SKIP	skstk [SSIZE]	;
LVAR	int	sktop		;
LVAR	char	*dlist[DSIZE]	;
LVAR	char	*ilist[ISIZE+1]	;

GVAR	FILE	*_el_errout	;

/*G el_addhdr	: Add header directory					*/
/*  name	: const char *	: Directory name			*/
/*  (returns)	: void		:					*/

GFUNC	void	el_addhdr
	(	const char	*name
	)
{
	int	ip	;

	for (ip = 0 ; ip < ISIZE ; ip += 1)
		if (ilist[ip] == NULL)
		{	if ((ilist[ip] = strdup (name)) != NULL) return	;
			errorE	("elc: out of memory\n") ;
		}

	errorE	("elc: too many header directories\n") ;
}

/*L openhdr	: Open a header file					*/
/*  name	: char *	: Header name				*/
/*  (returns)	: FILE *	: File or NULL on error			*/

LFUNC	FILE	*openhdr
	(	const char	*name
	)
{
	char	path[256] ;
	FILE	*fd	  ;
	int	ip	  ;

	if ((fd = fopen (name, "r")) != NULL) return fd ;

	for (ip = 0 ; ilist[ip] != NULL ; ip += 1)
	{
		sprintf	(path, "%s/%s", ilist[ip], name) ;
		if ((fd = fopen (path, "r")) != NULL) return fd ;
	}

	return	NULL ;
}

/*L el_defined	: Text if a symbol is defined				*/
/*  sym		: const char *	: Symbol				*/
/*  (returns)	: int		: Non-zero if defined			*/

LFUNC	int	el_defined
	(	const char	*sym
	)
{
	int	idx	;

	for (idx = 0 ; idx < DSIZE ; idx += 1)
		if ((dlist[idx] != NULL) && (strcmp (dlist[idx], sym) == 0))
			return	1 ;

	return	0	;
}

/*G el_define	: Define a symbol					*/
/*  sym		: const char *	: Symbol				*/
/*  (returns)	: void		:					*/

GFUNC	void	el_define
	(	const char	*sym
	)
{
	int	idx	;

	if (el_defined (sym)) return ;

	for (idx = 0 ; idx < DSIZE ; idx += 1)
		if (dlist[idx] == NULL)
		{	dlist[idx] = strdup (sym) ;
			return ;
		}

	errorE	("elc: too many #define'd symbols\n") ;
}

/*G el_undefine	: Undefine a symbol					*/
/*  sym		: const char *	: Symbol				*/
/*  (returns)	: void		:					*/

GFUNC	void	el_undefine
	(	const char	*sym
	)
{
	int	idx	;

	for (idx = 0 ; idx < DSIZE ; idx += 1)
		if ((dlist[idx] != NULL) && (strcmp (dlist[idx], sym) == 0))
		{	free ((void *)dlist[idx]) ;
			dlist[idx] = NULL ;
		}
}

/*L el_token	: Extract token from string				*/
/*  ptr		: char *	: String				*/
/*  others	: char *	: Acceptable characters in addition ...	*/
/*				  ... to alphanumerics			*/
/*  (returns)	: char *	: Token or "" if none			*/

LFUNC	const char *el_token
	(	char		*ptr,
		const char	*others
	)
{
	const char *res	;

	while (isspace (*ptr)) ptr += 1 ;
	res	= ptr	;
	while (isalnum (*ptr) || (strchr (others, *ptr) != NULL)) ptr += 1 ;
	*ptr	= 0	;

	return	res	;
}

/*L el_direc	: Handle a directive line				*/
/*  line	: char *	: Directive line			*/
/*  (returns)	: int		: 0 -> not directive			*/
/*				  1 -> include				*/
/*				  2 -> other directive			*/

LFUNC	int	el_direc
	(	char	*line
	)
{
	if (strncmp (line, "#include", 8) == 0)
	{
		const char *fn	= el_token (&line[8], ":\\/.$_") ;
		FILE	   *fd2	= openhdr  (fn) ;

		if (fd2 == NULL)
			errorE	("elc: unable to open \"%s\"\n", fn) ;

		source[fdp].lno	  = flineno + 1	;
		fdp	         += 1		;
		source[fdp].fid	  = fd2		;
		source[fdp].name  = strdup (fn) ;
		flineno	  = 1 ;

		return	1 ;
	}

	if (strncmp (line, "#ifdef", 6) == 0)
	{
		const char *tok	= el_token (&line[6], "_") ;
		int	cstat	= sktop == 0 ? SEXEC : skstk[sktop].state ;
		SKIP	*skip	= &skstk[++sktop] ;

		if (sktop >= SSIZE)
			errorE	("elc: #ifdef/#endif's nested too deply\n") ;

		skip->sym	= strdup (tok) ;
		skip->state	= cstat != SEXEC   ? SSKIP :
				  el_defined (tok) ? SEXEC : SSCAN ;
		return	2	;
	}

	if (strncmp (line, "#ifndef", 7) == 0)
	{
		const char *tok	= el_token (&line[7], "_") ;
		int	cstat	= sktop == 0 ? SEXEC : skstk[sktop].state ;
		SKIP	*skip	= &skstk[++sktop] ;

		if (sktop >= SSIZE)
			errorE	("elc: #ifdef/#endif's nested too deply\n") ;

		skip->sym	= strdup (tok) ;
		skip->state	= cstat != SEXEC   ? SSKIP :
				  el_defined (tok) ? SSCAN : SEXEC ;
		return	2	;
	}

	if (strncmp (line, "#else", 5) == 0)
	{
		SKIP	*skip	= &skstk[sktop] ;

		if (sktop <= 0)
			errorE	("elc: #else outside #ifdef/#ifndef ... #endif\n") ;

		skip->state	= skip->state == SEXEC ? SSKIP :
				  skip->state == SSCAN ? SEXEC : SSKIP ; 

		return	2 ;
	}

	if (strncmp (line, "#endif", 6) == 0)
	{
		if (sktop <= 0)
			errorE	("elc: #endif without matching #ifdef/#ifndef\n") ;

		free ((void *)skstk[sktop--].sym) ;

		return	2 ;
	}

	if (strncmp (line, "#define", 7) == 0)
	{
		el_define (el_token (&line[7], "_")) ;
		return	2 ;
	}

	if (strncmp (line, "#undef", 6) == 0)
	{
		el_undefine (el_token (&line[6], "_")) ;
		return	2 ;
	}

	/* Anything else is not a directive.				*/
	return	0 ;
}

/*  el_gets	: Get next input line					*/
/*  line	: char *	: Buffer				*/
/*  llen	: int		: Buffer length				*/
/*  fdp		: int		: File input stack pointer		*/
/*  (returns)	: char *	: Line or null at EOF			*/

LFUNC	char	*el_gets
	(	char	*line,
		int	llen,
		int	fdp
	)
{
	SOURCE	*sp	= &source[fdp] ;

	if (sp->fid != 0)
		return	fgets (line, llen, sp->fid) ;

	if (sp->bp[0] == 0)
		return	0 ;

	const char *ep	= strchr (sp->bp, '\n')   ;
	int	   cl	;

	if (ep == 0)
		ep  = &sp->bp[strlen(sp->bp)] ;
	else	ep += 1 ;

	cl	= ep - sp->bp ;
	if (cl > llen - 1) cl = llen - 1 ;

	memcpy	(line, sp->bp, cl) ;
	line[cl]  = 0  ;
	sp->bp   += cl ;

	return	line	;
}

/*G el_read	: Local read routine					*/
/*  buff	: char *	: Buffer				*/
/*  size	: size_t	: Buffer size				*/
/*  (returns)	: size_t	: Number of characters read		*/

GFUNC	size_t	el_read
	(	char	*buff,
		size_t	size
	)
{
	char	*rc	;
	int	llen	;
	int	got	= 0 ;

	/* The outer loop gets successive lines until either the input	*/
	/* is exhausted, or a line would overflow the buffer. This is	*/
	/* checked for inside the loop.					*/
	while (fdp >= 0)
	{
		/* If there is a pending line then that can be used. If	*/
		/* no line is pending get lines until one which is not	*/
		/* a directive is forthcoming. This can be used if we	*/
		/* are executing.					*/
		if (lpend)
		{	lpend	= 0	;
			rc	= line	;
		}
		else	if ((rc = el_gets (line, LSIZE, fdp)) != NULL)
			{
				flineno += 1 ;

				switch (el_direc (line))
				{
					case 0 :
						if (skstk[sktop].state != SEXEC)
							strcpy (line, "\n") ;
						break ;

					case 1 :
						sprintf (line, "#line %s 1\n", source[fdp].name) ;
						break ;

					case 2 :
						strcpy (line, "\n") ;
						break ;
				}
			}

		/* If the end-of-file was encountered then close the	*/
		/* current input file and pop the file stack. Loop to	*/
		/* try to get another line.				*/
		if (rc == NULL)
		{	if (source[fdp].fid != 0)
			{	fclose (source[fdp].fid ) ;
				free   ((void *)source[fdp].name) ;
			}
			if ((fdp -= 1) < 0) break  ;
			flineno = source[fdp].lno  ;
			sprintf	(line, "#line %s %d\n", source[fdp].name,
							source[fdp].lno) ;
		}

		/* If the latest line would cause a buffer overflow,	*/
		/* copy in what we can, flag that there is a line	*/
		/* pending, and exit the loop.				*/
		if ((size_t)(got + (llen = strlen (line))) > size)
		{	memcpy	(&buff[got], line, size - got) ;
			memmove	(&line[0], &line[size - got],
					   llen - (size - got)) ;
			line[llen - (size - got)] = 0 ;
			lpend	= 1    ;
			got	= size ;
			break	;
		}

		/* We have a line which will fit the buffer. Copy it in	*/
		/* and increment the count of characters read so far.	*/
		memcpy (&buff[got], line, llen) ;
		got	+= llen	;
	}

//	fprintf	(stderr, "el_read [%3d][%s]\n", got, buff) ; 
	return	got ;
}

GVAR	int	el_lineno ;
GVAR	char	*el_fname ;
GVAR	int	_el_error ;
GVAR	int	elyylineno;

/*L el_in_iinit	: Initialise input subsystem				*/
/*  ifd		: FILE *	: Initial input file descriptor		*/
/*  name	: const char *	: Initial input file name		*/
/*  (returns)	: void		:					*/

GFUNC	void	el_in_iinit
	(	FILE		*ifd,
		const char	*name
	)
{
	source[0].fid	= fdopen (dup (fileno (ifd)), "r") ;
	source[0].name	= strdup (name)	;
	source[0].buff	= 0		;
	source[0].bp	= 0		;
	el_fname	= strdup (name)	;
	fdp		= 0		;
	flineno		= 0		;
	bptr		= 0		;
	blen		= 0		;
	lpend		= 0		;
}

/*L el_in_sinit	: Initialise input subsystem				*/
/*  sstr	: const char *	: Source text				*/
/*  name	: const char *	: Initial input file name		*/
/*  (returns)	: void		:					*/

GFUNC	void	el_in_sinit
	(	const char	*sstr,
		const char	*name
	)
{
	source[0].fid	= 0 	     	;
	source[0].buff	= sstr		;
	source[0].bp	= sstr		;
	source[0].name	= strdup (name)	;
	el_fname	= strdup (name)	;
	fdp		= 0		;
	flineno		= 0		;
	bptr		= 0		;
	blen		= 0		;
	lpend		= 0		;
}

GFUNC	int	el_getc ()
{
	if (bptr >= blen)
	{	if ((blen = el_read (buff, sizeof(buff))) <= 0)
			return	EOF ;
		bptr	= 0  ;
	}

	return	buff[bptr++] ;
}

/*G el_yyerror	: Local replacement for yyerror				*/
/*  fmt		: char *	: Error message format			*/
/*  (returns)	: void		:					*/

GFUNC	void	el_yyerror
	(	char const *fmt,
		...
	)
{

	va_list	  aptr	     ;
	va_start (aptr, fmt) ;

	fprintf	  (_el_errout, "%s: %d: ", el_fname, el_lineno) ;
	vfprintf  (_el_errout, fmt, aptr) ;
	if (fmt[strlen(fmt)-1] != '\n') fprintf (_el_errout, "\n") ;
	_el_error += 1 ;
}

GFUNC	void	el_in_clean ()
{
	while (fdp >= 0)
	{
		if (source[fdp].fid != 0)
			fclose (source[fdp].fid) ;
		free ((void *)source[fdp].name) ;
		fdp -= 1 ;
	}
}
