/***************************************************************************
    file	         : kb_database.h
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#ifndef	_KB_DATABSE_H
#define	_KB_DATABSE_H


#ifdef _WIN32
#pragma warning(disable : 4251)
#endif

#include	<stdio.h>

#ifndef		_WIN32
#include	<sys/types.h>
#endif

#include	<qobject.h>
#include	<qstring.h>
#include	<qlist.h>
#include	<qvaluelist.h>
#include	<qintdict.h>
#include	<qtextcodec.h>


#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_dom.h"

#include	"libcommon_exports.h"


class	KBTable		;
class	KBDesignInfo	;
class	QTabWidget	;



/*  KBFieldSpec								*/
/*  -----------								*/
/*  The KBFieldSpec class is used to hold a description of a field	*/
/*  (column) in a database table.					*/

class 	LIBCOMMON_API	KBFieldSpec
{
public	:

	enum
	{	Primary	= 0x0001,	/* Field is primary key		*/
		NotNull	= 0x0002,	/* Field may not be null	*/
		Unique	= 0x0004,	/* Field is unique		*/
		Serial	= 0x0008,	/* Field gets serial values	*/
		Indexed = 0x0010,	/* Field is indexed		*/
		ReadOnly= 0x0020,	/* Field is read-only		*/
		InsAvail= 0x0100	/* Value available after insert	*/
	}	;

	KB::RState	m_state	    ;	/* Field spec state		*/
	uint	  	m_colno	    ;	/* Column number within table	*/
	QString	  	m_name	    ;	/* Field name			*/
	QString	  	m_typeName  ;	/* Field type as string		*/
	KB::IType	m_typeIntl  ;	/* Internal type		*/
	uint		m_flags	    ;	/* Primary, ... flags		*/
	uint	  	m_length    ;	/* Field maximum length		*/
	uint		m_prec	    ;	/* Precision			*/
	bool	  	m_dirty	    ;	/* Spec is dirty wrt. display	*/
	QString		m_defval    ;	/* Database default expression	*/
	KBType		*m_dbType   ;	/* Database derived type	*/

	KBTable		*m_table    ;	/* Used in query code		*/

	KBFieldSpec	(uint, cchar *, cchar *, KB::IType, uint, uint, uint) ;
	KBFieldSpec	(uint) ;
	KBFieldSpec	(uint, const QDomElement &) ;
	KBFieldSpec	(KBFieldSpec &) ;
	~KBFieldSpec	() ;

	void	toXML	(QDomElement &, KBDesignInfo * = 0) ;

	bool	operator == (const KBFieldSpec &other) const ;
}	;


/*  KBTableDetails							*/
/*  --------------							*/
/*  This class is used to return results from the ::listTables method	*/
/*  and contains global information about the table. It is also used	*/
/*  for table-like objects such as views.				*/

class	LIBCOMMON_API	KBTableDetails
{
public	:

	QString		m_name	;
	KB::TableType	m_type	;
	uint		m_perms	;
	QString		m_extra	;

	KBTableDetails	()	;
	KBTableDetails	(const QString &, KB::TableType, uint, const QString & = QString::null) ;
	KBTableDetails	(const KBTableDetails &) ;

	QString		typeText()	;

}	;

typedef	QValueList<KBTableDetails>	KBTableDetailsList	;


/*  KBTableSpec								*/
/*  -----------								*/
/*  Class used to hold table (and view) related information; the name,	*/
/*  the primary key (if any) column index, and the field specifications	*/

class 	LIBCOMMON_API	KBTableSpec
{
public	:

	KB::TableType	   m_type	;  /* Table/view type		*/
	QString		   m_name    	;  /* Table/view name		*/
	QList<KBFieldSpec> m_fldList  	;  /* List of all fields	*/
	bool		   m_keepsCase	;  /* Server preserves case	*/
	int		   m_prefKey  	;  /* Prefered key column	*/
	KBFieldSpec	   *m_fakeKey	;  /* Fake insert key		*/
	QString		   m_view	;  /* View SQL			*/

	KBTableSpec			(const QString & = QString::null) ;
	KBTableSpec			(const QDomElement &) ;
	KBTableSpec			(const KBTableSpec &) ;

	void		reset		(const QString & = QString::null) ;
	KBFieldSpec	*findField	(const QString &) ;

	KBTableSpec	&operator =	(const KBTableSpec &) ;

	void		toXML		(QDomElement &, QList<KBDesignInfo> * = 0) ;

	bool		insertKeyAvail	(int)	;

	KBFieldSpec	*findPrimary	()	;
	KBFieldSpec	*findUnique	()	;
}	;

/*  ------------------------------------------------------------------  */

/*  KBSequenceSpec							*/
/*  --------------							*/
/*  Specification for a sequence object in the database			*/

class	LIBCOMMON_API	KBSequenceSpec
{
public	:

	enum	Flags
	{	HasIncrement	= 0x01,
		HasMinValue	= 0x02,
		HasMaxValue	= 0x04,
		HasStart	= 0x08,
		CanCycle	= 0x80
	}	;

	QString		m_name		;
	int		m_increment	;
	int		m_minValue	;
	int		m_maxValue	;
	int		m_start		;
	uint		m_flags		;

	KBSequenceSpec			(const QString & = QString::null) ;
	KBSequenceSpec			(const QDomElement    &) ;
	KBSequenceSpec			(const KBSequenceSpec &) ;

	void		toXML		(QDomElement &) ;
}	;

/*  ------------------------------------------------------------------  */

/*  KBDriverDetails							*/
/*  ---------------							*/
/*  Driver details structure, used by the listDrivers routine.		*/

class	LIBCOMMON_API	KBDriverDetails
{
	QString		m_tag		;
	QString		m_comment	;
	QString		m_info		;
	uint		m_flags		;

public	:

	KBDriverDetails	()		;
	KBDriverDetails	(const QString &, const QString &, const QString &, uint) ;
	KBDriverDetails (const KBDriverDetails &) ;

	inline	const QString	&tag	() const
	{
		return	m_tag	  ;
	}
	inline	const QString	&comment() const
	{
		return	m_comment ;
	}
	inline	const QString	&info   () const
	{
		return	m_info    ;
	}
	inline	uint		flags   () const
	{
		return	m_flags   ;
	}
}	;

typedef	QValueList<KBDriverDetails>	KBDriverDetailsList	;

/*  ------------------------------------------------------------------  */

class	KBServer	;
class	KBSQLSelect	;
class	KBSQLUpdate	;
class	KBSQLInsert	;
class	KBSQLDelete	;
class	KBSQLCursor	;


#define	QP_SELECT	0x0001
#define	QP_INSERT	0x0002
#define	QP_UPDATE	0x0004
#define	QP_DELETE	0x0008

#define	FF_LENGTH	0x0001
#define	FF_PREC		0x0002
#define	FF_NOCREATE	0x0004

#define	AF_HOST		0x0001
#define	AF_PORTNUMBER	0x0002
#define	AF_SOCKETNAME	0x0004
#define	AF_FLAGS	0x0008
#define	AF_USERPASSWORD	0x0010
#define	AF_SSHTUNNEL	0x0020



/*  KBServer								*/
/*  --------								*/
/*  This class is encapsulates a connection to a server. It contains	*/
/*  some values which are pretty well common to the various SQL		*/
/*  engines, but most methods are pure virtual, and are implemented in	*/
/*  derived classes for specific databases.				*/

class	LIBCOMMON_API	KBServer : public QObject
{
protected :

	QString		m_serverName	;	/* Copy of server name	*/
	QString		m_host		;	/* Database host	*/
	QString		m_user		;	/* Login user name	*/
	QString		m_password	;	/* Login password	*/
	QString		m_database	;	/* Database name	*/
	QString		m_port		;	/* TCP/IP port		*/
	QString		m_disable	;	/* Server is disabled	*/
	bool		m_showAllTables	;	/* Show all tables	*/
	bool		m_cacheTables	;	/* Cache table details	*/
	bool		m_printQueries	;	/* Debug print queries	*/
	bool		m_pkReadOnly	;	/* Primary keys r/only	*/
	bool		m_fakeKeys	;	/* Fake insert keys	*/
	bool		m_readOnly	;	/* Read-only access	*/

	bool		m__conn		;	/* Connected flag	*/
	KBError		m_lError	;	/* Last error message	*/

	KBTableDetailsList
			m_tableList	;	/* Cached table list	*/
	QDict<KBTableSpec>
			m_tableCache	;	/* Cached table info	*/

	QTextCodec	*m_dataCodec	;	/* Non-default data	*/
	QTextCodec	*m_objCodec	;	/* Non-default object	*/

#ifndef	_WIN32
	QString		m_sshTarget	;
	pid_t		m_sshPID	;
	int		m_sshPort	;

	int		openSSHTunnel	(int = -1) ;
#endif

	void		noSequences	() ;
	void		noViews		() ;

	virtual	bool	doConnect    	(KBServerInfo  *)	 = 0 ;
	virtual	bool 	doListTables	(KBTableDetailsList&, uint = KB::IsTable|KB::IsView) = 0 ;
	virtual	bool	doListFields 	(KBTableSpec   &)	 = 0 ;
	virtual	bool	doCreateTable	(KBTableSpec   &,  bool, bool = false) = 0 ;
	virtual	bool	doRenameTable	(cchar *, cchar *, bool) = 0 ;
	virtual	bool	doDropTable  	(cchar *, 	   bool) = 0 ;

public	:

	enum	Transaction
	{
		BeginTransaction,
		CommitTransaction,
		RollbackTransaction
	}	;

	enum	Operator
	{
		EQ		= 0,
		NEQ		= 1,
		LE		= 2,
		GE		= 3,
		LT		= 4,
		GT		= 5,
		Like		= 6
	}	;

	enum	Syntax
	{	Limit
	}	;

	KBServer () ;
	virtual~KBServer () ;

	virtual	bool	 	connect    	(KBServerInfo  *) ;

	virtual	KBSQLSelect	*qrySelect	(bool, const QString &, bool) 		 = 0 ;
	virtual	KBSQLUpdate	*qryUpdate	(bool, const QString &, const QString &) = 0 ;
	virtual	KBSQLInsert	*qryInsert	(bool, const QString &, const QString &) = 0 ;
	virtual	KBSQLDelete	*qryDelete	(bool, const QString &, const QString &) = 0 ;
	virtual	KBSQLCursor	*qryCursor	(bool, const QString &, const QString &) ;

	virtual	bool		command		(bool, const QString &, uint, KBValue *, KBSQLSelect ** = 0) = 0 ;
	virtual	bool		transaction	(Transaction, void **) ;

	virtual	bool 		tableExists	(const QString &, bool &) = 0 ;
	virtual	bool 		listTables	(KBTableDetailsList&, uint = KB::IsTable|KB::IsView) ;

	virtual	bool		listDatabases	(QStringList &) ;

	//! \brief Get table information, including columns.
	//!
	//! This method retrieves information about a table or a view,
	//! including type, fields and, in the case of views, the SQL
	//! query that defines the view.
	bool 			listFields	(KBTableSpec &) ;

	bool			createTable	(KBTableSpec    &, bool, bool = false) ;
	bool			renameTable	(cchar *, cchar *, bool) ;
	bool			dropTable	(cchar *, bool) ;

	virtual	bool 		viewExists	(const QString &, bool &) ;
	virtual bool		createView	(KBTableSpec    &) ;
	virtual bool		renameView	(cchar *, cchar *) ;
	virtual bool		dropView	(cchar *) ;

	virtual	bool 		sequenceExists	(const QString &, bool &) ;
	virtual bool		descSequence	(KBSequenceSpec &) ;
	virtual bool		createSequence	(KBSequenceSpec &) ;
	virtual bool		renameSequence	(cchar *, cchar *) ;
	virtual bool		dropSequence	(cchar *) ;

	virtual	QString		rekallPrefix	(const QString & = QString::null) ;
	virtual	QString		mapExpression	(const QString &) ;

	virtual	QString		listTypes	()	;
	virtual	QString		placeHolder	(uint)	;
	virtual	bool		keepsCase	()	;

	virtual	uint 		optionFlags	()	    ;
	static	uint		defOperatorMap	(cchar **&) ;
	virtual	uint		operatorMap	(cchar **&) ;

	virtual	QString		setting		(const QString &) ;

	bool			execInitSQL	(const QString &) ;

	static	QString		syntaxToText	(Syntax);
	virtual	bool		getSyntax	(QString &, Syntax, ...) ;

	void			flushTableCache	()	;

	void			printQuery	(const QString &, uint, const KBValue *, bool) ;
	bool			subPlaceList	(const QString &, uint, const KBValue *, KBDataBuffer &, QTextCodec *, KBError &) ;
	QString			subPlaceList	(const QString &, uint, const KBValue *, KBError &) ;

	QString			doMapExpression	(const QString &, cchar *, cchar *, const QString &) ;

	const	KBError		&lastError	() ;

	inline	QTextCodec	*dataCodec	() { return m_dataCodec ; }
	inline	QTextCodec	*objCodec	() { return m_objCodec  ; }

	inline	QTextCodec	*getCodec
		(	bool	data
		)
	{
		return	data ? m_dataCodec : m_objCodec ;
	}

	static bool		listDrivers	(KBDriverDetailsList &, KBError &) ;
}	;


/*  KBSQLQuery								*/
/*  ---------								*/
/*  Base class for all queries.						*/

class	LIBCOMMON_API	KBSQLQuery
{
protected :

	QString			m_rawQuery	;
	QString			m_subQuery	;
	KBError			m_lError	;
	KBServer		*m_server	;
	int			m_nRows		;
	QTextCodec		*m_codec	;
	const KBValue::VTrans	*m_vtrans	;

public	:

	KBSQLQuery (KBServer *, bool, const QString &) ;
virtual~KBSQLQuery () ;

	virtual	bool		execute		(uint, const KBValue *) = 0 ; 

	const KBError		&lastError	() ;

	inline	const QString	&getRawQuery	() { return m_rawQuery ; }
	inline	const QString	&getSubQuery	() { return m_subQuery ; }

	virtual	KBSQLSelect	*isSelect	() { return 0 ; }
	virtual	KBSQLUpdate	*isUpdate	() { return 0 ; }
	virtual	KBSQLInsert	*isInsert	() { return 0 ; }
	virtual	KBSQLDelete	*isDelete	() { return 0 ; }

	inline	void	setVTrans
		(	const KBValue::VTrans	*vtrans
		)
	{
		m_vtrans = vtrans ;
	}

	/* The following values are be returned by "getNumRows" when	*/
	/* the number of rows is not available.				*/
	enum
	{
		RowsUnknown	= -1,
		RowsError	= -2
	}	;

}	;

/*  KBSQLSelect								*/
/*  -----------								*/
/*  The KBSQLSelect class is used as a base class for SELECT queries.	*/

class	LIBCOMMON_API	KBSQLSelect : public KBSQLQuery
{
protected :

	uint			m_nFields		;
	KBType			**m_types		;

	/* Value cache. The rekall query code needs to be able to move	*/
	/* at random amongst the rows in a query, but some (or most)	*/
	/* drivers only allow row-serial access to the data (and some 	*/
	/* may only allow column-serial access). To handle this, we	*/
	/* implement a cache to store fetched values. Note that the	*/
	/* dumpRow/dumpRowsTo/dumpAllRows calls should be used to	*/
	/* release space as soon as is possible.			*/
	QIntDict<KBValue>	m_valueCache		;

	void			putInCache	(uint,	     KBValue *) ;
	void			putInCache	(uint, uint, KBValue &) ;
	bool			getFromCache	(uint, uint, KBValue &) ;

public	:

	KBSQLSelect (KBServer *, bool, const QString &) ;
virtual~KBSQLSelect ()		   ;

	virtual	bool		execute		(uint, const KBValue *) = 0 ; 
	virtual	uint		getNumFields	() ;
	virtual	QString		getFieldName	(uint)	     = 0 ;

	virtual	KBValue		getField
				(	uint,
					uint,
					KBValue::VTrans = KBValue::VDefault
				)	= 0 ;

	virtual	bool		rowExists	(uint, bool  = false) ;
	virtual	void		dumpRow		(uint)	;
	virtual	void		dumpRowsTo	(uint)	;
	virtual	void		dumpAllRows	()	;

	KBType			*getFieldType	(uint)	;

	virtual	KBSQLSelect	*isSelect	() { return this ; }
	int			getNumRows	() ;
}	;


/*  KBSQLUpdate								*/
/*  -----------								*/
/*  The KBSQLUpdate class is used as a base class for UPDATE queries.	*/

class	LIBCOMMON_API	KBSQLUpdate : public KBSQLQuery
{
protected :

	QString			m_tabName	;

public	:

	KBSQLUpdate (KBServer *, bool, const QString &, const QString &) ;
	virtual~KBSQLUpdate ()		 ;

	virtual	bool	execute	     (uint, const KBValue *) = 0 ;

	virtual	KBSQLUpdate	*isUpdate	() { return this    ; }
	inline	int		getNumRows	() { return m_nRows ; }
}	;

/*  KBSQLInsert								*/
/*  -----------								*/
/*  The KBSQLInsert class is used as a base class for INSERT queries.	*/

class	LIBCOMMON_API	KBSQLInsert : public KBSQLQuery
{
protected :

	QString			m_tabName	;

public	:

	KBSQLInsert (KBServer *, bool, const QString &, const QString &)   ;
	virtual~KBSQLInsert ()		   ;

	virtual	bool	execute	  (uint, const KBValue *) = 0 ;
	virtual	bool	getNewKey (const QString &, KBValue &, bool) = 0 ;

	virtual	KBSQLInsert	*isInsert	() { return this    ; }
	inline	int		getNumRows	() { return m_nRows ; }
}	;

/*  KBSQLDelete								*/
/*  -----------								*/
/*  The KBSQLDelete class is used as a base class for delete queries.	*/

class	LIBCOMMON_API	KBSQLDelete : public KBSQLQuery
{
protected :

	QString			m_tabName	;

public	:

	KBSQLDelete (KBServer *, bool, const QString &, const QString &)   ;
	virtual~KBSQLDelete ()		   ;

	virtual	bool	execute	(uint, const KBValue *) = 0 ;

	virtual	KBSQLDelete	*isDelete	() { return this    ; }
	inline	int		getNumRows	() { return m_nRows ; }
}	;

/*  KBSQLCursor								*/
/*  -----------								*/
/*  The KBSQLCursor class is used as a base class for cursors.		*/

class	LIBCOMMON_API	KBSQLCursor : public KBSQLQuery
{
protected :

	QString			m_cursor	;
	uint			m_nFields	;
	KBType			**m_types	;

public	:

	KBSQLCursor (KBServer *, bool, const QString &, const QString &) ;
	virtual ~KBSQLCursor ()	;

	virtual	bool	fetch	(uint, KBValue *, bool &) = 0 ;
	virtual	bool	close	()			  = 0 ;

	inline	const	KBError	&lastError ()
	{
		return	m_lError ;
	}
}	;

/*  ------------------------------------------------------------------  */

/*  KBDBAdvanced							*/
/*  ------------							*/
/*  This class is a virtual base class for advanced RDBMS-specific	*/
/*  driver settings. It is used both to hold the settings and to	*/
/*  provide an interface to save and load settings via a DOM document	*/
/*  and to provide a dialog to control the settings.			*/

class	LIBCOMMON_API	KBDBAdvanced : public QObject
{
	Q_OBJECT

	QString		m_dbType	;

public	:

	KBDBAdvanced	(const QString &) ;
	virtual~KBDBAdvanced	()	;

	virtual	void		load		(const QDomElement &) = 0 ;
	virtual	void		save		(QDomElement	   &) = 0 ;
	virtual	void		setupDialog	(QTabWidget 	   *) = 0 ;
	virtual	void		saveDialog	() = 0 ;
	virtual	KBDBAdvanced	*copy		() = 0 ;

	inline	bool	isType
		(	const QString	&dbType
		)
	{
		return	dbType == m_dbType ;
	}
}	;


#endif	// _KB_DATABSE_H
