/***************************************************************************
    file	         : kb_basequery.cpp
    copyright            : (C) 1999,2000,2001,2002,2003,2004 by Mike Richardson
			   (C) 2001,2002,2003,2004 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/


#include	"kb_dblink.h"
#include	"kb_basequery.h"

KBBaseQueryValue::KBBaseQueryValue ()
	:
	m_type	('X')
{
}

KBBaseQueryValue::KBBaseQueryValue
	(	const QString	&field
	)
	:
	m_field	(field),
	m_type	('V')
{
}

KBBaseQueryValue::KBBaseQueryValue
	(	const QString	&field,
		const QString	&text
	)
	:
	m_field	(field),
	m_type	('S'),
	m_text	(text)
{
}

KBBaseQueryValue::KBBaseQueryValue
	(	const QString	&field,
		int		number
	)
	:
	m_field	(field),
	m_type	('D'),
	m_fixed	(number)
{
}

KBBaseQueryValue::KBBaseQueryValue
	(	const QString	&field,
		double		number
	)
	:
	m_field	(field),
	m_type	('F'),
	m_float	(number)
{
}

KBBaseQueryValue::KBBaseQueryValue
	(	const QString	&field,
		void		*
	)
	:
	m_field	(field),
	m_type	('P')
{
}

/*  KBBaseQueryValue							*/
/*  addToInsert	: Add query value to insert query			*/
/*  dbLink	: KBDBLink *	: Link to database			*/
/*  place	: uint		: Current place holder index		*/
/*  fields	: QStringList &	: List of fields being assembled	*/
/*  exprs	: QStringList &	: List of expressions being assembled	*/
/*  (returns)	: uint		: Possibly incremented place holder	*/

uint	KBBaseQueryValue::addToInsert
	(	KBDBLink	*dbLink,
		uint		place,
		QStringList	&fields,
		QStringList	&exprs
	)
{
	/* This call is used when building an insert query, so the	*/
	/* field and the expression are added to separate lists.	*/
	fields.append (dbLink->mapExpression (m_field)) ;

	switch (m_type)
	{
		case 'V' :
			exprs.append (dbLink->placeHolder(place)) ;
			place	+= 1 ;
			break	;

		case 'S' :
			exprs.append ("'" + m_text + "'") ;
			break	;

		case 'D' :
			exprs.append (QString::number (m_fixed)) ;
			break	;

		case 'F' :
			exprs.append (QString::number (m_float)) ;
			break	;

		default	:
			exprs.append ("null") ;
			break	;
	}

	return	place	;
}

/*  KBBaseQueryValue							*/
/*  addToUpdate	: Add query value to update query			*/
/*  dbLink	: KBDBLink *	: Link to database			*/
/*  place	: uint		: Current place holder index		*/
/*  terms	: QStringList &	: List of terms being assembled		*/
/*  where	: bool		: True for "where" expression		*/
/*  (returns)	: uint		: Possibly incremented place holder	*/

uint	KBBaseQueryValue::addToUpdate
	(	KBDBLink	*dbLink,
		uint		place,
		QStringList	&terms,
		bool		where
	)
{
	/* This is used for update queries, hence we form terms like	*/
	/* "a = b", except that for "where" terms and explicit nulls,	*/
	/* used "a is null"						*/
	QString	expr	;
	cchar	*oper	= "=" ;

	switch (m_type)
	{
		case 'V' :
			expr	 = dbLink->placeHolder(place) ;
			place	+= 1 ;
			break	;

		case 'S' :
			expr	= "'" + m_text + "'" ;
			break	;

		case 'D' :
			expr	= QString::number (m_fixed) ;
			break	;

		case 'F' :
			expr	= QString::number (m_float) ;
			break	;

		default	:
			expr	= "null" ;
			if (where) oper = "is" ;
			break	;
	}

	terms.append
	(	QString("%1 %2 %3")
			.arg(dbLink->mapExpression (m_field))
			.arg(oper)
			.arg(expr)
	)	;

	return	place	;
}


/*  ------------------------------------------------------------------	*/

KBBaseQuery::KBBaseQuery
	(	const QString	&table
	)
	:
	m_table	(table)
{
}

void	KBBaseQuery::setTable
	(	const QString	&table
	)
{
	m_table	= table	;
}

void	KBBaseQuery::addExpression
	(	const QString 	&field
	)
{
	m_values.append (KBBaseQueryValue (field)) ;
}

void	KBBaseQuery::addExpression
	(	const QString 	&field,
		const QString 	&text
	)
{
	m_values.append (KBBaseQueryValue (field, text)) ;
}

void	KBBaseQuery::addExpression
	(	const QString 	&field,
		int   		number
	)
{
	m_values.append (KBBaseQueryValue (field, number)) ;
}

void	KBBaseQuery::addExpression
	(	const QString 	&field,
		double		number
	)
{
	m_values.append (KBBaseQueryValue (field, number)) ;
}

void	KBBaseQuery::addExpression
	(	const QString 	&field,
		void 		*p
	)
{
	m_values.append (KBBaseQueryValue (field, p)) ;
}

/*  ------------------------------------------------------------------	*/

KBBaseUpdate::KBBaseUpdate
	(	const QString	&table
	)
	:
	KBBaseQuery (table)
{
}

void	KBBaseUpdate::addWhere
	(	const QString 	&field
	)
{
	m_where.append (KBBaseQueryValue (field)) ;
}

void	KBBaseUpdate::addWhere
	(	const QString 	&field,
		const QString 	&text
	)
{
	m_where.append (KBBaseQueryValue (field, text)) ;
}

void	KBBaseUpdate::addWhere
	(	const QString 	&field,
		int   		number
	)
{
	m_where.append (KBBaseQueryValue (field, number)) ;
}

void	KBBaseUpdate::addWhere
	(	const QString 	&field,
		double		number
	)
{
	m_where.append (KBBaseQueryValue (field, number)) ;
}

void	KBBaseUpdate::addWhere
	(	const QString 	&field,
		void 		*p
	)
{
	m_where.append (KBBaseQueryValue (field, p)) ;
}

QString	KBBaseUpdate::getQueryText
	(	KBDBLink	*dbLink
	)
{
	uint		place	= 0 ;
	QStringList	fields	;
	QStringList	where	;

	for (uint idx1 = 0 ; idx1 < m_values.count() ; idx1 += 1)
		place	= m_values[idx1].addToUpdate (dbLink, place, fields) ;
	for (uint idx2 = 0 ; idx2 < m_where .count() ; idx2 += 1)
		place	= m_where [idx2].addToUpdate (dbLink, place, where, true) ;

	QString	update	= QString("update %1 set %2")
				 .arg(dbLink->mapExpression(m_table))
				 .arg(fields.join(", "))
			  ;

	if (where.count() > 0)
		update += " where " + where.join(" and ") ;

	return	update	;
}

/*  ------------------------------------------------------------------	*/

KBBaseInsert::KBBaseInsert
	(	const QString	&table
	)
	:
	KBBaseQuery (table)
{
}

QString	KBBaseInsert::getQueryText
	(	KBDBLink	*dbLink
	)
{
	uint		place	= 0 ;
	QStringList	fields	;
	QStringList	exprs	;

	for (uint idx = 0 ; idx < m_values.count() ; idx += 1)
		place	= m_values[idx].addToInsert (dbLink, place, fields, exprs) ;

	return	QString("insert into %1 (%2) values (%3)")
			.arg(dbLink->mapExpression(m_table))
			.arg(fields.join(", "))
			.arg(exprs .join(", "))
			;
}
