/***************************************************************************
    file	         : kb_rawsql.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/


#include	<stdio.h>
#include	<qregexp.h>
#include	<qtabbar.h>


#ifndef		_WIN32
#include	"kb_rawsql.moc"
#else
#include	"kb_rawsql.h"
#endif

#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_options.h"
#include	"kb_dialog.h"



/*  KBRawSQLPage							*/
/*  KBRawSQLPage: Constructor for raw SQL execution widget		*/
/*  rawSQL	: KBRawSQL *	  : Parent object			*/
/*  caption	: const QString & : Tab caption				*/
/*  query	: const QString & : Initial query text			*/
/*  (returns)	: KBRawSQLPage	  :					*/

KBRawSQLPage::KBRawSQLPage
	(	KBRawSQL	*rawSQL,
		const QString	&caption,
		const QString	&query
	)
	:
	TKTextEditor	(new TKTextDocument (_KBDialog::getTextManager(KBOptions::getScriptFont())), rawSQL->tabWidget()),
	m_rawSQL	(rawSQL),
	m_caption	(caption)
{
	m_textView	= 0 ;
	m_listView	= 0 ;

	connect
	(	rawSQL->tabWidget(),
		SIGNAL(currentChanged(QWidget *)),
		SLOT  (slotTabChanged(QWidget *))
	)	;

	setText	 (query) ;
	showText ("")	 ;
}

/*  KBRawSQLPage							*/
/*  ~KBRawSQLPage: Destructor for raw SQL execution widget		*/
/*  (returns)	 :		:					*/

KBRawSQLPage::~KBRawSQLPage ()
{
	DELOBJ	(m_textView) ;
	DELOBJ	(m_listView) ;
}

/*  KBRawSQLPage							*/
/*  save	: Save configuration					*/
/*  config	: TKConfig *	: Configuration object			*/
/*  idx		: uint		: Index in configuration		*/
/*  (returns)	: void		:					*/

void	KBRawSQLPage::save
	(	TKConfig	*config,
		uint		idx
	)
{
	config->writeEntry (QString("caption_%1").arg(idx), m_caption) ;
	config->writeEntry (QString("query_%1"  ).arg(idx), text   ()) ;
}

/*  KBRawSQLPage							*/
/*  slotTabChanged							*/
/*		: Parent tabber changes tab				*/
/*  widget	: QWidget *	: New widget shown			*/
/*  (returns)	: void		:					*/

void	KBRawSQLPage::slotTabChanged
	(	QWidget		*widget
	)
{
	if (widget == this)
	{
		if (m_textView != 0)
			m_rawSQL->stack()->raiseWidget (m_textView) ;
		if (m_listView != 0)
			m_rawSQL->stack()->raiseWidget (m_listView) ;

		m_rawSQL->setCaption (m_caption) ;
	}
}

void	KBRawSQLPage::cleanPage ()
{
	if (m_listView != 0)
	{
		m_rawSQL->stack()->removeWidget (m_listView) ;
		DELOBJ	(m_listView) ;
	}
	if (m_textView != 0)
	{
		m_rawSQL->stack()->removeWidget (m_textView) ;
		DELOBJ	(m_textView) ;
	}
}

/*  KBRawSQLPage							*/
/*  showText	: Show text in this page				*/
/*  text	: const QString & : Text to display			*/
/*  (returns)	: void		  :					*/

void	KBRawSQLPage::showText
	(	const QString	&text
	)
{
	QWidgetStack *stack = m_rawSQL->stack() ;

	cleanPage () ;

	m_textView = new QTextView (stack) ;
	stack->addWidget (m_textView, -1) ;

	m_textView->setText	(text) ;
	m_textView->show	() ;
	stack	  ->raiseWidget (m_textView) ;
}

/*  KBRawSQLPage							*/
/*  showSelect	: Show select results					*/
/*  qrySelect	: KBSQLSelect *	: Select in question			*/
/*  (returns)	: void		:					*/

void	KBRawSQLPage::showSelect
	(	KBSQLSelect	*qrySelect
	)
{
	QWidgetStack *stack = m_rawSQL->stack() ;

	cleanPage () ;

	/* Create a new list view and set up columns for each column	*/
	/* returned by the query, plus an additional column for the row	*/
	/* number. Note that we disable updates until everything has	*/
	/* been updated.						*/
	m_listView = new QListView (stack) ;

	m_listView->setUpdatesEnabled(false) 	  ;
	m_listView->addColumn	     (TR("Row#")) ;

	for (uint col = 0 ; col < qrySelect->getNumFields() ; col += 1)
		m_listView->addColumn (qrySelect->getFieldName (col)) ;

	/* Now run through the table inserting the data from all the	*/
	/* returned rows.						*/
	for (uint row = 0 ; qrySelect->rowExists(row, true) ; row += 1)
	{
		QListViewItem *item = new QListViewItem(m_listView) ;

		item->setText (0, QString("%1").arg(row + 1)) ;

		for (uint col = 0 ; col < qrySelect->getNumFields() ; col += 1)
		{
			KBValue	value	=  qrySelect->getField(row, col) ;

			if (value.getType()->getIType() == KB::ITBinary)
			{
				item->setText (col + 1, "[Binary]") ;
				continue ;
			}

			QString	text	= value.getRawText() ;
			if (text.length() > 128) text = text.left(128) + " ..." ;

			item->setText (col + 1, text) ;
		}
	}

	m_listView->setUpdatesEnabled(true) ;

	stack	  ->addWidget	(m_listView, -1) ;
	m_listView->show	() ;
	stack	  ->raiseWidget (m_listView) ;
}

/*  KBRawSQLPage							*/
/*  executeCommand							*/
/*		: Execute a arbitrary SQL				*/
/*  query	: const QString & : Arbitrary SQL			*/
/*  (returns)	: void		  :					*/

void	KBRawSQLPage::executeCommand
	(	const QString	&query
	)
{
	KBSQLSelect	*qrySelect	= 0 ;

	if (!m_rawSQL->dbLink().command (true, query, 0, 0, &qrySelect))
	{
		showText
		(	QString	("<qt><ul><li>%1</li><li>%2</li></qt>")
				.arg(m_rawSQL->dbLink().lastError().getMessage())
				.arg(m_rawSQL->dbLink().lastError().getDetails())
		)	;
		return	;
	}

	if (qrySelect != 0)
	{
		showSelect (qrySelect) ;
		delete	qrySelect      ;
		return	;
	}

	showText(TR("OK")) ;
}

/*  KBRawSQLPage							*/
/*  executeSelect							*/
/*		: Execute a select query				*/
/*  query	: const QString & : Select query			*/
/*  (returns)	: void		  :					*/

void	KBRawSQLPage::executeSelect
	(	const QString	&query
	)
{
	KBSQLSelect	*qrySelect = m_rawSQL->dbLink().qrySelect (true, query) ;

	if (!qrySelect->execute (0, 0))
	{
		showText
		(	QString	("<qt><ul><li>%1</li><li>%2</li></qt>")
				.arg(qrySelect->lastError().getMessage())
				.arg(qrySelect->lastError().getDetails())
		)	;
		delete	qrySelect ;
		return	;
	}

	showSelect (qrySelect)	;
	delete	qrySelect	;
}

/*  KBRawSQLPage							*/
/*  executeUpdate							*/
/*		: Execute an update query				*/
/*  query	: const QString & : Select query			*/
/*  table	: const QString & : Table name				*/
/*  (returns)	: void		  :					*/

void	KBRawSQLPage::executeUpdate
	(	const QString	&query,
		const QString	&table
	)
{
	KBSQLUpdate	*qryUpdate = m_rawSQL->dbLink().qryUpdate (true, query, table) ;

	if (!qryUpdate->execute (0, 0))
	{
		showText
		(	QString	("<qt><ul><li>%1</li><li>%2</li></qt>")
				.arg(qryUpdate->lastError().getMessage())
				.arg(qryUpdate->lastError().getDetails())
		)	;
		delete	qryUpdate ;
		return	;
	}

	showText(QString(TR("Updated %1 records")).arg(qryUpdate->getNumRows())) ;
	delete	qryUpdate ;
}


/*  KBRawSQLPage							*/
/*  executeInsert							*/
/*		: Execute an insert query				*/
/*  query	: const QString & : Select query			*/
/*  table	: const QString & : Table name				*/
/*  (returns)	: void		  :					*/

void	KBRawSQLPage::executeInsert
	(	const QString	&query,
		const QString	&table
	)
{
	KBSQLInsert	*qryInsert = m_rawSQL->dbLink().qryInsert (true, query, table) ;

	if (!qryInsert->execute (0, 0))
	{
		showText
		(	QString	("<qt><ul><li>%1</li><li>%2</li></qt>")
				.arg(qryInsert->lastError().getMessage())
				.arg(qryInsert->lastError().getDetails())
		)	;
		delete	qryInsert ;
		return	;
	}

	showText(QString(TR("Inserted %1 records")).arg(qryInsert->getNumRows())) ;
	delete	qryInsert ;
}


/*  KBRawSQLPage							*/
/*  executeDelete							*/
/*		: Execute a delete query				*/
/*  query	: const QString & : Select query			*/
/*  table	: const QString & : Table name				*/
/*  (returns)	: void		  :					*/

void	KBRawSQLPage::executeDelete
	(	const QString	&query,
		const QString	&table
	)
{
	KBSQLDelete	*qryDelete = m_rawSQL->dbLink().qryDelete (true, query, table) ;

	if (!qryDelete->execute (0, 0))
	{
		showText
		(	QString	("<qt><ul><li>%1</li><li>%2</li></qt>")
				.arg(qryDelete->lastError().getMessage())
				.arg(qryDelete->lastError().getDetails())
		)	;
		delete	qryDelete ;
		return	;
	}

	showText(QString(TR("Deleted %1 records")).arg(qryDelete->getNumRows())) ;
	delete	qryDelete ;
}


/*  KBRawSQLPage							*/
/*  execute	: Execute query						*/
/*  (returns)	: void		:					*/

void	KBRawSQLPage::execute ()
{
	QString	query = text().stripWhiteSpace() ;
	if (query.isEmpty()) return ;

	static	QRegExp rxSelect   ("^select", 				false) ;
	static	QRegExp rxUpdate   ("^update\\s+([^\\s]+)", 		false) ;
	static	QRegExp rxInsert   ("^insert\\s+into\\s+([^\\s]+)",	false) ;
	static	QRegExp rxDelete   ("^delete\\s+from\\s+([^\\s]+)",	false) ;

	if (rxSelect.search(query) >= 0)
	{
		executeSelect (query) ;
		return	;
	}
	if (rxUpdate.search(query) >= 0)
	{
		executeUpdate (query, rxUpdate.cap(1)) ;
		return	;
	}
	if (rxInsert.search(query) >= 0)
	{
		executeInsert (query, rxInsert.cap(1)) ;
		return	;
	}
	if (rxDelete.search(query) >= 0)
	{
		executeDelete (query, rxDelete.cap(1)) ;
		return	;
	}

	executeCommand (query) ;
}

void	KBRawSQLPage::setCaption
	(	const QString	&caption
	)
{
	m_caption = caption ;
}

/*  ------------------------------------------------------------------  */

/*  KBRawSQL								*/
/*  KBRawSQL	: Constructor for raw SQL execution widget		*/
/*  parent	: QWidget *	  : Parent widget			*/
/*  dbInfo	: KBDBInfo &	  : Database information		*/
/*  server	: const QString & : Server name				*/
/*  ok		: bool		  : Success flag			*/
/*  (returns)	: KBRawSQL	  :					*/

KBRawSQL::KBRawSQL
	(	QWidget		*parent,
		KBDBInfo	*dbInfo,
		const QString	&server,
		bool		&ok
	)
	:
	KBasePart	(0, parent, WDestructiveClose|WStyle_NormalBorder),
	m_splitter	(m_partWidget),
	m_stack		(&m_splitter),
	m_queries	(&m_splitter),
	m_tabber	(&m_queries),
	m_name		(&m_queries),
	m_bAdd		(&m_queries),
	m_bCopy		(&m_queries),
	m_bRename	(&m_queries),
	m_bExecute	(&m_queries),
	m_bRemove	(&m_queries),
	m_dbInfo	(dbInfo),
	m_server	(server)
{
	m_topWidget = &m_splitter ;

	m_splitter.setOrientation (QSplitter::Vertical) ;
	m_splitter.setResizeMode  (&m_stack,   QSplitter::Stretch) ;
	m_splitter.setResizeMode  (&m_queries, QSplitter::Stretch) ;

	QVBoxLayout	*layQueries	= new QVBoxLayout (&m_queries) ;
	layQueries->addWidget	(&m_tabber  )	;

#if	__KB_EMBEDDED
	QGridLayout	*layButtons	= new QGridLayout (layQueries) ;
	layButtons->addMultiCellWidget	(&m_name,	0, 0, 0, 4) ;
	layButtons->addWidget		(&m_bAdd,	1, 0)	;
	layButtons->addWidget		(&m_bCopy,	1, 1)	;
	layButtons->addWidget		(&m_bRename,	1, 2)	;
	layButtons->addWidget		(&m_bExecute,	1, 3)	;
	layButtons->addWidget		(&m_bRemove,	1, 4)	;
#else
	QHBoxLayout	*layButtons	= new QHBoxLayout (layQueries) ;
	layButtons->addStretch	() ;
	layButtons->addWidget	(&m_name)	;
	layButtons->addWidget	(&m_bAdd)	;
	layButtons->addWidget	(&m_bCopy)	;
	layButtons->addWidget	(&m_bRename)	;
	layButtons->addWidget	(&m_bExecute)	;
	layButtons->addWidget	(&m_bRemove)	;
#endif
	_KBDialog::setupLayout	(layButtons, -1, -1) ;

	m_bAdd	  .setText	(TR("Add"    )) ;
	m_bCopy	  .setText	(TR("Copy"   )) ;
	m_bRename .setText	(TR("Rename" )) ;
	m_bExecute.setText	(TR("Execute")) ;
	m_bRemove .setText	(TR("Remove" )) ;

	QSize	bSize	= QSize(0,0) ;
	bSize	= bSize.expandedTo (m_bAdd    .sizeHint()) ;
	bSize	= bSize.expandedTo (m_bCopy   .sizeHint()) ;
	bSize	= bSize.expandedTo (m_bRename .sizeHint()) ;
	bSize	= bSize.expandedTo (m_bExecute.sizeHint()) ;
	bSize	= bSize.expandedTo (m_bRemove .sizeHint()) ;
	m_bAdd	  .setFixedSize   (bSize) ;
	m_bCopy	  .setFixedSize   (bSize) ;
	m_bRename .setFixedSize   (bSize) ;
	m_bExecute.setFixedSize   (bSize) ;
	m_bRemove .setFixedSize   (bSize) ;

	connect	(&m_bAdd,     SIGNAL(clicked()), SLOT(slotClickAdd    ())) ;
	connect	(&m_bCopy,    SIGNAL(clicked()), SLOT(slotClickCopy   ())) ;
	connect	(&m_bRename,  SIGNAL(clicked()), SLOT(slotClickRename ())) ;
	connect	(&m_bExecute, SIGNAL(clicked()), SLOT(slotClickExecute())) ;
	connect	(&m_bRemove,  SIGNAL(clicked()), SLOT(slotClickRemove ())) ;

	m_partWidget->show      () ;
	m_partWidget->setCaption("SQL") ;

	if (!m_dbLink.connect (m_dbInfo, m_server))
	{
		m_dbLink.lastError().DISPLAY() ;
		ok	= false	;
		return	;
	}

	/* Initially pick up the raw SQL global configuration. This	*/
	/* contains the initial size and splitter depths.		*/
	TKConfig *config = TKConfig::getConfig() ;
	config->setGroup ("Raw SQL Global") ;

	QSize		size	= config->readSizeEntry   ("geometry" ) ;
	QValueList<int> depths	= config->readIntListEntry("depths"   ) ;

	if (size.isEmpty()) size = QSize(500, 300) ;
	if (depths.count() == 2) m_splitter.setSizes (depths) ;

#if	__KB_EMBEDDED
	m_partWidget->showMaximized () ;
#else
	m_partWidget->resize	(size.width(), size.height(), true, false) ;
#endif

	/* Secondly see if there is a server-specific configuration	*/
	/* group, in which case reload the previous set of queries.	*/
	config->setGroup   (QString("Raw SQL: %1").arg(m_server)) ;
	uint	count	= config->readNumEntry ("numqueries") ;

	for (uint idx = 0 ; idx < count ; idx += 1)
	{
		QString		caption	= config->readEntry (QString("caption_%1").arg(idx)) ;
		QString		query	= config->readEntry (QString("query_%1"  ).arg(idx)) ;

		KBRawSQLPage	*page	= new KBRawSQLPage (this, caption, query) ;

		m_pages   .append     (page) ;
		m_tabber  .addTab     (page, caption) ;
		m_bExecute.setEnabled (true) ;
	}

	if (count == 0)
	{
		KBRawSQLPage *page = new KBRawSQLPage (this, TR("Query")) ;
		m_tabber.addTab (page, TR("Query")) ;
	}

//	m_bCopy   .setEnabled (count > 0) ;
//	m_bRename .setEnabled (count > 0) ;
//	m_bExecute.setEnabled (count > 0) ;
	m_bRemove .setEnabled (count > 1) ;
	ok	= true	;

	m_tabber  .setCurrentPage (0) ;
}

/*  KBRawSQL								*/
/*  queryClose	: Close handler						*/
/*  (returns)	: void			:				*/

bool	KBRawSQL::queryClose ()
{
	TKConfig	*config = TKConfig::getConfig () ;

	/* First save the geometry and splitter sizes to the global raw	*/
	/* SQL configuration group ....					*/
	config->setGroup   ("Raw SQL Global") ;
	config->writeEntry ("geometry",  m_partWidget->size ()) ;
	config->writeEntry ("depths",    m_splitter   .sizes()) ;


	/* ... and then save the individual queries to a server		*/
	/* specific group.						*/
	config->setGroup   (QString("Raw SQL: %1").arg(m_server)) ;
	config->writeEntry ("numqueries", m_pages.count()) ;

	uint	idx	= 0 ;

	LITER
	(	KBRawSQLPage,
		m_pages,
		page,

		page->save (config, idx) ;
		idx	+= 1 ;
	)

	config->sync () ;
	return	true	;
}

/*  KBRawSQLPage							*/
/*  slotClickAdd: Add a new query tab					*/
/*  (returns)	: void		:					*/

void	KBRawSQL::slotClickAdd ()
{
	QString	 name	= m_name.text () ;
	if (name.isEmpty())
		name	= QString(TR("Query %1")).arg(m_pages.count() + 1) ;

	KBRawSQLPage	*page	= new KBRawSQLPage (this, name) ;

	m_pages   .append     (page) ;
	m_tabber  .addTab     (page, name) ;
	m_bCopy   .setEnabled (true) ;
	m_bRename .setEnabled (true) ;
	m_bExecute.setEnabled (true) ;
	m_bRemove .setEnabled (true) ;
	m_name	  .clear      () ;

	m_tabber  .setCurrentPage (m_tabber.indexOf(page)) ;
}

/*  KBRawSQLPage							*/
/*  slotClickCopy							*/
/*		: Copy the current query tab				*/
/*  (returns)	: void		:					*/

void	KBRawSQL::slotClickCopy ()
{
	KBRawSQLPage  *currPage = (KBRawSQLPage *)m_tabber.currentPage() ;
	if (currPage == 0) return ;

	QString	 name	= m_name.text () ;
	if (name.isEmpty())
		name	= QString(TR("Query %1")).arg(m_pages.count() + 1) ;

	KBRawSQLPage	*newPage = new KBRawSQLPage (this, name, currPage->text()) ;

	m_pages   .append	  (newPage) ;
	m_tabber  .addTab    	  (newPage, name) ;
	m_name	  .clear      	  () ;
	m_tabber  .setCurrentPage (m_tabber.indexOf(newPage)) ;
}

/*  KBRawSQLPage							*/
/*  slotClickRename							*/
/*		: Rename the current query tab				*/
/*  (returns)	: void		:					*/

void	KBRawSQL::slotClickRename ()
{
	KBRawSQLPage 	*page	= (KBRawSQLPage *)m_tabber.currentPage() ;
	QString	 	name	= m_name.text() ;

	if ((page != 0) && !name.isEmpty())
	{
		m_tabber.changeTab	(page, name) ;
		page->setCaption 	(name) ;
		setCaption	 	(name) ;
	}
}


/*  KBRawSQLPage							*/
/*  slotClickExecute							*/
/*		: Execute the current tab				*/
/*  (returns)	: void		:					*/

void	KBRawSQL::slotClickExecute ()
{
	KBRawSQLPage  *currPage = (KBRawSQLPage *)m_tabber.currentPage() ;
	if (currPage != 0) currPage->execute () ;
}

/*  KBRawSQLPage							*/
/*  slotClickRemove							*/
/*		: Remove the current tab				*/
/*  (returns)	: void		:					*/

void	KBRawSQL::slotClickRemove ()
{
	if (m_pages.count() > 0)
	{
		KBRawSQLPage *page = (KBRawSQLPage *)m_tabber.currentPage() ;
		m_pages .remove     (page) ;
		m_tabber.removePage (page) ;
		delete	page  ;

		m_bCopy   .setEnabled (m_pages.count() > 0) ;
		m_bRename .setEnabled (m_pages.count() > 0) ;
		m_bExecute.setEnabled (m_pages.count() > 0) ;
		m_bRemove .setEnabled (m_pages.count() > 0) ;
	}
}

/*  KBRawSQLPage							*/
/*  setCaption	: Set window caption					*/
/*  caption	: const QString & : Caption text			*/
/*  (returns)	: void		  :					*/

void	KBRawSQL::setCaption
	(	const QString	&caption
	)
{
	m_partWidget->setCaption (QString("SQL: %1").arg(caption)) ;
}

