/***************************************************************************
    file	         : kb_pgsql.cpp
    copyright            : (C) 1999,2000,2001,2002,2003,2004 by Mike Richardson
			   (C) 2000,2001,2002,2003,2004 by theKompany.com
			   (C) 2001,2002,2003,2004 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>
#include	<stdarg.h>
#include	<time.h>
#include	<sys/types.h>

#include	<qarray.h>
#include	<qdict.h>
#include	<qstring.h>
#include	<qintdict.h>

#include	<libpq-fe.h>

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_databuffer.h"
#include	"kb_database.h"
#include	"kb_serverinfo.h"
#include	"kb_build.h"

#include	"kb_libloader.h"

#include	"kb_pgsql.h"
#include	"kb_pgadvanced.h"

static	PgSQLTypeMap	typeMap[] =
{
{	16,	KB::ITBool,	"Boolean",	0			},
{	17,	KB::ITBinary,	"Bytea",	0			},
{	18,	KB::ITString,	"Char",		FF_NOCREATE		},
{	19,	KB::ITString,	"Name",		FF_NOCREATE		},
{	20,	KB::ITFixed,	"Int8",		0			},
{	21,	KB::ITFixed,	"Int2",		0			},
{	22,	KB::ITDriver,	"Int2Vector",	FF_NOCREATE		},
{	23,	KB::ITFixed,	"Integer",	0			},
{	24,	KB::ITDriver,	"RegProc",	FF_NOCREATE		},
{	25,	KB::ITString,	"Text",		0			},
{	26,	KB::ITFixed,	"Oid",		0			},
{	28,	KB::ITDriver,	"Xid",		FF_NOCREATE		},
{	30,	KB::ITDriver,	"OidVector",	FF_NOCREATE		},
{	600,	KB::ITDriver,	"Point",	0			},
{	601,	KB::ITDriver,	"LSeg",		0			},
{	602,	KB::ITDriver,	"Path",		0			},
{	603,	KB::ITDriver,	"Box",		0			},
{	604,	KB::ITDriver,	"Polygon",	0			},
{	628,	KB::ITDriver,	"Line",		0			},
{	700,	KB::ITFloat,	"Float4",	0			},
{	701,	KB::ITFloat,	"Float8",	0			},
{	702,	KB::ITDriver,	"Abstime",	0			},
{	703,	KB::ITDriver,	"Reltime",	0			},
{	704,	KB::ITDriver,	"Interval",	0			},
{	718,	KB::ITDriver,	"Circle",	0			},
{	790,	KB::ITFloat,	"Money",	0			},
{	829,	KB::ITDriver,	"MacAddr",	0			},
{	869,	KB::ITDriver,	"INet",		0			},
{	650,	KB::ITDriver,	"Cidr",		0			},
{	1042,	KB::ITString,	"Char",		FF_LENGTH		},
{	1043,	KB::ITString,	"VarChar",	FF_LENGTH		},
{	1082,	KB::ITDate,	"Date",		0			},
{	1083,	KB::ITTime,	"Time",		0			},

#ifdef	__NOTDEF
{	1114,	KB::ITDateTime,	"DateTime",	0			},
{	1184,	KB::ITDateTime,	"DateTime",	FF_NOCREATE		},
{	1186,	KB::ITDriver,	"Interval",	FF_NOCREATE		},
#endif

{	1114,	KB::ITDateTime,	"TimeStamp",	0			},
{	1184,	KB::ITDateTime,	"TimeStamp",	FF_NOCREATE		},
{	1186,	KB::ITDriver,	"Interval",	0,			},
{	1700,	KB::ITFloat,	"Numeric",	FF_LENGTH|FF_PREC	},
}	;


/*  ------------------------------------------------------------------  */

/*  The following two routines are taken from the Postgres 7.4RC2 code	*/
/*  base. Added here (a) since we want to build for versions prior to	*/
/*  7.3 and (b) there is a possible allocate/release problem on windows	*/
/*  The license for this code is reproduced below.			*/

//  PostgreSQL Database Management System
//  (formerly known as Postgres, then as Postgres95)
//  
//  Portions Copyright (c) 1996-2003, The PostgreSQL Global Development Group
//  
//  Portions Copyright (c) 1994, The Regents of the University of California
//  
//  Permission to use, copy, modify, and distribute this software and its
//  documentation for any purpose, without fee, and without a written agreement
//  is hereby granted, provided that the above copyright notice and this
//  paragraph and the following two paragraphs appear in all copies.
//  
//  IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
//  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING
//  LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS
//  DOCUMENTATION, EVEN IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE
//  POSSIBILITY OF SUCH DAMAGE.
//  
//  THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
//  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
//  AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
//  ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATIONS TO
//  PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
//  


/*
 *		PQescapeBytea	- converts from binary string to the
 *		minimal encoding necessary to include the string in an SQL
 *		INSERT statement with a bytea type column as the target.
 *
 *		The following transformations are applied
 *		'\0' == ASCII  0 == \\000
 *		'\'' == ASCII 39 == \'
 *		'\\' == ASCII 92 == \\\\
 *		anything >= 0x80 ---> \\ooo (where ooo is an octal expression)
 */
unsigned char *
escapeBinary(const unsigned char *bintext, size_t binlen, size_t *bytealen)
{
	const unsigned char *vp;
	unsigned char *rp;
	unsigned char *result;
	size_t		i;
	size_t		len;

	/*
	 * empty string has 1 char ('\0')
	 */
	len = 1;

	vp = bintext;
	for (i = binlen; i > 0; i--, vp++)
	{
		if (*vp == 0 || *vp >= 0x80)
			len += 5;			/* '5' is for '\\ooo' */
		else if (*vp == '\'')
			len += 2;
		else if (*vp == '\\')
			len += 4;
		else
			len++;
	}

	rp = result = (unsigned char *) malloc(len);
	if (rp == NULL)
		return NULL;

	vp = bintext;
	*bytealen = len;

	for (i = binlen; i > 0; i--, vp++)
	{
		if (*vp == 0 || *vp >= 0x80)
		{
			(void) sprintf((char *)rp, "\\\\%03o", *vp);
			rp += 5;
		}
		else if (*vp == '\'')
		{
			rp[0] = '\\';
			rp[1] = '\'';
			rp += 2;
		}
		else if (*vp == '\\')
		{
			rp[0] = '\\';
			rp[1] = '\\';
			rp[2] = '\\';
			rp[3] = '\\';
			rp += 4;
		}
		else
			*rp++ = *vp;
	}
	*rp = '\0';

	return result;
}

#define ISFIRSTOCTDIGIT(CH) ((CH) >= '0' && (CH) <= '3')
#define ISOCTDIGIT(CH) ((CH) >= '0' && (CH) <= '7')
#define OCTVAL(CH) ((CH) - '0')

/*
 *		PQunescapeBytea - converts the null terminated string representation
 *		of a bytea, strtext, into binary, filling a buffer. It returns a
 *		pointer to the buffer (or NULL on error), and the size of the
 *		buffer in retbuflen. The pointer may subsequently be used as an
 *		argument to the function free(3). It is the reverse of PQescapeBytea.
 *
 *		The following transformations are made:
 *		\\	 == ASCII 92 == \
 *		\ooo == a byte whose value = ooo (ooo is an octal number)
 *		\x	 == x (x is any character not matched by the above transformations)
 */
unsigned char *
unescapeBinary(const unsigned char *strtext, size_t *retbuflen)
{
	size_t		strtextlen,
				buflen;
	unsigned char *buffer,
			   *tmpbuf;
	size_t		i,
				j;

	if (strtext == NULL)
		return NULL;

	strtextlen = strlen((const char *)strtext);
	/*
	 * Length of input is max length of output, but add one to avoid
	 * unportable malloc(0) if input is zero-length.
	 */
	buffer = (unsigned char *) malloc(strtextlen + 1);
	if (buffer == NULL)
		return NULL;

	for (i = j = 0; i < strtextlen; )
	{
		switch (strtext[i])
		{
			case '\\':
				i++;
				if (strtext[i] == '\\')
					buffer[j++] = strtext[i++];
				else
				{
					if ((ISFIRSTOCTDIGIT(strtext[i])) &&
						(ISOCTDIGIT(strtext[i + 1])) &&
						(ISOCTDIGIT(strtext[i + 2])))
					{
						int		byte;

						byte = OCTVAL(strtext[i++]);
						byte = (byte << 3) + OCTVAL(strtext[i++]);
						byte = (byte << 3) + OCTVAL(strtext[i++]);
						buffer[j++] = byte;
					}
				}
				/*
				 * Note: if we see '\' followed by something that isn't
				 * a recognized escape sequence, we loop around having
				 * done nothing except advance i.  Therefore the something
				 * will be emitted as ordinary data on the next cycle.
				 * Corner case: '\' at end of string will just be discarded.
				 */
				break;

			default:
				buffer[j++] = strtext[i++];
				break;
		}
	}
	buflen = j;					/* buflen is the length of the dequoted
								 * data */

	/* Shrink the buffer to be no larger than necessary */
	/* +1 avoids unportable behavior when buflen==0 */
	tmpbuf = (unsigned char *)realloc(buffer, buflen + 1);

	/* It would only be a very brain-dead realloc that could fail, but... */
	if (!tmpbuf)
	{
		free(buffer);
		return NULL;
	}

	*retbuflen = buflen;
	return tmpbuf;
}


/*  ------------------------------------------------------------------  */

/*  KBPgSqlType								*/
/*  KBPgSqlType	: Constructor for PostgreSQL type object		*/
/*  typeInfo	: PgSQLTypeMap * : Type information			*/
/*  length	: uint		 : Underlying database length		*/
/*  prec	: uint		 : Precision where applicable		*/
/*  nullOK	: bool		 : True if null is OK			*/
/*  (returns)	: KBPgSQLType	 :					*/

KBPgSQLType::KBPgSQLType
	(	PgSQLTypeMap	*typeInfo,
		uint	  	length,
		uint		prec,
		bool	  	nullOK
	)
	:
	KBType
	(	"PgSQL",
		typeInfo == 0 ? KB::ITUnknown : typeInfo->itype,
		length,
		prec,
		nullOK
	),
	m_typeInfo	(typeInfo)
{
}

/*  KBPgSQLType								*/
/*  isValid	: Test if value is valid for type			*/
/*  value	: const QString & : Value to check			*/
/*  pError	: KBError &	  : Error return			*/
/*  where	: const QString & : Caller				*/
/*  (returns)	: bool		  : Value				*/

bool	KBPgSQLType::isValid
	(	const QString	&value,
		KBError		&pError,
		const QString	&where
	)
{
	return	KBType::isValid (value, pError, where) ;
}

/*  KBPgSQLType								*/
/*  getQueryText							*/
/*  value	: KBDataArray  * : Raw text of value to convert		*/
/*  d		: KBShared     * : Decoded representation		*/
/*  buffer	: KBDataBuffer & : Results buffer			*/
/*  codec	: QTextCodec *	 : Non-default codec			*/
/*  (returns)	: QCString	 : Text					*/

void	KBPgSQLType::getQueryText
	(	KBDataArray	*value,
		KBShared	*d,
		KBDataBuffer	&buffer,
		QTextCodec	*codec
	)
{
	QCString res	= "" ;

//	fprintf
//	(	stderr,
//		"KBPgSQLType::getQueryText: iType=%d\n",
//		iType
//	)	;

	if (value != 0)
		switch (m_iType)
		{
			case KB::ITBinary :
			case KB::ITDriver :
			{
				size_t	tolen	;
				uchar	*b	= escapeBinary
						  (	(uchar *)value->m_data,
						  	value->m_length,
							&tolen
						  )	;
				buffer.append ("'")	;
				buffer.append ((cchar *)b, tolen - 1) ;
				buffer.append ("'")	;
				free	      (b) ;
			}
			return	;

			case KB::ITBool :
			{
				bool	v ;

//				fprintf
//				(	stderr,
//					"KBPgSQLType::getQueryText: KB::ITBool [%s]\n",
//					buffer.data()
//				)	;

				if	(qstricmp (value->m_data, "yes"  ) == 0) v = true  ;
				else if (qstricmp (value->m_data, "true" ) == 0) v = true  ;
				else if (qstricmp (value->m_data, "t"    ) == 0) v = true  ;
				else if	(qstricmp (value->m_data, "no"   ) == 0) v = false ;
				else if (qstricmp (value->m_data, "false") == 0) v = false ;
				else if (qstricmp (value->m_data, "f"    ) == 0) v = false ;
				else						 v = atoi(value->m_data) != 0 ;

//				KBType::getQueryText(value, d, buffer) ;
//				buffer.append ("::bool") ;

				buffer.append (v ? "'t'" : "'f'") ;

//				fprintf
//				(	stderr,
//					"KBPgSQLType::getQueryText: KB::ITBool [%s]\n",
//					buffer.data()
//				)	;
			}
			return	;

			default	:
				break	;
		}

	KBType::getQueryText (value, d, buffer, codec) ;
}

/*  getFieldTypes: Get type array corresponding to query results	*/
/*  pgQry	: PGresult *	: Query results				*/
/*  (returns)	: KBType **	: Type array				*/

static	KBType **getFieldTypes
	(	PGresult	*pgQry
	)
{
	int	nFields = PQnfields (pgQry) ;
	KBType	**types = new KBType *[nFields] ;

	for (int idx = 0 ; idx < nFields ; idx += 1)
	{
		Oid	  	oid	= PQftype  (pgQry, idx) ;
		int		length	= PQfsize  (pgQry, idx) ;
		int		prec	= 0 ;
		PgSQLTypeMap	*map	= dOidToType.find (oid) ;

		if	(length >= 0)
		{
			if ((length & 0x7fff0000) != 0)
			{
				prec	= length & 0x0000ffff ;
				length	= length >> 16 ;
			}
		}
		else if (oid == 17)
		{
			length	= 0x7fffffff ;
		}
		else if (oid == 25)
		{
			length	= 0x7fffffff ;
		}
		else	length	= 0 ;

		types[idx] = new KBPgSQLType (map, length, prec, true)  ;
	}

	return	types	;
}


/*  ------------------------------------------------------------------  */

/*  KBPgSQL								*/
/*  KBPgSQL	: Constructor for PostgreSQL database connection class	*/
/*  (returns)	: KBServer	:					*/

KBPgSQL::KBPgSQL ()
	:
	KBServer ()
{
	m_pgConn 	= 0 ;
}

/*  KBPgSQL								*/
/*  ~KBPgSQL	: Destructor for PostgresQL database connection class	*/
/*  (returns)	:		:					*/

KBPgSQL::~KBPgSQL ()
{
	if (m_pgConn != 0) PQfinish (m_pgConn) ;
}

/*  KBPgSQL								*/
/*  doConnect	: Open connection to database				*/
/*  svInfo	: KBServerInfo *: Server information			*/
/*  (returns)	: bool		: Success				*/

bool	KBPgSQL::doConnect
	(	KBServerInfo	*svInfo
	)
{
	m_readOnly	= svInfo->readOnly  () ;
	m_socket 	= svInfo->socketName() ;
	m_flags	 	= svInfo->flags     () ;

	if (svInfo->advanced() != 0)
	{
		if (svInfo->advanced()->isType("pgsql"))
		{
			KBPgAdvanced *a = (KBPgAdvanced *)svInfo->advanced() ;

			m_primaryIsSerial  = a->m_primaryIsSerial	;
			m_ignoreUser       = a->m_ignoreUser		;
			m_showPgSQLObjects = a->m_showPgSQLObjects	;
			m_logInternal	   = a->m_logInternal		;
			m_requireSSL	   = a->m_requireSSL		;
			m_caseInsensitive  = a->m_caseInsensitive	;
			m_mapExpressions   = a->m_mapExpressions	;
			m_useTimeouts	   = a->m_useTimeouts		;
			m_stmtTimeout	   = a->m_stmtTimeout		;
			m_lockTimeout	   = a->m_lockTimeout		;

			m_grants	   = a->m_grants		;
			m_grantSelect	   = a->m_grantSelect		;
			m_grantInsert	   = a->m_grantInsert		;
			m_grantUpdate	   = a->m_grantUpdate		;
			m_grantDelete	   = a->m_grantDelete		;
			m_grantTo	   = a->m_grantTo		;
			m_grantPopup	   = a->m_grantPopup		;
		}
		else	/* If the type is wrong then show a warning.	*/
			/* This should never happen unless someone	*/
			/* hand edits the XML database file.		*/
			KBError::EError
			(	TR("Driver error"),
				TR("Invalid advanced options, ignoring"),
				__ERRLOCN
			)	;
	}
	else
	{
		m_primaryIsSerial  = false ;
		m_ignoreUser       = false ;
		m_showPgSQLObjects = false ;
		m_logInternal	   = false ;
		m_requireSSL	   = false ;
		m_caseInsensitive  = false ;
		m_mapExpressions   = false ;
	}

	/* Check that we are not already connected. Note that the	*/
	/* notConnected routine sets an error message.			*/
	if (m_pgConn != 0)
	{	m_lError = KBError
			   (	KBError::Error,
				"Already connected to PostgreSQL database",
				QString::null,
				__ERRLOCN
			   ) 	;
		return	false ;
	}

	QString	conninfo	;
	QString	host		= m_host.stripWhiteSpace() ;
	QString	port		= m_port.stripWhiteSpace() ;

#ifndef	_WIN32
	/* The driver supports SSH tunneling. If there is a tunnel	*/
	/* target then attempt to open the tunnel; if this succeeds	*/
	/* then the host becomes the local host and the port is that	*/
	/* returned by the tunneler.					*/
	fprintf
	(	stderr,
		"KBPgSQL::doConnect: sshTarget=[%s]\n",
		(cchar *)m_sshTarget
	)	;

	if (!m_sshTarget.isEmpty())
	{
		int local = openSSHTunnel(5432) ;
		if (local < 0) return false ;

		host	= "127.0.0.1" ;
		port	= QString("%1").arg(local) ;
	}
#endif

	if (!host.isEmpty())
		if (m_host[0].isDigit())
			conninfo += QString(" hostaddr='%1'").arg(host) ;
		else	conninfo += QString(" host='%1'"    ).arg(host) ;

	if (!port.isEmpty())
		conninfo += QString(" port='%1'").arg(port) ;

	/* If the database name is empty then connect to the default	*/
	/* "postgres" database. In practice this is used when getting	*/
	/* a list of databases.						*/
	QString	database = m_database	;
	QString	user	 = m_user	;
	QString	password = m_password	;
	if (database.isEmpty())
	{
		database = "template1"	;
//		if (user.isEmpty()) user = "postgres" ;
	}

	if (!database.isEmpty())
		conninfo += QString(" dbname='%1'"  ).arg(database.stripWhiteSpace()) ;
	if (!user    .isEmpty())
		conninfo += QString(" user='%1'"    ).arg(user    .stripWhiteSpace()) ;
	if (!password.isEmpty())
		conninfo += QString(" password='%1'").arg(password.stripWhiteSpace()) ;

	if (m_requireSSL)
		conninfo += " requiressl=1" ;

	fprintf	(stderr, "KBPgSQL::doConnect: [%s]\n", (cchar *)conninfo) ;

//	m_pgConn = PQsetdbLogin
//		   (	m_host,
//			m_port,
//			0,
//			"/tmp/kbase.pgsql",
//			m_database,
//			m_user,
//			m_password
//		   )	;

	m_pgConn = PQconnectdb (conninfo) ;

	if (PQstatus (m_pgConn) == CONNECTION_BAD)
	{
		m_lError = KBError
			   (	KBError::Error,
				"Unable to connect to PostgreSQL server",
				PQerrorMessage (m_pgConn),
				__ERRLOCN
			   ) 	;
		return	false	;
	}

	PQexec (m_pgConn, "set datestyle to 'iso'") ;

	KBError	dummy	;
	setStmtTimeout	(dummy) ;

	m__conn		= true	;
	m_activeCookie	= 0	;
	return	true	;
}

/*  KBPgSQL								*/
/*  qrySelect	: Open a select query					*/
/*  data	: bool		: Querying for data			*/
/*  select	: const QString&: Select query				*/
/*  update	: bool		: Select for update			*/
/*  (returns)	: KBSQLSelect * : Select query class or NULL on error	*/

KBSQLSelect
	*KBPgSQL::qrySelect
	(	bool		data,
		const QString	&select,
		bool		update
	)
{
	return	new KBPgSQLQrySelect
		(	this,
			data,
			select,
			update
		)	;
}


/*  KBPgSQL								*/
/*  qryUpdate	: Open an update query					*/
/*  data	: bool		 : Querying for data			*/
/*  update	: const QString &: Update query				*/
/*  tabName	: const QString &: Table name				*/
/*  (returns)	: KNQryUpdate *  : Update query class or NULL on error	*/

KBSQLUpdate
	*KBPgSQL::qryUpdate
	(	bool		data,
		const QString	&update,
		const QString	&tabName
	)
{
	if (m_readOnly)
	{
		m_lError = KBError
			   (	KBError::Error,
			 	TR("Database is read-only"),
			 	TR("Attempting update query"),
			 	__ERRLOCN
			  )	;
		return	 0 ;
	}

	return	new KBPgSQLQryUpdate
		(	this,
			data,
			update,
			tabName
		)	;
}

/*  KBPgSQL								*/
/*  qryInsert	: Open an insert query					*/
/*  data	: bool		 : Querying for data			*/
/*  insert	: const QString &: Insert query				*/
/*  tabName	: const QString &: Table name				*/
/*  (returns)	: KNQryInsert *  : Insert query class or NULL on error	*/

KBSQLInsert
	*KBPgSQL::qryInsert
	(	bool		data,
		const QString	&insert,
		const QString	&tabName
	)
{
	if (m_readOnly)
	{
		m_lError = KBError
			   (	KBError::Error,
			 	TR("Database is read-only"),
			 	TR("Attempting insert query"),
			 	__ERRLOCN
			  )	;
		return	 0 ;
	}

	return	new KBPgSQLQryInsert
		(	this,
			data,
			insert,
			tabName
		)	;
}

/*  KBPgSQL								*/
/*  qryDelete	: Open an delete query					*/
/*  data	: bool		 : Querying for data			*/
/*  _delete	: const QString& : Delete query				*/
/*  tabName	: const QString &: Table name				*/
/*  (returns)	: KNQryDelete *  : Delete query class or NULL on error	*/

KBSQLDelete
	*KBPgSQL::qryDelete
	(	bool		data,
		const QString	&_delete,
		const QString	&tabName
	)
{
	if (m_readOnly)
	{
		m_lError = KBError
			   (	KBError::Error,
			 	TR("Database is read-only"),
			 	TR("Attempting delete query"),
			 	__ERRLOCN
			  )	;
		return	 0 ;
	}

	return	new KBPgSQLQryDelete
		(	this,
			data,
			_delete,
			tabName
		)	;
}

/*  KBPgSQL								*/
/*  transaction	: Server transaction request				*/
/*  op		: Transaction	: Specific operation			*/
/*  activeCookie: void **	: Pass/return active cookie		*/
/*  (returns)	: bool		: Success				*/

bool	KBPgSQL::transaction
	(	Transaction	op,
		void		**activeCookie
	)
{
	switch (op)
	{
		case BeginTransaction	:
			/* If we are being passed an active transaction	*/
			/* cookie *and* there is one in progress, then	*/
			/* pass back the in-progess cookie and return	*/
			/* an error. The caller can use this to figure	*/
			/* out what to abort or to warn about.		*/
			if ((activeCookie != 0) && (m_activeCookie != 0))
			{
				*activeCookie	= m_activeCookie ;
				m_lError	= KBError
						  (	KBError::Warning,
						  	TR("Transaction already in progress"),
						  	QString::null,
						  	__ERRLOCN
						  )	;
				return	false	;
			}
			

			if (!execSQL
				(	"begin",
					TR("Error starting transaction"),
					PGRES_COMMAND_OK,
					true
				))
				return	false	;

			/* If passed an active transaction cookie then	*/
			/* this is noted against future transactions.	*/
			if (activeCookie != 0)
				m_activeCookie	= *activeCookie ;

			return	true	;


		case CommitTransaction	:
			/* If passed an active transaction cookie then	*/
			/* clear it. Probably not used but behave in a	*/
			/* sensible way in case.			*/
			if (activeCookie != 0) *activeCookie = 0 ;
			m_activeCookie = 0 ;

			return	execSQL
				(	"commit",
					TR("Error committing work"),
					PGRES_COMMAND_OK,
					true
				)	;

		case RollbackTransaction:
			/* As above ....				*/
			if (activeCookie != 0) *activeCookie = 0 ;
			m_activeCookie = 0 ;

			return	execSQL
				(	"rollback",
					TR("Error rolling back work"),
					PGRES_COMMAND_OK,
					true
				)	;

		default	:
			break	;
	}

	m_lError = KBError
		   (	KBError::Fault,
			TR("Unknown driver transaction operation"),
			QString(TR("Code: %1")).arg(op),
			__ERRLOCN
		   )	;
	return	false	;
}

/*  KBPgSQL								*/
/*  qryCursor	: Create cursor						*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Query text				*/
/*  cursor	: const QString & : Cursor name				*/
/*  (returns)	: bool		  : Success				*/

KBSQLCursor
	*KBPgSQL::qryCursor
	(	bool		data,
		const QString	&query,
		const QString	&cursor
	)
{
	return	new KBPgSQLQryCursor (this, data, query, cursor) ;
}

/*  KBPgSQL								*/
/*  command	: Execute arbitrary SQL					*/
/*  data	: bool		  : Querying for data			*/
/*  rawSql	: const QString & : Query text				*/
/*  nvals	: uint		  : Number of substitution values	*/
/*  values	: KBValue *	  : Substitution values			*/
/*  select	: KBSQLSelect **  : Return for data queries		*/
/*  (returns)	: bool		  : Success				*/

bool	KBPgSQL::command
	(	bool		data,
		const QString	&rawSql,
		uint		nvals,
		KBValue		*values,
		KBSQLSelect	**
	)
{
	KBDataBuffer	exeSql	;
	PGresult 	*pgQry	;

	if (!subPlaceList (rawSql, nvals, values, exeSql, getCodec(true), m_lError))
		return 0 ;


	if ((pgQry = PQexec (m_pgConn, exeSql.data())) == 0)
	{
		fprintf
		(	stderr,
			"KBPgSQL::command: failed: PQexec returned null"
		)	;

		m_lError  = KBError
			  (	KBError::Error,
				TR("Command execution failed"),
				exeSql.data(),
				__ERRLOCN
			  )	;
		return	false	;
	}

	if (PQresultStatus (pgQry) == PGRES_COMMAND_OK)
	{
		PQclear (pgQry) ; 
		return	true	;
	}

	if (PQresultStatus (pgQry) == PGRES_TUPLES_OK )
	{
		PQclear (pgQry) ; 
		return	true	;
	}


	fprintf
	(	stderr,
		"KBPgSQL::command: failed: PQexec returned code %d",
		PQresultStatus (pgQry)
	)	;

	m_lError  = KBError
		  (	KBError::Error,
			TR("Command execution returned unknown code"),
			QString(TR("Code: %1\n%2")).arg(PQresultStatus(pgQry)).arg(exeSql.data()),
			__ERRLOCN
		  )	;

	PQclear (pgQry) ; 
	return	false	;
}


/*  KBPgSQL								*/
/*  execSQL	: Execute simple SQL statement				*/
/*  rawSql	: const QString &  : Raw SQL statement			*/
/*  subSql	: QString &	   : Substituted statement for logging	*/
/*  nvals	: uint		   : Number of execution values		*/
/*  values	: KBValue *	   : Execution values			*/
/*  code	: QTextCodec *	   : Non-default codec			*/
/*  emsg	: const QString &  : Failure error message		*/
/*  ok		: ExecStatusType   : OK return code			*/
/*  pError	: KBError &	   : Error location			*/
/*  userQuery	: bool		   : Query is a user query		*/
/*  (returns)	: PGresult *	   : Result block or null on error	*/

PGresult*KBPgSQL::execSQL
	(	const QString	&rawSql,
		QString		&subSql,
		uint		nvals,
		const KBValue	*values,
		QTextCodec	*codec,
		const QString	&emsg,
		ExecStatusType	ok,
		KBError		&pError,
		bool		userQuery
	)
{
	KBDataBuffer	exeSql	;

	if (!subPlaceList (rawSql, nvals, values, exeSql, codec, pError))
		return 0 ;

	subSql = subPlaceList (rawSql, nvals, values, pError) ;
	if (subSql.isNull())
		return 0 ;

//	fprintf
//	(	stderr,
//		"KBPgSQL::execSQL:\n"
//		"          raw=%s\n"
//		"          exe=%s\n"
//		"          sub=%s\n",
//		(cchar *)rawSql,
//		exeSql.data(),
//		(cchar *)subSql
//	)	;

	PGresult *pgQry = PQexec (m_pgConn, (exeSql.data())) ;
	if ((pgQry == 0) || (PQresultStatus (pgQry) != ok))
	{
		fprintf
		(	stderr,
			"          failed: %s\n",
			PQresultErrorMessage (pgQry)
		)	;

		pError	= KBError
			  (	KBError::Error,
				emsg,
				QString ("%1\n%2").arg(subSql).arg(PQresultErrorMessage (pgQry)),
				__ERRLOCN
			  )	;
		if (pgQry != 0) PQclear (pgQry) ; 
		pgQry	= 0 ;
	}

	if (userQuery || m_logInternal)
		printQuery (subSql, nvals, values, pgQry != 0) ;

	return	pgQry ;
}


/*  KBPgSQL								*/
/*  execSQL	: Execute simple SQL statement				*/
/*  sql		: const QString &  : SQL statement			*/
/*  emsg	: const QString &  : Failure error message		*/
/*  ok		: ExecStatusType   : OK return code			*/
/*  userQuery	: bool		   : Query is a user query		*/
/*  (returns)	: bool		   : Success				*/

bool	KBPgSQL::execSQL
	(	const QString	&sql,
		const QString	&emsg,
		ExecStatusType	ok,
		bool		userQuery
	)
{
	bool	 rc	= true ;
	PGresult *pgQry = PQexec (m_pgConn, (cchar *)sql) ;

	if ((pgQry == 0) || (PQresultStatus (pgQry) != ok))
	{
		rc	 = false;
		m_lError = KBError
			   (	KBError::Error,
				emsg,
				QString ("%1\n%2").arg(sql).arg(PQresultErrorMessage (pgQry)),
				__ERRLOCN
			   )	;
	}

	if (pgQry != 0)
		PQclear (pgQry) ; 

	if (userQuery || m_logInternal)
		printQuery (sql, 0, 0, rc) ;

	return	rc ;
}


/*  KBPgSQL								*/
/*  objectExists: See if named object exists				*/
/*  object	: const QString & : Object name				*/
/*  relkind	: cchar *	  : Required relation kind		*/
/*  exists	: bool &	  : True if object exists		*/
/*  (returns)	: bool		  : Success				*/

bool	KBPgSQL::objectExists
	(	const QString	&table,
		cchar		*relkind,
		bool		&exists
	)
{
	PGresult *pgQry ;
	QString	 query	;
	QString	 dummy	;

	query	= QString
		  (	"select relname "
			"from   pg_class, pg_user "
			"where  pg_user.usesysid = pg_class.relowner "
			"and    relname          = '%1' "
			"and    pg_class.relkind = '%2' "
		  )
		  .arg(m_mapExpressions ? table : table.lower())
		  .arg(relkind) ;

	if (!m_ignoreUser)
		query	+= QString("and    pg_user.usename  = '%3' ")
				   .arg(m_user) ;

//	fprintf
//	(	stderr,
//		"KBPgSQL::objectExists: [%s]\n",
//		(cchar *)query
//	)	;

	if ((pgQry = execSQL (query,
			      dummy,
			      0,
			      0,
			      0,
			      "Error verifying object existance",
			      PGRES_TUPLES_OK,
			      m_lError)) == 0) return false ;

	exists	= PQntuples (pgQry) == 1 ;
	PQclear (pgQry) ;
	return	true	;
}

bool	KBPgSQL::listDatabases
	(	QStringList	&dbList
	)
{

	QString	 subqry ;
	PGresult *pgQry ;

	if ((pgQry = execSQL
		(	"select pg_database.datname	"
			"from	pg_database		"
			"order	by pg_database.datname	",
			subqry,
			0,
			0,
			0,
			"List databases query failed",
			PGRES_TUPLES_OK,
			m_lError,
		     	true
		     )) == 0) return false ;

	for (int idx1 = 0 ; idx1 < PQntuples (pgQry) ; idx1 += 1)
		dbList.append (PQgetvalue (pgQry, idx1, 0)) ;

	return	true	;
}

/*  KBPgSQL								*/
/*  listTypes	: Get list of types with information flags		*/
/*  (returns)	: QString	: List as bar-separated string		*/

QString	KBPgSQL::listTypes ()
{
	static	QString	typeList ;

	if (typeList.isNull())
	{
		typeList = "Primary Key,0|Foreign Key,0" ;

		for (uint idx = 0 ; idx < sizeof(typeMap)/sizeof(PgSQLTypeMap) ; idx += 1)
		{
			PgSQLTypeMap *m = &typeMap[idx] ;

			if ((m->flags & FF_NOCREATE) == 0)
				typeList += QString("|%1,%2").arg(m->ptype).arg(m->flags) ;
		}
	}

	return	typeList ;
}

/*  KBPgSQL								*/
/*  tableExists	: See if named table exists				*/
/*  table	: const QString & : Table name				*/
/*  exists	: bool &	  : True if table exists		*/
/*  (returns)	: bool		  : Success				*/

bool	KBPgSQL::tableExists
	(	const QString	&table,
		bool		&exists
	)
{
	return	objectExists (table, "r", exists) ;
}

/*  KBPgSQL								*/
/*  viewExists	: See if named view exists				*/
/*  view	: const QString & : View name				*/
/*  exists	: bool &	  : True if view exists			*/
/*  (returns)	: bool		  : Success				*/

bool	KBPgSQL::viewExists
	(	const QString	&view,
		bool		&exists
	)
{
	return	objectExists (view, "v", exists) ;
}

/*  KBPgSQL								*/
/*  sequenceExists							*/
/*		: See if named sequence exists				*/
/*  sequence	: const QString & : Sequence name			*/
/*  exists	: bool &	  : True if table exists		*/
/*  (returns)	: bool		  : Success				*/

bool	KBPgSQL::sequenceExists
	(	const QString	&sequence,
		bool		&exists
	)
{
	return	objectExists (sequence, "S", exists) ;
}

bool	KBPgSQL::listForType
	(	KBTableDetailsList	&tabList,
		const QString		&query,
		KB::TableType		type,
		uint			permissions
	)
{
	PGresult *pgQry ;
	QString	dummy	;

	if ((pgQry = execSQL
		     (		query,
		     		dummy,
				0,
				0,
				0,
				TR("Error getting list of database objects"),
				PGRES_TUPLES_OK,
				m_lError
	   	     )) == 0)	return false ;

	for (int idx1 = 0 ; idx1 < PQntuples (pgQry) ; idx1 += 1)
	{
		QString	name	= PQgetvalue (pgQry, idx1, 0) ;

		if (!m_showAllTables)
			if (name.left(8) == "__rekall")
				continue ;

		if (!m_showPgSQLObjects)
			if (name.left(3) == "pg_")
				continue ;

		tabList.append (KBTableDetails (name, type, permissions)) ;
	}

	PQclear (pgQry) ;
	return	true	;
}

/*  KBPgSQL								*/
/*  doListTables: List tables in database				*/
/*  tabList	: KBTableDetailsList &					*/
/*				: Result list				*/
/*  yype	: uint		: Type flags				*/
/*  (returns)	: bool		: Success				*/

bool	KBPgSQL::doListTables
	(	KBTableDetailsList	&tabList,
		uint			type
	)
{
	QString	 query	;

	if ((type & KB::IsTable) != 0)
	{
		query	= "select tablename from pg_tables " ;

		if (!m_ignoreUser)
			query	+= QString("where tableowner = '%1' ").arg(m_user) ;

		query  += "order by tablename" ;

		if (!listForType (tabList, query, KB::IsTable, QP_SELECT|QP_INSERT|QP_UPDATE|QP_DELETE))
			return	false	;
	}

	if ((type & KB::IsView) != 0)
	{
		query	= "select viewname from pg_views " ;

		if (!m_ignoreUser)
			query	+= QString("where viewowner = '%1' ").arg(m_user) ;

		query  += "order by viewname" ;

		if (!listForType (tabList, query, KB::IsView,  QP_SELECT))
			return	false	;
	}

	if ((type & KB::IsSequence) != 0)
	{
		query	= "select relname from pg_class where relkind = 'S'::\"char\" " ;

		if (!m_ignoreUser)
			query	+= QString("and pg_get_userbyid(relowner) = '%1' ").arg(m_user) ;

		query  += "order by relname" ;

		if (!listForType (tabList, query, KB::IsSequence,  QP_SELECT))
			return	false	;
	}


	return	true	;
}


/*  KBPgSQL								*/
/*  doListFields: List fields in table					*/
/*  tabSpec	: KBTableSpec &	: Table specification			*/
/*  (returns)	: bool		: Success				*/

bool	KBPgSQL::doListFields
	(	KBTableSpec	&tabSpec
	)
{
	PGresult 	*pgQry  ;
	QString	 	dummy	;
	int	 	unique	;
	QString	 	pkName	;
	QString	 	tabName	= tabSpec.m_name;

	tabSpec.m_prefKey   = -1    		;
	tabSpec.m_keepsCase = m_mapExpressions	;
	unique		    = -1    		;

	if (!m_mapExpressions) tabName = tabName.lower() ;

#if	0
	if ((pgQry = execSQL (QString (	"select	relkind 	"
					"from	pg_class	"
					"where relname = '%1'	"
				      )
				      .arg(tabName),
			      dummy,
			      0,
			      0,
			      0,
			      QString::null,
			      PGRES_TUPLES_OK,
			      m_lError)) != 0)
	{
		cchar	*v = PQgetvalue (pgQry, 0, 0) ;
		if ((v != 0) && (v[0] == 'v'))
			iaflag	= 0 ;
		PQclear (pgQry) ;

//		fprintf
//		(	stderr,
//			"KBPgSQL::doListFields: [%s] iaflag=%d\n",
//			(cchar *)tabName,
//			iaflag
//		)	;
	}
#endif

	/* First stage is to get the column and column information. The	*/
	/* query is figured from the PostgreSQL sources and		*/
	/* documentation, and may not be optimal.			*/
	if ((pgQry = execSQL (QString ( "select attname, atttypid, attlen, attnotnull, attnum, atttypmod "
					"from   pg_attribute, pg_class  "
					"where  pg_attribute.attrelid = pg_class.oid "
					"and    pg_attribute.attnum > 0 "
					"and    pg_class.relname = '%1' "
					"and	attname not ilike '%pg.dropped%' "
					"order by pg_attribute.attnum"
				      )
				      . arg(tabName),
			      dummy,
			      0,
			      0,
			      0,
			      "Error getting list of fields in table",
			      PGRES_TUPLES_OK,
			      m_lError)) == 0) return false ;

	for (int idx1 = 0 ; idx1 < PQntuples (pgQry) ; idx1 += 1)
	{
		cchar		*coid	= PQgetvalue (pgQry, idx1, 1)	;
		PgSQLTypeMap	*ptr	= dOidToType.find (atoi(coid))	;

		bool		nn	= PQgetvalue(pgQry, idx1, 3)[0] == 't' ; 
		int		length	= atoi (PQgetvalue (pgQry, idx1, 5)) - 4 ;
		int		prec	= 0 ;

		QString		ptype	;
		KB::IType	itype	;

		if (ptr != 0)
		{
			ptype	= ptr->ptype	;
			itype	= ptr->itype	;
		}
		else
		{
			ptype	= QString("<Unknown %1>").arg(coid) ;
			itype	= KB::ITUnknown	;
		}

		if	(length >= 0)
		{
			if ((length & 0x7fff0000) != 0)
			{
				prec	= length & 0x0000ffff ;
				length	= length >> 16 ;
			}
		}
		else if ((ptr != 0) && (ptr->oid == 17))
		{
			length	= 0x7fffffff ;
		}
		else if ((ptr != 0) && (ptr->oid == 25))
		{
			length	= 0x7fffffff ;
		}
		else	length	= 0 ;

		KBFieldSpec *fSpec = new KBFieldSpec
				     (		(uint)idx1,
						PQgetvalue (pgQry, idx1, 0),
						ptype,
						itype,
						nn ? KBFieldSpec::InsAvail|KBFieldSpec::NotNull : KBFieldSpec::InsAvail,
						length,
						prec
				     ) ;

		fSpec->m_dbType	   = new KBPgSQLType (ptr, length, prec, !nn) ;

		tabSpec.m_fldList.append (fSpec) ;

//		fprintf	(stderr, "%12s %8s %8s %8s %3s\n",
//				 PQgetvalue (pgQry, idx1, 0),
//				 PQgetvalue (pgQry, idx1, 1),
//				 PQgetvalue (pgQry, idx1, 2),
//				 PQgetvalue (pgQry, idx1, 3),
//				 PQgetvalue (pgQry, idx1, 4)) ;
	}

	PQclear (pgQry) ;


	/* Now look for indexing and primary key information, Again it	*/
	/* is extracted from PostreSQL's internal tables.		*/
	if ((pgQry = execSQL (QString ( "select pg_index.indkey[0], pg_index.indisunique, pg_index.indisprimary "
					"from   pg_index, pg_class "
					"where  pg_index.indrelid = pg_class.oid "
					"and    pg_class.relname  = '%1'"
				      )
				      .arg (tabName),
			      dummy,
			      0,
			      0,
			      0,
			      "Error determining table key information",
			      PGRES_TUPLES_OK,
			      m_lError)) == 0) return false ;

	for (int idx2 = 0 ; idx2 < PQntuples (pgQry) ; idx2 += 1)
	{
		int kcol = atoi(PQgetvalue (pgQry, idx2, 0)) - 1 ;
		if ((kcol < 0) || (kcol >= (int)tabSpec.m_fldList.count()))
			continue ;

		KBFieldSpec *fSpec = tabSpec.m_fldList.at (kcol) ;

		fSpec->m_flags |= KBFieldSpec::Indexed ;

		if (PQgetvalue(pgQry, idx2, 1)[0] == 't')
		{	fSpec->m_flags |= KBFieldSpec::Unique  ;
			if (unique < 0) unique = kcol ;
		}

		if (PQgetvalue(pgQry, idx2, 2)[0] == 't')
		{	fSpec->m_flags |= KBFieldSpec::Primary ;
			tabSpec.m_prefKey = kcol ;
			pkName = fSpec->m_name ;
		}
	}


	/* If the preferred key is set then we have found a primary key	*/
	/* column. In this case, see if there is a sequence named as	*/
	/* the table with "_seq" appended. If so we presume it is a	*/
	/* rekall-generated sequence, and we will need to pre-load	*/
	/* values.							*/
	if (tabSpec.m_prefKey >= 0)
	{
		bool	exists	;

		if (!objectExists (tabSpec.m_name + "_seq", "S", exists))
			return false ;

		if (exists)
		{
			KBFieldSpec *pSpec = tabSpec.m_fldList.at(tabSpec.m_prefKey) ;
			pSpec->m_typeName  = "Primary Key"	;
			pSpec->m_flags    |= KBFieldSpec::Serial;
		}
	}
	else	tabSpec.m_prefKey = unique ;



	PQclear (pgQry) ;

	/* Thirdly, see if any columns have default expressions. We	*/
	/* have to find out about these since we cannot insert or	*/
	/* update with a null value to get the default.			*/
	if ((pgQry = execSQL (QString (	"select pg_attribute.attname, pg_attrdef.adsrc "
					"from   pg_class, pg_attribute, pg_attrdef "
					"where  pg_attribute.attrelid = pg_class.oid "
					"and    pg_attribute.attrelid = pg_attrdef.adrelid "
					"and    pg_attribute.attnum   = pg_attrdef.adnum "
					"and    pg_class.relname      = '%1' "
		  		      )
				      .arg (tabName),
			      dummy,
			      0,
			      0,
			      0,
			      "Error finding column defaults",
			      PGRES_TUPLES_OK,
			      m_lError)) == 0) return false ;

	int	nDefs	= PQntuples (pgQry) ;

	for (int def = 0 ; def < nDefs ; def += 1)
		LITER
		(	KBFieldSpec,
			tabSpec.m_fldList,
			fSpec,

			if (fSpec->m_name != PQgetvalue (pgQry, def, 0))
				continue ;

			fSpec->m_defval = PQgetvalue (pgQry, def, 1) ;

			if ((fSpec->m_flags & KBFieldSpec::Primary) != 0)
				fSpec->m_flags |= KBFieldSpec::Serial|KBFieldSpec::ReadOnly   ;

			break ;
		)	;

	PQclear (pgQry) ;

	if ((pgQry = execSQL (QString ( "select definition	"
					"from	pg_views	"
					"where	viewname = '%1'	"
				      )
				      .arg (tabName),
			      dummy,
			      0,
			      0,
			      0,
			      QString::null,
			      PGRES_TUPLES_OK,
			      m_lError)) != 0)
	{
		if (PQntuples (pgQry) > 0)
		{
			tabSpec.m_type	= KB::IsView		   ;
			tabSpec.m_view	= PQgetvalue (pgQry, 0, 0) ;
		}
	
		PQclear (pgQry) ;
	}

#if	0
	LITER
	(	KBFieldSpec,
		tabSpec.fldList,
		fSpec,

		fSpec->Printf (stderr) ;
		fputs ("\n",   stderr) ;
	)
#endif
	return	true	;
}

/*  KBPgSQL								*/
/*  tblCreateSQL: Generate SQL for a table create statement		*/
/*  fldList	: QList<KBFieldSpec>&: Field specification		*/
/*  table	: const QString &    : Table name			*/
/*  create	: QString &	     : SQL text				*/
/*  best	: bool		     : Use best column match		*/
/*  (returns)	: bool		     : Success				*/

bool	KBPgSQL::tblCreateSQL
	(	QList<KBFieldSpec>	&fldList,
		const QString		&table,
		QString			&create,
		bool			best
	)
{
	KBFieldSpec	*fSpec	;
	PgSQLTypeMap	*mapp	;
	cchar		*sep	= ""	;
	cchar		*quote	= m_mapExpressions ? "\"" : "" ;

	create	= QString ("create table %1%2%3\n(")
				.arg(quote)
				.arg(table)
				.arg(quote) ;

	for (uint col = 0 ; col < fldList.count() ; col += 1)
	{
		fSpec	= fldList.at (col) ;
		mapp	= 0	;

		/* Special case. If the field is "Primary Key" then we	*/
		/* create an "Int4" column marked not null and primary	*/
		/* key (and create a sequence below) ...		*/
		QString		ftype	= fSpec->m_typeName ;
		KB::IType	itype	= fSpec->m_typeIntl ;

		if (ftype == "Primary Key")
		{
			cchar	*type	= m_primaryIsSerial ? "serial" : "int4" ;

			create += QString ("%1\t%2%3%4 %5 not null primary key")
					  .arg(sep	    )
					  .arg(quote	    )
					  .arg(fSpec->m_name)
					  .arg(quote	    )
					  .arg(type	    ) ;
			sep	= ",\n"	;
			continue   ;
		}
		/* ... while a foreign key is also "Int4" not null.	*/
		if (ftype == "Foreign Key")
		{
			create += QString ("%1\t%2%3%4 int not null")
					  .arg(sep	    )
					  .arg(quote	    )
					  .arg(fSpec->m_name)
					  .arg(quote	    ) ;
			sep	= ",\n"	;
			continue   ;
		}

		/* Map the types used when creating the object and	*/
		/* design dictionary tables.				*/
		if	(ftype == "_Text"   ) ftype = "Text"    ;
		else if (ftype == "_Integer") ftype = "Integer" ;
		else if (ftype == "_Binary" ) ftype = "Bytea"	;

		/* Scan though the mapping table looking for the type	*/
		/* type ....						*/
		for (uint typ = 0 ; typ < sizeof(typeMap)/sizeof(PgSQLTypeMap) ; typ += 1)
			if (typeMap[typ].ptype == ftype)
				if ((typeMap[typ].flags & FF_NOCREATE) == 0)
				{
					mapp	= &typeMap[typ] ;
					break	;
				}

		/* If there is no mapping but the use-best-match flag	*/
		/* is set, then look for a mapping based on the		*/
		/* internal types.					*/
		if ((mapp == 0) && best)
			for (uint typ = 0 ; typ < sizeof(typeMap)/sizeof(PgSQLTypeMap) ; typ += 1)
				if (typeMap[typ].itype == itype)
					if ((typeMap[typ].flags & FF_NOCREATE) == 0)
					{
						mapp	= &typeMap[typ] ;
						break	;
					}

		/* It is an internal error if no entry at all was	*/
		/* found ....						*/
		if (mapp == 0)
		{	m_lError = KBError
				   (	KBError::Fault,
				  	"Error mapping column type",
				  	QString ("Type %1 for column %2 not known")
						.arg(ftype)
						.arg(fSpec->m_name),
				  	__ERRLOCN
				   )	;
			return	false	;
		}

		create += QString ("%1\t%2%3%4 %5")
				  .arg(sep	      )
				  .arg(quote	      )
				  .arg(fSpec->m_name  )
				  .arg(quote	      )
				  .arg(mapp ->ptype   )
				  ;

		if ((mapp->flags & FF_LENGTH) != 0)
		{
			create	+= QString("(%1").arg(fSpec->m_length) ;
			if ((mapp->flags & FF_PREC) != 0)
				create	+= QString(",%1").arg(fSpec->m_prec) ;
			create	+= ")" ;
		}

		if ((fSpec->m_flags & KBFieldSpec::NotNull) != 0) create += " not null"    ;
		if ((fSpec->m_flags & KBFieldSpec::Primary) != 0) create += " primary key" ;
		if ((fSpec->m_flags & KBFieldSpec::Unique ) != 0) create += " unique" 	 ;

		if (!fSpec->m_defval.isEmpty())
			if (fSpec->m_defval.lower() == "null")
				create += " default null" ;
			else	create += QString(" default '%1'").arg(fSpec->m_defval) ;

		sep	= ",\n"	;
	}

	create	+= "\n)";

//	fprintf	(stderr, "%s\n", (cchar *)create) ;

	return	true	;
}

/*  KBPgSQL								*/
/*  doGrants	: Apply grants to object				*/
/*  grants	: const QString & : Skeleton GRANT .... text		*/
/*  object	: const QString & : Object name				*/
/*  type	: const QString & : Object type for error messages	*/
/*  (returns)	: bool		  : Success				*/

bool	KBPgSQL::doGrants
	(	const QString	&grants,
		const QString	&object,
		const QString	&type
	)
{
	PGresult	*pgQry	;
	QString		dummy	;

	if (!grants.isEmpty())
	{
		if ((pgQry = execSQL
			     (	QString (grants).arg(object),
				dummy,
				0,
				0,
				0,
				QString (TR("Error setting grants on %1 %2")).arg(type).arg(object),
				PGRES_COMMAND_OK,
				m_lError,
				true
			     )) == 0) return false ;
		PQclear (pgQry) ;
	}

	return	true ;
}

/*  KBPgSQL								*/
/*  doCreateTable: Create a new table					*/
/*  tabSpec	 : KBTableSpec &: Table specification			*/
/*  assoc	 : bool		: Create associated stuff		*/
/*  best	 : bool		: Use best match			*/
/*  (returns)	 : bool		: Success				*/

bool	KBPgSQL::doCreateTable
	(	KBTableSpec	&tabSpec,
		bool		assoc,
		bool		best
	)
{
	QString		create	;
	QString		dummy	;
	QString		grants	;
	QString		tname	;
	PGresult	*pgQry	;

	/* If grants setting is enabled then get the grants SQL text.	*/
	/* Note that the dialog is used even if it is not show as a	*/
	/* popup, since it contains the code to build the SQL text.	*/
	if (m_grants)
	{
		KBPgGrantsDlg	gDlg
				(	m_grantSelect,
					m_grantInsert,
					m_grantUpdate,
					m_grantDelete,
					m_grantTo,
					m_mapExpressions
				)	;

		if (m_grantPopup)
			switch (gDlg.exec())
			{
				case 0 :
					/* User cancels completely, so	*/
					/* return an error to caller.	*/
					m_lError = KBError
						   (	KBError::Warning,
						   	QString(TR("User cancelled creation")),
						   	QString::null,
						   	__ERRLOCN
						   )	;
					return	false	;

				case 1 :
					/* User skips. Creation occurs	*/
					/* but no grants are set.	*/
					break	;

				case 2 :
					/* User OKs, so pick up the SQL	*/
					/* grant text.			*/
					grants	= gDlg.grantText () ;
					break	;

				default	:
					break	;
			}
		else
			/* No popup, just get the grants SQL text.	*/
			grants	= gDlg.grantText () ;
	}

	/* If we are automatically creating the associated stuff then	*/
	/* initially create the table with a modified name. This is	*/
	/* needed so that we can later rename the table, then create a	*/
	/* new table with the same name, without conficts over index	*/
	/* names.							*/
	if (assoc)
	{
		static uint	loop	;
		if (loop 	== 0) loop = time (0) ;
		if ((loop += 1) == 0) loop = 1 ;

		tname	= QString("_%1_%2")
				.arg(tabSpec.m_name.left(4))
				.arg(QString::number(loop).right(5)) ;
	}
	else	tname	= tabSpec.m_name ;


	if (!tblCreateSQL
		(	tabSpec.m_fldList,
			tname,
			create,
			best
		)) return false ;

	if ((pgQry = execSQL
		     (	create,
			dummy,
			0,
			0,
			0,
			QString ("Error creating table (as %1)").arg(tname),
			PGRES_COMMAND_OK,
			m_lError,
			true
		     )) == 0) return false ;
	PQclear (pgQry) ;


	/* If the table was created under the modified name, rename it	*/
	/* to the real specified name.					*/
	cchar	*q	= m_mapExpressions ?
				"alter table \"%1\" rename to \"%2\"" 	:
				"alter table %1 rename to %2" 		;
	if (assoc)
	{
		if ((pgQry = execSQL
			     (	QString (q)
					.arg(tname)
					.arg(tabSpec.m_name),
				dummy,
				0,
				0,
				0,
				QString ("Error renaming table (from %1)").arg(tname),
				PGRES_COMMAND_OK,
				m_lError,
				true
			     )) == 0) return false ;
		PQclear (pgQry) ;
	}

	if (!doGrants (grants, tabSpec.m_name, TR("table")))
		return	false	;


	/* If we are creating associated stuff, and we are not using	*/
	/* the "serial" option to create the primary key, then create	*/
	/* out own sequence, named for the table with "_seq" appended.	*/
	if (assoc && !m_primaryIsSerial)
		LITER
		(	KBFieldSpec,
			tabSpec.m_fldList,
			fSpec,

			if (fSpec->m_typeName == "Primary Key")
			{
				cchar	*q	= m_mapExpressions ?
							"create sequence \"%1_seq\" minvalue 1"	:
							"create sequence %1_seq minvalue 1"	;

				create	= QString (q).arg(tabSpec.m_name) ;

				if ((pgQry = execSQL
					     (	create,
						dummy,
						0,
						0,
						0,
						"Error creating associated sequence",
						PGRES_COMMAND_OK,
						m_lError,
						true
					     )) == 0) return false ;

				PQclear (pgQry) ;

				if (!doGrants (grants, QString("%1_seq").arg(tabSpec.m_name), TR("sequence")))
					return	false	;
			}
		)

	/* Next stage is to create indexes. Note that we only handle	*/
	/* single-column indexes. Also, ignore primary key columns as	*/
	/* PostgreSQL will have automatically have created an index	*/
	/* for these.							*/
	LITER
	(	KBFieldSpec,
		tabSpec.m_fldList,
		fSpec,

		if (fSpec->m_typeName == "Primary Key")
			continue ;
		if ((fSpec->m_flags & KBFieldSpec::Primary) != 0)
			continue ;
		if ((fSpec->m_flags & KBFieldSpec::Indexed) == 0)
			continue ;


		cchar	*unique	= "" ;

		if ((fSpec->m_flags & KBFieldSpec::Unique ) != 0)
			unique	= "unique" ;

		cchar	*q	= m_mapExpressions ?
					"create %1 index \"%2_idx_%3\" on \"%4\" (\"%5\")"	:
					"create %1 index %2_idx_%3 on %4 (%5)"			;

		create	= QString (q)
				  .arg(unique)
				  .arg(tname)
				  .arg(fSpec ->m_name)
				  .arg(tabSpec.m_name)
				  .arg(fSpec ->m_name) ;

		if ((pgQry = execSQL
			     (	create,
				dummy,
				0,
				0,
				0,
				QString("Error creating column %1 index")
					.arg(fSpec->m_name),
				PGRES_COMMAND_OK,
				m_lError,
				true
			     )) == 0) return false ;

		PQclear (pgQry) ;
	)

	return	true	;
}

/*  KBPgSQL								*/
/*  doRenameTable: Rename a  table					*/
/*  oldName	 : cchar *	: Current table name			*/
/*  newName	 : cchar *	: New table name			*/
/*  assoc	 : bool		: Rename associated stuff - none here	*/
/*  (returns)	 : bool		: Success				*/

bool	KBPgSQL::doRenameTable
	(	cchar	*oldName,
		cchar	*newName,
		bool	assoc
	)
{
	PGresult *pgQry ;
	QString	 dummy	;
	cchar	 *q1	= m_mapExpressions ?
				"alter table \"%1\" rename to \"%2\"" 	:
				"alter table %1 rename to %2" 		;

	if ((pgQry = execSQL
		     (	QString (q1).arg(oldName).arg(newName),
			dummy,
			0,
			0,
			0,
			"Error renaming table",
			PGRES_COMMAND_OK,
			m_lError,
			true
		     )) == 0) return false ;
	PQclear (pgQry) ;

	if (!assoc) return true  ;

	cchar	*q2	= m_mapExpressions ?
				"alter table \"%1_seq\" rename to \"%2_seq\""	:
				"alter table %1_seq rename to %2_seq"		;
	if ((pgQry = execSQL
		     (	QString (q2).arg(oldName).arg(newName),
			dummy,
			0,
			0,
			0,
			"Error renaming associated sequence",
			PGRES_COMMAND_OK,
			m_lError,
			true
		     )) == 0) return false ;
	PQclear (pgQry) ;
	return	true	;
}

/*  KBPgSQL								*/
/*  doDropTable	: Drop a table						*/
/*  table	: cchar *	: Table name				*/
/*  assoc	: bool		: Drop associated stuff - none here	*/
/*  (returns)	: bool		: Success				*/

bool	KBPgSQL::doDropTable
	(	cchar	*table,
		bool	assoc
	)
{
	PGresult	*pgQry	;
	QString		dummy	;
	QString		seqName	;

	KBFieldSpec	*primary;
	KBTableSpec	tabSpec	(table) ;


	if (!doListFields (tabSpec)) return false ;
	primary	= tabSpec.findPrimary() ;

	cchar	*q	= m_mapExpressions ?
				"drop table \"%1\""	:
				"drop table %1"		;
	if ((pgQry = execSQL
		     (	QString (q).arg(table),
			dummy,
			0,
			0,
			0,
			"Error dropping table",
			PGRES_COMMAND_OK,
			m_lError,
			true
		     )) == 0) return false ;
	PQclear (pgQry) ;


	if ((primary != 0) && assoc)
	{
		if (m_primaryIsSerial)
		{
			cchar	*q	= m_mapExpressions ?
						"drop sequence \"%1_%2_seq\""	:
						"drop sequence %1_%2_seq"	;
			if ((pgQry = execSQL
				     (	QString (q)
						.arg(table)
						.arg(primary->m_name),
					dummy,
					0,
					0,
					0,
					"Error dropping serial sequence",
					PGRES_COMMAND_OK,
					m_lError,
					true
				     )) == 0) return false ;
			PQclear (pgQry) ;
		}

		if (!m_primaryIsSerial)
		{
			cchar	*q	= m_mapExpressions ?
						"drop sequence \"%1_seq\""	:
						"drop sequence %1_seq"		;

			if ((pgQry = execSQL
				     (	QString (q).arg(table),
					dummy,
					0,
					0,
					0,
					"Error dropping associated sequence",
					PGRES_COMMAND_OK,
					m_lError,
					true
				     )) == 0) return false ;
			PQclear (pgQry) ;
		}
	}

	return	true	;
}

/*  KBPgSQL								*/
/*  createView	: Create a view						*/
/*  viewSpec	: KBTableSpec &	: Specification	8			*/
/*  (returns)	: bool		: Success				*/

bool	KBPgSQL::createView
	(	KBTableSpec	&viewSpec
	)
{
	cchar	 *q	= m_mapExpressions ?
				"create view \"%1\" as %2"	:
				"create view %1 as %2"		;

	QString	 query	= QString (q)
				  .arg (viewSpec.m_name)
				  .arg (viewSpec.m_view) ;
	QString	 dummy	;
	PGresult *pgQry	;

	if ((pgQry = execSQL
		     (	query,
			dummy,
			0,
			0,
			0,
			"Error creating view",
			PGRES_COMMAND_OK,
			m_lError,
			true
		     )) == 0) return false ;
	PQclear (pgQry) ;
	return	true	;
}

/*  KBPgSQL								*/
/*  renameView	: Rename a view						*/
/*  oldName	: cchar *	: Current view name			*/
/*  newName	: cchar *	: New view name				*/
/*  (returns)	: bool		: Success				*/

bool	KBPgSQL::renameView
	(	cchar	*,
		cchar	*
	)
{
	m_lError = KBError
		   (	KBError::Error,
			TR("Cannot rename views"),
			QString::null,
			__ERRLOCN
		   )	;
	return	false	;
}


/*  KBPgSQL								*/
/*  dropView	: Drop a view						*/
/*  view	: cchar *	: View name				*/
/*  (returns)	: bool		: Success				*/

bool	KBPgSQL::dropView
	(	cchar	*view
	)
{
	PGresult *pgQry	;
	QString	 dummy	;
	cchar	 *q	= m_mapExpressions ?
				"drop view \"%1\""	:
				"drop view %1"		;

	if ((pgQry = execSQL
		     (	QString (q).arg(view),
			dummy,
			0,
			0,
			0,
			"Error dropping view",
			PGRES_COMMAND_OK,
			m_lError,
			true
		     )) == 0) return false ;

	PQclear (pgQry) ;
	return	true	;
}

/*  KBPgSQL								*/
/*  descSequence: Describe a sqquence					*/
/*  viewSpec	: KBSequenceSpec & : Specification			*/
/*  (returns)	: bool		   : Success				*/

bool	KBPgSQL::descSequence
	(	KBSequenceSpec	&seqSpec
	)
{
	cchar	 *quote	= m_mapExpressions ? "\"" : "" ;

	PGresult *pgQry	;
	QString	 dummy	;
	QString	 query	= QString
			  (	"select last_value, 	"
				"	increment_by,	"
				"	min_value,	"
				"	max_value,	"
				"	is_cycled	"
				"from	%1%2%3		"
			  )
			  .arg	(quote)
			  .arg	(seqSpec.m_name)
			  .arg	(quote) ;

	if ((pgQry = execSQL
		     (	query,
			dummy,
			0,
			0,
			0,
			"Error getting sequence details",
			PGRES_TUPLES_OK,
			m_lError,
			true
		     )) == 0) return false ;

	if (PQntuples (pgQry) == 0)
	{
		m_lError = KBError
			   (	KBError::Error,
				QString	(TR("Sequence %1 does not exist"))
					.arg (seqSpec.m_name),
				QString::null,
				__ERRLOCN
			   )	;
		PQclear	(pgQry)	;
		return	false	;
	}

	seqSpec.m_start		= atoi(PQgetvalue (pgQry, 0, 0)) ;
	seqSpec.m_increment	= atoi(PQgetvalue (pgQry, 0, 1)) ;
	seqSpec.m_minValue	= atoi(PQgetvalue (pgQry, 0, 2)) ;
	seqSpec.m_maxValue	= atoi(PQgetvalue (pgQry, 0, 3)) ;

	seqSpec.m_flags		= KBSequenceSpec::HasIncrement|
				  KBSequenceSpec::HasMinValue |
				  KBSequenceSpec::HasMaxValue |
				  KBSequenceSpec::HasStart    ;

	if (PQgetvalue (pgQry, 0, 3)[0] == 't')
		seqSpec.m_flags |= KBSequenceSpec::CanCycle ;

	PQclear	(pgQry)	;
	return	true	;
}

/*  KBPgSQL								*/
/*  createSequence							*/
/*		: Create a sqquence					*/
/*  viewSpec	: KBSequenceSpec & : Specification			*/
/*  (returns)	: bool		   : Success				*/

bool	KBPgSQL::createSequence
	(	KBSequenceSpec	&seqSpec
	)
{
	cchar	 *q	= m_mapExpressions ?
				"create sequence \"%1\""	:
				"create sequence %1"		;

	QString	 query	= QString (q).arg (seqSpec.m_name) ;
	QString	 dummy	;
	PGresult *pgQry	;

	if ((seqSpec.m_flags & KBSequenceSpec::HasIncrement) != 0)
		query += QString(" increment %1").arg(seqSpec.m_increment) ;
	if ((seqSpec.m_flags & KBSequenceSpec::HasMinValue ) != 0)
		query += QString(" minvalue  %1").arg(seqSpec.m_minValue ) ;
	if ((seqSpec.m_flags & KBSequenceSpec::HasMaxValue ) != 0)
		query += QString(" maxvalue  %1").arg(seqSpec.m_maxValue ) ;
	if ((seqSpec.m_flags & KBSequenceSpec::HasStart    ) != 0)
		query += QString(" start     %1").arg(seqSpec.m_start    ) ;

	if ((seqSpec.m_flags & KBSequenceSpec::CanCycle    ) != 0)
		query += " cycle" ;

	if ((pgQry = execSQL
		     (	query,
			dummy,
			0,
			0,
			0,
			"Error creating sequence",
			PGRES_COMMAND_OK,
			m_lError,
			true
		     )) == 0) return false ;
	PQclear (pgQry) ;
	return	true	;
}

/*  KBPgSQL								*/
/*  renameSequence							*/
/*		: Rename a sequuence					*/
/*  oldName	: cchar *	: Current view sequence			*/
/*  newName	: cchar *	: New sequence name			*/
/*  (returns)	: bool		: Success				*/

bool	KBPgSQL::renameSequence
	(	cchar	*,
		cchar	*
	)
{
	m_lError = KBError
		   (	KBError::Error,
			TR("Cannot rename sequences"),
			QString::null,
			__ERRLOCN
		   )	;
	return	false	;
}


/*  KBPgSQL								*/
/*  dropSequence: Drop a sequence					*/
/*  sequence	: cchar *	: Sequence name				*/
/*  (returns)	: bool		: Success				*/

bool	KBPgSQL::dropSequence
	(	cchar	*sequence
	)
{
	PGresult *pgQry	;
	QString	 dummy	;
	cchar	 *q	= m_mapExpressions ?
				"drop sequence \"%1\""	:
				"drop sequence %1"	;

	if ((pgQry = execSQL
		     (	QString (q).arg(sequence),
			dummy,
			0,
			0,
			0,
			"Error dropping sequence",
			PGRES_COMMAND_OK,
			m_lError,
			true
		     )) == 0) return false ;

	PQclear (pgQry) ;
	return	true	;
}

/*  KBPgSQL								*/
/*  keepsCase	: Check whether server preserves name case		*/
/*  (returns)	: bool		: True if so				*/

bool	KBPgSQL::keepsCase ()
{
	/* Postgres doesn't by default ...				*/
	return	m_mapExpressions ;
}

/*  KBPgSQL								*/
/*  optionFlags	: Get server option flags				*/
/*  (returns)	: uint		: Flags					*/

uint	KBPgSQL::optionFlags ()
{
	/* Standard flags, plus we support SSH tunnelling.		*/
	return	KBServer::optionFlags() | AF_SSHTUNNEL ;
}

/*  KBPgSQL								*/
/*  mapExpression: Map expressions with double-quotes			*/
/*  expr	 : const QString & : Expression				*/
/*  (returns)	 : QString	   : Mapped expression			*/

QString	KBPgSQL::mapExpression
	(	const QString	&expr
	)
{
	static	QString	spec ("_") ;

	return	m_mapExpressions ?
			doMapExpression (expr, "\"", "\"", spec) :
			expr ;
}

/*  KBPgSQL								*/
/*  getSyntax	: Get text for syntax element				*/
/*  result	: QString &	: Return resule string			*/
/*  syntax	: Syntax	: Element				*/
/*  ...		: ...		: Arguments				*/
/*  (returns)	: QString	: Text					*/

bool	KBPgSQL::getSyntax
	(	QString			&result,
		KBServer::Syntax	syntax,
		...
	)
{
	va_list	 ap ;
	va_start (ap, syntax) ;

	switch (syntax)
	{
		case Limit :
			{
				int	limit	= va_arg(ap, int) ;
				int	offset	= va_arg(ap, int) ;

				result	= QString(" limit %1 offset %2 ").arg(limit).arg(offset) ;
			}
			return	true	;

		default	:
			break	;
	}

	m_lError = KBError
		   (	KBError::Error,
			QString(TR("Driver does not support %1")).arg(syntaxToText(syntax)),
			QString::null,
			__ERRLOCN
		   )	;
	return	false	;
}

/*  KBPgSQL								*/
/*  operatorMap	: Get the operator map for the driver			*/
/*  map		: cchar *&	: Vector of textual operators		*/
/*  (returns)	: uint		: Number of operators in map		*/

uint	KBPgSQL::operatorMap
	(	cchar		**&map
	)
{
	static	cchar	*opMap[] =
	{
		"=",		// EQ
		"!=",		// NEQ
		"<=",		// LE
		">=",		// GE
		"<",		// LT
		">",		// GT
		"like",		// Like
	}	;

	memcpy	(m_opMap, opMap, sizeof(m_opMap)) ;
	if (m_caseInsensitive) m_opMap[Like] = "ilike" ;

	map	= m_opMap ;
	return	sizeof(opMap)/sizeof(cchar *) ;
}

bool	KBPgSQL::setStmtTimeout
	(	KBError		&pError
	)
{
	if (m_useTimeouts)
	{
		PGresult *pgQry ;
		QString	 query	= QString("set statement_timeout to %1").arg(m_stmtTimeout) ;

		if ((pgQry = execSQL
		     (		query,
		     		query,
		     		0,
		     		0,
		     		0,
		     		"Error setting statement timeout",
		     		PGRES_COMMAND_OK,
		     		pError,
		     		true
		     )) == 0) return false ;

		PQclear	(pgQry) ;
		return	true	;
	}

	return	true	;
}

bool	KBPgSQL::setLockTimeout
	(	KBError		&pError
	)
{
	if (m_useTimeouts)
	{
		PGresult *pgQry ;
		QString	 query	= QString("set statement_timeout to %1").arg(m_lockTimeout) ;

		if ((pgQry = execSQL
		     (		query,
		     		query,
		     		0,
		     		0,
		     		0,
		     		"Error setting update lock timeout",
		     		PGRES_COMMAND_OK,
		     		pError,
		     		true
		     )) == 0) return false ;

		PQclear	(pgQry) ;
		return	true	;
	}

	return	true	;
}



/*  ------------------------------------------------------------------- */


/*  KBPgSQLQrySelect							*/
/*  KBPgSQLQrySelect							*/
/*		: Constructor for select query object			*/
/*  server	: KBPgSQL *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Select query text			*/
/*  update	: bool		  : Select for update			*/
/*  (returns)	: KBPgSQLQrySelect:					*/

KBPgSQLQrySelect::KBPgSQLQrySelect
	(	KBPgSQL		*server,
		bool		data,
		const QString	&select,
		bool		update
	)	
	:
	KBSQLSelect	(server, data, select),
	m_server	(server),
	m_update	(update)
{
	m_nRows		= 0 ;
	m_nFields	= 0 ;
	m_pgQry		= 0 ;
}

/*  KBPgSQLQrySelect							*/
/*  ~KBPgSQLQrySelect							*/
/*		: Destructor for select query object			*/
/*  (returns)	:		:					*/

KBPgSQLQrySelect::~KBPgSQLQrySelect ()
{
	if (m_pgQry != 0)
		PQclear (m_pgQry) ;

//	if (m_types != 0)
//		for (uint idx = 0 ; idx < m_nFields ; idx += 1)
//			m_types[idx]->deref() ;
}

/*  KBPgSQLQrySelect							*/
/*  execute	: Execute the query					*/
/*  nvals	: uint		: Number of substitution values	*/
/*  value	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBPgSQLQrySelect::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	if (m_pgQry != 0)
		PQclear (m_pgQry) ;

	if (m_update)
		if (!m_server->setLockTimeout (m_lError))
			return	false	;

	if ((m_pgQry = m_server->execSQL
		       (	m_rawQuery,
				m_subQuery,
				nvals,
		     		values,
		     		m_codec,
		     		"Select query failed",
		     		PGRES_TUPLES_OK,
		     		m_lError,
		     		true
		       )) == 0)
	{
		if (m_update)
		{	KBError	dummy	;
			m_server->setStmtTimeout (dummy) ;
		}

		return	false	;
	}

	m_nRows	  = PQntuples (m_pgQry)	;
	m_nFields = PQnfields (m_pgQry)	;

	if (m_types == 0)
		m_types	  = getFieldTypes (m_pgQry) ;

	if (m_update)
		if (!m_server->setStmtTimeout (m_lError))
			return	false	;

	return	true ;
}

/*  KBPgSQLQrySelect							*/
/*  getField	: Get a specified field from the query results		*/
/*  qrow	: uint		  : Row number				*/
/*  qcol	: uint		  : Column number			*/
/*  vtrans	: KBValue::VTrans : Translation mode			*/
/*  (returns)	: KBValue	  : Value				*/

KBValue	KBPgSQLQrySelect::getField
	(	uint		qrow,
		uint		qcol,
		KBValue::VTrans
	)
{
	/* First check that the request is within the range of rows	*/
	/* and fields returned by the query.				*/
	if ((int)qrow >= m_nRows  ) return KBValue() ;
	if (     qcol >= m_nFields) return KBValue() ;

	if (PQgetisnull (m_pgQry, qrow, qcol))
		return KBValue (m_types[qcol]) ;

	char	*value	= PQgetvalue (m_pgQry, qrow, qcol) ;

	if (m_types[qcol]->getIType() == KB::ITBinary)
	{
		size_t	tolen	;
		uchar	*b	= unescapeBinary
				  (	(uchar *)value,
					&tolen
				  )	;
		KBValue	v	((cchar *)b, tolen) ;
		free	(b)	;
		return	v	;
	}

	/* Postgres insists on 'f' and 't' for false and true while	*/
	/* 0 and 1 are much more universal, so silently convert. Note	*/
	/* that KBPgSQLType::getQueryText reverses this.		*/
	if (m_types[qcol]->getIType() == KB::ITBool)
	{
		if (value[0] == 'f') return KBValue (0, m_types[qcol]) ;
		if (value[0] == 't') return KBValue (1, m_types[qcol]) ;
	}

	return	KBValue (value, m_types[qcol], m_codec) ;
}

/*  KBPgSQLQrySelect							*/
/*  getFieldName: Get a specified field name from the query results	*/
/*  qcol	: uint		: Column number				*/
/*  (returns)	: QString	: Value					*/

QString	KBPgSQLQrySelect::getFieldName
	(	uint	qcol
	)
{
	/* First check that the request is within the range of rows	*/
	/* and fields returned by the query.				*/
	if (qcol >= m_nFields) return QString() ;

	return	PQfname (m_pgQry, qcol) ;
}

/*  KBPgSQLQryUpdate							*/
/*  KBPgSQLQryUpdate							*/
/*		: Constructor for update query object			*/
/*  server	: KBPgSQL *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Update query text			*/
/*  tabName	: const QString & : Table being updated			*/
/*  codec	: QTextCodec *	  : Non-default codec			*/
/*  (returns)	: KBPgSQLQryUpdate:					*/

KBPgSQLQryUpdate::KBPgSQLQryUpdate
	(	KBPgSQL		*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)	
	:
	KBSQLUpdate (server, data, query, tabName),
	m_server    (server)
{
	QString	    dummy  ;
	PGresult    *pgQry ;

	m_nRows	 = 0	;
	m_isView = 0	;

	if ((pgQry = server->execSQL
			(	QString
				(	"select	relkind 	"
					"from	pg_class	"
					"where relname = '%1'	"
				)
					.arg(tabName),
			 	dummy,
			 	0,
			 	0,
			 	0,
			 	QString::null,
			 	PGRES_TUPLES_OK,
			 	m_lError)) != 0)
	{
		cchar	*v = PQgetvalue (pgQry, 0, 0) ;
		if ((v != 0) && (v[0] == 'v')) m_isView = true ;
		PQclear (pgQry) ;
	}

}

/*  KBPgSQLQryUpdate							*/
/*  execute	: Execute the query					*/
/*  nvlaus	: uint		: Number of substitution values	*/
/*  values	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBPgSQLQryUpdate::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	PGresult *pgQry ;

	if ((pgQry = m_server->execSQL
		     (		m_rawQuery,
		     		m_subQuery,
		     		nvals,
		     		values,
		     		m_codec,
		     		"Update query failed",
		     		PGRES_COMMAND_OK,
		     		m_lError,
		     		true
		     )) == 0) return false ;


	m_nRows	= m_isView ? 1 : atoi (PQcmdTuples (pgQry)) ;
	PQclear	(pgQry) ;
	return	true	;
}

/*  KBPgSQLQryUpdate							*/
/*  ~KBPgSQLQryUpdate							*/
/*		: Destructor for update query object			*/
/*  (returns)	:		:					*/

KBPgSQLQryUpdate::~KBPgSQLQryUpdate ()
{
}


/*  KBPgSQLQryInsert							*/
/*  KBPgSQLQryInsert							*/
/*		: Constructor for insert query object			*/
/*  server	: KBPgSQL *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Insert query			*/
/*  tabName	: const QString & : Table being updated			*/
/*  codec	: QTextCodec *	  : Non-default codec			*/
/*  (returns)	: KBPgSQLQryInsert:					*/

KBPgSQLQryInsert::KBPgSQLQryInsert
	(	KBPgSQL		*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)	
	:
	KBSQLInsert (server, data, query, tabName),
	m_server    (server)
{
	m_nRows	  = 0 	;
	m_usePKey = -1	;
}

/*  KBPgSQLQryInsert							*/
/*  ~KBPgSQLQryInsert							*/
/*		: Destructor for insert query object			*/
/*  (returns)	:		:					*/

KBPgSQLQryInsert::~KBPgSQLQryInsert ()
{
}

/*  KBPgSQLQryInsert							*/
/*  execute	: Execute the query					*/
/*  nvlaus	: uint		: Number of substitution values		*/
/*  values	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBPgSQLQryInsert::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	PGresult *pgQry ;

	if ((pgQry = m_server->execSQL
		     (		m_rawQuery,
		     		m_subQuery,
		     		nvals,
		     		values,
		     		m_codec,
		     		"Insert query failed",
		     		PGRES_COMMAND_OK,
		     		m_lError,
		     		true
		     )) == 0) return false ;

	m_nRows	  = atoi (PQcmdTuples (pgQry)) ;
	m_lastOid = PQoidStatus (pgQry) ;
	PQclear	(pgQry) ;
	return	true	;
}

/*  KBPgSQLQryInsert							*/
/*  getNewKey	: Get new insert key					*/
/*  primary	: const QString & : Key column name			*/
/*  newKey	: KBValue &	  : New key				*/
/*  prior	: bool		  : Call is pre-insert			*/
/*  (returns)	: bool		  : Success				*/

bool	KBPgSQLQryInsert::getNewKey
	(	const QString	&primary,
		KBValue		&newKey,
		bool		prior
	)
{
	QString	tabName	= m_tabName	;
	QString	priName	= primary	;

	if (!m_server->m_mapExpressions)
	{
		tabName	= tabName.lower() ;
		priName	= priName.lower() ;
	}

	/* Before we execute the query for the first time, we need to	*/
	/* see the primary key column is sequence-derived. If so then	*/
	/* we can get the primary key ahead of the insert, if not then	*/
	/* we will retrieve it afterwards via the row Oid.		*/
	if (m_usePKey < 0)
	{
		KBTableSpec tabSpec (m_tabName) ;
		if (!m_server->listFields (tabSpec))
		{	m_lError = m_server->lastError() ;
			return	 false ;
		}

		if ( (tabSpec.m_prefKey >= 0) &&
		     (tabSpec.m_fldList.at(tabSpec.m_prefKey)->m_name.lower() == primary.lower()) )
		{
			bool exists  ;
			if (!m_server->objectExists (m_tabName + "_seq", "S", exists))
			{	m_lError = m_server->lastError() ;
				return	 false ;
			}

			m_usePKey = exists ;
		}
		else	m_usePKey = false  ;
	}

	/* If this is the pre-insert call and we can use the default	*/
	/* primary key, then retrieve and return the next value from	*/
	/* the associated query.					*/
	if (prior)
	{
		if (m_usePKey)
		{
			QString	 dummy	;
			PGresult *pgQry ;

			cchar	*q	= m_server->m_mapExpressions ?
						"select nextval('\"%1_seq\"')"	:
						"select nextval('%1_seq')"	;

			if ((pgQry = m_server->execSQL
				     (	QString(q).arg(tabName),
				     	dummy,
					0,
					0,
					0,
					"Failed to get next value for primary key",
					PGRES_TUPLES_OK,
					m_lError
				     )) == 0) return false ;

			newKey	= KBValue (PQgetvalue (pgQry, 0, 0), &_kbFixed) ;
			PQclear	(pgQry) ;
			return	true	;
		}

		newKey	= KBValue()	;
		return	true		;
	}

	/* This is the post-insert call then we should be able to	*/
	/* retrieve the primary key using the saved Oid from the insert	*/
	QString	 dummy	;
	PGresult *pgQry ;

	cchar	*q	= m_server->m_mapExpressions ?
				"select \"%1\" from \"%2\" where oid = %3"	:
				"select %1 from %2 where oid = %3"		;

	if ((pgQry = m_server->execSQL
		     (		QString(q).arg(priName).arg(tabName).arg(m_lastOid),
		     		dummy,
				0,
				0,
				0,
				"Failed to retrieve primary key via Oid",
				PGRES_TUPLES_OK,
				m_lError
		     )) == 0) return false ;

	if (PQntuples(pgQry) != 1)
	{
		m_lError = KBError
			   (	KBError::Error,
				QString	("Unexpectedly got %1 row(s) while retrieving via Oid")
					.arg(PQntuples(pgQry)),
				QString::null,
				__ERRLOCN
			   )	;
  		PQclear	(pgQry) ;
		return	false	;
	}

	if (PQnfields(pgQry) != 1)
	{
		m_lError = KBError
			   (	KBError::Error,
				QString	("Unexpectedly got %1 fields(s) while retrieving via Oid")
					.arg(PQnfields(pgQry)),
				QString::null,
				__ERRLOCN
			   )	;
  		PQclear	(pgQry) ;
		return	false	;
	}

	newKey	= KBValue (PQgetvalue (pgQry, 0, 0), &_kbFixed) ;
	PQclear	(pgQry) ;
	return	true	;

//	m_lError = KBError
//		   (	KBError::Error,
//			"Unable to retrieve primary key",
//			QString("Table:%1, Primary key: %2").arg(tabName).arg(primary),
//			__ERRLOCN
//		   )	;
//	return	false	;
}

/*  KBPgSQLQryDelete							*/
/*  KBPgSQLQryDelete							*/
/*		: Constructor for delete query object			*/
/*  server	: KBPgSQL *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Delete query			*/
/*  tabName	: const QString & : Table being updated			*/
/*  codec	: QTextCodec *	  : Non-default codec			*/
/*  (returns)	: KBPgSQLQryDelete:					*/

KBPgSQLQryDelete::KBPgSQLQryDelete
	(	KBPgSQL		*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)	
	:
	KBSQLDelete (server, data, query, tabName),
	m_server    (server)
{
	m_nRows	= 0 ;
}

/*  KBPgSQLQryDelete							*/
/*  execute	: Execute the query					*/
/*  nvlaus	: uint		: Number of substitution values	*/
/*  values	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBPgSQLQryDelete::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	PGresult *pgQry ;

	if ((pgQry = m_server->execSQL
		     (		m_rawQuery,
		     		m_subQuery,
		     		nvals,
		     		values,
		     		m_codec,
		     		"Delete query failed",
		     		PGRES_COMMAND_OK,
		     		m_lError,
		     		true
		     )) == 0) return false ;

	m_nRows	= atoi (PQcmdTuples (pgQry)) ;
	PQclear	(pgQry) ;
	return	true	;
}

/*  KBPgSQLQryDelete							*/
/*  ~KBPgSQLQryDelete							*/
/*		: Destructor for delete query object			*/
/*  (returns)	:		:					*/

KBPgSQLQryDelete::~KBPgSQLQryDelete ()
{
}

/*  ------------------------------------------------------------------  */

/*  KBPgSQLQryCursor							*/
/*  KBPgSQLQryCursor							*/
/*		: Constructor for cursor base class			*/
/*  server	: KBServer *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Cursor text				*/
/*  cursor	: const QString & : Cursor name				*/
/*  (returns)	: KBSQLCursor	  :					*/

KBPgSQLQryCursor::KBPgSQLQryCursor
	(	KBPgSQL		*server,
		bool		data,
		const QString	&query,
		const QString	&cursor
	)
	:
	KBSQLCursor	(server, data, query, cursor),
	m_server	(server)
{
	m_types	  = 0	;
	m_nFields = 0	;
}

/*  KBPgSQLQryCursor							*/
/*  ~KBPgSQLQryCursor: Destructor for cursor base class			*/

KBPgSQLQryCursor::~KBPgSQLQryCursor ()
{
	close	()	;
}

/*  KBPgSQLQryCursor							*/
/*  execute	: Open the cursor					*/
/*  nvals	: uint		  : Number of substitution values	*/
/*  values	: KBValue *	  : Substitution values			*/
/*  (returns)	: bool		  : Success				*/

bool	KBPgSQLQryCursor::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	close	()	;

	PGresult *pgQry = m_server->execSQL
		          (	m_rawQuery,
				m_subQuery,
				nvals,
		     		values,
		     		m_codec,
		     		"Open cursor failed",
		     		PGRES_COMMAND_OK,
		     		m_lError,
		     		true
		          )	;

	if (pgQry == 0) return false ;

	PQclear (pgQry) ;
	return	true	;
}

/*  KBPgSQLQryCursor							*/
/*  fetch	: Fetch a row from the cursor				*/
/*  nvals	: uint		: Number of result values		*/
/*  values	: KBValue *	: Result values				*/
/*  got		: bool		: Set true if a row is forthcoming	*/
/*  (returns)	: bool		: Success				*/

bool	KBPgSQLQryCursor::fetch
	(	uint		nvals,
		KBValue		*values,
		bool		&got
	)
{
	QString	dummy	;

	/* Execute the fetch, checking for errors and noting the number	*/
	/* or rows and fields. If the fetch succeeds but returns zero	*/
	/* rows the return success bit with "got" set to false.		*/
	PGresult *pgQry = m_server->execSQL
		          (	QString("fetch next from %1").arg(m_cursor),
				dummy,
				0,
		     		0,
		     		0,
		     		"Cursor fetched failed",
		     		PGRES_TUPLES_OK,
		     		m_lError,
		     		true
		          )	;

	if (pgQry == 0) return false ;

	int	nRows	= PQntuples (pgQry) ;
	int	nFields	= PQnfields (pgQry) ;

	fprintf	
	(	stderr,
		"KBPgSQLQryCursor::fetch: nRows=%d nFields=%d\n",
		nRows,
		nFields
	)	;

	if (nRows <= 0)
	{	got	= false	;
		return	true	;
	}

	/* The first time this method is called (after opening the	*/
	/* cursor) build the array of column types. We can assume that	*/
	/* these would not change from row to row!			*/
	if (m_types == 0)
	{
		m_types	  = getFieldTypes (pgQry) ;
		m_nFields = nFields ;
	}

	/* Fetch the specified number of values. If the caller asks	*/
	/* for more than are avaialable then silently pad out with	*/
	/* nulls.							*/
	for (uint qcol = 0 ; qcol < nvals ; qcol += 1)
	{
		if (qcol >= (uint)nFields)
		{
			values[qcol] = KBValue() ;
			continue  ;
		}

		char	*value	= PQgetvalue (pgQry, 0, qcol) ;
		values[qcol] = KBValue (value, m_types[qcol], m_codec) ;

	}

	PQclear (pgQry) ;
	got	= true	;
	return	true	;
}

bool	KBPgSQLQryCursor::update
	(	const QString	&,
		uint		,
		const KBValue	*
	)
{
	m_lError = KBError
		   (	KBError::Error,
			"Unimplemented: KBPgSQLQryCursor::update",
			QString::null,
			__ERRLOCN
		   )	;

	return	false	;
}

/*  KBPgSQLQryCursor							*/
/*  close	: Close the cursor					*/
/*  (returns)	: bool		: Success				*/

bool	KBPgSQLQryCursor::close ()
{
	/* Null operation so far as PostgreSQL is concerned, we just	*/
	/* clear the types array.					*/
	if (m_types != 0)
	{
		for (uint idx = 0 ; idx < m_nFields ; idx += 1)
			m_types[idx]->deref() ;
		delete	m_types	;

		m_types	  = 0 ;
		m_nFields = 0 ;
	}

	return	true	;
}

/*  ------------------------------------------------------------------  */

#	ifndef _WIN32


KBFACTORY
(	KBPgSQLFactory,
	"driver_pgsql"
)

KBFACTORYIMPL
(	KBPgSQLFactory,
	driver_pgsql,
	"Rekall PgSQL driver",
	"Plugin",
	"0",
	"7",
	"0"
)

#else
class	KBPgSQLFactory : public KBFactory
{
public:
	inline	KBPgSQLFactory() : KBFactory ()
	{
	}

	virtual	QObject	*create(QObject * = 0, const char *	= 0, const char * = 0, 
		const QStringList &	= QStringList());
	virtual	const char *ident() ;
};

extern	"C"	__declspec(dllexport) void *init_libkbase_driver_pgsql()			
{							
	return	new KBPgSQLFactory;				
}							
#endif

QObject	*KBPgSQLFactory::create
	(	QObject		  *parent,
		cchar		  *object,
		cchar		  *,
		const QStringList &
	)
{
	if (dOidToType.count() == 0)
		for (uint idx = 0 ; idx < sizeof(typeMap)/sizeof(PgSQLTypeMap) ; idx += 1)
		{	PgSQLTypeMap *m = &typeMap[idx] ;
			dOidToType.insert (m->oid, m)   ;
		}

	if ((parent != 0) && !parent->inherits ("QWidget"))
	{
		fprintf	(stderr, "KBPgSQLFactory: parent does not inherit QWidget\n") ;
		return	0  ;
	}

	if (strcmp (object, "driver"  ) == 0) return new KBPgSQL () ;

	if (strcmp (object, "advanced") == 0) return new KBPgAdvanced () ;

	return	0 ;
}

cchar	*KBPgSQLFactory::ident ()
{
	return	__KB_BUILD_IDENT	;
}
