/***************************************************************************
    file	         : kb_odbc.cpp
    copyright            : (C) 1999,2000,2001 by Mike Richardson
			   (C) 2000,2001 by theKompany.com
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *     This program is licensed under the terms contained in the file      *
 *     LICENSE which is contained with the source code distribution.       *
 *                                                                         *
 ***************************************************************************/



#include	<qregexp.h>

#include	"kb_odbc.h"
#include	"kb_odbcadv.h"
#include	"kb_build.h"


using	namespace ODBC_NS ; 

/*  ODBCTypeMap								*/
/*  -----------								*/
/*  Structure used to map between ODBC and internal types. The global	*/
/*  array holds all possible mappings; each time a connection is	*/
/*  opened, a driver specific map is created containing just those	*/
/*  types applicable to the server.					*/


static	ODBCTypeMap	typeMap[] =
{

{	SQL_BIT,		"",	KB::ITFixed,	"Bool",		0			},

{	SQL_TINYINT,		"",	KB::ITFixed,	"TinyInt",	0			},
{	SQL_SMALLINT,		"",	KB::ITFixed,	"SmallInt",	0			},
{	SQL_INTEGER,		"",	KB::ITFixed,	"Integer",	0			},
{	SQL_BIGINT,		"",	KB::ITFixed,	"BigInt",	0			},

{	SQL_REAL,		"",	KB::ITFloat,	"Real",		FF_LENGTH|FF_PREC	},
{	SQL_FLOAT,		"",	KB::ITFloat,	"Float",	FF_LENGTH|FF_PREC	},
{	SQL_DOUBLE,		"",	KB::ITFloat,	"Double",	FF_LENGTH|FF_PREC	},

{	SQL_DECIMAL,		"",	KB::ITFloat,	"Decimal",	FF_LENGTH|FF_PREC	},
{	SQL_NUMERIC,		"",	KB::ITFloat,	"Numeric",	FF_LENGTH|FF_PREC	},

{	SQL_CHAR,		"",	KB::ITString,	"Char",		FF_LENGTH		},
{	SQL_VARCHAR,		"",	KB::ITString,	"VarChar",	FF_LENGTH		},
{	SQL_LONGVARCHAR,	"",	KB::ITString,	"LongVarChar",	FF_LENGTH		},

#if	0
{	SQL_WCHAR,		"",	KB::ITString,	"",		0			},
{	SQL_WVARCHAR,		"",	KB::ITString,	"",		0			},
{	SQL_WLONGVARCHAR,	"",	KB::ITString,	"",		0			},
#endif

{	SQL_BINARY,		"",	KB::ITBinary,	"Binary",	0			},
{	SQL_VARBINARY,		"",	KB::ITBinary,	"VarBinary",	0			},
{	SQL_LONGVARBINARY,	"",	KB::ITBinary,	"LongVarBinary",0			},

{	SQL_DATE,		"",	KB::ITDate,	"Date",		0			},
{	SQL_TIME,		"",	KB::ITTime,	"Time",		0			},
{	SQL_TIMESTAMP,		"",	KB::ITDateTime,	"Date/Time",	0			},

{	SQL_TYPE_DATE,		"",	KB::ITDate,	"Date",		0			},
{	SQL_TYPE_TIME,		"",	KB::ITTime,	"Time",		0			},
{	SQL_TYPE_TIMESTAMP,	"",	KB::ITDateTime,	"Date/Time",	0			},

{	SQL_GUID,		"",	KB::ITDriver,	"GUID",		0,			},
}	;

static	QIntDict<ODBCTypeMap>		dIdentToType	;
static	QList	<ODBCDriverExtn>	dDriverList	;


/*  ------------------------------------------------------------------  */

/*  getInternalType: Get corresponding internal type			*/
/*  type	   : SQLSMALLINT: ODBC type				*/
/*  (returns)	   : KB::IType	: Internal type				*/

static	KB::IType getInternalType
	(	  SQLSMALLINT	type
	)
{
	ODBCTypeMap *ptr = dIdentToType.find (type)    ;
	return	ptr == 0 ? KB::ITUnknown : ptr->kbType ;
}


/*  sqlCodeToCode: Debugging, covert type code to string		*/
/*  code	 : SQLSMALLINT	: Type code				*/
/*  (returns)	 : QString	: Type as string			*/

static	QString	sqlCodeToCode
	(	SQLSMALLINT	code
	)
{
	switch (code)
	{
		case SQL_TINYINT	: return "SQL_TINYINT"		;
		case SQL_SMALLINT	: return "SQL_SMALLINT"		;
		case SQL_INTEGER	: return "SQL_INTEGER"		;
		case SQL_BIGINT		: return "SQL_BIGINT"		;
		case SQL_REAL		: return "SQL_REAL"		;
		case SQL_FLOAT		: return "SQL_FLOAT"		;
		case SQL_DOUBLE		: return "SQL_DOUBLE"		;
		case SQL_BIT		: return "SQL_BIT"		;
		case SQL_DECIMAL	: return "SQL_DECIMAL"		;
		case SQL_NUMERIC	: return "SQL_NUMERIC"		;
		case SQL_CHAR		: return "SQL_CHAR"		;
		case SQL_VARCHAR	: return "SQL_VARCHAR"		;
		case SQL_LONGVARCHAR	: return "SQL_LONGVARCHAR"	;
		case SQL_BINARY		: return "SQL_BINARY"		;
		case SQL_VARBINARY	: return "SQL_VARBINARY"	;
		case SQL_LONGVARBINARY	: return "SQL_LONGVARBINARY"	;
		case SQL_DATE		: return "SQL_DATE"		;
		case SQL_TIME		: return "SQL_TIME"		;
		case SQL_TIMESTAMP	: return "SQL_TIMESTAMP"	;
		case SQL_TYPE_DATE	: return "SQL_TYPE_DATE"	;
		case SQL_TYPE_TIME	: return "SQL_TYPE_TIME"	;
		case SQL_TYPE_TIMESTAMP	: return "SQL_TYPE_TIMESTAMP"	;
		case SQL_GUID		: return "SQL_GUID"		;
		default			: break	 ;
	}

	return	QString("SQL_%1").arg(code) ;
}


inline	int	charToHex
	(	int	ch
	)
{
	return	ch <= '9' ? ch - '0' : ch - 'A' + 10 ;
}

static	SQLINTEGER packBinary
	(	char		*buffer,
		SQLINTEGER	bufflen
	)
{
	SQLINTEGER p1 = 0 ;
	SQLINTEGER p2 = 0 ;

	while (bufflen >= 2)
	{
		buffer[p1] = (charToHex(buffer[p2]) << 4) | charToHex(buffer[p2+1]) ;
		p1	  += 1 ;
		p2	  += 2 ;
		bufflen	  -= 2 ;
	}

	return	p1 ;
}

#define		DBTAG		"KBODBC"
#define		DBTYPE		KBODBC
#define         QRYSELECT	KBODBCQrySelect

#include	"kb_odbcval.cpp"
#include	"kb_odbccheck.cpp"
		
/*  ------------------------------------------------------------------  */

/*  KBODBCType								*/
/*  KBODBCType	: Constructor for PostgreSQL type object		*/
/*  odbcType	: SQLSMALLINT	: KBase internal type			*/
/*  length	: uint		: Database field length			*/
/*  nullOK	: bool		: True if null is OK			*/
/*  (returns)	: KBODBCType	:					*/

KBODBCType::KBODBCType
	(	SQLSMALLINT	odbcType,
		uint		length,
		bool	  	nullOK
	)
	:
	KBType	   ("ODBC", getInternalType(odbcType), length, 0, nullOK),
	m_odbcType (odbcType)
{
}

/*  KBODBCType								*/
/*  isValid	: Test if value is valid for type			*/
/*  value	: const QString & : Value to check			*/
/*  pError	: KBError &	  : Error return			*/
/*  where	: const QString & : Caller				*/
/*  (returns)	: bool		  : Value				*/

bool	KBODBCType::isValid
	(	const QString	&value,
		KBError		&pError,
		const QString	&where
	)
{
	return	KBType::isValid (value, pError, where) ;
}

/*  KBODBCType								*/
/*  getQueryText: Get text for insertion into a query			*/
/*  value	: KBDataArray  * : Raw text of value to convert		*/
/*  d		: KBShared     * : Decoded representation		*/
/*  buffer	: KBDataBuffer & : Result buffer			*/
/*  codec	: QTextCodec *	 : Non-default codec			*/
/*  (returns)	: void		 :					*/

void	KBODBCType::getQueryText
	(	KBDataArray	*value,
		KBShared	*d,
		KBDataBuffer	&buffer,
		QTextCodec	*codec
	)
{
	return	KBType::getQueryText (value, d, buffer, codec) ;
}

/*  ------------------------------------------------------------------  */

namespace ODBC_NS
{
extern	struct	ODBCDriverExtn	ODBCMySQLDriverExtn ;
extern	struct	ODBCDriverExtn	ODBCMSJetDriverExtn ;
}

/*  KBODBC								*/
/*  KBODBC	: Constructor for ODBC database connection class	*/
/*  (returns)	: KBServer	:					*/

KBODBC::KBODBC	 ()
	:
	KBServer ()
{
	if (dIdentToType.count() == 0)
	{
		for (uint idx = 0 ; idx < sizeof(typeMap)/sizeof(ODBCTypeMap) ; idx += 1)
		{
			ODBCTypeMap *m = &typeMap[idx] ;
			dIdentToType.insert (m->odbcType, m) ;
		}

		dDriverList.append (&ODBC_NS::ODBCMySQLDriverExtn) ;
		dDriverList.append (&ODBC_NS::ODBCMSJetDriverExtn) ;
	}

	m_connected	= false ;
	m_envHandle	= 0	;
	m_conHandle	= 0	;
	m_driverExtn	= 0	;
	m_mapCRLF	= false ;
	m_showSysTables	= false ;
	m_mapExpressions= false ;

	m_typeDict.setAutoDelete (true) ;
}

/*  KBODBC								*/
/*  ~KBODBC	: Destructor for PostgresQL database connection class	*/
/*  (returns)	:		:					*/

KBODBC::~KBODBC ()
{
	fprintf
	(	stderr,
		"KBODBC::~KBODBC: con=%d env=%d\n",
		(int)m_conHandle,
		(int)m_envHandle
	)	;

	if (m_conHandle != 0)
	{
		SQLDisconnect (m_conHandle) ;
                SQLFreeHandle (SQL_HANDLE_DBC, m_conHandle) ;
                SQLFreeHandle (SQL_HANDLE_ENV, m_envHandle) ;
	}
}

QString	KBODBC::getAvailableType
	(	int	dummy,
		...
	)
{
	va_list 	aptr	;
	SQLSMALLINT	type	;

	va_start (aptr, dummy)  ;

	while ((type = va_arg (aptr, int)) != 0)
	{
		ODBCTypeMap *t = m_typeDict.find (type) ;
		if (t != 0) return t->odbcName ;
	}

	va_end	 (aptr) ;
	return	 QString::null ;

}

/*  KBODBC								*/
/*  getTypeInfo	: Retrieve type information for underlying RDBMS	*/
/*  (returns)	: bool		: Success				*/

bool	KBODBC::getTypeInfo ()
{
	SQLHSTMT	stmHandle	;
	long		odbcRC		;

	SQLCHAR         typeName	[101]	;
	SQLSMALLINT	typeCode	;
	SQLSMALLINT	typeAuto	;

	if (!getStatement (stmHandle)) return false ;

	odbcRC	= SQLGetTypeInfo (stmHandle, SQL_ALL_TYPES) ;
	if (!ODBCOK(odbcRC))
	{
		SQLFreeStmt (stmHandle, SQL_DROP) ;

		m_lError = KBError
			   (	KBError::Error,
				"Failed to get ODBC type info",
				QString::null,
				__ERRLOCN
			   )	;
		return	false	;
	}

	for (odbcRC = SQLFetch (stmHandle) ; ODBCOK(odbcRC) ; odbcRC = SQLFetch (stmHandle))
	{
		SQLGetData
		(	stmHandle,
			1,
			SQL_C_CHAR,
			&typeName[0],
			sizeof(typeName),
			0
		)	;
		SQLGetData
		(	stmHandle,
			2,
			SQL_C_SHORT,
			&typeCode,
			sizeof(typeCode),
			0
		)	;
		SQLGetData
		(	stmHandle,
			12,
			SQL_C_SHORT,
			&typeAuto,
			sizeof(typeAuto),
			0
		)	;


		fprintf
		(	stderr,
			"ODBC: %4d: %s -> %s [%d]\n",
			typeCode,
			(cchar *)sqlCodeToCode(typeCode),
			typeName,
			typeAuto
			)	;

		if (m_typeDict.find (typeCode) == 0)
		{
			ODBCTypeMap *newMap = new ODBCTypeMap ;
			ODBCTypeMap *oldMap = dIdentToType.find (typeCode) ;

			if (oldMap != 0)
			{
				newMap->odbcType = typeCode ;
				strncpy (newMap->odbcName, (cchar *)typeName, 64) ;
				newMap->odbcName[63] = 0 ;

				newMap->kbType = oldMap->kbType ;
				newMap->kbName = oldMap->kbName ;
				newMap->flags  = oldMap->flags  ;

				m_typeDict.insert (typeCode, newMap) ;
			}
			else	fprintf	(stderr, "ODBC: *** not mapped ***\n") ;
		}

		/* If this type is an auto_unique column then add the	*/
		/* type name to the auto types list. This is separate	*/
		/* from the actual type since the database may have one	*/
		/* type used for both auto_unique and not_auto_unique 	*/
		/* columns (eg., Jet).					*/
		if (typeAuto != 0)
			m_autoTypes.append((cchar *)typeName) ;
	}


	SQLFreeStmt (stmHandle, SQL_DROP) ;

	m_primaryType = getAvailableType (0, SQL_INTEGER, 0) ;
	m_textType    = getAvailableType (0, SQL_VARCHAR, 0) ;
	m_integerType = getAvailableType (0, SQL_INTEGER, 0) ;
	m_blobType    = getAvailableType
			(	0,
				SQL_LONGVARBINARY,
				SQL_VARBINARY,
				SQL_LONGVARCHAR,
				SQL_VARCHAR,
				0
			)	;

	if (m_autoTypes.count() > 0)
		m_primaryType = m_autoTypes[0] ;

	fprintf	(stderr, "m_primaryType : %s\n", (cchar *)m_primaryType) ;
	fprintf	(stderr, "m_textType    : %s\n", (cchar *)m_textType   ) ;
	fprintf	(stderr, "m_integerType : %s\n", (cchar *)m_integerType) ;
	fprintf	(stderr, "m_blobType    : %s\n", (cchar *)m_blobType   ) ;

	return	true ;
}

/*  KBODBC								*/
/*  checkRCOK	: Check whether SQL statement executed correctly	*/
/*  stmHandle	: SQLHSTMT	: Statement handle			*/
/*  odbcRC	: SQLRETURN	: ODBC return code			*/
/*  where	: cchar *	: First text string for possible error	*/
/*  htype	: SQLSMALLINT	: Handle type				*/
/*  (returns)	: bool		: True if statement executed correctly	*/

bool	KBODBC::checkRCOK
	(	SQLHSTMT	stmHandle,
		SQLRETURN	odbcRC,
		cchar		*where,
		SQLSMALLINT	htype
	)
{
	return	::checkRCOK (stmHandle, odbcRC, where, htype, m_lError) ;
}

/*  KBODBC								*/
/*  checkDataOK	 : Check whether SQL statement executed correctly	*/
/*		 : Allows SQL_NO_DATA as OK (eg., update and delete)	*/
/*  stmHandle	 : SQLHSTMT	: Statement handle			*/
/*  db2cliRC	 : long		: ODBC return code			*/
/*  where	 : cchar *	: First text string for possible error	*/
/*  (returns)	 : bool		: True if statement executed correctly	*/

bool	KBODBC::checkDataOK
	(	SQLHSTMT	stmHandle,
		SQLRETURN	odbcRC,
		cchar		*where
	)
{
	return	::checkDataOK (stmHandle, odbcRC, where, m_lError) ;
}

/*  KBODBC								*/
/*  findDataSource							*/
/*		: Find selected data source				*/
/*  (returns)	: void		:					*/

void	KBODBC::findDataSource ()
{
	/* The main purpose of this routine is to scan the ODBC data	*/
	/* sources to locate the one to which we have connected and	*/
	/* thence to find if there is a driver extension for it.	*/
	SQLSMALLINT	direction = SQL_FETCH_FIRST ;
	long		odbcRC	  ;

	SQLCHAR		serverName[256] ;
	SQLSMALLINT	serverLen	;
	SQLCHAR		descrText [256] ;
	SQLSMALLINT	descrLen	;

	m_driverExtn = 0 ;

	for (;;)
	{
		odbcRC = SQLDataSources
			 (	m_envHandle,
				direction,
				serverName,
				sizeof(serverName),
				&serverLen,
				descrText,
				sizeof(descrText),
				&descrLen
			 )	;

		if (!ODBCOK(odbcRC)) break ;

		fprintf
		(	stderr,
			"KBODBC::findDataSource: got [%s][%s]\n",
			(cchar *)serverName,
			(cchar *)descrText
		)	;

		if ((cchar *)serverName == m_database)
		{
			fprintf
			(	stderr,
				"KBODBC::findDataSource: matched [%s][%s]\n",
				(cchar *)serverName,
				(cchar *)descrText
			)	;

			LITER
			(	ODBCDriverExtn,
				dDriverList,
				de,

				fprintf
				(	stderr,
					"KBODBC::findDataSource: check [%s][%s]\n",
					(cchar *)descrText,
					(cchar *)de->match
				)	;

				if (QString((cchar *)descrText).find(QRegExp(de->match, false)) >= 0)
				{
					m_driverExtn = de ;
					fprintf	(stderr, "........ mapped driver\n") ;
					break	;
				}
			)
		}

		direction = SQL_FETCH_NEXT ;
	}	
}

/*  KBODBC								*/
/*  doConnect	: Open connection to database				*/
/*  svInfo	: KBServerInfo *: Server information			*/
/*  (returns)	: KBServer	:					*/

bool	KBODBC::doConnect
	(	KBServerInfo	*svInfo
	)
{
	long	odbcRC	;

	m_readOnly = svInfo->readOnly() ;

	if (svInfo->advanced() != 0)
		if (svInfo->advanced()->isType("odbc"))
		{
			KBODBCAdvanced *a = (KBODBCAdvanced *)svInfo->advanced() ;
			m_mapCRLF	  = a->m_mapCRLF	;
			m_showSysTables   = a->m_showSysTables  ;
			m_mapExpressions  = a->m_mapExpressions	;
			m_odbcType	  = a->m_odbcType	;
		}
		else	/* If the type is wrong then show a warning.	*/
			/* This should never happen unless someone	*/
			/* hand edits the XML database file.		*/
			KBError::EError
			(	TR("Driver error"),
				TR("Invalid advanced options, ignoring"),
				__ERRLOCN
			)	;

	/* Check that we are not already connected. Note that the	*/
	/* notConnected routine sets an error message.			*/
	if (m_conHandle != 0)
	{	m_lError  = KBError
			  (	KBError::Error,
				"Already connected to ODBC database",
				QString::null,
				__ERRLOCN
			  )	;
		return	false	;
	}

	/* Allocate Environment handle and register our version with	*/
	/* the ODBC subsystem.						*/
        odbcRC	= SQLAllocHandle
		  (	SQL_HANDLE_ENV,
			SQL_NULL_HANDLE,
			&m_envHandle
		  )	;

        if (!ODBCOK(odbcRC))
	{
		m_lError  = KBError
			  (	KBError::Error,
				"Failed to allocate ODBC environment handle",
				QString::null,
				__ERRLOCN
			  )	;
		return	false	;
        }

	odbcRC	= SQLSetEnvAttr
		  (	m_envHandle,
			SQL_ATTR_ODBC_VERSION, 
                        (void *)SQL_OV_ODBC3,
			0
		  )	; 

        if (!checkRCOK(m_envHandle, odbcRC, "Error registering with ODBC", SQL_HANDLE_ENV))
        {
		SQLFreeHandle (SQL_HANDLE_ENV, m_envHandle) ;
		return	false ;
        }

	fprintf
	(	stderr,
		"KBODBC::doConnect - about to connect\n"
		"        database [%s]\n"
		"        user     [%s]\n"
		"        password [%s]\n",
		(cchar *)m_database,
		(cchar *)m_user,
		(cchar *)m_password
	)	;


	/* SPECIAL CASE:						*/
	/* If the database name is empty then we are nominally opening	*/
	/* simply in order to list databases, or in the case of ODBC,	*/
	/* data source names. In this case do not actually connect.	*/
	if (m_database.isEmpty())
	{
		m_connected = true ;
		return	true	   ;
	}

	/* Next allocate a connection handle, and set a timeout on the	*/
	/* connection. Then we can see about getting the actual		*/
	/* connection itself.						*/
        odbcRC	= SQLAllocHandle
		  (	SQL_HANDLE_DBC,
			m_envHandle,
			&m_conHandle
		  )	; 

        if (!checkRCOK(m_envHandle, odbcRC, "Error getting ODBC connection handle", SQL_HANDLE_ENV))
        {
		SQLFreeHandle (SQL_HANDLE_ENV, m_envHandle) ;
		return	false ;
        }

        SQLSetConnectAttr
	(	m_conHandle,
		SQL_LOGIN_TIMEOUT,
		(SQLPOINTER *)5,
		0
	)	;

        odbcRC	= SQLConnect
		  (	m_conHandle,
			(SQLCHAR *)(cchar *)m_database, SQL_NTS,
			(SQLCHAR *)(cchar *)m_user,     SQL_NTS,
			(SQLCHAR *)(cchar *)m_password, SQL_NTS
		  )	;

        if (!checkRCOK(m_conHandle, odbcRC, "Error connecting to ODBC data source", SQL_HANDLE_DBC))
        {
                SQLFreeHandle (SQL_HANDLE_DBC, m_conHandle) ;
                SQLFreeHandle (SQL_HANDLE_ENV, m_envHandle) ;
		m_envHandle	= 0	;
		m_conHandle	= 0	;
		return	false	;
        }

	m_connected = true	;

	char	    infob[64]	;
	SQLSMALLINT infolen	;

	odbcRC	= SQLGetInfo
		(	m_conHandle,
			SQL_IDENTIFIER_CASE,
			infob,
			sizeof(infob),
			&infolen
		)	;

        if (!checkRCOK(m_conHandle, odbcRC, "Error getting case preservation", SQL_HANDLE_DBC))
        {
		SQLDisconnect (m_conHandle) ;
                SQLFreeHandle (SQL_HANDLE_DBC, m_conHandle) ;
                SQLFreeHandle (SQL_HANDLE_ENV, m_envHandle) ;
		m_envHandle	= 0	;
		m_conHandle	= 0	;
		return	false	;
        }

	m_caseMapping = *(SQLUSMALLINT *)&infob[0] ;
	switch (m_caseMapping)
	{
		case SQL_IC_UPPER :
		case SQL_IC_LOWER :
		case SQL_IC_MIXED :
			fprintf	(stderr, "KBODBC::doConnect: looses case\n") ;
			m_keepsCase = false ;
			break	  ;

		default	:
			fprintf	(stderr, "KBODBC::doConnect: keeps case\n" ) ;
			m_keepsCase = true  ;
			break	  ;
	}

	fprintf
	(	stderr,
		"KBODBC::doConnect(%s): keeps_case=%d (%d)\n",
		(cchar *)svInfo->serverName(),
		m_keepsCase,
		*(SQLUSMALLINT *)&infob[0]
	)	;

	
	if (!getTypeInfo ())
	{
		SQLDisconnect (m_conHandle) ;
                SQLFreeHandle (SQL_HANDLE_DBC, m_conHandle) ;
                SQLFreeHandle (SQL_HANDLE_ENV, m_envHandle) ;
		m_envHandle	= 0	;
		m_conHandle	= 0	;
		return	false	;
	}

	if	(m_odbcType == "MySQL")
		m_driverExtn = &ODBCMySQLDriverExtn ;
	else if (m_odbcType == "Jet"  )
		m_driverExtn = &ODBCMSJetDriverExtn ;
	else
		findDataSource () ;

	return	true	;
}

/*  KBODBC								*/
/*  keepsCase	: Check whether server preserves name case		*/
/*  (returns)	: bool		: True if so				*/

bool	KBODBC::keepsCase ()
{
	return	m_keepsCase	;
}

/*  KBODBC								*/
/*  qrySelect	: Open a select query					*/
/*  data	: bool		: Query for data			*/
/*  select	: const QString&: Select query				*/
/*  update	: bool		: Select for update			*/
/*  (returns)	: KBSQLSelect * : Select query class or NULL on error	*/

KBSQLSelect
	*KBODBC::qrySelect
	(	bool		data,
		const QString	&select,
		bool		update
	)
{
	return	m_driverExtn == 0 ?
			new KBODBCQrySelect 	   (this, data, select, update) :
			(*m_driverExtn->qrySelect) (this, data, select, update) ;
}


/*  KBODBC								*/
/*  qryUpdate	: Open an update query					*/
/*  data	: bool		 : Query for data			*/
/*  update	: const QString &: Update query				*/
/*  tabName	: const QString &: Table name				*/
/*  (returns)	: KNQryUpdate *  : Update query class or NULL on error	*/

KBSQLUpdate
	*KBODBC::qryUpdate
	(	bool		data,
		const QString	&update,
		const QString	&tabName
	)
{
	if (m_readOnly)
	{
		m_lError = KBError
			   (	KBError::Error,
			 	TR("Database is read-only"),
			 	TR("Attempting update query"),
			 	__ERRLOCN
			  )	;
		return	 0 ;
	}

	return	m_driverExtn == 0 ?
			new KBODBCQryUpdate 	   (this, data, update, tabName) :
			(*m_driverExtn->qryUpdate) (this, data, update, tabName) ;
}

/*  KBODBC								*/
/*  qryInsert	: Open an insert query					*/
/*  data	: bool		 : Query for data			*/
/*  insert	: const QString &: Insert query				*/
/*  tabName	: const QString &: Table name				*/
/*  (returns)	: KNQryInsert *  : Insert query class or NULL on error	*/

KBSQLInsert
	*KBODBC::qryInsert
	(	bool		data,
		const QString	&insert,
		const QString	&tabName
	)
{
	if (m_readOnly)
	{
		m_lError = KBError
			   (	KBError::Error,
			 	TR("Database is read-only"),
			 	TR("Attempting insert query"),
			 	__ERRLOCN
			  )	;
		return	 0 ;
	}

	return	m_driverExtn == 0 ?
			new KBODBCQryInsert 	   (this, data, insert, tabName) :
			(*m_driverExtn->qryInsert) (this, data, insert, tabName) ;
}

/*  KBODBC								*/
/*  qryDelete	: Open a delete query					*/
/*  data	: bool		 : Query for data			*/
/*  _delete	: const QString &: Delete query				*/
/*  tabName	: const QString &: Table name				*/
/*  (returns)	: KNQryDelete *  : Delete query class or NULL on error	*/

KBSQLDelete
	*KBODBC::qryDelete
	(	bool		data,
		const QString	&_delete,
		const QString	&tabName
	)
{
	if (m_readOnly)
	{
		m_lError = KBError
			   (	KBError::Error,
			 	TR("Database is read-only"),
			 	TR("Attempting delete query"),
			 	__ERRLOCN
			  )	;
		return	 0 ;
	}

	return	m_driverExtn == 0 ?
			new KBODBCQryDelete 	   (this, data, _delete, tabName) :
			(*m_driverExtn->qryDelete) (this, data, _delete, tabName) ;
}

/*  KBODBC								*/
/*  command	: Execute arbitrary SQL					*/
/*  data	: bool		  : Querying for data			*/
/*  query	: const QString & : Query text				*/
/*  nvals	: uint		  : Number of substitution values	*/
/*  values	: KBValue *	  : Substitution values			*/
/*  select	: KBSQLSelect **  : Return for data queries		*/
/*  (returns)	: bool		  : Success				*/

bool	KBODBC::command
	(	bool		data,
		const QString	&query,
		uint		,
		KBValue		*,
		KBSQLSelect	**select
	)
{
	SQLHSTMT stmHandle ;
	long	 odbcRC	   ;

	if (!getStatement (stmHandle))
		return false ;

	SQLCHAR	*q = (SQLCHAR *)(cchar *)query ;

	odbcRC	= SQLExecDirect (stmHandle, q, strlen((cchar *)q)) ;
	if (!checkRCOK(stmHandle, odbcRC, "SQL command execution"))
	{
		SQLFreeStmt (stmHandle, SQL_DROP) ;
		return	false	;
	}

	if (select == 0)
	{
		SQLFreeStmt (stmHandle, SQL_DROP) ;
		return	true	;
	}

	SQLSMALLINT	_m_nFields	;
	SQLNumResultCols(stmHandle, &_m_nFields) ;

	if (_m_nFields == 0)
	{
		*select	= 0	;
		SQLFreeStmt (stmHandle, SQL_DROP) ;
		return	true	;
	}

	bool		ok ;
	KBODBCQrySelect	*s ;

	s = new KBODBCQrySelect (this, stmHandle, data, query, ok) ;

	if (!ok)
	{
		m_lError = s->lastError() ;
		delete	 s	 ;
		*select	 = 0	 ;
		return	 false	 ;
	}

	*select	= s	;
	return	true	;
}


/*  KBODBC								*/
/*  getStatement: Get statement handle					*/
/*  handle	: SQLHSTMT &	: Statement handle return		*/
/*  (returns)	: bool		: Success				*/

bool	KBODBC::getStatement
	(	SQLHSTMT	&handle
	)
{
	long	odbcRC	= SQLAllocStmt (m_conHandle, &handle) ;

	if (!checkRCOK(m_conHandle, odbcRC, "Unable to get ODBC statement handle", SQL_HANDLE_DBC))
	{
		handle	= 0	;
		return	false	;
	}

	return	true	;
}

/*  KBODBC								*/
/*  tableExists	: See if named table exists				*/
/*  tabName	: const QString & : Table name				*/
/*  exists	: bool &	  : True if table exists		*/
/*  (returns)	: bool		  : Success				*/

bool	KBODBC::tableExists
	(	const QString	&tabName,
		bool		&exists
	)
{
	KBTableDetailsList	tabList ;

	fprintf	(stderr, "KBODBC::tableExists (%s).....\n", (cchar *)tabName) ;
	if (!doListTables (tabList, tabName, true, KB::IsTable))
		return	false	;

	exists	= tabList.count() > 0 ;
	fprintf	(stderr, "KBODBC::tableExists -> %d\n", exists) ;
	return	true ;
}

bool	KBODBC::listDatabases
	(	QStringList	&dbList
	)
{
	/* The main purpose of this routine is to scan the ODBC data	*/
	/* sources to locate the one to which we have connected and	*/
	/* thence to find if there is a driver extension for it.	*/
	SQLSMALLINT	direction = SQL_FETCH_FIRST ;
	long		odbcRC	  ;

	SQLCHAR		serverName[256] ;
	SQLSMALLINT	serverLen	;
	SQLCHAR		descrText [256] ;
	SQLSMALLINT	descrLen	;

	for (;;)
	{
		odbcRC = SQLDataSources
			 (	m_envHandle,
				direction,
				serverName,
				sizeof(serverName),
				&serverLen,
				descrText,
				sizeof(descrText),
				&descrLen
			 )	;

		if (!ODBCOK(odbcRC)) return false ;

		fprintf
		(	stderr,
			"KBODBC::listDatabases: got [%s][%s]\n",
			(cchar *)serverName,
			(cchar *)descrText
		)	;

		dbList.append ((cchar *)serverName) ;
		direction = SQL_FETCH_NEXT ;
	}

	return	true	;
}

/*  KBODBC								*/
/*  doListTables: List tables in database				*/
/*  tabList	: KBTableDetailsList &					*/
/*				  : Result list				*/
/*  tabName	: const QString & : Specific table name			*/
/*  alltables	: bool		  : List all tables			*/
/*  getTypes	: uint		  : Type flags				*/
/*  (returns)	: bool		  : Success				*/

bool	KBODBC::doListTables
	(	KBTableDetailsList	&tabList,
		const QString		&tabName,
		bool			alltables,
		uint			getTypes
	)
{
	SQLHSTMT        stmHandle	;
	SQLCHAR		name	[101]   = "" ;
	SQLCHAR		type	[101]   = "" ;
	SQLCHAR		remarks	[301]	= "" ;
	SQLINTEGER	indName		;
	SQLINTEGER	indType		;
	SQLINTEGER	indRemarks	;

	long		odbcRC		;

	KB::TableType	tabType		;

	/* Create an SQL statement and then execute the SQLTables	*/
	/* routine whicn retrieves the list of tables.			*/
	if (!getStatement (stmHandle)) return false ;


	odbcRC	= SQLTables (stmHandle, 0, 0, 0, 0, 0, 0, 0, 0 ) ;

	if (!checkRCOK(stmHandle, odbcRC, "Failed to retrieve ODBC table list"))
	{
		SQLFreeStmt   (stmHandle, SQL_DROP) ;
		return	false ;
	}


	SQLBindCol (stmHandle, SQLTables_TABLE_NAME, SQL_C_CHAR, name,    sizeof(name   ), &indName   ) ;
	SQLBindCol (stmHandle, SQLTables_TABLE_TYPE, SQL_C_CHAR, type,    sizeof(type   ), &indType   ) ;
	SQLBindCol (stmHandle, SQLTables_REMARKS,    SQL_C_CHAR, remarks, sizeof(remarks), &indRemarks) ;

	name	[0] = 0 ;
	type	[0] = 0 ;
	remarks	[0] = 0 ;
	odbcRC	    = SQLFetch (stmHandle) ;

	while (ODBCOK(odbcRC))
	{
		QString	thisName ;

		if (indName == SQL_NULL_DATA)
			thisName = "UnknownTableName" ;
		else	thisName = (cchar *)name ;

		fprintf
		(	stderr,
			"KBODBC::doListTables: [%s][%s][%s]\n",
			(cchar *)thisName,
			type,
			remarks
		) ;

		if	(strcmp ((cchar *)type,        "VIEW" ) == 0)
		{
			if ((getTypes & KB::IsView ) == 0)
				goto getNext ;

			tabType	= KB::IsView  ;
		}
		else if	(strcmp ((cchar *)type, "SYSTEM TABLE") == 0)
		{
			if ((getTypes & KB::IsTable) == 0)
				goto getNext ;

			if (!m_showSysTables)
				goto getNext ;

			tabType	= KB::IsTable ;
		}
		else if (strcmp ((cchar *)type,        "TABLE") == 0)
		{
			if ((getTypes & KB::IsTable) == 0)
				goto getNext ;

			tabType	= KB::IsTable ;
		}
		else
		{
			goto getNext ;
		}

		if (!alltables)
			if (thisName.left(8).lower() == "__rekall")
				thisName = QString::null ;

		if (!tabName.isEmpty())
			if (m_keepsCase ? thisName != tabName : thisName.lower() != tabName.lower())
				thisName = QString::null ;

		if (!thisName.isNull())
			tabList.append
			(	KBTableDetails
				(	thisName,
					tabType,
					QP_SELECT|QP_INSERT|QP_UPDATE|QP_DELETE
			)	)	;


		getNext:

			name	[0] = 0 ;
			type	[0] = 0 ;
			remarks	[0] = 0 ;
			odbcRC	    = SQLFetch (stmHandle) ;
	}

	SQLFreeStmt  (stmHandle, SQL_DROP) ;

	return	true	;
}

/*  KBODBC								*/
/*  doListTables: List tables in database				*/
/*  tabList	: KBTableDetailsList &					*/
/*				: Result list				*/
/*  alltables	: bool		: List all tables			*/
/*  type	: uint		: Type flags				*/
/*  (returns)	: bool		: Success				*/

bool	KBODBC::doListTables
	(	KBTableDetailsList	&tabList,
		uint			type
	)
{
	return	doListTables (tabList, QString::null, m_showAllTables, type) ;
}


/*  KBODBC								*/
/*  doListFields: List fields in table					*/
/*  tabSpec	: KBTableSpec &	: Table specification			*/
/*  (returns)	: bool		: Success				*/

bool	KBODBC::doListFields
	(	KBTableSpec	&tabSpec
	)
{
	SQLHSTMT	stmHandle	;
	SQLCHAR		table	[101]	= "" 	;
	SQLCHAR         column	[101]	= "" 	;
	SQLCHAR		typeName[101]	= "" 	;
	SQLCHAR         priv	[101]	= "" 	;

	SQLSMALLINT	type		;
	SQLINTEGER	size		;
	SQLSMALLINT	decimal		;
	SQLSMALLINT	nullable	;
	SQLCHAR		defval	[512]	;

	uint		idx		= 0	;
	long		odbcRC1		;
	long		odbcRC2		;

	int		possibleKey	= -1	;
	bool		hasColDef	= false	;
	uint		flags		= m_driverExtn == 0 ?
						0 : m_driverExtn->flags ;

	fprintf
	(	stderr,
		"KBODBC::doListFields: m_driverExtn=%p flags=%08x\n",
		(void *)m_driverExtn,
		flags
	)	;

	if ((flags & ODBC_FLAGS_USED) != 0)
	{
		hasColDef = (flags & ODBC_FLAGS_COLDEF) != 0 ;
	}


	tabSpec.m_fldList.clear () ;
	tabSpec.m_prefKey = -1  ;


	/* ------------------------------------------------------------ */
	/* Part A:							*/
	/* Start of the obvious way, use the SQLColumns call to pick up	*/
	/* the list of columns and all the basic column inforamtion.	*/
	if (!getStatement (stmHandle)) return false ;

	switch (m_caseMapping)
	{
		case SQL_IC_UPPER :
			strncpy	((char *)table, tabSpec.m_name.upper(), 100) ;
			break	;

		case SQL_IC_LOWER :
			strncpy	((char *)table, tabSpec.m_name.lower(), 100) ;
			break	;

		default	:
			strncpy	((char *)table, tabSpec.m_name, 100) ;
			break	;
	}


	table[100] = 0 ;

	odbcRC1	= SQLColumns (stmHandle, 0, 0, 0, 0, table, SQL_NTS, 0, 0 ) ;

	if (!checkRCOK(stmHandle, odbcRC1, "Failed to get ODBC column list"))
	{
		SQLFreeStmt (stmHandle, SQL_DROP) ;
		return	false ;
	}

	for (odbcRC1 = SQLFetch (stmHandle) ; ODBCOK(odbcRC1) ; odbcRC1 = SQLFetch (stmHandle))
	{
		column	[0] = 0 ;
		defval	[0] = 0 ;

		odbcRC2  = SQLGetData
			   (	stmHandle,
				SQLColumns_COLUMN_NAME,
				SQL_C_CHAR,
				&column[0],
				sizeof(column),
				0
			   )	;

		if (!ODBCOK(odbcRC2))
			strcpy ((char *)column, "Unknown") ;

		SQLGetData
		(	stmHandle,
			SQLColumns_DATA_TYPE,
			SQL_C_SHORT,
			&type,
			sizeof(type),
			0
		)	;
		SQLGetData
		(	stmHandle,
			SQLColumns_TYPE_NAME,
			SQL_C_CHAR,
			&typeName,
			sizeof(typeName),
			0
		)	;
		SQLGetData
		(	stmHandle,
			SQLColumns_COLUMN_SIZE,
			SQL_C_SLONG,
			&size,
			sizeof(size),
			0
		)	;
		SQLGetData
		(	stmHandle,
			SQLColumns_DECIMAL_DIGITS,
			SQL_C_SHORT,
			&decimal,
			sizeof(decimal),
			0
		)	;
		SQLGetData
		(	stmHandle,
			SQLColumns_NULLABLE,
			SQL_C_SHORT,
			&nullable,
			sizeof(nullable),
			0
		)	;

		if (hasColDef)
			SQLGetData
			(	stmHandle,
				SQLColumns_COLUMN_DEF,
				SQL_C_CHAR,
				&defval,
				sizeof(defval),
				0
			)	;


		ODBCTypeMap	*ptr	= dIdentToType.find (type) ;
		QString		ktype	= ptr == 0 ? QString("<%1>").arg(type) : QString(ptr->kbName) ;
		KB::IType	itype	= ptr == 0 ? KB::ITUnknown : ptr->kbType ;
		uint		flags	= 0 ;


		if (nullable == SQL_NO_NULLS)
			flags |= KBFieldSpec::NotNull ;


		/* Any column which is auto_unique is marked as serial	*/
		/* and as read-only, the latter since this is almost	*/
		/* certainly what we want.				*/
		if (m_autoTypes.findIndex((cchar *)typeName) >= 0)
			flags |= KBFieldSpec::Serial  | KBFieldSpec::ReadOnly ;


		fprintf
		(	stderr,
			"ODBC: field   [%s]->[%s][%s][%s]\n",
			(cchar *)tabSpec.m_name,
			(cchar *)column,
			(cchar *)typeName,
			(cchar *)defval
		)	;

		KBFieldSpec *s = new KBFieldSpec
				 (	(uint)idx,
					(cchar *)column,
					ktype,
					itype,
					flags,
					size,
					decimal
				 )	;

		if (defval[0] != 0)
			s->m_defval = (cchar *)defval ;


		/* If the type name is an auto_unique type then note	*/
		/* this column as possibly the preferred key; we defer	*/
		/* the actual decision to after the SQLPrimaryKeys call	*/
		if (m_autoTypes.findIndex((cchar *)typeName) >= 0)
			possibleKey = tabSpec.m_fldList.count() ;


		tabSpec.m_fldList.append (s) ;
		idx	+= 1	;
	}

	SQLFreeStmt (stmHandle, SQL_DROP) ;


	/* Scan table for primary key columns. This uses the ODBC call	*/
	/* SQLPrimaryKeys, though why this can't be returned with the	*/
	/* SQLColumns call is anybodies guess.				*/
	if (!getStatement (stmHandle)) return false ;

	odbcRC1	= SQLPrimaryKeys (stmHandle, 0, 0, 0, 0, table, SQL_NTS) ;

	if (!checkRCOK(stmHandle, odbcRC1, "Failed to get ODBC primary key list"))
	{
		SQLFreeStmt (stmHandle, SQL_DROP) ;
		return	false ;
	}

	for (odbcRC1 = SQLFetch (stmHandle) ; ODBCOK(odbcRC1) ; odbcRC1 = SQLFetch (stmHandle))
	{
		odbcRC2  = SQLGetData
			   (	stmHandle,
				4,
				SQL_C_CHAR,
				&column[0],
				sizeof(column),
				0
			   )	;

		if (!ODBCOK(odbcRC2)) continue ;

		fprintf
		(	stderr,
			"ODCB: primary [%s]->[%s]\n",
			(cchar *)tabSpec.m_name,
			(cchar *)column
		)	;

		for (idx = 0 ; idx < tabSpec.m_fldList.count() ; idx += 1)
		{
			KBFieldSpec *fspec = tabSpec.m_fldList.at(idx) ;

			if (fspec->m_name != QString((cchar *)column))
				continue ;

			fprintf	(stderr, "    : matched %d\n", idx) ;

			fspec->m_flags |= KBFieldSpec::Primary |
					  KBFieldSpec::Unique  |
					  KBFieldSpec::NotNull ;
			tabSpec.m_prefKey = idx ;
			break	;
		}
	}

	SQLFreeStmt (stmHandle, SQL_DROP) ;


	/* If we don't now have a primary key *but* there is a possible	*/
	/* one (based on auto_unique columns) then set that one now.	*/
	if (tabSpec.m_prefKey < 0)
		if (possibleKey > 0)
			tabSpec.m_prefKey = possibleKey ;



	/* Scan table for set of columns that will uniquely identify a	*/
	/* row in the table.						*/
	/*								*/
	if (!getStatement (stmHandle)) return false ;

	fprintf	(stderr, "ODBC: calling SQLSpecialColumns(SQL_BEST_ROWID)\n") ;
	odbcRC1	= SQLSpecialColumns (stmHandle, SQL_BEST_ROWID, 0, 0, 0, 0, table, SQL_NTS, 0, 0) ;

	if (!checkRCOK(stmHandle, odbcRC1, "Failed to get ODBC special columns"))
	{
		SQLFreeStmt (stmHandle, SQL_DROP) ;
		return	false ;
	}

	for (odbcRC1 = SQLFetch (stmHandle) ; ODBCOK(odbcRC1) ; odbcRC1 = SQLFetch (stmHandle))
	{
		odbcRC2  = SQLGetData
			   (	stmHandle,
				2,
				SQL_C_CHAR,
				&column[0],
				sizeof(column),
				0
			   )	;

		if (!ODBCOK(odbcRC2)) continue ;

		fprintf	(stderr, "ODBC: SQL_BEST_ROWID [%s]\n", (cchar *)column) ;

		KBFieldSpec *fSpec = tabSpec.findField ((cchar *)column) ;
		if (fSpec != 0) fSpec->m_flags |= KBFieldSpec::Unique  ;
	}

	SQLFreeStmt (stmHandle, SQL_DROP) ;

	/* Scan table for columns that are automatically updated when a	*/
	/* row is updated in the database.				*/
	/*								*/
	if (!getStatement (stmHandle)) return false ;

	fprintf	(stderr, "ODBC: calling SQLSpecialColumns(SQL_ROWVER)\n") ;
	odbcRC1	= SQLSpecialColumns (stmHandle, SQL_ROWVER, 0, 0, 0, 0, table, SQL_NTS, 0, 0) ;

	if (!checkRCOK(stmHandle, odbcRC1, "Failed to get ODBC special columns"))
	{
		SQLFreeStmt (stmHandle, SQL_DROP) ;
		return	false ;
	}

	for (odbcRC1 = SQLFetch (stmHandle) ; ODBCOK(odbcRC1) ; odbcRC1 = SQLFetch (stmHandle))
	{
		odbcRC2  = SQLGetData
			   (	stmHandle,
				2,
				SQL_C_CHAR,
				&column[0],
				sizeof(column),
				0
			   )	;

		if (!ODBCOK(odbcRC2)) continue ;

		fprintf	(stderr, "ODBC: SQL_ROWVER [%s]\n", (cchar *)column) ;
	}

	SQLFreeStmt (stmHandle, SQL_DROP) ;

	/* Scan table for column privileges, so we can find things like	*/
	/* read-only columns.						*/
	if (!getStatement (stmHandle)) return false ;

	fprintf	(stderr, "ODBC: calling SQLColumnPrivileges\n") ;
	odbcRC1	= SQLColumnPrivileges (stmHandle, 0, 0, 0, 0, table, SQL_NTS, 0, 0) ;

	if (!checkRCOK(stmHandle, odbcRC1, "Failed to get ODBC column privileges"))
	{
		SQLFreeStmt (stmHandle, SQL_DROP) ;
		return	false ;
	}

	for (odbcRC1 = SQLFetch (stmHandle) ; ODBCOK(odbcRC1) ; odbcRC1 = SQLFetch (stmHandle))
	{
		odbcRC2  = SQLGetData
			   (	stmHandle,
				4,
				SQL_C_CHAR,
				&column[0],
				sizeof(column),
				0
			   )	;

		if (!ODBCOK(odbcRC2)) continue ;

		odbcRC2  = SQLGetData
			   (	stmHandle,
				7,
				SQL_C_CHAR,
				&priv[0],
				sizeof(priv),
				0
			   )	;

		if (!ODBCOK(odbcRC2)) continue ;

		fprintf
		(	stderr,
			"ODBC: Priv: [%s]->[%s]\n",
			(cchar *)column,
			(cchar *)priv
		)	;

//		KBFieldSpec *fSpec = tabSpec.findField ((cchar *)column) ;
	}

	SQLFreeStmt (stmHandle, SQL_DROP) ;


	if (m_driverExtn)
		if (!m_driverExtn->doListFields (this, tabSpec, m_lError))
			return	false	;


	/* If there is no primary key column then the preferred key can	*/
	/* be a unique key column.					*/
	if (tabSpec.m_prefKey < 0)
		for (uint idx = 0 ; idx < tabSpec.m_fldList.count() ; idx += 1)
		{	KBFieldSpec *fSpec = tabSpec.m_fldList.at(idx) ;
			if ((fSpec->m_flags & KBFieldSpec::Unique) != 0)
			{	tabSpec.m_prefKey = idx ;
				break	;
			}
		}


	return	true	;
}


/*  KBODBC								*/
/*  execSQL	: Simple SQL execution with no placeholders		*/
/*  sql		: const QString & : SQL statement			*/
/*  where	: cchar *	: First text string for possible error	*/
/*  (returns)	: bool		: True if statement executed correctly	*/

bool	KBODBC::execSQL
	(	const QString	&sql,
		cchar		*where
	)
{
	SQLHSTMT stmHandle ;
	long	 odbcRC	   ;

	if (!getStatement (stmHandle))
		return false ;

	SQLCHAR	*s = (SQLCHAR *)(cchar *)sql ;

	odbcRC	= SQLExecDirect (stmHandle, s, strlen((cchar *)s)) ;
	if (!checkRCOK(stmHandle, odbcRC, where))
	{
		SQLFreeStmt (stmHandle, SQL_DROP) ;
		return	false	;
	}

	printQuery (sql, 0, 0, true) ;

	SQLFreeStmt (stmHandle, SQL_DROP) ;
	return	true ;
}

/*  KBODBC								*/
/*  doCreateTable: Create a new table					*/
/*  tabSpec	 : KBTableSpec & : Table specification			*/
/*  assoc	 : bool		 : Create associated stuff		*/
/*  best	 : bool		 : Use best match			*/
/*  (returns)	 : bool		 : Success				*/

bool	KBODBC::doCreateTable
	(	KBTableSpec	&tabSpec,
		bool		,
		bool
	)
{
	cchar	*sep	= ""	;
	QString	create	= QString ("create table %1\n(").arg(tabSpec.m_name) ;


	/* Main loop generates SQL create for the columns and column	*/
	/* types, plus options like auto-increment and primary key.	*/
	LITER
	(	KBFieldSpec,
		tabSpec.m_fldList,
		fSpec,

		/* Special cases. If the field is "Primary Key" then we	*/
		/* create an "Int4" column marked as primary key, not	*/
		/* null and auto increment ...				*/
		QString	ftype	= fSpec->m_typeName ;
		if (ftype == "Primary Key")
		{
			create += QString ("%1\t%2 %3 primary key")
					  .arg(sep	    )
					  .arg(fSpec->m_name)
					  .arg(m_primaryType) ;
			sep	= ",\n"	;
			continue   ;
		}
		/* ... while a foreign key is also "Int4" not null.	*/
		if (ftype == "Foreign Key")
		{
			create += QString ("%1\t%2 %3 not null")
					  .arg(sep	    )
					  .arg(fSpec->m_name)
					  .arg(m_primaryType) ;
			sep	= ",\n"	;
			continue   ;
		}

		/* Map the types used when creating the object and	*/
		/* design dictionary tables.				*/
		if	(ftype == "_Text"   ) ftype = "VarChar"	;
		else if (ftype == "_Integer") ftype = "Integer"	;
		else if (ftype == "_Binary" ) ftype = "Binary"	;


		/* Scan though the RDBMS specific mapping table	looking	*/
		/* for the type type.					*/
		QIntDictIterator<ODBCTypeMap> typeIter (m_typeDict) ;
		ODBCTypeMap *map ;

		while ((map = typeIter.current()) != 0)
		{
			if (map->kbName == ftype)
				if ((map->flags & FF_NOCREATE) == 0)
					break	;

			typeIter += 1 ;
		}


		if (map == 0)
		{	m_lError = KBError
				   (	KBError::Fault,
				  	"Error mapping column type",
				  	QString ("Type %1 for column %2 not known")
						.arg(ftype)
						.arg(fSpec->m_name),
				  	__ERRLOCN
				   )	;
			return	false	;
		}


		create += QString ("%1\t%2 %3").arg(sep        )
					       .arg(fSpec->m_name)
					       .arg(map  ->odbcName) ;

		if ((map->flags & FF_LENGTH) != 0)
		{
			create	+= QString("(%1").arg(fSpec->m_length) ;
			if ((map->flags & FF_PREC) != 0)
				create	+= QString(",%1").arg(fSpec->m_prec) ;
			create	+= ")" ;
		}

		if ((fSpec->m_flags & KBFieldSpec::NotNull) != 0) create += " not null" ;
		if ((fSpec->m_flags & KBFieldSpec::Primary) != 0) create += " primary key auto_increment" ;

		sep	= ",\n"	;
	)

#if	0
	/* Now add any unique columns. Note that we do not handle	*/
	/* multiple-column uniqueness.					*/
	LITER
	(	KBFieldSpec,
		fldList,
		fSpec,

		if ((fSpec->m_flags & KBFieldSpec::Unique ) != 0)
		{
			create	+= sep	;
			create	+= QString("unique (%1)").arg(fSpec->m_name) ;
			sep	= ",\n"	;
		}
		if ((fSpec->m_flags & KBFieldSpec::Indexed) != 0)
		{
			create	+= sep	;
			create	+= QString("index  (%1)").arg(fSpec->m_name) ;
			sep	= ",\n"	;
		}
	)
#endif
	create	+= "\n)";

	fprintf	(stderr, "ODBC:\n%s\n", (cchar *)create) ;

	return	execSQL (create, "Error creating table") ;
}

/*  KBODBC								*/
/*  doRenameTable: Rename a  table					*/
/*  oldName	 : cchar *	: Current table name			*/
/*  newName	 : cchar *	: New table name			*/
/*  assoc	 : bool		: Rename associated stuff - none here	*/
/*  (returns)	 : bool		: Success				*/

bool	KBODBC::doRenameTable
	(	cchar	*oldName,
		cchar	*newName,
		bool	
	)
{
	QString	rename	= QString("alter table %1 rename as %2")
				 .arg(oldName)
				 .arg(newName) ;

	return	execSQL
		(	rename,
			QString("Failed to rename table \"%1\" as \"%2\"")
				 .arg(oldName)
				 .arg(newName)
		)	;
}

/*  KBODBC								*/
/*  doDropTable	: Drop a table						*/
/*  table	: cchar *	: Table name				*/
/*  assoc	: bool		: Drop associated stuff - none here	*/
/*  (returns)	: bool		: Success				*/

bool	ODBC_NS::KBODBC::doDropTable
	(	cchar	*table,
		bool	
	)
{
	QString	rename	= QString("drop table %1").arg(table) ;

	return	execSQL
		(	rename,
			QString("Failed to drop table \"%1\"")
				 .arg(table)
		)	;
}

/*  KBServer								*/
/*  listTypes	: Return a list of all column types			*/
/*  (returns)	: QString	: Types					*/

QString	KBODBC::listTypes ()
{
	static	QString	typeList ;

	if (typeList.isNull())
	{
		typeList = "Primary Key,0|Foreign Key,0" ;

		for (uint idx = 0 ; idx < sizeof(typeMap)/sizeof(ODBCTypeMap) ; idx += 1)
		{
			ODBCTypeMap *m = &typeMap[idx] ;

			if ((m->flags & FF_NOCREATE) != 0)
				continue ;

			if (m_typeDict.find (m->odbcType) == 0)
				continue ;

			typeList += QString("|%1,%2").arg(m->kbName).arg(m->flags) ;
		}
	}

	return	typeList ;
}

/*  KBODBC								*/
/*  bindParamaters							*/
/*		: Bind placeholder parameters to statement		*/
/*  stmHandle	: SQLHSTMT	: Statement handle			*/
/*  nvals	: uint		: Number of substitution values		*/
/*  value	: KBValue *	: Substitution values			*/
/*  vList	: KBODBCValue &	: Substituted values list		*/
/*  codec	: QTextCodec *	: Non-default codec			*/
/*  (returns)	: bool		: Success				*/

bool	KBODBC::bindParameters
	(	SQLHSTMT		stmHandle,
		uint			nvals,
		const KBValue		*values,
		QList<KBODBCValue>	&vList,
		QTextCodec		*codec
	)
{
	long	odbcRC	;

	for (uint idx = 0 ; idx < nvals ; idx += 1)
	{
		KBODBCValue *odbcVal = new KBODBCValue (values[idx], codec) ;

		vList.append (odbcVal) ;

		odbcRC	= SQLBindParameter
			  (	stmHandle,
				idx + 1,
				SQL_PARAM_INPUT,
				odbcVal->vtype(),
				odbcVal->ptype(),
				20,
				0,
				odbcVal->vptr (),
				odbcVal->vlen (),
				odbcVal->slind()
			  )	;

		if (!checkRCOK(stmHandle, odbcRC, "Error binding ODBC parameter"))
			return	false ;
	}

	return	true	;
}

/*  KBODBC								*/
/*  getRowCount	: Get query row count					*/
/*  stmHandle	: SQLHSTMT	: Query statement handle		*/
/*  nRows	: int &		: Return row count			*/
/*  (returns)	: bool		: Success				*/

bool	KBODBC::getRowCount
	(	SQLHSTMT	stmHandle,
		int		&nRows
	)
{
	SQLINTEGER sqlRows ;
	long	   odbcRC  = SQLRowCount (stmHandle, &sqlRows) ;

	if (!checkRCOK(stmHandle, odbcRC, "Error finding ODBC row count"))
		return	false	;

	nRows	= sqlRows	;
	fprintf	(stderr, "ODBC: returned %d rows\n", (int)nRows) ;
	return	true ;
}

QString	ODBC_NS::KBODBC::mapExpression
	(	const QString	&expr
	)
{
	static	QString	spec ("_") ;

	return	m_mapExpressions ?
			doMapExpression (expr, "[", "]", spec) :
			expr ;
}

/*  KBODBC								*/
/*  getSyntax	: Get text for syntax element				*/
/*  result	: QString &	: Return resule string			*/
/*  syntax	: Syntax	: Element				*/
/*  ...		: ...		: Arguments				*/
/*  (returns)	: QString	: Text					*/

bool	KBODBC::getSyntax
	(	QString			&result,
		KBServer::Syntax	syntax,
		...
	)
{
	va_list	 ap ;
	va_start (ap, syntax) ;

	switch (syntax)
	{
		case Limit :
			{
				int	limit	= va_arg(ap, int) ;
				int	offset	= va_arg(ap, int) ;

				if (m_driverExtn != 0)
					if (m_driverExtn->limitOffset != 0)
					{	result	= (*m_driverExtn->limitOffset) (limit, offset) ;
						return	true ;
					}

				result	= QString(" limit %1 offset %2 ").arg(limit).arg(offset) ;
			}
			return	true	;

		default	:
			break	;
	}

	m_lError = KBError
		   (	KBError::Error,
			QString(TR("Driver does not support %1")).arg(syntaxToText(syntax)),
			QString::null,
			__ERRLOCN
		   )	;
	return	false	;
}



/*  ------------------------------------------------------------------- */


/*  KBODBCQrySelect							*/
/*  KBODBCQrySelect							*/
/*		: Constructor for select query object			*/
/*  server	: KBODBC *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Select query text			*/
/*  update	: bool		  : Select for update			*/
/*  (returns)	: KBODBCQrySelect :					*/

KBODBCQrySelect::KBODBCQrySelect
	(	KBODBC		*server,
		bool		data,
		const QString	&select,
		bool
	)	
	:
	KBSQLSelect (server, data, select),
	m_pServer   (server)
{
	m_nRows	    = 0  ;
	m_nFields   = 0  ;
	m_CRow	    = -1 ;

	if (!m_pServer->getStatement (m_stmHandle))
		return ;

	QCString qryText = m_rawQuery.utf8() ;
	long	 odbcRC	 = SQLPrepare (m_stmHandle, (uchar *)qryText.data(), qryText.length()) ;

	if (!m_pServer->checkRCOK (m_stmHandle, odbcRC, "Error preparing statement from ODBC"))
	{
		SQLFreeStmt (m_stmHandle, SQL_DROP) ;
		m_stmHandle = 0	    ;
		m_lError    = m_pServer->lastError() ;
	}

	fprintf	(stderr, "ODBC: [%s]\n", (cchar *)select) ;
}

/*  KBODBCQrySelect							*/
/*  KBODBCQrySelect							*/
/*		: Constructor for select query object			*/
/*  server	: KBODBC *	  : Server connection object		*/
/*  stmHandle	: SQLHSTMT	  : Extant stamement handle		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Select query text			*/
/*  ok		: bool		  : Success				*/
/*  (returns)	: KBODBCQrySelect :					*/

KBODBCQrySelect::KBODBCQrySelect
	(	KBODBC		*server,
		SQLHSTMT	stmHandle,
		bool		data,
		const QString	&select,
		bool		&ok
	)	
	:
	KBSQLSelect (server, data, select),
	m_pServer   (server)
{
	m_nRows	    = 0  ;
	m_nFields   = 0  ;
	m_CRow	    = -1 ;

	m_stmHandle = stmHandle ;

	SQLSMALLINT	_m_nFields	;
	SQLNumResultCols(m_stmHandle, &_m_nFields) ;
	m_nFields	= _m_nFields	;

	m_types		= new KBType *[m_nFields]  ;

	for (uint idx = 0 ; idx < m_nFields ; idx += 1)
	{
		SQLCHAR		colName[101]	;
		SQLSMALLINT	nameLen		;	
		SQLSMALLINT	colType		;
		SQLUINTEGER	colSize		;
		SQLSMALLINT	decimal		;
		SQLSMALLINT	nullable	;
		long		odbcRC		;

		odbcRC	= SQLDescribeCol
			  (	m_stmHandle,
				idx + 1,
				colName,
				sizeof(colName),
				&nameLen,
				&colType,
				&colSize,
				&decimal,
				&nullable
			  )	;

		if (!ODBCOK(odbcRC))
		{
			m_lError  = KBError
				  (	KBError::Error,
					"Error finding ODBC select column type",
					QString::null,
					__ERRLOCN
				  )	  ;
			ok	  = false ;
			return	  ;
		}

		/* Types retured by SQLDescribeCol are like		*/
		/*	SQL_CHAR					*/
		/*	SQL_VARCHAR					*/
		/*	...						*/
		m_colNames.append ((cchar *)colName) ;
		m_dbTypes .append (colType) ;
		m_types[idx] = new KBODBCType (colType, colSize, nullable ) ;

		switch (colType)
		{
//			case SQL_TYPE_DATE:
//				m_cTypes.append (SQL_C_TYPE_DATE     ) ;
//				break	;
//
//			case SQL_TYPE_TIME:
//				m_cTypes.append (SQL_C_TYPE_TIME     ) ;
//				break	;
//
//			case SQL_TYPE_TIMESTAMP:
//				m_cTypes.append (SQL_C_TYPE_TIMESTAMP) ;
//				break	;
//
			case SQL_NUMERIC  :
			case SQL_DECIMAL  :
				/* For numeric and decimal types, we	*/
				/* get the driver to do the conversion	*/
				/* for us. Much easier!			*/
				m_cTypes.append (SQL_C_DOUBLE	     )  ;
				break	;

			default:
				m_cTypes.append (SQL_C_DEFAULT	     )  ;
				break;
		}

		fprintf
		(	stderr,
			"ODBC: %3d: %5d: [%3d] %s\n",
			idx,
			colType,
			m_dbTypes[idx],
			(cchar *)m_types[idx]->getDescrip(true)
		)	;
	}

	m_nRows	= RowsUnknown ;
	m_CRow	= -1	;
	ok	= true	;
}

/*  KBODBCQrySelect							*/
/*  ~KBODBCQrySelect							*/
/*		: Destructor for select query object			*/
/*  (returns)	:		:					*/

KBODBCQrySelect::~KBODBCQrySelect ()
{
	if (m_stmHandle != 0)
		SQLFreeStmt  (m_stmHandle, SQL_DROP) ;

//	if (m_types != 0)
//		for (uint idx = 0 ; idx < m_nFields ; idx += 1)
//			m_types[idx]->deref() ;
}

/*  KBODBCQrySelect							*/
/*  execute	: Execute the query					*/
/*  nvals	: uint		: Number of substitution values		*/
/*  value	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBODBCQrySelect::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	long	odbcRC	;

	if (m_stmHandle == 0)
		return false ;

	SQLCloseCursor	(m_stmHandle) ;

	m_dbTypes .clear () ;
	m_cTypes  .clear () ;
	m_colNames.clear () ;

	QList<KBODBCValue>  vList  ;
	vList.setAutoDelete (true) ;

	if (!m_pServer->bindParameters (m_stmHandle, nvals, values, vList, m_codec))
	{
		m_lError = m_pServer->lastError () ;
		return	 false ;
	}

	odbcRC	= SQLExecute (m_stmHandle) ;
	m_pServer->printQuery (m_rawQuery, nvals, values, ODBCOK(odbcRC)) ;

	if (!m_pServer->checkRCOK(m_stmHandle, odbcRC, "Error executing ODBC select query"))
	{
		m_lError = m_pServer->lastError() ;
		return	 false	;
	}

	SQLSMALLINT	_m_nFields	;
	SQLNumResultCols(m_stmHandle, &_m_nFields) ;
	m_nFields	= _m_nFields	;

	if (m_types == 0)
	{
	m_types		= new KBType *[m_nFields]  ;

	for (uint idx = 0 ; idx < m_nFields ; idx += 1)
	{
		SQLCHAR		colName[101]	;
		SQLSMALLINT	nameLen		;	
		SQLSMALLINT	colType		;
		SQLUINTEGER	colSize		;
		SQLSMALLINT	decimal		;
		SQLSMALLINT	nullable	;

		odbcRC	= SQLDescribeCol
			  (	m_stmHandle,
				idx + 1,
				colName,
				sizeof(colName),
				&nameLen,
				&colType,
				&colSize,
				&decimal,
				&nullable
			  )	;

		if (!ODBCOK(odbcRC))
		{
			m_lError  = KBError
				  (	KBError::Error,
					"Error finding ODBC select column type",
					QString::null,
					__ERRLOCN
				  )	;
			return	false	;
		}

		/* Types retured by SQLDescribeCol are like		*/
		/*	SQL_CHAR					*/
		/*	SQL_VARCHAR					*/
		/*	...						*/
		m_colNames.append ((cchar *)colName) ;
		m_dbTypes .append (colType) ;
		m_types[idx] = new KBODBCType (colType, colSize, nullable ) ;

		switch (colType)
		{
//			case SQL_TYPE_DATE:
//				m_cTypes.append (SQL_C_TYPE_DATE     ) ;
//				break	;
//
//			case SQL_TYPE_TIME:
//				m_cTypes.append (SQL_C_TYPE_TIME     ) ;
//				break	;
//
//			case SQL_TYPE_TIMESTAMP:
//				m_cTypes.append (SQL_C_TYPE_TIMESTAMP) ;
//				break	;
//
			case SQL_NUMERIC  :
			case SQL_DECIMAL  :
				/* For numeric and decimal types, we	*/
				/* get the driver to do the conversion	*/
				/* for us. Much easier!			*/
				m_cTypes.append (SQL_C_DOUBLE	     )  ;
				break	;

			default:
				m_cTypes.append (SQL_C_DEFAULT	     )  ;
				break;
		}

		fprintf
		(	stderr,
			"ODBC: %3d: %5d: [%3d] %s\n",
			idx,
			colType,
			m_dbTypes[idx],
			(cchar *)m_types[idx]->getDescrip(true)
		)	;
	}
	}

	m_nRows	= RowsUnknown ;
	m_CRow	= -1	;
	return	true	;
}

#include	"kb_odbcrow.cpp"

/*  KBODBCQrySelect							*/
/*  getField	: Get a specified field from the query results		*/
/*  qrow	: uint		  : Row number				*/
/*  qcol	: uint		  : Column number			*/
/*  vtrans	: KBValue::VTrans : Translation mode			*/
/*  (returns)	: KBValue	  : Value				*/

KBValue	KBODBCQrySelect::getField
	(	uint		qrow,
		uint		qcol,
		KBValue::VTrans
	)
{
	if (rowExists (qrow))
	{
		KBValue	value	;
		return	getFromCache (qrow, qcol, value) ? value : KBValue() ;
	}

	return	KBValue() ;
}

/*  KBODBCQrySelect							*/
/*  getFieldName: Get a specified field name from the query results	*/
/*  qcol	: uint		: Column number				*/
/*  (returns)	: QString	: name					*/

QString	KBODBCQrySelect::getFieldName
	(	uint	//qcol
	)
{
#if	0
	/* First check that the request is within the range of rows	*/
	/* and fields returned by the query.				*/
	if (qcol >= m_nFields) return QString() ;
	return	xSelect->getFieldName (qcol)  ;
#endif
	return	"NoName" ;
}

/*  KBODBCQryUpdate							*/
/*  KBODBCQryUpdate							*/
/*		: Constructor for update query object			*/
/*  server	: KBODBC *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Update query text			*/
/*  tabName	: const QString & : Table being updated			*/
/*  codec	: QTextCodec *	  : Non-default codec			*/
/*  (returns)	: KBODBCQryUpdate:					*/

KBODBCQryUpdate::KBODBCQryUpdate
	(	KBODBC		*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)	
	:
	KBSQLUpdate (server, data, query, tabName),
	m_pServer   (server)
{
	m_nRows	 = 0 ;

	if (!m_pServer->getStatement (m_stmHandle))
		return ;

	QCString qryText = m_rawQuery.utf8() ;
	long	 odbcRC	 = SQLPrepare (m_stmHandle, (uchar *)qryText.data(), qryText.length()) ;

	if (!m_pServer->checkRCOK (m_stmHandle, odbcRC, "Error preparing statement from ODBC"))
	{
		SQLFreeStmt (m_stmHandle, SQL_DROP) ;
		m_stmHandle = 0	    ;
		m_lError    = m_pServer->lastError() ;
	}

	fprintf	(stderr, "ODBC: [%s]\n", (cchar *)query) ;
}

/*  KBODBCQryUpdate							*/
/*  execute	: Execute the query					*/
/*  nvals	: uint		: Number of substitution values		*/
/*  value	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBODBCQryUpdate::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	long	odbcRC	;

	if (m_stmHandle == 0)
		return false ;

	SQLCloseCursor	(m_stmHandle) ;

	QList<KBODBCValue>  vList  ;
	vList.setAutoDelete (true) ;

	if (!m_pServer->bindParameters (m_stmHandle, nvals, values, vList, m_codec))
	{
		m_lError = m_pServer->lastError () ;
		return	 false ;
	}

	odbcRC	= SQLExecute (m_stmHandle) ;
	m_pServer->printQuery (m_rawQuery, nvals, values, ODBCOK(odbcRC)) ;

	if (!m_pServer->checkDataOK(m_stmHandle, odbcRC, "Error executing ODBC update query"))
	{
		m_lError = m_pServer->lastError() ;
		return	 false	;
	}

	if (!m_pServer->getRowCount(m_stmHandle, m_nRows))
	{
		m_lError = m_pServer->lastError() ;
		return	 false	;
	}

	return	true	;
}

/*  KBODBCQryUpdate							*/
/*  ~KBODBCQryUpdate							*/
/*		: Destructor for update query object			*/
/*  (returns)	:		:					*/

KBODBCQryUpdate::~KBODBCQryUpdate ()
{
	if (m_stmHandle != 0)
		SQLFreeStmt  (m_stmHandle, SQL_DROP) ;
}


/*  KBODBCQryInsert							*/
/*  KBODBCQryInsert							*/
/*		: Constructor for insert query object			*/
/*  server	: KBODBC *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Insert query			*/
/*  tabName	: const QString & : Table being updated			*/
/*  codec	: QTextCodec *	  : Non-default codec			*/
/*  (returns)	: KBODBCQryInsert :					*/

KBODBCQryInsert::KBODBCQryInsert
	(	KBODBC		*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)	
	:
	KBSQLInsert (server, data, query, tabName),
	m_pServer   (server)
{
	m_nRows	 = 0 ;

	if (!m_pServer->getStatement (m_stmHandle)) return ;

	QCString qryText = m_rawQuery.utf8() ;
	long	 odbcRC	 = SQLPrepare (m_stmHandle, (uchar *)qryText.data(), qryText.length()) ;

	if (!m_pServer->checkRCOK (m_stmHandle, odbcRC, "Error preparing statement from ODBC"))
	{
		SQLFreeStmt (m_stmHandle, SQL_DROP) ;
		m_stmHandle = 0	    ;
		m_lError    = m_pServer->lastError() ;
		return	;
	}

	fprintf	(stderr, "ODBC: [%s]\n", (cchar *)query) ;
}

/*  KBODBCQryInsert							*/
/*  ~KBODBCQryInsert							*/
/*		: Destructor for insert query object			*/
/*  (returns)	:		:					*/

KBODBCQryInsert::~KBODBCQryInsert ()
{
	if (m_stmHandle != 0)
		SQLFreeStmt  (m_stmHandle, SQL_DROP) ;
}

/*  KBODBCQryInsert							*/
/*  execute	: Execute the query					*/
/*  nvals	: uint		: Number of substitution values		*/
/*  value	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBODBCQryInsert::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	long	odbcRC	;

	if (m_stmHandle == 0)
		return false ;

	SQLCloseCursor	(m_stmHandle) ;

	QList<KBODBCValue>  vList  ;
	vList.setAutoDelete (true) ;

	if (!m_pServer->bindParameters (m_stmHandle, nvals, values, vList, m_codec))
	{
		m_lError = m_pServer->lastError () ;
		return	 false ;
	}

	odbcRC	= SQLExecute (m_stmHandle) ;
	m_pServer->printQuery (m_rawQuery, nvals, values, ODBCOK(odbcRC)) ;

	if (!m_pServer->checkDataOK(m_stmHandle, odbcRC, "Error executing ODBC insert query"))
	{
		m_lError = m_pServer->lastError() ;
		return	 false	;
	}

	if (!m_pServer->getRowCount(m_stmHandle, m_nRows))
	{
		m_lError = m_pServer->lastError() ;
		return	 false	;
	}

	return	true	;
}

/*  KBODBC								*/
/*  getNewKey	: Get new primary key value				*/
/*  primary	: const QString & : Key column name			*/
/*  _newKey	: KBValue &	  : New key value			*/
/*  prior	: bool		  : Pri-insert call			*/
/*  (returns)	: bool		  : Success				*/

bool	KBODBCQryInsert::getNewKey
	(	const QString	&,
		KBValue		&,
		bool		
	)
{
	UNIMPL	("KBODBCQryInsert::getNewKey") ;
	return	false ;
}

/*  KBODBCQryDelete							*/
/*  KBODBCQryDelete							*/
/*		: Constructor for delete query object			*/
/*  server	: KBODBC *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Delete query			*/
/*  tabName	: const QString & : Table being updated			*/
/*  codec	: QTextCodec *	  : Non-default codec			*/
/*  (returns)	: KBODBCQryDelete :					*/

KBODBCQryDelete::KBODBCQryDelete
	(	KBODBC		*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)	
	:
	KBSQLDelete (server, data, query, tabName),
	m_pServer   (server)
{
	m_nRows	 = 0 ;

	if (!m_pServer->getStatement (m_stmHandle))
		return ;

	QCString qryText = m_rawQuery.utf8() ;
	long	 odbcRC	 = SQLPrepare (m_stmHandle, (uchar *)qryText.data(), qryText.length()) ;

	if (!m_pServer->checkRCOK (m_stmHandle, odbcRC, "Error preparing statement from ODBC"))
	{
		SQLFreeStmt (m_stmHandle, SQL_DROP) ;
		m_stmHandle = 0	    ;
		m_lError    = m_pServer->lastError() ;
	}

	fprintf	(stderr, "ODBC: [%s]\n", (cchar *)query) ;
}

/*  KBODBCQryDelete							*/
/*  execute	: Execute the query					*/
/*  nvals	: uint		: Number of substitution values		*/
/*  value	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBODBCQryDelete::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	long	odbcRC	;

	if (m_stmHandle == 0)
		return false ;

	SQLCloseCursor	(m_stmHandle) ;

	QList<KBODBCValue>  vList  ;
	vList.setAutoDelete (true) ;

	if (!m_pServer->bindParameters (m_stmHandle, nvals, values, vList, m_codec))
	{
		m_lError = m_pServer->lastError () ;
		return	 false ;
	}

	odbcRC	= SQLExecute (m_stmHandle) ;
	m_pServer->printQuery (m_rawQuery, nvals, values, ODBCOK(odbcRC)) ;

	if (!m_pServer->checkDataOK(m_stmHandle, odbcRC, "Error executing ODBC delete query"))
	{
		m_lError = m_pServer->lastError() ;
		return	 false	;
	}

	if (!m_pServer->getRowCount(m_stmHandle, m_nRows))
	{
		m_lError = m_pServer->lastError() ;
		return	 false	;
	}

	return	true	;
}

/*  KBODBCQryDelete							*/
/*  ~KBODBCQryDelete							*/
/*		: Destructor for delete query object			*/
/*  (returns)	:		:					*/

KBODBCQryDelete::~KBODBCQryDelete ()
{
	if (m_stmHandle != 0)
		SQLFreeStmt  (m_stmHandle, SQL_DROP) ;
}


