/***************************************************************************
    file	         : tkc_pydebugwidget.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>

#include	"kb_python.h"

#include	<qapp.h>
#include	<qcursor.h>
#include 	<qframe.h>
#include 	<qsplitter.h>
#include 	<qlayout.h>


#include	"tkc_pydebugbase.h"

#ifndef 	_WIN32
#include	"tkc_pydebugwidget.moc"
#else
#include 	"tkc_pydebugwidget.h"
#include	<qapplication.h>
#endif

#include	"tkc_pyeditor.h"
#include	"tkc_excskipdlg.h"

#include	"tk_icons.h"
#include	"tk_messagebox.h"
#include	"tk_config.h"


TKCPyDebugWidget	*TKCPyDebugWidget::debWidget ;


#define	ICONNAME	"rekall"
static	QStringList	excSkipList	;

/*  ------------------------------------------------------------------  */

/*  TKCPyDebugWidget							*/
/*  TKCPyDebugWidget							*/
/*		: Constructor for the debugger widget			*/
/*  parent	: QWidget *	   : Parent widget			*/
/*  window	: QWidget *	   : Top window				*/
/*  (returns)	: TKCPyDebugWidget :					*/

TKCPyDebugWidget::TKCPyDebugWidget
	(	QWidget 	*parent,
		KBSDIMainWindow	*window
	)
	:
	QWidget		(parent, "tk_pydebugwidget"),
	parent		(parent),
	window		(window),
	errExp		(": *([0-9]*):")
{
	QPixmap		icon	= getSmallIcon (ICONNAME) ;
	QIconSet	iconSet	(icon) ;

	inModalLoop	= false	;
	aborting	= 0	;

#if	__KB_EMBEDDED
	m_pDockTabGroup = new QTabWidget (this, 	 "TabGroup");
#else
	m_pSplitMain	= new QSplitter  (Qt::Horizontal, this)     ;
	m_pDockTabGroup = new QTabWidget (m_pSplitMain, "TabGroup") ;
#endif
	
  	m_pObjectView 	= new TKCPyValueList(m_pDockTabGroup, this) ;
  	m_pFuncView 	= new TKCPyFuncList (m_pDockTabGroup, this) ;
	m_pTraceView	= new TKCPyValueList(m_pDockTabGroup, this) ;
	m_pStackView	= new TKCPyValueList(m_pDockTabGroup, this) ;


	m_pDockTabGroup->addTab(m_pObjectView, TR("Objects"    )) ;
	m_pDockTabGroup->addTab(m_pFuncView,   TR("Functions"  )) ;
	m_pDockTabGroup->addTab(m_pTraceView,  TR("Breakpoints")) ;
	m_pDockTabGroup->addTab(m_pStackView,  TR("Backtrace"  )) ;

#if	__KB_EMBEDDED
	m_pRightSide	= new QWidget (m_pDockTabGroup) ;
	m_pDockTabGroup->addTab(m_pRightSide,  TR("Source"     )) ;
#else
	m_pRightSide	= new QWidget (m_pSplitMain   ) ;
#endif

	m_pReason	= new QLabel(m_pRightSide) ;
	m_pReason->setFixedHeight(18);

	m_pSplitRight	= new QSplitter  (Qt::Vertical, m_pRightSide) ;
	m_pEditors	= new QTabWidget (m_pSplitRight) ;
	m_pErrWin	= new QListBox   (m_pSplitRight) ;

	m_pLayMain	= new QVBoxLayout(this ) ;

#if	__KB_EMBEDDED
	m_pLayMain ->addWidget (m_pDockTabGroup) ;
#else
	m_pLayMain ->addWidget (m_pSplitMain   ) ;
#endif

	m_pLayRight	= new QVBoxLayout(m_pRightSide ) ;
	m_pLayRight->addWidget (m_pReason    ) ;
	m_pLayRight->addWidget (m_pSplitRight) ;

	setTraceMessage(QString::null);
	
	m_pObjectView->addColumn (TR("Name" )) ;
	m_pObjectView->addColumn (TR("Type" )) ;
	m_pObjectView->addColumn (TR("Value")) ;

	m_pFuncView  ->addColumn (TR("Name" )) ;
	m_pFuncView  ->addColumn (TR("Type" )) ;

	m_pStackView ->addColumn (TR("Name" )) ;
	m_pStackView ->addColumn (TR("Type" )) ;
	m_pStackView ->addColumn (TR("Value")) ;
	m_pStackView ->addColumn (TR("Line" )) ;
	m_pStackView ->setSorting(-1);

  	m_pTraceView ->addColumn (TR("Name" )) ;
	m_pTraceView ->addColumn (TR("Type" )) ;
	m_pTraceView ->addColumn (TR("Line" )) ;
	m_pTraceView ->addColumn (TR("Bpt"  )) ;
	m_pTraceView ->addColumn (TR("Count")) ;

	m_pTraceView ->setRootIsDecorated (false) ;

	connect	(m_pObjectView, SIGNAL(mouseButtonPressed (int, QListViewItem *, const QPoint &, int)),
			        SLOT  (showContextMenu	  (int, QListViewItem *, const QPoint &, int))) ;
	connect	(m_pFuncView,   SIGNAL(mouseButtonPressed (int, QListViewItem *, const QPoint &, int)),
			        SLOT  (showContextMenu	  (int, QListViewItem *, const QPoint &, int))) ;
	connect	(m_pStackView,  SIGNAL(mouseButtonPressed (int, QListViewItem *, const QPoint &, int)),
			        SLOT  (showContextMenu	  (int, QListViewItem *, const QPoint &, int))) ;
	connect	(m_pTraceView,  SIGNAL(mouseButtonPressed (int, QListViewItem *, const QPoint &, int)),
			        SLOT  (showContextMenu	  (int, QListViewItem *, const QPoint &, int))) ;

	connect (m_pEditors, 	SIGNAL(currentChanged(QWidget *)),
			        SLOT  (editorChanged (QWidget *))) ;

	QWidget::show()	 ;

	debWidget = this ;
	inDialog  = 0	 ;
	m_pManager= 0 	 ;
}

/*  TKCPyDebugWidget							*/
/*  TKCPyDebugWidget							*/
/*		: Destructor for the debugger widget			*/
/*  (returns)	:		:					*/

TKCPyDebugWidget::~TKCPyDebugWidget ()
{
	TKCPyTraceItem *trace = (TKCPyTraceItem *)m_pTraceView->firstChild() ;

	while (trace != 0)
	{
		TKCPyDebugBase::clearTracePoint
		(	trace->value()->value(),
			trace->getLineno     ()
		)	;
		trace = (TKCPyTraceItem *)trace->nextSibling() ;
	}

	debWidget = 0 ;
	DELOBJ	(m_pManager) ;
}


/*  TKCPyDebugWidget							*/
/*  widget	: Return debugger widget				*/
/*  (returns)	: TKCPyDebugWidget * : Widget or null			*/

TKCPyDebugWidget
	*TKCPyDebugWidget::widget ()
{
	return	debWidget ;
}

/*  TKCPyDebugWidget							*/
/*  getObjectModule							*/
/*		: Get module associated with object			*/
/*  pyObj	: PyObject *	: Object				*/
/*  lno		: uint &	: Line number or zero if not found	*/
/*  (returns)	: TKCPyCookie *	: Module cookie				*/

TKCPyCookie
	*TKCPyDebugWidget::getObjectModule
	(	PyObject	*pyObj,
		uint		&lno
	)
{
	/* If this is a module then we can pick up the filename from it	*/
	/* and map this back to the module as a cookie.			*/	
	if (PyModule_Check (pyObj))
	{	lno	= 0 ;
		return	TKCPyModuleToCookie (PyModule_GetFilename (pyObj)) ;
	}

	/* If the object is a function or code object then we need to	*/
	/* go via a python code ans string objects ...			*/
	PyCodeObject *pyCode ;

	if	(PyFunction_Check (pyObj))
		pyCode	= (PyCodeObject *)((PyFunctionObject *)pyObj)->func_code ;
	else if (PyCode_Check	  (pyObj))
		pyCode	= (PyCodeObject *)pyObj ;
	else	return	0 ;

	lno	= pyCode->co_firstlineno ;

	return	TKCPyModuleToCookie
		(	TKCPyDebugBase::getPythonString(pyCode->co_filename)
		)	;
}

void	TKCPyDebugWidget::setTraceMessage
	(	const QString	&msg
	)
{
	QPalette pal = QApplication::palette () ;

	pal.setColor (QColorGroup::Foreground, white  ) ;
	pal.setColor (QColorGroup::Background, msg.isNull() ? darkBlue : darkRed) ;

	m_pReason->setPalette(pal) ;
	m_pReason->setText(msg);
}

/*  TKCPyDebugWidget							*/
/*  showEvent	: Handle widget becoming visible			*/
/*  e		: QShowEvent *	: Show event				*/
/*  (returns)	: void		:					*/

void	TKCPyDebugWidget::showEvent
	(	QShowEvent	*
	)
{
	/* We want to (re)load the objects and function views. Start	*/
	/* from the module dictionary and invalidate all items that	*/
	/* are on display at the moment.				*/
	QDict<TKCPyValue> modDict ;
	TKCPyDebugBase::getModuleDict(modDict) ;

	m_pObjectView->invalidate () ;
	m_pFuncView->invalidate () ;

	QDictIterator<TKCPyValue> modIter (modDict) ;
	while (modIter.current())
	{
		/* Get each value and see if it is present in each	*/
		/* view at the top level (no recursion). For each, if	*/
		/* it is present then mark as valid, if not then add a	*/
		/* a new item if it should be shown.			*/
		TKCPyValue	*value	= modIter.current() ;
		TKCPyValueItem	*item1	= m_pObjectView->scanForObject (value->value(), false) ;
		TKCPyValueItem	*item2	= m_pFuncView->scanForObject(value->value(), false) ;

		if	(item1 != 0)
			item1->setValid () ;
		else if (m_pObjectView->showObject (value->value()))
			new TKCPyValueItem (m_pObjectView, modIter.currentKey(), value) ;

		if	(item2 != 0)
			item2->setValid () ;
		else if (m_pFuncView->showObject (value->value()))
			new TKCPyValueItem (m_pFuncView,   modIter.currentKey(), value) ;

		/* Decrement the reference count. If it was already	*/
		/* present, or was not wanted, in both views, then the	*/
		/* count will drop to zero and we will delete it.	*/
		if (value->deref()) delete value ;

		modIter	+= 1 ;
	}

	/* Finish off by cleaning up any entries that are no longer	*/
	/* valid.							*/
	m_pObjectView->clean () ;
	m_pFuncView->clean () ;
}


/*  TKCPyDebugWidget							*/
/*  showTrace	: Show traceback					*/
/*  pyFrame	: PyFrameObject * : Stack frame				*/
/*  what	: const QString & : Reason for trace			*/
/*  (returns)	: void		  :					*/

void	TKCPyDebugWidget::showTrace
	(	PyFrameObject	*pyFrame,
		const QString	&what
	)
{
	/* Generate the stack backtrace. We display the calls with the	*/
	/* path name to the code if we can determin it (so that we get	*/
	/* the full module/class/function identification) or the code	*/
	/* name if not.							*/
	QListViewItem	*last   = 0 ;
	PyFrameObject	*pyBack	= pyFrame ;
	m_pStackView->clear () ;


	while (pyBack != 0)
	{
		PyCodeObject	*pyCode	= pyBack->f_code ;
		QString		name	= TKCPyDebugBase::getObjectName
					  (	(PyObject *)pyCode
					  )	;

		if (name == QString::null)
			name	= TKCPyDebugBase::getPythonString(pyCode->co_name) ;

		last	= new TKCPyStackItem
			  (	m_pStackView,
				last,
				name,
				TKCPyValue::allocValue ((PyObject *)pyBack),
				pyBack->f_lineno
			  )	;

		pyBack	= pyBack->f_back ;
	}

	PyObject    *pyCode = (PyObject *)pyFrame->f_code ;
	TKCPyEditor *editor = showObjectCode (pyCode) ;

	for (uint idx = 0 ; idx < editList.count() ; idx += 1)
	{
		TKCPyEditor *ed = editList.at(idx) ;
		ed->setCurrentLine (ed == editor ? pyFrame->f_lineno : 0) ;
	}

	setTraceMessage
	(	QString (TR("  %1: %2, line %3"))
			.arg(what)
			.arg(TKCPyDebugBase::getObjectName (pyCode))
			.arg(pyFrame->f_lineno)
	)	;

}

/*  TKCPyDebugWidget							*/
/*  loadErrorText: Load error text into error window			*/
/*  eText	 : const QString & : Error text				*/
/*  (returns)	 : void		   :					*/

void	TKCPyDebugWidget::loadErrorText
	(	const QString	&eText
	)
{
	/* Disconnect the selected signal first, else we get a burst	*/
	/* as we load ...						*/
	disconnect (m_pErrWin, SIGNAL(selected(int)), this, SLOT(errSelected(int))) ;

	m_pErrWin->clear();

	int	ptr1	= 0 ;
	int	ptr2	= eText.find ('\n') ;

	while (ptr2 > 0)
	{	
	  m_pErrWin->insertItem (eText.mid (ptr1, ptr2 - ptr1 - 1)) ;
		ptr1	= ptr2 + 1 ;
		ptr2	= eText.find ('\n', ptr1) ;
	}
	if (ptr1 < (int)eText.length())
		m_pErrWin->insertItem (eText.mid (ptr1)) ;

	connect (m_pErrWin, SIGNAL(selected(int)), this, SLOT(errSelected(int))) ;
}

/*  TKCPyDebugWidget							*/
/*  editModule	: Edit module						*/
/*  module	: TKCPyCookie *	  : Module cookie			*/
/*  eCompile	: const QString & : Compilation error text		*/
/*  (returns)	: bool		  : Success				*/

TKCPyEditor
	*TKCPyDebugWidget::editModule
	(	TKCPyCookie	*module,
		const QString	&eCompile
	)
{
	TKCPyEditor	*editor	= 0		;

	/* Start by seeing if the module is already loaded into an	*/
	/* editor. If not then create a new one.			*/
	for (uint idx = 0 ; idx < editList.count() ; idx += 1)
		if (*editList.at(idx)->getModule() == *module)
		{	editor	= editList.at(idx) ;
			m_pEditors->setCurrentPage  (m_pEditors->indexOf(editor)) ;
			break	;
		}

	if (editor == 0)
	{
		editor = new TKCPyEditor  (m_pEditors, this, module) ;
		editList.append  	  (editor) ;
		m_pEditors->addTab 	  (editor, module->title()) ;

		connect(editor, SIGNAL(changed(int)), SLOT(moduleChanged(int))) ;
		emit	showingFile (true) ;
	}

	m_pEditors     ->setCurrentPage(m_pEditors->indexOf(editor)) ;
#if	__KB_EMBEDDED
	m_pDockTabGroup->setCurrentPage(m_pEditors->indexOf(m_pRightSide)) ;
#endif

	/* Load the module into the editor. This will overwrite any	*/
	/* existing content, in case it has changed (but, keep extant	*/
	/* breakpoint markers). Also save any error text with the	*/
	/* editor, and load it into the error window.			*/
	QString	modText	 ;
	QString	eError   ;
	QString	eDetails ;

	if (!module->get (modText, eError, eDetails))
	{
		TKCPyDebugError (eError, eDetails) ;
		return	editor	;
	}

	editor->showText   (modText ) ;
	editor->setErrText (eCompile) ;
	loadErrorText	   (eCompile) ;

	return	editor ;
}

/*  TKCPyDebugWidget							*/
/*  showObjectCode: Show code associated with object			*/
/*  pyObj	  : PyObject *	: Object				*/
/*  (returns)	  : void	:					*/

TKCPyEditor
	*TKCPyDebugWidget::showObjectCode
	(	PyObject	*pyObj
	)
{
	TKCPyCookie	*module	 ;
	uint		__lno	 ;

	if ((module = getObjectModule (pyObj, __lno)) != 0)
	{
		TKCPyEditor *editor = editModule (module, "") ;
		delete	    module  ;
		return	    editor  ;
	}

	return	0	;
}

/*  TKCPyDebugWidget							*/
/*  showAsDialog: Show as modal dialog					*/
/*  exception	: bool		: In an exception			*/
/*  (returns)	: TKCPyTraceOpt	: Continuation mode			*/
TKCPyTraceOpt
	TKCPyDebugWidget::showAsDialog
	(	bool	exception
	)
{
extern	void	qt_enter_modal (QWidget *) ;
extern	void	qt_leave_modal (QWidget *) ;

	/* We want to restore the currently active window on exit from	*/
	/* the modal loop. This should just be the window returned by	*/
	/* "qApp->activeWindow()", but if we return in single-step then	*/
	/* on entering here again, the active window is ourself, maybe	*/
	/* because we never return to the main event loop. So, there is	*/
	/* a fudge to remember the last active window, which is used if	*/
	/* we cannot get one this time.					*/
	static	QWidget	*lastActive ;

	QWidget *active	    = qApp->activeWindow () ;
	if ((active != 0) && (active != window)) lastActive = active ;

	emit	enterTrap (!exception, true, !exception) ;

	fprintf	(stderr, "TKCPyDebugWidget: going modal\n") ;

	/* Mark the debugger's top-level window as modal, then make it	*/
	/* active. The show()/raise()/setActiveWindow() sequence should	*/
	/* cover all situations.					*/
	window  ->setModal        () ;
	window  ->show	     	  () ;
	window  ->raise		  () ;
	window	->setActiveWindow () ;

	/* Tell QT that we are entering a modal loop for the debugger	*/
	/* window, then do so. On return (after the user has selected	*/
	/* a continuation option) leave the loop and clear the modal	*/
	/* setting. The "inModalLoop" flag is used by the continuation	*/
	/* slot routines to verify that we are in the modal loop.	*/
	inModalLoop	= true	;
	qt_enter_modal (window) ;
	qApp	->enter_loop () ;
	qt_leave_modal (window) ;

	window  ->clearModal () ;
	inModalLoop	= false	;

	fprintf	(stderr, "TKCPyDebugWidget: back from modal\n") ;

	/* If we have a last-active window then restore this, if not	*/
	/* then just lower the debugger window.				*/
	if (lastActive != 0)
	{	lastActive->show	    () ;
		lastActive->raise	    () ;	
		lastActive->setActiveWindow () ;
	}
	else	window->lower () ;

	/* Reset the continuation buttons to disabled, clear the stack	*/
	/* backtrace and trace message, and remove any current-line	*/
	/* markers.							*/
	emit	exitTrap () ;


	m_pStackView->clear();
	setTraceMessage(QString::null) ;

	for (uint idx = 0 ; idx < editList.count() ; idx += 1)
		editList.at(idx)->setCurrentLine (0) ;

	/* If there was an exception or the user is aborting, then	*/
	/* indicate that the code has been debugged (well, maybe) and	*/
	/* does not need to be displayed.				*/
	if (exception || (modalEndRC == OptAbort)) TKCPySetErrDebugged () ;


	/* If we are aborting, then set the aborting count to two, so	*/
	/* that two exceptions will be skipped. This is needed so that	*/
	/* when the user aborts, they are not immediately presented	*/
	/* with two exceptions.						*/
	aborting = modalEndRC == OptAbort ? 2 : 0 ;
	return	   modalEndRC ;
}

/*  TKCPyDebugWidget							*/
/*  breakAction	: Continue after breakpoint				*/
/*  action	: int		: Action code				*/
/*  (returns)	: void		:					*/

void	TKCPyDebugWidget::breakAction
	(	int		action
	)
{
	/* If we really are in the modal loop then set the continuation	*/
	/* code and tell QT that we wish to exit the loop.		*/
	if (inModalLoop)
	{	modalEndRC = (TKCPyTraceOpt)action ;
		qApp->exit_loop () ;
	}
}

/*  TKCPyDebugWidget							*/
/*  errSelected	: User selects an error					*/
/*  idx		: int		: Index into error list box		*/
/*  (returns)	: void		:					*/

void	TKCPyDebugWidget::errSelected
	(	int	
	)
{
	TKCPyEditor *editor = (TKCPyEditor *)m_pEditors->currentPage();
	if (editor == 0) return  ;

	/* This is pretty basic. Get the line number from the error	*/
	/* message and move the cursor to that line in the edit window.	*/
	/* This may not be the right line if the user has moved any	*/
	/* lines about ....						*/
	if (errExp.search(m_pErrWin->text(m_pErrWin->currentItem())) < 0)
		return	;

	editor->gotoLine (errExp.cap(1).toInt()) ;
}

/*  TKCPyDebugWidget							*/
/*  moduleChanged: Module text changed handler				*/
/*  id		 : int		: Change event				*/
/*  (returns)	 : void		:					*/

void	TKCPyDebugWidget::moduleChanged
	(	int	id
	)
{
	TKCPyEditor *editor = (TKCPyEditor *)m_pEditors->currentPage() ;

	if ((editor != 0) && (id == TKTextEditor::Modification))
		emit fileChanged (true) ;
}

/*  TKCPyDebugWidget							*/
/*  setExcSkipList							*/
/*		: Set the exception skip list				*/
/*  (returns)	: void		:					*/

void	TKCPyDebugWidget::setExcSkipList ()
{
	TKCExcSkipDlg excDlg (excSkipList) ;
	excDlg.exec () ;
}

/*  TKCPyDebugWidget							*/
/*  saveModule	: Save text of currently displayed module		*/
/*  (returns)	: bool		: Success				*/

bool	TKCPyDebugWidget::saveModule ()
{
	TKCPyEditor *editor = (TKCPyEditor *)m_pEditors->currentPage() ;
	if (editor == 0) return false ;

	QString	eError	;
	QString	eText	;

	if (!editor->save (eError, eText))
	{	
	  TKCPyDebugError(eError, eText);
		return	false	;
	}

	emit	fileChanged (false) ;
	return	true	;
}

/*  TKCPyDebugWidget							*/
/*  doCompile	: Compile the currently displayed module		*/
/*  (returns)	: void		:					*/

void	TKCPyDebugWidget::doCompile ()
{
	TKCPyEditor *editor = (TKCPyEditor *)m_pEditors->currentPage() ;
	if (editor == 0) return ;

	if (editor->isModified() && !saveModule()) return ;

	QString	eText	 ;
	QString	eError	 ;
	QString	eDetails ;
	bool	pyErr	 ;

	if (!TKCPyCompileAndLoad
		(	editor->getModule(),
			eText,
			eError,
			eDetails,
			pyErr
		))
		TKCPyDebugError (eError, eDetails) ;

	editor->setErrText (eText) ;
	loadErrorText	   (eText) ;
}

/*  TKCPyDebugWidget							*/
/*  closeModule	: Close currently displayed module			*/
/*  (returns)	: void		:					*/

void	TKCPyDebugWidget::closeModule ()
{
	TKCPyEditor *editor = (TKCPyEditor *)m_pEditors->currentPage() ;
	if (editor == 0) return ;

	if (editor->isModified())
		if (TKMessageBox::questionYesNo
			(	0,
				QString (TR("Module \"%1\" has been changed: close anyway?"))
					.arg(editor->getModule()->title()),
				TR("Module editor")
			)
			!= TKMessageBox::Yes) return ;

	editList.remove (editor) ;
	delete	editor	;

	m_pErrWin->clear() ;
	emit	showingFile (editList.count() > 0) ;
}

/*  TKCPyDebugWidget							*/
/*  editorChanged: User selects different editor			*/
/*  widget	 : QWidget *	: Tab widget, actually an editor	*/
/*  (returns)	 : void		:					*/

void	TKCPyDebugWidget::editorChanged
	(	QWidget		*widget
	)
{
	TKCPyEditor *editor = (TKCPyEditor *)widget ;
	if (editor == 0) return ;

	loadErrorText    (editor->getErrText ()) ;
	emit fileChanged (editor->isModified ()) ;
}

/*  TKCPyDebugWidget							*/
/*  doFuncTrace	: Handle function trace call				*/
/*  pyObh	: PyObject *	: Call frame				*/
/*  pyMsg	: PyObject *	: Trace message				*/
/*  pyArg	: PyObject *	: Trace arguments			*/
/*  userPtr	: void *	: Used pointer				*/
/*  (returns)	: TKCPyTraceOpt	: Trace continuation code		*/

TKCPyTraceOpt
	TKCPyDebugWidget::doFuncTrace
	(	PyObject	*pyObj,
		PyObject	*,
		PyObject	*,
		void		*userPtr
	)
{
	aborting = 0 ;

	/* The first argument should be a frame. If not then ignore the	*/
	/* call; also ignore it if we cannot locate the trace item for	*/
	/* the code being executed.					*/
	if (!PyFrame_Check(pyObj))
		return	OptContinue ;

	PyFrameObject	*pyFrame = (PyFrameObject  *)pyObj	;
	TKCPyTraceItem	*trace	 = (TKCPyTraceItem *)userPtr	;

	/* Increment the trace call count, then return if the		*/
	/* breakpoint is not enabled (ie., if we are watching).		*/
	trace->increment () ;
	if (!trace->isEnabled ())
		return	OptContinue ;

	showObjectCode ((PyObject *)pyFrame->f_code) ;
	showTrace      (pyFrame, TR("Func bpt")) ;

	/* Invoke the debug dialog. This switches the debug display to	*/
	/* be modal.							*/
	return	showAsDialog (false) ;
}

/*  TKCPyDebugWidget							*/
/*  doLineTrace	: Handle function trace call				*/
/*  pyObj	: PyObject *	: Call frame				*/
/*  pyMsg	: PyObject *	: Trace message				*/
/*  pyArg	: PyObject *	: Trace arguments			*/
/*  userPtr	: void *	: Used pointer				*/
/*  (returns)	: TKCPyTraceOpt	: Trace continuation code		*/

TKCPyTraceOpt
	TKCPyDebugWidget::doLineTrace
	(	PyObject	*pyObj,
		PyObject	*,
		PyObject	*,
		void		*userPtr
	)
{
	aborting = 0 ;

	/* The first argument should be a frame. If not then ignore the	*/
	/* call; also ignore it if we cannot locate the trace item for	*/
	/* the code being executed.					*/
	if (!PyFrame_Check(pyObj))
		return	OptContinue ;

	PyFrameObject	*pyFrame = (PyFrameObject  *)pyObj   ;
	TKCPyTraceItem	*trace	 = (TKCPyTraceItem *)userPtr ;

	/* Unless line stepping, increment the trace call count, then	*/
	/* return if the breakpoint is not enabled (ie., if we are	*/
	/* watching).							*/
	if (trace != 0)
	{	trace->increment () ;
		if (!trace->isEnabled ()) return OptContinue ;
	}

	showObjectCode ((PyObject *)pyFrame->f_code) ;
	showTrace      (pyFrame, TR("Line bpt")) ;

	/* Invoke the debug dialog. This switches the debug display to	*/
	/* be modal.							*/
	return	showAsDialog (false) ;
}

/*  TKCPyDebugWidget							*/
/*  doProfTrace	: Handle profile trace call				*/
/*  pyObh	: PyObject *	: Call frame				*/
/*  pyMsg	: PyObject *	: Trace message				*/
/*  pyArg	: PyObject *	: Trace arguments			*/
/*  userPtr	: void *	: Used pointer				*/
/*  (returns)	: TKCPyTraceOpt	: Trace continuation code		*/

TKCPyTraceOpt
	TKCPyDebugWidget::doProfTrace
	(	PyObject	*pyObj,
		PyObject	*,
		PyObject	*pyArg,
		void		*
	)
{
	/* NOTE: Currently this is just an exception ....		*/
	fprintf
	(	stderr,
		"TKCPyDebugWidget::doProfTrace: aborting=%d\n",
		aborting
	)	;

	if (!m_excTrap)
		return	OptContinue ;

	/* The first argument should be a frame. If not then ignore the	*/
	/* call.							*/
	if (!PyFrame_Check(pyObj))
		return	OptContinue ;

	/* Also ignore the call if we are marked as aborting, so that	*/
	/* the user does not get an exception every time they abort.	*/
	if (aborting > 0)
	{	aborting -= 1	    ;
		return	OptContinue ;
	}

	PyFrameObject	*pyFrame = (PyFrameObject *)pyObj 	;
	PyObject    	*pyCode  = (PyObject *)pyFrame->f_code	;
	QString		locn	 = TKCPyDebugBase::getObjectName (pyCode) ;

	/* See if the exception location is one of those which should	*/
	/* skipped ...							*/
	for (uint idx = 0 ; idx < excSkipList.count() ; idx += 1)
		if (locn.find (excSkipList[idx]) == 0)
		{
			fprintf	(stderr, "Skipping exceptions [%s] on [%s]\n",
					 (cchar *)locn,
					 (cchar *)excSkipList[idx]) ;
			return	OptContinue ;
		}


	PyObject *except = PyTuple_GetItem (pyArg, 0) ;
	PyObject *value  = PyTuple_GetItem (pyArg, 1) ;
	PyObject *traceb = PyTuple_GetItem (pyArg, 2) ;

	PyErr_NormalizeException (&except, &value, &traceb) ;
	QString	 name	 = PyString_AsString (((PyClassObject *)except)->cl_name) ;
	QString	 what    = QString (TR("Exception %1")).arg(name) ;


	showObjectCode ((PyObject *)pyFrame->f_code) ;
	showTrace      (pyFrame,  what) ;

	/* Invoke the debug dialog. This switches the debug display to	*/
	/* be modal.							*/
	return	showAsDialog (true) ;
}


/*  TKCPyDebugWidget							*/
/*  doDebugHook	: Handle user invoked debugger entry			*/
/*  pyObj	: PyObject *	: Call frame				*/
/*  msg		: const char *	: User message				*/
/*  (returns)	: TKCPyTraceOpt	: Trace continuation code		*/

TKCPyTraceOpt
	TKCPyDebugWidget::doDebugHook
	(	PY_FRAME_TYPE	*pyObj,
		const char	*msg
	)
{
	fprintf
	(	stderr,
		"TKCPyDebugWidget::doDebugHook: [%s]\n",
		msg
	)	;

	/* The first argument should be a frame. If not then ignore the	*/
	/* call.							*/
	if (!PyFrame_Check(pyObj))
		return	OptContinue ;


	PyFrameObject	*pyFrame = (PyFrameObject *)pyObj 	;
	PyObject    	*pyCode  = (PyObject *)pyFrame->f_code	;
	QString	 	what     = QString (TR("User debug: %1")).arg(msg) ;


	showObjectCode (pyCode) ;
	showTrace      (pyFrame,  what) ;

	/* Invoke the debug dialog. This switches the debug display to	*/
	/* be modal.							*/
	return	showAsDialog (true) ;
}


/*  TKCPyDebugWidget							*/
/*  codeLoaded	: Handle nofitication that code has been loaded		*/
/*  (returns)	: void		:					*/

void	TKCPyDebugWidget::codeLoaded ()
{
	showEvent (0) ;
}

/*  TKCPyDebugWidget							*/
/*  addBreakOptions							*/
/*		: Add tracepoint options to popup menu			*/
/*  popup	: QPopupMenu &	: Popup menu				*/
/*  pyObj	: PyObject *	: Object for which to add		*/
/*  (returns)	: void		:					*/

void	TKCPyDebugWidget::addBreakOptions
	(	QPopupMenu	&popup,
		PyObject	*pyObj
	)
{
	traceItem = (TKCPyTraceItem *)m_pTraceView->scanForObject(pyObj) ;

	if (popup.count() > 0)
		popup.insertSeparator () ;

	if	(traceItem == 0)
	{	popup.insertItem (TR("Set breakpoint"),    this, SLOT(setBreakpoint    ())) ;
		popup.insertItem (TR("Set watchpoint"),    this, SLOT(setWatchpoint    ())) ;
	}
	else if (traceItem->isEnabled ())
	{	popup.insertItem (TR("Clear breakpoint"),  this, SLOT(disableBreakpoint())) ;
		popup.insertItem (TR("Remove watchpoint"), this, SLOT(removeWatchpoint ())) ;
	}
	else
	{	popup.insertItem (TR("Enable breakpoint"), this, SLOT(enableBreakpoint ())) ;
		popup.insertItem (TR("Remove watchpoint"), this, SLOT(removeWatchpoint ())) ;
	}
}

PyObject*TKCPyDebugWidget::getCode
	(	PyObject	*pyObj
	)
{
	if (PyMethod_Check  (pyObj)) pyObj = ((PyMethodObject   *)pyObj)->im_func   ;
	if (PyFunction_Check(pyObj)) pyObj = ((PyFunctionObject *)pyObj)->func_code ;
	if (PyCode_Check    (pyObj)) return pyObj ;
	return	0 ;
}

/*  TKCPyDebugWidget							*/
/*  showContextMenu							*/
/*		: Handler for mouse-click in list views			*/
/*  button	: int		  : Button pressed			*/
/*  item	: QListViewItem * : Item in which pressed		*/
/*  pos		: const QPoint &  : Position at click			*/
/*  col		: int		  : Column number			*/
/*  (returns)	: void		  :					*/

void	TKCPyDebugWidget::showContextMenu
	(	int		button,	
		QListViewItem	*item,
		const QPoint	&,
		int
	)
{
	QPopupMenu	popup ;

	/* We are only interested in right-clicks in actual items ...	*/
	if ((item == 0) || (button != Qt::RightButton)) return ;
	popItem	= (TKCPyValueItem *)item ;

	TKCPyValue	*value	= popItem->value() ;
	PyObject	*pyObj	= value  ->value() ;

	TKCPyCookie	*cookie	= 0 ;
	uint		__lno	;

	switch (value->type()->typeCode)
	{
		case TKCPyType::Module 	:
			cookie	= getObjectModule (pyObj, __lno) ;
			break	;

		case TKCPyType::Function:
			cookie	= getObjectModule (pyObj, __lno) ;
			break	;

		case TKCPyType::Code	:
			cookie	= getObjectModule (pyObj, __lno) ;
			break	;

		default	:
			break	;
	}

	if (cookie != 0)
	{
		popup.insertItem (TR("Show source"), this, SLOT(showSource())) ;
		delete	cookie	 ;
	}

	/* There are a number of cases which effectively refer to	*/
	/* python code which can be breakpointed.			*/
	PyObject *pyCode = getCode (pyObj) ;
	if (pyCode != 0) addBreakOptions (popup, pyCode) ;

	if (popup.count() > 0) popup.exec (QCursor::pos()) ;
}

void	TKCPyDebugWidget::trapExceptions
	(	bool	excTrap
	)
{
	m_excTrap = excTrap ;

#if	__PY22PLUS
	TKCPyDebugBase::trapExceptions (m_excTrap) ;
#endif
}

/*  TKCPyDebugWidget							*/
/*  setBreakpoint: Set breakpoint					*/
/*  (returns)	 : void		:					*/

void	TKCPyDebugWidget::setBreakpoint ()
{
	TKCPyValue	*value	= popItem->value() ;
	PyObject	*code	= getCode (value->value()) ;
	int		lno	= ((PyCodeObject *)code)->co_firstlineno ;
	value->ref() ;

	TKCPyTraceItem	*trace	= new TKCPyTraceItem
				  (	m_pTraceView,
					popItem->key(),
					TKCPyValue::allocValue (code),
					true,
					lno
				  )	;

	TKCPyDebugBase::setTracePoint (code, trace, lno) ;

	TKCPyEditor *editor = showObjectCode (code) ;
	if (editor != 0)
		editor->setBreakpoint (lno) ;


}

/*  TKCPyDebugWidget							*/
/*  setWatchpoint: Set watchpoint					*/
/*  (returns)	 : void		:					*/

void	TKCPyDebugWidget::setWatchpoint ()
{
	TKCPyValue	*value	= popItem->value() ;
	PyObject	*code	= getCode (value->value()) ;

	value->ref() ;

	TKCPyTraceItem	*trace	= new TKCPyTraceItem
				  (	m_pTraceView,
					popItem->key(),
					TKCPyValue::allocValue (code),
					false
				  )	;

	TKCPyDebugBase::setTracePoint (code, trace) ;
}

/*  TKCPyDebugWidget							*/
/*  enableBreakpoint	: Enable breakpoint				*/
/*  (returns)	 	: void		:				*/

void	TKCPyDebugWidget::enableBreakpoint ()
{
	traceItem->enable (true) ;
}

/*  TKCPyDebugWidget							*/
/*  disableBreakpoint	: Disable breakpoint				*/
/*  (returns)	 	: void		:				*/

void	TKCPyDebugWidget::disableBreakpoint ()
{
	traceItem->enable (false) ;
}

/*  TKCPyDebugWidget							*/
/*  removeWatchpoint	: Remove a watchpoint				*/
/*  (returns)	 	: void		:				*/

void	TKCPyDebugWidget::removeWatchpoint ()
{
	TKCPyValue	*value	= popItem->value() ;
	PyObject	*code	= getCode (value->value()) ;

	TKCPyEditor *editor = showObjectCode (code) ;
	if (editor != 0)
		editor->clearBreakpoint (traceItem->getLineno()) ;

	TKCPyDebugBase::clearTracePoint
	(	traceItem->value()->value(),
		traceItem->getLineno	 ()
	)	;
	delete	traceItem ;


}

/*  TKCPyDebugWidget							*/
/*  showSource	: Show source code					*/
/*  (returns)	: void		:					*/

void	TKCPyDebugWidget::showSource ()
{
	showObjectCode (popItem->value()->value()) ;

}

void	TKCPyDebugWidget::clearBreakpoints
	(	TKCPyCookie	*
	)
{
}

/*  TKCPyDebugWidget							*/
/*  toggleBreakpoint							*/
/*		: Toggle breakpoint at specified line			*/
/*  module	: TKCPyCookie *	: Module cookie				*/
/*  lineNo	: uint		: Line number				*/
/*  editor	: TKCPyEditor *	: Associated editor			*/
/*  (returns)	: void		:					*/

void	TKCPyDebugWidget::toggleBreakpoint
	(	TKCPyCookie	*module,
		uint		lineNo,
		TKCPyEditor	*editor
	)
{
	TKCPyTraceItem	*trace	;
	PyObject	*pyMod	;

	if ((pyMod = TKCPyCookieToModule (module)) == 0)
		return	;

	trace	= (TKCPyTraceItem *)m_pTraceView->firstChild() ;
	while (trace != 0)
	{
		if ( (trace->value()->value() == pyMod )  &&
		     (trace->getLineno()      == lineNo) ) break ;

		trace	= (TKCPyTraceItem *)trace->nextSibling() ;
	}

	if (trace != 0)
	{
		delete	trace	;
		TKCPyDebugBase::clearTracePoint (pyMod, lineNo) ;
		editor->clearBreakpoint (lineNo) ;
		return	;
	}

	trace	= new TKCPyTraceItem
			  (	m_pTraceView,
				PyModule_GetName       (pyMod),
				TKCPyValue::allocValue (pyMod),
				true,
				lineNo
			  )	;

	TKCPyDebugBase::setTracePoint (pyMod, trace, lineNo) ;
	editor->setBreakpoint (lineNo) ;
}


/*  TKCPyDebugWidget							*/
/*  toggleBreakpoint							*/
/*		: Toggle breakpoint at current line			*/
/*  (returns)	: void		:					*/

void	TKCPyDebugWidget::toggleBreakpoint ()
{
	TKCPyEditor *editor = (TKCPyEditor *)m_pEditors->currentPage() ;

	if (editor != 0)
		toggleBreakpoint
		(	editor->getModule	(),
			editor->getCurrentLine	(),
			editor
		)	;
}

void	TKCPyDebugWidget::dropSource
	(	TKCPyCookie	*module
	)
{
	for (uint idx = 0 ; idx < editList.count() ; idx += 1)
		if (*editList.at(idx)->getModule() == *module)
		{	TKCPyEditor *editor = editList.at(idx) ;
			editList.remove (editor) ;
			delete	editor	;
			break	;
		}
}

/*  TKCPyDebugWidget							*/
/*  init	: Initialise from configuration				*/
/*  config	: TKConfig *	: Configuration				*/
/*  (returns)	: void		:					*/

void	TKCPyDebugWidget::init
	(	TKConfig	*config
	)
{
	QValueList<int> wMain   = config->readIntListEntry ("splitMain" ) ;
	QValueList<int> wRight  = config->readIntListEntry ("splitRight") ;

#if	! __KB_EMBEDDED
	m_pSplitMain ->setSizes (wMain ) ;
#endif
	m_pSplitRight->setSizes (wRight) ;

	excSkipList		= config->readListEntry    ("excSkipList") ;
}

/*  TKCPyDebugWidget							*/
/*  save	: Save to configuration					*/
/*  config	: TKConfig *	: Configuration				*/
/*  (returns)	: void		:					*/

void	TKCPyDebugWidget::save
	(	TKConfig	*config
	)
{
#if	! __KB_EMBEDDED
	config->writeEntry ("splitMain",   m_pSplitMain  ->sizes()) ;
#endif
	config->writeEntry ("splitRight",  m_pSplitRight ->sizes()) ;
	config->writeEntry ("excSkipList", excSkipList) ;
}


