/***************************************************************************
    file	         : kb_pyscript.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>
#include	<sys/stat.h>

#include	"sip.h"

#ifndef 	_WIN32
#include	"kb_pyscript.moc"
#include	<unistd.h>
#else
#include 	"kb_pyscript.h"
#include 	<process.h>
#include	<qsettings.h>
#include	<qmessagebox.h>
#include	<qfileinfo.h>
#include	<qdir.h>
#endif

#include	<errno.h>
#include	<time.h>

#include	<qdict.h>
#include	<qfile.h>
#include	<qtextstream.h>
#include 	<qmessagebox.h>

#include	"kb_python.h"

#include	"kb_classes.h"
#include	"kb_location.h"
#include	"kb_dbinfo.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_node.h"
#include	"kb_block.h"
#include	"kb_script.h"
#include	"kb_build.h"
#include	"kb_slot.h"

#include	"kb_locator.h"

#include	"kb_pydebug.h"

#include	"tkc_pydebug.h"
#include	"tkc_pydebugwidget.h"

#include	"tktexteditor.h"

#include	"pybase.h"


#define	EFUNC	"__expr_%1"
#define	PYHDR	"import RekallDefs.py\n\n"
#undef	PYHDR
#define	PYHDR	""
#define	EWRAP	"def %1 (_ctrl) :\n\t return %2\n"
#define	FWRAP	"%1"

extern	"C"
{
extern	void	initlibRekallc ()  ;
}


#ifdef	_SHOWREF
#define	SHOWREF(p,i)	fprintf (stderr, "Ref count %3d on %08x %s\n",	\
					 (p)->ob_refcnt,		\
					 (int)(p),			\
					 i) ;
#else
#define	SHOWREF(p,i)
#endif	// _SHOWREF

#ifdef	_NODECREF
#undef	Py_DECREF
#undef	Py_XDECREF
#define	Py_DECREF(x)
#define	Py_XDECREF(x)
#endif	// _NODECREF

#ifdef _WIN32
extern	__declspec(dllimport)PyObject	*PyExc_IOError	;
#else
extern	PyObject	*PyExc_IOError	;
#endif

/*  KBPYModule								*/
/*  ----------								*/
/*  Simple class used to record information about loaded python modules	*/
/*  Basically just contains the location and a time stamp used to	*/
/*  check for validity.							*/

class	KBPYModule
{
public	:

	KBLocation	location	;
	PyObject	*pyMod		;
	QString		stamp		;

	inline	KBPYModule
		(	KBLocation	&location,
			PyObject 	*pyMod,
			QString		stamp	= QString::null
		)
		:
		location (location),
		pyMod	 (pyMod),
		stamp	 (stamp)
	{
	}
}	;


	KBNotifier		*kbNotifier	;
	KBCallback		*kbCallback	;
	long			pyCookie	;
	QDict<PyObject>		pyClasses	;


static	QString	errMsg		;	/* Execution error message	*/
static	QString	errIdent	;	/* Execution error source	*/
static	uint	errLno		;	/* Execution error line number	*/
static	QString	errText		;	/* Execution traceback file	*/

static	uint	exprSeq		;	/* Unique expression index	*/

static	QDict<KBPYModule>	loadMap		;
static	QDict<KBPYModule>	nameMap		;
static	QDict<KBPYScriptCode>	inlineMap	;

static	PyObject		*pyBuiltin	;
static	PyObject		*rekallMod	;
static	PyObject		*rekallMain	;
static	PyObject		*rekallDict	;
static	QString			pyPath		;

static	PyObject 		*pyEventName	;
static	PyObject 		*pySlotsName	;
static	PyObject 		*pyCallName	;

static	TKCPyDebug		debugger	;
static	bool			debuggerOK	;
static	bool			errDebugged	;

static	KBPYScriptIF		*scrIface	;

static	QString	getPythonError	(cchar *)	;
static	QString	getCompileError	(KBLocation &, cchar *, QString &) ;

	PyObject		*pyThisName	;


/*  setErrDebugged							*/
/*		: Set error debugged flag				*/
/*  (returns)	: void		:					*/

void	TKCPySetErrDebugged ()
{
	/* Normally, after an error occurs, the error passed back to	*/
	/* the caller will indicate that the caller should display the	*/
	/* script file which contains the error. However, when a	*/
	/* debugger is running (which displays the script itself),	*/
	/* calling this routine will stop the caller displaying it.	*/
	errDebugged = true ;
}

/*  compileAndLoad							*/
/*		: Compile and load a script				*/
/*  module	: TKCPyCookie *	: Module cookie				*/
/*  eText	: QString &	: Python error text return		*/
/*  eError	: QString &	: Error return				*/
/*  eDetails	: QString *	: Error details return			*/
/*  pyErr	: bool &	: Python error occurred			*/
/*  (returns)	: bool		: Success				*/

bool	TKCPyCompileAndLoad
	(	TKCPyCookie	*module,
		QString		&eText,
		QString		&eError,
		QString		&eDetails,
		bool		&pyErr
	)
{
	/* This routine is used so that the debugger can have a script	*/
	/* compiled and loaded. It is done via a routine so that the	*/
	/* KBScriptIF interface does not have to be visible to the	*/
	/* debugger.							*/
	if (scrIface == 0)
	{
		eError	 = "Cannot compile: no python interface loaded?" ;
		eDetails = QString::null ;
		return	 false	;
	}

	QString	ePatt	;
	KBError	error	;

	if (!scrIface->load
		(	((TKCPyRekallCookie *)module)->getLocation(),
			eText,
			ePatt,
			error,
			pyErr
		))
	{
		eError   = error.getMessage () ;
		eDetails = error.getDetails () ;
		return	 false ;
	}

	return	true ;
}

/*  TKCPyModuleToCookie							*/
/*		: Get module cookie given identification		*/
/*  ident	: const QString & : Identification			*/
/*  (returns)	: TKCPyCookie *	  : Module or null if not found		*/

TKCPyCookie
	*TKCPyModuleToCookie
	(	const QString	&ident
	)
{
	KBPYModule	*module	;
	KBPYScriptCode	*code	;

	if ((module = loadMap  .find(ident)) != 0)
		return	new TKCPyRekallCookie (module->location) ;

	if ((code   = inlineMap.find(ident)) != 0)
		return	new TKCPyRekallCookie (code  ->location) ;

	return	0	;
}	

/*  TKCPyCookieToModule							*/
/*		: Get module for specified module cookie		*/
/*  cookie	: TKCPyCookie *	: Module cookie				*/
/*  (returns)	: PyObject *	: Module or null if not found		*/

PyObject*TKCPyCookieToModule
	(	TKCPyCookie *cookie
	)
{
	const QString	&ident	= ((TKCPyRekallCookie *)cookie)->getLocation().ident() ;
	KBPYModule	*module	;

	if ((module = loadMap.find (ident)) != 0)
		return	module->pyMod	;

	return	0 ;
}	

/*  getPythonString							*/
/*		: Get string for python object				*/
/*  pyobj	: PyObject *	: Python object				*/
/*  returns	: QString	: String				*/

QString	getPythonString
	(	PyObject	*pyobj
	)
{
	if (pyobj == 0)
		return	"<null>" ;

	if (PyString_Check (pyobj))
		return	PyString_AsString (pyobj) ;

	QString	 res	;
	PyObject *str	= PyObject_Str (pyobj) ;
	res	= PyString_AsString (str) ;
	Py_DECREF (str) ;
	return	res	;
}


/*  TKCPySetupEditor							*/
/*		: Set up editor for python				*/
/*  editor	: TKTextEditor * : Editor				*/
/*  (returns)	: void		 :					*/

void	TKCPySetupEditor
	(	TKTextEditor	*editor
	)
{
	editor->setHighlight ("Python") ;
}

/*  findPythonClass							*/
/*		: Find python class definition and add to dictionary	*/
/*  pyname	: cchar *	: Python class name			*/
/*  kbname	: cchar *	: Corresponding rekall element name	*/
/*  aliases	: cchar **	: Aliases				*/
/*  (returns)	: void		:					*/

void	findPythonClass
	(	cchar	*pyname,
		cchar	*kbname,
		cchar	**aliases
	)
{
	PyObject *clobj = PyDict_GetItemString(rekallDict, _C(pyname)) ;

	fprintf
	(	stderr,
		"findPythonClass: [%s][%s]\n",
		pyname,
		kbname
	)	;

	if (clobj == 0)
		KBError::EFatal
		(	QString("Unable to locate python class %1").arg(pyname),
			getPythonError (""),
			__ERRLOCN
		)	;
	if (!PyClass_Check (clobj))
		KBError::EFatal
		(	QString("%1 is not a python class").arg(pyname),
			QString::null,
			__ERRLOCN
		)	;

	pyClasses.insert (kbname, clobj) ;

	if (aliases != 0)
	{
//		fprintf	(stderr, "    ....") ;

		while (*aliases != 0)
		{
//			fprintf	(stderr, " %s", *aliases) ;
			pyClasses.insert (*aliases, clobj) ;
			aliases += 1 ;
		}

//		fprintf	(stderr, "\n") ;
	}
}

/*  getPythonError							*/
/*		: Get error details					*/
/*  deferr	: cchar *	: Default error text			*/
/*  (returns)	: QString	: Error details				*/

static	QString	getPythonError
	(	cchar	*deferr
	)
{
	PyObject *except ;
	PyObject *value  ;
	PyObject *traceb ;

	PyErr_Fetch		 (&except, &value, &traceb) ;
	if (except == 0) return deferr ;
	PyErr_NormalizeException (&except, &value, &traceb) ;

	Py_XDECREF(except) ;

	QString	 valtext = QString::null ;
	QString	 trbtext = QString::null ;
	if (value  != 0) valtext = getPythonString (value ) ;
	if (traceb != 0) trbtext = getPythonString (traceb) ;

	Py_XDECREF(value ) ;
	Py_XDECREF(traceb) ;

	return	QString("%1 %2").arg(valtext).arg(trbtext) ;
}

/*  getCompileError							*/
/*		: Get compilation error details				*/
/*  location	: KBLocation &	: Location of failed script		*/
/*  deferr	: cchar *	: Default error text			*/
/*  ePatt	: QString &	: Error pattern				*/
/*  (returns)	: QString	: Error details				*/

static	QString	getCompileError
	(	KBLocation	&location,
		cchar		*deferr,
		QString		&ePatt
	)
{
	QString	 eText	 ;

	PyObject *object ;
	PyObject *value  ;
	PyObject *traceb ;

	PyObject *emsg	 ;
	PyObject *detail ;
	PyObject *lnoN	 ;
	PyObject *lnoS	 ;

	ePatt	= ": *([0-9]*):" ;

	PyErr_Fetch (&object, &value, &traceb) ;
	if (value == 0) return deferr ;

	Py_XDECREF(object) ;
	Py_XDECREF(traceb) ;

	/* The error format seems to be					*/
	/*								*/
	/* 	( error_text, ( unknown, line_no, col_no, line_text ))	*/
	/*								*/
	/* If this is appears to be the case then format the error as	*/
	/*								*/
	/* 	source: line_no: error_text				*/
	/*								*/
	/* If not then convert the error object to its representation	*/
	/* as a string and use that.					*/
	if (!PyTuple_Check (value))
	{
		eText	= getPythonString (value) ;
		Py_XDECREF(value) ;
		return	eText	  ;
	}

	if (PyTuple_Size (value) != 2) 
	{
		eText	= getPythonString (value) ;
		Py_XDECREF(value) ;
		return	eText	  ;
	}

	emsg	= PyTuple_GetItem (value, 0) ;
	detail	= PyTuple_GetItem (value, 1) ;

	if ( !PyString_Check (emsg  ) ||
	     !PyTuple_Check  (detail) || (PyTuple_Size (detail) < 4) )
	{
		eText	= getPythonString (value) ;
		Py_XDECREF(value) ;
		return	eText	  ;
	}

	lnoN	= PyTuple_GetItem (detail, 1) ;
	lnoS	= PyObject_Str    (lnoN	    ) ;

	eText	= QString("%1 : %2: %2")
			 .arg(location.isFile() ? location.path () : location.docName)
			 .arg(PyString_AsString(lnoS))
			 .arg(PyString_AsString(emsg)) ;

	Py_XDECREF(value ) ;
	Py_XDECREF(lnoS  ) ;

	return	eText ;
}

/*  saveExeError: Save execution error					*/
/*  (returns)	: void		:					*/

static	void	saveExeError ()
{
	PyObject *except ;
	PyObject *value  ;
	PyObject *traceb ;

	errMsg	 = "Unknown execution error" ;
	errLno	 = 0  ;
	errIdent = "" ;

	PyErr_Fetch (&except, &value, &traceb) ;
	if (except == 0) return ;

	PyErr_NormalizeException (&except, &value, &traceb) ;
	if ((traceb == 0) || !PyTraceBack_Check (traceb))
	{
		errMsg	= QString("%1<br>%2")
				.arg(getPythonString (except))
				.arg(getPythonString (value )) ;
		return	;
	}

	PyObject *pyLineno = (PyTraceBack_Type.tp_getattr) (traceb, _C("tb_lineno")) ;
	PyObject *pyFrame  = (PyTraceBack_Type.tp_getattr) (traceb, _C("tb_frame" )) ;

	errMsg	 = PyString_AsString (((PyClassObject *)except)->cl_name) ;
	errLno	 = PyInt_AsLong      (pyLineno) ;
	errIdent = PyString_AsString (((PyFrameObject *)pyFrame)->f_code->co_filename) ;
#ifndef _WIN32
	QString	 errFile = QString  ("/tmp/rekall.py.%1").arg(getpid())  ;
#else
	QString errFile = QString(getenv("TMP")); 
	errFile	+= QString("\\rekall.py.%1").arg(getpid());
#endif
	PyObject *efd    = PyFile_FromString (_C(errFile), _C("w")) ;

	if (efd != 0)
	{
		PyTraceBack_Print (traceb, efd) ;
		Py_DECREF (efd)	;
	}
	else	errFile	= "" ;

	QFile	f  (errFile) ;
	f.open (IO_ReadOnly) ;
	errText = QTextStream (&f).read() ;
	f.remove () ;
}

/*  addPathElement							*/
/*		: Add element to python path				*/
/*  elem	: const QString & : Element				*/
/*  (returns)	: QString	  : New python path			*/

static	QString	addPathElement
	(	const QString	&elem
	)
{
	PyObject *path	= PySys_GetObject (_C("path")) ;
	int	 plen	= PyList_Size	  (path) ;
	QString	 text	= elem ;

	for (int idx = 0 ; idx < plen ; idx += 1)
	{
		PyObject *item = PyList_GET_ITEM (path, idx) ;

		text	+= DELIM ;
		text 	+= getPythonString (item) ;
	}
	fprintf	(stderr, "Python path: [%s]\n", (cchar *)text) ;
	return	text ;
}

/*  makePythonInst: Generate python class instance for thing		*/
/*  aclobj	  : PYObject *	: Python class object			*/
/*  name	  : QString &	: Name of thing				*/
/*  thing	  : long	: Thing in question			*/
/*  (returns)	  : PyObject *	: Python instance			*/

static	PyObject *makePythonInst
	(	 PyObject	*aclobj,
		 const QString	&name,
		 long		thing
	)
{
	if (aclobj == 0)
	{
		KBError::EFault
		(	QString(TR("Failed to locate python class for %1")).arg(name),
			QString::null,
			__ERRLOCN
		)	;
		return	0 ;
	}

	/* Generate a Python for the object. The cookie is used to try	*/
	/* to catch users trying to create such objects directly from	*/
	/* their own scripts.						*/
	PyObject *aclarg = Py_BuildValue(_C("(lls)"), pyCookie, (long)thing, (cchar *)name) ;
	PyObject *pyinst = PyInstance_New (aclobj,  aclarg,  0) ;
	SHOWREF	 (aclarg, "constructor args") ;
	Py_DECREF(aclarg) ;

	if (pyinst == 0)
	{
		KBError::EFault
		(	QString(TR("Unable to create instance for class %1")).arg(name),
			getPythonError(""),
			__ERRLOCN
		)	;
		return	0 ;
	}

	return	pyinst	;
}

/*  makePythonInst: Generate python class instance for database link	*/
/*  node	  : KBNode *	: Node to be mapped to python instance	*/
/*  (returns)	  : PyObject *	: Python instance			*/

PyObject*makePythonInst
	(	KBDBLink	*dbLink
	)
{
	PyObject *obj =	makePythonInst
			(	pyClasses.find ("KBDBLink"),
				"KBDBLink",
				(long)dbLink
			)	;

//	fprintf
//	(	stderr,
//		"makePythonInst(dbLink) ----> [%p]\n",
//		(void *)obj
//	)	;
	return	 obj	;
}

/*  makePythonInst: Generate python class instance for query handle	*/
/*  node	  : KBSQLQuery *: Query to be converted			*/
/*  (returns)	  : PyObject *	: Python instance			*/

PyObject*makePythonInst
	(	KBSQLQuery	*query
	)
{
	cchar		*name	;

	if	(query->isSelect()) name = "KBSQLSelect" ;
	else if	(query->isInsert()) name = "KBSQLInsert" ;
	else if	(query->isUpdate()) name = "KBSQLUpdate" ;
	else if	(query->isDelete()) name = "KBSQLDelete" ;
	else
	{
		KBError::EFault
		(	TR("makePythonInst called on unknown query object"),
			QString::null,
			__ERRLOCN
		)	;
		return	0 ;

	}

	return	makePythonInst (pyClasses.find(name), name, (long)query) ;
}

/*  makePythonInst: Generate python class instance for event		*/
/*  event	  : KBEvent *	: Event to be mapped to python instance	*/
/*  (returns)	  : PyObject *	: Python instance			*/

PyObject*makePythonInst
	(	KBEvent		*event
	)
{
	PyObject *obj =	makePythonInst
			(	pyClasses.find ("KBEvent"),
				"KBEvent",
				(long)event
			)	;

//	fprintf
//	(	stderr,
//		"makePythonInst(event) ----> [%p]\n",
//		(void *)obj
//	)	;
	return	 obj	;
}

/*  makePythonInst: Generate python class instance for slot		*/
/*  slot	  : KBSlot *	: Slot to be mapped to python instance	*/
/*  (returns)	  : PyObject *	: Python instance			*/

PyObject*makePythonInst
	(	KBSlot		*slot
	)
{
	PyObject *obj =	makePythonInst
			(	pyClasses.find ("KBSlot"),
				"KBSlot",
				(long)slot
			)	;

	fprintf
	(	stderr,
		"makePythonInst(slot) ----> [%p]\n",
		(void *)obj
	)	;
	return	 obj	;
}

/*  makePythonInst: Generate python class instance for node		*/
/*  node	  : KBNode *	: Node to be mapped to python instance	*/
/*  (returns)	  : PyObject *	: Python instance			*/

PyObject*makePythonInst
	(	KBNode		*node
	)
{
	if (node == 0)
		return	Py_None	;

	/* If the node is actually an object then see if it already	*/
	/* contains a corresponding script object, in which case we	*/
	/* have a corresponding persistent python object.		*/
	if (node->isObject())
	{
		KBPYScriptObject *so = (KBPYScriptObject *)node->isObject()->scriptObject() ;
		if (so != 0) return so->object() ;
	}

	const QString	&elem	= node->getElement    () ;
	PyObject 	*aclobj = pyClasses.find  (elem) ;

	/* If the class was not found then see if the node is an item	*/
	/* or an object. This should pick up any other nodes types and	*/
	/* provide some standard functionality for them.		*/
	if (aclobj == 0)
		if (node->isItem     ())
			aclobj = pyClasses.find ("KBItem"  ) ;
	if (aclobj == 0)
		if (node->isObject   ())
			aclobj = pyClasses.find ("KBObject") ;

	PyObject *pyObj	= makePythonInst (aclobj, elem, (long)node) ;

	/* If we now have a python object, and the node is actually an	*/
	/* object, make the python object persistent by saving a	*/
	/* pointer to it.						*/
	if ((pyObj != 0) && (node->isObject() != 0))
		node->isObject()->setScriptObject (new KBPYScriptObject (pyObj)) ;


	/* Next stage is to add access to the objects events and slots.	*/
	/* We will do this by adding two dummy objects, so we first	*/
	/* need to build dummy classes, containing the events and slots	*/
	/* respectively as methods.					*/
	PyObject *pyEventDict  = PyDict_New() ;
	PyObject *pySlotsDict  = PyDict_New() ;

	PyDict_SetItemString (pyEventDict, _C("__module__"), pyEventName) ;
	PyDict_SetItemString (pySlotsDict, _C("__module__"), pySlotsName) ;


	LITER
	(
		KBAttr,
		node->getAttribs(),
		attr,

		if (attr->isEvent() == 0) continue ;

		/* Attribute is an event. Make a python instance of the	*/
		/* event, then extract the "call" method which we use	*/
		/* to do the work. This gives us a bound method call	*/
		/* which we insert into the disctionary.		*/
		PyObject *obj   = makePythonInst      (attr->isEvent())	    ;
		PyObject *call  = PyObject_GetAttr    (obj, pyCallName)     ;
		PyObject *aname = PyString_FromString (_C(attr->getName())) ;

		PyDict_SetItem
		(
			pyEventDict,
			aname,
			call
		)	;

		Py_DECREF(obj  ) ;
		Py_DECREF(aname) ;
		Py_DECREF(call ) ;
	)

	LITER
	(
		KBSlot,
		node->getSlots(),
		slot,

		PyObject *obj	= makePythonInst      (slot) ;
		PyObject *call  = PyObject_GetAttr    (obj, pyCallName)  ;
		PyObject *aname = PyString_FromString (_C(slot->name())) ;

		PyDict_SetItem
		(
			pySlotsDict,
			aname,
			call
		)	;

		Py_DECREF(obj  ) ;
		Py_DECREF(aname) ;
		Py_DECREF(call ) ;
	)

	PyObject *pyInstDict	= ((PyInstanceObject *)pyObj)->in_dict ;
	PyObject *pyEventClass	= PyClass_New (0, pyEventDict, PyString_FromString("event")) ;
	PyObject *pySlotsClass	= PyClass_New (0, pySlotsDict, PyString_FromString("slots")) ;

	PyDict_SetItemString	(pyInstDict, _C("__events__"), pyEventClass) ;
	PyDict_SetItemString	(pyInstDict, _C("__slots__" ), pySlotsClass) ;

	Py_DECREF	(pyEventDict ) ;
	Py_DECREF	(pySlotsDict ) ;
	Py_DECREF	(pyEventClass) ;
	Py_DECREF	(pySlotsClass) ;

	return	pyObj	;
}

/*  execFunc	: Execute function					*/
/*  func	: PyObject *	  : Function object			*/
/*  node	: PyObject *	  : Invoking node			*/
/*  argc	: uint		  : Argument count			*/
/*  argv	: const KBValue * : Argument vector			*/
/*  resval	: KBValue &	  : Result value			*/
/*  source	: KBNode *	  : Source object			*/
/*  event	: const QString & : Source event			*/
/*  (returns)	: KBScript::ExeRC : Return code				*/

static	KBScript::ExeRC	execFunc
	(	PyObject	*func,
		PyObject	*node,
		uint		argc,
		const KBValue	*argv,
		KBValue		&resval,
		KBNode		*source	= 0,
		const QString	event	= QString::null
	)
{
	uint	 extra	  = source == 0 ? 0 : 2 ;

	PyObject	*argvec = PyTuple_New (argc + 1 + extra) ;
	PyTuple_SetItem (argvec, 0, node) ;

	SHOWREF	 (argvec, "argument list"  ) ;
	SHOWREF	 (pyinst, "object instance") ;

	if (source != 0)
	{
		PyObject *pyinst = makePythonInst (source) ;
		if (pyinst == 0) return KBScript::ExeError ;

		PyTuple_SetItem (argvec, 1,  pyinst) ;
		PyTuple_SetItem (argvec, 2,  PyString_FromString (event)) ;
	}

	for (uint idx = 0 ; idx < argc ; idx += 1)
	{
		const QString	&text	= argv[idx].getRawText() ;
		KBType		*type	= argv[idx].getType   () ;
		PyObject	*pyobj	;

		if (text.isNull())
		{
			PyTuple_SetItem (argvec, idx + extra + 1, Py_None) ;
			continue ;
		}

		switch (type->getIType())
		{
			case KB::ITString :
				pyobj	= PyString_FromString (text) ;
				break	;

			case KB::ITFixed  :
				pyobj	= PyLong_FromLong     (atol (text)) ;
				break	;

			case KB::ITFloat  :
				pyobj	= PyFloat_FromDouble  (strtod (text, 0)) ;
				break	;

			default	:
				pyobj	= PyString_FromString ("") ;
				break	;
		}

		PyTuple_SetItem (argvec, idx + extra + 1, pyobj) ;
	}

	/* Clear the error-debugged flag. The debugger will set this if	*/
	/* it handles an error in while executing the function.		*/
	errDebugged = false  ;

	PyObject *res = PyEval_CallObject (func, argvec) ;
	SHOWREF	 (argvec, "argument list") ;
	Py_DECREF(argvec) ;

	if (res == 0)
	{
		KBScript::ExeRC rc =  errDebugged ? KBScript::ExeFail  :
						    KBScript::ExeError ;
		saveExeError () ;
		errDebugged  = false ;
		
		return	rc ;
	}

	bool	rc	;

	if	(PyLong_Check   (res))
	{	long	r = PyLong_AsLong     (res) ;
		resval	  = KBValue ((int)r) ;
		rc	  = r != 0 ;
	}
	else if	(PyInt_Check    (res))
	{	int	i = PyInt_AsLong      (res) ;
		resval	  = KBValue ((int)i) ;
		rc	  = i != 0 ;
	}
	else if	(PyFloat_Check  (res))
	{	double  d = PyFloat_AsDouble  (res) ;
		resval	  = KBValue (d) ;
		rc	  = d != 0.0    ;
	}
	else if (PyString_Check (res))
	{	char   *s = PyString_AsString (res) ;
		resval	  = KBValue (s) ;
		rc	  = s[0] != 0	;
	}
	else
	{	resval	  = KBValue ()	;
		rc	  = true	;
	}

	SHOWREF	 (res, "result") ;
	Py_DECREF(res) ;
	return	rc ? KBScript::ExeTrue : KBScript::ExeFalse ;
}

/*  execFunc	: Execute function					*/
/*  func	: PyObject *	  : Function object			*/
/*  node	: KBNode *	  : Invoking node			*/
/*  argc	: uint		  : Argument count			*/
/*  argv	: const KBValue * : Argument vector			*/
/*  resval	: KBValue &	  : Result value			*/
/*  source	: KBNode *	  : Source object			*/
/*  event	: const QString & : Source event			*/
/*  (returns)	: KBScript::ExeRC : Return code				*/

static	KBScript::ExeRC	execFunc
	(	PyObject	*func,
		KBNode		*node,
		uint		argc,
		const KBValue	*argv,
		KBValue		&resval,
		KBNode		*source	= 0,
		const QString	event	= QString::null
	)
{
	/* Create a python class instance for the sepecified node. This	*/
	/* can fail, for instance of the node type is unknown here.	*/
	PyObject *pyinst = makePythonInst (node)   ;
	if (pyinst == 0) return KBScript::ExeError ;

	return	execFunc
		(	func,
			pyinst,
			argc,
			argv,
			resval,
			source,
			event
		)	;
}

/*  ------------------------------------------------------------------  */

/*  KBPYScriptObject							*/
/*  KBPYScriptObject							*/
/*		: Constructor for persistent script object		*/
/*  pyObject	: PyObject *	   : Associated python object		*/
/*  (returns)	: KBPYScriptObject :					*/

KBPYScriptObject::KBPYScriptObject
	(	PyObject	*pyObject
	)
	:
	m_pyObject (pyObject)
{
	/* Since we are keeping a pointer to the python object, we need	*/
	/* to increment its use count.					*/
	Py_XINCREF (pyObject) ;
}

/*  KBPYScriptObject							*/
/*  KBPYScriptObject							*/
/*		: Destructor for persistent script object		*/
/*  (returns)	:		:					*/

KBPYScriptObject::~KBPYScriptObject ()
{
	/* Decrement the use count. Most likely this will result in the	*/
	/* desctruction of the python object.				*/
//	fprintf
//	(	stderr,
//		"KBPYScriptObject::~KBPYScriptObject: count %d\n",
//		m_pyObject->ob_refcnt
//	)	;
	Py_XDECREF (m_pyObject) ;
}

/*  KBPYScriptObject							*/
/*  object	: Retrieve python object				*/
/*  (returns)	: PyObject *	: Python object				*/

PyObject
	*KBPYScriptObject::object ()
{
	/* Increment the use count on the object before returning it	*/
	/* to the caller.						*/
//	fprintf
//	(	stderr,
//		"KBPYScriptObject::object: count %d\n",
//		m_pyObject->ob_refcnt
//	)	;
	Py_XINCREF (m_pyObject) ;
	return	m_pyObject	;
}



/*  ------------------------------------------------------------------  */

/*  KBPYScriptCode							*/
/*  KBPYScriptCode: Constructor for compiled (inline) script functions	*/
/*  pyDunc	  : PyObject *	    : Python function			*/
/*  location	  : KBLocation &    : Nominal inline location		*/
/*  inherit	  : KBEvent *	    : Inherited event			*/
/*  (returns)	  : KBPYScriptCode  :					*/
	
KBPYScriptCode::KBPYScriptCode
	(	PyObject	*pyFunc,
		KBEvent		*inherit,
		KBLocation	&location
	)
	:
	KBScriptCode	(inherit),
	pyFunc		(pyFunc),
	location	(location)
{
PyCodeObject *pyCode = (PyCodeObject *)((PyFunctionObject *)pyFunc)->func_code ;
fprintf	(stderr, "****\n**** %s\n****\n",
(cchar *)getPythonString(pyCode->co_filename)) ;
	inlineMap.insert (location.ident(), this) ;
}

/*  KBPYScriptCode							*/
/*  ~KBPYScriptCode: Destructor for compiled script object		*/
/*  (returns)	   :			:				*/

KBPYScriptCode::~KBPYScriptCode ()
{
	TKCPyDebugWidget *debWidget ;
	if ((debWidget = TKCPyDebugWidget::widget()) != 0)
	{
		TKCPyRekallCookie cookie (location) ;
		debWidget->dropSource (&cookie) ;
	}


	inlineMap.remove (location.ident()) ;
	Py_XDECREF	 (pyFunc) ;
}

/*  KBPYScriptCode							*/
/*  execute	: Execute compiled code					*/
/*  node	: KBNode *	  : Invoking node			*/
/*  argc	: uint		  : Argument count			*/
/*  argv	: const KBValue * : Argument vector			*/
/*  resval	: KBValue &	  : Result value			*/
/*  (returns)	: KBScript::ExeRC : Success				*/

KBScript::ExeRC	KBPYScriptCode::execute
	(	KBNode		*node,
		uint		argc,
		const KBValue	*argv,
		KBValue		&resval
	)
{

	/* Create a python class instance for the sepecified node. This	*/
	/* can fail, for instance of the node type is unknown here.	*/
	PyObject *pyinst = makePythonInst (node)	;
	if (pyinst == 0) return KBScript::ExeError	;

	PyObject *sipObj = node   == 0 ? 0 : PyDict_GetItem(((PyInstanceObject *)pyinst)->in_dict, pyThisName) ;
	PyKBBase *kbBase = sipObj == 0 ? 0 : (PyKBBase *)((sipThisType *)sipObj)->u.cppPtr ;
	KBEvent	 *saved	 = 0 ;

	if (kbBase != 0)
	{
		fprintf
		(	stderr,
			"KBPYScriptCode::execute: saving  event: %p <- %p\n",
			(void *)kbBase->inherit,
			(void *)m_inherit
		)	;

		saved		= kbBase->inherit ;
		kbBase->inherit	= m_inherit	  ;
	}

	KBScript::ExeRC	rc = execFunc
			     (		pyFunc,
					pyinst,
					argc,
					argv,
					resval,
					0,
					QString::null
			     )	;

	if (kbBase != 0)
	{
		fprintf
		(	stderr,
			"KBPYScriptCode::execute: restore event: %p\n",
			(void *)saved
		)	;

		kbBase->inherit	= saved ;
	}

	return	rc ;
}

/*  KBPYScriptCode							*/
/*  execute	: Execute compiled code					*/
/*  node	: KBNode *	  : Invoking node			*/
/*  source	: KBNode *	  : Source node				*/
/*  event	: const QString & : Source event			*/
/*  argc	: uint		  : Argument count			*/
/*  argv	: const KBValue * : Argument vector			*/
/*  resval	: KBValue &	  : Result value			*/
/*  (returns)	: KBScript::ExeRC : Success				*/

KBScript::ExeRC	KBPYScriptCode::execute
	(	KBNode		*node,
		KBNode		*source,
		const QString	&event,
		uint		argc,
		const KBValue	*argv,
		KBValue		&resval
	)
{
	/* This method is used for event-slot script invocation.	*/
	return	execFunc (pyFunc, node, argc, argv, resval, source, event) ;
}

void	KBPYScriptCode::setBreakpoints
	(	const QValueList<int>	&bptList
	)
{
	fprintf
	(	stderr,
		"KBPYScriptCode::setBreakpoints: %d breakpoints\n",
		bptList.count()
	)	;

//	TKCPyDebugWidget *debWidget ;
//	if ((debWidget = TKCPyDebugWidget::widget()) != 0)
//		debWidget->setBreakpoint
//		(	pyFunc,
//			bptList,
//			location.ident()
//		)	;
}

/*  ------------------------------------------------------------------  */

/*  KBPYScriptIF							*/
/*  KBPYScriptIF: Constructor for Python script interface language	*/
/*  (returns)	: KBPYScriptIF	:					*/

KBPYScriptIF::KBPYScriptIF ()
{
	if (Py_IsInitialized()) return ;


	PyExc_IOError	= 0 ;
	Py_Initialize() ;

	QString	err	;
	if ((err = debugger.init()) != QString::null)
		KBError::EError
		(	TR("Python debugger initialisation error"),
			err,
			__ERRLOCN
		)	;
	else	debuggerOK = true ;

	if ((pyBuiltin = PyImport_AddModule (_C("__builtin__"))) == 0)
		KBError::EFatal
		(	TR("Inline python script execution failed"),
			TR("Failed to find python module __builtin__"),
			__ERRLOCN
		)	;

#ifndef _WIN32
	QString	hdrdir	= locateDir ("appdata", "script/py/Rekall.py") ;
#else
	QString	hdrdir	= locateDir ("appdata", "script\\py\\Rekall.py") ;
#endif

	if (hdrdir.isEmpty())
		KBError::EFatal
		(	TR("Unable to locate main python script directory"),
			QString::null,
			__ERRLOCN
		)	;

#ifndef _WIN32
	pyPath	= addPathElement (hdrdir += "script/py") ;
#else
	
	char tempStr[_MAX_PATH];
	ZeroMemory(tempStr, _MAX_PATH);
	HKEY hKey;
	unsigned long numBytes = sizeof(tempStr);
	unsigned long type;
	long retCode;

	//extern	const QString &getInstDir() ;
	extern	const QString &getRootDir() ;

	//const	QString	&instDir = getInstDir() ;
	const	QString	&rootDir = getRootDir() ;

	//pyPath	= QDir::convertSeparators(instDir + "\\python\\python22;");
	//pyPath += QDir::convertSeparators(instDir + "\\python\\python22\\Lib;");
	//pyPath += QDir::convertSeparators(instDir + "\\python\\python22\\DLLs;");
	//pyPath += QDir::convertSeparators(rootDir + "\\data\\script\\py");
	
	retCode = RegOpenKeyExA(HKEY_LOCAL_MACHINE, "SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\App Paths\\Python.exe", 
		0, KEY_ALL_ACCESS, &hKey);

	if(retCode == ERROR_SUCCESS)
	{
		RegQueryValueExA(hKey, "", NULL, &type, (unsigned char*)tempStr, (unsigned long*)&numBytes);
		fprintf(stderr, "The value of tempStr is %s\n", (char*)tempStr);
		QString pathToPython(tempStr);
	
		int pos = pathToPython.findRev('\\');
	
		const QString instDir = pathToPython.left(pos);

		pyPath	= QDir::convertSeparators(instDir + ";");
		pyPath += QDir::convertSeparators(instDir + "\\Lib;");
		pyPath += QDir::convertSeparators(instDir + "\\DLLs;");
		pyPath += QDir::convertSeparators(rootDir + "\\data\\script\\py");
	}
	else
	{
		QString msgStr = "The Python interpreter can not be found.\n";
		msgStr += "Please visit http://www.python.org\n";
		msgStr += "and downlaod a copy of Python V2.2.x";

		QMessageBox::critical(0L, "Python Interpretor Error", msgStr);
		return;
	}

	RegCloseKey(hKey);
#endif

	fprintf	(stderr, "PYPATH=%s\n", (cchar *)pyPath) ;
	PySys_SetPath (_C(strdup(pyPath.latin1()))) ;

	/* Initialise the Rekall Python interface, and load the main	*/
	/* Rekall module. The absense of the latter is a fatal error.	*/
	initlibRekallc () ;

	KBPYModule *module;
	KBLocation locn	  ;

	rekallMod  = PyImport_ImportModule (_C("Rekall")) ;
	if (rekallMod == 0)
		KBError::EFault
		(	TR("Failed to load python module 'Rekall'"),
			getPythonError ("Import module failed"),
			__ERRLOCN
		)	;
	rekallDict = PyModule_GetDict (rekallMod) ;

	locn	   = KBLocation (0, "script", "builtin", "Rekall") ;
	module	   = new KBPYModule   (locn, rekallMod, (QString)"") ;
	nameMap.insert ("Rekall", module) ;

	rekallMain = PyImport_ImportModule (_C("RekallMain")) ;
	if (rekallMain == 0)
		KBError::EFault
		(	TR("Failed to load python module 'RekallMain'"),
			getPythonError ("Import module failed"),
			__ERRLOCN
		)	;

	locn	   = KBLocation (0, "script", "builtin", "RekallMain") ;
	module	   = new KBPYModule   (locn, rekallMain, (QString)"") ;
	nameMap.insert ("RekallMain", module) ;

	pyEventName  = PyString_FromString("Events") ;
	pySlotsName  = PyString_FromString("Slots" ) ;
	pyCallName   = PyString_FromString("call"  ) ;

	pyCookie   = time(0) * getpid() * (long)&pyCookie ;

	/* Now locate the class objects for each of the Python classes	*/
	/* which correspond to Rekall objects; an error in any of these	*/
	/* will be treated as fatal.					*/
	extern	void	registerClasses () ;
	registerClasses	() ;

	extern	void	initPYBasic 	() ;
	initPYBasic	() ;

	/* SIP uses "sipThis" as the entry name to point to its own	*/
	/* python object. We need to access this at some points.	*/
	pyThisName = PyString_FromString("sipThis") ;

	scrIface = this ;
}

/*  KBPYScriptIF							*/
/*  ~KBPYScriptIF: Destructor for Python script interface language	*/

KBPYScriptIF::~KBPYScriptIF ()
{
	scrIface = 0	;
}

/*  KBPYScriptIF							*/
/*  getIface	: Get interface object if any				*/
/*  (returns)	: KBPYScriptIF * : Interface object or null		*/

KBPYScriptIF
	*KBPYScriptIF::getIface ()
{
	return	scrIface	;
}

/*  KBPYScriptIF							*/
/*  setApp	: Set application pointers				*/
/*  _kbNotifer	: KBNotifier *	: Application notifier			*/
/*  _kbCallback	: KBCallback *	: Application callback			*/
/*  (returns)	: void		:					*/

void	KBPYScriptIF::setApp
	(	KBNotifier	*_kbNotifier,
		KBCallback	*_kbCallback
	)
{
	kbNotifier	= _kbNotifier ;
	kbCallback	= _kbCallback ;
}


/*  KBPYScriptIF							*/
/*  load	: Load a module into the interpreter instance		*/
/*  location	: KBLocation &	: Database location			*/
/*  eText	: QString &	: Error text				*/
/*  ePatt	: QString &	: Error pattern				*/
/*  pError	: KBError &	: Error return				*/
/*  pyErr	: bool &	: Error is python error			*/
/*  (returns)	: bool		:					*/

bool	KBPYScriptIF::load
	(	KBLocation	&location,
		QString		&eText,
		QString		&ePatt,
		KBError		&pError,
		bool		&pyErr
	)
{
	KBPYModule  *module 	;
	PyObject    *pyCode	;
	PyObject    *pyMod	;

	fprintf
	(	stderr,
		"KBPYScriptIF::load: location.dbInfo=%p\n",
		(void *)location.dbInfo
	)	;

	/* Error in getting the timestamp or in getting the script text	*/
	/* are not python errors; everything else is.			*/
	QString	    stamp	= location.timestamp (pError) ;
	if (stamp == QString::null)
	{	pyErr	= false	;
		return	false	;
	}

	/* First see if the actual module has been loaded, so that it	*/
	/* can be shared between multiple instances, and if present	*/
	/* that it has not become out-of-date.				*/
	if ((module = loadMap.find (location.ident())) != 0)
	{
		fprintf	(stderr, "KBScriptIF::load(%s) %s/%s\n",
				 (cchar *)location.ident(),
				 (cchar *)stamp,
				 (cchar *)module->stamp) ;

		if (module->stamp == stamp)
		{	pyErr	= false	;
			return	true	;
		}
	}

	/* Get the module text (and check for errors), and then compile	*/
	/* it to a code object; check at this point for compilation	*/
	/* errors.							*/
	QString	   text	= location.contents  (pError) ;
	if (text.isNull())
	{	pyErr	= false	;
		return	false	;
	}

	if (location.dbInfo != 0)
#ifndef _WIN32
		PySys_SetPath (_C(QString("%1:%2").arg(location.dbInfo->getDBPath()).arg(pyPath))) ;
#else
		PySys_SetPath(_C(QString("%1;%2").arg(location.dbInfo->getDBPath()).arg(pyPath)));
#endif
	else	PySys_SetPath (_C(pyPath)) ;

	if ((pyCode = compileText
		      	(	location,
				text,
				eText,
				ePatt,
				pError
			)) == 0)
	{
		pyErr	= true	;
		return	false	;
	}

	/* OK to load the module. The use of ExecCodeModuleEx is based	*/
	/* on a reading for the Python 2.1 code, but seems to work OK	*/
	/* with Python 1.5 as well.					*/
	if ((pyMod  = PyImport_ExecCodeModuleEx
			(	_C(location.docName),
				pyCode,
				_C(location.ident())
			)) == 0)
	{
		pError	= KBError
			  (	KBError::Error,
			  	QString (TR("Error loading python module %1"))
					.arg(location.docName),
				getPythonError ("Failed to import module"),
				__ERRLOCN
			  )	;
		Py_DECREF (pyCode) ;
		pyErr	  = true   ;
		return	  false	   ;
	}

	/* Create a new module object, and insert (or replace) it in	*/
	/* the module dictionary.					*/
	module	= new KBPYModule (location, pyMod, stamp) ;
	loadMap.replace (location.ident(), module) ;

	QString	name	= location.docName	   ;
	int	lsep	= name.findRev ('/') 	   ;
	if (lsep >= 0) name = name.mid (lsep + 1)  ;
	nameMap.replace (name, module)		   ;
	fprintf	(stderr, "Namemap added [%s]\n", (cchar *)name) ;

	TKCPyDebugWidget *debWidget ;
	if ((debWidget = TKCPyDebugWidget::widget()) != 0)
		debWidget->codeLoaded () ;

	pyErr	= false	;
	return	true	;
}

/*  KBPYScriptIF							*/
/*  load	: Load a module into the interpreter instance		*/
/*  location	: KBLocation &	: Database location			*/
/*  eText	: QString &	: Error text				*/
/*  ePatt	: QString &	: Error pattern				*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: bool		:					*/

bool	KBPYScriptIF::load
	(	KBLocation	&location,
		QString		&eText,
		QString		&ePatt,
		KBError		&pError
	)
{
	bool	pyErr	;
	return	load	(location, eText, ePatt, pError, pyErr) ;
}

/*  KBELScriptIF							*/
/*  compileText	: Compile script text					*/
/*  location	: KBLocation &	  : Document location			*/
/*  script	: const QString & : Script text				*/
/*  eText	: QString &	  : Error text				*/
/*  ePat	: QString &	  : Error pattern			*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: bool		  : Success				*/

PyObject*KBPYScriptIF::compileText
	(	KBLocation	&location,
		const QString	&script,
		QString		&eText,
		QString		&ePatt,
		KBError		&pError
	)
{
	/* Compile the text. If this succeeds then pass back the code	*/
	/* object with no more ado.					*/
	PyObject *pyobj = Py_CompileString
			  (	_C(script),
				_C(location.ident()),
				Py_file_input
			  )	;

	if (pyobj != 0) return pyobj ;

	/* Got an error. If possible then get the error message, which	*/
	/* is passed back via the error object.				*/
	eText	= getCompileError
		  (	location,
		  	"Unknown python compilation error occurred",
		  	ePatt
		  )	;

	pError	= KBError
		  (	KBError::Error,
			TR("Error compiling python script"),
			eText,
			__ERRLOCN
		  )	;

	return	0	;
}

/*  KBPYScriptIF							*/
/*  compileInline: Compile inline script				*/
/*  script	 : const QString & : Script text			*/
/*  fname	 : const QString & : Entry function name		*/
/*  eText	 : QString &	   : Error text				*/
/*  ePat	 : QString &	   : Error pattern			*/
/*  iList	 : QStringList &   : Import list			*/
/*  inherit	 : KBEvent *	   : Inherited event			*/
/*  pError	 : KBError &	   : Error return			*/
/*  (returns)	 : KBScriptCode *  : Compiled script or null on error	*/

KBScriptCode
	*KBPYScriptIF::compileInline
	(	const QString		&script,
		const QString		&path,
		const QString		&fname,
		QString			&eText,
		QString			&ePatt,
		const QStringList	&iList,
		KBEvent			*inherit,
		KBError			&pError
	)
{
	PyObject	*pyTDict   = PyDict_New() ;
	PyObject	*pyInline  = 0 ;
	PyObject	*pyCode    = 0 ;
	PyObject	*pyFunc    = 0 ;
	PyObject	*pyFDict   ;
	PyObject	*pyModule  ;

	KBLocation	inlineLoc
			(	0,
				"script",
				KBLocation::m_pInline,
				path + ":" + fname,
				script
			)	;

	/* First compile the text of the inline script. This should	*/
	/* yeild code which represents the script itself.		*/
	if ((pyInline = compileText (inlineLoc, script, eText, ePatt, pError)) == 0)
		goto	error	;

	/* Next evaluate the code with a temporary dictionary as its	*/
	/* environment. Since it should do nothing other than define a	*/
	/* function, this should work.					*/
	if ((pyCode = PyEval_EvalCode ((PyCodeObject *)pyInline, pyTDict, pyTDict)) == 0)
	{	pError	= KBError
			  (	KBError::Error,
				TR("Error creating inline code function"),
				script,
				__ERRLOCN
			  )	;
		goto	error	;
	}

	/* Now we should be able to locate the code of the function as	*/
	/* specified by the function name ...				*/
	if ((pyFunc = PyDict_GetItemString (pyTDict, _C(fname))) == 0)
	{	pError	= KBError
			  (	KBError::Error,
				TR("Failed to locate inline code function"),
				script,
				__ERRLOCN
			  )	;
		goto	error	;
	}

	Py_INCREF(pyFunc) ;


	/* OK. We now need to locate each module in the imports list	*/
	/* and store it in the function's dictionary. This should be	*/
	/* like the code doing an import.				*/
	Py_DECREF (pyTDict) ;
	pyTDict	  = 0	    ;
	pyFDict   = PyFunction_GET_GLOBALS(pyFunc) ;


	for (QStringList::ConstIterator it = iList.begin() ; it != iList.end() ; ++it)
	{
		/* Skip RekallMain. It is now imported by default, just	*/
		/* below this loop.					*/
		if (*it == "RekallMain")
			continue ;

		if ((pyModule = PyImport_ImportModule (_C(*it))) == 0)
		{
			pError	= KBError
				  (	KBError::Error,
					QString	(TR("Cannot import module %1")).arg(*it),
					QString::null,
					__ERRLOCN
				  )	;
			fprintf	(stderr, "Inline: lacking  [%s]\n", (cchar *)(*it)) ;
			goto	error	;
		}

		Py_INCREF(pyModule) ;

		PyDict_SetItem
		(	pyFDict,
			PyString_FromString (_C(*it)),
			pyModule
		)	;

		fprintf	(stderr, "Inline: imported [%s]\n", (cchar *)(*it)) ;
	}

	/* As of release, always import "RekallMain". It is so often,	*/
	/* and particulary in components.				*/
	if ((pyModule = PyImport_ImportModule (_C("RekallMain"))) == 0)
	{
		pError	= KBError
			  (	KBError::Error,
				QString	(TR("Cannot import module RekallMain")),
				QString::null,
				__ERRLOCN
			  )	;
		fprintf	(stderr, "Inline: lacking  [RekallMain]\n") ;
		goto	error	;
	}

	Py_INCREF	(pyModule) ;
	PyDict_SetItem	(pyFDict, PyString_FromString("RekallMain"), pyModule) ;


	Py_INCREF (pyBuiltin) ;

	PyDict_SetItem
	(	pyFDict,
		PyString_FromString (_C("__builtins__")),
		pyBuiltin
	)	;

	return	new KBPYScriptCode (pyFunc, inherit, inlineLoc) ;

	error	:
		Py_XDECREF (pyTDict ) ;
		Py_XDECREF (pyInline) ;
		Py_XDECREF (pyCode  ) ;
		Py_XDECREF (pyFunc  ) ;

		fprintf	(stderr, "Compile inline failed\n"
				 "----\n"
				 "%s"
				 "----\n", (cchar *)script) ;
		return	false	;
}

/*  KBPYScriptIF							*/
/*  compileFunc	: Compile inline script function			*/
/*  script	: const QString & : Script text				*/
/*  fname	: const QString & : Entry function name			*/
/*  eText	: QString &	  : Error text				*/
/*  ePat	: QString &	  : Error pattern			*/
/*  iList	: QStringList &	  : Import list				*/
/*  inherit	: KBEvent *	  : Inherited event			*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: KBScriptCode *  : Compiled script or null on error	*/

KBScriptCode *KBPYScriptIF::compileFunc
	(	const QString		&script,
		const QString		&path,
		const QString		&fname,
		QString			&eText,
		QString			&ePatt,
		const QStringList	&iList,
		KBEvent			*inherit,
		KBError			&pError
	)
{
	return	compileInline
		(	QString(FWRAP).arg(script),
			path,
			fname,
			eText,
			ePatt,
			iList,
			inherit,
			pError
		)	;
}

/*  KBPYScriptIF							*/
/*  compileExpr	: Compile inline script expression			*/
/*  script	: const QString & : Script text				*/
/*  eText	: QString &	  : Error text				*/
/*  ePat	: QString &	  : Error pattern			*/
/*  iList	: QStringList &	  : Import list				*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: KBScriptCode *  : Compiled script or null on error	*/

KBScriptCode *KBPYScriptIF::compileExpr
	(	const QString		&script,
		const QString		&path,
		QString			&eText,
		QString			&ePatt,
		const QStringList	&iList,
		KBError			&pError
	)
{
	QString	fname	= QString(EFUNC).arg(exprSeq++) ;

	return	compileInline
		(	QString(EWRAP).arg(fname).arg(script),
			path,
			fname,
			eText,
			ePatt,
			iList,
			0,
			pError
		)	;
}


/*  KBPYScriptIF							*/
/*  compile	: Compile a script 					*/
/*  location	: KBLocation &	: Script location			*/
/*  eText	: QString &	: Error text				*/
/*  ePat	: QString &	: Error pattern				*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: bool		: Success				*/

bool	KBPYScriptIF::compile
	(	KBLocation	&location,
		QString		&eText,
		QString		&ePatt,
		KBError		&pError
	)
{

	QString	doc = location.contents (pError) ;
	if (doc == QString::null) return false	 ;

	PyObject *pyobj = compileText (location,
				       doc,
				       eText,
				       ePatt,
				       pError) ;

	if (pyobj != 0)
	{
		/* Seems OK, so bin the compiled object and return	*/
		/* success.						*/
		Py_DECREF (pyobj) ;
		return	  true	  ;
	}

	return	false	;
}


/*  findFunction: Find a function					*/
/*  scrList	: const QStringList &					*/
/*				  : List of script modules to search	*/
/*  fName	: const QString & : Function name			*/
/*  (returns)	: PyObject *	  : Function object or null		*/

PyObject*KBPYScriptIF::findFunction
	(	const QStringList	&scrList,
		const QString		&fName
	)
{
	for (QStringList::ConstIterator it = scrList.begin() ; it != scrList.end() ; ++it)
	{
		KBPYModule  *module ;
		QString	    mName   = *it ;
		int	    lsep    = mName.findRev ('/') ;

		if (lsep >= 0) mName = mName.mid (lsep + 1)  ;

		fprintf	(stderr, "Namemap search [%s]\n", (cchar *)(mName)) ;
		if ((module = nameMap.find (mName)) == 0)
		{
			errIdent = "" ;
			errLno	 = 0  ;
			errMsg	 = QString("Module %1 not found for function %2")
					  .arg(mName)
					  .arg(fName) ;
			return	 0   ;
		}

		/* Get the module dictionary (which is claimed never	*/
		/* to fail) and then look for the function therein.	*/
		/* The latter will fail if the script is messed up.	*/
		PyObject *dict	= PyModule_GetDict    (module->pyMod  ) ;
		PyObject *func	= PyDict_GetItemString(dict, _C(fName)) ;

		if (func != 0) return func ;
	}

	errIdent = "" ;
	errLno	 = 0  ;
	errMsg	 = QString ("Script function %1 not found").arg(fName) ;
	return	 0    ;
}

/*  KBPYScriptIF							*/
/*  execute	: Execute an EL script					*/
/*  scrList	: const QStringList &					*/
/*				  : List of script modules to search	*/
/*  fname	: const QString & : Function name			*/
/*  node	: KBNode *	  : Invoking node			*/
/*  argc	: uint		  : Argument count			*/
/*  argv	: const KBValue * : Argument vector			*/
/*  resval	: KBValue &	  : Result value			*/
/*  (returns)	: KBScript::ExeRC : Result				*/

KBScript::ExeRC	KBPYScriptIF::execute
	(	const QStringList	&scrList,
		const QString		&fname,
		KBNode			*node,
		uint			argc,
		const KBValue		*argv,
		KBValue			&resval
	)
{
	QString	_errMsg		;
	QString	_errIdent	;
	uint	_errLno		= 0 ;

	if (!scrList.isEmpty())
	{
		PyObject *func = findFunction (scrList, fname) ;

		fprintf
		(	stderr,
			"KBPYScriptIF::execute: [%s]->[%p]\n",
			(cchar *)fname,
			(void  *)func
		)	;

		if (func != 0)
		{	SHOWREF	  (func, "function") ;
			KBScript::ExeRC	rc = execFunc (func, node, argc, argv, resval) ;
			return	  rc	 ;
		}

		_errMsg	  = errMsg   ;
		_errIdent = errIdent ;
		_errLno	  = errLno   ;
	}

	PyObject *func = findFunction ("RekallMain", fname) ;

	if (func != 0)
	{	SHOWREF	  (func, "function") ;
		KBScript::ExeRC	rc = execFunc (func, node, argc, argv, resval) ;
		return	  rc	 ;
	}

	if (!scrList.isEmpty())
	{
		errMsg	 = _errMsg   ;
		errIdent = _errIdent ;
		errLno	 = _errLno   ;
	}

	return	  KBScript::ExeError ;
}

/*  KBPYScriptIF							*/
/*  deleteScript: Delete script file(s)					*/
/*  location	: KBLocation &	  : Document location			*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: bool		  : Success				*/

bool	KBPYScriptIF::unlink
	(	KBLocation	&location,
		KBError		&pError
	)
{
#ifndef _WIN32
	QString	path	= location.dbInfo->getDBPath() + "/" + location.docName ;
#else
	QString	path	= location.dbInfo->getDBPath() + "\\" + location.docName ;
#endif

	if (QFile::exists(path + ".pyc") && (::unlink (path + ".pyc") != 0))
	{
		pError	= KBError
			  (	KBError::Error,
				QString (TR("Failed to delete script code %1.pyc")).arg(path),
				strerror(errno),
				__ERRLOCN
			  )	;
		return	false	;
	}

	if (::unlink (path + ".py") != 0)
	{
		pError	= KBError
			  (	KBError::Error,
			 	QString (TR("Failed to delete script %1")).arg(path),
			 	strerror(errno),
			 	__ERRLOCN
			  )	;
		return	false	;
	}

	return	true	;
}

/*  KBPYScriptIF							*/
/*  rename	: Rename script file(s)					*/
/*  location	: KBLocation &	  : Document location			*/
/*  newName	: const QString & : New script name			*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: bool		  : Success				*/

bool	KBPYScriptIF::rename
	(	KBLocation	&location,
		const QString	&newName,
		KBError		&pError
	)
{
#ifndef _WIN32
	QString	oldPath	= location.dbInfo->getDBPath() + "/" + location.docName ;
	QString	newPath	= location.dbInfo->getDBPath() + "/" + newName ;
#else
	QString	oldPath	= location.dbInfo->getDBPath() + "\\" + location.docName ;
	QString	newPath	= location.dbInfo->getDBPath() + "\\" + newName ;
#endif

	if (QFile::exists(oldPath + ".pyc") && (::rename (oldPath + ".pyc", newPath + ".pyc") != 0))
	{
		pError	= KBError
			  (	KBError::Error,
				QString (TR("Failed to rename script code %1.pyc")).arg(oldPath),
				strerror(errno),
				__ERRLOCN
			  )	;
		return	false	;
	}

	if (::rename (oldPath + ".py", newPath + ".py") != 0)
	{
		pError	= KBError
			  (	KBError::Error,
			 	QString (TR("Failed to rename script %1")).arg(oldPath),
			 	strerror(errno),
			 	__ERRLOCN
			  )	;
		return	false	;
	}

	return	true	;
}

/*  KBPYScriptIF							*/
/*  exeError	: Retrieve execution error details			*/
/*  _errMsg	: QString &	: Return error message			*/
/*  _errLno	: uint &	: Return error line number		*/
/*  _errText	: QString &	: Error details text			*/
/*  (returns)	: KBLocation	: Return location			*/

KBLocation KBPYScriptIF::exeError
	   (	QString	&_errMsg,
		uint	&_errLno,
		QString	&_errText
	   )
{
	_errMsg	 = errMsg  ;
	_errLno	 = errLno  ;
	_errText = errText ;

	/* If the error identifier is empty then it must be a failure	*/
	/* to locate the error. In this case we have nothing to go on.	*/
	if (errIdent == "")
		return	KBLocation
			(	0,
				"script",
				KBLocation::m_pInline,
				"[unknown]",
				"py"
			) ;

	/* If the error was in inline code then return just enough to	*/
	/* correspond to this. The original caller should know where	*/
	/* it is.							*/
	if (inlineMap.find (errIdent) != 0)
		return	KBLocation
			(	0,
				"script",
				KBLocation::m_pInline,
				KBLocation::m_pInline,
				"py"
			) ;

	/* OK, got a real looking identifier. It should exist in the	*/
	/* module map, in which case return the location for the map	*/
	/* entry.							*/
	KBPYModule *module ;
	if ((module = loadMap.find (errIdent)) != 0)
	{	fprintf	(stderr, "KBPYScriptIF::exeError -> [%s]\n",
				 (cchar *)module->location.ident()) ;
		return	module->location ;
	}

	/* Should not get here!						*/
	fprintf	(stderr, "KBPYScriptIF::exeError failed for [%s]\n",
			 (cchar *)errIdent) ;
	return	KBLocation () ;
}

/*  KBPyScriptIF:							*/
/*  editorInit	: Set up editor for python 				*/
/*  editor	: TKTextEditor * : Editor				*/
/*  (returns)	: void		 :					*/

void	KBPYScriptIF::editorInit
	(	TKTextEditor	*editor
	)
{
	TKCPySetupEditor (editor) ;
}

/*  KBPyScriptIF:							*/
/*  showDebug	: Show debugger						*/
/*  action	: TKToggleAction *: Display toggle action		*/
/*  (returns)	: KBDebug *	  : Debugger				*/

KBDebug	*KBPYScriptIF::showDebug
	(	TKToggleAction	*action
	)
{
	bool	ok	;
	KBDebug	*debug	= 0;
	debug = new KBPYDebug (action, ok) ;

	if (!ok)
	{	
		if(debug != 0)
			delete	debug	;

		debug = 0;
		return	0	;
	}

	return	debug	;
}


/*  KBPyScriptIF:							*/
/*  debugScript	: Load script into debugger				*/
/*  location	: KBLocation &	: Script location			*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: bool		: Success				*/

bool	KBPYScriptIF::debugScript
	(	KBLocation	&location,
		KBError		&pError
	)
{
	if (!debuggerOK)
	{
		pError	= KBError
			  (	KBError::Error,
				TR("Cannot load: debugger was not loaded"),
				QString::null,
				__ERRLOCN
			  )	;
		return	false	;
	}

	TKCPyDebugWidget *debWidget ;
	if ((debWidget = TKCPyDebugWidget::widget()) == 0)
	{
		pError	= KBError
			  (	KBError::Error,
				TR("Cannot load: debugger not open"),
				QString::null,
				__ERRLOCN
			  )	;
		return	false	;
	}

	QString	eText	;
	QString	ePatt	;
	bool	pyErr	;
	bool	rc	= load	(location, eText, ePatt, pError, pyErr) ;

	if (rc || pyErr)
	{
		if (pyErr) pError.DISPLAY() ;

		TKCPyRekallCookie module (location) ;
		debWidget->editModule (&module, eText) ;

	}

	return	rc	;
}

QString	KBPYScriptIF::ident ()
{
	return	QString	("%1 (Python %2.%3)")
			.arg(__KB_BUILD_IDENT)
			.arg(PY_MAJOR_VERSION)
			.arg(PY_MINOR_VERSION) ;
}

extern	"C"
{
#ifndef _WIN32
	QObject	*makeScriptIF (cchar *pypath)
#else
	__declspec(dllexport) QObject* makeScriptIF(cchar* pypath)
#endif
	{
		loadMap.setAutoDelete    (true) ;
		Py_SetProgramName	 (_C(pypath)) ;
		return	new KBPYScriptIF ()	;
	}
}

