/***************************************************************************
    file	         : kb_qrylevel.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<ctype.h>

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_dblink.h"
#include	"kb_database.h"
#include	"kb_queryset.h"
#include	"kb_qrylevel.h"
#include	"kb_block.h"
#include	"kb_table.h"
#include	"kb_options.h"

#include	"tk_messagebox.h"


#define	EXPRCOL	(0xffff)	/* Dummy column for expression items	*/


/*  As items are added to the query structure, they are associated with	*/
/*  a table. There are two special cases (a) no table, ie., the item	*/
/*  expression is constant and (b) mixed tables, ie., the expression	*/
/*  refers to more than one table.					*/
/*									*/
static	KBTable	*noTable	= (KBTable *)(sizeof(KBTable) * 1) ;
static	KBTable	*mixTable	= (KBTable *)(sizeof(KBTable) * 2) ;


/*  fieldPart	: Get field part from an item expression		*/
/*  expr	: const QString & : Item expression			*/
/*  (returns)	: QString	  : Field part				*/

static	QString	fieldPart
	(	const QString	&expr
	)
{
	int	dot	= expr.find ('.') ;
	return	dot < 0 ? expr : expr.mid (dot + 1) ;
}

static	uint	fieldPermissions
	(	KBFieldSpec	*fSpec,
		QString		reason
	)
{
	if ((fSpec->m_flags & KBFieldSpec::InsAvail) != 0)
	{
		reason	+= TR("<li>Inserted key available</li>") ;
		return	QP_INSERT|QP_UPDATE|QP_DELETE ;
	}
	if ((fSpec->m_flags & KBFieldSpec::Unique  ) != 0)
	{
		reason	+= TR("<li>Column is unique</li>") ;
		return	QP_UPDATE|QP_DELETE ;
	}

	return	0 ;
}

static	QString	permissionToText
	(	uint	permission
	)
{
	QStringList	what	;
	if ((permission & QP_SELECT) != 0) what.append ("select") ;
	if ((permission & QP_INSERT) != 0) what.append ("insert") ;
	if ((permission & QP_UPDATE) != 0) what.append ("update") ;
	if ((permission & QP_DELETE) != 0) what.append ("delete") ;
	return	TR("Permissions are: ") + what.join(", ") ;
}


/*  getNextToken: Get next token from expression			*/
/*  expr	: const QString & : Expression				*/
/*  offset	: uint &	  : Expression offset			*/
/*  (returns)	: QString	  : Token or QString::null at end	*/

static	QString	getNextToken
	(	const QString	&expr,
		uint		&offset
	)
{
	QString	res	= QString::null ;
	bool	quoted	= false ;

	for ( ; offset < expr.length() ; offset += 1)
	{
		QChar ch = expr.at(offset) ;

		if (ch == '\'')
		{	quoted	= !quoted ;
			continue  ;
		}
		if (quoted)
			continue  ;

		if (ch == '.')
		{	offset	+= 1	;
			return	"."	;
		}

		if (ch.isLetterOrNumber() || (ch == '_'))
			break	  ;
	}

	if (offset >= expr.length())
		return QString::null ;

	for ( ; offset < expr.length() ; offset += 1)
	{
		QChar ch = expr.at(offset) ;

		if (!ch.isLetterOrNumber() && (ch != '_'))
			return res  ;

		res	+= ch ;
	}

	return	res	;
}

static	void	findTableColumnPairs
	(	const QString	&expr,
		QStringList	&tables,
		QStringList	&columns
	)
{
	QString		tok1	= QString::null ;
	QString		tok2	= QString::null ;
	QString		token	= QString::null ;
	uint		offset	= 0 ;

	while ((token = getNextToken (expr, offset)) != QString::null)
	{
		if (token == ".")
		{	tok1	= tok2		;
			tok2	= QString::null ;
			continue  ;
		}

		if (tok1 != QString::null)
		{
			tables .append (tok1 )	;
			columns.append (token)	;
			tok1	= QString::null ;
			continue  ;
		}

		if (tok2 != QString::null)
		{
			tables .append (tok1 )	;
			columns.append (tok2 )	;
			tok2	= token		;
			continue ;
		}

		tok2 = token ;
	}

	if (tok2 != QString::null)
	{
		tables .append (QString::null) ;
		columns.append (tok2) ;
	}

//	fprintf	(stderr, "findTableColumnPairs: [%s]\n", (cchar *)expr) ;
//	for (uint idx = 0 ; idx < tables.count() ; idx += 1)
//		fprintf
//		(	stderr,
//			"findTableColumnPairs: [%s][%s]\n",
//			(cchar *)tables [idx],
//			(cchar *)columns[idx]
//		)	;
//	fprintf	(stderr, "findTableColumnPairs: ----\n") ;
}

/*  ------------------------------------------------------------------  */

/*  KBQryLevelSet							*/
/*  KBQryLevelSet: Constructor for query level table set		*/
/*  dbLink	 : KBDBLink &	 : Link to server			*/
/*  table	 : KBTable *	 : Associated table			*/
/*  (returns)	 : KBQryLevelSet :					*/

KBQryLevelSet::KBQryLevelSet
	(	KBDBLink	&dbLink,
		KBTable		*table
	)
	:
	m_dbLink	(dbLink),
	m_table		(table)
{
	m_uniqueItem	= 0  ;
	m_uniqueCol	= -1 ;
	m_uniqueType	= KBTable::None ;
	m_qryInsert	= 0  ;
	m_qryNewkey	= 0  ;
	m_qryUpdate	= 0  ;
	m_permission	= QP_SELECT  ;
	m_values	= 0  ;

//	fprintf
//	(	stderr,
//		"KBQryLevelSet::KBQryLevelSet: %s (%s)\n",
//		(cchar *)m_table->getTable(),
//		(cchar *)m_table->getAlias()
//	)	;
}

/*  KBQryLevelSet							*/
/*  ~KBQryLevelSet: Destructor for query level table set		*/
/*  (returns)	  :		:					*/

KBQryLevelSet::~KBQryLevelSet ()
{
	DELOBJ	(m_qryInsert) ;
	DELOBJ	(m_qryNewkey) ;
	DELOBJ	(m_qryUpdate) ;

	if (m_values != 0) delete [] m_values ;
}

/*  KBQryLevelSet							*/
/*  addItem	: Add item to this level set				*/
/*  item	: KBItem *	  : Item in question			*/
/*  defval	: const QString & : Database level default		*/
/*  (returns)	: void		  :					*/

void	KBQryLevelSet::addItem
	(	KBItem		*item,
		const QString	&defval
	)
{
	/* Note that all items added are update items. We note the	*/
	/* default value if there is one.				*/
	m_items.append  (item  ) ;
	item->setDefVal (defval) ;
}


/*  KBQryLevelSet							*/
/*  uniqueDisplayed							*/
/*		: See if unique column is displayed			*/
/*  name	: const QString & : Column name				*/
/*  needed	: bool		  : True if column value will be needed	*/
/*  uexpr	: const QString & : Retrieval expression		*/
/*  (returns)	: bool		  : True if column is displayed		*/

bool	KBQryLevelSet::uniqueDisplayed
	(	const QString	&name,
		bool		needed,
		const QString	&uexpr
	)
{
	m_uniqueItem = 0 		;
	m_uniqueExpr = QString::null	;

	LITER
	(	KBItem,
		m_items,
		item,

		/* If this field shows the required column then set it	*/
		/* as the unique item provided that it is updatable (so	*/
		/* the client can have a hidden field that contains the	*/
		/* primary key, for example).				*/
		if (item->getExpr() == name)
		{
			if (item->isUpdateVal())
			{
				m_uniqueItem = item  ;
				m_uniqueExpr = uexpr ;
				if (needed) item->setNeeded (true) ;
				return	true ;
			}

			return	false	;
		}
	)

	return	false	;
}

/*  KBQryLevelSet								*/
/*  findPermissions							*/
/*		: Get permissions for this level set			*/
/*  reason	: QString &	: Reason text				*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: int		: Permissions or minus one on error	*/

int	KBQryLevelSet::findPermissions
	(	QString		&reason,
		KBError		&pError
	)
{
	KBTableSpec		tabSpec	(m_table->getTable  ()) ;
	QStringList		uNames	;
	QString			uExpr	;
	KBFieldSpec 		*fSpec  ;

	if (!m_dbLink.listFields (tabSpec))
	{	pError	= m_dbLink.lastError() ;
		return	-1	;
	}

	m_permission = 0 ;
	m_uniqueType = m_table->getUnique (uNames, uExpr) ;

	/* Sort out permissions and such based on the uniqueness type	*/
	/* stored with the table....					*/
	reason	+= QString(TR("Table <b>%1</b>")).arg(m_table->getTable()) ;

	/* Primary key specified. Check that the table has a primary	*/
	/* key column and that is is the named column.			*/
	if (m_uniqueType == KBTable::PrimaryKey)
	{
		fSpec = tabSpec.findPrimary() ;
		if ((fSpec == 0) || (fSpec->m_name != uNames[0]))
		{	
			pError	= KBError
				  (	KBError::Error,
					TR("Table does not have specified primary key"),
				  	QString	(TR("Table: %1, key %2"))
				  		.arg(m_table->getTable())
				  		.arg(uNames[0]),
				  	__ERRLOCN
				  )	;
			return	-1	;
		}

		reason += QString(TR(": using primary key <b>%1</b>")).arg(fSpec->m_name) ;
		reason += "<ul>" ;

		m_table->setUnique (uNames[0], fSpec->m_flags, QString::null) ;
		m_permission = fieldPermissions (fSpec, reason) | QP_SELECT ;

		if (uniqueDisplayed  (uNames[0], false, QString::null))
		{	reason       += TR("<li>Column is displayed</li>") ;
			m_permission |= QP_INSERT ;
		}

		reason += "<li>" + permissionToText (m_permission) + "</li>" ;
		reason += "</ul>" ;

		return	m_permission ;
	}

	/* Any unique column. Check that the table has the named	*/
	/* column and that it is unique.				*/
	if (m_uniqueType == KBTable::AnyUnique)
	{
		fSpec = tabSpec.findField (uNames[0]) ;
		if ((fSpec == 0) || ((fSpec->m_flags & KBFieldSpec::Unique) == 0))
		{
			pError	= KBError
				  (	KBError::Error,
				  	TR("Table does not have specified unique column"),
				  	QString	(TR("Table: %1, column %2"))
				  		.arg(m_table->getTable ())
				  		.arg(uNames[0]),
				  	__ERRLOCN
				  )	;
			return	-1	;
		}

		reason += QString(TR(": using unique column <b>%1</b>")).arg(fSpec->m_name) ;
		reason += "<ul>" ;

		m_table->setUnique (uNames[0], fSpec->m_flags, QString::null) ;
		m_permission = fieldPermissions (fSpec, reason) | QP_SELECT ;

		/* Assume a primary key is not updated by a trigger,	*/
		/* in which case insert is possible if the column is	*/
		/* displayed.						*/
		if (uniqueDisplayed  (uNames[0], true, QString::null))
		{	reason       += TR("<li>Column is displayed</li>") ;
			m_permission |= QP_INSERT ;
		}

		reason += "<li>" + permissionToText (m_permission) + "</li>" ;
		reason += "</ul>" ;

		return	m_permission ;
	}

	/* User specifies a single column. This is like any-unique but	*/
	/* we assume uniqueness. Caveat Emptor!				*/
	if (m_uniqueType == KBTable::AnySingle)
	{
		fSpec = tabSpec.findField (uNames[0]) ;
		if (fSpec == 0)
		{
			pError	= KBError
				  (	KBError::Error,
				  	TR("Table does not have specified column"),
				  	QString	(TR("Table: %1, column %2"))
				  		.arg(m_table->getTable ())
			  			.arg(uNames[0]),
				  	__ERRLOCN
				  )	;
			return	-1	;
		}

		reason += QString(TR(": using specified column <b>%1</b>")).arg(fSpec->m_name) ;
		reason += "<ul>" ;

		/* We can do everything, except that to insert, the	*/
		/* column must be displayed or the value must be	*/
		/* available after an insert.				*/
		m_table->setUnique (uNames[0], fSpec->m_flags, QString::null) ;
		m_permission  = QP_SELECT|QP_UPDATE|QP_DELETE  ;

		if (uniqueDisplayed (uNames[0], true, QString::null))
		{	reason       += TR("<li>Key column is displayed</li>") ;
			m_permission |= QP_INSERT ;
		}

		if ((fSpec->m_flags & KBFieldSpec::InsAvail) != 0)
		{	reason       += TR("<li>Inserted key available</li>") ;
			m_permission |= QP_INSERT ;
		}

		reason += "<li>" + permissionToText (m_permission) + "</li>" ;
		reason += "</ul>" ;

		return	m_permission ;
	}

	/* User specifies a single column, but with an expression to	*/
	/* generate (::PreExpression) or retrieve (::PostExpression)	*/
	/* the inserted value.						*/
	if ((m_uniqueType == KBTable::PreExpression) || (m_uniqueType == KBTable::PostExpression))
	{
		fSpec = tabSpec.findField (uNames[0]) ;
		if (fSpec == 0)
		{
			pError	= KBError
				  (	KBError::Error,
				  	TR("Table does not have specified column"),
				  	QString	(TR("Table: %1, column %2"))
				  		.arg(m_table->getTable ())
			  			.arg(uNames[0]),
				  	__ERRLOCN
				  )	;
			return	-1	;
		}

		reason += QString(TR(": using specified column <b>%1</b>")).arg(fSpec->m_name) ;
		reason += QString(TR("  with expression <b>%1</b>"       )).arg(uExpr	     ) ;
		reason += "<ul>" ;

		/* With this setting we can do everything, since we	*/
		/* assume that the exer has got it right!		*/
		m_table->setUnique (uNames[0], fSpec->m_flags, uExpr) ;
		m_permission  = QP_SELECT|QP_UPDATE|QP_DELETE|QP_INSERT ;

		if (uniqueDisplayed (uNames[0], true, uExpr))
			reason       += TR("<li>Key column is displayed</li>") ;

		reason += "<li>" + permissionToText (m_permission) + "</li>" ;
		reason += "</ul>" ;

		return	m_permission ;
	}

	/* Next is a setting which was made prior to this release. The	*/
	/* semantics are to use the specified column if we can, but to	*/
	/* fall back to Auto if not.					*/
	if (m_uniqueType == KBTable::Legacy100)
	{
		reason += QString(TR(": using pre-2.0.0 spec on column <i>%1</i>")).arg(uNames[0]) ;
		reason += "<ul>" ;

		m_permission = QP_SELECT ;

		if ((fSpec = tabSpec.findField (uNames[0])) != 0)
		{
			if ((fSpec->m_flags & KBFieldSpec::InsAvail) != 0)
			{
				reason	     += TR("<li>Inserted key available</li>") ;
				m_permission |= QP_INSERT|QP_UPDATE|QP_DELETE ;
			}
			if ((fSpec->m_flags & KBFieldSpec::Unique  ) != 0)
			{
				reason	     += TR("<li>Column is unique</li>") ;
				m_permission |= QP_UPDATE|QP_DELETE ;
			}
		}

		if (m_permission == (QP_SELECT|QP_INSERT|QP_UPDATE|QP_DELETE))
		{
			m_table->setUnique (uNames[0], fSpec->m_flags, QString::null) ;
			uniqueDisplayed  (uNames[0], false, QString::null) ;
	
			reason += "<li>" + permissionToText (m_permission) + "</li>" ;
			reason += "</ul>" ;

			m_uniqueType = KBTable::AnySingle ;
			return	m_permission ;
		}

		reason += TR("<li>Falling back on automatic selection ....</li>") ;
		reason += "</ul>" ;
		reason += QString(TR("table <b>%1</b>")).arg(m_table->getTable()) ;

		m_uniqueType = KBTable::Auto ;
	}

	/* Auto selection. In this try the various possibilities in	*/
	/* descending order of usefulness.				*/
	if (m_uniqueType == KBTable::Auto)
	{
		reason += QString(TR(": using automatic column selection")) ;
		reason += "<ul>" ;

		/* First case is table has a primary key column. This	*/
		/* is the preferred case, whatever anyone else may say	*/
		/* about it.						*/
		if ((fSpec = tabSpec.findPrimary ()) != 0)
		{
			reason	+= QString (TR("<li>Using primary key <i>%1</i></li>")).arg(fSpec->m_name) ;

			m_permission = fieldPermissions (fSpec, reason) | QP_SELECT ;
			m_table->setUnique (fSpec->m_name, fSpec->m_flags, QString::null) ;
			bool ud	= uniqueDisplayed   (fSpec->m_name, false, QString::null) ;

			/* Assume a primary key is not updated by a	*/
			/* trigger, in which case insert is possible if	*/
			/* the column is displayed in all cases.	*/
			if (ud)
			{	reason       += TR("<li>Column is displayed</li>") ;
				m_permission |= QP_INSERT ;
			}

			reason	+= "<li>" + permissionToText (m_permission) + "</li>" ;
			reason	+= "</ul>" ;

			m_uniqueType = KBTable::PrimaryKey ;
			return	m_permission ;
		}

		/* No primary key so see if there is a unique column.	*/
		/* If there are more that one then we will arbitrarily	*/
		/* use the first one discovered.			*/
		if ((fSpec = tabSpec.findUnique  ()) != 0)
		{
			reason	+= QString (TR("<li>Using unique column <i>%1</i></li>")).arg(fSpec->m_name) ;

			m_permission = fieldPermissions (fSpec, reason) | QP_SELECT ;
			m_table->setUnique (fSpec->m_name, fSpec->m_flags, QString::null) ;
			bool ud	= uniqueDisplayed    (fSpec->m_name, true, QString::null) ;

			/* Assume a unique column will not be changed	*/
			/* by a trigger. If so then we can do insert	*/
			/* if the column is displayed.			*/
			if (ud)
			{	reason       += TR("<li>Column is displayed</li>") ;
				m_permission |= QP_INSERT ;
			}

			reason	+= "<li>" + permissionToText (m_permission) + "</li>" ;
			reason	+= "</ul>" ;

			m_uniqueType = KBTable::AnyUnique ;
			return	m_permission ;
		}

		/* No primary, no unique, so the best that we can do is	*/
		/* select. Tough.					*/
		reason	+= QString(TR("<li>No primary key or unique column</li>")) ;
		reason	+= "<li>" + permissionToText (QP_SELECT) + "</li>" ;
		reason	+= "</ul>" ;

		m_uniqueType = KBTable::None	 ;
		return	m_permission = QP_SELECT ;
	}

	/* Shound not get here! 					*/
	pError	= KBError
		  (	KBError::Error,
			TR("Unrecognised table uniqueness setting"),
			QString(TR("Uniqueness: %1")).arg(m_uniqueType),
			__ERRLOCN
		  )	;
	return	-1 	;
}

/*  KBQryLevelSet								*/
/*  doUpdate	: Update values in specified row			*/
/*  querySet	: KBQuerySet *	: Query set				*/
/*  qrow	: uint		: Query row number			*/
/*  priKey	: KBValue &	: Return primary key value		*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: bool		: Success				*/

bool	KBQryLevelSet::doUpdate
	(	KBQuerySet	*querySet,
		uint		qrow,
		KBValue		&priKey,
		KBError		&pError
	)
{
	/* We will only do a real update if there actually are changed	*/
	/* values. The "m_updated" flag will be set if this is true.	*/
	m_updated = false	;

	/* We should never be called if there is no corresponding	*/
	/* primary key column in the query set.				*/
	if (m_uniqueCol < 0)
		KBError::EFatal
		(	TR("Update attempted without known primary column"),
			QString(TR("Table: %1")).arg(m_table->getTable()),
			__ERRLOCN
		)	;

	/* First time, build the update query. Run through all the	*/
	/* fields, adding "name = value" pairs to an update query for	*/
	/* each one which is applicable to the update.			*/
	if (m_qryUpdate == 0)
	{
		QString	update	;
		uint	qryIdx	= 0  ;
		cchar	*sep	= "" ;

		update	 = QString ("update %1 set ")
					.arg(m_dbLink.mapExpression(m_table->getTable())) ;

		LITER
		(	KBItem,
			m_items,
			item,

			if ((item->flags() & KBFieldSpec::ReadOnly) != 0)
				continue ;

			update	+= QString("%1%2 = %3")
					  .arg(sep)
					  .arg(m_dbLink.mapExpression(fieldPart(item->getExpr())))
					  .arg(m_dbLink.placeHolder  (qryIdx)) ;
			qryIdx	+= 1	;
			sep	 = ", "	;
		)

		/* Add a where clause to the update for the current	*/
		/* value of the primary key. Note that although we	*/
		/* should never allow a user to update a primary key,	*/
		/* we still go via the overrides since we may be	*/
		/* updating a row which we have just inserted.		*/
		update	+= QString(" where %1 = %2")
				  .arg(m_dbLink.mapExpression(m_table->getUnique()))
				  .arg(m_dbLink.placeHolder  (qryIdx)) ;

		m_qryUpdate = m_dbLink.qryUpdate
				(	true,
					update, 
					m_table->getName()
				)	;

		if (m_qryUpdate == 0)
		{
			pError	= m_dbLink.lastError() ;
			return	false	;
		}
	}


	/* Execute the update query with the appropriate set of values	*/
	/* and check for errors. Note that the first column in the	*/
	/* query set is always the primary key. As we go, see if any	*/
	/* of the values are dirty.					*/
	if (m_values == 0)
		m_values = new KBValue[m_items.count() + 2] ;

	uint	slot	= 0	;
	bool	dirty	= false	;

	LITER
	(	KBItem,
		m_items,
		item,

		KBValue	&vSlot	= m_values[slot] ;
		bool	d ;

		if ((item->flags() & KBFieldSpec::ReadOnly) != 0)
			continue ;

		vSlot	= querySet->getField (qrow, item->getQueryIdx(), d) ;
		if (d) dirty = true ;

		if (vSlot.isEmpty() && !item->getDefVal().isEmpty())
				vSlot = KBValue (item->getDefVal(), &_kbRaw) ;

		slot	+= 1 ;
	)

	/* If no values are dirty then we do not need to actually do an	*/
	/* update.							*/
	if (!dirty)
		return 	true	;

	m_updated	 = true	;
	m_values[slot++] = priKey = querySet->getField (qrow, m_uniqueCol, dirty) ;

	if (!m_qryUpdate->execute (slot, m_values))
	{
		pError	 = m_qryUpdate->lastError() ;
		return	 false ;
	}

	
	if (m_qryUpdate->getNumRows () != 1)
	{
		pError	= KBError
			  (	KBError::Error,
				QString (TR("Unexpectedly updated %1 rows")).arg(m_qryUpdate->getNumRows()),
			  	m_qryUpdate->getSubQuery(),
			  	__ERRLOCN
			  )	;
		return	 false	;
	}

	/* If the primary key is displayed in a field then copy the	*/
	/* value to our private primary key column, in case the user	*/
	/* updated it.							*/
	if (m_uniqueItem != 0)
		querySet->setField
		(	qrow,
			m_uniqueCol,
			querySet->getField (qrow, m_uniqueItem->getQueryIdx(), dirty),
			true
		)	;

	return	true	;
}

KBValue	KBQryLevelSet::keyFromExpr
	(	KBError		&pError,
		const QString	&uexpr
	)
{
	/* User has supplied an explicit expression. Execute the	*/
	/* expression (which should be a select query), and check that	*/
	/* we get back some data with a single column.			*/
	if (!m_qryNewkey->execute (0, 0))
	{
		pError	= m_qryNewkey->lastError() ;
		return	KBValue	() ;
	}
	if (!m_qryNewkey->rowExists (0))
	{
		pError	= KBError
			  (	KBError::Error,
				TR("New key query for insert returned no data"),
				uexpr,
				__ERRLOCN
			  )	;
		return	KBValue	() ;
	}
	if (m_qryNewkey->getNumFields() != 1)
	{
		pError	= KBError
			  (	KBError::Error,
				QString(TR("New key query for insert returned %1 columns")).arg(m_qryNewkey->getNumFields()),
				TR("Expected one column"),
				__ERRLOCN
			  )	;
		return	KBValue	() ;
	}

	/* Executed OK, get the key value from the results of the	*/
	/* query.							*/
	KBValue	newKey	= m_qryNewkey->getField (0, 0) ;
	if (newKey.isNull())
	{
		pError	= KBError
			  (	KBError::Error,
				TR("New key query for insert returned null"),
				TR("Expected single non-null value"),
				__ERRLOCN
			  )	;
		return	KBValue	() ;
	}

	fprintf
	(	stderr,
		"KBQryLevelSet::keyFromExpr: got expression key [%s]\n",
		(cchar *)newKey.getRawText()
	)	;

	return	newKey	;
}

/*  KBQryLevelSet							*/
/*  doInsert	: Insert values into new row				*/
/*  querySet	: KBQuerySet *	  : Query set				*/
/*  qrow	: uint		  : Current row in query		*/
/*  pValue	: KBValue *	  : Parent value if any			*/
/*  cExpr	: const QString & : Child expression			*/
/*  newKey	: KBValue &	  : Return primary key value of new row	*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: bool		  : Success				*/

bool	KBQryLevelSet::doInsert
	(	KBQuerySet	*querySet,
		uint		qrow,
		KBValue		*pValue,
		const QString	&cExpr,
		KBValue		&newKey,
		KBError		&pError
	)
{
	/* We should never be called if there is no corresponding	*/
	/* primary key column in the query set.				*/
	if (m_uniqueCol < 0)
		KBError::EFatal
		(	TR("Insert attempted without known primary column"),
			QString(TR("Table: %1")).arg(m_table->getTable()),
			__ERRLOCN
		)	;

	QString	name	;
	KBValue	value	;
	QString	text	;

	/* First time, build the insert query, much as for the update	*/
	/* query above. Here we also need to include the link value to	*/
	/* the parent block, if there is one, plus possible a value for	*/
	/* the primary key.						*/
	if (m_qryInsert == 0)
	{
		QString	insert	;
		QString	place	;
		uint	qryIdx	= 0  ;
		cchar	*sep	= "" ;

		insert	 = QString("insert into %1 (")
				  .arg(m_dbLink.mapExpression (m_table->getTable())) ;

		LITER
		(	KBItem,
			m_items,
			item,

			/* If the item is read-only then skip it. In	*/
			/* this case the RDBMS *must* be generating a	*/
			/* value.					*/
			if ((item->flags() & KBFieldSpec::ReadOnly) != 0)
				continue ;

			/* Also skip the unique item. This, if present,	*/
			/* is handled specially below.			*/
			if (item == m_uniqueItem)
				continue ;

			insert	+= QString("%1%2")
					.arg(sep)
					.arg(m_dbLink.mapExpression(fieldPart(item->getExpr()))) ;
			place	+= QString("%1%2")
					.arg(sep)
					.arg(m_dbLink.placeHolder  (qryIdx)) ;
			qryIdx	+= 1	;
			sep	 = ", "	;
		)

		if (pValue != 0)
		{	insert	+= QString("%1%2")
					.arg(sep)
					.arg(m_dbLink.mapExpression(cExpr )) ;
			place	+= QString("%1%2")
					.arg(sep)
					.arg(m_dbLink.placeHolder  (qryIdx)) ;
			qryIdx	+= 1	;
			sep      = ", "	;
		}


		/* Now see about the table unique column. We include	*/
		/* this in the insert, unless (a) is is marked as	*/
		/* read-only or (b) there is a unique expression.	*/
		uint	uflags	= 0 ;
		QString	uname	;
		QString	uexpr	;

		if (m_uniqueItem != 0)
		{	uname	= m_uniqueItem->getExpr () ;
			uflags	= m_uniqueItem->flags   () ;
			uexpr	= m_uniqueExpr		   ;
		}
		else	uname	= m_table->getUnique    (uflags, uexpr) ;

		if (((uflags & (KBFieldSpec::ReadOnly)) == 0) && (m_uniqueType != KBTable::PostExpression))
		{
			insert	+= QString("%1%2")
					.arg(sep)
					.arg(m_dbLink.mapExpression(uname )) ;
			place	+= QString("%1%2")
					.arg(sep)
					.arg(m_dbLink.placeHolder  (qryIdx)) ;
			qryIdx	+= 1	;
			sep      = ", "	;
		}

		insert	+= QString(") values (%1)").arg(place) ;

		m_qryInsert = m_dbLink.qryInsert
				(	true,
					insert,
					m_table->getName()
				)	;

		if (m_qryInsert == 0)
		{
			pError	= m_dbLink.lastError() ;
			return	false	;
		}

		if (!uexpr.isEmpty())
		{
			fprintf
			(	stderr,
				"KBQryLevelSet::doInsert: newkey query [%s]\n",
				(cchar *)uexpr
			)	;

			if ((m_qryNewkey = m_dbLink.qrySelect (true, uexpr)) == 0)
			{
				pError	= m_dbLink.lastError() ;
				return	false	;
			}
		}
		else	m_qryNewkey = 0 ;
	}


	/* Now assemble the actual values to the query. Similarly, add	*/
	/* the parent link value and new primary key if we have it.	*/
	/* Then off and execute it.					*/
	if (m_values == 0)
		m_values = new KBValue[m_items.count() + 2] ;


	uint	slot	= 0 ;
	bool	dirty	;

	LITER
	(	KBItem,
		m_items,
		item,

		/* Matching unique item and read-only check used when	*/
		/* building the	query above; similarly skip any unique	*/
		/* item here.						*/
		if ((item->flags() & KBFieldSpec::ReadOnly) != 0)
			continue ;

		if (item == m_uniqueItem)
			continue ;

		KBValue	&vSlot	= m_values[slot] ;
		vSlot	= querySet->getField (qrow, item->getQueryIdx(), dirty) ;

		/* If the value is null and the item has a non-null	*/
		/* (RDBMS) default then we use that default instead.	*/
		if (vSlot.isNull() && !item->getDefVal().isNull())
			vSlot = KBValue (item->getDefVal(), &_kbRaw) ;

		slot	+= 1 ;
	)

	if (pValue != 0)
		m_values[slot++] = *pValue ;


	/* Now the table default column. As above, this is skipped in	*/
	/* its entirity (as far was the insert query itself is		*/
	/* concerned) if the column is read-only or if there is a	*/
	/* unique value retrieval post-expression.				*/
	uint		uflags	 = 0	 ;
	bool		keyFetch = false ;
	QString		uname	 ;
	QString		uexpr	 ;

	if (m_uniqueItem != 0)
	{	uname	= m_uniqueItem->getExpr () ;
		uflags	= m_uniqueItem->flags   () ;
		uexpr	= m_uniqueExpr		   ;
	}
	else	uname	= m_table->getUnique    (uflags, uexpr) ;

	fprintf
	(	stderr,
		"KBQryLevelSet::doInsert: m_uniqueItem=[%p] uname=[%s] uflags=%08x ro=%d\n",
		(void  *)m_uniqueItem,
		(cchar *)uname,
		uflags,
		(uflags & (KBFieldSpec::ReadOnly)) != 0
	)	;

	if (((uflags & (KBFieldSpec::ReadOnly)) == 0) && (m_uniqueType != KBTable::PostExpression))
	{
		KBValue	&vSlot	= m_values[slot] ;

		if (m_uniqueType != KBTable::PreExpression)
		{
			/* If the unique column is on view then pick	*/
			/* up the value from it, if not then set to	*/
			/* null (note, this assign is needed since the	*/
			/* array will be used by successive inserts.	*/
			if (m_uniqueItem != 0)
				vSlot	= querySet->getField (qrow, m_uniqueItem->getQueryIdx(), dirty) ;
			else	vSlot	= KBValue () ;

			newKey	= vSlot	;

			/* If the value is still null and there is a	*/
			/* non-null default value for the item, then	*/
			/* use that.					*/
			if (vSlot.isNull() && (m_uniqueItem != 0) && !m_uniqueItem->getDefVal().isNull())
			{
				vSlot	 = KBValue (m_uniqueItem->getDefVal(), &_kbRaw) ;

				/* If the column has the insert-avail	*/
				/* flag set then we can retrieve the	*/
				/* key after the insert.		*/
				keyFetch = (uflags & KBFieldSpec::InsAvail) != 0 ;
			}

			/* If the value is still null and the item is	*/
			/* a primary key, then see if we can get a new	*/
			/* key from the query prior to the insert.	*/
			if (vSlot.isNull() && ((uflags & KBFieldSpec::Primary) != 0))
			{
				if (!m_qryInsert->getNewKey (uname, newKey, true))
				{
					pError	= m_qryInsert->lastError() ;
					return	false ;
				}

				vSlot	 = newKey ;

				/* If the returned new key is null but	*/
				/* the column is marked as		*/
				/* insert-available then note that we	*/
				/* should get the key post-insert.	*/
				keyFetch = newKey.isNull() && ((uflags & KBFieldSpec::InsAvail) != 0) ;
			}

			slot	+= 1 ;
		}
		else
		{
			/* Using a pre-expression to get the new key,	*/
			/* so evaluate the expression now and note the	*/
			/* new key value.				*/
			newKey	= keyFromExpr (pError, uexpr) ;
			if (newKey.isNull()) return false ;

			vSlot	= newKey ;
			slot   += 1	 ;
		}
	}
	else
		/* If the column is readonly but the insert-available	*/
		/* flag is set, then note that the key should be	*/
		/* retrieved after the insert.				*/
		if ((uflags & KBFieldSpec::InsAvail) != 0) keyFetch = true ;


	/* OK then, time to do the business. Execute the insert and	*/
	/* verify that exactly ine row was processed.			*/
	if (!m_qryInsert->execute (slot, m_values))
	{
		pError	= m_qryInsert->lastError() ;
		return	false ;
	}

	if (m_qryInsert->getNumRows () != 1)
	{
		pError	= KBError
			  (	KBError::Error,
				QString (TR("Unexpectedly inserted %1 rows")).arg(m_qryInsert->getNumRows()),
				m_qryInsert->getSubQuery (),
				__ERRLOCN
			  )	;
		return	 false	;
	}

	/* In order to be able to keep the display synchronised, we	*/
	/* fetch the updated row back from the database. This may also	*/
	/* require retrieval of the primary key. There are two cases	*/
	/* for this (a) an explicit new key retrieval or (b) a key	*/
	/* automatically generated in the insert.			*/
	if (m_uniqueType == KBTable::PostExpression)
	{
		newKey = keyFromExpr (pError, uexpr) ;
		if (newKey.isNull()) return false ;
	}
	else if (keyFetch)
	{
		if (!m_qryInsert->getNewKey (uname, newKey, false))
		{	pError	= m_qryInsert->lastError() ;
			return	false ;
		}
	}

	querySet->setField (qrow, m_uniqueCol, newKey) ;
	return	true	;
}



/*  ------------------------------------------------------------------  */

/*  KBQryLevel								*/
/*  KBQryLevel	: Constructor for query level object			*/
/*  parent	: KBNode *	: Parent node				*/
/*  next	: KBQryLevel *	: Next level down or null		*/
/*  dbLink	: KBDBLink &	: Link to server			*/
/*  qryLvl	: uint		: Query level				*/
/*  table	: KBTable *	: Table object				*/
/*  topTable	: KBTable *	: Top table for top level set		*/
/*  (returns)	: KBQryLevel	:					*/

KBQryLevel::KBQryLevel
	(	KBNode		*parent,
		KBQryLevel	*next,
		KBDBLink	&dbLink,
		uint		qryLvl,
		KBTable		*table,
		KBTable		*topTable
	)
	:
	m_parent	(parent),
	m_next		(next),
	m_dbLink	(dbLink),
	m_qryLvl	(qryLvl),
	m_table		(table),
	m_topTable	(topTable == 0 ? table : topTable)
{
	/* NOTE: The top table defaults to the main table object. It	*/
	/* is specifically set when the block uses an SQL query.	*/
	m_distinct 	= false ;

	m_qryDelete	= 0 ;
	m_qryFetch 	= 0 ;
	m_qryForUpdate	= 0 ;
	m_querySet	= 0 ;
	m_locking	= KBQryBase::NoLocking ;
	m_limit		= 0 ;

	m_levelSets.setAutoDelete (true) ;
	m_topLevelSet	= 0 ;
}

/*  KBQryLevel								*/
/*  ~KBQryLevel	: Destructor for query level object			*/
/*  (returns)	:		:					*/

KBQryLevel::~KBQryLevel ()
{
	DELOBJ	(m_qryDelete   ) ;
	DELOBJ	(m_qryFetch    ) ;
	DELOBJ	(m_qryForUpdate) ;

	/* Only delete the query set at the top-level. All other query	*/
	/* sets are chained off it as subsets and will be deleted	*/
	/* within this deletion.					*/
	if (m_qryLvl == 0) DELOBJ (m_querySet) ;
}

/*  KBQryLevel								*/
/*  clear	: Clear contents					*/
/*  (returns)	: void		:					*/

void	KBQryLevel::clear ()
{
	m_locking	= KBQryBase::NoLocking ;

	m_querySet  = 0	      ;

	m_fldList  .clear ()  ;

	m_allItems .clear ()  ;
	m_getItems .clear ()  ;
	m_updItems .clear ()  ;

	if (m_topLevelSet != 0)
		m_topLevelSet->getTable()->setGrouped (false) ;

	m_levelSets.clear ()  ;
	m_topLevelSet = 0     ;

	DELOBJ	(m_qryDelete   ) ;
	DELOBJ	(m_qryFetch    ) ;
	DELOBJ	(m_qryForUpdate) ;

	if (m_next != 0) m_next->clear () ;
}

/*  KBQryLevel								*/
/*  findLevel	: Find level at which SQL component is row constant	*/
/*  item	: KBItem *	  : Associated item			*/
/*  ident	: const QString & : SQL component			*/
/*  iTable	: KBTable *&	  : Set to table for the item		*/
/*  (returns)	: KBQryLevel *	  : Level or null if not matched	*/

KBQryLevel
	*KBQryLevel::findLevel
	(	KBItem		*item,
		const QString	&ident,
		KBTable		*&iTable
	)
{	
	if (m_fldList.count() == 0)
	{
		if (!m_table->getFieldList (m_fldList, m_dbLink, true))
		{
//			fprintf
//			(	stderr,
//				"KBQryLevel::findLevel: [%s] NO FIELD LIST\n",
//				(cchar *)ident
//			)	;
			m_table->lastError().DISPLAY() ;
			return	0 ;
		}

//		fprintf
//		(	stderr,
//			"KBQryLevel::findLevel: [%s] has %d fields\n",
//			(cchar *)ident,
//			m_fldList.count()
//		)	;
	}

	LITER
	(	KBFieldSpec,
		m_fldList,
		fSpec,

//		fprintf
//		(	stderr,
//			"findLevel: %d: [%s/%s] [%s/%s]\n",
//			m_dbLink.keepsCase(),
//			(cchar *)ident,
//			(cchar *)dbLink.fixCase(ident),
//			(cchar *)fSpec->name,
//			(cchar *)dbLink.fixCase(fSpec->name)
//		)	;
//
		if (m_dbLink.fixCase (ident) == m_dbLink.fixCase(fSpec->m_name))
		{
//			fprintf
//			(	stderr,
//				"[%s] matched at %d [%04x] with %s\n",
//				(cchar *)ident,
//				m_qryLvl,
//				fSpec->m_flags,
//				iTable == noTable 	 ? "noTable" :
//				iTable != fSpec->m_table ? "mixed"   : "setting"
//			)	;

			if (iTable == noTable)
			{	iTable = fSpec->m_table ;
				item->setFlags (fSpec->m_flags) ;
				return	this   ;
			}

			if (iTable != fSpec->m_table)
			{	iTable = mixTable    ;
				item->setFlags (KBFieldSpec::ReadOnly) ;
				return	this	;
			}

			item->setFlags (fSpec->m_flags);
			return	this ;
		}
	)

	return	m_next == 0 ? 0 : m_next->findLevel (item, ident, iTable) ;
}

/*  KBQryLevel								*/
/*  rowConstant	: Find level at which SQL component is row constant	*/
/*  item	: KBItem *	  : Associated item			*/
/*  tName	: const QString & : Table or alias name			*/
/*  fName	: const QString & : Field name				*/
/*  iTable	: KBTable *&	  : Set to table for the item		*/
/*  (returns)	: KBQryLevel *	  : Appropriate level			*/

KBQryLevel
	*KBQryLevel::rowConstant
	(	KBItem		*item,
		const QString	&tName,
		const QString	&fName,
		KBTable		*&iTable
	)
{
	QString			ident	;
	KBQryLevel		*level	;

	if (tName == QString::null)
		ident	= m_table->getName() + "." + fName ;
	else	ident	= tName		     	  + "." + fName ;

//	fprintf	(stderr, "Row constant [%s.%s][%s] in [%s]\n",
//			 (cchar *)tName,
//			 (cchar *)fName,
//			 (cchar *)ident,
//			 (cchar *)table->getName()) ;

	level = findLevel (item, ident, iTable) ;
	return	level == 0 ? this : level ;
}

/*  KBQryLevel								*/
/*  rowConstant	: Determin level at which expression is row constant	*/
/*  item	: KBItem *	: Associated item			*/
/*  tables	: QStringList &	: Tables .....				*/
/*  columns	: QStringList &	: .... and columns lists		*/
/*  iTable	: KBTable *&	: Set to table for the item		*/
/*  (returns)	: KBQryLevel *	: Appropriate level			*/

KBQryLevel
	*KBQryLevel::rowConstant
	(	KBItem			*item,
		const QStringList	&tables,
		const QStringList	&columns,
		KBTable		*	&iTable
	)
{
	KBQryLevel	*level	= this		;
	for (uint idx = 0 ; idx < tables.count() ; idx += 1)
		level	= level->rowConstant
			  (	item,
				tables [idx],
				columns[idx],
				iTable
			  )	;

	return	level	;
}

/*  KBQryLevel								*/
/*  placeItem	: Place an item at this query level			*/
/*  item	: KBItem *	: Item in question			*/
/*  (returns)	: void		:					*/

void	KBQryLevel::placeItem
	(	KBItem	*item
	)
{
	QString	expr	= item->getExpr() ;
	QString	field	= fieldPart(expr) ;

	/* All items are noted in the "allItems" list for setting, and	*/
	/* in the "m_getItems" list for query retrieval. After this we	*/
	/* do nothing more for non-updatable items.			*/
	m_allItems.append (item) ;
	m_getItems.append (item) ;

	if (!item->isUpdateVal ())
	{
		item->setFlags (KBFieldSpec::ReadOnly) ;
		return ;
	}

	/* Get the table which is associated with the item. Note that	*/
	/* if it can be updated it must be a simple table column, and	*/
	/* that table should be at this level.				*/
	KBTable	*iTable = item->getTable () ;

	if (iTable == noTable)
	//	KBError::EFatal
	//	(	"No table found for updatable item",
	//		QString("Item expression: %1").arg(expr),
	//		__ERRLOCN
	//	)	;
		return	;

	if (iTable == 0)
		KBError::EFatal
		(	TR("No table set for updatable item"),
			QString(TR("Item expression: %1")).arg(expr),
			__ERRLOCN
		)	;
	if (iTable == mixTable)
		KBError::EFatal
		(	TR("Updatable item with multiple tables"),
			QString(TR("Item expression: %1")).arg(expr),
			__ERRLOCN
		)	;

	/* Add the item to the list of all updatable items. Whether or	*/
	/* not they are fixed will be decided when the permissions are	*/
	/* known.							*/
	m_updItems.append (item) ;


	/* Add the item to the appropriate table set for this level,	*/
	/* which is created if necessary. We note the top table level	*/
	/* set for this level once we have one.				*/
	KBQryLevelSet	*ls	= m_levelSets.find (iTable) ;
	QString		defval	= QString::null	    ;
	QString		qname	= QString("%1.%2")
					.arg(iTable->getName())
					.arg(field) ;
	if (ls == 0)
		m_levelSets.insert
		(	iTable,
			ls = new KBQryLevelSet (m_dbLink, iTable)
		)	;

	LITER
	(	KBFieldSpec,
		m_fldList,
		fspec,

		if (fspec->m_name == qname)
		{
//			fprintf	(stderr, "Default [%s]->[%s]\n",
//					 (cchar *)expr,
//					 (cchar *)fspec->m_defval) ;
			defval	= fspec->m_defval ;
			break	;
		}
	)

	ls->addItem (item, defval) ;

//	fprintf
//	(	stderr,
//		"KBQryLevel::placeItem [%s:%s] [%p:%s][%p:%s]\n",
//		(cchar *)item->getName(),
//		(cchar *)item->getExpr(),
//		(void  *)iTable,
//		iTable  == 0 ? "" : (cchar *)iTable ->getTable(),
//		(void  *)m_topTable,
//		m_topTable == 0 ? "" : (cchar *)m_topTable->getTable()
//	)	;

	if (iTable == m_topTable) m_topLevelSet = ls ;

//	fprintf	(stderr, "placeItem [%s]->[%s]\n",
//			 (cchar *)expr,
//			 (cchar *)iTable->getTable()) ;
}

/*  KBQryLevel								*/
/*  addItem	: Add a form item					*/
/*  item	: KBItem *	: Item in question			*/
/*  (returns)	: bool		: Item will fetch from database		*/

bool	KBQryLevel::addItem
	(	KBItem	*item
	)
{
	/* An item pointer of zero indicates that the owning block is	*/
	/* going to reload all item pointers, in which case we need to	*/
	/* clear up - but only allow this call via the top level.	*/
	if (item == 0)
	{	if (m_qryLvl == 0) clear () ;
		return 	 false ;
	}

	/* Assume that the item cannot be updated in the query set, and	*/
	/* that is is not a primary key. This will be udated later if	*/
	/* it proves otherwise.						*/
	item->setFlags  (0) ;
	item->setDefVal (QString::null) ;

	if (item->isRowMark() != 0) return false ;

	/* If the expression is empty then this is an unbound control	*/
	/* and we ignore it (presumably a script will set it). If the	*/
	/* expression starts with the "=" character then it is an	*/
	/* on-the-fly expression, in which case we do not need to get	*/
	/* it from the database.					*/
	QString	expr = item->getExpr() ;
	if (expr.isEmpty()) return false ;


	if (expr.at(0) == '=')
	{
		item->setQueryIdx (KBQryIdx (m_qryLvl, EXPRCOL)) ;
		item->setTable    (0) ;
		item->setFlags	  (KBFieldSpec::ReadOnly) ;
		m_allItems.append (item) ;
		return	false ;
	}

	/* Item contains a database query expression, so place it into	*/
	/* the query level hierarchy as appropriate. In the process we	*/
	/* identify which single table (if any) it is associated with.	*/
	/* We also mark tables which appear in group expressions.	*/
	QStringList	tables	;
	QStringList	columns	;

	findTableColumnPairs	(expr, tables, columns) ;

	KBTable		*iTable = noTable ;
	KBQryLevel	*qLevel = rowConstant(item, tables, columns, iTable) ;

	item  ->setTable (iTable) ;
	qLevel->placeItem(item  ) ;

//	fprintf	(stderr, "qlvl %3d: %-24s: %s\n",
//			 m_qryLvl,
//			 (cchar *)expr,
//			 iTable == 0        ? "Null"  :
//			 iTable == noTable  ? "None"  :
//			 iTable == mixTable ? "Mixed" : (cchar *)iTable->getTable()) ;

	return	true ;
}

/*  KBQryLevel								*/
/*  remItem	: Remove a form item					*/
/*  item	: KBItem *	: Item in question			*/
/*  (returns)	: void		:					*/

void	KBQryLevel::remItem
	(	KBItem	*item
	)
{
#if	0
	if (m_allItems.find (item) != 0)
	{
		m_allItems.remove (item) ;
		m_getItems.remove (item) ;
		m_updItems.remove (item) ;
		if (item == prItem) prItem = 0 ;
		return	;
	}
#endif
	if (m_next != 0) m_next->remItem (item) ;
}

/*  KBQryLevel								*/
/*  resetData	: Reset data to original state				*/
/*  qrow	: uint		: Query row _at this level_		*/
/*  (returns)	: void		:					*/

void	KBQryLevel::resetData
	(	uint	qrow
	)
{
	m_querySet->resetData (qrow) ;
}

/*  KBQryLevel								*/
/*  loadItems	: Load all items in this level				*/
/*  qrow	: uint		: Query row _at this level_		*/
/*  (returns)	: bool		: Success				*/

bool	KBQryLevel::loadItems
	(	uint	qrow
	)
{
	fprintf
	(	stderr,
		"KBQryLevel::loadItems: row=%d\n",
		qrow
	)	;

	/* NOTE: The argument is the query row _at this level_, however	*/
	/* we load items based on the query row _in their block_, since	*/
	/* they may not actually be at this level.			*/
	LITER
	(	KBItem,
		m_allItems,
		item,

		uint	qryIdx	= item->getQueryIdx() ;
		uint	irow	= item->getBlock   ()->getCurQRow() ;

		if (!item->setValue (irow, getField (qrow, qryIdx)))
			return	false	;
	)

	return	true	;
}

/*  KBQryLevel								*/
/*  clearItems	: Clear all items in this level				*/
/*  qrow	: uint		: Query row _at this level_		*/
/*  (returns)	: bool		: Success				*/

bool	KBQryLevel::clearItems
	(	uint	
	)
{
	/* NOTE: The argument is the query row _at this level_, however	*/
	/* we clear items based on the query row _in their block_ since	*/
	/* they may not actually be at this level.			*/
	LITER
	(	KBItem,
		m_allItems,
		item,

		item->clearValue (item->getBlock()->getCurQRow(), true) ;
	)

	return	true	;
}

/*  KBQryLevel								*/
/*  getNumRows	: Get number of rows in query set			*/
/*  (returns)	: uint		: Number of rows			*/

uint	KBQryLevel::getNumRows ()
{
//	if (m_querySet != 0)
//		fprintf	(stderr, "KBQryLevel::getNumRows %d -> %d\n",
//				 m_qryLvl,
//				 m_querySet->getNumRows ()) ;
//	else	fprintf	(stderr, "KBQryLevel::getNumRows %d -> none\n",
//				 m_qryLvl) ;

	return	m_querySet == 0 ? 0 : m_querySet->getNumRows () ;
}

/*  KBQryLevel								*/
/*  setRowState	: Set state of a row					*/
/*  qrow	: uint		: Query row				*/
/*  state	: KB::RState	: State to set				*/
/*  (returns)	: void		:					*/

void	KBQryLevel::setRowState
	(	uint		qrow,
		KB::RState	state
	)
{
	if (m_querySet != 0) m_querySet->setRowState (qrow, state) ;
}

/*  KBQryLevel								*/
/*  getRowState	: Get state of specified row				*/
/*  qrow	: uint		: Row number				*/
/*  (returns)	: KB::RState	: Row state				*/

KB::RState KBQryLevel::getRowState
	(	uint	qrow
	)
{
	return	m_querySet == 0 ? KB::RSInSync : m_querySet->getRowState (qrow) ;
}

/*  KBQryLevel								*/
/*  getField	: Set field value					*/
/*  qrow	: uint		  : Query row				*/
/*  qcol	: uint		  : Query column			*/
/*  initial	: bool		  : Get initial value			*/
/*  (returns)	: const KBValue	& : Field value				*/

const KBValue
	&KBQryLevel::getField
	(	uint	qrow,
		uint	qcol,
		bool	initial
	)
{
	static	KBValue	nullVal	;
	bool	dirty	 	;

	return	m_querySet == 0	    ? nullVal :
		qcol	 == EXPRCOL ? nullVal :
				      m_querySet->getField (qrow, qcol, dirty, initial) ;
}

/*  KBQryLevel								*/
/*  getWidth	: Get nominal column width				*/
/*  qcol	: uint		: Query column				*/
/*  (returns)	: uint		: Nominal width				*/

uint	KBQryLevel::getWidth
	(	uint	qcol
	)
{
	return	m_querySet == 0	    ? 0 :
		qcol	 == EXPRCOL ? 0 :
				      m_querySet->getWidth (qcol) ;
}

/*  KBQryLevel								*/
/*  setField	: Set field value					*/
/*  qrow	: uint		  : Query row				*/
/*  qcol	: uint		  : Query column			*/
/*  value	: const KBValue & : Value				*/
/*  (returns)	: void		  :					*/

void	KBQryLevel::setField
	(	uint		qrow,
		uint		qcol,
		const KBValue	&value
	)
{
	if ((m_querySet == 0) || (qcol != EXPRCOL)) return ;

	m_querySet->setField (qrow, qcol, value) ;
	if (m_querySet->getRowState (qrow) != KB::RSInserted)
		m_querySet->setRowState (qrow, KB::RSChanged) ;

	LITER
	(	KBItem,
		m_allItems,
		item,

		if (item->getQueryIdx() == qcol)
		{	item->setValue (qrow, value) ;
			break	;
		}
	)
}

/*  KBQryLevel								*/
/*  insertRow	: Insert empty row into query set			*/
/*  qrow	: uint		: Query row				*/
/*  (returns)	: void;		:					*/

void	KBQryLevel::insertRow
	(	uint	qrow
	)
{
	if (m_querySet != 0) m_querySet->insertRow (qrow) ;
}

/*  KBQryLevel								*/
/*  newRowEmpty	: Test if new row at end of query set is empty		*/
/*  qrow	: uint		: Query row number			*/
/*  (returns)	: bool		: True if empty				*/

bool	KBQryLevel::newRowEmpty
	(	uint	qrow
	)
{
	if (m_querySet == 0)
		return	true	;

	if ( (qrow < m_querySet->getNumRows ()) &&
	     (m_querySet->getRowState (qrow) != KB::RSInserted))
		return	false	;

	LITER
	(	KBItem,
		m_updItems,
		item,
		if (!item->isEmpty(qrow)) return false ;
	)

	return	true	;
}

/*  KBQryLevel								*/
/*  rowIsDirty	: Check if row is dirty					*/
/*  qrow	: uint		: Query row number			*/
/*  reset	: bool		: True to reset dirty flag		*/
/*  (returns)	: bool		: True if dirty				*/

bool	KBQryLevel::rowIsDirty
	(	uint	qrow,
		bool	reset
	)
{
	return	m_querySet == 0 ? false : m_querySet->rowIsDirty (qrow, reset) ;
}

/*  KBQryLevel								*/
/*  setCurrentRow: Set current query row				*/
/*  qrow	 : uint		: Query row				*/
/*  (returns)	 : uint		: Total rows here and below		*/

uint	KBQryLevel::setCurrentRow
	(	uint	qrow
	)
{
	if ((m_next != 0) && (m_querySet != 0))
	{
		if (qrow <  m_querySet->getNumRows ())
		{	KBQuerySet *subset = m_querySet->getSubset (qrow, 0) ;
			m_next  ->setQuerySet  (subset) ;
			return	subset->getTotalRows () ;
		}

		if (qrow == m_querySet->getNumRows ())
		{	m_next->setQuerySet (0) ;
			return	0 ;
		}

		KBError::EFatal
		(	TR("KBQryLevel::setCurrentRow: query set overrun"),
			QString::null,
			__ERRLOCN
		)	;
	}

	return	1 ;
}

/*  KBQryLevel								*/
/*  setQuerySet	: Set current query set for this level			*/
/*  qs		: KBQuerySet *	: Query set				*/
/*  (returns)	: void		:					*/

void	KBQryLevel::setQuerySet
	(	KBQuerySet	*qs
	)
{
	/* Note that the query set will be null if the parent is at the	*/
	/* notional new row at the end of the set, or if a new empty	*/
	/* row has just be inserted into the parent.			*/
	m_querySet = qs ;

	if (qs != 0)
		qs->markAllDirty () ;

	if (m_next != 0)
		m_next->setQuerySet (qs == 0 ? 0 : qs->getSubset (0, 0)) ;
}

/*  KBQryLevel								*/
/*  checkUpdate	: Check that insert/update looks OK			*/
/*  offset	: uint		: Field offset				*/
/*  qwidth	: uint		: Query width (number of fields)	*/
/*  (returns)	: void		: Exits on error			*/

void	KBQryLevel::checkUpdate
	(	uint		offset,
		uint		qwidth
	)
{
	/* This routine is really for debugging, so that we can get	*/
	/* sensible errors rather than bombing. All errors detected	*/
	/* here are treated as fatal.					*/

	/* (a)	If this is the bottom-most level then we should be	*/
	/*	using exactly the remaining set of query fields.	*/
	if ((m_next == 0) && (m_getItems.count() + m_levelSets.count() != qwidth - offset))
		KBError::EFatal
		(
			QString	(TR("QuerySet bottom level %1 at %2 of %3 columns"))
				.arg(m_getItems.count() + m_levelSets.count())
				.arg(offset)
				.arg(qwidth),
			QString::null,
			__ERRLOCN
		)	;

	/* (b)	If there are insufficient unused query columns to	*/
	/*	supply all the fields we require at this level, then	*/
	/*	the query has returned less fields that we expected.	*/
	if (m_getItems.count() + m_levelSets.count() > qwidth - offset)
		KBError::EFatal
		(
			QString	(TR("QuerySet insert underrun of %1 at %2 of %3 columns"))
				.arg(m_getItems.count() + m_levelSets.count())
				.arg(offset)
				.arg(qwidth),
			QString::null,
			__ERRLOCN
		)	;
}

/*  KBQryLevel								*/
/*  insertRows	: Insert query result rows into query set		*/
/*  select	: KBSQLSelect *	: Select query containing values	*/
/*  qs		: KBQuerySet  *	: Query set into which to insert	*/
/*  rowNum	: uint		: Starting row number in query		*/
/*  forRows	: uint		: Number of rows to insert		*/
/*  offset	: uint		: Field offset				*/
/*  limit	: uint		: Insert row limit			*/
/*  progress	: KBProgress *	: Progress display			*/
/*  (returns)	: InsertRC	: Return code indicates result		*/

KBQryLevel::InsertRC
	KBQryLevel::insertRows
	(	KBSQLSelect	*select,
		KBQuerySet	*qs,
		uint		rowNum,
		uint		forRows,
		uint		offset,
		uint		limit,
		TKProgress	*progress
	)
{
	uint	 nFields = m_getItems.count() + m_levelSets.count() ;
	InsertRC rc	 = OK ;

	m_querySet = qs ;
	m_querySet->setTotalRows (forRows) ;

	checkUpdate (offset, select->getNumFields()) ;

	/* Run though the items being retrieved, setting the field	*/
	/* types from the query, and storing the item translation modes	*/
	/* in the query set.						*/
	for (uint idx = 0 ; idx < m_getItems.count() ; idx += 1)
	{
		KBItem	*item	= m_getItems.at(idx) ;

		item->setFieldType    (select->getFieldType (offset + item->getQueryIdx ())) ;
		m_querySet->setVTrans (idx, item->getVTrans()) ;
	}

	/* NOTE	We assume that the primary key is the first field at	*/
	/*	this level, so insert by blocks based on the values in	*/
	/*	that field.						*/
	while (forRows > 0)
	{
		/* Handle the situation where the number of rows is not	*/
		/* known in advance. If at the top level then we can	*/
		/* dump all rows prior to the next one to be fetched.	*/
		if (!select->rowExists(rowNum, m_qryLvl == 0))
		{
			fprintf
			(	stderr,
				"KBQryLevel::insertRows: out of data at row %u\n",
				rowNum
			)	;
			rc	= OK ;
			break	;
		}

		/* If there is a record limit set and we have reached	*/
		/* the limit then bail out.				*/
		if ((limit > 0) && (rowNum >= limit))
		{
			fprintf
			(	stderr,
				"KBQryLevel::insertRows: hit limit at %d\n",
				rowNum
			)	;
			rc	= Limit ;
			break	;
		}

		KBValue	pkValue	= select  ->getField   (rowNum, offset) ;
		uint	qsRow	= m_querySet->getNumRows () ;
		uint	useRows	= 1 ;


		for (uint qCol = 0 ; qCol < nFields ; qCol += 1)
			m_querySet->setField (qsRow, qCol, select->getField (rowNum, offset + qCol), true) ;
		m_querySet->setRowState (qsRow, KB::RSInSync) ;

		if (m_next == 0)
		{
//			fprintf
//			(	stderr,
//				"KBQryLevel::insertRows: level %d: %-8s: f=%4d u=%4d r=%4d o=%4d\n",
//				m_qryLvl,
//				(cchar *)pkValue.getRawText(),
//				forRows,
//				1,
//				rowNum,
//				offset
//			)	;
 
			rowNum  += 1 ;
			forRows -= 1 ;

#if	__KB_TKC || (QT_VERSION >= 300)

			if (progress != 0)
			{
				progress->setDone (progress->getDone() + 1) ;
				if (progress->cancelled())
				{
					rc	= Cancel ;
					break	;
				}
			}
#endif
			continue     ;
		}

		while (useRows < forRows)
		{
			if (!select->rowExists (rowNum + useRows))
				break	;
			if ( select->getField  (rowNum + useRows, offset) != pkValue)
				break	;

			useRows	+= 1 ;
		}

//		fprintf
//		(	stderr,
//			"KBQryLevel::insertRows: level %d: %-8s: f=%4d u=%4d r=%4d o=%4d\n",
//			m_qryLvl,
//			(cchar *)pkValue.getRawText(),
//			forRows,
//			useRows,
//			rowNum,
//			offset
//		)	;
 
		uint	   nNext   = m_next->m_getItems.count() + m_next->m_levelSets.count() ;
		KBQuerySet *subset = m_querySet->getSubset (qsRow, nNext) ;

		subset->clear () ;

		rc = m_next->insertRows
				(	select,
					subset,
					rowNum,
					useRows,
					offset + nFields,
					limit,
					progress
				)	;
		if (rc != OK) break	;

		rowNum	+= useRows ;
		forRows	-= useRows ;
	}

	/* At the top level, set the total rows correctly (since we are	*/
	/* called with forRows=0x7fffffff). This is fixed to be correct	*/
	/* if we have hit the limit, since with a rekall-level query	*/
	/* the limit may have been hit in a sublevel, whence rowNum	*/
	/* will not be correct.						*/
	if (m_qryLvl == 0)
	{
		uint	tr = rc == Limit ? limit : rowNum ;
		fprintf
		(	stderr,
			"KBQryLevel::insertRows: m_qryLvl=0 setTotalRows(%d)\n",
			tr
		)	;
		m_querySet->setTotalRows (tr) ;
	}

	return	rc	;
}

/*  KBQueryLevel							*/
/*  updateRow	: Update query set with a single row			*/
/*  select	: KBSQLSelect *	: Select query containing values	*/
/*  qs		: KBQuerySet  *	: Query set into which to insert	*/
/*  rowNum	: uint		: Row number to update			*/
/*  (returns)	: bool		: Any value changed			*/

bool	KBQryLevel::updateRow
	(	KBSQLSelect	*select,
		uint		rowNum
	)
{
	uint	nFields	= m_getItems.count() + m_levelSets.count() ;
	bool	changed	= false ;

	checkUpdate (0, select->getNumFields()) ;

	for (uint qCol = 0 ; qCol < nFields ; qCol += 1)
		if (m_querySet->setField (rowNum, qCol, select->getField (0, qCol), true))
			changed	= true	;

	m_querySet->setRowState (rowNum, KB::RSInSync) ;
	return	changed	;
}

/*  KBQryLevel								*/
/*  getFieldList: Get field list for table				*/
/*  fldList	: QList<KBFieldSpec> & : Field specification list	*/
/*  pKey	: int &		       : Return primary key		*/
/*  addName	: bool		       : Add table/alias name		*/
/*  pError	: KBError &	       : Error return			*/
/*  (returns)	: bool		       : Success			*/

bool	KBQryLevel::getFieldList
	(	QList<KBFieldSpec>	&fldList,
		int			&pKey,
		bool			addName,
		KBError			&pError
	)
{
	if (!m_table->getFieldList (fldList, m_dbLink, addName))
	{	pError	= m_table->lastError() ;
		return	false	;
	}

	pKey	= -1	;
	return	true	;
}

/*  KBQryLevel								*/
/*  saveRow	: Save row into query set				*/
/*  qrow	: uint		: Query row				*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: bool		: Success				*/

bool	KBQryLevel::saveRow
	(	uint	qrow,
		KBError	&pError
	)
{
	KB::RState	state	= m_querySet->getRowState (qrow, KB::RSInserted) ;

	/* We need to know if any items have changed before saving any	*/
	/* values. In order to save having to call "item->changed()"	*/
	/* twice for each update item, build a list of all changes.	*/
	QList<KBItem>	changed	  ;

	/* Check the update items, criteria are:			*/
	/* (a)	Skip block items					*/
	/* (b)	Skip primary key					*/
	/* (c)	Other update items must be valid			*/
	LITER
	(	KBItem,
		m_updItems,
		item,

		if (item->isBlock() == 0)
			if (!item->isValid (qrow, false))
			{	pError	= item->lastError () ;
				return  false  ;
			}

		/* Also add the item if the row is being inserted, in	*/
		/* order that the rowset entries get the value types.	*/
		if ((state == KB::RSInserted) || item->changed (qrow))
			changed.append (item) ;
	)

	/* If the query set will be extended then check that we have	*/
	/* insert permission and return an error if not. Similarly, if	*/
	/* any fields have changed, check that we have update		*/
	/* permission.							*/
	if ((qrow >= m_querySet->getNumRows()) && ((m_permission & QP_INSERT) == 0))
	{
		pError	= KBError
			  (	KBError::Warning,
				TR("Cannot insert rows"),
				QString	(TR("Table %1: no unique key available after insert"))
					.arg(m_table->getName()),
				__ERRLOCN
			  )	;
		return	false	;
	}

	if ((changed.count() > 0) && ((m_permission & QP_UPDATE) == 0))
	{
		pError	= KBError
			  (	KBError::Warning,
				TR("Cannot update rows"),
				QString	(TR("Table %1: no unique key column"))
					.arg(m_table->getName()),
				__ERRLOCN
			  )	;
		return	false	;
	}

	/* Get the set of values from the display fields and update	*/
	/* the query set with any that have changed.			*/
	LITER
	(	KBItem,
		changed,
		item,

		m_querySet->setField
		(	qrow,
			item->getQueryIdx (),
			item->getValue    (qrow)
		)
	)

	/* Note that synchronisation of the database will be performed	*/
	/* on a subsequent syncRow or syncAll.				*/
	return	true ;
}

/*  KBQryLevel								*/
/*  doUpdate	: Update values in specified row			*/
/*  qrow	: uint		 : Query row number			*/
/*  pValue	: KBValue *	 : Parent value if any			*/
/*  cExpr	: const QString &: Child expression			*/
/*  block	: KBBlock *	 : Requesting block			*/
/*  pError	: KBError &	 : Error return				*/
/*  priKey	: KBValue &	: Return primary key value		*/
/*  (returns)	: bool		 : Success				*/

bool	KBQryLevel::doUpdate
	(	uint		qrow,
		KBValue		*,
		const QString	&,
		KBBlock		*,
		KBValue		&priKey,
		KBError		&pError
	)
{
	QPtrDictIterator<KBQryLevelSet>	levelIter (m_levelSets) ;
	KBQryLevelSet			*ls   	;
	KBValue				dummy 	;

	while ((ls = levelIter.current()) != 0)
	{
		if (!ls->doUpdate
			(	m_querySet,
				qrow,
				ls == m_topLevelSet ? priKey : dummy,
				pError
			))
		{
			return	false	;
		}

		levelIter += 1	;
	}

	bool	changed	;
	return	getUpdates (qrow, false, changed, pError) ;
}

/*  KBQryLevel								*/
/*  doInsert	: Insert values into new row				*/
/*  qrow	: uint		 : Current row in query			*/
/*  pValue	: KBValue *	 : Parent value if any			*/
/*  cExpr	: const QString &: Child expression			*/
/*  block	: KBBlock *	 : Requesting block			*/
/*  priKey	: KBValue &	 : Return primary key value		*/
/*  pError	: KBError &	 : Error return				*/
/*  (returns)	: bool		 : Success				*/

bool	KBQryLevel::doInsert
	(	uint		qrow,
		KBValue		*pValue,
		const QString	&cExpr,
		KBBlock		*,
		KBValue		&priKey,
		KBError		&pError
	)
{
	if (m_topLevelSet == 0)
	{
		pError	= KBError
			  (	KBError::Error,
			  	TR("Unable to insert record"),
			  	TR("Rekall could not determine into which table to insert"),
			  	__ERRLOCN
			  )	;
		return	false	;
	}

	if (!m_topLevelSet->doInsert (m_querySet, qrow, pValue, cExpr, priKey, pError))
		return	false	;

	bool	changed	;
	return	getUpdates (qrow, true, changed, pError) ;
}

/*  KBQryLevel								*/
/*  doDelete	: Delete the specified row				*/
/*  qrow	: uint		: Row to delete				*/
/*  priKey	: KBValue &	: Return primary key value		*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: bool		: Success				*/

bool	KBQryLevel::doDelete
	(	uint		qrow,
		KBValue		&priKey,
		KBError		&pError
	)
{
	bool	dirty	;

	if (m_topLevelSet == 0)
	{
		pError	= KBError
			  (	KBError::Error,
			  	TR("Unable to delete record"),
			  	TR("Rekall could not determine from which table to delete"),
			  	__ERRLOCN
			  )	;
		return	false	;
	}

	/* This is the easy one. Just assemble the query from the	*/
	/* table name, the primary key name, and the primary key value	*/
	/* for the specified row.					*/
	if (m_qryDelete == 0)
	{
		QString	dele = QString("delete from %1 where %2 = %3")
				  .arg(m_dbLink.mapExpression(m_topTable->getTable   ( )))
				  .arg(m_dbLink.mapExpression(m_topTable->getUnique  ( )))
				  .arg(m_dbLink.placeHolder(0)) ;

		m_qryDelete = m_dbLink.qryDelete
				(	true,
					dele,
					m_topTable->getName()
				)	;

		if (m_qryDelete == 0)
		{
			pError	= m_dbLink.lastError() ;
			return	false	;
		}
	}

	priKey	 = m_querySet->getField(qrow, m_topLevelSet->getUniqueCol(), dirty) ;

	if (!m_qryDelete->execute (1, &priKey))
	{
		pError	= m_qryDelete->lastError() ;
		return	false ;
	}

	if (m_qryDelete->getNumRows () != 1)
	{
		pError	= KBError
			  (	KBError::Error,
			  	QString (TR("Unexpectedly deleted %1 rows")).arg(m_qryDelete->getNumRows()),
			  	m_qryDelete->getSubQuery(),
			  	__ERRLOCN
			  )	;
		return	 false	;
	}

	return	true ;
}

/*  KBQryLevel								*/
/*  startUpdate	: Check if update can be started			*/
/*  qrow	: uint			: Query row			*/
/*  locking	: KBQryBase::Locking	: Locking type			*/
/*  pError	: KBError &		: Error return			*/
/*  (returns)	: bool			: True if OK to update		*/

bool	KBQryLevel::startUpdate
	(	uint			qrow,
		KBQryBase::Locking	locking,
		KBError			&pError
	)
{
	bool	changed	;
	bool	intrans		= true	;
	void	*activeCookie	= this	;

	fprintf
	(	stderr,
		"KBQryLevel::startUpdate: qrow=%d locking=%d\n",
		qrow,
		(int)m_locking
	)	;

	if (m_qryForUpdate == 0)
		m_qryForUpdate = makeFetchSelect (true) ;


	switch (locking)
	{
		case KBQryBase::LockRowUpdate :
			if (!m_dbLink.transaction (KBServer::BeginTransaction, &activeCookie))
			{
				pError	= m_dbLink.lastError() ;
				return	false	;
			}

			intrans	= true	;
			break	;

		default	:
			break	;
	}

	/* Retrieve the latest state of the record. If this fails then	*/
	/* return false, indicating that update is not allowed.		*/
	KBError	dummy	;
	if (!getUpdates (m_qryForUpdate, qrow, false, changed, dummy))
	{
		fprintf
		(	stderr,
			"KBQryLevel::startUpdate: failed [%s][%s]\n",
			(cchar *)dummy.getMessage(),
			(cchar *)dummy.getDetails()
		)	;

		if (intrans)
			m_dbLink.transaction (KBServer::RollbackTransaction, 0) ;

		pError	= KBError
			  (	dummy.getEType(),
				"Unable to lock record for update",
				dummy.getDetails(),
				__ERRLOCN
			  )	;
		return	false	;
	}


	/* If the data was changed then the ::getUpdates call will have	*/
	/* updated the data in the query set. Returning false with the	*/
	/* set error will cause the form block to update the display	*/
	/* and to inform the user that the data has changed. The user	*/
	/* can then try again.						*/
	if (changed)
	{
		if (intrans)
			m_dbLink.transaction (KBServer::RollbackTransaction, 0) ;

		pError	= KBError
			  (	KBError::Warning,
				"Record has been changed by another user",
				QString::null,
				__ERRLOCN
			  )	;
		return	false	;
	}

	m_locking  = locking	;
	return	     true	;
}

/*  KBQryLevel								*/
/*  endUpdate	: End update						*/
/*  commit	: bool		: True to commit transaction if any	*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: bool		: Success				*/

bool	KBQryLevel::endUpdate
	(	bool		commit,
		KBError		&pError
	)
{
	fprintf
	(	stderr,
		"KBQryLevel::endUpdate: commit=%d locking=%d\n",
		commit,
		(int)m_locking
	)	;

	switch (m_locking)
	{
		case KBQryBase::NoLocking	:
			/* No locking in progress so there is nothing	*/
			/* to do.					*/
			return	true	;

		case KBQryBase::LockRowUpdate	:
			/* Locking in progress so either commit or	*/
			/* rollback the transaction.			*/
			m_locking = KBQryBase::NoLocking ;
			if (!m_dbLink.transaction
			          (	commit ?
						KBServer::CommitTransaction :
						KBServer::RollbackTransaction,
					0
			 	  ))
			{
				pError	= m_dbLink.lastError() ;
				return	false	;
			}
			return	true	;

		default	:
			/* Error case. Should not get here ...		*/
			break	;
	}

	m_locking = KBQryBase::NoLocking ;
	if (!m_dbLink.transaction (KBServer::RollbackTransaction, 0))
	{
		pError	  = m_dbLink.lastError() ;
		return	  false	;
	}

	return	true	;
}

bool	KBQryLevel::verifyChange
	(	const QString	&what,
		KBError		&pError
	)
{
	QString	errText	;

	if (m_parent->isItem()) errText = m_parent->isItem()->getErrText() ;
	if (errText.isEmpty ()) errText = TR("record") ;

	if (TKMessageBox::questionYesNo
		(	0,
			QString (TR("You are about to %1 a %2: proceed?"))
				.arg(what)
				.arg(errText),
			QString	(TR("Database %1")).arg(what)
		)
		!= TKMessageBox::Yes)
	{
		pError	= KBError
			  (	KBError::None,
				QString(TR("User cancelled %1")).arg(what),
				QString::null,
				__ERRLOCN
			  )	;
		return	false	;
	}

	return	true	;
}

/*  KBQryLevel								*/
/*  syncRow	: Synchronise row					*/
/*  qrow	: uint		 : Query row number			*/
/*  pValue	: KBValue *	 : Parent value if any			*/
/*  cExpr	: const QString &: Child expression			*/
/*  block	: KBBlock *	 : Requesting block			*/
/*  pError	: KBError &	 : Error return				*/
/*  oper	: KB::Action &	 : Operation performed			*/
/*  priKey	: KBValue &	 : Return row primary key		*/
/*  (returns)	: bool		 : Success				*/

bool	KBQryLevel::syncRow
	(	uint		qrow,
		KBValue		*pValue,
		const QString	&cExpr,
		KBBlock		*block,
		KBError		&pError,
		KB::Action	&oper,
		KBValue		&priKey
	)
{
	oper	= KB::Null	;

	switch (m_querySet->getRowState (qrow))
	{
		case KB::RSInserted	:

			if (KBOptions::getVerInsert() == KBOptions::VerifyAlways)
				if (!verifyChange (TR("insert"), pError))
					goto	unlockAndFail ;

			if (!doInsert (qrow, pValue, cExpr, block, priKey, pError))
				goto	unlockAndFail ;

			m_querySet->setRowState (qrow, KB::RSInSync) ;
			oper	= KB::Insert ;
			break	;

		case KB::RSChanged	:

			if (KBOptions::getVerUpdate() == KBOptions::VerifyAlways)
				if (!verifyChange (TR("update"), pError))
					goto	unlockAndFail ;

			if (!doUpdate (qrow, pValue, cExpr, block, priKey, pError))
				goto	unlockAndFail ;

			m_querySet->setRowState (qrow, KB::RSInSync) ;
			oper	= KB::Save ;
			break	;

		case KB::RSDeleted	:

			if (KBOptions::getVerDelete() == KBOptions::VerifyAlways)
				if (!verifyChange (TR("delete"), pError))
					goto	unlockAndFail ;

			if (!doDelete (qrow, priKey, pError))
				goto	unlockAndFail ;

			m_querySet->deleteRow (qrow)  ;
			oper	= KB::Delete ;
			break	;

		default	:
			break	;
	}

	return	endUpdate (true, pError) ;

	unlockAndFail :

		{
			KBError	dummy	;
			endUpdate (false, dummy) ;
		}

		return	false	;
}

/*  KBQryLevel								*/
/*  syncAll	: Synchronise all rows					*/
/*  pValue	: KBValue *	 : Parent value if any			*/
/*  cExpr	: const QString &: Child expression			*/
/*  block	: KBBlock *	 : Requesting block			*/
/*  pError	: KBError &	 : Error return				*/
/*  (returns)	: bool		 : Success				*/

bool	KBQryLevel::syncAll
	(	KBValue		*pValue,
		const QString	&cExpr,
		KBBlock		*block,
		KBError		&pError
	)
{
	KBValue	dummy	;
	uint	qrow2	;

	if (KBOptions::getVerMultiple())
	{
		uint	nInsert	= 0 ;
		uint	nUpdate	= 0 ;
		uint	nDelete	= 0 ;

		for (uint qrow1 = 0 ; qrow1 < m_querySet->getNumRows () ; qrow1 += 1)
			switch (m_querySet->getRowState (qrow1))
			{
				case KB::RSChanged  : nUpdate += 1 ; break ;
				case KB::RSInserted : nInsert += 1 ; break ;
				case KB::RSDeleted  : nDelete += 1 ; break ;
				default		    :		     break ;
			}

		if ((nInsert == 0) && (nUpdate == 0) && (nDelete == 0))
			return	endUpdate (true, pError) ;

		QString	errText	;

		if (m_parent->isItem()) errText = m_parent->isItem()->getErrText() ;
		if (errText.isEmpty ()) errText = TR("record") ;

		if (TKMessageBox::questionYesNo
			(	0,
				QString (TR("You are about to insert %1, update %2 and delete %3 %4%5: proceed?"))
					.arg(nInsert)
					.arg(nUpdate)
					.arg(nDelete)
					.arg(errText)
					.arg(TR("(s)")),
				TR("Database update")
			)
			!= TKMessageBox::Yes)
		{
			if (endUpdate (false, pError))
				pError	= KBError
					  (	KBError::None,
						TR("User cancelled multirecord update"),
						QString::null,
						__ERRLOCN
					  )	;
			return	false	;
		}
	}

	
	for (uint qrow1 = 0 ; qrow1 < m_querySet->getNumRows () ; qrow1 += 1)
		switch (m_querySet->getRowState (qrow1))
		{
			case KB::RSChanged  :
				if (!doUpdate (qrow1, pValue, cExpr, block, dummy, pError))
					goto	unlockAndFail ;
				break	;

			case KB::RSInserted :
				if (!doInsert (qrow1, pValue, cExpr, block, dummy, pError))
					goto	unlockAndFail ;
				break	;

			default	:
				break	;
		}

	qrow2	= 0 ;
	while (qrow2 < m_querySet->getNumRows ())
	{
		if (m_querySet->getRowState (qrow2) != KB::RSDeleted)
		{	m_querySet->setRowState (qrow2, KB::RSInSync) ;
			qrow2 += 1 ;
			continue   ;
		}

		if (!doDelete (qrow2, dummy, pError))
			goto	unlockAndFail	;

		m_querySet->deleteRow (qrow2)	;
	}

	return	endUpdate (true, pError) ;

	unlockAndFail :

		{
			KBError	dummy	;
			endUpdate (false, dummy) ;
		}

		return	false	;
}


/*  KBQryLevel								*/
/*  findPermissions							*/
/*		: Get permissions at this level				*/
/*  pError	: KBError &	 : Error return				*/
/*  (returns)	: bool		 : Success				*/

bool	KBQryLevel::findPermissions
	(	KBError	&pError
	)
{
	/* Get the permissions at this level. Although we might be	*/
	/* able to handle different permissions where there is more	*/
	/* than one table, it would be a real pain, so just go for the	*/
	/* lowest common denominator.					*/
	QPtrDictIterator<KBQryLevelSet>	levelIter (m_levelSets) ;
	KBQryLevelSet			*ls  ;
	int				perm ;

	m_reason     = QString::null ;
	m_permission = QP_SELECT|QP_INSERT|QP_DELETE|QP_UPDATE ;

	while ((ls = levelIter.current()) != 0)
	{
		if ((perm = ls->findPermissions (m_reason, pError)) == -1)
			return false ;

		m_permission &= perm ;
		levelIter    += 1	   ;
	}

	m_reason += TR("<b>Summary</b><ul>") ;

	/* TO BE FIXED:							*/
	/* Only allow insertion if there is a single table ...		*/
//	if (m_levelSets.count() > 1)
//	{
//		m_reason     += TR("<li>Multiple tables, no insertion</li>") ;
//		m_permission &= ~(QP_INSERT) ;
//	}
	QList<KBTable> stl ;
	m_table->getQueryInfo (stl) ;
	if (stl.count() > 0)
	{
		m_reason     += TR("<li>Multiple tables, no insertion</li>") ;
		m_permission &= ~(QP_INSERT) ;
	}

	if (m_distinct)
	{
		m_reason     += TR("<li>Distinct query, no insertion, update or delete</li>") ;
		m_permission &= ~(QP_INSERT|QP_UPDATE|QP_DELETE) ;
	}
	if (!m_glbGroup.isEmpty())
	{
		m_reason     += TR("<li>Grouped query, no insertion, update or delete</li>") ;
		m_permission &= ~(QP_INSERT|QP_UPDATE|QP_DELETE) ;
	}

	m_reason += "<li><b>" + permissionToText(m_permission) + "</b></li>" ;
	m_reason += "</ul>" ;

	if ((m_permission & QP_UPDATE) == 0)
		LITER
		(	KBItem,
			m_updItems,
			item,

			item->setFlags (KBFieldSpec::ReadOnly) ;
		)

	return	m_next == 0 ? true : m_next->findPermissions (pError) ;
}


/*  KBQryLevel								*/
/*  buildSelect	: Build select query for the node			*/
/*  select	: KBSelect &	: Select object accumulating query	*/
/*  descend	: bool		: Descend into sub-levels		*/
/*  distinct	: bool		: Select query containts "distinct"	*/
/*  (returns)	: void		:					*/

void	KBQryLevel::buildSelect
	(	KBSelect	&select,
		bool		descend,
		bool		distinct
		
	)
{
	int	prCols	= 0	;

	m_table->addToSelect (select, descend) ;

	/* We add a primary key column for each table, so that the	*/
	/* user can (if they really want to) add a control do display	*/
	/* and update the primary key value. If there is no primary	*/
	/* key then just fetch zero; this means that we always fetch	*/
	/* (valSets.count() + m_getItems.count()) values.		*/
	QPtrDictIterator<KBQryLevelSet>	levelIter (m_levelSets) ;
	KBQryLevelSet			*ls  ;

	while ((ls = levelIter.current()) != 0)
	{
		KBTable	*lsTable = ls->getTable () ;

		/* Include the primary key if				*/
		/* (a) The query does not contain "distinct"		*/
		/* (b) There is no expression that groups the table	*/
		/* (c) There is a unique column				*/

//		fprintf
//		(	stderr,
//			"KBQryLevel::buildSelect: [%s] [%d][%d][%s]\n",
//			(cchar *)lsTable->getName(),
//			distinct,
//			lsTable->getGrouped(),
//			(cchar *)lsTable->getUnique()
//		)	;

		if (!distinct && !lsTable->getGrouped() && !lsTable->getUnique().isEmpty())
		{	select.appendExpr (lsTable->getName() + "." + lsTable->getUnique()) ;
			ls->setUniqueCol  (prCols) ;
		}
		else	select.appendExpr ("0") ;

		levelIter += 1	;
		prCols	  += 1	;
	}

	for (uint idx = 0 ; idx < m_getItems.count() ; idx += 1)
	{
		KBItem *item	= m_getItems.at(idx) ;
		item->setQueryIdx (KBQryIdx (m_qryLvl, idx + prCols)) ;
		select.appendExpr (item->getExpr()) ;
	}

	if (descend && (m_next != 0))
		m_next->buildSelect (select, true, distinct) ;
}

/*  KBSQLSelect								*/
/*  addQueryterms: Add query terms to select query			*/
/*  select	 : KBSelect &	: Select query being assembled		*/
/*  values	 : KBValue *	: Query values array			*/
/*  qrow	 : uint		: Query row originating query		*/
/*  slot	 : uint &	: Next slot in values array		*/
/*  (returns)	 : void		:					*/

void	KBQryLevel::addQueryTerms
	(	KBSelect	&select,
		KBValue		*values,
		uint		qrow,
		uint		&slot
	)
{
	LITER
	(	KBItem,
		m_getItems,
		item,

		if (item->isEmpty(qrow)) continue ;

		KBValue	value	= item->getValue   (qrow) ;
		QString	vtext	= value.getRawText () ;
		cchar	*oper	= m_dbLink.mapOperator (KBServer::EQ, "=") ;


		if	(vtext.startsWith ("<>"))
		{
			value	= KBValue (vtext.mid(2).stripWhiteSpace(), value.getType()) ;
			oper	= m_dbLink.mapOperator (KBServer::NEQ,  "!=") ;
		}
		else if	(vtext.startsWith ("!="))
		{
			value	= KBValue (vtext.mid(2).stripWhiteSpace(), value.getType()) ;
			oper	= m_dbLink.mapOperator (KBServer::NEQ,  "!=") ;
		}
		else if	(vtext.startsWith ("<="))
		{
			value	= KBValue (vtext.mid(2).stripWhiteSpace(), value.getType()) ;
			oper	= m_dbLink.mapOperator (KBServer::LE,   "<=") ;
		}
		else if	(vtext.startsWith (">="))
		{
			value	= KBValue (vtext.mid(2).stripWhiteSpace(), value.getType()) ;
			oper	= m_dbLink.mapOperator (KBServer::GE,   ">=") ;
		}
		else if	(vtext.startsWith ("<") )
		{
			value	= KBValue (vtext.mid(1).stripWhiteSpace(), value.getType()) ;
			oper	= m_dbLink.mapOperator (KBServer::LT,   "<" ) ;
		}
		else if	(vtext.startsWith (">") )
		{
			value	= KBValue (vtext.mid(1).stripWhiteSpace(), value.getType()) ;
			oper	= m_dbLink.mapOperator (KBServer::GT,   ">" ) ;
		}
		else if (value.getRawText().find('%') >= 0)
			oper	= m_dbLink.mapOperator (KBServer::Like, "like") ;


		select.appendWhere
		(	item->getExpr() + " " + oper + " " + m_dbLink.placeHolder(slot)
		)	;

		values[slot] = value ;
		slot	+= 1	     ;
	)
}

bool	KBQryLevel::getSelect
	(	KBSelect	&select
	)
{
	/* If there is any grouping then identify all tables in the	*/
	/* group expression and mark all such tables as grouped.	*/
	if (!m_glbGroup .isEmpty ())
	{
		QStringList	tables	;
		QStringList	columns	;

		findTableColumnPairs	(m_glbGroup, tables, columns) ;
		markGroups		(tables) ;
	}



	select.setDistinct (m_distinct) ;
	buildSelect 	   (select, true, m_distinct) ;

	if (!m_glbWhere.isEmpty ())
		select.appendWhere  (m_glbWhere ) ;

	if (!m_glbGroup .isEmpty ())
		select.appendGroup  (m_glbGroup ) ;

	if (!m_glbHaving.isEmpty ())
		select.appendHaving (m_glbHaving) ;

	if (!m_glbOrder.isEmpty())
		select.appendOrder  (m_glbOrder ) ;

	return	true	;
}

/*  KBQryLevel								*/
/*  doSelect	: Execute associated select statement			*/
/*  pValue	: KBValue *	 : Parent value if any			*/
/*  cExpr	: const QString &: Child expression			*/
/*  filter	: const QString &: Filter if any			*/
/*  sorting	: const QString &: Sorting if any			*/
/*  isQuery	: bool		 : True to add query terms		*/
/*  qrow	: uint		 : Query row number			*/
/*  pError	: KBError &	 : Error return				*/
/*  (returns)	: bool		 : Success				*/

bool	KBQryLevel::doSelect
	(	KBValue		*pValue,
		const QString	&cExpr,
		const QString	&filter,
		const QString	&sorting,
		bool		isQuery,
		uint		qrow,
		KBError		&pError
	)
{
	/* doSelect is always called before the user gets a chance to	*/
	/* do anything, so now is the time to get the level permissions	*/
	/* If update is allowed then the update items are no longer	*/
	/* fixed.							*/
	if (!findPermissions (pError)) return false ;

	fprintf	(stderr, "doSelect: permission=%04x\n", m_permission) ;

	/* Unlike insert, update and delete, we build a new query each	*/
	/* time since (a) most likely we will only execute a select for	*/
	/* all rows once and (b) selects with addition query terms will	*/
	/* differ each time.						*/
	KBSQLSelect	*qrySelect ;
	KBSelect	select	   ;

	if (!getSelect (select))
		return	false	;

	KBValue	*values	= new KBValue[m_getItems.count() + m_levelSets.count() + 1] ;
	uint	slot	= 0	   ;


	if (pValue != 0)
	{	select.appendWhere  (cExpr + " = " + m_dbLink.placeHolder (slot)) ;
		values[slot] = *pValue	;
		slot	    += 1	;
	}


	/* If this is a form query then add in any query terms from the	*/
	/* form.							*/
	if (isQuery)
		addQueryTerms (select, values, qrow, slot) ;


	if (!filter .isEmpty())
		select.appendWhere  (filter  ) ;
	if (!sorting.isEmpty())
		select.appendOrder  (sorting ) ;

	if (m_limit > 0)
		select.setLimit	    (m_limit + 1, 0) ;

	/* Allocate a new query set, passing it the list if items to be	*/
	/* trieved so that it can note the translation modes.		*/
	if (m_querySet == 0)
		m_querySet = new KBQuerySet
			     (		m_getItems.count() + m_levelSets.count()
			     )	;

	/* Prepare, and then execute the query, checking in case of	*/
	/* execution errors.						*/
	if ((qrySelect = m_dbLink.qrySelect (true, select.getQueryText(&m_dbLink))) == 0)
	{
		pError	= m_dbLink.lastError() ;
		return	false ;
	}
	if (!qrySelect->execute (slot, values))
	{
		pError	= qrySelect->lastError() ;
		delete	qrySelect ;
		return	false	  ;
	}

	m_querySet->clear	() ;

	TKProgress *progress	= 0  ;
	uint	   nrows	= 0  ; //qrySelect->getNumRows() ;

#if	__KB_TKC || (QT_VERSION >= 300)
	progress = new TKProgress
		   (	TR("Loading records ...."),
			TR("Record"),
			TR("of"),
			nrows > 0,
			200
		   )	;
	if (nrows >  0) progress->setTotal (nrows) ;
	if (nrows > 40) progress->show	   () ;
#endif

	InsertRC  rc = insertRows
		  (	qrySelect,
			m_querySet,
			0,
			0x7fffffff,
			0,
			m_limit,
			progress
		  )	;

#if	__KB_TKC || (QT_VERSION >= 300)
	delete	progress  ;
#endif

	delete	qrySelect ;
	delete	[]values  ;

	switch (rc)
	{
		case Cancel :
			KBError::EWarning
			(	TR("User cancelled: not all data loaded"),
				QString::null,
				__ERRLOCN
			)	;
			break	;

		case Limit :
			KBError::EWarning
			(	TR("Query record limit reached: not all data loaded"),
				QString(TR("Loaded %1 records")).arg(m_querySet->getTotalRows()),
				__ERRLOCN
			)	;
			break	;

		default	:
			break	;
	}

	return	true	  ;
}

KBSQLSelect
	*KBQryLevel::makeFetchSelect
	(	bool			forUpdate
	)
{
	KBSelect    select ;
	buildSelect (select, false, false) ;

	QPtrDictIterator<KBQryLevelSet>	levelIter (m_levelSets) ;
	KBQryLevelSet			*ls 	  ;
	int				prColumn  ;
	uint				colidx	  = 0 ;

	for ( ; (ls = levelIter.current()) != 0 ; levelIter += 1)
		if ((prColumn = ls->getUniqueCol()) >= 0)
		{
			KBTable *lvlTbl = ls->getTable() ;
			QString	tt	= lvlTbl->getName() + "." + lvlTbl->getUnique() ;

			select.appendWhere
			(	tt + " = " + m_dbLink.placeHolder(colidx)
			)	;

			colidx	+= 1 ;
		}

	select.setForUpdate (forUpdate) ;

	fprintf
	(	stderr,
		"KBQryLevel::makeFetchSelect: [%d]->[%s]\n",
		forUpdate,
		(cchar *)select.getQueryText(&m_dbLink)
	)	;

	return	m_dbLink.qrySelect
		(	true,
			select.getQueryText(&m_dbLink),
			forUpdate
		)	;
}

/*  KBQryLevel								*/
/*  getUpdates	: Retrieve updates from database			*/
/*  fetch	: KBSQLSelect *	: Fetch select query			*/
/*  qrow	: uint		: Query row number, ie., the cursor	*/
/*  postInsert	: bool		: Just done an insert			*/
/*  changed	: bool		: Set if any value changed		*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: bool		: Success				*/

bool	KBQryLevel::getUpdates
	(	KBSQLSelect	*fetch,
		uint		qrow,
		bool		postInsert,
		bool		&changed,
		KBError		&pError
	)
{
	/* Scan the query set to see which rows need to be refetched.	*/
	/* These are any that have any primary key value in common with	*/
	/* the value in the updated row (and which should include the	*/
	/* updated row itself). First scan and flag such rows, so that	*/
	/* we need only fetch each one once.				*/
	uint	nRows	= m_querySet->getNumRows() ;
	bool	*refetch= new bool[nRows]	 ;
	bool	dirty	;

	uint	*pCols	= new uint   [m_levelSets.count()] ;
	KBValue	*pVals	= new KBValue[m_levelSets.count()] ;
	uint	pCnt	= 0			 ;

	memset	(refetch, 0, nRows * sizeof(bool)) ;

	QPtrDictIterator<KBQryLevelSet>	levelIter (m_levelSets) ;
	KBQryLevelSet			*ls 	  ;
	int				prColumn  ;

	for ( ; (ls = levelIter.current()) != 0 ; levelIter += 1)
	{
		fprintf
		(	stderr,
			"KBQryLevel::getUpdates: pi=%d ls=%s up=%d pc=%d\n",
			postInsert,
			(cchar *)ls->getTable()->getTable(),
			ls->updated	(),
			ls->getUniqueCol()
		)	;

		if ((prColumn = ls->getUniqueCol()) < 0)
			continue ;

		/* As we go, make a note of the query set columns	*/
		/* which contain the primary key values			*/
		pCols[pCnt] = prColumn ;
		pCnt	   += 1 ;


		/* Note that we ignore the level set if it was not	*/
		/* updated; if it was not updated then any data fetched	*/
		/* from the associated table for the corresponding	*/
		/* primary key cannot have changed.			*/
		if (!postInsert && !ls->updated())
			continue ;

		const KBValue &prival = m_querySet->getField (qrow, prColumn, dirty) ;

		for (uint r = 0 ; r < nRows ; r += 1)
			if (m_querySet->getField (r, prColumn, dirty) == prival)
				refetch[r] = true ;
	}

	/* Now go and retrieve the rows. Each fetch is qualified by the	*/
	/* actual primary key values, so it should return just the row	*/
	/* in question, which can then be updated.			*/
	changed	= false ;

	for (uint r = 0 ; r < nRows ; r += 1)
		if (refetch[r])
		{
			for (uint idx = 0 ; idx < pCnt ; idx += 1)
				pVals[idx] = m_querySet->getField (r, pCols[idx], dirty) ;

			if (!fetch->execute (pCnt, pVals))
			{	
				pError	= fetch->lastError() ;
				return	false	;
			}

			if (updateRow (fetch, r)) changed = true ;
		}


	delete	[]refetch ;
	delete	[]pCols	  ;
	delete	[]pVals	  ;

	return	true  ;
}

/*  KBQryLevel								*/
/*  getUpdates	: Retrieve updates from database			*/
/*  qrow	: uint		: Query row number, ie., the cursor	*/
/*  postInsert	: bool		: Just done an insert			*/
/*  changed	: bool		: Set if any value changed		*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: bool		: Success				*/

bool	KBQryLevel::getUpdates
	(	uint		qrow,
		bool		postInsert,
		bool		&changed,
		KBError		&pError
	)
{
	/* After an insert or update query, we fetch new or updated	*/
	/* row(s) back from the database. This has the dual benefits of	*/
	/* automatically getting values for any query columns which are	*/
	/* expressions rather then simple column names, and retrieving	*/
	/* values which may result from things like database triggers.	*/

	/* NOTE - To be sorted						*/
	/*	The code here only retrieves values for this level,	*/
	/*	which is correct if this is the bottom-most level.	*/
	/*	Really, w we should do a query all the way down ....	*/

	/* First time round, build the query. This is a select for this	*/
	/* level, qualified by expressions for all the primary key	*/
	/* columns.							*/
	if (m_qryFetch == 0)
		m_qryFetch = makeFetchSelect (false) ;


	return	getUpdates (m_qryFetch, qrow, postInsert, changed, pError) ;

	/* HACK ALERT!							*/
	/* This needs fixing in XBSQL: Fixed with xbsql-0.11		*/
	// if (m_dbLink.getDBType() == "xbase")
	// {
	// 	DELOBJ	(m_qryFetch ) ;
 	//	fprintf	(stderr, "Deleted fetch query for xbase\n") ;
 	// }
}

/*  KBQryLevel								*/
/*  sortByColumn: Sort data by column					*/
/*  qrow	: uint		: Query row number			*/
/*  asc		: bool		: Sort ascending			*/
/*  item	: KBItem *	: Item associated with column		*/
/*  (returns)	: uint		: Total rows here and below		*/

void	KBQryLevel::sortByColumn
	(	uint	qcol,
		bool	asc,
		KBItem	*item
	)
{
	m_querySet->sortByColumn (qcol, asc, item) ;
}

/*  KBQryLevel								*/
/*  setRowMarked: Set row marking					*/
/*  qrow	: uint		: Query row				*/
/*  op		: KB::MarkOp	: Mark operation			*/
/*  (returns)	: void		:					*/

void	KBQryLevel::setRowMarked
	(	uint		qrow,
		KB::MarkOp	op
	)
{
	if (m_querySet != 0) m_querySet->setRowMarked (qrow, op) ;
}

/*  KBQryLevel								*/
/*  getRowMarked: Set row mark state					*/
/*  qrow	: uint		: Query row				*/
/*  (returns)	: bool		: True if marked			*/

bool	KBQryLevel::getRowMarked
	(	uint		qrow
	)
{
	return	m_querySet == 0 ? false : m_querySet->getRowMarked (qrow) ;
}

/*  KBQryLevel								*/
/*  deleteAllMarked							*/
/*		: Delete all marked rows				*/
/*  nrows	: uint &	: Return number of rows deleted		*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: bool		: Success				*/

bool	KBQryLevel::deleteAllMarked
	(	uint		&nrows,
		KBError		&pError
	)
{
	if (m_querySet == 0)
	{	nrows	= 0	;
		return	true	;
	}

	return	m_querySet->deleteAllMarked (nrows, m_parent, pError) ;
}

/*  KBQryLevel								*/
/*  getSQLText	: Get SQL text						*/
/*  pretty	: bool		: Get in pretty format			*/
/*  (returns)	: QString	: Text					*/

QString	KBQryLevel::getSQLText
	(	bool	pretty
	)
{
	if (m_qryLvl == 0)
	{
		KBSelect	select ;

		select.setDistinct (m_distinct) ;
		select.setLimit	   (m_limit, 0) ;
		buildSelect 	   (select, true, !m_distinct && m_glbGroup.isEmpty()) ;

		if (!m_glbWhere .isEmpty()) select.appendWhere  (m_glbWhere ) ;
		if (!m_glbGroup .isEmpty()) select.appendGroup  (m_glbGroup ) ;
		if (!m_glbHaving.isEmpty()) select.appendHaving (m_glbHaving) ;
		if (!m_glbOrder .isEmpty()) select.appendOrder  (m_glbOrder ) ;

		return	pretty ?
				select.getPrettyText (false, &m_dbLink) :
				select.getQueryText  (&m_dbLink) ;
	}

	return	QString::null	;
}

/*  KBQryLevel								*/
/*  getSQLReason: Get SQL explanation text				*/
/*  (returns)	: QString	:					*/

QString	KBQryLevel::getSQLReason ()
{
	KBError	error	;

	if (!findPermissions (error))
		return	QString	(TR("Error getting permissions:<br/>%1<br/>%2"))
				.arg(error.getMessage())
				.arg(error.getDetails()) ;

	return	m_reason ;
}

/*  KBQryLevel								*/
/*  markGroups	: Mark tables that appear in "group by" 		*/
/*  tables	: QStringList &	: List of tables			*/
/*  (returns)	: void		:					*/

void	KBQryLevel::markGroups
	(	const QStringList	&tables
	)
{
	/* Check the top level set's table against the list of tables	*/
	/* that are involved in grouping to see if it is amongst them,	*/
	/* in which case mark it so.					*/
	if (m_topLevelSet != 0)
	{
		KBTable	*table	= m_topLevelSet->getTable() ;

		for (uint idx = 0 ; idx < tables.count() ; idx += 1)
			if (table->getName() == tables[idx])
			{
				fprintf
				(	stderr,
					"KBQryLevel::markGroups: marking [%s]\n",
					(cchar *)tables[idx]
				)	;
				table->setGrouped (true) ;
				break	;
			}
	}

	/* Pass the list down through all levels ...			*/
	if (m_next != 0)
		m_next->markGroups (tables) ;
}
