/***************************************************************************
    file	         : kb_node.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>

#ifndef 	_WIN32
#include	"kb_node.moc"
#else
#include	"kb_node.h"
#endif

#include	"kb_param.h"
#include	"kb_config.h"
#include	"kb_block.h"
#include	"kb_nodemonitor.h"
#include	"kb_slot.h"

#if		! __KB_RUNTIME
#include	"kb_propdlg.h"
#endif

extern	LIBKBASE_API	uint GetNodeFlags(const QString &);


/*  KBNode								*/
/*  KBNode	: Constructor for node base class			*/
/*  parent	: KBNode *	: Parent node if any			*/
/*  element	: cchar *	: Element tag				*/
/*  (returns)	: KBNode	:					*/

KBNode::KBNode
	(	KBNode	*parent,
		cchar	*element
	)
	:
	parent	(parent),
	element	(element)
{
	flags	= GetNodeFlags (element) ;
	showing	= KB::ShowAsUnknown	 ;
	root	= 0	;
	monitor	= 0	;
	attrCnt	= 0	;

	if (parent != 0)
	{	root = parent->getRoot () ;
		parent->addChild   (this) ;
	}
	else	root	= this	;

	m_attrNotes	= new KBAttrStr (this, "notes", "", KAF_NOTES) ;

}

/*  KBNode								*/
/*  KBNode	: Constructor for node base class			*/
/*  parent	: KBNode *	: Parent node if any			*/
/*  element	: cchar *	: Element tag				*/
/*  aList	: const QDict<QString> &				*/
/*				: List of attributes			*/
/*  (returns)	: KBNode	:					*/

KBNode::KBNode
	(	KBNode			*parent,
		cchar			*element,
		const QDict<QString>	&aList
	)
	:
	parent	(parent),
	element	(element)
{
	flags	= GetNodeFlags (element) ;
	showing	= KB::ShowAsUnknown	 ;
	root	= 0	;
	monitor	= 0	;
	attrCnt	= 0	;

	if (parent != 0)
	{	root = parent->getRoot () ;
		parent->addChild   (this) ;
	}
	else	root	= this	;

	m_attrNotes	= new KBAttrStr (this, "notes", aList, KAF_NOTES) ;
}

/*  KBNode								*/
/*  KBNode	: Constructor for node base class			*/
/*  parent	: KBNode *	: Parent node if any			*/
/*  node	: KBNode *	: Extant node				*/
/*  (returns)	: KBNode	:					*/

KBNode::KBNode
	(	KBNode	*parent,
		KBNode	*node
	)
	:
	parent	(parent),
	element	(node->element)
{
	flags	= GetNodeFlags (element) ;
	showing	= KB::ShowAsUnknown	 ;
	root	= 0	;
	monitor	= 0	;

	if (parent != 0)
	{	root = parent->getRoot () ;
		parent->addChild   (this) ;
	}
	else	root	= this	;

	m_attrNotes	= new KBAttrStr (this, "notes", node, KAF_NOTES) ;

	/* Normally we use ::replicateBelow() when replicating nodes	*/
	/* that can have children, but handle children that are		*/
	/* configuration objects as a special case, since they are	*/
	/* quite pervasive *and* so not depend on their parent (ie.,	*/
	/* this node) being completely constructed.			*/
	LITER
	(	KBNode,
		node->getChildren(),
		child,

		if (child->isConfig() != 0) child->replicate (this)
	)
}

/*  KBNode								*/
/*  ~KBNode	: Destructor for node base class			*/
/*  (returns)	:		:					*/

KBNode::~KBNode ()
{
	LITER
	(	KBAttr,
		attribs,
		attrib,
		if ((attrib->getFlags() & KAF_USER) != 0) delete attrib ;
	)


	while (children.count() > 0) delete children.first() ;
	if (parent != 0) parent->remChild (this) ;
	DELOBJ	(monitor) ;
	DELOBJ	(m_attrNotes) ;
}

/*  KBNode								*/
/*  startParse	: Start parsing for this node				*/
/*  (returns)	: void		:					*/

void	KBNode::startParse ()
{
}

/*  KBNode								*/
/*  endParse	: End parsing for this node				*/
/*  (returns)	: KBNode *	: Parent element			*/

KBNode *KBNode::endParse ()
{
	return	parent	;
}

/*  KBNode								*/
/*  addAttr	: Add an attribute					*/
/*  attr	: KBAttr *	  : The attribute			*/
/*  attrIdx	: uint &	  : Return attribute index		*/
/*  (returns)	: KBNodeMonitor * : Current node monitor if any		*/

KBNodeMonitor
	*KBNode::addAttr
	(	KBAttr		*attr,
		uint		&attrIdx
	)
{
	attribs.append (attr) ;

	attrCnt += 1	   ;
	attrIdx	 = attrCnt ;

	return	monitor ;
}

/*  KBNode								*/
/*  remAttr	: Remove an attribute					*/
/*  attr	: KBAttr *	: The attribute				*/
/*  (returns)	: void		:					*/

void	KBNode::remAttr
	(	KBAttr	*attr
	)
{
	attribs.remove (attr) ;
}

/*  KBNode								*/
/*  getAttrVal	: Find a named attribute value				*/
/*  name	: const QString & : Attribute name			*/
/*  (returns)	: QString	  : Value or null			*/

QString	KBNode::getAttrVal
	(	const QString	&name
	)
{
	KBAttr	*attr	;
	if ((attr = getAttr (name)) != 0) return attr->getValue() ;
	return	QString	() ;
}

/*  KBNode								*/
/*  setAttrVal	: Set a named attribute value				*/
/*  name	: const QString & : Attribute name			*/
/*  value	: const QString & : Attribute value			*/
/*  mkuser	: bool		  : Create user attribute if needed	*/
/*  update	: bool		  : Update objects			*/
/*  (returns)	: bool		  : Success				*/

bool	KBNode::setAttrVal
	(	const QString	&name,
		const QString	&value,
		bool		mkuser,
		bool		update
	)
{
	KBAttr	*attr	;

	if ((attr = getAttr (name)) == 0)
		if (mkuser)
		{
			new	KBAttrStr (this, name, value, KAF_USER) ;
			return	true   ;
		}
		else	return	false  ;

	attr->setValue (value)	;
	if (update) updateProps () ;
	return	true   ;
}

/*  KBNode								*/
/*  getAttr	: Find a named attribute				*/
/*  name	: const QString & : Attribute name			*/
/*  (returns)	: KBAttr *	  : Attribute or null			*/

KBAttr *KBNode::getAttr
	(	const QString	&name
	)
{
	for (KBAttr *attr = attribs.first() ; attr != 0 ; attr = attribs.next())
		if (attr->getName() == name)
			return	attr	;

	return	0 ;
}

/*  KBNode								*/
/*  addChild	: Add a child node					*/
/*  child	: KBNode *	: Child in question			*/
/*  (returns)	: void		:					*/

void	KBNode::addChild
	(	KBNode	*child
	)
{
	children.append (child) ;

	/* If we are displaying a monitor then locate the children	*/
	/* sub-entry and get a monitor displayed for the child. Note:	*/
	/* at this stage, the new node will not have any attributes,	*/
	/* these will be attached to the monitor as the are created.	*/
	if (monitor != 0)
		for (QListViewItem *entry  = monitor->firstChild () ;
				    entry != 0 ;
				    entry  = entry  ->nextSibling())
			if (entry->text (0) == "Children")
			{
				child->showMonitor (entry) ;
				break	;
			}
}

/*  KBNode								*/
/*  remChild	: Remove a child node					*/
/*  child	: KBNode *	: Child in question			*/
/*  (returns)	: void		:					*/

void 	KBNode::remChild
	(	KBNode	*child
	)
{
	if (!children.remove (child))
		fprintf	(stderr, "remChild: lost child\n") ;
}

/*  KBNode								*/
/*  showAs	: Set or clear design mode				*/
/*  mode	: KB::ShowAs	: Design mode				*/
/*  (returns)	: void		:					*/

void	KBNode::showAs
	(	KB::ShowAs	mode
	)
{
	bool	ac	= false	;

	showing	= mode ;

	LITER
	(	KBAttr,
		attribs,
		attrib,

		if ((attrib->getFlags() & KAF_USER) != 0)
		{	attribs.removeRef (attrib) ;
			delete	attrib	;
			continue	;
		}

		if (attrib->showAs (mode))
			ac	= true	;
	)

	/* The default action is just to recurse down through all the	*/
	/* nodes.							*/
	LITER
	(	KBNode,
		children,
		child,
		child->showAs (mode)
	)

	if (ac) updateProps () ;
}

void	KBNode::updateProps ()
{
}

/*  KBNode								*/
/*  prepare	: Prepare prior prior to execution			*/
/*  (returns)	: void		:					*/

void	KBNode::prepare ()
{
	/* Note that here we will be recursing down to child nodes, to	*/
	/* the first value is meaningless.				*/
	LITER
	(	KBNode,
		children,
		child,
		child->prepare ()
	)
}

/*  KBNode								*/
/*  printNode	: Output node XML					*/
/*  text	: QString &	: Result string				*/
/*  indent	: int		: Indent level of this node		*/
/*  (returns)	: void		:					*/

void	KBNode::printNode
	(	QString	&text,
		int	indent
	)
{
	bool	anyChild = (children.count() > 0) || (m_slotList.count() > 0) ;
	QString	nodeText ;

	text	+= QString("%1<%2").arg("", indent).arg(element) ;
	for (uint idx = 0 ; idx < attribs .count () ; idx += 1)
		attribs.at(idx)->printAttr(text, nodeText, indent + 2) ;

	if (!anyChild && nodeText.isEmpty())
	{
		text	+= "/>\n" ;
		return	;
	}

	text	+= ">\n" ;

	for (uint idx1 = 0 ; idx1 <   children.count () ; idx1 += 1)
		children  .at(idx1)->printNode (text, indent + 2) ;

	for (uint idx2 = 0 ; idx2 < m_slotList.count () ; idx2 += 1)
		m_slotList.at(idx2)->printNode (text, indent + 2) ;


	text	+= nodeText ;
	text	+= QString("%1</%2>\n").arg("", indent).arg(element) ;
}

#if	! __KB_RUNTIME

/*  KBNode								*/
/*  propertyDlg	: Show property dialog for node				*/
/*  caption	: cchar *	: Caption for dialog			*/
/*  iniattr	: cchar *	: Attribute to show			*/
/*  (returns)	: bool		: Used clicked OK			*/

bool	KBNode::propertyDlg
	(	cchar	*caption,
		cchar	*iniattr
	)
{
	/* This is the default, so that we always get a properties	*/
	/* dialog. Most node types will have their own derived property	*/
	/* dialogs, which handle specific attributes - the dialog here	*/
	/* will treat everything as a text field.			*/
	KBPropDlg propDlg (this, caption, attribs, iniattr) ;
	return	  propDlg.exec () ;
}
#endif

/*  KBNode								*/
/*  applyDialog	: Apply attributes, called from property dialogs	*/
/*  (returns)	: void		:					*/

void	KBNode::applyDialog ()
{
	/* This is a dummy. Interested parties will implement it.	*/
}

/*  KBNode								*/
/*  replicate	: Replicate this node					*/
/*  _parent	: KBNode *	: Parent node				*/
/*  (returns)	: KBNode *	: Always null				*/

KBNode	*KBNode::replicate
	(	KBNode	*
	)
{
	/* This is an error case. The replicate method must be		*/
	/* implemented in the final derived class.			*/
	KBError::EFault
	(	QString(TR("KBNode::replicate() called for %1")).arg(element),
		QString::null,
		__ERRLOCN
	)	;
	return	0 ;
}

/*  KBNode								*/
/*  replicateBelow							*/
/*		: Replicate children of this node			*/
/*  _parent	: KBNode *	: Parent node				*/
/*  (returns)	: KBNode *	: Always null				*/

KBNode	*KBNode::replicateBelow
	(	KBNode	*_parent
	)
{
	LITER
	(	KBNode,
		children,
		child,

		/* Don't replicate configuration objects, this will	*/
		/* have been done in the KBNode replication constructor	*/
		if (child->isConfig() == 0)
			child->replicate (_parent) ;
	)

	return	_parent	;
}

/*  KBObject								*/
/*  write	: Write object to report writer				*/
/*  writer	: KBWriter *	: Writer object				*/
/*  offset	: QPoint	: Geometry offset			*/
/*  first	: bool		: First on page				*/
/*  extra	: init &	: Return extra space			*/
/*  (returns)	: void		:					*/

bool	KBNode::write
	(	KBWriter	*writer,
		QPoint		offset,
		bool		first,
		int		&extra
	)
{
	LITER
	(	KBNode,
		children,
		child,
		if (!child->write (writer, offset, first, extra)) return false	;
	)

	return	true	;
}

/*  KBNode								*/
/*  setMonitor	: Set monitor values for this node			*/
/*  myitem	: KBNodeMonitor * : Monitor item for this node		*/
/*  (returns)	: void		  :					*/

void	KBNode::setMonitor
	(	KBNodeMonitor	*myitem
	)
{
	KBNodeMonitor	*attritem = 0 ;
	KBNodeMonitor	*nodeitem = 0 ;

	if (myitem != 0)
	{
		KBAttr	*name	= getAttr ("name") ;
		myitem->setText (0, element) ;
		if (name != 0) myitem->setText (1, name->getValue()) ;

		attritem = new KBNodeMonitor  (0, myitem) ;
		attritem->setText	(0, "Attributes") ;
		attritem->setSelectable (false) ;

		nodeitem = new KBNodeMonitor  (0, myitem) ;
		nodeitem->setText	(0, "Children"  ) ;
		nodeitem->setSelectable (false) ;
	}

	LITER
	(	KBAttr,
		attribs,
		attr,
		attr->showMonitor (attritem)
	)
	LITER
	(	KBNode,
		children,
		node,
		node->showMonitor (nodeitem)
	)
}

/*  KBNode								*/
/*  showMonitor	: Show or clear monitor for this node			*/
/*  parent	: QListView *	: Parent monitor			*/
/*  (returns)	: void		:					*/

void	KBNode::showMonitor
	(	QListView	*parent
	)
{
	if (parent == 0)
		setMonitor (monitor = 0) ;
	else	setMonitor (monitor = new KBNodeMonitor (this, parent)) ;
}

/*  KBNode								*/
/*  showMonitor	: Show or clear monitor for this node			*/
/*  parent	: QListViewItem * : Parent monitor			*/
/*  (returns)	: void		  :					*/

void	KBNode::showMonitor
	(	QListViewItem	*parent
	)
{
	if (parent == 0)
		setMonitor (monitor = 0) ;
	else	setMonitor (monitor = new KBNodeMonitor (this, parent)) ;
}

/*  KBNode								*/
/*  setMonitorSelect							*/
/*		: Set whether this nodes moditor is selected		*/
/*  select	: bool		: Selected or not			*/
/*  (returns)	: void		  :					*/

void	KBNode::setMonitorSelect
	(	bool	select
	)
{
	if (monitor != 0)
		monitor->listView()->setSelected (monitor, select) ;
}

/*  KBNode								*/
/*  findAllParams: Locate all paramaters in document			*/
/*  paramSet	 : QDict<KBParamSet> &  : Dictionary of found entries	*/
/*  (returns)	 : void			:				*/

void	KBNode::findAllParams
	(	QDict<KBParamSet>	&paramSet
	)
{
	/* Default is just to iterate down. The paramater node class	*/
	/* will override this method and add itself to the dictionary.	*/
	LITER
	(	KBNode,
		children,
		child,
		child->findAllParams (paramSet)
	)
}


/*  KBNode								*/
/*  objType	: Get object type for this node				*/
/*  (returns)	: KB::ObjType	: Object type				*/

KB::ObjType KBNode::objType ()
{
	/* Default is to return the type of the root object. This is	*/
	/* overridden by top-level types (KBForm, ...)			*/
	return	getRoot()->objType() ;
}

#if	! __KB_RUNTIME

/*  KBNode								*/
/*  doMultiProp	: Handle multiple properties editing			*/ 
/*  nodeList	: QList<KBNode> & : Nodes to be handled			*/
/*  (returns)	: bool		  : Success				*/

bool	KBNode::doMultiProp
	(	QList<KBNode> &nodeList
	)
{
	QList<KBAttr>	attrList	 ;
	KBNode	multi	(0, "MultiProp") ;
	bool	gotone	= false		 ;

	multi.attribs.find	(multi.m_attrNotes) ;
	multi.attribs.take	()	 ;
	attrList.setAutoDelete	(true)	 ;

	/* Iterate over each of the attributes which this node has. For	*/
	/* each, see if _all_ of the nodes in the list also have that	*/
	/* attribute; if so then replicate the attribute into the new	*/
	/* node.							*/
	LITER
	(	KBAttr,
		attribs,
		attr,

		bool	missing	= false ;

		LITER
		(	KBNode,
			nodeList,
			node,

			if (node->getAttr (attr->getName()) == 0)
			{	missing	= true ;
				break	;
			}
			if ((attr->getFlags() & KAF_COMMON) == 0)
			{	missing	= true ;
				break	;
			}
		)

		if (!missing)
		{	attrList.append (attr->replicate (&multi)) ;
			gotone = true ;
		}
	)

	if (!gotone)
		return	false ;

	if (!multi.propertyDlg (TR("Common properties")))
		return	false ;

	LITER
	(
		KBNode,
		nodeList,
		node,

		node->setMultiProp (&multi)
	)

	return	true ;
}

/*  KBNode								*/
/*  setMultiProp: Set multiple properties from node			*/
/*  multi	: KBNode *	: Node from which to set		*/
/*  (returns)	: void		:					*/

void	KBNode::setMultiProp
	(	KBNode	*multi
	)
{
	/* For each of our attributes, see if the setting node defines	*/
	/* it, in which case copy the value.				*/
	LITER
	(
		KBAttr,
		attribs,
		attr,

		if (multi->getAttr(attr->getName()) != 0)
			attr->setValue (multi->getAttrVal (attr->getName())) ;
	)

	if (isObject()) isObject()->updateProps () ;
}
#endif
